// Sound Effects Utility
// Plays different sounds based on generation outcomes when soundEffects is enabled

const SOUNDS = {
    success: 'https://assets.mixkit.co/active_storage/sfx/2869/2869-preview.mp3', // Success chime
    error: 'https://assets.mixkit.co/active_storage/sfx/2955/2955-preview.mp3', // Error buzz
};

export const playSound = async (type: 'success' | 'error', volume: number = 0.3) => {
    try {
        // Import authService dynamically to avoid circular dependencies
        const { authService } = await import('../services/authService');

        const currentUser = authService.getCurrentUser();
        if (!currentUser?.uid) {
            return; // Not logged in
        }

        // Use service method which handles API_URL correctly
        const preferences = await authService.getUserPreferences(currentUser.uid);
        // Navigate safe structure: data.uiConfig or just uiConfig depending on API return
        // authService.getUserPreferences returns "data || {}" from response.
        // The API response is { data: { theme... } } usually, so getUserPreferences returns the inner object?
        // Let's check authService again. 
        // getUserPreferences: async (uid) => { ... const { data } = await response.json(); return data || {}; }
        // So it returns the preference object directly.

        const soundEffectsEnabled = preferences?.uiConfig?.soundEffects;

        if (soundEffectsEnabled) {
            const audio = new Audio(SOUNDS[type]);
            audio.volume = volume;
            await audio.play().catch(err => console.log('Sound play failed:', err));
        }
    } catch (error) {
        // Silently fail - sounds are optional
        console.log('Sound check failed:', error);
    }
};

export const playSoundSync = (type: 'success' | 'error', soundEffectsEnabled: boolean, volume: number = 0.3) => {
    if (soundEffectsEnabled) {
        const audio = new Audio(SOUNDS[type]);
        audio.volume = volume;
        audio.play().catch(err => console.log('Sound play failed:', err));
    }
};
