
export const extractVideoFrames = async (file: File): Promise<string[]> => {
    return new Promise((resolve, reject) => {
        // Safety Timeout (5 seconds max per video)
        const timeoutId = setTimeout(() => {
            reject(new Error("Video framing timed out"));
        }, 5000);

        const video = document.createElement('video');
        video.preload = 'metadata';
        video.src = URL.createObjectURL(file);
        video.muted = true;
        video.playsInline = true;

        video.onloadedmetadata = async () => {
            try {
                const duration = video.duration;
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');

                if (!ctx) {
                    clearTimeout(timeoutId);
                    reject(new Error("Canvas context failed"));
                    return;
                }

                // Resize logic (Max 512px height for speed/token efficiency)
                const scale = Math.min(1, 512 / video.videoHeight);
                canvas.width = video.videoWidth * scale;
                canvas.height = video.videoHeight * scale;

                const frames: string[] = [];
                const seekPoints = [0.1, duration / 2, duration - 0.5].filter(t => t >= 0 && t <= duration);

                // If short video, just take start
                if (duration < 1) {
                    seekPoints.length = 0;
                    seekPoints.push(0);
                }

                for (const time of seekPoints) {
                    await new Promise<void>((r) => {
                        video.currentTime = time;
                        video.onseeked = () => r();
                    });
                    ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
                    frames.push(canvas.toDataURL('image/jpeg', 0.8));
                }

                URL.revokeObjectURL(video.src);
                clearTimeout(timeoutId);
                resolve(frames);
            } catch (e) {
                clearTimeout(timeoutId);
                reject(e);
            }
        };

        video.onerror = (e) => {
            clearTimeout(timeoutId);
            reject(new Error("Video load error"));
        };
    });
};

export const svgToPng = (file: File): Promise<string> => {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = (e) => {
            const img = new Image();
            img.onload = () => {
                const canvas = document.createElement('canvas');
                canvas.width = img.width;
                canvas.height = img.height;
                const ctx = canvas.getContext('2d');
                if (!ctx) return reject('No ctx');
                ctx.drawImage(img, 0, 0);
                resolve(canvas.toDataURL('image/png'));
            };
            img.src = e.target?.result as string;
        };
        reader.readAsDataURL(file);
    });
};
