import JSZip from 'jszip';

export interface DownloadableItem {
    url: string;
    filename: string;
}

/**
 * Downloads a list of items.
 * 
 * @param items List of objects with url and filename
 * @param zipName Name of the zip file (without .zip extension)
 * @param forceSeparate If true, forces individual downloads. If false, always creates a ZIP.
 */
export const downloadItems = async (items: DownloadableItem[], zipName: string, forceSeparate: boolean = false) => {
    if (items.length === 0) return;

    // Helper to trigger a single download
    const triggerDownload = (url: string, filename: string) => {
        const link = document.createElement('a');
        link.href = url;
        link.download = filename;
        link.target = "_blank"; // Safer for some browsers
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    };

    // If forceSeparate is true, download files individually
    if (forceSeparate) {
        for (const item of items) {
            try {
                const response = await fetch(item.url);
                const blob = await response.blob();
                const blobUrl = URL.createObjectURL(blob);
                triggerDownload(blobUrl, item.filename);
                URL.revokeObjectURL(blobUrl);
            } catch (e) {
                console.warn(`Fetch failed for ${item.url}, falling back to direct link`, e);
                triggerDownload(item.url, item.filename);
            }
            await new Promise(resolve => setTimeout(resolve, 500)); // Small delay to prevent browser blocking
        }
    } else {
        // Always create ZIP when forceSeparate is false (user clicked "Zip All")
        try {
            const zip = new JSZip();
            const folder = zip.folder(zipName) || zip;

            // Fetch to blobs sequentially
            for (const item of items) {
                try {
                    const response = await fetch(item.url);
                    if (!response.ok) throw new Error(`Failed to fetch ${item.url}`);
                    const blob = await response.blob();
                    folder.file(item.filename, blob);
                } catch (err) {
                    console.error(`Failed to add ${item.url} to zip:`, err);
                }
            }

            const content = await zip.generateAsync({ type: "blob" });
            const zipUrl = URL.createObjectURL(content);
            triggerDownload(zipUrl, `${zipName}.zip`);
            URL.revokeObjectURL(zipUrl);

        } catch (error) {
            console.error("ZIP creation failed:", error);
            alert("Failed to create ZIP archive. Downloading individually as fallback.");
            // Fallback
            for (const item of items) {
                triggerDownload(item.url, item.filename);
                await new Promise(resolve => setTimeout(resolve, 500));
            }
        }
    }
};
