

export enum AppView {
  LANDING = 'LANDING',
  AUTH = 'AUTH',
  DASHBOARD = 'DASHBOARD',
  CHAT_INTELLIGENCE = 'CHAT_INTELLIGENCE',
  IMAGE_GENERATION = 'IMAGE_GENERATION',
  VIDEO_GENERATION = 'VIDEO_GENERATION', // Veo
  MEDIA_ANALYSIS = 'MEDIA_ANALYSIS',
  IMAGE_EDITOR = 'IMAGE_EDITOR', // Nano Banana
  LIBRARY = 'LIBRARY',
  FAVORITES = 'FAVORITES',
  TRASH = 'TRASH',
  SETTINGS = 'SETTINGS',
  ADMIN = 'ADMIN',
  REVIEWER_AI = 'REVIEWER_AI',
  PROMPT_GENERATOR = 'PROMPT_GENERATOR',
  HISTORY = 'HISTORY'
}

export enum AdminSection {
  OVERVIEW = 'OVERVIEW',
  USERS = 'USERS',
  MESSAGES = 'MESSAGES',
  BROADCASTS = 'BROADCASTS', // New Broadcast Section
  INBOX = 'INBOX', // Simulated Email Inbox
  EMAIL_SETTINGS = 'EMAIL_SETTINGS',
  AUTH_SETTINGS = 'AUTH_SETTINGS', // New Tab
  SITE_SETTINGS = 'SITE_SETTINGS',
  LANDING_PAGE = 'LANDING_PAGE',
  AUDIT = 'AUDIT',
  BUG_REPORTS = 'BUG_REPORTS',
  TIPS = 'TIPS', // New Tips Management Section
  TUTORIAL_STEPS = 'TUTORIAL_STEPS', // New Tutorial Steps Section
  MARKET_INTELLIGENCE = 'MARKET_INTELLIGENCE', // New Market Intelligence Admin
  TRASH = 'TRASH' // New Admin Trash Section
}

export enum AuthView {
  LOGIN = 'LOGIN',
  REGISTER = 'REGISTER',
  VERIFY_EMAIL = 'VERIFY_EMAIL',
  FORGOT_PASSWORD = 'FORGOT_PASSWORD'
}

export interface User {
  uid: string;
  displayName: string;
  email: string;
  photoURL: string | null;
  country?: string;
  role: "admin" | "user";
  isVerified: boolean;
  isBanned?: boolean;
  isDeleted?: boolean; // Soft delete status
  createdAt: number;
  lastLoginAt: number;
  ipAddresses?: string[];
  subscription: "free" | "pro" | "enterprise";
  usageStats: {
    imagesGenerated: number;
    videosCreated: number;
    metadataGenerated?: number;
    trendSearches?: number;
    imagesReviewed?: number; // Reviewer AI
    imagesEdited?: number; // Image Editor
    promptsGenerated?: number; // Prompt Generator
    apiCallsThisMonth: number;
    apiCallsLimit: number;

    // New Daily/Monthly Stats Structure
    apiCalls?: number;
    daily?: { [key: string]: any };
    monthly?: { [key: string]: any };
    history?: {
      lastDay?: { [key: string]: any };
      lastMonth?: { [key: string]: any };
    };

    // New model-specific tracking
    modelUsage?: { [key: string]: number };
    modelLimits?: { [key: string]: number };
  };
}

export interface ActivityLog {
  id: string;
  type: 'generate' | 'system' | 'setting' | 'error';
  text: string;
  timestamp: number;
}

export interface ChatMessage {
  id: string;
  role: 'user' | 'model';
  text: string;
  timestamp: number;
  attachments?: Attachment[];
  isThinking?: boolean;
  groundingMetadata?: GroundingChunk[];
}

export interface Attachment {
  mimeType: string;
  data: string; // Base64
}

export interface GroundingChunk {
  web?: { uri: string; title: string };
  maps?: { uri: string; title: string };
}

export enum ModelType {
  // Thinking / Text
  FLASH_LITE = 'gemini-2.5-flash-lite-preview-09-2025',
  FLASH = 'gemini-2.5-flash',

  FLASH_3_0 = 'gemini-3-flash-preview',
  PRO = 'gemini-3-pro-preview', // Gemini 3 Pro (Text)
  PRO_2_5 = 'gemini-2.5-pro', // Gemini 2.5 Pro
  PRO_1_5 = 'gemini-1.5-pro-latest', // Gemini 1.5 Pro Latest


  // Creative / Image
  IMAGEN_FAST = 'imagen-4.0-fast-generate-preview',
  IMAGEN_STD = 'imagen-4.0-generate-001',
  IMAGEN_ULTRA = 'imagen-4.0-generate-ultra-preview',

  FLASH_IMAGE_2_0 = 'gemini-2.0-flash-image',
  FLASH_IMAGE_2_5 = 'gemini-2.5-flash-image',
  PRO_IMAGE_3_0 = 'gemini-3-pro-image-preview', // Nano Banana

  // Video
  VEO = 'veo-3.1-fast-generate-preview', // Veo 3.1 Fast
  VEO_HQ = 'veo-3.1-generate-preview', // Veo 3.1
  VEO_3_0 = 'veo-3.0-generate-preview', // Veo 3.0
  VEO_2_0 = 'veo-2.0-generate-preview', // Veo 2.0
}

export interface GeneratedImage {
  url: string;
  prompt: string;
}

export interface GeneratedVideo {
  uri: string;
  prompt: string;
}

export interface MetadataSettings {
  titleLength: number;
  descLength: number;
  keywordCount: number;
  keywordType: 'single' | 'double' | 'mixed';
  enableGrounding: boolean;
  exportFormat: 'txt' | 'json' | 'csv';
  useFilenameMode?: boolean;
  videoFrameOptions?: ('start' | 'middle' | 'end')[]; // Control which frames are used for generation (multi-select)
  selectedStockSites?: string[]; // IDs of enabled stock sites
}

export interface MetadataResult {
  title: string;
  description: string;
  commonKeywords: string[];
  longTailKeywords: string[];
  iStockTags: string[];
  category?: string;
  sources?: { title: string; uri: string }[];
  filename?: string;
  modelUsed?: string; // Track which model actually generated this result
}

export interface BatchItem {
  id: string;
  file: File;
  preview: string;
  status: 'pending' | 'processing' | 'complete' | 'error' | 'waiting';
  result?: MetadataResult;
  error?: string;
  progress: number; // 0-100
  generationInfo?: {
    model: string;
    apiKey: string;
  };
}

// --- Market Intelligence Types ---

export interface StockSite {
  id: string;
  name: string;
  url: string;
  category: string;
  description: string;
  contributors: string;
  librarySize: string;
  commission: string;
  isAiPowered?: boolean;
  isVisible?: boolean; // Controlled by Admin
  isVisibleMarket?: boolean; // New independent toggle for Market Intelligence
  pros?: string[];
  cons?: string[];
  isDeleted?: boolean;
}

export interface TrendReport {
  id: string;
  query: string;
  timestamp: number;
  selectedPlatforms: string[]; // List of names or IDs

  // 1. Insights
  insights: {
    create: { title: string; reason: string }[];
    caution: { title: string; reason: string }[];
    avoid: { title: string; reason: string }[];
    takeaways: string[];
    confidenceScore: number;
  };

  // 2. Saturation
  saturation: {
    level: 'Low' | 'Medium' | 'High';
    score: number; // 1-10
    competitionLevel: number;
    totalVolume: number;
    supplyData: { name: string; supply: number; demand: number }[];
  };

  // 3. Trends
  trend: {
    direction: 'Rising' | 'Stable' | 'Declining';
    data: { name: string; val: number }[]; // Line chart data
    seasonality: string;
    subTrends: string[];
    forecast: string;
  };

  // 4. Platforms
  platformInsights?: {
    name: string;
    bestContent: string;
    pricing: string;
    tips: string;
    acceptanceRate: string;
  }[];

  // 5. Creative
  creative: {
    ideas: string[];
    colorPalette: string[]; // Hex codes
    prompts: string[];
    moodBoardUrls: string[]; // Placeholders for now
  };

  // 6. Pricing
  pricing: {
    avgPrice: string;
    potentialMonthly: string;
    roi: string;
    licensing: string;
  };

  // 7. Community
  community: {
    quotes: { text: string; author: string }[];
    sentiment: string;
  };

  // Citations
  sources: { title: string; url: string; date: string }[];
}

// --- Settings & Admin Types ---

export interface ApiKeyData {
  id: string;
  key: string; // masked
  fullKey: string;
  status: 'valid' | 'invalid' | 'validating' | 'unchecked';
  isEnabled?: boolean; // New toggle status
  tier?: 'FREE' | 'PAID' | 'unknown'; // API key tier based on RPM limits
  addedAt: number;
  lastUsed?: number;
  usageCount: number;
  validationAttempts: number;
  usage?: number; // Usage count from backend (api_key_usage)
}

export interface ValidationLog {
  id: string;
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'error' | 'warning' | 'network' | 'process';
}

export interface SupportMessage {
  id: string;
  sender: 'user' | 'admin';
  text: string;
  timestamp: number;
  status: 'sending' | 'sent' | 'read' | 'failed';
  attachments?: { name: string; type: string; url: string; size: string }[];
  isEdited?: boolean;
  userId?: string; // For admin view
  userEmail?: string; // For admin view
  userName?: string; // For admin view notifications
}

export interface SiteSettings {
  // General
  appStaticApiKey?: string; // For system-wide trend generation
  systemApiUsage?: number; // Track usage of system key
  appIcon?: string; // Custom app icon (favicon) URL
  appLogo?: string; // Custom app logo URL
  adobeStockInstruction?: string; // Custom instruction for Adobe Stock category
  titleInstruction?: string; // Deprecated in favor of specific types below
  titleInstructionImage?: string;
  titleInstructionVideo?: string;
  titleInstructionSvg?: string;
  titleInstructionSvg?: string;
  descriptionInstruction?: string; // Deprecated
  descriptionInstructionImage?: string;
  descriptionInstructionVideo?: string;
  descriptionInstructionSvg?: string;
  keywordInstruction?: string;
  keywordGenerationPrompt?: string;
  emailVerificationRequired: boolean;
  allowRegistrations: boolean;
  adminApprovalRequired: boolean;
  defaultApiLimit: number;
  creditsPerApiKey: number; // Legacy total

  // New Model Limits Config
  modelLimits: {
    // Thinking
    flash_lite_2_5: number;
    flash_2_5: number;
    pro_2_5: number;
    pro_3_0: number;

    // Creative
    gemini_2_0_flash_image: number;
    gemini_2_5_flash_image: number;
    gemini_3_pro_image: number; // Nano Banana
    imagen_4_fast: number;
    imagen_4_std: number;
    imagen_4_ultra: number;

    // Video
    veo_3_1: number;
    veo_3_1_fast: number;
    veo_3_0: number;
    veo_2_0: number;
  };

  // Dynamic Model ID Overrides
  modelIds: {
    [key: string]: string;
  };

  // Model Visibility Toggle
  modelVisibility?: {
    [key: string]: boolean; // key is the model ID, value is whether it's visible
  };

  // Model Access by Tier - which models are enabled for FREE vs PAID keys
  modelAccessTiers?: {
    [modelId: string]: {
      free: boolean;  // Is this model enabled for FREE API keys?
      paid: boolean;  // Is this model enabled for PAID API keys (Tier 1/2/3)?
    };
  };

  // Model Limits by Tier - different credit amounts for FREE vs PAID keys
  modelLimitsByTier?: {
    [modelKey: string]: {
      free: number;   // Credits for FREE API keys
      paid: number;   // Credits for PAID API keys
    };
  };

  maintenanceMode: boolean;
  maintenanceMessage: string;
  maintenancePreCode: string;
  maintenanceEndTime?: number; // Timestamp for countdown
  safetyFilters: boolean;

  // Google
  googleClientId?: string;
  enableSimulatedGoogleLogin: boolean;
  enableGoogleLogin: boolean;

  // Facebook
  facebookClientId?: string;
  enableSimulatedFacebookLogin: boolean;
  enableFacebookLogin: boolean;

  // Twitter
  twitterClientId?: string;
  enableSimulatedTwitterLogin: boolean;
  enableTwitterLogin: boolean;

  smtpConfig: {
    host: string;
    port: number;
    user: string;
    pass: string;
    fromName: string;
    fromEmail: string;
    secure: boolean;
  };

  // Sidebar Management
  sidebarSettings?: {
    [key: string]: SidebarItemSetting;
  };
}

export interface SidebarItemSetting {
  id: string;
  enabled: boolean;
  comingSoon: boolean;
}

export interface SystemEmail {
  id: string;
  to: string;
  subject: string;
  body: string; // HTML
  type: 'verification' | 'reset' | 'welcome' | 'broadcast';
  sentAt: number;
  status: 'delivered' | 'bounced' | 'opened';
}

export interface Broadcast {
  id: string;
  title: string;
  content: string;
  type: 'announcement' | 'feature' | 'notice' | 'urgent';
  targetAudience: string; // 'All', 'Free', 'Pro', 'Enterprise'
  sentAt: number;
  recipients: number;
  openRate: number;
  readCount?: number; // Track number of views
  openedBy?: string[]; // IDs of users who opened it
  isDeleted?: boolean; // Soft delete
}

export interface BugReport {
  id: string;
  userId: string;
  userEmail: string;
  title: string;
  description: string;
  steps: string;
  category?: string;
  severity: 'low' | 'medium' | 'high' | 'critical';
  status: 'open' | 'in_progress' | 'resolved';
  timestamp: number;
  isDeleted?: boolean;
}

export interface AuditLog {
  id: string;
  action: string;
  admin: string;
  target: string;
  timestamp: number;
  status: 'success' | 'failure';
  details: string;
  ip: string;
}

export interface Tip {
  id: string;
  title: string;
  text: string;
  category?: string;
  createdAt: number;
  isDeleted?: boolean; // Soft delete
}

export interface TutorialStep {
  id: string;
  title: string;
  description: string;
  actionLabel?: string;
  actionLink?: string; // AppView or specific tab ID
  order: number;
  isDeleted?: boolean; // Soft delete
}

// --- Image Reviewer Types ---

export interface AnatomyCheckResult {
  score: number; // 0 or 100
  hasHumans: boolean;
  defects: string[];
  analysis: string;
}

export interface HandDetectionResult {
  hasHands: boolean;
  confidence: number;
  details: string;
  fixedImageBase64?: string;
}

export interface QualityCheckResult {
  score: number; // 0-100
  issues: string[];
  reasoning: string;
}

export interface TwoStepReviewResult {
  verdict: 'ACCEPTED' | 'REJECTED';
  totalScore: number;
  scoreA: number; // Anatomy check score
  scoreB: number; // Quality check score
  killSwitchActivated: boolean;
  anatomyAnalysis: AnatomyCheckResult;
  qualityAnalysis: QualityCheckResult;
  handDetection?: HandDetectionResult;
  reason?: string;
}