
import { GoogleGenAI, Modality, Type } from "@google/genai";
import { Attachment, GroundingChunk, MetadataResult, MetadataSettings, ModelType, TrendReport } from "../types";
import { MODEL_TYPE_TO_SETTINGS } from "../constants";

// Helper to initialize AI with specific key
const getAI = (apiKey: string) => {
    const raw = apiKey || process.env.API_KEY || '';
    const clean = raw.replace(/[^\x20-\x7E]/g, '').trim();
    return new GoogleGenAI({ apiKey: clean });
};

// --- Model Fallback Helpers ---

export const getPreferredModels = (category: 'thinking' | 'creative' | 'video'): string[] => {
    try {
        let key = 'sf_thinking_models';
        if (category === 'creative') key = 'sf_creative_models';
        if (category === 'video') key = 'sf_video_models';

        const stored = localStorage.getItem(key);
        if (stored) {
            let models = JSON.parse(stored);

            // Migration map for old/incorrect model names
            const MODEL_MIGRATIONS: { [key: string]: string } = {
                'gemini-2.5-pro-preview': 'gemini-2.5-pro',
                'gemini-2.5-pro-preview-09-2025': 'gemini-2.5-pro',
                'gemini-2.0-flash-thinking-exp-1219': 'gemini-2.5-pro', // Old placeholder
                'gemini-2.0-flash-thinking-exp': ModelType.FLASH_3_0, // Data cleanup: Migrate ghost model
                'gemini-2.0-flash': ModelType.FLASH_3_0, // AUTOMATIC MIGRATION: 2.0 Flash -> 3.0 Flash
                'gemini-2.0-flash-exp': ModelType.FLASH_3_0, // AUTOMATIC MIGRATION: 2.0 Flash Exp -> 3.0 Flash
            };

            // Replace old model names with new ones
            if (Array.isArray(models)) {
                let migrated = false;
                models = models.map(model => {
                    if (MODEL_MIGRATIONS[model]) {
                        migrated = true;
                        console.log(`Migrating model: ${model} -> ${MODEL_MIGRATIONS[model]}`);
                        return MODEL_MIGRATIONS[model];
                    }
                    return model;
                });

                // Filter out any Gemini 1.5 models - REMOVED to allow 1.5 Pro again
                // const originalLength = models.length;
                // models = models.filter(model => !model.includes('gemini-1.5'));
                // if (models.length !== originalLength) {
                //     console.log('Filtered out deprecated Gemini 1.5 models');
                //     migrated = true;
                // }

                // Remove duplicates
                models = [...new Set(models)];

                // Save migrated models back to localStorage
                if (migrated) {
                    localStorage.setItem(key, JSON.stringify(models));
                }

                if (models.length > 0) return models;
            }
        }
    } catch (e) { console.error("Error reading preferred models", e); }

    // Defaults if local storage is empty
    if (category === 'thinking') return [ModelType.FLASH];
    if (category === 'video') return ['veo-3.1-fast-generate-preview'];
    return [ModelType.IMAGEN_STD];
};

export const setPreferredModel = (category: 'thinking' | 'creative' | 'video', newPrimaryAndOnly: string) => {
    try {
        let key = 'sf_thinking_models';
        if (category === 'creative') key = 'sf_creative_models';
        if (category === 'video') key = 'sf_video_models';

        const models = getPreferredModels(category);
        if (models[0] === newPrimaryAndOnly) return;

        // Reorder: Move newPrimary to front
        const newOrder = [newPrimaryAndOnly, ...models.filter(m => m !== newPrimaryAndOnly)];
        console.log(`[GeminiService] 🔄 Auto-switching preferred model to: ${newPrimaryAndOnly} for category ${category}`);
        localStorage.setItem(key, JSON.stringify(newOrder));
    } catch (e) {
        console.error("Failed to update preferred model persistence", e);
    }
};

const executeWithFallback = async <T>(
    category: 'thinking' | 'creative',
    operation: (model: string) => Promise<T>
): Promise<{ result: T; modelUsed: string }> => {
    const models = getPreferredModels(category);
    let lastError: any;

    for (const model of models) {
        try {
            console.log(`Attempting with model: ${model}`);
            const result = await operation(model);
            return { result, modelUsed: model };
        } catch (error: any) {
            console.warn(`Model ${model} failed:`, error);
            lastError = error;

            // Check if error is likely rate limit (429) or other transient issue to justify retry
            // Simple logic: try next model for any error to be robust
            continue;
        }
    }

    // --- Safety Net Fallback REMOVED ---
    // Strict enforcement: if preferred models fail, we do NOT fallback to hidden models.


    throw lastError || new Error("All selected models failed.");
};

// --- Text & Multimodal Chat ---

// Helper for rate limit retries
const retryOperation = async <T>(operation: () => Promise<T>, maxRetries = 2, baseDelay = 1000): Promise<T> => {
    let delay = baseDelay;
    for (let i = 0; i < maxRetries; i++) {
        try {
            return await operation();
        } catch (error: any) {
            const msg = error?.message || '';
            const status = error?.status || error?.response?.status;
            // Check for 429 (Too Many Requests) or 503 (Service Unavailable)
            const isRateLimit = msg.includes('429') || status === 429 || msg.includes('Too Many Requests');
            const isServerOverload = msg.includes('503') || status === 503;

            if ((isRateLimit || isServerOverload) && i < maxRetries - 1) {
                console.warn(`⚠️ API Limit hit (${status || '429'}). Retrying in ${delay}ms... (Attempt ${i + 1}/${maxRetries})`);
                await new Promise(resolve => setTimeout(resolve, delay));
                delay *= 2; // Exponential backoff
                continue;
            }
            throw error;
        }
    }
    throw new Error(`Operation failed after ${maxRetries} retries.`);
};


export const generateChatResponse = async (
    prompt: string,
    baseModel: string, // Kept for backward compatibility or specific overrides
    history: { role: string; parts: { text: string }[] }[],
    attachments: Attachment[] = [],
    useThinking: boolean = false,
    useSearch: boolean = false,
    useMaps: boolean = false,
    location?: { latitude: number; longitude: number },
    apiKeys?: string | string[] // Added optional apiKeys (default to env if missing for compat)
): Promise<{ text: string; groundingChunks?: GroundingChunk[] }> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for chat.");

    // Construct parts
    const parts: any[] = [];

    attachments.forEach(att => {
        parts.push({
            inlineData: {
                mimeType: att.mimeType,
                data: att.data
            }
        });
    });
    parts.push({ text: prompt });

    // Use Thinking models list if Thinking is requested, otherwise use passed model or default
    const modelsToTry = useThinking ? getPreferredModels('thinking') : [baseModel];

    let lastError;

    // --- NESTED LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[Chat] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            // console.log(`[Chat] Trying with Key: ${maskedKey} on Model: ${model}`); // Reduced log noise

            try {
                const ai = getAI(apiKey);
                const config: any = {};

                // Thinking Config
                if (useThinking && (model === ModelType.PRO || model === ModelType.PRO_2_5 || model.includes('thinking'))) {
                    config.thinkingConfig = { thinkingBudget: 32768 };
                }

                // Tools Config (Search / Maps)
                const tools: any[] = [];
                if (useSearch) {
                    tools.push({ googleSearch: {} });
                }
                // Maps can be used with Search
                if (useMaps) {
                    tools.push({ googleMaps: {} });
                    if (location) {
                        config.toolConfig = {
                            retrievalConfig: {
                                latLng: {
                                    latitude: location.latitude,
                                    longitude: location.longitude
                                }
                            }
                        }
                    }
                }

                if (tools.length > 0) {
                    config.tools = tools;
                }

                const response = await ai.models.generateContent({
                    model: model,
                    contents: { role: 'user', parts },
                    config
                });

                const groundingChunks = response.candidates?.[0]?.groundingMetadata?.groundingChunks as GroundingChunk[] | undefined;

                if (useThinking) setPreferredModel('thinking', model);

                return {
                    text: response.text || "No response generated.",
                    groundingChunks
                };
            } catch (e: any) {
                lastError = e;
                console.warn(`[Chat] Error with Key ${maskedKey}: ${e.message}`);

                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue; // Try next key
                }
                continue; // Try next key for other errors too? Yes, usually safer.
            }
        }
    }
    throw lastError || new Error("All chat models and keys failed.");
};

// --- Metadata Generation ---

// Custom Error for Exhaustion
export class RateLimitExhaustedError extends Error {
    constructor(message: string) {
        super(message);
        this.name = "RateLimitExhaustedError";
    }
}

export const generateMetadata = async (
    apiKeys: string | string[],
    base64Image: string | string[],
    mimeType: string,
    settings: MetadataSettings,
    context?: { keywords?: string; excludeKeywords?: string; description?: string; filename?: string; adobeStockInstruction?: string; titleInstruction?: string; descriptionInstruction?: string; keywordInstruction?: string; keywordGenerationPrompt?: string },
    modelId?: string,
    signal?: AbortSignal
): Promise<MetadataResult> => {
    // Normalize apiKeys to array
    const keys = Array.isArray(apiKeys) ? apiKeys : [apiKeys];
    if (keys.length === 0) throw new Error("No API keys provided.");

    // Determine the effective Keyword Prompt
    // Priority: 1. Context (passed from caller) 2. LocalSettings (backend/admin override stored locally) 3. Default
    let effectiveKeywordPrompt = context?.keywordGenerationPrompt || '';

    if (!effectiveKeywordPrompt) {
        try {
            const settingsStr = localStorage.getItem('site_settings');
            if (settingsStr) {
                const siteSettings = JSON.parse(settingsStr);
                effectiveKeywordPrompt = siteSettings.keywordGenerationPrompt || '';
            }
        } catch (e) {
            console.error('Error reading keyword generation prompt:', e);
        }
    }

    // --- ROBUST EXECUTION STRATEGY ---
    // 1. Determine list of models to try
    //    - If modelId is provided, try it FIRST.
    //    - Then try other preferred models as backup.
    const preferred = getPreferredModels('thinking');
    const modelsToTry = modelId
        ? [modelId, ...preferred.filter(m => m !== modelId)]
        : preferred;

    // Strict enforcement: Do not push Flash default if user hasn't enabled it
    if (modelsToTry.length === 0) throw new Error("No enabled models available for metadata generation.");

    let lastError;

    // --- DOUBLE CHECK STRATEGY: 2 PASS LOOP ---
    for (let attempt = 1; attempt <= 2; attempt++) {
        if (attempt === 2) {
            console.warn("[generateMetadata] ⚠️ All keys/models failed pass 1. Retrying (Pass 2/2)...");
        }

        // Outer Loop: Mode/Model Version
        for (const model of modelsToTry) {
            if (signal?.aborted) throw new Error("Aborted");
            console.log(`[generateMetadata] Switching to Model Version: ${model} (Pass ${attempt}/2)`);

            // Inner Loop: API Keys
            for (const apiKey of keys) {
                if (signal?.aborted) throw new Error("Aborted");
                const ai = getAI(apiKey);
                const maskedKey = apiKey.substring(0, 8) + '...';
                // console.log(`[generateMetadata] Trying with Key: ${maskedKey} on Model: ${model}`);

                const prompt = `
    Analyze this image for stock photography metadata.
    ${settings.useFilenameMode && context?.filename ? `
    PRIORITY CONTEXT: Filename-Based Generation.
    The filename is: "${context.filename.replace(/"/g, '')}".
    Focus on the subject matter implied by this filename for the Title, Description, and Keywords.
    Use the filename as the ground truth for the subject.
    ` : ''}
    ${context?.description ? `Context: ${context.description}` : ''}
    ${(context?.adobeStockInstruction || `Determine the most appropriate "category" for this image.
    CRITICAL: The category MUST be exactly one of the values from this list (choose the best match):
    Animals, Buildings and Architecture, Business, Drinks, The Environment, States of Mind, Food, Graphic Resources, Hobbies and Leisure, Industry, Landscapes, Lifestyle, People, Plants and Flowers, Culture and Religion, Science, Social Issues, Sports, Technology, Transport, Travel.`) ? `
    ADOBE STOCK CATEGORY INSTRUCTION (Primary Focus for Category field):
    ${context?.adobeStockInstruction || `Determine the most appropriate "category" for this image.
    CRITICAL: The category MUST be exactly one of the values from this list (choose the best match):
    Animals, Buildings and Architecture, Business, Drinks, The Environment, States of Mind, Food, Graphic Resources, Hobbies and Leisure, Industry, Landscapes, Lifestyle, People, Plants and Flowers, Culture and Religion, Science, Social Issues, Sports, Technology, Transport, Travel.`}
    ` : ''}
    ${context?.keywords ? `Additional focus keywords: ${context.keywords}` : ''}
    ${context?.excludeKeywords ? `STRICT NEGATIVE CONSTRAINTS (EXCLUDE THESE KEYWORDS): ${context.excludeKeywords}` : ''}
    
    Requirements:
    1. Title: Target approximately ${settings.titleLength} characters.
       - MINIMUM LENGTH: ${Math.floor(settings.titleLength * 0.8)} characters. If the generated title is too short, EXPAND it with more descriptive adjectives and details.
       - MAXIMUM LENGTH: ${settings.titleLength + 5} characters.
       - ${context?.titleInstruction || "Engaging, descriptive, and SEO-optimized."}
    2. Description: Target approximately ${settings.descLength} characters. ${context?.descriptionInstruction || "Write a detailed, rich description that fills this space."} MINIMUM LENGTH: ${Math.floor(settings.descLength * 0.9)} characters. IF IT IS TOO SHORT, ADD MORE VISUAL DETAILS. Do NOT exceed ${settings.descLength} characters.
    3. Keywords: ${effectiveKeywordPrompt ? effectiveKeywordPrompt.replace(/{{keywordCount}}/g, String(settings.keywordCount)) : `🚨 CRITICAL PRIORITY 🚨 - You MUST generate EXACTLY ${settings.keywordCount} keywords TOTAL when combining 'commonKeywords' and 'longTailKeywords'. ${context?.keywordInstruction || "Provide a mix of common and long-tail keywords relevant to the subject."}
       
       📊 COUNT ENFORCEMENT (NON-NEGOTIABLE):
       -  ⚠️ MANDATORY: commonKeywords.length + longTailKeywords.length MUST BE BETWEEN ${settings.keywordCount - 3} AND ${settings.keywordCount + 10}
       - IF YOU GENERATE ONLY ${Math.floor(settings.keywordCount / 2)} KEYWORDS (e.g., ${Math.floor(settings.keywordCount / 2)} when ${settings.keywordCount} requested), THIS IS A CRITICAL FAILURE
       - EXAMPLE: User wants 50 keywords → You MUST generate 47-60 keywords. Generating 20-25 is COMPLETELY UNACCEPTABLE
       - IT IS BETTER TO HAVE ${settings.keywordCount + 10} KEYWORDS THAN ${settings.keywordCount - 10} KEYWORDS. AIM HIGH, NOT LOW.
       
       ✅ KEYWORD SEO ORDER (CRITICAL FOR STOCK PLATFORMS):
       First 5 keywords are MOST IMPORTANT for ranking. Order keywords by priority:
       1. Main subject (what is it?)
       2. Secondary subject (what else is in the image?)
       3. Usage context (background, copy space, banner, header)
       4. Concept/emotion (happiness, freedom, success, celebration)
       5. Descriptive attributes (colors, lighting, composition)
       6. Style (modern, vintage, minimalist, realistic)
       7. Location/season (urban, beach, summer, winter, indoor, outdoor)
       
       🔟 NEGATIVE CONTROL (What to AVOID):
       ❌ Wrong/unrelated locations
       ❌ Objects not present in the image
       ❌ Famous brand names or trademarks
       ❌ Spam repetition or duplicate keywords
       ❌ Misleading or irrelevant terms that hurt SEO ranking
       
       💡 KEYWORD GENERATION STRATEGY:
       - Think like a BUYER searching for this image, not the creator
       - Use highly relevant, specific keywords
       - NO duplicates or near-duplicates
       - When low on ideas: Add color variations (blue, azure, navy, cerulean), lighting (bright, golden hour, dusk, soft light), mood (cheerful, serene, energetic, peaceful), textures (smooth, rough, grainy), composition (centered, symmetrical, rule-of-thirds), style (modern, vintage, minimalist), seasonal (summer, winter, festive), contextual (indoor, outdoor, urban, rural), conceptual (freedom, growth, innovation, teamwork)
       
       🎯 FINAL VERIFICATION BEFORE RESPONDING:
       Count commonKeywords.length + longTailKeywords.length. If not between ${settings.keywordCount - 3} and ${settings.keywordCount + 10}, ADD MORE KEYWORDS IMMEDIATELY using synonyms and related concepts.`}
    
    ${settings.keywordType === 'single' ? "STRICT REQUIREMENT: All keywords must be SINGLE words only. No phrases." : ''}
    ${settings.keywordType === 'double' ? "STRICT REQUIREMENT: All keywords must be TWO-WORD phrases only. No single words." : ''}
    ${settings.keywordType === 'mixed' ? "Generate a natural mix of single words and phrases." : ''}
    ${context?.keywords ? `IMPORTANT: You MUST include the following keywords in the 'commonKeywords' or 'longTailKeywords' list: ${context.keywords}` : ''}
    ${context?.excludeKeywords ? `CRITICAL: You MUST NOT use the following keywords (or close variations) in ANY part of the result: ${context.excludeKeywords}` : ''}
    
    4. Categorize keywords into 'common', 'longTail', and 'iStockTags'.
       - 'iStockTags': Generate 5-10 EXTRA keywords optimized for iStock vocabulary. These are BONUS and do NOT count towards the ${settings.keywordCount} limit.
    
    ${settings.enableGrounding ? 'Use Google Search to find relevant trending topics or location data if applicable.' : ''}
  `;

                const parts: any[] = [];

                if (Array.isArray(base64Image)) {
                    base64Image.forEach(img => {
                        parts.push({
                            inlineData: {
                                data: img,
                                mimeType: mimeType
                            }
                        });
                    });
                } else {
                    parts.push({
                        inlineData: {
                            data: base64Image,
                            mimeType: mimeType
                        }
                    });
                }

                parts.push({ text: prompt });

                const config: any = {
                    temperature: 0.4,
                };

                if (settings.enableGrounding) {
                    console.log("🌍 SEARCH GROUNDING ENABLED: Performing live Google Search for facts/location verification...");
                    config.tools = [{ googleSearch: {} }];
                    parts.push({ text: "\n\nCRITICAL: You are a JSON generator. Use Google Search to verify facts/locations, then output the result as a STRICT, RAW JSON object (no markdown, no backticks). Keys: title, description, commonKeywords, longTailKeywords, iStockTags, category." });
                } else {
                    config.responseMimeType = "application/json";
                    config.responseSchema = {
                        type: Type.OBJECT,
                        properties: {
                            title: { type: Type.STRING },
                            description: { type: Type.STRING },
                            commonKeywords: { type: Type.ARRAY, items: { type: Type.STRING } },
                            longTailKeywords: { type: Type.ARRAY, items: { type: Type.STRING } },
                            iStockTags: { type: Type.ARRAY, items: { type: Type.STRING } },
                            category: { type: Type.STRING }
                        },
                        required: ["title", "description", "commonKeywords", "longTailKeywords", "iStockTags", "category"]
                    };
                }

                try {
                    // Perform generation with simple retry for transient 503s within the SAME key/model combo
                    let response;
                    for (let i = 0; i < 3; i++) {
                        if (signal?.aborted) throw new Error("Aborted");
                        try {
                            response = await Promise.race([
                                ai.models.generateContent({
                                    model: model,
                                    contents: { parts },
                                    config
                                }),
                                new Promise<any>((_, reject) => setTimeout(() => reject(new Error("Timeout: Generation took longer than 60s")), 60000))
                            ]);
                            break; // Success
                        } catch (e: any) {
                            // 503 Service Unavailable - Retry same key/model a few times
                            if ((e.message?.includes('503') || e.status === 503) && i < 2) {
                                const delay = 500 * Math.pow(2, i);
                                console.info(`[Network] 503 Service Unavailable on ${model} (Key: ${maskedKey}). Retrying in ${delay}ms...`);
                                await new Promise(r => setTimeout(r, delay));
                                continue;
                            }
                            // If it's a 429, THROW immediately to switch key
                            if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests')) {
                                throw e;
                            }
                            throw e; // Other errors
                        }
                    }
                    if (!response) throw new Error("Failed to generate content after internal retries.");

                    let resultText = response.text || "{}";

                    // Robust cleanup
                    resultText = resultText.replace(/```json/g, '').replace(/```/g, '').trim();

                    let parsed;
                    try {
                        parsed = JSON.parse(resultText);
                    } catch (e) {
                        console.error("JSON Parse Error:", e, "Received:", resultText);
                        // Fallback: try to find json-like structure
                        const match = resultText.match(/\{[\s\S]*\}/);
                        if (match) {
                            try {
                                parsed = JSON.parse(match[0]);
                            } catch (e2) { throw new Error("Failed to parse JSON response"); }
                        } else {
                            throw new Error("Invalid JSON response");
                        }
                    }

                    const sources: { title: string; uri: string }[] = [];
                    if (settings.enableGrounding && response.candidates?.[0]?.groundingMetadata?.groundingChunks) {
                        response.candidates[0].groundingMetadata.groundingChunks.forEach((chunk: any) => {
                            if (chunk.web) {
                                sources.push({ title: chunk.web.title, uri: chunk.web.uri });
                            }
                        });
                    }

                    // --- STRICT ENFORCEMENT ---
                    // 0. Enforce Title Length
                    const maxTitleLen = settings.titleLength + 5;
                    if (parsed.title && parsed.title.length > maxTitleLen) {
                        // Smart truncate: Cut at max len, then backtrack to last space
                        let cutTitle = parsed.title.substring(0, maxTitleLen);
                        const lastSpace = cutTitle.lastIndexOf(' ');
                        // Ensure we don't cut too much (e.g. if one super long word)
                        if (lastSpace > maxTitleLen * 0.7) {
                            cutTitle = cutTitle.substring(0, lastSpace);
                        }
                        parsed.title = cutTitle;
                    }

                    // 1. Enforce Description Length
                    if (parsed.description && parsed.description.length > settings.descLength) {
                        // Hard cut at limit
                        let cutDesc = parsed.description.substring(0, settings.descLength);
                        // Try to back up to the last sentence end to be polite, but only if we don't lose too much
                        const lastPeriod = cutDesc.lastIndexOf('.');
                        const lastExclaim = cutDesc.lastIndexOf('!');
                        const lastQuestion = cutDesc.lastIndexOf('?');
                        const bestEnd = Math.max(lastPeriod, lastExclaim, lastQuestion);

                        if (bestEnd > settings.descLength * 0.7) {
                            cutDesc = cutDesc.substring(0, bestEnd + 1);
                        } else {
                            // If no sentence end nearby, cut at last word
                            const lastSpace = cutDesc.lastIndexOf(' ');
                            if (lastSpace > settings.descLength * 0.8) {
                                cutDesc = cutDesc.substring(0, lastSpace) + '...';
                            }
                        }
                        parsed.description = cutDesc;
                    }

                    // 2. Enforce Keyword Count (Trim excess ONLY if significantly over)
                    // Only count common + longTail for the limit
                    let currentCount = (parsed.commonKeywords?.length || 0) + (parsed.longTailKeywords?.length || 0);

                    // Allow a buffer of up to 10 extra keywords (better to have 55 than 25 when user wants 50)
                    const maxAllowed = settings.keywordCount + 10;
                    if (currentCount > maxAllowed) {
                        let toRemove = currentCount - settings.keywordCount;

                        // Priority for removal: longTail -> common (Do NOT remove iStockTags as they are bonus)
                        if (parsed.longTailKeywords && toRemove > 0) {
                            const canRemove = Math.min(parsed.longTailKeywords.length, toRemove);
                            parsed.longTailKeywords = parsed.longTailKeywords.slice(0, parsed.longTailKeywords.length - canRemove);
                            toRemove -= canRemove;
                        }
                        if (parsed.commonKeywords && toRemove > 0) {
                            const canRemove = Math.min(parsed.commonKeywords.length, toRemove);
                            parsed.commonKeywords = parsed.commonKeywords.slice(0, parsed.commonKeywords.length - canRemove);
                            toRemove -= canRemove;
                        }
                    }

                    // SUCCESS: Return result immediately
                    setPreferredModel('thinking', model);
                    return {
                        ...parsed,
                        sources: sources.length > 0 ? sources : undefined,
                        modelUsed: model // Return the actual model that succeeded
                    };

                } catch (e: any) {
                    lastError = e;
                    console.warn(`[generateMetadata] Error with Key ${maskedKey} on Model ${model}:`, e.message);

                    // If Content Blocked/Safety, fail fast for this model (unlikely to change with key)
                    // Actually, safety block might be safer to just try next model? Let's treat as skippable.

                    // If 429, we loop to NEXT KEY.
                    if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests')) {
                        console.log(`[generateMetadata] 429 Rate Limit hit. Switching to next API Key...`);
                        continue; // Loop to next key
                    }

                    // For other errors (except maybe auth), we also try next key just in case it's key-specific?
                    // Or if it's 503 that persisted?
                    // Robust approach: Try next key for ANY error.
                    continue;
                }
            }
            // End of Keys loop. If we are here, ALL keys failed for this model.
            console.warn(`[generateMetadata] All keys failed for model ${model}. Switching to next model...`);
        }
    } // End of attempt loop

    // End of Models loop. If we are here, ALL models and ALL keys failed after 2 passes.
    console.error("[generateMetadata] FATAL: All API keys and Models exhausted after 2 passes.");
    throw new RateLimitExhaustedError("All API keys and models are rate limited or unavailable. Please add a new API key or wait for credits to reset.");
};

export const regenerateTitle = async (
    apiKeys: string | string[],
    currentTitle: string,
    settings: MetadataSettings
): Promise<string> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for title regeneration.");

    const prompt = `Rewrite this stock photo title to be more engaging and SEO friendly. Max ${settings.titleLength} chars. Title: "${currentTitle}"`;

    // Determine models to try (Thinking models preferred)
    const modelsToTry = getPreferredModels('thinking');

    let lastError;
    let success = false;
    let result = currentTitle;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[RegenerateTitle] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';

            try {
                const ai = getAI(apiKey);
                const config: any = {
                    responseMimeType: 'application/json',
                    responseSchema: {
                        type: Type.OBJECT,
                        properties: { title: { type: Type.STRING } }
                    }
                };

                // Thinking Config
                if (model.includes('thinking') || model === ModelType.PRO || model === ModelType.PRO_2_5) {
                    config.thinkingConfig = { thinkingBudget: 4096 };
                }

                const response = await ai.models.generateContent({
                    model: model,
                    contents: { parts: [{ text: prompt }] },
                    config
                });

                const cleanText = (response.text || "{}").replace(/^```json\n?|```$/g, '').trim();
                const parsed = JSON.parse(cleanText);
                return parsed.title;

            } catch (e: any) {
                lastError = e;
                console.warn(`[RegenerateTitle] Error with Key ${maskedKey}: ${e.message}`);

                // If Rate Limit (429) or Overloaded (503), continue to NEXT KEY
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue; // Next key
                }

                continue;
            }
        }
    }

    console.error("[RegenerateTitle] All keys exhausted or failed.");
    return currentTitle; // Fail safe return original title
}


// --- Imagen 4.0 Image Generation ---

interface ImageGenOptions {
    aspectRatio: string;
    numberOfImages: number;
    negativePrompt?: string;
    modelId?: string;
}

export const generateImage = async (
    apiKeys: string | string[], // Updated to accept array
    prompt: string,
    options: ImageGenOptions = { aspectRatio: "1:1", numberOfImages: 1 }
): Promise<string[]> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for image generation.");

    // Final prompt construction
    const finalPrompt = options.negativePrompt
        ? `${prompt} --negative_prompt: ${options.negativePrompt}`
        : prompt;

    // Determine models to try
    let modelsToTry: string[] = [];
    if (options.modelId) {
        modelsToTry = [options.modelId];
    } else {
        modelsToTry = getPreferredModels('creative');
    }

    let lastError;

    // --- NESTED LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[ImageGen] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            // console.log(`[ImageGen] Trying with Key: ${maskedKey} on Model: ${model}`);

            try {
                const ai = getAI(apiKey);

                // Handle Imagen (generateImages) vs Gemini Flash Image (generateContent)
                if (model.startsWith('imagen')) {
                    const response = await ai.models.generateImages({
                        model: model,
                        prompt: finalPrompt,
                        config: {
                            numberOfImages: options.numberOfImages,
                            outputMimeType: 'image/jpeg',
                            aspectRatio: options.aspectRatio,
                        },
                    });
                    return response.generatedImages.map(img =>
                        `data:image/jpeg;base64,${img.image.imageBytes}`
                    );
                } else {
                    // Fallback to Gemini Flash Image using generateContent
                    const response = await ai.models.generateContent({
                        model: model,
                        contents: { parts: [{ text: finalPrompt }] },
                        config: {
                            responseModalities: [Modality.IMAGE],
                        }
                    });

                    const images: string[] = [];
                    if (response.candidates?.[0]?.content?.parts) {
                        for (const part of response.candidates[0].content.parts) {
                            if (part.inlineData) {
                                images.push(`data:image/png;base64,${part.inlineData.data}`);
                            }
                        }
                    }
                    if (images.length === 0) throw new Error("No image generated from Flash Image fallback");
                    return images;
                }
            } catch (e: any) {
                lastError = e;
                console.warn(`[ImageGen] Error with Key ${maskedKey}: ${e.message}`);

                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue;
                }
                continue;
            }
        }
    }

    throw lastError || new Error("All image generation models and keys failed.");
};

export const enhancePrompt = async (apiKeys: string | string[], draft: string): Promise<string> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for prompt enhancement.");

    const prompt = `Rewrite the following image generation prompt to be more detailed, descriptive, and optimized for high-quality AI image generation. Keep it under 300 words. Original: "${draft}"`;

    // Uses thinking model for text enhancement (preferred) or creative fallback
    const modelsToTry = getPreferredModels('thinking');

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[EnhancePrompt] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                const ai = getAI(apiKey);
                const config: any = {};
                // Thinking Config
                if (model.includes('thinking') || model === ModelType.PRO || model === ModelType.PRO_2_5) {
                    config.thinkingConfig = { thinkingBudget: 4096 };
                }

                const response = await ai.models.generateContent({
                    model: model,
                    contents: { parts: [{ text: prompt }] },
                    config
                });
                return response.text || draft;

            } catch (e: any) {
                lastError = e;
                console.warn(`[EnhancePrompt] Error with Key ${maskedKey}: ${e.message}`);
                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue;
                }
                continue;
            }
        }
    }

    console.error("[EnhancePrompt] All keys exhausted or failed.");
    return draft; // Fail safe return original
}

// --- Veo Video Generation ---

export const checkVeoKey = async (): Promise<boolean> => {
    const win = window as any;
    if (win.aistudio && win.aistudio.hasSelectedApiKey) {
        return await win.aistudio.hasSelectedApiKey();
    }
    return false;
};

export const promptVeoKey = async () => {
    const win = window as any;
    if (win.aistudio && win.aistudio.openSelectKey) {
        await win.aistudio.openSelectKey();
    }
};

interface VideoGenOptions {
    aspectRatio: '16:9' | '9:16';
    resolution: '720p' | '1080p';
    duration?: number; // Veo currently generates fixed durations but we pass intent
    modelId?: string;
}

export const generateVideo = async (
    apiKeys: string | string[],
    prompt: string,
    options: VideoGenOptions,
    imageBytes?: string
): Promise<string> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for video generation.");

    // Determine models
    const selectedModel = options.modelId || getPreferredModels('video')[0];
    // Video doesn't really have "fallback models" as easily as text/image unless we define them,
    // but we can try the same model with different keys.
    const modelsToTry = [selectedModel];
    // Could add fallbacks here if we had them, e.g. VEO_HQ -> VEO

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[VideoGen] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                // Initialize AI instance per key
                // Note: Veo uses GoogleGenAI class directly in this service impl
                const ai = new GoogleGenAI({ apiKey: apiKey });

                const config: any = {
                    numberOfVideos: 1,
                    resolution: options.resolution,
                    aspectRatio: options.aspectRatio
                };

                let operation;
                console.log(`[VideoGen] Starting generation with Key: ${maskedKey}`);

                if (imageBytes) {
                    operation = await ai.models.generateVideos({
                        model: model,
                        prompt: prompt || "Animate this image naturally.",
                        image: {
                            imageBytes: imageBytes,
                            mimeType: 'image/png',
                        },
                        config
                    });
                } else {
                    operation = await ai.models.generateVideos({
                        model: model,
                        prompt: prompt,
                        config
                    });
                }

                // If we get here, the operation started successfully.
                // Now we poll. We MUST use the SAME key for polling as for creation.
                console.log(`[VideoGen] Operation started. Polling...`);

                // Polling loop
                while (!operation.done) {
                    await new Promise(resolve => setTimeout(resolve, 5000));
                    try {
                        operation = await ai.operations.getVideosOperation({ operation });
                    } catch (pollErr: any) {
                        // If polling fails (e.g. network), we might want to retry polling a few times?
                        // But if 429 on poll, backoff?
                        console.warn(`[VideoGen] Polling error (non-fatal?): ${pollErr.message}`);
                        // If it's a 4xx error (not found/perm), it might be fatal.
                        if (pollErr.status === 404 || pollErr.status === 403) throw pollErr;
                    }
                }

                const downloadLink = operation.response?.generatedVideos?.[0]?.video?.uri;
                if (!downloadLink) throw new Error("No video URI returned.");

                // Fetch final video blob
                const videoRes = await fetch(`${downloadLink}&key=${apiKey}`);
                const blob = await videoRes.blob();
                return URL.createObjectURL(blob);

            } catch (e: any) {
                lastError = e;
                console.warn(`[VideoGen] Error with Key ${maskedKey}: ${e.message}`);
                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue;
                }
                // If operation failed mid-polling, we might iterate to next key (restart generation).
                // This is expensive but correct for robustness.
                continue;
            }
        }
    }

    console.error("[VideoGen] All keys exhausted or failed.");
    throw lastError || new Error("All video generation keys and models failed.");
};


// --- Image Editing (Nano Banana / Flash Image) ---

export const editImage = async (
    apiKeys: string | string[],
    base64Image: string,
    prompt: string,
    strength: number = 0.5,
    modelId?: string
): Promise<string> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for image editing.");

    // Determine models
    let modelsToTry: string[] = [];
    if (modelId) {
        modelsToTry = [modelId];
    } else {
        modelsToTry = getPreferredModels('creative');
    }

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[EditImage] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                const ai = getAI(apiKey);

                // For Flash Image, we guide intensity via prompt if the model API doesn't explicitly support 'strength' param in this version
                const guidePrompt = strength > 0.7 ? `Heavily modified: ${prompt}` : prompt;

                // Config specific to the model if needed (usually default for creative is fine)
                const config: any = {
                    responseModalities: [Modality.IMAGE]
                };

                const response = await ai.models.generateContent({
                    model: model,
                    contents: {
                        parts: [
                            {
                                inlineData: {
                                    data: base64Image,
                                    mimeType: 'image/png',
                                }
                            },
                            { text: guidePrompt }
                        ]
                    },
                    config
                });

                for (const part of response.candidates?.[0]?.content?.parts || []) {
                    if (part.inlineData) {
                        setPreferredModel('creative', model);
                        return `data:image/png;base64,${part.inlineData.data}`;
                    }
                }
                throw new Error("No image generated from edit request.");

            } catch (e: any) {
                lastError = e;
                console.warn(`[EditImage] Error with Key ${maskedKey}: ${e.message}`);
                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue;
                }
                continue;
            }
        }
    }

    console.error("[EditImage] All keys exhausted or failed.");
    throw lastError || new Error("All image editing models and keys failed.");
}

// --- Dashboard Analytics (Simulation using System Key) ---

export interface DashboardTrendData {
    chartData: any[];
    insights: {
        topPattern: string;
        bestKeyword: string;
        bestBiKeyword: string;
        keywordMix: string;
        bestTime: string;
        topCategory: string;
        emergingColor: string;
        deadNiche: string;
    };
}

export const generateDashboardTrends = async (apiKey: string, mode: 'global' | 'regional' = 'global', region?: string): Promise<DashboardTrendData> => {
    // Use the provided API key to simulate fetching/generating trend data
    if (!apiKey) {
        return {
            chartData: [],
            insights: {
                topPattern: '-', bestKeyword: '-', bestBiKeyword: '-', keywordMix: '-',
                bestTime: '-', topCategory: '-', emergingColor: '#ccc', deadNiche: '-'
            }
        };
    }

    // Simulating API latency for the "generation" process
    await new Promise(resolve => setTimeout(resolve, 800));

    let chartData: any[] = [];
    let insights: any = {};

    if (mode === 'regional' && region) {
        const r = region.toLowerCase();
        // Simulate specific regional nuances
        if (r.includes('germany') || r.includes('de')) {
            chartData = [
                { name: 'Industrial Tech', growth: 45, data: [20, 30, 45, 50, 60, 75, 85] },
                { name: 'Sustainable Energy', growth: 32, data: [25, 35, 40, 45, 55, 65, 80] },
                { name: 'Automotive Detail', growth: 18, data: [40, 42, 45, 48, 50, 55, 60] },
                { name: 'Minimalist Office', growth: 12, data: [10, 15, 20, 22, 25, 30, 35] },
            ];
            insights = {
                topPattern: 'Industry + Clean Backdrop',
                bestKeyword: 'Präzision (Precision)',
                bestBiKeyword: 'Renewable Energy',
                keywordMix: 'Solar, Wind, Engineer, Future',
                bestTime: 'Tue 10:00 AM',
                topCategory: 'Business/Tech',
                emergingColor: '#F59E0B', // Amber
                deadNiche: 'Generic Handshakes'
            };
        } else if (r.includes('japan') || r.includes('jp')) {
            chartData = [
                { name: 'Minimalist Zen', growth: 55, data: [30, 40, 50, 60, 70, 80, 90] },
                { name: 'Urban Nightlife', growth: 25, data: [15, 25, 35, 45, 55, 60, 65] },
                { name: 'Seasonal Nature', growth: 40, data: [10, 20, 40, 60, 50, 70, 85] },
                { name: 'Tech Lifestyle', growth: 20, data: [20, 22, 25, 30, 35, 40, 45] },
            ];
            insights = {
                topPattern: 'Subject + Negative Space',
                bestKeyword: 'Harmony',
                bestBiKeyword: 'Cherry Blossom',
                keywordMix: 'Zen, Wood, Tea, Silence',
                bestTime: 'Fri 6:00 PM',
                topCategory: 'Lifestyle',
                emergingColor: '#EC4899', // Pink
                deadNiche: 'Crowded Tourist Spots'
            };
        } else if (r.includes('brazil') || r.includes('br')) {
            chartData = [
                { name: 'Vibrant Outdoors', growth: 60, data: [20, 35, 50, 60, 75, 80, 90] },
                { name: 'Diverse Business', growth: 35, data: [10, 20, 30, 40, 50, 60, 70] },
                { name: 'Fitness & Health', growth: 45, data: [15, 25, 35, 45, 55, 65, 75] },
                { name: 'Carnival Abstract', growth: 20, data: [5, 10, 15, 20, 30, 40, 50] },
            ];
            insights = {
                topPattern: 'Action + Emotion + Sun',
                bestKeyword: 'Joy',
                bestBiKeyword: 'Outdoor Fitness',
                keywordMix: 'Nature, Sports, Fun, Green',
                bestTime: 'Sat 9:00 AM',
                topCategory: 'People/Leisure',
                emergingColor: '#10B981', // Green
                deadNiche: 'Winter Landscapes'
            };
        } else {
            // Generic Regional
            chartData = [
                { name: `Local Culture (${region})`, growth: 42, data: [15, 30, 45, 50, 65, 70, 85] },
                { name: 'Regional Food', growth: 28, data: [20, 25, 30, 40, 50, 60, 70] },
                { name: 'Local Business', growth: 15, data: [30, 35, 38, 40, 42, 45, 50] },
                { name: 'Architecture', growth: 10, data: [10, 12, 15, 18, 20, 22, 25] },
            ];
            insights = {
                topPattern: 'Local Subject + Action',
                bestKeyword: `${region} Lifestyle`,
                bestBiKeyword: 'Local Travel',
                keywordMix: 'Culture, Food, People, City',
                bestTime: 'Mon 8:00 AM',
                topCategory: 'Travel/Local',
                emergingColor: '#3B82F6', // Blue
                deadNiche: 'Generic International Flags'
            };
        }
    } else {
        // Global Defaults
        chartData = [
            { name: 'Authentic Diversity', growth: 15, data: [20, 35, 45, 50, 60, 70, 80] },
            { name: 'AI & Human Collaboration', growth: 28, data: [10, 20, 30, 50, 70, 90, 100] },
            { name: 'Sustainable Living', growth: 10, data: [50, 52, 54, 56, 58, 60, 65] },
            { name: 'Remote Work Lifestyle', growth: 8, data: [60, 62, 65, 68, 70, 72, 75] },
            { name: 'Mental Health Awareness', growth: 12, data: [30, 35, 40, 45, 50, 55, 60] },
        ];
        insights = {
            topPattern: 'Subject + Action + Context',
            bestKeyword: 'Authenticity',
            bestBiKeyword: 'Remote Work',
            keywordMix: 'Eco, Green, Solar, Future',
            bestTime: 'Wed 2:00 PM (UTC)',
            topCategory: 'Business',
            emergingColor: '#8B5CF6', // Purple
            deadNiche: 'Isolated Objects White Bg'
        };
    }

    return { chartData, insights };
};

// --- Trend Analysis (User Key Required) ---

export const generateTrendAnalysis = async (
    apiKeys: string | string[],
    query: string,
    platforms: string[]
): Promise<{ report: TrendReport; modelUsed: string }> => {
    // Normalize apiKeys to array
    const keys = Array.isArray(apiKeys) ? apiKeys : [apiKeys];
    // Filter out empties and clean
    const cleanKeys = keys.map(k => k.replace(/[^\x20-\x7E]/g, '').trim()).filter(k => k.length > 0);

    console.log(`[GeminiService] Initializing with ${cleanKeys.length} keys`);

    if (cleanKeys.length === 0) throw new Error("No valid API keys provided");

    const today = new Date().toLocaleDateString();

    // Determine models to try (Thinking models preferred)
    const modelsToTry = getPreferredModels('thinking');

    // Define Prompt
    const prompt = `
    ACT AS A SENIOR MARKET ANALYST AND STRATEGIST FOR THE MICROSTOCK PHOTOGRAPHY INDUSTRY.
    
    CORE OBJECTIVE:
    Conduct a DEEP DIVE market analysis for the trend/niche: "${query}".
    Current Date: ${today}.
    Target Platforms: ${platforms.join(', ')}.

    INSTRUCTIONS:
    1.  **Analyze the specific keywords** provided in the query. Do not just give generic advice.
    2.  **Platform-Specific Insights:** For EACH selected platform (${platforms.join(', ')}), provide specific nuances. (e.g., "Adobe Stock buyers prefer authentic lifestyle," "Shutterstock moves high volume of isolated objects").
    3.  **Simulate Advanced Statistics:** Since you don't have real-time API access to these stock sites, use your vast training data to SIMULATE realistic, statistically probable data for "Saturation," "Competition," and "Pricing" based on the nature of this niche.
    4.  **Content Depth:** The user wants "longer, better, more content." Do not be brief. Be exhaustive.
    5.  **Creative Direction:** Provide highly specific visual descriptions, lighting setups, and casting advice.

    REQUIRED JSON OUTPUT FORMAT (Strict JSON, no markdown):
    {
      "id": "generated-${Date.now()}",
      "query": "${query}",
      "timestamp": ${Date.now()},
      "selectedPlatforms": ${JSON.stringify(platforms)},
      "insights": {
        "create": [
            {"title": "Specific Concept 1", "reason": "Detailed explanation of why this specific concept sells on ${platforms[0] || 'stock sites'}."},
            {"title": "Specific Concept 2", "reason": "Detailed market reasoning..."},
            {"title": "Specific Concept 3", "reason": "Detailed market reasoning..."}
        ], 
        "caution": [
            {"title": "Oversaturated Cliché", "reason": "Explain exactly what visual trope is dead and why."}
        ], 
        "avoid": [
            {"title": "Technical/Legal Pitfall", "reason": "Specific rejection reasons relevant to this niche."}
        ], 
        "takeaways": [
            "Comprehensive strategic takeaway 1...",
            "Comprehensive strategic takeaway 2...",
            "Comprehensive strategic takeaway 3...",
            "Comprehensive strategic takeaway 4..."
        ],
        "confidenceScore": 85
      },
      "saturation": {
        "level": "Low|Medium|High",
        "score": 5, 
        "competitionLevel": 5, 
        "totalVolume": 10000, 
        "supplyData": [
            {"name": "Month 1", "supply": 10, "demand": 20}, 
            {"name": "Month 2", "supply": 15, "demand": 25}, 
            {"name": "Month 3", "supply": 20, "demand": 18}, 
            {"name": "Month 4", "supply": 25, "demand": 30}, 
            {"name": "Month 5", "supply": 30, "demand": 45}, 
            {"name": "Month 6", "supply": 35, "demand": 40}
        ] 
      },
      "trend": {
        "direction": "Rising|Stable|Declining",
        "data": [
            {"name": "W1", "val": 10}, {"name": "W2", "val": 20}, {"name": "W3", "val": 15}, 
            {"name": "W4", "val": 30}, {"name": "W5", "val": 25}, {"name": "W6", "val": 40}
        ], 
        "seasonality": "Detailed explanation of seasonal factors affecting this niche.",
        "subTrends": ["Specific Sub-trend 1", "Specific Sub-trend 2", "Specific Sub-trend 3"],
        "forecast": "Long-term prediction for this niche over the next 12 months."
      },
      "creative": {
        "ideas": [
            "Highly descriptive visual idea 1...", 
            "Highly descriptive visual idea 2...",
            "Highly descriptive visual idea 3..."
        ],
        "colorPalette": ["#Hex1", "#Hex2", "#Hex3", "#Hex4", "#Hex5"],
        "prompts": [
            "Detailed AI generation prompt 1...", 
            "Detailed AI generation prompt 2..."
        ],
        "moodBoardUrls": []
      },
      "pricing": {
        "avgPrice": "$0.00 - $0.00",
        "potentialMonthly": "$000 - $000",
        "roi": "High|Medium|Low",
        "licensing": "Detailed breakdown of Commercial vs Editorial potential."
      },
      "community": {
        "quotes": [{"text": "Relevant forum/buyer sentiment quote", "author": "Stock Buyer/Contributor"}],
        "sentiment": "Positive|Neutral|Negative"
      },
      "sources": [{"title": "Market Analysis Source", "url": "#", "date": "${today.split('/')[2]}"}]
    }
  `;

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[generateTrendAnalysis] Switching to Model: ${model}`);

        for (const apiKey of cleanKeys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            console.log(`[generateTrendAnalysis] Trying with Key: ${maskedKey} on Model: ${model}`);

            try {
                // Initialize AI instance per key
                const ai = getAI(apiKey);

                const config: any = {
                    responseMimeType: 'application/json'
                };

                // Thinking Config
                if (model.includes('thinking') || model.includes('flash') || model.includes('pro')) {
                    // Conservatively apply budget if supported, or if 2.5/3.0 flash
                    config.thinkingConfig = { thinkingBudget: 4096 };
                }

                const response = await ai.models.generateContent({
                    model: model,
                    contents: { parts: [{ text: prompt }] },
                    config
                });

                const text = response.text || "{}";
                // Cleanup markdown
                const cleanText = text.replace(/^```json\n?|```$/g, '').trim();
                let parsed;
                try {
                    parsed = JSON.parse(cleanText);
                } catch (e) {
                    // Fallback grab json
                    const match = cleanText.match(/\{[\s\S]*\}/);
                    if (match) parsed = JSON.parse(match[0]);
                    else throw new Error("Invalid JSON response");
                }

                parsed.timestamp = Date.now();

                // SUCCESS
                setPreferredModel('thinking', model);
                return { report: parsed, modelUsed: model };

            } catch (e: any) {
                lastError = e;
                console.warn(`[generateTrendAnalysis] Error with Key ${maskedKey} on Model ${model}:`, e.message);

                // If Rate Limit (429) or Overloaded (503), continue to NEXT KEY
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue; // Next key
                }

                // For other errors, also continue to be safe?
                continue;
            }
        }
        console.warn(`[generateTrendAnalysis] All keys failed for model ${model}. Switching model...`);
    }

    console.error("[generateTrendAnalysis] FATAL: All keys and models exhausted.");
    throw lastError || new Error("All API keys and models exhausted.");
};

// --- Image Reviewer: Two-Step Verification ---

// Default system prompts (can be overridden by admin config)
const DEFAULT_ANATOMY_PROMPT = `CRITICAL TASK: Analyze this stock photo for ANATOMICAL ERRORS and AI ARTIFACTS.
This is a STRICT "PASS/FAIL" check. A score of 0 means REJECTION.

SCOPE:
[✅ CHECK] Human Faces, Eyes, Expressions
[✅ CHECK] General AI Artifacts, Glitches, Bugs, Errors, Floating objects
[❌ IGNORE] Fingers, Hands, Arms (Handled by a separate check)
[❌ IGNORE] Legs, Feet (Handled by a separate check)

STEP 1: 🔴 INITIAL SCAN FOR HUMANS 🔴
Are there humans in the image?
- If NO (landscapes, objects, abstract) → Score: 100. STOP.
- If YES → Continue to Step 2.

═══════════════════════════════════════════════════════

STEP 2: 🔴 CHECK FACE FOR ARTIFACTS 🔴
Analyze the face(s) closely.

Check for:
- Distorted/warped features → score: 0
- AI artifacts/glitches on skin/face → score: 0
- Asymmetry (severely lopsided features) → score: 0
- Duplicated features (double noses, etc) → score: 0
- Missing features → score: 0
- Unnatural merging of features → score: 0
- "Melting" textures on the face → score: 0

If face is clean, GO TO STEP 3.

═══════════════════════════════════════════════════════

STEP 3: 🔴 CHECK EYES FOR DEFECTS 🔴
Analyze the eyes strictly.

Check for:
- Pupil size mismatch (unless lighting explains it) → score: 0
- Wrong eye count (not 2 per face) → score: 0
- Blurred/corrupted eyes or irises → score: 0
- Drastically different shapes → score: 0
- Missing pupils/iris (white eyes) → score: 0
- Squinting that looks like a glitch → score: 0

If eyes are clean, GO TO STEP 4.

═══════════════════════════════════════════════════════

STEP 4: 🔴 GENERAL ARTIFACTS SCAN 🔴
Look for non-anatomical AI errors.

Check for:
- Floating objects / phantom pixels around the head/body.
- Text/Watermark gibberish.
- Impossible physics or objects merging into the person.

If ANY issue found → Score: 0.

If ALL STEPS PASS → Score: 100.

OUTPUT MUST BE VALID JSON (no extra text):
{
    "score": 0 or 100,
    "hasHumans": true or false,
    "hands": [], 
    "defects": ["list SPECIFIC defects found"],
    "analysis": "Brief summary of face/eye/artifact check."
}

🔴 CRITICAL: Do NOT reject based on hands/legs. REJECT IMMEDIATELY for any Face/Eye distortion.`;

const STRICT_FACE_INSTRUCTIONS = `
═══════════════════════════════════════════════════════
🔴 STRICT "UNCANNY VALLEY" & HORROR CHECK ENABLED 🔴
═══════════════════════════════════════════════════════
You must apply EXTREME SCRUTINY to facial features.
REJECT if the face has any "Zombie", "Horror", "Melted", or "Uncanny Valley" traits.

ADDITIONAL STRICT RULES:
- ASYMMETRY: One eye lower/larger than other → REJECT.
- EXPRESSION: Dead/Soulless stare or "Zombie" look → REJECT.
- TEXTURE: Plastic/Wax skin or "Melting" patches → REJECT.
- DENTIST: Too many teeth, merged teeth → REJECT.

If the face looks even SLIGHTLY unnatural or creepy → Score: 0.
`;

const DEFAULT_QUALITY_PROMPT = `CRITICAL TASK: TECHNICAL QUALITY ASSESSMENT for stock photography standards (Adobe Stock, Shutterstock, iStock).
This is a STRICT "ZERO TOLERANCE" QUALITY CONTROL CHECK.
You must be EXTREMELY CRITICAL. Do not give the benefit of the doubt.

IMPORTANT: Analyze this image for TECHNICAL QUALITY ONLY. Do NOT consider anatomy or content - only technical aspects.

EVALUATION CRITERIA (Score each 0-10, then multiply):

1. NOISE / GRAIN (×10 points):
- 10: Pristine. No noise even at 100% zoom.
- 8-9: Almost perfect, negligible grain.
- 6-7: Very slight grain, acceptable for standard stock.
- 4-5: VISIBLE NOISE or grain at normal size. (REJECT)
- 2-3: Heavy noise.
- 0-1: Unusable.

2. SHARPNESS / FOCUS (×10 points):
- 10: Razor sharp details.
- 8-9: Excellent sharpness.
- 6-7: Good sharpness.
- 4-5: SOFT or slightly out of focus. (REJECT)
- 2-3: Blurry.
- 0-1: Unusable.

3. EXPOSURE (×10 points):
- 10: Perfect balance.
- 8-9: Excellent dynamic range.
- 6-7: Minor clipping but acceptable.
- 4-5: Noticeably over/under exposed. (REJECT)
- 0-3: Severe exposure issues.

4. COLOR ACCURACY (×10 points):
- 10: Perfect, natural.
- 8-9: Excellent.
- 6-7: Slight cast but natural.
- 4-5: Unnatural saturation or bad, strong cast. (REJECT)
- 0-3: Broken colors.

5. COMPOSITION (×10 points):
- 10: Award-winning.
- 8-9: Strong leading lines, rule of thirds.
- 6-7: Balanced, standard stock.
- 4-5: Unbalanced, awkward crop. (REJECT)
- 0-3: Bad.

6. LIGHTING QUALITY (×10 points):
- 10: Professional lighting, perfect highlights and shadows
- 8-9: Very attractive.
- 6-7: Standard, flat but clean.
- 4-5: Harsh shadows, amatuer lighting. (REJECT)
- 0-3: Bad.

7. TECHNICAL DEFECTS (×10 points):
- 10: Zero defects.
- 8-9: Zero visible defects.
- 6-7: Negligible compression artifacts.
- 4-5: Visible JPEG artifacts or banding. (REJECT)
- 0-3: Corrupted.

SCORING CALCULATION:
1. Evaluate each criterion.
2. If ANY criterion falls into the "REJECT" (4-5 or lower) category, the final average MUST reflect this.
3. Average all 7 scores.
4. Multiply by 10 to get final score (0-100).

CONSISTENCY RULES:
- ZERO TOLERANCE: If you see a flaw, PUNISH the score.
- Do NOT be generous.
- 60/100 is the PASS threshold. If an image has VISIBLE flaws, it MUST score BELOW 60.
- BE CONSISTENT: The same image must always receive the same score.

OUTPUT JSON ONLY:
{
    "score": 0-100,
    "issues": ["list of specific technical issues found"],
    "reasoning": "detailed breakdown of each criterion score"
}
`;

export const analyzeImageAnatomyCheck = async (
    apiKeys: string | string[],
    base64Image: string,
    mimeType: string,
    modelId?: string,
    customPrompt?: string,
    enableStrictFaceCheck?: boolean
): Promise<import('../types').AnatomyCheckResult> => {
    let prompt = customPrompt || DEFAULT_ANATOMY_PROMPT;

    if (enableStrictFaceCheck && !customPrompt) {
        prompt += STRICT_FACE_INSTRUCTIONS;
    }

    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for Anatomy Check.");

    // Models logic
    const preferred = getPreferredModels('thinking');
    const modelsToTry = modelId
        ? [modelId, ...preferred.filter(m => m !== modelId)]
        : preferred;
    if (modelsToTry.length === 0) throw new Error("No enabled models available for Anatomy Check.");

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[AnatomyCheck] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                // Per-key/model attempt
                const ai = getAI(apiKey);

                const response = await ai.models.generateContent({
                    model: model,
                    contents: {
                        role: "user",
                        parts: [
                            { text: prompt },
                            { inlineData: { mimeType, data: base64Image } }
                        ]
                    },
                    config: {
                        responseMimeType: "application/json",
                        temperature: 0.0,
                        topP: 1.0,
                        topK: 1
                    }
                });

                const resultText = response.text || "";
                if (!resultText) throw new Error("No text result");

                const cleanedText = resultText.replace(/```json/g, '').replace(/```/g, '').trim();
                let parsed;
                try {
                    parsed = JSON.parse(cleanedText);
                } catch (e) {
                    // Fallback grab json
                    const match = cleanedText.match(/\{[\s\S]*\}/);
                    if (match) parsed = JSON.parse(match[0]);
                    else throw e;
                }

                // Safety check: ensure defects is an array
                if (!parsed.defects || !Array.isArray(parsed.defects)) {
                    parsed.defects = [];
                }

                // CODE-LEVEL VERIFICATION: Check hands array if present
                if (parsed.hands && Array.isArray(parsed.hands)) {
                    parsed.hands.forEach((hand: any) => {
                        if (hand.fingerCount > 5) {
                            parsed.score = 0;
                            parsed.defects.push(`Code Verification: Detected ${hand.fingerCount} fingers on ${hand.side} hand`);
                        }
                        if (hand.fingerCount < 5 && hand.isVisible) {
                            parsed.score = 0;
                            parsed.defects.push(`Code Verification: Detected only ${hand.fingerCount} fingers on visible ${hand.side} hand`);
                        }
                    });
                }

                return parsed;

            } catch (e: any) {
                lastError = e;
                console.warn(`[AnatomyCheck] Error with Key ${maskedKey}: ${e.message}`);
                // If 429, continue to next key immediately
                if (e.message?.includes('429') || e.status === 429) continue;
                // Loop continues for other errors too in robust mode
                continue;
            }
        }
        console.warn(`[AnatomyCheck] All keys failed for model ${model}. Switching...`);
    }

    console.error("[AnatomyCheck] All keys exhausted or failed.");
    throw lastError || new Error("Anatomy Check failed: All keys/models exhausted.");
};

const ANIME_STYLE_QUALITY_INSTRUCTIONS = `
═══════════════════════════════════════════════════════
🎨 ANIME & CARTOON STYLE MODE ENABLED 🎨
═══════════════════════════════════════════════════════
This image is INTENDED to be stylized (Anime, Cartoon, 3D Render, Illustration).
DO NOT PENALIZE FOR:
- "Plastic" or smooth skin textures (this is correct for the style).
- Lack of photorealistic skin pores or grain.
- Stylized lighting or vibrant colors.
- "Soft" appearing edges if they fit the art style (e.g. painterly look).

JUDGE QUALITY BASED ON:
- Clean lines and freedom from artifacts.
- Composition and visual impact.
- Absence of unintentional AI glitches (merging, distortion) rather than lack of realism.
`;

export const analyzeImageQualityCheck = async (
    apiKeys: string | string[],
    base64Image: string,
    mimeType: string,
    modelId?: string,
    customPrompt?: string,
    enableAnimeStyle?: boolean
): Promise<import('../types').QualityCheckResult> => {
    let prompt = customPrompt || DEFAULT_QUALITY_PROMPT;

    if (enableAnimeStyle && !customPrompt) {
        prompt += ANIME_STYLE_QUALITY_INSTRUCTIONS;
    }

    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for Quality Check.");

    const preferred = getPreferredModels('thinking');
    const modelsToTry = modelId
        ? [modelId, ...preferred.filter(m => m !== modelId)]
        : preferred;
    if (modelsToTry.length === 0) throw new Error("No enabled models available for Quality Check.");

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[QualityCheck] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                const ai = getAI(apiKey);
                const response = await ai.models.generateContent({
                    model: model,
                    contents: {
                        role: "user",
                        parts: [
                            { text: prompt },
                            { inlineData: { mimeType, data: base64Image } }
                        ]
                    },
                    config: {
                        responseMimeType: "application/json",
                        temperature: 0.0,
                        topP: 1.0,
                        topK: 1
                    }
                });

                const resultText = response.text || "";
                if (!resultText) throw new Error("No text result");

                const cleanedText = resultText.replace(/```json/g, '').replace(/```/g, '').trim();
                let parsed;
                try {
                    parsed = JSON.parse(cleanedText);
                } catch (e) {
                    // Fallback grab json
                    const match = cleanedText.match(/\{[\s\S]*\}/);
                    if (match) parsed = JSON.parse(match[0]);
                    else throw e;
                }

                if (!parsed.issues || !Array.isArray(parsed.issues)) {
                    parsed.issues = [];
                }
                return parsed;

            } catch (e: any) {
                lastError = e;
                console.warn(`[QualityCheck] Error with Key ${maskedKey}: ${e.message}`);
                // If 429, continue to next key immediately
                if (e.message?.includes('429') || e.status === 429) continue;
                // Loop continues for other errors too
                continue;
            }
        }
        console.warn(`[QualityCheck] All keys failed for model ${model}. Switching...`);
    }

    console.error("[QualityCheck] All keys exhausted or failed.");
    throw lastError || new Error("Quality Check failed: All keys/models exhausted.");
};

// --- Hand Detection & Fix (Step 1 & 2) ---

const HAND_DETECTION_PROMPT = `
You are an expert anatomical analyzer. Your SOLE purpose is to detect human hands or arms in an image.

INSTRUCTIONS:
1. Scan the image for any presence of human hands, fingers, or arms.
2. Ignore other body parts (faces, legs) - focus ONLY on hands/arms.
3. Be strict: Even a small partial hand counts as "detected".

OUTPUT JSON:
{
    "hasHands": true/false,
    "confidence": 0-100,
    "location": "description of where hands are found",
    "details": "brief description of what was found"
}
`;

export const detectHands = async (
    apiKeys: string | string[],
    base64Image: string,
    mimeType: string,
    modelId?: string
): Promise<{ hasHands: boolean; confidence: number; details: string }> => {

    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) return { hasHands: false, confidence: 0, details: "No API keys" };

    const preferred = getPreferredModels('thinking');
    const modelsToTry = modelId
        ? [modelId, ...preferred.filter(m => m !== modelId)]
        : preferred;
    if (modelsToTry.length === 0) modelsToTry.push(ModelType.FLASH);

    for (const model of modelsToTry) {
        console.log(`[HandDetect] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                const ai = getAI(apiKey);
                const response = await ai.models.generateContent({
                    model: model,
                    contents: {
                        parts: [
                            { text: HAND_DETECTION_PROMPT },
                            { inlineData: { mimeType, data: base64Image } }
                        ]
                    },
                    config: {
                        responseMimeType: "application/json",
                        temperature: 0.0,
                        topP: 1.0,
                    }
                });

                const text = response.text || "{}";
                const cleanText = text.replace(/^```json\n?|```$/g, '').trim();
                let result;
                try {
                    result = JSON.parse(cleanText);
                } catch (e) {
                    const match = cleanText.match(/\{[\s\S]*\}/);
                    if (match) result = JSON.parse(match[0]);
                    else throw e;
                }

                return {
                    hasHands: result.hasHands === true,
                    confidence: result.confidence || 0,
                    details: result.details || "No details provided"
                };
            } catch (e: any) {
                console.warn(`[HandDetect] Failed model ${model} key ${maskedKey}:`, e.message);
                if (e.message?.includes('429') || e.status === 429) continue;
                continue;
            }
        }
    }

    // If all fail, return negative result instead of throwing (optional feature)
    console.warn("[HandDetect] Hand detection exhausted all keys/models");
    return { hasHands: false, confidence: 0, details: "Detection exhausted" };
};

export const fixImageHands = async (
    apiKeys: string | string[],
    base64Image: string,
    mimeType: string,
    modelId?: string
): Promise<string | null> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) return null;

    const preferred = getPreferredModels('creative'); // Fix hands uses creative models
    const modelsToTry = modelId
        ? [modelId, ...preferred.filter(m => m !== modelId)]
        : preferred;
    if (modelsToTry.length === 0) modelsToTry.push('imagen-3.0-generate-001');

    const FIX_PROMPT = `
    TASK: Detect and REGENERATE human hands/arms to be photorealistic and anatomically perfect.
    1. Identify any human hands or arms in the image.
    2. REGENERATE them completely. Do NOT simply remove them or blur them.
    3. The new hands must have correct finger counts (5 fingers), natural skin texture, and lighting that matches the scene.
    4. If the hands are distorted, replace them with natural-looking hands in a similar pose.
    5. Keep the rest of the image (faces, background, clothes) EXACTLY as is.
    Output: The modified image with fixed hands.
    `;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[HandFix] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                console.log(`🛠️ Attempting hand fix with model: ${model} (Key: ${maskedKey})`);
                const ai = getAI(apiKey);
                const response = await ai.models.generateContent({
                    model: model,
                    contents: {
                        parts: [
                            { text: FIX_PROMPT },
                            { inlineData: { mimeType, data: base64Image } }
                        ]
                    }
                });

                const generatedPart = (response as any).response?.candidates?.[0]?.content?.parts?.[0];
                if (generatedPart && 'inlineData' in generatedPart) {
                    setPreferredModel('creative', model);
                    return generatedPart.inlineData?.data || null;
                }
            } catch (e: any) {
                console.warn(`[HandFix] Failed with model ${model}/key ${maskedKey}:`, e.message);
                if (e.message?.includes('429') || e.status === 429) continue;
                continue;
            }
        }
    }

    // Hand fix is optional/best effort, so we don't throw exhaust error, just return null.
    console.warn("[HandFix] Hand fix failed on all attempts.");
    return null;
};

export const performTwoStepReview = async (
    apiKeys: string | string[],
    base64Image: string,
    mimeType: string,
    model?: string,
    customPrompts?: { anatomy?: string; quality?: string },
    options?: { enableAnatomyCheck?: boolean; enableQualityCheck?: boolean; enableHandFix?: boolean; enableStrictFaceCheck?: boolean; enableAnimeStyle?: boolean }
): Promise<import('../types').TwoStepReviewResult> => {
    // Use provided model or get preferred thinking model
    const selectedModel = model || getPreferredModels('thinking')[0] || 'gemini-2.0-flash';
    const keys = Array.isArray(apiKeys) ? apiKeys : [apiKeys];

    // Get custom prompts from localStorage if available
    const anatomyPrompt = customPrompts?.anatomy || localStorage.getItem('sf_reviewer_prompt_anatomy_v3') || undefined;
    const qualityPrompt = customPrompts?.quality || localStorage.getItem('sf_reviewer_prompt_quality_v4') || undefined;

    const enableAnatomyCheck = options?.enableAnatomyCheck !== false; // Default true
    const enableQualityCheck = options?.enableQualityCheck !== false; // Default true
    const enableHandFix = options?.enableHandFix === true; // Default false (opt-in)
    const enableStrictFaceCheck = options?.enableStrictFaceCheck === true; // Default false
    const enableAnimeStyle = options?.enableAnimeStyle === true; // Default false

    console.log('🔍 Review Options:', {
        enableAnatomyCheck,
        enableQualityCheck,
        enableHandFix,
        enableStrictFaceCheck,
        enableAnimeStyle,
        model: selectedModel
    });

    let anatomyResult;
    let scoreA = 100; // Default pass if disabled

    // ═══════════════════════════════════════════════════════
    // STEP 1: ANATOMY CHECK (Face, Eyes, Artifacts)
    // ═══════════════════════════════════════════════════════
    if (enableAnatomyCheck) {
        console.log('✅ [1/3] Running Anatomy Check...');
        anatomyResult = await analyzeImageAnatomyCheck(
            keys,
            base64Image,
            mimeType,
            selectedModel,
            anatomyPrompt,
            enableStrictFaceCheck
        );
        scoreA = anatomyResult.score;
        console.log('Anatomy Result:', anatomyResult);

        // Kill Switch Logic
        if (scoreA === 0) {
            return {
                verdict: 'REJECTED',
                totalScore: 0,
                scoreA,
                scoreB: 0,
                killSwitchActivated: true,
                anatomyAnalysis: anatomyResult,
                qualityAnalysis: {
                    score: 0,
                    issues: ['Skipped due to anatomy check failure'],
                    reasoning: 'Quality check skipped - Image failed anatomy validation'
                }
            };
        }

        // 5-second delay to prevent rate limiting (simple delay, maybe reduce if rotating?)
        // With rotation we might not need this as much, but safer to keep for now.
        console.log('⏳ Waiting 2s to prevent rate limit...');
        await new Promise(resolve => setTimeout(resolve, 2000));
    } else {
        anatomyResult = {
            score: 100,
            hasHumans: false,
            defects: [],
            analysis: 'Anatomy check disabled by user'
        };
    }

    // ═══════════════════════════════════════════════════════
    // STEP 2: QUALITY CHECK (Technical: Noise, Blur, Lighting)
    // ═══════════════════════════════════════════════════════
    let qualityResult;
    let scoreB = 100; // Default pass if disabled

    if (enableQualityCheck) {
        console.log('✅ [2/3] Running Quality Check...');
        qualityResult = await analyzeImageQualityCheck(
            keys,
            base64Image,
            mimeType,
            selectedModel,
            qualityPrompt,
            enableAnimeStyle // Pass the flag
        );
        scoreB = qualityResult.score;

        // 2-second delay
        console.log('⏳ Waiting 2s to prevent rate limit...');
        await new Promise(resolve => setTimeout(resolve, 2000));
    } else {
        qualityResult = {
            score: 100,
            issues: [],
            reasoning: 'Quality check disabled by user'
        };
    }


    // ═══════════════════════════════════════════════════════
    // STEP 3: HAND DETECTION & FIX (Final Check)
    // ═══════════════════════════════════════════════════════
    let handDetectionResult;

    // Only run if enabled AND previous checks passed
    if (enableHandFix) {
        console.log('✅ [3/3] Running Hand Detection...');
        handDetectionResult = await detectHands(
            keys,
            base64Image,
            mimeType,
            selectedModel
        );

        console.log('Hand Detection Result:', handDetectionResult);

        // If hands detected -> REJECT and signal for fix
        if (handDetectionResult.hasHands) {
            return {
                verdict: 'REJECTED',
                totalScore: 0,
                scoreA,
                scoreB,
                killSwitchActivated: true, // Treat as kill switch to stop further processing
                anatomyAnalysis: anatomyResult,
                qualityAnalysis: qualityResult,
                handDetection: handDetectionResult, // Signal UI to trigger Step 2 (Fix)
                reason: "Humans Hands detected - Triggering Auto-Fix"
            };
        }
    }

    const totalScore = scoreA + scoreB;

    // Final Verdict: Total Score >= 160 = ACCEPTED (Anatomy 100 + Quality 60 minimum)
    const verdict = totalScore >= 160 ? 'ACCEPTED' : 'REJECTED';

    return {
        verdict,
        totalScore,
        scoreA,
        scoreB,
        killSwitchActivated: false,
        anatomyAnalysis: anatomyResult,
        qualityAnalysis: qualityResult,
        handDetection: handDetectionResult
    };
};

export const generateStockPrompts = async (
    apiKeys: string | string[],
    topic: string,
    style: string,
    keywords: string[] = [],
    count: number = 5,
    modelId?: string
): Promise<string[]> => {
    // Normalize keys
    const rawKeys = apiKeys ? (Array.isArray(apiKeys) ? apiKeys : [apiKeys]) : [process.env.API_KEY || ''];
    const keys = rawKeys.filter(k => k && k.trim().length > 0);

    if (keys.length === 0) throw new Error("No API keys provided for prompt generation.");

    const prompt = `
    Generate ${count} high-quality, commercially viable stock photography prompts for the topic: "${topic}".
    
    Style: ${style}
    Keywords to include: ${keywords.join(', ')}
    
    Requirements:
    1. Optimized for AI generation (Flux, Midjourney, Imagen).
    2. Detailed lighting, composition, and subject description.
    3. Diverse selection (different angles, lighting setups).
    4. Commercial value focused (clean backgrounds, copy space).
    
    Output Format:
    Return ONLY a JSON array of strings. Example: ["Prompt 1...", "Prompt 2..."]
    `;

    // Determine models to try
    let modelsToTry: string[] = [];
    if (modelId) {
        modelsToTry = [modelId];
    } else {
        modelsToTry = getPreferredModels('thinking');
    }

    let lastError;

    // --- ROBUST LOOP: Models -> Keys ---
    for (const model of modelsToTry) {
        console.log(`[StockPrompts] Switching to Model: ${model}`);

        for (const apiKey of keys) {
            const maskedKey = apiKey.substring(0, 8) + '...';
            try {
                const ai = getAI(apiKey);
                const config: any = {
                    responseMimeType: 'application/json',
                    responseSchema: {
                        type: Type.ARRAY,
                        items: { type: Type.STRING }
                    }
                };

                // Thinking Config
                if (model.includes('thinking') || model === ModelType.PRO || model === ModelType.PRO_2_5) {
                    config.thinkingConfig = { thinkingBudget: 4096 };
                }

                const response = await ai.models.generateContent({
                    model: model,
                    contents: { parts: [{ text: prompt }] },
                    config
                });

                const text = response.text || "[]";
                try {
                    const parsed = JSON.parse(text);
                    setPreferredModel('thinking', model);
                    return parsed;
                } catch (e) {
                    // Fallback grab json if schema failed strict return (unlikely with schema mode but safe)
                    const match = text.match(/\[([\s\S]*)\]/);
                    if (match) return JSON.parse(match[0]);
                    throw e;
                }
            } catch (e: any) {
                lastError = e;
                console.warn(`[StockPrompts] Error with Key ${maskedKey}: ${e.message}`);
                // Failover logic
                if (e.message?.includes('429') || e.status === 429 || e.message?.includes('Too Many Requests') || e.message?.includes('503')) {
                    continue;
                }
                continue;
            }
        }
    }

    console.error("[StockPrompts] All keys exhausted or failed.");
    throw lastError || new Error("All prompt generation models and keys failed.");
};

const THINKING_MODELS = [
    { id: ModelType.FLASH, name: 'Gemini 2.5 Flash (Balanced)' },
    { id: ModelType.PRO, name: 'Gemini 3 Pro (Quality)' },
    { id: ModelType.PRO_2_5, name: 'Gemini 2.5 Pro (Advanced)' },
];
