import { authService } from './authService';

// Backend URL
const API_URL = 'http://localhost:3001/api';

export interface Asset {
  id: string;
  type: 'image' | 'video' | 'metadata' | 'report' | 'review' | 'prompt';
  url: string;
  prompt: string;
  createdAt: number;
  isFavorite: boolean;
  isDeleted: boolean;
  metadata?: any; // For metadata results
  aspectRatio?: string;
  userId?: string;
  batchId?: string; // For batch operations
  modelUsed?: string; // Track which model generated this (e.g., for metadata history)
  generationSettings?: { // Track generation settings snapshot
    enableGrounding?: boolean;
    descLength?: number;
    keywordCount?: number;
    titleLength?: number;
    exportFormat?: 'txt' | 'json' | 'csv'; // Track chosen export format
    selectedStockSites?: string[]; // Track targeted microstock platforms
  };
  title?: string;
}

export interface TrashConfig {
  enabled: boolean;
  frequency: 'minute' | 'hourly' | 'daily' | 'weekly' | 'monthly';
}

const getUid = (): string | null => {
  const user = authService.getCurrentUser();
  return user ? user.uid : null;
};

export const dbService = {
  // --- Query Methods ---

  getAll: async (targetUid?: string): Promise<Asset[]> => {
    const uid = targetUid || getUid();
    if (!uid) return [];
    try {
      const res = await fetch(`${API_URL}/assets/list?uid=${uid}`, { headers: authService.getHeaders() });
      const data = await res.json();
      return data.assets || [];
    } catch (e) { return []; }
  },

  getLibrary: async (targetUid?: string, options?: { type?: string; limit?: number; excludeUrl?: boolean }): Promise<Asset[]> => {
    const uid = targetUid || getUid();
    if (!uid) return [];
    try {
      let url = `${API_URL}/assets/list?uid=${uid}&isDeleted=false`;
      if (options?.type) url += `&type=${options.type}`;
      if (options?.limit) url += `&limit=${options.limit}`;
      if (options?.excludeUrl) url += `&excludeUrl=${options.excludeUrl}`;

      const res = await fetch(url, { headers: authService.getHeaders() });
      const data = await res.json();
      return data.assets || [];
    } catch (e) { return []; }
  },

  getFavorites: async (targetUid?: string): Promise<Asset[]> => {
    const uid = targetUid || getUid();
    if (!uid) return [];
    try {
      // Fetch specific endpoint or filter? Server has general list.
      // We can fetch all and filter or add isFavorite query param to server.
      // Server list support: uid, type, isDeleted. 
      // We'll fetch library and filter client side for now to minimize server changes
      const res = await fetch(`${API_URL}/assets/list?uid=${uid}&isDeleted=false`, { headers: authService.getHeaders() });
      const data = await res.json();
      return (data.assets || []).filter((a: Asset) => a.isFavorite);
    } catch (e) { return []; }
  },

  getTrash: async (targetUid?: string): Promise<Asset[]> => {
    const uid = targetUid || getUid();
    if (!uid) return [];
    try {
      const res = await fetch(`${API_URL}/assets/list?uid=${uid}&isDeleted=true`, { headers: authService.getHeaders() });
      const data = await res.json();
      return data.assets || [];
    } catch (e) { return []; }
  },

  getByType: async (type: Asset['type'], targetUid?: string): Promise<Asset[]> => {
    const uid = targetUid || getUid();
    if (!uid) return [];
    try {
      const res = await fetch(`${API_URL}/assets/list?uid=${uid}&type=${type}&isDeleted=false`, { headers: authService.getHeaders() });
      const data = await res.json();
      return data.assets || [];
    } catch (e) { return []; }
  },

  // --- Mutation Methods ---

  uploadFile: async (file: File): Promise<string | null> => {
    const formData = new FormData();
    formData.append('file', file);

    try {
      const res = await fetch(`${API_URL}/upload`, {
        method: 'POST',
        body: formData,
        // No Content-Type header needed; fetch sets it for FormData
      });
      if (!res.ok) throw new Error('Upload failed');
      const data = await res.json();
      return data.url;
    } catch (e) {
      console.error("Upload error", e);
      return null;
    }
  },

  add: async (asset: Asset) => {
    const uid = getUid();
    if (!uid) return;
    try {
      const res = await fetch(`${API_URL}/assets/create`, {
        method: 'POST',
        headers: authService.getHeaders(),
        body: JSON.stringify({ asset, uid })
      });
      if (!res.ok) throw new Error('Create failed');
    } catch (e) { console.error("Add error", e); }
  },

  update: async (id: string, updates: Partial<Asset>) => {
    const uid = getUid();
    if (!uid) return;
    try {
      await fetch(`${API_URL}/assets/update`, {
        method: 'POST',
        headers: authService.getHeaders(),
        body: JSON.stringify({ uid, assetId: id, updates })
      });
    } catch (e) { console.error("Update error", e); }
  },

  toggleFavorite: async (id: string, currentStatus?: boolean) => {
    // Need current status if not provided
    // In MySQL version, just pass the inverted status if known, or fail if unknown?
    // UI usually knows.
    if (currentStatus === undefined) return; // Can't toggle blindly yet
    await dbService.update(id, { isFavorite: !currentStatus });
  },

  moveToTrash: async (id: string) => {
    await dbService.update(id, { isDeleted: true });
  },

  restore: async (id: string) => {
    await dbService.update(id, { isDeleted: false });
  },

  permanentDelete: async (id: string) => {
    const uid = getUid();
    if (!uid) return;
    try {
      await fetch(`${API_URL}/assets/delete`, {
        method: 'POST',
        headers: authService.getHeaders(),
        body: JSON.stringify({ uid, assetId: id })
      });
    } catch (e) { console.error("Delete error", e); }
  },

  emptyTrash: async () => {
    const trash = await dbService.getTrash();
    for (const item of trash) {
      await dbService.permanentDelete(item.id);
    }
  },

  clearByType: async (type: Asset['type'], force: boolean = false) => {
    // Fetch all of type and delete
    const assets = await dbService.getByType(type);
    await Promise.all(assets.map(async (item) => {
      if (force || !item.isFavorite) {
        await dbService.moveToTrash(item.id);
      }
    }));
  },

  // --- Config ---
  getTrashConfig: async (): Promise<TrashConfig> => {
    // Uses preferences endpoint
    const uid = getUid();
    if (!uid) return { enabled: true, frequency: 'hourly' };
    try {
      const res = await fetch(`${API_URL}/user/preferences?uid=${uid}&section=general`, { headers: authService.getHeaders() });
      const json = await res.json();
      return json.data.trashConfig || { enabled: true, frequency: 'hourly' };
    } catch (e) { return { enabled: true, frequency: 'hourly' }; }
  },

  setTrashConfig: async (config: TrashConfig) => {
    const uid = getUid();
    if (!uid) return;
    try {
      await fetch(`${API_URL}/user/preferences`, {
        method: 'POST',
        headers: authService.getHeaders(),
        body: JSON.stringify({ uid, section: 'general', data: { trashConfig: config } })
      });

      // Reset timer when frequency changes - always start fresh countdown
      const lastClearKey = 'trash_last_clear';
      localStorage.setItem(lastClearKey, Date.now().toString());
    } catch (e) { }
  },

  // Helpers
  checkAutoClearTrash: async () => {
    const config = await dbService.getTrashConfig();
    if (!config.enabled) return false;

    const lastClearKey = 'trash_last_clear';
    const lastClearStr = localStorage.getItem(lastClearKey);
    const lastClear = lastClearStr ? parseInt(lastClearStr) : 0;

    const nextTime = await dbService.getNextTrashClearTime();
    const now = Date.now();

    if (now >= nextTime) {
      // Time to clear!
      await dbService.emptyTrash();
      localStorage.setItem(lastClearKey, now.toString());
      return true;
    }
    return false;
  },

  getNextTrashClearTime: async (configOverride?: TrashConfig) => {
    const config = configOverride || await dbService.getTrashConfig();
    if (!config.enabled) return Date.now() + 1000 * 60 * 60; // 1 hour default

    const lastClearKey = 'trash_last_clear';
    const lastClearStr = localStorage.getItem(lastClearKey);
    const lastClear = lastClearStr ? parseInt(lastClearStr) : Date.now();

    // Calculate next clear time based on frequency
    const intervals = {
      'minute': 60 * 1000,
      'hourly': 60 * 60 * 1000,
      'daily': 24 * 60 * 60 * 1000,
      'weekly': 7 * 24 * 60 * 60 * 1000,
      'monthly': 30 * 24 * 60 * 60 * 1000,
    };

    const interval = intervals[config.frequency] || intervals.hourly;
    return lastClear + interval;
  },

  getGlobalStats: async () => {
    // Should use api/admin/stats but that is admin only.
    // Need public stats endpoint or use defaults.
    return { users: 0, assets: 0 };
  },

  subscribeToGlobalStats: (cb: any) => {
    // MySQL doesn't support realtime. Just fetch once.
    dbService.getGlobalStats().then(cb);
    return () => { };
  }
};