import { User, SystemEmail, ActivityLog, SiteSettings, ApiKeyData, ModelType } from "../types";
import { SETTINGS_TO_MODEL_TYPE } from "../constants";
import { geoService } from './geoService';

// Backend URL - change if deployed
const API_URL = '/api';

// Helper to get date string YYYY-MM-DD (LOCAL TIME)
const getDateStr = (offsetDays: number = 0) => {
    const date = new Date();
    date.setDate(date.getDate() + offsetDays);
    return date.getFullYear() + '-' +
        String(date.getMonth() + 1).padStart(2, '0') + '-' +
        String(date.getDate()).padStart(2, '0');
};

export const authService = {
    // --- Auth State Management ---
    getToken: (): string | null => {
        return localStorage.getItem('sf_token');
    },

    setToken: (token: string) => {
        localStorage.setItem('sf_token', token);
    },

    clearToken: () => {
        localStorage.removeItem('sf_token');
        localStorage.removeItem('sf_user'); // Cache clear
    },

    getHeaders: () => {
        const token = authService.getToken();
        return {
            'Content-Type': 'application/json',
            ...(token ? { 'Authorization': `Bearer ${token}` } : {})
        };
    },

    // --- Observer Mimic ---
    // In a real React app, you'd use a Context Provider to load user on mount.
    // Here we provide a subscriber for components.
    onAuthStateChanged: (callback: (user: User | null) => void) => {
        // Check immediately
        const cached = localStorage.getItem('sf_user');
        if (cached) {
            try {
                callback(JSON.parse(cached));
            } catch (e) { callback(null); }
        }

        // Always revalidate with server to get latest stats
        if (authService.getToken()) {
            authService.getUserProfile('me', true).then(user => {
                if (user) callback(user);
            }).catch(() => {
                // If fetch fails and no cache, clear state
                if (!cached) callback(null);
            });
        } else if (!cached) {
            callback(null);
        }

        // Listen for internal events
        const listener = (event: CustomEvent) => {
            callback(event.detail); // detail is user or null
        };
        window.addEventListener('sf_user_updated', listener as EventListener);

        // Return unsubscribe
        return () => window.removeEventListener('sf_user_updated', listener as EventListener);
    },

    // --- DAILY RESET LOGIC ---
    checkDailyReset: async (uid: string) => {
        try {
            const response = await fetch(`${API_URL}/user/check-daily-reset`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid })
            });

            if (response.ok) {
                const { updated, stats } = await response.json();

                // Always sync stats to ensure we have the latest structure (e.g. daily/monthly objects)
                // even if no date-based reset occurred.
                if (stats) {
                    const user = authService.getCurrentUser();
                    if (user) {
                        // Create fresh reference for React
                        const updatedUser = JSON.parse(JSON.stringify(user));
                        updatedUser.usageStats = stats;
                        localStorage.setItem('sf_user', JSON.stringify(updatedUser));
                        window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));
                    }
                }

                if (updated) {
                    console.log("Daily/Monthly stats reset successfully.");
                }
            }
        } catch (e) {
            console.error("Failed to perform daily reset check", e);
        }
    },

    getCurrentUser: (): User | null => {
        const cached = localStorage.getItem('sf_user');
        return cached ? JSON.parse(cached) : null;
    },

    getUserProfile: async (uid: string, silent = false): Promise<User | null> => {
        try {
            const endpoint = uid === 'me' ? '/auth/me' : `/user/profile?uid=${uid}`;
            const response = await fetch(`${API_URL}${endpoint}`, {
                headers: authService.getHeaders()
            });

            if (!response.ok) return null;
            const data = await response.json();

            // Cache if it's me
            if (uid === 'me' || (data.user && data.user.uid === authService.getCurrentUser()?.uid)) {
                localStorage.setItem('sf_user', JSON.stringify(data.user));
                // Only dispatch update if NOT called from onAuthStateChanged to prevent infinite loop
                if (!silent) {
                    window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: data.user }));
                }
            }
            return data.user as User;
        } catch (error) {
            console.error("Get Profile Error:", error);
            return null;
        }
    },

    // --- USER DATA HELPERS ---

    getUserApiKeys: async (uid: string): Promise<ApiKeyData[]> => {
        try {
            const response = await fetch(`${API_URL}/user/keys?uid=${uid}`, {
                headers: authService.getHeaders(),
                cache: 'no-store' // Disable caching to always get fresh usage data
            });
            if (!response.ok) throw new Error('Failed to fetch keys');
            const data = await response.json();

            if (data.keys) {
                let keys = data.keys.map((k: any) => ({
                    id: k.id,
                    key: k.key ? k.key.trim() : '', // Masked for display
                    fullKey: k.fullKey ? k.fullKey.trim() : '', // Real key for API usage
                    status: 'unchecked', // Default validation state
                    tier: k.tier || 'FREE',
                    isEnabled: k.isEnabled,
                    addedAt: Date.now(), // Fallback if no specific date
                    usageCount: 0,
                    validationAttempts: 0,
                    usage: k.usage // Map usage from API
                }));

                // Random Rotation Logic - Shuffle keys if enabled
                // This ensures load balancing across multiple keys for every request
                if (localStorage.getItem('sf_random_key_rotation') === 'true') {
                    for (let i = keys.length - 1; i > 0; i--) {
                        const j = Math.floor(Math.random() * (i + 1));
                        [keys[i], keys[j]] = [keys[j], keys[i]];
                    }
                    console.log('[AuthService] API Keys Shuffled due to enabled rotation.');
                }

                return keys;
            }
            return [];
        } catch (error) {
            console.error('Fetch keys error:', error);
            return [];
        }
    },

    saveUserApiKeys: async (uid: string, keys: ApiKeyData[]) => {
        const response = await fetch(`${API_URL}/user/keys`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ uid, keys })
        });
        if (!response.ok) throw new Error('Failed to save keys');

        const data = await response.json();

        // Update user stats with returned model usage and limits
        const currentUser = authService.getCurrentUser();
        if (currentUser && data.modelUsage && data.modelLimits) {
            currentUser.usageStats.modelUsage = data.modelUsage;
            currentUser.usageStats.modelLimits = data.modelLimits;
            currentUser.usageStats.apiCallsLimit = data.newLimit;
            currentUser.usageStats.apiCallsThisMonth = data.newUsage;

            // Update localStorage and dispatch event
            localStorage.setItem('sf_user', JSON.stringify(currentUser));
            window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: currentUser }));
            console.log('[saveUserApiKeys] Updated user stats from API response:', data.modelUsage);
        }
    },

    // --- RECALCULATE LIMITS ---
    recalculateUserLimits: async (uid: string) => {
        // Triggered by saveUserApiKeys, so we just refresh profile (silent to avoid double dispatch)
        await authService.getUserProfile(uid, true);
    },

    deductCredits: async (uid: string, amount: number = 1, apiKey?: string, modelId?: string) => {
        console.log(`[AuthService] deductCredits: uid=${uid}, amount=${amount}, modelId="${modelId}"`);
        try {
            const response = await fetch(`${API_URL}/user/deduct-credits`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid, amount, apiKey, modelId })
            });
            console.log('[AuthService] deductCredits server response:', response.status);

            // Optimistic Local Update
            const user = authService.getCurrentUser();
            if (user) {
                if (!user.usageStats) user.usageStats = { apiCallsThisMonth: 0, daily: {}, monthly: {} } as any;

                // Update Legacy
                user.usageStats.apiCallsThisMonth = (user.usageStats.apiCallsThisMonth || 0) + amount;

                // Update New Structure (if exists, or init)
                if (!user.usageStats.daily) user.usageStats.daily = { date: getDateStr() };
                user.usageStats.daily.apiCalls = (user.usageStats.daily.apiCalls || 0) + amount; // For dashboard

                if (!user.usageStats.monthly) user.usageStats.monthly = { month: new Date().toISOString().slice(0, 7) };
                user.usageStats.monthly.apiCalls = (user.usageStats.monthly.apiCalls || 0) + amount; // For dashboard

                // Update Model Usage
                if (modelId) {
                    if (!user.usageStats.modelUsage) user.usageStats.modelUsage = {};
                    user.usageStats.modelUsage[modelId] = (user.usageStats.modelUsage[modelId] || 0) + amount;
                    console.log(`[AuthService] Local stats updated. ${modelId} usage: ${user.usageStats.modelUsage[modelId]}`);
                }

                // Create fresh reference for React
                const updatedUser = JSON.parse(JSON.stringify(user));
                localStorage.setItem('sf_user', JSON.stringify(updatedUser));
                window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));
            }
        } catch (e) { console.error("Deduct credits failed", e); }
    },

    getSiteSettings: async (): Promise<SiteSettings> => {
        try {
            const response = await fetch(`${API_URL}/settings`);
            if (response.ok) {
                const json = await response.json();
                if (json.data && json.data.length > 0) {
                    // The row has 'settings' column which is JSON string
                    // OR maybe it has individual columns?
                    // In server.js: const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 1');
                    // system_settings schema likely has a 'settings' text column or columns for each?
                    // Based on AdminDashboard usage: it seems to expect an object structure. 
                    // Let's assume the 'settings' column holds the JSON.
                    const row = json.data[0];
                    const parsed = typeof row.settings === 'string' ? JSON.parse(row.settings) : row.settings;
                    return parsed || {};
                }
            }
        } catch (e) {
            console.error("Failed to fetch settings", e);
        }

        // Default fallback if fetch fails
        return {
            creditsPerApiKey: 2640,
            modelLimits: {},
            maintenanceMode: false,
            emailVerificationRequired: false,
            allowRegistrations: true,
            modelAccessTiers: {}
        } as SiteSettings;
    },

    canUserAccessModel: (modelId: string, userKeys: ApiKeyData[], settings: SiteSettings): boolean => {
        if (!settings || !settings.modelAccessTiers || !userKeys || userKeys.length === 0) return true; // Default to visible if unknown

        const config = settings.modelAccessTiers[modelId];
        if (!config) return true; // No restriction defined

        const hasFreeKey = userKeys.some(k => k.tier !== 'PAID' && (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        const hasPaidKey = userKeys.some(k => k.tier === 'PAID' && (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);

        const freeAllowed = config.free !== false;
        const paidAllowed = config.paid !== false;

        // User can see model if:
        // 1. They have a Free key AND Free tier is allowed
        // 2. OR They have a Paid key AND Paid tier is allowed
        return (hasFreeKey && freeAllowed) || (hasPaidKey && paidAllowed);
    },

    // --- Activity Logging ---
    logActivity: async (uid: string, type: 'generate' | 'system' | 'setting' | 'error', text: string, cost?: number, metadata: any = {}) => {
        try {
            await fetch(`${API_URL}/user/activity`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid, type, description: text, cost: cost || 0, metadata })
            });
        } catch (e) { console.error("Failed to log activity", e); }
    },

    getActivities: async (uid: string, days: number = 30): Promise<ActivityLog[]> => {
        try {
            const response = await fetch(`${API_URL}/user/activity?uid=${uid}&limit=20`, {
                headers: authService.getHeaders()
            });
            if (response.ok) {
                const data = await response.json();
                return data.activities || [];
            }
            return [];
        } catch (e) {
            console.error("Failed to get activities", e);
            return [];
        }
    },

    incrementUserStat: async (stat: keyof User['usageStats'], amount: number = 1, modelId?: string) => {
        console.log(`[authService] incrementUserStat called: stat=${stat}, amount=${amount}, modelId=${modelId}`);

        const user = authService.getCurrentUser();
        if (!user) {
            console.error('[authService] No current user - aborting increment');
            return;
        }

        console.log(`[authService] Current user: ${user.uid}`);

        // Optimistic Update (Use any to bypass strict type checks for partial updates)
        const safeStats: any = user.usageStats || {};

        // Only increment scalar stats (not objects like modelUsage)
        if (stat !== 'modelUsage' && stat !== 'modelLimits') {
            const val = amount;
            safeStats[stat] = (safeStats[stat] || 0) + val;

            // Optimistically update daily/monthly for dashboard responsiveness
            if (!safeStats.daily) safeStats.daily = { date: getDateStr() };
            safeStats.daily[stat] = (safeStats.daily[stat] || 0) + val;

            if (!safeStats.monthly) safeStats.monthly = { month: new Date().toISOString().slice(0, 7) };
            safeStats.monthly[stat] = (safeStats.monthly[stat] || 0) + val;

            console.log(`[authService] Optimistic update: ${stat} = ${safeStats.daily[stat]}`);
        }

        if (modelId) {
            if (!safeStats.modelUsage) safeStats.modelUsage = {};
            safeStats.modelUsage[modelId] = (safeStats.modelUsage[modelId] || 0) + amount;
        }
        user.usageStats = safeStats;

        // Create fresh reference for React to detect change
        const updatedUser = JSON.parse(JSON.stringify(user));
        localStorage.setItem('sf_user', JSON.stringify(updatedUser));
        window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));

        console.log(`[authService] Making HTTP request to ${API_URL}/user/stats/increment`);
        console.log(`[authService] Request payload:`, { uid: user.uid, statName: stat, amount, modelId });

        try {
            const response = await fetch(`${API_URL}/user/stats/increment`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid: user.uid, statName: stat, amount, modelId })
            });

            console.log(`[authService] Response status: ${response.status}`);

            if (response.ok) {
                const data = await response.json();
                console.log(`[authService] ✅ INCREMENT SUCCESS - Server confirmed`);
                // Sync with server truth - create fresh reference
                const syncedUser = JSON.parse(JSON.stringify(user));
                syncedUser.usageStats = data.stats;
                localStorage.setItem('sf_user', JSON.stringify(syncedUser));
                window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: syncedUser }));
            } else {
                const errorText = await response.text();
                console.error(`[authService] ❌ INCREMENT FAILED: ${response.status}`, errorText);
            }
        } catch (e) {
            console.error(`[authService] ❌ INCREMENT EXCEPTION:`, e);
            // Ideally revert optimistic update here if critical
        }
    },

    getDailyTrend: async (stat: string): Promise<number> => 0,
    getDashboardStats: async (uid: string, days: number = 30): Promise<Record<string, any>> => {
        try {
            console.log('[authService] Fetching dashboard stats for uid:', uid);
            const response = await fetch(`${API_URL}/user/stats/dashboard?uid=${uid}&days=${days}`, {
                headers: authService.getHeaders()
            });
            if (!response.ok) {
                console.log('[authService] Dashboard stats fetch failed:', response.status);
                return {};
            }
            // New backend format: already returns date-keyed object
            const statsMap = await response.json();
            console.log('[authService] Dashboard stats received:', Object.keys(statsMap).length, 'days');
            return statsMap;
        } catch (e) {
            console.error('[authService] Dashboard stats error:', e);
            return {};
        }
    },
    getDailyHistory: async (stat: string, days: number = 7): Promise<number[]> => [],

    // --- Login / Register ---
    login: async (email: string, password: string): Promise<User> => {
        const response = await fetch(`${API_URL}/auth/login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email, password })
        });

        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || 'Login failed');
        }

        const data = await response.json();
        authService.setToken(data.token);
        localStorage.setItem('sf_user', JSON.stringify(data.user)); // Cache

        window.dispatchEvent(new CustomEvent('sf_auth_state_change', { detail: data.user }));
        return data.user;
    },

    register: async (name: string, email: string, password: string, isSocial = false, photoURL: string | null = null, country: string = 'Unknown'): Promise<User> => {
        const geoData = await geoService.getIpAndCountry();
        const detectedCountry = country !== 'Unknown' ? country : geoData.country;

        const response = await fetch(`${API_URL}/auth/register`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                displayName: name, email, password, photoURL, country: detectedCountry
            })
        });

        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || 'Registration failed');
        }

        const data = await response.json();
        authService.setToken(data.token);
        localStorage.setItem('sf_user', JSON.stringify(data.user));

        window.dispatchEvent(new CustomEvent('sf_auth_state_change', { detail: data.user }));
        return data.user;
    },

    // Social Login Simulations/Handlers
    handleGoogleCredential: async (credential: string): Promise<User> => { throw new Error("Not implemented in MySQL version"); },
    loginWithGoogleSimulation: async (): Promise<User> => { return authService.register('Google User', `google_${Date.now()}@example.com`, 'social_pass', true, null); },
    loginWithTwitterSimulation: async (): Promise<User> => { return authService.register('Twitter User', `twitter_${Date.now()}@example.com`, 'social_pass', true, null); },
    loginWithFacebookSimulation: async (): Promise<User> => { return authService.register('Facebook User', `fb_${Date.now()}@example.com`, 'social_pass', true, null); },

    updateProfile: async (uid: string, data: { displayName?: string; photoURL?: string | null; country?: string }): Promise<User> => {
        const response = await fetch(`${API_URL}/user/update`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ uid, data })
        });
        if (!response.ok) throw new Error('Update failed');
        const resData = await response.json();

        localStorage.setItem('sf_user', JSON.stringify(resData.user));
        window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: resData.user }));
        return resData.user;
    },

    updateUserLocation: async (uid: string) => {
        // Optional implementation
    },

    changePassword: async (uid: string, currentPass: string, newPass: string): Promise<void> => {
        const response = await fetch(`${API_URL}/user/change-password`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ uid, currentPassword: currentPass, newPassword: newPass })
        });
        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || 'Password change failed');
        }
    },

    verifyEmail: async (code: string): Promise<boolean> => {
        return true;
    },

    resendCode: async () => { },

    deleteAccount: async (uid: string): Promise<void> => {
        const response = await fetch(`${API_URL}/user/delete`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ uid })
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || 'Account deletion failed');
        }

        await authService.logout();
    },

    uploadFile: async (file: File): Promise<string> => {
        const formData = new FormData();
        formData.append('file', file);

        const response = await fetch(`${API_URL}/upload`, {
            method: 'POST',
            // headers: authService.getHeaders(), // Multer handles content-type, but auth might fail if server expects bearer? 
            // The server.js middleware setup: `app.use('/uploads', express.static...)`
            // `app.post('/api/upload', upload.single('file')...)` 
            // It does NOT have `requireAuth` on `/api/upload` in server.js! (Line 626).
            // So we don't strictly need headers, but if we did, we'd need to NOT set Content-Type so browser sets boundary.
            body: formData
        });

        if (!response.ok) throw new Error('Upload failed');
        const data = await response.json();
        return data.url;
    },

    // Get User Preferences
    getUserPreferences: async (uid: string) => {
        try {
            const response = await fetch(`${API_URL}/user/preferences?uid=${uid}`, {
                headers: authService.getHeaders()
            });
            if (!response.ok) throw new Error('Failed to fetch preferences');
            const { data } = await response.json();
            return data || {};
        } catch (error) {
            console.error("Get Preferences Error:", error);
            // Fallback to empty if fails
            return {};
        }
    },

    // Update User Preferences
    updateUserPreferences: async (uid: string, preferences: any) => {
        try {
            const response = await fetch(`${API_URL}/user/preferences`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid, data: preferences })
            });

            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.error || 'Failed to update preferences');
            }

            const result = await response.json();

            // Update local cache if successful
            const currentUser = authService.getCurrentUser();
            if (currentUser && currentUser.uid === uid) {
                const updatedUser = {
                    ...currentUser,
                    preferences: { ...(currentUser as any).preferences, ...preferences }
                };
                localStorage.setItem('sf_user', JSON.stringify(updatedUser));
                window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));
            }

            return result;
        } catch (error) {
            console.error("Update Preferences Error:", error);
            throw error;
        }
    },

    resetUserStats: async (uid: string) => {
        try {
            const response = await fetch(`${API_URL}/user/reset-stats`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ uid })
            });

            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.error || 'Failed to reset stats');
            }

            const result = await response.json();

            // Update local cache
            if (result.user) {
                localStorage.setItem('sf_user', JSON.stringify(result.user));
                window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: result.user }));
            }
            return result;
        } catch (error) {
            console.error("Reset Stats Error:", error);
            throw error;
        }
    },

    logout: async (): Promise<void> => {
        authService.clearToken();
        localStorage.removeItem('sf_thinking_models');
        localStorage.removeItem('sf_creative_models');
        localStorage.removeItem('sf_video_models');
        window.dispatchEvent(new CustomEvent('sf_auth_state_change', { detail: null }));
    }
};