import { User, SiteSettings, SystemEmail, Broadcast, SupportMessage, BugReport, AuditLog, Tip, TutorialStep, StockSite } from "../types";
import { authService } from "./authService";
import { UserDatabase } from "./USER-database";

const API_URL = '/api';

// Default Constants
const DEFAULT_TIPS: Tip[] = [
    // --- VIDEO ---
    { id: 'tip_v1', title: 'Mastering Veo Camera', text: 'Use terms like "pan left", "zoom in", or "orbit" to control the Veo director cam.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v2', title: 'Cinematic Frame Rates', text: 'Use 24fps for a film look, or 60fps for smooth sports motion.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v3', title: 'Veo Lighting', text: 'Specify "golden hour", "soft studio lighting", or "neon cyberpunk" for better video atmosphere.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v4', title: 'Drone Shots', text: 'Prompt "FPV drone shot" or "aerial view" for stunning flyover videos.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v5', title: 'Slow Motion', text: 'Add "slow motion" or "high speed camera" to capture detailed action shots.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v6', title: 'Time Lapse', text: 'Create time-lapse videos by prompting "time lapse of city traffic" or "clouds moving".', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v7', title: 'Macro Video', text: 'Use "macro lens" to get extreme close-ups of insects, eyes, or textures.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v8', title: 'Consistent Characters', text: 'Describe your character in detail (age, clothes, hair) to keep them consistent across shots.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v9', title: 'Camera Movement', text: 'Try "truck left" or "dolly forward" for dynamic, movie-like movement.', category: 'Video', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_v10', title: 'Video Resolution', text: 'Veo supports 1080p. Upscale externally for 4K stock footage delivery.', category: 'Video', createdAt: Date.now(), isDeleted: false },

    // --- IMAGE ---
    { id: 'tip_i1', title: 'Aspect Ratios', text: 'Use --ar 16:9 for desktop wallpapers or --ar 9:16 for social media stories.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i2', title: 'Photorealism', text: 'Add "shot on 35mm", "f/1.8", and "bokeh" for realistic photography results.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i3', title: 'Negative Prompts', text: 'Specify what you DON\'T want (e.g., "no blur", "no text", "no distortion") to clean up images.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i4', title: 'Lighting Styles', text: 'Experiment with "rembrandt lighting", "volumetric lighting", or "god rays".', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i5', title: 'Art Styles', text: 'Try "oil painting", "watercolor", "cyberpunk", or "ukiyo-e" for artistic variations.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i6', title: 'Texture Detail', text: 'Use "8k resolution", "highly detailed", and "unreal engine 5" for sharp textures.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i7', title: 'Commercial Stock', text: 'For stock photos, use "clean background", "isolated", and "commercial lighting".', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i8', title: 'Double Exposure', text: 'Prompt "double exposure of wolf and forest" for artistic blending effects.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i9', title: 'Isometric View', text: 'Great for 3D icons and game assets: "isometric view of a cute cafe".', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i10', title: 'Flat Lay', text: 'Use "knolling" or "flat lay photography" for organized product showcases.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i11', title: 'Minimalism', text: 'Use "minimalist", "negative space", and "pastel colors" for modern, clean looks.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i12', title: 'Portrait Lenses', text: 'Try "85mm lens" or "105mm lens" for flattering portraits with blurred backgrounds.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i13', title: 'Action Shots', text: 'Use "motion blur", "frozen in time", or "high shutter speed" for dynamic energy.', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i14', title: 'Food Photography', text: 'Keywords: "steam rising", "fresh ingredients", "overhead shot", "appetizing".', category: 'Image', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_i15', title: 'Abstract backgrounds', text: 'Prompt "abstract liquid 3d render", "glass distortion", or "geometric shapes".', category: 'Image', createdAt: Date.now(), isDeleted: false },

    // --- MARKET INTEL ---
    { id: 'tip_m1', title: 'Trend Spotting', text: 'Check the "Rising" trends in Market Intelligence to find low-competition niches.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m2', title: 'Seasonal Content', text: 'Start creating holiday content (Christmas, Halloween) 3 months in advance.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m3', title: 'Keyword Research', text: 'Use the Metadata tool to find "long-tail keywords" that buyers are actually searching for.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m4', title: 'Platform Specifics', text: 'Adobe Stock loves "lifestyle" and "business" content. Shutterstock prefers "news" and "vectors".', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m5', title: 'Underserved Niches', text: 'Look for high search volume but low result counts in the Analysis tab.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m6', title: 'Color Trends', text: 'Check the "Color Palette" section in Trend Reports to match current design aesthetics.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m7', title: 'Authenticity', text: 'Buyers want "authentic" and "candid" moments. Avoid overly staged or plastic looks.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m8', title: 'Diversity', text: 'Ensure your portfolio represents diverse ages, ethnicities, and abilities.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m9', title: 'Copy Space', text: 'Leave empty "copy space" in your compositions for designers to add text.', category: 'Market', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_m10', title: 'Tech Trends', text: 'Keep up with new tech (VR headsets, EVs, AI) and create relevant stock concepts.', category: 'Market', createdAt: Date.now(), isDeleted: false },

    // --- WORKFLOW ---
    { id: 'tip_w1', title: 'Batch Processing', text: 'Upload multiple images to the Metadata Generator to tag them all at once.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w2', title: 'Save Your Prompts', text: 'Star your favorite generations to save the prompt for later reuse.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w3', title: 'API Limits', text: 'Monitor your usage in Settings. Upgrade to Pro for higher daily limits.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w4', title: 'Export Formats', text: 'Download as PNG for highest quality, or JPG for smaller file sizes.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w5', title: 'Dark Mode', text: 'Toggle Dark Mode in the top bar to reduce eye strain during late-night sessions.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w6', title: 'Keyboard Shortcuts', text: 'Press "/" to quickly focus the search bar or prompt input.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w7', title: 'Mobile View', text: 'StockForge is fully responsive. Manage your portfolio from your phone.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w8', title: 'Support', text: 'Found a bug? Submit a report directly from the sidebar "Bug" icon.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w9', title: 'Image History', text: 'Your "Assets" library keeps your last 100 generations. Download them before they expire!', category: 'Workflow', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_w10', title: 'Reviewer AI', text: 'Run your images through Reviewer AI to check for technical flaws before submitting.', category: 'Workflow', createdAt: Date.now(), isDeleted: false },

    // --- METADATA & REVIEW ---
    { id: 'tip_r1', title: 'Title accuracy', text: 'Reviewer AI checks if your title matches the visual content. Keep it descriptive.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r2', title: 'Keyword Spamming', text: 'Avoid adding irrelevant keywords. Reviewer AI will flag this as "spamming".', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r3', title: 'Model Releases', text: 'Even AI people need "Property Release" marked on some platforms. Check local laws.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r4', title: 'Editorial vs Commercial', text: 'Use "Editorial" for news/brands, "Commercial" for generic creative stock.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r5', title: 'Noise Reduction', text: 'Reviewer AI detects high noise. Use an upscaler or "denoise" prompt to fix it.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r6', title: 'Artifacts', text: 'Check for "extra fingers" or "weird eyes". Reviewer AI often spots these common AI glitches.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r7', title: 'Composition Score', text: 'Reviewer AI rates composition. Aim for Rule of Thirds or Centered symmetry.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_r8', title: 'Commercial Value', text: 'The "Commercial Viability" score predicts how well an image might sell.', category: 'Metadata', createdAt: Date.now(), isDeleted: false },

    // --- APP FEATURES ---
    { id: 'tip_a1', title: 'Model Selection', text: 'Use "Imagen 3" for consistency, or "Flux" for artistic creativity.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a2', title: 'Settings', text: 'Customize your default aspect ratio and model in the Settings page.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a3', title: 'Broadcasts', text: 'Read system broadcasts for important updates and new feature announcements.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a4', title: 'Feedback', text: 'We love feedback! Use the Support chat to request new features.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a5', title: 'Usage Stats', text: 'Track your daily and monthly generation habits in the Dashboard Overview.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a6', title: 'Live Trends', text: 'Market trends are updated frequently. Refresh to see the latest search data.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a7', title: 'Security', text: 'Your API keys are encrypted. Only you can use them.', category: 'App', createdAt: Date.now(), isDeleted: false },
    { id: 'tip_a8', title: 'Favicon', text: 'Admins can custom branding images in the Admin Panel settings.', category: 'App', createdAt: Date.now(), isDeleted: false },
];

const DEFAULT_TUTORIAL_STEPS: TutorialStep[] = [
    { id: 'step_1', title: 'Complete Profile', description: 'Update your name and avatar.', actionLabel: 'Go to Profile', actionLink: 'SETTINGS:profile', order: 1, isDeleted: false },
    { id: 'step_2', title: 'Add API Key', description: 'Connect Gemini API to unlock features.', actionLabel: 'Add Key', actionLink: 'SETTINGS:api', order: 2, isDeleted: false },
    { id: 'step_3', title: 'Create Image', description: 'Generate your first AI image.', actionLabel: 'Create', actionLink: 'IMAGE_GENERATION', order: 3, isDeleted: false },
    { id: 'step_4', title: 'Make Video', description: 'Turn an image into a video with Veo.', actionLabel: 'Animate', actionLink: 'VIDEO_GENERATION', order: 4, isDeleted: false },
    { id: 'step_5', title: 'Analyze Trends', description: 'Check market trends for a keyword.', actionLabel: 'Explore', actionLink: 'CHAT_INTELLIGENCE', order: 5, isDeleted: false },
];

const DEFAULT_SETTINGS: SiteSettings = {
    appStaticApiKey: "AIzaSyBOI4M-nrYVvYgXgpfNZXRdWIOauwA8xzk",
    systemApiUsage: 0,
    modelIds: {},
    emailVerificationRequired: false,
    allowRegistrations: true,
    adminApprovalRequired: false,
    defaultApiLimit: 1000,
    creditsPerApiKey: 2620,
    modelLimits: {
        flash_lite_2_5: 990, flash_2_5: 240,
        pro_2_5: 50, pro_3_0: 50, gemini_2_0_flash_image: 90, gemini_2_5_flash_image: 490,
        gemini_3_pro_image: 200, imagen_4_fast: 60, imagen_4_std: 60, imagen_4_ultra: 10,
        veo_3_1: 5, veo_3_1_fast: 5, veo_3_0: 5, veo_2_0: 5
    },
    maintenanceMode: false,
    maintenanceMessage: "We are currently performing system maintenance. Please check back later.",
    maintenancePreCode: "admin123",
    maintenanceEndTime: Date.now() + 24 * 60 * 60 * 1000,
    safetyFilters: true,
    googleClientId: "",
    enableSimulatedGoogleLogin: true,
    enableGoogleLogin: true,
    facebookClientId: "",
    enableSimulatedFacebookLogin: true,
    enableFacebookLogin: true,
    twitterClientId: "",
    enableSimulatedTwitterLogin: true,
    enableTwitterLogin: true,
    smtpConfig: {
        host: 'smtp.example.com',
        port: 587,
        user: 'admin',
        pass: 'password',
        fromName: 'StockForge AI',
        fromEmail: 'noreply@stockforge.ai',
        secure: true
    },
    sidebarSettings: {
        'DASHBOARD': { id: 'DASHBOARD', enabled: true, comingSoon: false },
        'CHAT_INTELLIGENCE': { id: 'CHAT_INTELLIGENCE', enabled: true, comingSoon: false },
        'MEDIA_ANALYSIS': { id: 'MEDIA_ANALYSIS', enabled: true, comingSoon: false },
        'IMAGE_GENERATION': { id: 'IMAGE_GENERATION', enabled: true, comingSoon: false },
        'IMAGE_EDITOR': { id: 'IMAGE_EDITOR', enabled: true, comingSoon: false },
        'VIDEO_GENERATION': { id: 'VIDEO_GENERATION', enabled: true, comingSoon: false },
        'REVIEWER_AI': { id: 'REVIEWER_AI', enabled: true, comingSoon: false },
        'LIBRARY': { id: 'LIBRARY', enabled: true, comingSoon: false },
        'FAVORITES': { id: 'FAVORITES', enabled: true, comingSoon: false },
        'TRASH': { id: 'TRASH', enabled: true, comingSoon: false }
    },
    titleInstructionImage: `Act as a top-ranked Adobe Stock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (must be first in title)
2. Supporting visual elements
3. Environment or background
4. Commercial usage intent
5. Conceptual meaning (if relevant)

Create ONE SEO title that:
- matches buyer search intent
- is descriptive, not poetic
- starts with the main subject
- uses clear, searchable keywords
- includes "background", "copy space", or "template" ONLY if visible
- avoids adjectives that do not describe visuals
- avoids emotional storytelling
- avoids branding and metaphors
- is concise, professional, and under 70 characters`,

    titleInstructionVideo: `You are an expert in stock video SEO.
Analyze the provided video and identify:
- moving elements
- camera angle or motion
- scene type (interior/exterior)
- lighting and time of day

Generate ONE SEO-friendly video title that:
- starts with the main visual subject
- includes motion words only if visible
- avoids cinematic or narrative language
- stays under 75 characters
- follows Adobe Stock video title best practices`,

    titleInstructionSvg: `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style (flat, outline, 3D, minimal)
- usage context (UI, web, app, print)
- color and theme

Generate ONE SEO-optimized title that:
- starts with the main object name
- includes "vector", "icon", or "illustration" if relevant
- is suitable for stock marketplaces
- avoids artistic language
- remains under 65 characters`,

    descriptionInstructionImage: `Act as a top microstock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (first phrase)
2. Supporting objects or details
3. Scene type (interior/exterior/isolated/background)
4. Lighting, colors, and style
5. Commercial usage intent

Write ONE description that:
- clearly describes only what is visible
- opens with the main subject in the first 5 words
- naturally includes common buyer search terms
- includes "copy space" or "background" ONLY if visible
- avoids emotional adjectives unless visually clear
- avoids filler words and repetition
- remains factual and professional
- is optimized for stock search engines
- is between 2–3 concise sentences`,

    descriptionInstructionVideo: `You are an expert in stock video SEO.
Analyze the video and identify:
- moving subjects
- camera motion or angle
- environment and time of day
- lighting and scene style

Write ONE SEO-friendly description that:
- starts with the main moving subject
- describes visible motion accurately
- avoids cinematic or narrative language
- avoids assumptions about intent
- is suitable for Adobe Stock video listings
- uses 1–2 concise sentences`,

    descriptionInstructionSvg: `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style (flat, outline, minimal, 3D)
- color palette
- usage context (web, app, UI, print)

Write ONE SEO-optimized description that:
- starts with the main object
- includes "vector", "icon", or "illustration" if relevant
- focuses on usability and clarity
- avoids artistic or emotional language
- is suitable for microstock platforms`
};

const DEFAULT_LANDING_CONTENT = {
    general: { appName: "StockForge AI", appLogo: "", footerText: "© 2024 StockForge AI. Made with 💛 and Gemini." },
    hero: { badge: "AI POWERED", titlePrefix: "Forge Your", titleHighlight: "Creative Future", titleSuffix: "with AI Intelligence", subtitle: "The ultimate microstock creator toolkit. Generate, analyze, and optimize your portfolio with the power of Gemini 2.0 and Imagen 4.", buttonPrimary: "Start Creating", buttonSecondary: "Watch Demo", stat1Value: 13000, stat1Label: "Creators", stat2Value: 1500000, stat2Label: "Assets Generated", stat3Value: "4.9/5", stat3Label: "Rating" },
    features: { title: "Features", subtitle: "Everything you need to dominate the microstock market.", card1Title: "Imagen 4.0 Generation", card1Text: "Create stunning, photorealistic images with the latest Imagen 4.0 model. Optimized for commercial stock standards.", card2Title: "Veo Video", card2Text: "Generate cinematic 4K video clips.", card3Title: "Metadata AI", card3Text: "Auto-tagging with SEO precision.", card4Title: "Magic Editor", card4Text: "In-painting and upscaling tools.", card5Title: "Global Trend Analysis", card5Text: "Real-time market insights from major stock platforms to guide your next creation.", card6Title: "Deep Reasoning", card6Text: "Gemini 2.0 analyzes your portfolio performance." },
    about: { title: "Why <span class='text-amber-500'>StockForge?</span>", description: "We bridge the gap between AI generation and commercial viability. Our tools are designed specifically for stock contributors.", whyTitle: "Built for Contributors", box1Title: "Speed", box1Text: "Generate assets 10x faster than traditional workflows.", box2Title: "Quality", box2Text: "Commercial-grade outputs ready for upload.", box3Title: "Intelligence", box3Text: "Data-driven decisions based on market gaps." },
    pricing: { isVisible: true, title: "Simple Pricing", subtitle: "Start for free, upgrade for power.", starterVisible: true, starterName: "Starter", starterPrice: "$0", starterFeatures: "50 Images/mo\nStandard Resolution\nBasic Metadata\nCommunity Support", starterButton: "Start Free", proVisible: true, proName: "Pro Creator", proTag: "POPULAR", proPrice: "$29", proFeatures: "Unlimited Images\n4K Upscaling\nVeo Video Access\nTrend Analysis\nPriority Support", proButton: "Go Pro", agencyVisible: false, agencyName: "Agency", agencyPrice: "$99", agencyFeatures: "Team Access\nAPI Access\nCustom Models\nDedicated Account Manager", agencyButton: "Contact Sales" },
    globalImpact: { title: "Global Impact", slides: [{ id: 1, type: 'image', url: 'https://images.unsplash.com/photo-1618005182384-a83a8bd57fbe?q=80&w=2564&auto=format&fit=crop', title: 'Abstract Tech' }, { id: 2, type: 'image', url: 'https://images.unsplash.com/photo-1550751827-4bd374c3f58b?q=80&w=2670&auto=format&fit=crop', title: 'Cyberpunk City' }, { id: 3, type: 'image', url: 'https://images.unsplash.com/photo-1451187580459-43490279c0fa?q=80&w=2672&auto=format&fit=crop', title: 'Global Network' }, { id: 4, type: 'video', url: 'https://assets.mixkit.co/videos/preview/mixkit-digital-animation-of-a-technological-interface-9884-large.mp4', title: 'Digital Interface' }] },
    poweredBy: { title: "Powered By", chip1: "2.0 Flash", chip2: "Video FX", chip3: "4.0 Ultra" },
    cta: { title: "Ready to Start?", buttonText: "Join Now" },
    navigation: { featuresLabel: "Features", pricingLabel: "Pricing", aboutLabel: "About" }
};

const methodCache = new Map<string, { data: any; timestamp: number }>();
const CACHE_DURATION = 1000;

const getCached = (key: string): any | null => {
    const cached = methodCache.get(key);
    if (cached && (Date.now() - cached.timestamp < CACHE_DURATION)) {
        return cached.data;
    }
    return null;
};

const setCache = (key: string, data: any): void => {
    methodCache.set(key, { data, timestamp: Date.now() });
};

const checkIsAdmin = (): boolean => {
    const user = authService.getCurrentUser();
    return user?.role === 'admin';
};

export const adminService = {
    _fetchData: async (table: string) => {
        const response = await fetch(`${API_URL}/admin/data/${table}`, { headers: authService.getHeaders() });
        if (!response.ok) return [];
        const json = await response.json();
        return json.data || [];
    },

    _createItem: async (table: string, item: any) => {
        await fetch(`${API_URL}/admin/data/${table}`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ item })
        });
    },

    _updateItem: async (table: string, id: string, updates: any) => {
        await fetch(`${API_URL}/admin/data/${table}/update`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ id, updates })
        });
    },

    _deleteItem: async (table: string, id: string) => {
        await fetch(`${API_URL}/admin/data/${table}/delete`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ id })
        });
    },

    getUsers: async (): Promise<User[]> => {
        const rows = await adminService._fetchData('users');
        return rows.map((r: any) => ({
            uid: r.uid,
            displayName: r.display_name || 'User',
            email: r.email,
            photoURL: r.photo_url || null,
            role: r.role,
            isVerified: !!r.is_verified,
            isBanned: !!r.is_banned,
            isDeleted: !!r.is_deleted,
            createdAt: Number(r.created_at),
            lastLoginAt: Number(r.last_login_at),
            subscription: r.subscription || 'free',
            country: r.country,
            ipAddress: r.ip_address,
            usageStats: typeof r.usage_stats === 'string' ? JSON.parse(r.usage_stats) : (r.usage_stats || {}),
            preferences: typeof r.preferences === 'string' ? JSON.parse(r.preferences) : (r.preferences || {})
        }));
    },

    toggleUserBan: async (uid: string): Promise<void> => {
        const users = await adminService.getUsers();
        const user = users.find(u => u.uid === uid);
        if (user) {
            const newStatus = !user.isBanned;
            await adminService._updateItem('users', uid, { is_banned: newStatus });

            if ((user as any).ipAddress) {
                try {
                    await fetch(`${API_URL}/admin/banned-ips`, {
                        method: 'POST',
                        headers: authService.getHeaders(),
                        body: JSON.stringify({
                            ip: (user as any).ipAddress,
                            action: newStatus ? 'ban' : 'unban',
                            reason: newStatus ? `Banned along with user ${user.email}` : `Unbanned along with user ${user.email}`
                        })
                    });
                } catch (e) {
                    console.error("Failed to update IP ban status", e);
                }
            }

            await adminService.logAction(newStatus ? 'USER_BAN' : 'USER_UNBAN', `User ${user.email} (${uid}) was ${newStatus ? 'banned' : 'unbanned'}`);
        }
    },

    updateUserRole: async (uid: string, role: 'admin' | 'user'): Promise<void> => {
        await adminService._updateItem('users', uid, { role });
        await adminService.logAction('USER_ROLE_UPDATE', `User ${uid} role updated to ${role}`);
    },

    deleteUser: async (uid: string): Promise<void> => {
        await adminService._updateItem('users', uid, { is_deleted: true });
        await adminService.logAction('USER_DELETE', `User ${uid} marked as deleted`);
    },

    restoreUser: async (uid: string): Promise<void> => {
        await adminService._updateItem('users', uid, { is_deleted: false });
        await adminService.logAction('USER_RESTORE', `User ${uid} restored`);
    },

    permanentDeleteUser: async (uid: string): Promise<void> => {
        await fetch(`${API_URL}/admin/users/permanent-delete`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ uid })
        });
        await adminService.logAction('USER_PERMANENT_DELETE', `User ${uid} permanently deleted`);
    },

    getSiteSettings: async (): Promise<SiteSettings> => {
        try {
            const response = await fetch(`${API_URL}/settings`);
            if (!response.ok) return DEFAULT_SETTINGS;
            const json = await response.json();
            const settingsRow = json.data && json.data.length > 0 ? json.data[0] : null;
            return settingsRow ? (typeof settingsRow.settings === 'string' ? JSON.parse(settingsRow.settings) : settingsRow.settings) : DEFAULT_SETTINGS;
        } catch { return DEFAULT_SETTINGS; }
    },

    saveSiteSettings: async (settings: SiteSettings): Promise<void> => {
        await adminService._updateItem('system_settings', '1', { settings: JSON.stringify(settings) });
        await adminService.logAction('SETTINGS_UPDATE', 'Site settings updated');
        window.dispatchEvent(new CustomEvent('sf_settings_updated', { detail: settings }));
    },

    incrementSystemApiUsage: async (): Promise<void> => {
        try {
            await fetch(`${API_URL}/increment-usage`, { method: 'POST' });
        } catch (e) { console.error('Failed to increment usage', e); }
    },

    getAllTips: async (): Promise<Tip[]> => {
        const cached = getCached('getAllTips');
        if (cached) return cached;
        const isAdmin = checkIsAdmin();
        let result: Tip[];
        if (isAdmin) {
            try {
                const tips = await adminService._fetchData('tips');
                result = tips.map((t: any) => ({
                    id: t.id,
                    title: t.title,
                    text: t.text,
                    category: t.category,
                    createdAt: Number(t.created_at),
                    isDeleted: !!t.is_deleted
                }));
            } catch { result = DEFAULT_TIPS; }
        } else {
            try {
                const response = await fetch(`${API_URL}/tips`);
                if (response.ok) {
                    const json = await response.json();
                    const rows = json.data || [];
                    result = rows.map((t: any) => ({
                        id: t.id,
                        title: t.title,
                        text: t.text,
                        category: t.category,
                        createdAt: Number(t.created_at),
                        isDeleted: !!t.is_deleted
                    }));
                } else { result = DEFAULT_TIPS; }
            } catch { result = DEFAULT_TIPS; }
        }
        setCache('getAllTips', result);
        return result;
    },

    getRandomTips: async (count: number): Promise<Tip[]> => {
        try {
            const response = await fetch(`${API_URL}/tips`);
            if (!response.ok) return [];
            const json = await response.json();
            const activeTips = (json.data || []).map((t: any) => t);
            const shuffled = [...activeTips].sort(() => 0.5 - Math.random());
            return shuffled.slice(0, count);
        } catch { return []; }
    },

    getAllActiveTips: async (): Promise<Tip[]> => {
        try {
            const response = await fetch(`${API_URL}/tips`);
            if (!response.ok) return [];
            const json = await response.json();
            return json.data || [];
        } catch { return []; }
    },

    saveTip: async (tip: Tip): Promise<void> => {
        const dbItem = {
            title: tip.title,
            text: tip.text,
            category: tip.category,
            created_at: tip.createdAt,
            is_deleted: tip.isDeleted ? 1 : 0
        };

        if (tip.id) {
            const existing = (await adminService.getAllTips()).find(t => t.id === tip.id);
            if (existing) {
                await adminService._updateItem('tips', tip.id, dbItem);
            } else {
                await adminService._createItem('tips', { ...dbItem, id: tip.id });
            }
        } else {
            const newId = 'tip_' + Date.now();
            await adminService._createItem('tips', { ...dbItem, id: newId });
        }
        await adminService.logAction('TIP_UPDATE', `Tip "${tip.title}" saved`);
    },

    deleteTip: async (id: string) => {
        await adminService._updateItem('tips', id, { is_deleted: 1 });
        await adminService.logAction('TIP_DELETE', `Tip ${id} deleted`);
    },
    restoreTip: async (id: string) => adminService._updateItem('tips', id, { is_deleted: 0 }),
    permanentDeleteTip: async (id: string) => adminService._deleteItem('tips', id),

    populateDefaultTips: async () => {
        const tips = await adminService.getAllTips();

        console.log("Populating default tips...");
        for (const tip of DEFAULT_TIPS) {
            const exists = tips.some(t => t.title === tip.title);
            if (!exists) {
                await adminService.saveTip(tip);
            }
        }
    },

    getTutorialSteps: async (): Promise<TutorialStep[]> => {
        try {
            const response = await fetch(`${API_URL}/tutorial_steps`);
            if (!response.ok) throw new Error('Failed to fetch');
            const json = await response.json();
            const rows = json.data || [];
            return rows.map((r: any) => ({
                id: r.id, title: r.title, description: r.description,
                actionLabel: r.action_label, actionLink: r.action_link,
                order: r.step_order, isDeleted: !!r.is_deleted
            })).sort((a: any, b: any) => a.order - b.order);
        } catch (e) {
            console.warn("Using default tutorial steps", e);
            return DEFAULT_TUTORIAL_STEPS;
        }
    },

    saveTutorialStep: async (step: TutorialStep) => {
        if (!step.id) {
            step.id = 'step_' + Date.now();
            await adminService._createItem('tutorial_steps', step);
        } else {
            await adminService._updateItem('tutorial_steps', step.id, step);
        }
        await adminService.logAction('TUTORIAL_SAVE', `Step ${step.title} saved`);
    },

    deleteTutorialStep: async (id: string) => {
        await adminService._updateItem('tutorial_steps', id, { is_deleted: true });
        await adminService.logAction('TUTORIAL_DELETE', `Step ${id} deleted`);
    },
    restoreTutorialStep: async (id: string) => adminService._updateItem('tutorial_steps', id, { is_deleted: false }),
    reorderTutorialSteps: async (steps: TutorialStep[]): Promise<void> => {
        for (let i = 0; i < steps.length; i++) {
            await adminService._updateItem('tutorial_steps', steps[i].id, { step_order: i + 1 });
        }
        await adminService.logAction('TUTORIAL_REORDER', 'Tutorial steps reordered');
    },
    permanentDeleteTutorialStep: async (id: string) => adminService._deleteItem('tutorial_steps', id),

    getBroadcasts: async (): Promise<Broadcast[]> => {
        const cached = getCached('getBroadcasts');
        if (cached) return cached;
        const isAdmin = checkIsAdmin();
        let result: Broadcast[];
        if (isAdmin) {
            try {
                // Fetch users for reach calculation
                const users = await adminService.getUsers();
                const totalUsers = users.length;
                const paidUsers = users.filter(u => u.subscription && u.subscription !== 'free').length;

                const rows = await adminService._fetchData('broadcasts');
                result = rows.map((b: any) => {
                    const target = (b.target_audience || 'All').toLowerCase();
                    let reach = totalUsers;

                    if (target.includes('pro') || target.includes('paid')) {
                        reach = paidUsers;
                    }
                    // Add other segments if needed

                    const readCount = b.read_count || 0;
                    const openRate = reach > 0 ? Math.round((readCount / reach) * 100) : 0;

                    return {
                        id: b.id, title: b.title, content: b.message, type: b.type,
                        sentAt: b.sent_at, readCount: readCount, openRate: openRate,
                        targetAudience: b.target_audience || 'All', isDeleted: !!b.is_deleted,
                        recipients: reach
                    };
                }).sort((a: any, b: any) => b.sentAt - a.sentAt);
            } catch { result = []; }
        } else {
            try {
                const response = await fetch(`${API_URL}/broadcasts`, { headers: authService.getHeaders() });
                if (response.ok) {
                    const json = await response.json();
                    const rows = json.data || [];
                    result = rows.map((b: any) => ({
                        id: b.id, title: b.title, content: b.message, type: b.type,
                        sentAt: b.sent_at, isDeleted: !!b.is_deleted
                    })).sort((a: any, b: any) => b.sentAt - a.sentAt);
                } else { result = []; }
            } catch { result = []; }
        }
        setCache('getBroadcasts', result);
        return result;
    },

    getActiveBroadcasts: async (): Promise<Broadcast[]> => {
        try {
            const response = await fetch(`${API_URL}/broadcasts`, { headers: authService.getHeaders() });
            if (!response.ok) return [];
            const json = await response.json();
            const rows = json.data || [];
            return rows.map((b: any) => ({
                id: b.id, title: b.title, content: b.message, type: b.type,
                sentAt: b.sent_at, isDeleted: !!b.is_deleted
            })).sort((a: any, b: any) => b.sentAt - a.sentAt);
        } catch { return []; }
    },

    sendBroadcast: async (broadcast: Broadcast) => {
        const dbItem = {
            id: "bc_" + Date.now(),
            title: broadcast.title,
            message: broadcast.content,
            type: broadcast.type,
            target_audience: broadcast.targetAudience,
            sent_at: Date.now(),
            read_count: 0,
            open_rate: 0,
            is_deleted: 0
        };
        await adminService._createItem("broadcasts", dbItem);
        await adminService.logAction('BROADCAST_SEND', `Broadcast "${broadcast.title}" sent`);
    },

    deleteBroadcast: async (id: string) => {
        await adminService._updateItem('broadcasts', id, { is_deleted: true });
        await adminService.logAction('BROADCAST_DELETE', `Broadcast ${id} soft-deleted`);
    },
    restoreBroadcast: async (id: string) => {
        await adminService._updateItem('broadcasts', id, { is_deleted: false });
        await adminService.logAction('BROADCAST_RESTORE', `Broadcast ${id} restored`);
    },
    permanentDeleteBroadcast: async (id: string) => {
        await adminService._deleteItem('broadcasts', id);
        await adminService.logAction('BROADCAST_PERMANENT_DELETE', `Broadcast ${id} permanently deleted`);
    },

    getBugReports: async (): Promise<BugReport[]> => {
        const cached = getCached('getBugReports');
        if (cached) return cached;
        const isAdmin = checkIsAdmin();
        if (!isAdmin) {
            const emptyResult: BugReport[] = [];
            setCache('getBugReports', emptyResult);
            return emptyResult;
        }
        let result: BugReport[];
        try {
            const reports = await adminService._fetchData('bug_reports');
            result = reports.map((r: any) => ({
                id: r.id, userId: r.user_uid, title: r.title, description: r.description,
                steps: r.steps, severity: r.severity, status: r.status, timestamp: r.timestamp,
                isDeleted: !!r.is_deleted
            }));
        } catch { result = []; }
        setCache('getBugReports', result);
        return result;
    },

    submitBugReport: async (data: any) => {
        const id = 'bug_' + Date.now();
        const dbItem = {
            id, user_uid: data.userId, title: data.title, description: data.description,
            steps: data.steps, severity: data.severity, status: 'open', timestamp: Date.now(), is_deleted: 0
        };
        await adminService._createItem('bug_reports', dbItem);
        await adminService.logAction('BUG_REPORT_SUBMIT', `Bug report "${data.title}" submitted by ${data.userId}`);
    },

    updateBugStatus: async (id: string, status: string) => {
        await adminService._updateItem('bug_reports', id, { status });
        await adminService.logAction('BUG_REPORT_STATUS_UPDATE', `Bug report ${id} status updated to ${status}`);
    },

    getSupportMessages: async (userId?: string): Promise<SupportMessage[]> => {
        try {
            const user = authService.getCurrentUser();
            const isAdmin = user?.role === 'admin';
            let msgs = [];
            if (isAdmin) {
                msgs = await adminService._fetchData('support_messages');
            } else {
                const response = await fetch(`${API_URL}/user/messages`, { headers: authService.getHeaders() });
                if (response.ok) {
                    const json = await response.json();
                    msgs = json.data || [];
                }
            }
            const mapped = msgs.map((m: any) => ({
                id: m.id, userId: m.user_uid, userEmail: m.email, userName: m.user_name || 'User',
                subject: m.subject, text: m.text || m.message, status: m.status,
                timestamp: Number(m.timestamp), sender: m.sender || 'user',
            }));
            if (userId && isAdmin) return mapped.filter((m: any) => m.userId === userId);
            return mapped;
        } catch (e) {
            console.error("Failed to get messages", e);
            return [];
        }
    },

    sendSupportMessage: async (msg: SupportMessage) => {
        const id = msg.id || 'supp_' + Date.now();
        const dbItem = {
            id, user_uid: msg.userId, email: msg.userEmail, message: msg.text,
            subject: 'Support Chat', status: msg.status || 'unread',
            timestamp: msg.timestamp, sender: msg.sender
        };
        await adminService._createItem('support_messages', dbItem);
    },

    // --- AUDIT ---
    getAuditLogs: async (): Promise<AuditLog[]> => adminService._fetchData('audit_logs'),

    logAction: async (action: string, details: string, admin_uid?: string) => {
        const admin = admin_uid || authService.getCurrentUser()?.uid || 'system';
        const id = "audit_" + Date.now();
        await adminService._createItem("audit_logs", { id, action, details, admin_uid: admin, timestamp: Date.now() });
    },

    // --- LANDING CONTENT ---
    getLandingContent: async (): Promise<any> => {
        try {
            const response = await fetch(`${API_URL}/landing-content`);
            if (!response.ok) return DEFAULT_LANDING_CONTENT;
            const json = await response.json();
            return json.data || DEFAULT_LANDING_CONTENT;
        } catch (e) {
            console.warn('[adminService] Failed to fetch landing content, using defaults', e);
            return DEFAULT_LANDING_CONTENT;
        }
    },

    saveLandingContent: async (content: any): Promise<void> => {
        const rows = await adminService._fetchData('system_settings');
        const exists = rows.some((r: any) => r.id === 2);
        if (exists) {
            await adminService._updateItem('system_settings', '2', { settings: JSON.stringify(content) });
        } else {
            await adminService._createItem('system_settings', { id: 2, settings: JSON.stringify(content) });
        }
        await adminService.logAction('LANDING_UPDATE', 'Landing page content updated');
    },

    // --- MISSING METHODS ---
    restoreBugReport: async (id: string) => adminService._updateItem('bug_reports', id, { is_deleted: false }),
    permanentDeleteBugReport: async (id: string) => adminService._deleteItem('bug_reports', id),
    deleteBugReport: async (id: string) => adminService._updateItem('bug_reports', id, { is_deleted: true }),

    deleteConversation: async (userId: string) => {
        const msgs = await adminService.getSupportMessages(userId);
        for (const m of msgs) {
            await adminService._deleteItem('support_messages', m.id);
        }
    },
    deleteSupportMessage: async (id: string) => adminService._deleteItem('support_messages', id),
    markMessageAsRead: async (id: string) => adminService._updateItem('support_messages', id, { status: 'read' }),

    saveBroadcast: async (broadcast: any) => {
        if (broadcast.id) {
            const updates: any = {};
            if (broadcast.title !== undefined) updates.title = broadcast.title;
            if (broadcast.content !== undefined) updates.message = broadcast.content;
            if (broadcast.type !== undefined) updates.type = broadcast.type;
            if (broadcast.sentAt !== undefined) updates.sent_at = broadcast.sentAt;
            if (broadcast.targetAudience !== undefined) updates.target_audience = broadcast.targetAudience;
            await adminService._updateItem('broadcasts', broadcast.id, updates);
        } else {
            await adminService.sendBroadcast(broadcast);
        }
    },

    resetStats: async () => {
        await fetch(`${API_URL}/admin/stats/reset`, { method: 'POST', headers: authService.getHeaders() });
        await adminService.logAction('STATS_RESET', 'All admin stats reset');
    },

    markBroadcastRead: async (broadcastId: string, uid: string) => {
        try {
            // Optimistic Client Update
            const prefs = await UserDatabase.getPreferences(uid);
            const dismissed = prefs.dismissedBroadcasts || [];
            if (!dismissed.includes(broadcastId)) {
                await UserDatabase.updatePreferences(uid, {
                    dismissedBroadcasts: [...dismissed, broadcastId]
                });
            }

            // Server Analytics Update
            await fetch(`${API_URL}/user/${uid}/dismiss-broadcast`, {
                method: 'POST',
                headers: authService.getHeaders(),
                body: JSON.stringify({ broadcastId })
            });
        } catch (e) {
            console.error("Failed to mark broadcast as read (Analytics)", e);
        }
    },

    // --- STOCK SITES (MARKET INTEL) ---
    getMicrostockSites: async (): Promise<StockSite[]> => {
        const cached = getCached('getMicrostockSites');
        if (cached) return cached;

        try {
            const isAdmin = checkIsAdmin();
            let rows;

            if (isAdmin) {
                console.log('[adminService] Fetching ALL sites (Admin Mode)');
                rows = await adminService._fetchData('stock_sites');
            } else {
                console.log('[adminService] Fetching VISIBLE sites (User Mode)');
                const response = await fetch(`${API_URL}/stock-sites`, { headers: authService.getHeaders() });
                if (!response.ok) {
                    console.error('[adminService] Failed to fetch stock sites:', response.status);
                    return [];
                }
                const json = await response.json();
                console.log(`[adminService] Fetched ${json.data?.length} sites`);
                rows = json.data || [];
            }

            const sites = rows.map((s: any) => ({
                id: s.id,
                name: s.name,
                url: s.url,
                category: s.category,
                description: s.description,
                contributors: s.contributors,
                librarySize: s.library_size,
                commission: s.commission,
                isVisible: !!s.is_visible,
                isVisibleMarket: !!s.is_visible_market, // New field mapping
                isAiPowered: !!(s.pros && JSON.stringify(s).toLowerCase().includes('ai')),
                pros: typeof s.pros === 'string' ? JSON.parse(s.pros) : (s.pros || []),
                cons: typeof s.cons === 'string' ? JSON.parse(s.cons) : (s.cons || []),
                isDeleted: !!s.is_deleted
            })).filter(s => !s.isDeleted);

            setCache('getMicrostockSites', sites);
            return sites;
        } catch (e) {
            console.error("Failed to fetch sites", e);
            return [];
        }
    },

    saveMicrostockSite: async (site: StockSite) => {
        const dbItem = {
            name: site.name,
            url: site.url,
            category: site.category,
            description: site.description,
            contributors: site.contributors,
            library_size: site.librarySize,
            commission: site.commission,
            is_visible: site.isVisible ? 1 : 0,
            is_visible_market: site.isVisibleMarket ? 1 : 0,
            pros: JSON.stringify(site.pros),
            cons: JSON.stringify(site.cons)
        };

        if (site.id) {
            const existing = (await adminService.getMicrostockSites()).find(s => s.id === site.id);
            if (existing) {
                await adminService._updateItem('stock_sites', site.id, dbItem);
            } else {
                await adminService._createItem('stock_sites', { ...dbItem, id: site.id });
            }
        } else {
            const newId = 'site_' + Date.now();
            await adminService._createItem('stock_sites', { ...dbItem, id: newId });
        }
        await adminService.logAction('SITE_UPDATE', `Site ${site.name} updated`);
        methodCache.delete('getMicrostockSites');
    },

    deleteMicrostockSite: async (id: string) => {
        await adminService._updateItem('stock_sites', id, { is_deleted: 1 });
        await adminService.logAction('SITE_DELETE', `Site ${id} deleted`);
        methodCache.delete('getMicrostockSites');
    },

    toggleMicrostockSite: async (id: string, field: 'reviewer' | 'market' = 'reviewer') => {
        const sites = await adminService.getMicrostockSites();
        const site = sites.find(s => s.id === id);
        if (site) {
            const dbField = field === 'market' ? 'is_visible_market' : 'is_visible';
            const currentValue = field === 'market' ? site.isVisibleMarket : site.isVisible;

            await adminService._updateItem('stock_sites', id, { [dbField]: !currentValue ? 1 : 0 });
            await adminService.logAction('SITE_TOGGLE', `Site ${id} ${field} visibility toggled`);
            methodCache.delete('getMicrostockSites');
        }
    },

    setAllMicrostockVisibility: async (field: 'reviewer' | 'market', value: boolean) => {
        const dbField = field === 'market' ? 'is_visible_market' : 'is_visible';
        await fetch(`${API_URL}/admin/stock-sites/bulk-visibility`, {
            method: 'POST',
            headers: authService.getHeaders(),
            body: JSON.stringify({ field: dbField, value })
        });
        await adminService.logAction('SITE_TOGGLE', `All sites ${field} visibility set to ${value}`);
        methodCache.delete('getMicrostockSites');
    },

    populateDefaultSites: async () => {
        const sites = await adminService.getMicrostockSites();
        // Removed length check to ensure missing sites are added even if some exist

        console.log("Populating default stock sites...");
        const DEFAULT_SITES: Partial<StockSite>[] = [
            // --- MAJORS (The Big 4) ---
            {
                name: 'Shutterstock',
                url: 'https://submit.shutterstock.com',
                category: 'Major',
                description: 'The largest microstock agency with massive traffic.',
                contributors: '1M+', librarySize: '450M+', commission: '15-40%',
                isAiPowered: true, isVisible: true,
                pros: ['Huge customer base & traffic', 'Reliable monthly payouts', 'Good API integration'],
                cons: ['High competition', 'Low royalty rates for new contributors', 'Strict metadata rules']
            },
            {
                name: 'Adobe Stock',
                url: 'https://contributor.stock.adobe.com',
                category: 'Major',
                description: 'Integrated directly into Adobe Creative Cloud apps.',
                contributors: '1M+', librarySize: '350M+', commission: '33%',
                isAiPowered: true, isVisible: true,
                pros: ['Integrated into Creative Cloud', 'Higher average price per sale', 'Accepts AI content'],
                cons: ['Strict technical review', 'Payment threshold can be high', 'Slower review times']
            },
            {
                name: 'Getty Images / iStock',
                url: 'https://esp.gettyimages.com',
                category: 'Major',
                description: 'Premium agency with high standards and exclusive options.',
                contributors: '500k+', librarySize: '500M+', commission: '15-45%',
                isAiPowered: true, isVisible: true,
                pros: ['Prestigious reputation', 'Exclusive contributor options', 'Global reach via Getty'],
                cons: ['Convoluted upload system (ESP)', 'Low stats visibility', 'Complex royalty structure']
            },
            {
                name: 'Alamy',
                url: 'https://www.alamy.com/contributor',
                category: 'Major',
                description: 'British agency known for editorial content and higher royalties.',
                contributors: '200k+', librarySize: '300M+', commission: '40-50%',
                isAiPowered: false, isVisible: true,
                pros: ['High commission rates', 'Good for editorial/news', 'No exclusivity required'],
                cons: ['Slower checking times', 'Outdated interface', 'Requires precise keywording']
            },

            // --- MID-TIER & WELL KNOWN ---
            {
                name: 'Dreamstime', url: 'https://www.dreamstime.com', category: 'Mid-Tier', description: 'One of the oldest and most resilient stock agencies.',
                contributors: '500k+', librarySize: '200M+', commission: '25-50%', isVisible: true, isAiPowered: true,
                pros: ['Reliable long-term earner', 'Good community forums', 'Accepts editorial content'],
                cons: ['Search algorithm is tough', 'Upload process is slow', 'Dated user interface']
            },
            {
                name: 'Depositphotos', url: 'https://depositphotos.com', category: 'Mid-Tier', description: 'Fast-growing agency with strong marketing campaigns.',
                contributors: '100k+', librarySize: '250M+', commission: '34-42%', isVisible: true, isAiPowered: true,
                pros: ['Aggressive marketing', 'Frequent bulk sales', 'User-friendly mobile app'],
                cons: ['Low per-image rates', 'Review times vary', 'Generic catalog feel']
            },
            {
                name: '123RF', url: 'https://www.123rf.com', category: 'Mid-Tier', description: 'Asian-based agency with global reach, good for vectors.',
                contributors: '300k+', librarySize: '200M+', commission: '30-60%', isVisible: true, isAiPowered: true,
                pros: ['Steady passive income', 'Good for vector artists', 'Global distribution'],
                cons: ['Lower traffic recently', 'Low commission tiers', 'Site redesign bugs']
            },
            {
                name: 'Pond5', url: 'https://www.pond5.com', category: 'Video', description: 'The absolute market leader for stock footage and video.',
                contributors: '100k+', librarySize: '30M+', commission: '40-60%', isVisible: true, isAiPowered: true,
                pros: ['Best for video sales', 'Set your own prices', 'Reference for video pricing'],
                cons: ['Photo sales are weak', 'High competition in 4K', 'Interface needs update']
            },
            {
                name: 'Storyblocks', url: 'https://www.storyblocks.com', category: 'Video', description: 'Subscription-based model popular with YouTubers.',
                contributors: 'Selected', librarySize: '1M+', commission: 'Share', isVisible: true, isAiPowered: true,
                pros: ['High volume video usage', 'Member library model', 'Steady subscription checks'],
                cons: ['Application based entry', 'Cannibalizes single sales', 'Fixed pricing pool']
            },
            {
                name: 'Vecteezy', url: 'https://www.vecteezy.com', category: 'Vector', description: 'Specialized in vector art, icons, and illustrations.',
                contributors: '20k+', librarySize: '10M+', commission: 'Pro Rata', isVisible: true, isAiPowered: true,
                pros: ['Great for vector artists', 'Free/Pro hybrid model', 'Simple upload process'],
                cons: ['Pay per download is low', 'Requires high volume', 'Strict vector standards']
            },
            {
                name: 'Freepik', url: 'https://contributor.freepik.com', category: 'Vector', description: 'The highest volume site for free and premium assets.',
                contributors: '50k+', librarySize: '50M+', commission: 'Downloads', isVisible: true, isAiPowered: true,
                pros: ['Massive traffic volume', 'Accepts AI content', 'King of vector downloads'],
                cons: ['Very low pay per DL', 'Requires huge portfolio', 'Subscription dominance']
            },
            {
                name: 'Canva', url: 'https://www.canva.com/contributors', category: 'Design', description: 'The design platform used by millions of non-designers.',
                contributors: 'Selective', librarySize: '100M+', commission: 'Pool', isVisible: true, isAiPowered: true,
                pros: ['Huge user base usage', 'Integration in design tool', 'Growing rapidly'],
                cons: ['Hard to get accepted', 'Low transparency on pay', 'Payment delay']
            },
            {
                name: 'Stocksy', url: 'https://www.stocksy.com', category: 'Premium', description: 'An artist-owned cooperative with a curated aesthetic.',
                contributors: '2k+', librarySize: '2M+', commission: '50-75%', isVisible: true, isAiPowered: false,
                pros: ['Artistic & authentic focus', 'Highest royalties', 'Co-op dividends'],
                cons: ['Exclusive content only', 'Hard to join/audition', 'Strict aesthetic rules']
            },
            {
                name: 'Envato Elements', url: 'https://author.envato.com', category: 'Subscription', description: 'The "Netflix of creative assets" - huge subscription value.',
                contributors: 'Invite', librarySize: '10M+', commission: 'Sub Share', isVisible: true, isAiPowered: true,
                pros: ['Consistent monthly income', 'Broad asset types accepted', 'Sticky customer base'],
                cons: ['Invite only system', 'Cannibalizes Market sales', 'High update requirement']
            },
            {
                name: 'MotionArray', url: 'https://motionarray.com', category: 'Video', description: 'All-in-one platform for video creators and editors.',
                contributors: '5k+', librarySize: '500k+', commission: 'PPU', isVisible: true, isAiPowered: false,
                pros: ['Video templates sell well', 'Premiere Pro assets', 'Artlist ecosystem'],
                cons: ['Long review times', 'Strict quality control', 'Specific template structure']
            },

            // --- NICHE & SPECIALTY ---
            {
                name: 'Artlist', url: 'https://artlist.io', category: 'Video', description: 'High-quality royalty-free music and SFX for video.',
                contributors: 'Curated', librarySize: '100k+', commission: 'Flat/Share', isVisible: true, isAiPowered: false,
                pros: ['Quality over quantity', 'Video/Music focus', 'Simple licensing'],
                cons: ['Highly curated entry', 'Buyout models common', 'Limited slots']
            },
            {
                name: 'Epidemic Sound', url: 'https://www.epidemicsound.com', category: 'Audio', description: 'The standard for YouTuber background music.',
                contributors: 'Selective', librarySize: '40k+', commission: 'Buyout', isVisible: true, isAiPowered: false,
                pros: ['Music market dominance', 'Good for full-time musicians', 'Stable work'],
                cons: ['Music only - no photos', 'Often 100% buyout', 'Exclusivity required']
            },
            {
                name: 'MusicBed', url: 'https://www.musicbed.com', category: 'Audio', description: 'Cinematic quality music licensing for filmmakers.',
                contributors: '1k+', librarySize: '20k+', commission: 'High', isVisible: true, isAiPowered: false,
                pros: ['Cinematic quality', 'High licensing fees', 'Placement in TV/Film'],
                cons: ['Very selective application', 'Niche formatting', 'Lengthy onboarding']
            },
            {
                name: 'Vimeo Stock', url: 'https://vimeo.com/stock', category: 'Video', description: 'Stock footage platform powered by the Vimeo community.',
                contributors: 'Open', librarySize: '1M+', commission: '60-70%', isVisible: true, isAiPowered: false,
                pros: ['High quality connection', 'Generous commission', 'Professional audience'],
                cons: ['Niche traffic volume', 'Overshadowed by giants', 'Video only']
            },
            {
                name: 'Dissolve', url: 'https://dissolve.com', category: 'Premium', description: 'Lifestyle and commercial footage for high-end brands.',
                contributors: 'Selective', librarySize: '3M+', commission: 'High', isVisible: true, isAiPowered: false,
                pros: ['Premium footage prices', 'Ad agency clients', 'Respects creators'],
                cons: ['Exclusive heavy', 'Slow acceptance', 'Low volume high price']
            },
            {
                name: 'EyeEm', url: 'https://www.eyeem.com', category: 'Mobile', description: 'Berlin-based agency focused on authentic mobile photography.',
                contributors: '20M+', librarySize: '100M+', commission: '50%', isVisible: true, isAiPowered: true,
                pros: ['Partner distribution (Getty)', 'Mobile first workflow', 'Authentic style'],
                cons: ['Slow payouts (history)', 'Company stability issues', 'Low direct sales']
            },
            {
                name: '500px', url: 'https://500px.com', category: 'Community', description: 'Photography community with a licensing marketplace attached.',
                contributors: '15M+', librarySize: '100M+', commission: '60%', isVisible: true, isAiPowered: false,
                pros: ['Great photography community', 'Distribution partners', 'Exposure'],
                cons: ['Declining sales', 'More social than stock', 'China ownership concerns']
            },
            {
                name: 'SmugMug', url: 'https://www.smugmug.com', category: 'Portfolio', description: 'Portfolio hosting that allows selling prints and downloads.',
                contributors: 'N/A', librarySize: 'Hosted', commission: '85%', isVisible: true, isAiPowered: false,
                pros: ['Print sales fulfillment', 'Total control of site', 'High keep rate (85%)'],
                cons: ['Monthly subscription fee', 'No marketplace traffic', 'Self-marketing needed']
            },
            {
                name: 'Fine Art America', url: 'https://fineartamerica.com', category: 'POD', description: 'The largest print-on-demand site for wall art.',
                contributors: '500k+', librarySize: '10M+', commission: 'Custom', isVisible: true, isAiPowered: false,
                pros: ['Best for wall prints', 'Global manufacturing', 'Set your own markup'],
                cons: ['Marketing required', 'Not strictly stock', 'Base prices can be high']
            },
            {
                name: 'Redbubble', url: 'https://www.redbubble.com', category: 'POD', description: 'Marketplace for independent artists selling on products.',
                contributors: '800k+', librarySize: '5M+', commission: 'Margin %', isVisible: true, isAiPowered: true,
                pros: ['Easy to use / upload', 'Merch focus (Stickers/Tees)', 'Organic organic traffic'],
                cons: ['Copycats / Theft common', 'Variable print quality', 'Lower margins']
            },
            {
                name: 'Society6', url: 'https://society6.com', category: 'POD', description: 'Premium POD focused on home decor and lifestyle goods.',
                contributors: '400k+', librarySize: '3M+', commission: '10%', isVisible: true, isAiPowered: false,
                pros: ['Artistic / Decor vibe', 'High average order', 'Trend setting'],
                cons: ['Low margins (10%)', 'Upload process tedious', 'Strict file reqs']
            },
            {
                name: 'Zazzle', url: 'https://www.zazzle.com', category: 'POD', description: 'Deeply customizable products for gifts and business.',
                contributors: '1M+', librarySize: '200M+', commission: 'Custom', isVisible: true, isAiPowered: false,
                pros: ['Extreme customization', 'High volume potential', 'Set your own royalties'],
                cons: ['Dated interface', 'Steep learning curve', 'Complex product creation']
            },
            {
                name: 'Displate', url: 'https://displate.com', category: 'POD', description: 'Magnet-mounted metal prints. Extremely popular with gamers.',
                contributors: '40k+', librarySize: '1M+', commission: 'Flat/Share', isVisible: true, isAiPowered: true,
                pros: ['Metal prints niche', 'Unique product', 'High collector value'],
                cons: ['Specific format only', 'Upload limits', 'Review process']
            },
            {
                name: 'Cavan Images', url: 'https://cavanimages.com', category: 'Premium', description: 'Ultra-premium lifestyle content for advertising.',
                contributors: 'Selected', librarySize: '500k+', commission: 'High', isVisible: true, isAiPowered: false,
                pros: ['Ad agency favorite', 'High licensing fees', 'Offline sales'],
                cons: ['Exclusive focus', 'Hard to get in', 'Low volume']
            },
            {
                name: 'Westend61', url: 'https://www.westend61.de', category: 'Premium', description: 'Europe’s leading premium lifestyle agency.',
                contributors: 'Selected', librarySize: '500k+', commission: '40-50%', isVisible: true, isAiPowered: false,
                pros: ['Strong European focus', 'Authentic Lifestyle', 'Personal support'],
                cons: ['Strict quality control', 'Mostly exclusive', 'Smaller market share']
            },
            {
                name: 'Arcangel', url: 'https://arcangel.com', category: 'Specialty', description: 'The premier agency for book cover photography.',
                contributors: 'Selected', librarySize: '700k+', commission: '50%', isVisible: true, isAiPowered: true,
                pros: ['Book covers pay well', 'Global publishing clients', 'Creative freedom'],
                cons: ['Very strict standards', 'Deep backlog', 'Long wait for sales']
            },
            {
                name: 'Trevillion', url: 'https://www.trevillion.com', category: 'Specialty', description: 'Creative stock photography for the publishing industry.',
                contributors: 'Selected', librarySize: '200k+', commission: '50%', isVisible: true, isAiPowered: false,
                pros: ['Creative & Emotional', 'Book cover niche', 'Respected brand'],
                cons: ['Selective admission', 'Niche market', 'Slower sales cycle']
            },
            {
                name: 'Wildberries', url: 'https://wildberries.com', category: 'Regional', description: 'Russian e-commerce giant with stock capabilities.',
                contributors: 'Local', librarySize: 'Unknown', commission: 'Varies', isVisible: false, isAiPowered: false,
                pros: ['Russian market access', 'Huge volume', 'Apparel focus'],
                cons: ['Regional strictness', 'Language barrier', 'Payment complex']
            },
            {
                name: 'Pixta', url: 'https://www.pixta.jp', category: 'Specialty', description: 'Leading microstock agency in Japan and Asia.',
                contributors: '300k+', librarySize: '50M+', commission: 'Varies', isVisible: true, isAiPowered: true,
                pros: ['Japan market leader', 'Asia specific content', 'Growing demand'],
                cons: ['Language barrier', 'Specific aesthetic needed', 'Yen payments']
            },
            {
                name: 'Topic Images', url: 'https://www.topicimages.com', category: 'Specialty', description: 'Korean stock agency specializing in local content.',
                contributors: 'Local', librarySize: '5M+', commission: 'Varies', isVisible: true, isAiPowered: false,
                pros: ['Korea focus', 'High local prices', 'Less competition'],
                cons: ['Niche geography', 'Korean language site', 'Limited access']
            },
            {
                name: 'Visual China Group', url: 'https://www.vcg.com', category: 'Major', description: 'The "Getty Images of China" - massive market reach.',
                contributors: 'Selected', librarySize: '200M+', commission: 'Varies', isVisible: true, isAiPowered: false,
                pros: ['China market dominance', 'Huge volume', 'Strategic partnerships'],
                cons: ['Complex access', 'Government regulations', 'Payment logistics']
            },
            {
                name: 'Mostphotos', url: 'https://www.mostphotos.com', category: 'Niche', description: 'Stock agency with a strong Nordic presence.',
                contributors: '50k+', librarySize: '15M+', commission: '50%', isVisible: true, isAiPowered: false,
                pros: ['Nordic focus', 'Fair 50/50 split', 'Easy upload'],
                cons: ['Smaller traffic', 'Europe centric', 'Slower growth']
            },
            {
                name: 'Colourbox', url: 'https://www.colourbox.com', category: 'Niche', description: 'Danish agency with a focus on education clients.',
                contributors: '20k+', librarySize: '25M+', commission: 'High', isVisible: true, isAiPowered: false,
                pros: ['Education market', 'Bulk sales', 'Simple license'],
                cons: ['Strict metadata', 'Dated interface', 'Lower single sales']
            },
            {
                name: 'PantherMedia', url: 'https://panthermedia.net', category: 'Niche', description: 'German agency with a vast distribution network.',
                contributors: '10k+', librarySize: '100M+', commission: '30-50%', isVisible: true, isAiPowered: false,
                pros: ['German market', 'Hundreds of partners', 'Passive distribution'],
                cons: ['Smaller direct sales', 'Complex reporting', 'Delayed stats']
            },
            {
                name: 'Zoonar', url: 'https://www.zoonar.com', category: 'Niche', description: 'Distributor that pushes files to many other agencies.',
                contributors: '50k+', librarySize: '8M+', commission: 'Varies', isVisible: true, isAiPowered: false,
                pros: ['One upload -> Many sites', 'Distribution partners', 'Metadata tools'],
                cons: ['Dated UI', 'Slow reporting', 'Partner cuts take a share']
            },
            {
                name: 'YayImages', url: 'https://yayimages.com', category: 'Subscription', description: 'Streaming-model stock site for startups.',
                contributors: '12k+', librarySize: '12M+', commission: 'Streaming', isVisible: true, isAiPowered: true,
                pros: ['Startups love it', 'Unique visual search', 'Fair split logic'],
                cons: ['Low per DL revenue', 'Small market share', 'Volume dependent']
            },
            {
                name: 'StockXpert', url: '#', category: 'Budget', description: 'Legacy agency, now mostly defunct or integrated.',
                contributors: 'Closed', librarySize: 'Archived', commission: '0%', isVisible: false, isAiPowered: false,
                pros: ['Historical significance', 'Was a major player', 'Learn from history'],
                cons: ['Closed business', 'No uploads', 'Redirects elsewhere']
            },
            {
                name: 'FreeImages', url: 'https://www.freeimages.com', category: 'Free', description: 'Formerly sxc.hu, a pioneer of free stock.',
                contributors: 'Open', librarySize: '3M+', commission: 'None', isVisible: true, isAiPowered: false,
                pros: ['Exposure only', 'Easy entry', 'Traffic funnel'],
                cons: ['No payment', 'Devalued work', 'Getty owned funnel']
            },
            {
                name: 'Fotopro', url: '#', category: 'Niche', description: 'General stock site with standard offerings.',
                contributors: 'Unknown', librarySize: 'Small', commission: 'Standard', isVisible: true, isAiPowered: false,
                pros: ['Generic backup', 'Another basket', 'Simple'],
                cons: ['Small traffic', 'Low visibility', 'Generic']
            },
            {
                name: 'Bigstock', url: 'https://www.bigstockphoto.com', category: 'Mid-Tier', description: 'Shutterstock subsidiary with a credit-based model.',
                contributors: 'See SS', librarySize: '100M+', commission: '30%', isVisible: true, isAiPowered: false,
                pros: ['Shutterstock owned', 'Credit model', 'Easier acceptance'],
                cons: ['Dying slowly', 'Uploads via SS now', 'Low priority']
            },
            {
                name: 'VectorStock', url: 'https://www.vectorstock.com', category: 'Vector', description: 'Dedicated exclusively to vector graphics.',
                contributors: '10k+', librarySize: '30M+', commission: '25%', isVisible: true, isAiPowered: false,
                pros: ['Vectors only', 'Targeted buying', 'Simple licensing'],
                cons: ['Small per-sale cut', 'No photos allowed', 'Specific format reqs']
            },
            {
                name: 'Crestock', url: 'https://www.crestock.com', category: 'Budget', description: 'Small agency focused on low-cost images.',
                contributors: 'Closed', librarySize: 'Small', commission: '20%', isVisible: true, isAiPowered: false,
                pros: ['Simple', 'Good for buyers', 'Cheap'],
                cons: ['Inactive mostly', 'Masterfile owned', 'Low payouts']
            },
            {
                name: 'Can Stock Photo', url: '#', category: 'Defunct', description: 'Was a popular agency, closed operations in 2023.',
                contributors: 'Closed', librarySize: 'Archived', commission: '0%', isVisible: false, isAiPowered: false,
                pros: ['History'], cons: ['Closed']
            },
            {
                name: 'Wirestock', url: 'https://wirestock.io', category: 'AI', description: 'One-stop shop to upload to many agencies + sell AI art.',
                contributors: 'Aggregator', librarySize: 'Aggregated', commission: '15% cut', isVisible: true, isAiPowered: true,
                pros: ['One upload to all', 'AI tagging & keywording', 'Accepts AI art'],
                cons: ['They take a 15% cut', 'You don\'t own the accounts', 'Delay in stats']
            },
            {
                name: 'BlackBox', url: 'https://blackbox.global', category: 'Video', description: 'Co-op platform for video creators to distribute footage.',
                contributors: 'Co-op', librarySize: 'Video', commission: 'Split', isVisible: true, isAiPowered: false,
                pros: ['Video aggregator', 'Metadata help', 'Community focus'],
                cons: ['Complex splits', 'Video only', 'Platform fee']
            },
            {
                name: 'Picfair', url: 'https://www.picfair.com', category: 'Specialty', description: 'Host your own photography store with zero commission.',
                contributors: 'Open', librarySize: 'User', commission: 'You Set', isVisible: true, isAiPowered: false,
                pros: ['Your own store', 'Set your own prices', 'Fair simple license'],
                cons: ['You drive traffic', 'Not a marketplace', 'Subscription for Plus']
            },
            {
                name: 'Foap', url: 'https://www.foap.com', category: 'Mobile', description: 'App that turns your phone photos into cash.',
                contributors: 'Mobile', librarySize: 'User', commission: '50/50', isVisible: true, isAiPowered: false,
                pros: ['Brand missions pay well', 'Directly from phone', 'Simple app'],
                cons: ['Low sales outside missions', 'Photo quality varies', 'Crowded']
            },
            {
                name: 'VideoHive', url: 'https://videohive.net', category: 'Video', description: 'Stock footage and motion graphics templates.',
                contributors: '52k+', librarySize: '2M+', commission: '55-87%', isVisible: true, isAiPowered: true,
                pros: ['Market leader for templates', 'High traffic', 'Good for motion graphics'],
                cons: ['High competition', 'Price dumping', 'Strict technical review']
            }
        ];

        for (const site of DEFAULT_SITES) {
            const existing = sites.find(s => s.name === site.name);

            const siteWithDefaults = {
                category: 'Other', description: '', contributors: '?', librarySize: '?', commission: '?',
                isAiPowered: false, isVisible: true, pros: [], cons: [],
                ...site,
                id: existing ? existing.id : undefined
            };

            await adminService.saveMicrostockSite(siteWithDefaults as StockSite);
        }
    },

    getEmailLogs: async () => [],
    updateSystemStats: async () => { },
    getAnalytics: async () => {
        const response = await fetch(`${API_URL}/admin/stats`, { headers: authService.getHeaders() });
        return response.json();
    },
};
