import { Asset, TrashConfig } from '../dbService';
import { ApiKeyData, ActivityLog } from '../types';

// Type definitions for User Database Schema
export interface UserNotification {
    id: string;
    title: string;
    text: string;
    type: 'info' | 'success' | 'warning' | 'urgent';
    category: 'System' | 'Feature' | 'Announcement' | 'Support';
    timestamp: number;
    read: boolean;
    messageData?: any; // Optional: stores full message details for Support notifications
}

export interface UserPreferences {
    theme: 'dark' | 'light';
    trashConfig: TrashConfig;
    dismissedTutorial: boolean;
    tutorialProgress: string[]; // IDs of completed steps
    dismissedBroadcasts?: string[]; // IDs of dismissed broadcasts
    dismissedMessages?: string[]; // IDs of dismissed message notifications
}

const DB_NAME = 'StockForgeDB';
const DB_VERSION = 1;
const STORES = ['assets', 'api_keys', 'notifications', 'activity_logs', 'preferences'];

const getDB = (): Promise<IDBDatabase> => {
    return new Promise((resolve, reject) => {
        if (typeof window === 'undefined' || !window.indexedDB) {
            reject(new Error('IndexedDB is not supported in this browser'));
            return;
        }
        const request = window.indexedDB.open(DB_NAME, DB_VERSION);
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve(request.result);
        request.onupgradeneeded = (event) => {
            const db = (event.target as IDBOpenDBRequest).result;
            STORES.forEach(store => {
                if (!db.objectStoreNames.contains(store)) {
                    db.createObjectStore(store);
                }
            });
        };
    });
};

const performTransaction = <T>(storeName: string, mode: IDBTransactionMode, callback: (store: IDBObjectStore) => IDBRequest): Promise<T> => {
    return new Promise(async (resolve, reject) => {
        try {
            const db = await getDB();
            const tx = db.transaction(storeName, mode);
            const store = tx.objectStore(storeName);
            const request = callback(store);
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        } catch (error) {
            reject(error);
        }
    });
};

// The DB Driver
export const UserDatabase = {

    // --- Core Helpers ---

    // We use UID as the key for each object store since each user has one entry containing their array of data.
    // This mimics the previous localStorage structure but allows for much larger payloads.

    // --- Generic CRUD ---

    save: <T>(uid: string, collection: string, data: T): Promise<void> => {
        return performTransaction(collection, 'readwrite', (store) => store.put(data, uid));
    },

    get: <T>(uid: string, collection: string, defaultValue: T): Promise<T> => {
        return new Promise(async (resolve) => {
            try {
                const result = await performTransaction<T>(collection, 'readonly', (store) => store.get(uid));
                resolve(result === undefined ? defaultValue : result);
            } catch (e) {
                console.error(`IndexedDB Error [${collection}]:`, e);
                resolve(defaultValue);
            }
        });
    },

    // --- Specific Collections Accessors ---

    // 1. ASSETS (Library, Trash, Favorites)
    getAssets: (uid: string): Promise<Asset[]> => {
        return UserDatabase.get<Asset[]>(uid, 'assets', []);
    },

    saveAssets: (uid: string, assets: Asset[]): Promise<void> => {
        return UserDatabase.save(uid, 'assets', assets);
    },

    // 2. API KEYS
    getApiKeys: (uid: string): Promise<ApiKeyData[]> => {
        return UserDatabase.get<ApiKeyData[]>(uid, 'api_keys', []);
    },

    saveApiKeys: (uid: string, keys: ApiKeyData[]): Promise<void> => {
        return UserDatabase.save(uid, 'api_keys', keys);
    },

    // 3. NOTIFICATIONS
    getNotifications: (uid: string): Promise<UserNotification[]> => {
        return UserDatabase.get<UserNotification[]>(uid, 'notifications', []);
    },

    saveNotifications: (uid: string, notifs: UserNotification[]): Promise<void> => {
        return UserDatabase.save(uid, 'notifications', notifs);
    },

    addNotification: async (uid: string, notif: UserNotification): Promise<void> => {
        const current = await UserDatabase.getNotifications(uid);
        // Avoid duplicates by ID
        if (!current.find(n => n.id === notif.id)) {
            await UserDatabase.saveNotifications(uid, [notif, ...current]);
        }
    },

    markNotificationRead: async (uid: string, notifId: string): Promise<void> => {
        const current = await UserDatabase.getNotifications(uid);
        const updated = current.map(n => n.id === notifId ? { ...n, read: true } : n);
        await UserDatabase.saveNotifications(uid, updated);
    },

    markAllNotificationsRead: async (uid: string): Promise<void> => {
        const current = await UserDatabase.getNotifications(uid);
        const updated = current.map(n => ({ ...n, read: true }));
        await UserDatabase.saveNotifications(uid, updated);
    },

    // 4. ACTIVITY LOGS
    getActivityLogs: (uid: string): Promise<ActivityLog[]> => {
        return UserDatabase.get<ActivityLog[]>(uid, 'activity_logs', []);
    },

    logActivity: async (uid: string, log: ActivityLog): Promise<void> => {
        const logs = await UserDatabase.getActivityLogs(uid);
        // Keep last 100 logs
        const newLogs = [log, ...logs].slice(0, 100);
        await UserDatabase.save(uid, 'activity_logs', newLogs);
    },

    // 5. PREFERENCES & CONFIG
    getPreferences: (uid: string): Promise<UserPreferences> => {
        return UserDatabase.get<UserPreferences>(uid, 'preferences', {
            theme: 'dark',
            trashConfig: { enabled: true, frequency: 'hourly' },
            dismissedTutorial: false,
            tutorialProgress: [],
            dismissedBroadcasts: []
        });
    },

    updatePreferences: async (uid: string, updates: Partial<UserPreferences>): Promise<void> => {
        const current = await UserDatabase.getPreferences(uid);
        await UserDatabase.save(uid, 'preferences', { ...current, ...updates });
    },

    // 6. TRASH CONFIG (Specific Helper)
    getTrashConfig: async (uid: string): Promise<TrashConfig> => {
        const prefs = await UserDatabase.getPreferences(uid);
        return prefs.trashConfig;
    },

    setTrashConfig: async (uid: string, config: TrashConfig): Promise<void> => {
        await UserDatabase.updatePreferences(uid, { trashConfig: config });
    },

    // 7. CLEAR NOTIFICATIONS
    clearNotifications: async (uid: string): Promise<void> => {
        // 1. Get current notifications to identify broadcasts and messages
        const current = await UserDatabase.getNotifications(uid);
        const broadcastIds = current
            .filter(n => n.id.startsWith('broadcast_'))
            .map(n => n.id.replace('broadcast_', ''));

        const messageIds = current
            .filter(n => n.id.startsWith('message_'))
            .map(n => n.id.replace('message_', ''));

        // 2. Update dismissed broadcasts and messages in preferences
        if (broadcastIds.length > 0 || messageIds.length > 0) {
            const prefs = await UserDatabase.getPreferences(uid);
            const newDismissedBroadcasts = [...new Set([...(prefs.dismissedBroadcasts || []), ...broadcastIds])];
            const newDismissedMessages = [...new Set([...(prefs.dismissedMessages || []), ...messageIds])];
            await UserDatabase.updatePreferences(uid, {
                dismissedBroadcasts: newDismissedBroadcasts,
                dismissedMessages: newDismissedMessages
            });
        }

        // 3. Clear notifications store
        await UserDatabase.saveNotifications(uid, []);
    },

    // --- Maintenance ---

    // Clear all data for a specific user (Account Deletion)
    deleteUserDatabase: async (uid: string): Promise<void> => {
        const db = await getDB();
        const tx = db.transaction(STORES, 'readwrite');

        // Fire and forget deletions for each store
        STORES.forEach(store => {
            try {
                tx.objectStore(store).delete(uid);
            } catch (e) {
                console.error(`Failed to delete ${store} for ${uid}`, e);
            }
        });

        return new Promise<void>((resolve, reject) => {
            tx.oncomplete = () => resolve();
            tx.onerror = () => resolve(); // Resolve anyway
        });
    }
};