import express from 'express';
import cors from 'cors';
import dotenv from 'dotenv';
import multer from 'multer';
import path from 'path';
import fs from 'fs';
import { fileURLToPath } from 'url';
import bcrypt from 'bcrypt';
import jwt from 'jsonwebtoken';
import { randomUUID, createHash } from 'crypto';
import db, { checkConnection } from './database/db.js';
import nodemailer from 'nodemailer';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

dotenv.config();

const app = express();
app.use(cors());
app.use(express.json({ limit: '50mb' }));

// --- Configuration ---
const JWT_SECRET = process.env.JWT_SECRET || 'fallback_secret_dev_only_change_in_prod';
const UPLOAD_DIR = 'uploads/';

// Pacific Time Utilities
const getPacificDate = () => {
    const now = new Date();
    return new Date(now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' }));
};

const getPacificYMD = () => {
    const ptDate = getPacificDate();
    const year = ptDate.getFullYear();
    const month = String(ptDate.getMonth() + 1).padStart(2, '0');
    const day = String(ptDate.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
};

const getNextPTMidnight = () => {
    const ptDate = getPacificDate();
    const nextMidnight = new Date(ptDate);
    nextMidnight.setDate(nextMidnight.getDate() + 1);
    nextMidnight.setHours(0, 0, 0, 0);
    // Convert back to user's local time (server time) timestamp
    // Ideally we return the timestamp that represents PT midnight
    // Since we want the timestamp, we need to be careful.
    // The timestamp of 'nextMidnight' object is created from ptDate constructor which might be tricky with offsets.
    // Safer to use the date string construction.
    return nextMidnight;
};

// --- Database Init ---
checkConnection();

// --- Middleware ---

// File Upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, UPLOAD_DIR)
    },
    filename: function (req, file, cb) {
        const safeName = file.originalname.replace(/[^a-z0-9.]/gi, '_').toLowerCase();
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, uniqueSuffix + '-' + safeName);
    }
});
const upload = multer({ storage: storage });
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Upload Endpoint
app.post('/api/upload', upload.single('file'), (req, res) => {
    if (!req.file) return res.status(400).json({ error: 'No file uploaded' });

    // Construct full URL since frontend is on different port (5173 vs 3001)
    const protocol = req.protocol;
    const host = req.get('host');
    const fullUrl = `${protocol}://${host}/uploads/${req.file.filename}`;

    res.json({ url: fullUrl }); // Frontend expects { url: string }
});

// Auth Middleware
const requireAuth = (req, res, next) => {
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
        console.log('[AUTH] Missing or invalid header:', authHeader);
        return res.status(401).json({ error: 'Unauthorized' });
    }
    const token = authHeader.split('Bearer ')[1];
    try {
        const decoded = jwt.verify(token, JWT_SECRET);
        req.user = decoded;
        next();
    } catch (error) {
        return res.status(401).json({ error: 'Invalid token' });
    }
};

const requireAdmin = async (req, res, next) => {
    if (!req.user) return res.status(401).json({ error: 'Unauthorized' });

    try {
        const [rows] = await db.query('SELECT role FROM users WHERE uid = ?', [req.user.uid]);
        if (rows.length === 0 || rows[0].role !== 'admin') {
            return res.status(403).json({ error: 'Forbidden: Admins only' });
        }
        next();
    } catch (error) {
        res.status(500).json({ error: 'Database check failed' });
    }
};

// --- GENERIC ADMIN CRUD ENDPOINTS ---
// GET /api/admin/data/:table - Fetch all rows from admin table
app.get('/api/admin/data/:table', requireAuth, requireAdmin, async (req, res) => {
    try {
        const { table } = req.params;
        const allowedTables = ['tips', 'tutorial_steps', 'broadcasts', 'bug_reports', 'audit_logs', 'users', 'support_messages', 'system_settings', 'stock_sites'];
        if (!allowedTables.includes(table)) {
            return res.status(400).json({ error: 'Invalid table' });
        }
        const [rows] = await db.query(`SELECT * FROM ${table}`);
        res.json({ data: rows });
    } catch (error) {
        console.error('[ADMIN GET] Error:', error);
        res.status(500).json({ error: 'Failed to fetch data' });
    }
});

// POST /api/admin/data/:table - Create new item in admin table
app.post('/api/admin/data/:table', requireAuth, requireAdmin, async (req, res) => {
    try {
        const { table } = req.params;
        const { item } = req.body;
        const allowedTables = ['tips', 'tutorial_steps', 'broadcasts', 'bug_reports', 'audit_logs', 'support_messages', 'system_settings', 'stock_sites'];
        if (!allowedTables.includes(table)) {
            return res.status(400).json({ error: 'Invalid table' });
        }
        const keys = Object.keys(item);
        const values = Object.values(item);
        const placeholders = keys.map(() => '?').join(',');
        const colNames = keys.join(',');
        await db.query(`INSERT INTO ${table} (${colNames}) VALUES (${placeholders})`, values);
        res.json({ success: true });
    } catch (error) {
        console.error('[ADMIN CREATE] Error:', error);
        res.status(500).json({ error: 'Failed to create item' });
    }
});

// POST /api/admin/data/:table/update - Update existing item in admin table
app.post('/api/admin/data/:table/update', requireAuth, requireAdmin, async (req, res) => {
    try {
        const { table } = req.params;
        const { id, updates } = req.body;
        const allowedTables = ['tips', 'tutorial_steps', 'broadcasts', 'bug_reports', 'users', 'support_messages', 'system_settings', 'stock_sites'];
        if (!allowedTables.includes(table)) {
            return res.status(400).json({ error: 'Invalid table' });
        }
        const setClause = Object.keys(updates).map(k => `${k} = ?`).join(', ');
        const values = [...Object.values(updates), id];

        let query = `UPDATE ${table} SET ${setClause} WHERE id = ?`;
        if (table === 'users') {
            query = `UPDATE ${table} SET ${setClause} WHERE uid = ?`;
        } else if (table === 'api_keys') { // Future proofing if needed
            query = `UPDATE ${table} SET ${setClause} WHERE id = ?`;
        }

        await db.query(query, values);
        res.json({ success: true });
    } catch (error) {
        console.error('[ADMIN UPDATE] Error:', error);
        res.status(500).json({ error: 'Failed to update item' });
    }
});

// POST /api/admin/data/:table/delete - Permanently delete item from admin table
app.post('/api/admin/data/:table/delete', requireAuth, requireAdmin, async (req, res) => {
    try {
        const { table } = req.params;
        const { id } = req.body;
        const allowedTables = ['tips', 'tutorial_steps', 'broadcasts', 'bug_reports', 'support_messages', 'users', 'stock_sites'];
        if (!allowedTables.includes(table)) {
            return res.status(400).json({ error: 'Invalid table' });
        }

        let query = `DELETE FROM ${table} WHERE id = ?`;
        if (table === 'users') {
            query = `DELETE FROM ${table} WHERE uid = ?`;
        }

        await db.query(query, [id]);
        res.json({ success: true });
    } catch (error) {
        console.error('[ADMIN DELETE] Error:', error);
        res.status(500).json({ error: 'Failed to delete item' });
    }
});

// --- Helper Functions ---

// Helper: Reset all number fields in an object to 0
const resetStatsObj = (obj) => {
    Object.keys(obj).forEach(k => {
        if (typeof obj[k] === 'number') obj[k] = 0;
    });
};

// Helper: Check IP Ban
const checkIpBan = async (ip) => {
    try {
        if (!ip) return false;
        const cleanIp = ip.split(',')[0].trim();
        const [rows] = await db.query('SELECT id FROM banned_ips WHERE ip_address = ?', [cleanIp]);
        return rows.length > 0;
    } catch (e) {
        // console.error("Check IP Ban Failed (Table might be missing):", e.message);
        return false;
    }
};

// Helper: Sync User Usage Stats (Daily Reset & Key Check)
const syncUserUsageStats = async (uid) => {
    try {
        const [uRows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        if (uRows.length === 0) return null;
        let stats = typeof uRows[0].usage_stats === 'string' ? JSON.parse(uRows[0].usage_stats) : uRows[0].usage_stats;
        if (!stats) stats = {};

        // Initialize structure if missing
        const pacificToday = getPacificYMD();
        if (!stats.daily) stats.daily = { date: pacificToday, apiCalls: 0 };
        if (!stats.daily.date) stats.daily.date = pacificToday;

        // Get Active Keys
        const [kRows] = await db.query('SELECT key_value FROM api_keys WHERE user_uid = ? AND is_enabled = 1', [uid]);

        let monthlyCalculatedUsage = 0;
        let dailyCalculatedUsage = 0;

        if (kRows.length > 0) {
            const hashes = kRows.map(k => createHash('sha256').update(k.key_value).digest('hex'));
            const placeholders = hashes.map(() => '?').join(',');
            const monthStart = pacificToday.substring(0, 7) + '-01'; // YYYY-MM-01

            // Monthly Total
            const [monthlyRows] = await db.query(
                `SELECT SUM(api_calls) as total FROM api_key_usage WHERE usage_date >= ? AND key_hash IN (${placeholders})`,
                [monthStart, ...hashes]
            );
            monthlyCalculatedUsage = Number(monthlyRows[0].total) || 0;

            // Daily Total (Today Only)
            const [dailyRows] = await db.query(
                `SELECT SUM(api_calls) as total FROM api_key_usage WHERE usage_date = ? AND key_hash IN (${placeholders})`,
                [pacificToday, ...hashes]
            );
            dailyCalculatedUsage = Number(dailyRows[0].total) || 0;
        }

        // --- NEW: Sync Asset Stats (Trend, Images, Videos) for TODAY & THIS MONTH ---
        const pacificOffsetMs = getPacificDate().getTime() - new Date().getTime();
        const pacificOffsetSeconds = Math.floor(pacificOffsetMs / 1000);
        const monthStart = pacificToday.substring(0, 7) + '-01'; // YYYY-MM-01

        const [assetStats] = await db.query(`
            SELECT 
                type,
                COUNT(CASE WHEN CAST(DATE(FROM_UNIXTIME((created_at/1000) + ?)) AS CHAR) = ? THEN 1 END) as today_count,
                COUNT(CASE WHEN CAST(DATE(FROM_UNIXTIME((created_at/1000) + ?)) AS CHAR) >= ? THEN 1 END) as monthly_count
            FROM assets 
            WHERE user_uid = ?
            GROUP BY type
        `, [pacificOffsetSeconds, pacificToday, pacificOffsetSeconds, monthStart, uid]);

        let trendsToday = 0, imagesToday = 0, videosToday = 0, reviewsToday = 0, metadataToday = 0, editsToday = 0, promptsToday = 0;
        let trendsMonth = 0, imagesMonth = 0, videosMonth = 0, reviewsMonth = 0, metadataMonth = 0, editsMonth = 0, promptsMonth = 0;

        assetStats.forEach(r => {
            const today = parseInt(r.today_count) || 0;
            const month = parseInt(r.monthly_count) || 0;
            if (r.type === 'report') { trendsToday = today; trendsMonth = month; }
            else if (r.type === 'image') { imagesToday = today; imagesMonth = month; }
            else if (r.type === 'video') { videosToday = today; videosMonth = month; }
            else if (r.type === 'review') { reviewsToday = today; reviewsMonth = month; }
            else if (r.type === 'metadata') { metadataToday = today; metadataMonth = month; }
            else if (r.type === 'edit') { editsToday = today; editsMonth = month; }
            else if (r.type === 'prompt') { promptsToday = today; promptsMonth = month; }
        });

        // Detect if update is needed
        let updated = false;

        // 1. Sync Monthly Stat (apiCallsThisMonth)
        if (stats.apiCallsThisMonth !== monthlyCalculatedUsage) {
            stats.apiCallsThisMonth = monthlyCalculatedUsage;
            updated = true;
        }

        // --- NEW: Sync Model Usage (Per Model Stats) ---
        // Recalculate model usage based on ACTIVE keys only
        // This ensures that if a key is deleted, its usage no longer counts towards the displayed limits
        if (kRows.length > 0) {
            const hashes = kRows.map(k => createHash('sha256').update(k.key_value).digest('hex'));
            const placeholders = hashes.map(() => '?').join(',');
            const monthStart = pacificToday.substring(0, 7) + '-01'; // YYYY-MM-01

            const [modelRows] = await db.query(
                `SELECT model_id, SUM(api_calls) as total FROM api_key_usage WHERE usage_date >= ? AND key_hash IN (${placeholders}) GROUP BY model_id`,
                [monthStart, ...hashes]
            );

            // Rebuild modelUsage object from scratch based on DB truth
            // We use a temporary object to compare changes
            const validModelUsage = {};
            modelRows.forEach(row => {
                if (row.model_id && row.model_id !== 'unknown') {
                    validModelUsage[row.model_id] = Number(row.total);
                }
            });

            // Compare and update if different
            // Simple stringify comparison for deep object equality check
            if (JSON.stringify(stats.modelUsage) !== JSON.stringify(validModelUsage)) {
                stats.modelUsage = validModelUsage;
                updated = true;
            }
        } else {
            // No active keys? Clear model usage (unless we want to keep it? User wants old stats gone though)
            if (Object.keys(stats.modelUsage || {}).length > 0) {
                stats.modelUsage = {};
                updated = true;
            }
        }

        // 2. Sync Daily Stat (daily.apiCalls)
        // Ensure date matches first
        if (stats.daily.date !== pacificToday) {
            // It's a new day, reset daily stats (but keep monthly)
            stats.history.lastDay = { ...stats.daily };
            resetStatsObj(stats.daily);
            stats.daily.date = pacificToday;
            stats.daily.apiCalls = dailyCalculatedUsage;

            // RESET MODEL USAGE DAILY
            stats.modelUsage = {};

            updated = true;
        } else {
            // Same day, just sync the number
            if (stats.daily.apiCalls !== dailyCalculatedUsage) {
                stats.daily.apiCalls = dailyCalculatedUsage;
                updated = true;
            }
        }

        // 3. Sync Asset Stats to Daily Object (Overwrite or Set)
        if (stats.daily.trendSearches !== trendsToday) { stats.daily.trendSearches = trendsToday; updated = true; }
        if (stats.daily.imagesGenerated !== imagesToday) { stats.daily.imagesGenerated = imagesToday; updated = true; }
        if (stats.daily.videosCreated !== videosToday) { stats.daily.videosCreated = videosToday; updated = true; }
        if (stats.daily.imagesReviewed !== reviewsToday) { stats.daily.imagesReviewed = reviewsToday; updated = true; }
        if (stats.daily.metadataGenerated !== metadataToday) { stats.daily.metadataGenerated = metadataToday; updated = true; }
        if (stats.daily.imagesEdited !== editsToday) { stats.daily.imagesEdited = editsToday; updated = true; }
        if (stats.daily.promptsGenerated !== promptsToday) { stats.daily.promptsGenerated = promptsToday; updated = true; }

        // 4. Sync Asset Stats to Monthly Object
        if (!stats.monthly) stats.monthly = { month: pacificToday.substring(0, 7) };
        if (stats.monthly.month !== pacificToday.substring(0, 7)) {
            // New month, reset logic could go here but we overwrite anyway
            stats.monthly = { month: pacificToday.substring(0, 7) };
        }

        if (stats.monthly.trendSearches !== trendsMonth) { stats.monthly.trendSearches = trendsMonth; updated = true; }
        if (stats.monthly.imagesGenerated !== imagesMonth) { stats.monthly.imagesGenerated = imagesMonth; updated = true; }
        if (stats.monthly.videosCreated !== videosMonth) { stats.monthly.videosCreated = videosMonth; updated = true; }
        if (stats.monthly.imagesReviewed !== reviewsMonth) { stats.monthly.imagesReviewed = reviewsMonth; updated = true; }
        if (stats.monthly.metadataGenerated !== metadataMonth) { stats.monthly.metadataGenerated = metadataMonth; updated = true; }
        if (stats.monthly.imagesEdited !== editsMonth) { stats.monthly.imagesEdited = editsMonth; updated = true; }
        if (stats.monthly.promptsGenerated !== promptsMonth) { stats.monthly.promptsGenerated = promptsMonth; updated = true; }

        if (updated) {
            await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), uid]);
        }
        return stats;
    } catch (e) {
        console.error("Sync Usage Stats Failed:", e);
        return null; // Fail safe
    }
};

// Helper: Map DB User to Frontend (CamelCase)
const toUserResponse = (row) => {
    if (!row) return null;
    const stats = typeof row.usage_stats === 'string' ? JSON.parse(row.usage_stats) : row.usage_stats;
    const prefs = typeof row.preferences === 'string' ? JSON.parse(row.preferences) : row.preferences;

    return {
        uid: row.uid,
        email: row.email,
        displayName: row.display_name || '',
        photoURL: row.photo_url || null,
        role: row.role,
        isVerified: !!row.is_verified,
        isDeleted: !!row.is_deleted,
        isBanned: !!row.is_banned,
        createdAt: Number(row.created_at),
        lastLoginAt: Number(row.last_login_at),
        subscription: row.subscription || 'free',
        country: row.country || 'Unknown',
        ipAddress: row.ip_address || 'Unknown',
        usageStats: stats || {},
        preferences: prefs || {}
    };
};

// Helper: Get Geo Info from IP
const getGeoInfo = async (ip) => {
    try {
        // Handle localhost or empty
        if (!ip || ip.includes('127.0.0.1') || ip.includes('::1')) {
            // User requested that localhost maps to Slovenia country, but IP should still be shown as 127.0.0.1
            return { country: 'Slovenia', city: 'Slovenia' };
        }

        // Clean IP (x-forwarded-for can contain multiple)
        const cleanIp = ip.split(',')[0].trim();

        const response = await fetch(`http://ip-api.com/json/${cleanIp}`);
        if (!response.ok) return { country: 'Unknown', city: 'Unknown' };

        const data = await response.json();
        if (data.status === 'fail') return { country: 'Unknown', city: 'Unknown' };

        return { country: data.country || 'Unknown', city: data.city || 'Unknown' };
    } catch (e) {
        console.warn('Geo lookup failed:', e.message);
        return { country: 'Unknown', city: 'Unknown' };
    }
};

// ... AUTH ROUTES ...

// Register
app.post('/api/auth/register', async (req, res) => {
    const { email, password, displayName, photoURL } = req.body;
    const ip = req.headers['x-forwarded-for'] || req.socket.remoteAddress;

    // Check IP Ban
    if (await checkIpBan(ip)) {
        return res.status(403).json({ error: 'Access denied. Your IP has been banned.' });
    }

    if (!email || !password) return res.status(400).json({ error: 'Email and password required' });

    try {
        const [existing] = await db.query('SELECT uid FROM users WHERE email = ?', [email]);
        if (existing.length > 0) return res.status(409).json({ error: 'Email already registered' });

        // GEO TRACKING
        const ip = req.headers['x-forwarded-for'] || req.socket.remoteAddress || '';
        const geo = await getGeoInfo(ip);
        console.log(`[REGISTER] New User IP: ${ip}, Location: ${geo.country}`);

        const uid = randomUUID();
        const hash = await bcrypt.hash(password, 10);
        const now = Date.now();
        const usageStats = { imagesGenerated: 0, videosCreated: 0, apiCallsThisMonth: 0, modelUsage: {}, modelLimits: {} };
        // Default Preferences for New Users (Auto-Download, Sound, Notifications OFF)
        const preferences = {
            theme: 'dark',
            generationConfig: { globalCooldown: 5, imageDelay: 0, videoDelay: 0, metadataDelay: 0 },
            uiConfig: { autoDownload: false, soundEffects: false, notifications: false, reducedMotion: false }
        };

        await db.query(
            `INSERT INTO users (uid, email, password_hash, display_name, photo_url, role, created_at, last_login_at, usage_stats, preferences, country, ip_address) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [uid, email, hash, displayName || 'User', photoURL || '', 'user', now, now, JSON.stringify(usageStats), JSON.stringify(preferences), geo.country, ip]
        );

        const token = jwt.sign({ uid, email, role: 'user' }, JWT_SECRET, { expiresIn: '7d' });

        // Return structured user
        const newUser = {
            uid, email, displayName, photoURL, role: 'user',
            isVerified: false, createdAt: now, lastLoginAt: now,
            usageStats, preferences, country: geo.country
        };

        res.json({ success: true, user: newUser, token });
    } catch (error) {
        console.error("Register Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// Login
app.post('/api/auth/login', async (req, res) => {
    try {
        const { email, password } = req.body;
        const ip = req.headers['x-forwarded-for'] || req.socket.remoteAddress;

        // Check IP Ban
        if (await checkIpBan(ip)) {
            return res.status(403).json({ error: 'Access denied. Your IP has been banned.' });
        }
        const [rows] = await db.query('SELECT * FROM users WHERE email = ?', [email]);
        if (rows.length === 0) return res.status(401).json({ error: 'Invalid credentials' });

        const userRow = rows[0];
        if (userRow.is_deleted) return res.status(403).json({ error: 'Account deleted' });
        if (userRow.is_banned) return res.status(403).json({ error: 'Account suspended' });

        const match = await bcrypt.compare(password, userRow.password_hash);
        if (!match) return res.status(401).json({ error: 'Invalid credentials' });

        // GEO TRACKING UPDATE
        // ip is already captured above

        // Update login time, IP, and potentially country if it was Unknown or changed (optional, but good)
        // We will just update IP and country every login to keep it fresh
        const geo = await getGeoInfo(ip);
        console.log(`[LOGIN] User: ${email}, IP: ${ip}, Country: ${geo.country}`);

        await db.query('UPDATE users SET last_login_at = ?, ip_address = ?, country = ? WHERE uid = ?', [Date.now(), ip, geo.country, userRow.uid]);

        // Update local object for response
        userRow.country = geo.country;

        const token = jwt.sign({ uid: userRow.uid, email: userRow.email, role: userRow.role }, JWT_SECRET, { expiresIn: '7d' });

        res.json({ success: true, user: toUserResponse(userRow), token });
    } catch (error) {
        console.error("Login Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// Get Current User (Me)
app.get('/api/auth/me', requireAuth, async (req, res) => {
    try {
        await syncUserUsageStats(req.user.uid);
        const [rows] = await db.query('SELECT * FROM users WHERE uid = ?', [req.user.uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });
        res.json({ user: toUserResponse(rows[0]) });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// --- USER ROUTES ---

app.get('/api/user/profile', async (req, res) => {
    const { uid } = req.query;
    try {
        await syncUserUsageStats(uid);
        const [rows] = await db.query('SELECT * FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });
        res.json({ user: toUserResponse(rows[0]) });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});


// Get User Preferences
app.get('/api/user/preferences', requireAuth, async (req, res) => {
    const { uid, section } = req.query;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403).json({ error: 'Forbidden' });

    try {
        const [rows] = await db.query('SELECT preferences FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });

        let prefs = typeof rows[0].preferences === 'string' ? JSON.parse(rows[0].preferences) : rows[0].preferences;
        if (!prefs) prefs = {};

        // Return specific section or all
        const data = section ? (prefs[section] || {}) : prefs;
        res.json({ data });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Update User Preferences
app.post('/api/user/preferences', requireAuth, async (req, res) => {
    const { uid, section, data } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403).json({ error: 'Forbidden' });

    try {
        const [rows] = await db.query('SELECT preferences FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });

        let prefs = typeof rows[0].preferences === 'string' ? JSON.parse(rows[0].preferences) : rows[0].preferences;
        if (!prefs) prefs = {};

        // Update specific section or merge all
        if (section) {
            prefs[section] = { ...prefs[section], ...data };
        } else {
            prefs = { ...prefs, ...data };
        }

        await db.query('UPDATE users SET preferences = ? WHERE uid = ?', [JSON.stringify(prefs), uid]);
        res.json({ success: true, preferences: prefs });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});


// Update Profile
app.post('/api/user/update', requireAuth, async (req, res) => {
    const { uid, data } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        console.log(`[UPDATE PROFILE] Request for UID: ${uid} (Auth User: ${req.user.uid})`);

        // --- START NEW: Profile Image Deletion Logic ---
        // 1. Fetch current user data to see old photo_url
        const [currentRows] = await db.query('SELECT photo_url FROM users WHERE uid = ?', [uid]);
        if (currentRows.length > 0) {
            const oldPhotoUrl = currentRows[0].photo_url;
            const newPhotoUrl = data.photoURL;

            // Check if photo is changing (new URL provided and different from old)
            // Note: If data.photoURL is explicitly set (even to empty string), we might need to delete old.
            // Adjust condition based on how frontend sends data. 
            // If data.photoURL is undefined, it means "don't change". 
            // If it is string (empty or new url), it means "change to this".

            if (data.photoURL !== undefined && oldPhotoUrl && oldPhotoUrl !== newPhotoUrl) {
                // Check if old photo was a local upload
                if (oldPhotoUrl.includes('/uploads/')) {
                    try {
                        // Extract filename. Format is usually "http://host/uploads/filename" 
                        // or just "/uploads/filename" depending on how it's stored.
                        // We safely grab the part after "uploads/"
                        const filename = oldPhotoUrl.split('/uploads/').pop();
                        if (filename) {
                            const filePath = path.join(__dirname, 'uploads', filename);
                            if (fs.existsSync(filePath)) {
                                fs.unlinkSync(filePath);
                                console.log(`[UPDATE PROFILE] Deleted old image: ${filePath}`);
                            } else {
                                console.log(`[UPDATE PROFILE] Old image file not found: ${filePath}`);
                            }
                        }
                    } catch (err) {
                        console.error(`[UPDATE PROFILE] Failed to delete old image:`, err);
                        // We do NOT block the update if deletion fails, just log it.
                    }
                }
            }
        }
        // --- END NEW ---

        console.log(`[UPDATE PROFILE] Data:`, JSON.stringify(data));

        const fields = [];
        const values = [];

        if (data.displayName !== undefined) { fields.push('display_name = ?'); values.push(data.displayName); }
        if (data.photoURL !== undefined) { fields.push('photo_url = ?'); values.push(data.photoURL); }
        if (data.country !== undefined) { fields.push('country = ?'); values.push(data.country); }
        if (data.isVerified !== undefined) { fields.push('is_verified = ?'); values.push(data.isVerified ? 1 : 0); }

        if (fields.length > 0) {
            values.push(uid);
            await db.query(`UPDATE users SET ${fields.join(', ')} WHERE uid = ?`, values);
        }

        const [rows] = await db.query('SELECT * FROM users WHERE uid = ?', [uid]);
        res.json({ success: true, user: toUserResponse(rows[0]) });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/user/change-password', requireAuth, async (req, res) => {
    const { uid, currentPassword, newPassword } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        const [rows] = await db.query('SELECT password_hash FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });

        const match = await bcrypt.compare(currentPassword, rows[0].password_hash);
        if (!match) return res.status(400).json({ error: 'Incorrect current password' });

        const hash = await bcrypt.hash(newPassword, 10);
        await db.query('UPDATE users SET password_hash = ? WHERE uid = ?', [hash, uid]);

        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/user/delete', requireAuth, async (req, res) => {
    const { uid } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        await db.query('UPDATE users SET is_deleted = 1 WHERE uid = ?', [uid]);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// GET /api/stock-sites - Public/User endpoint for fetching visible sites
app.get('/api/stock-sites', requireAuth, async (req, res) => {
    console.log(`[API] GET /api/stock-sites called by user: ${req.user.uid} (${req.user.email})`);
    try {
        const [rows] = await db.query('SELECT * FROM stock_sites WHERE (is_visible = 1 OR is_visible_market = 1) AND is_deleted = 0');
        console.log(`[API] stock-sites found ${rows.length} visible sites`);
        res.json({ data: rows });
    } catch (error) {
        console.error('Failed to fetch stock sites:', error);
        res.status(500).json({ error: 'Failed to fetch stock sites' });
    }
});

// Bulk Update Stock Sites Visibility
app.post('/api/admin/stock-sites/bulk-visibility', requireAuth, requireAdmin, async (req, res) => {
    const { field, value } = req.body;
    if (!['is_visible', 'is_visible_market'].includes(field)) {
        return res.status(400).json({ error: 'Invalid field' });
    }

    try {
        await db.query(`UPDATE stock_sites SET ${field} = ? WHERE is_deleted = 0`, [value ? 1 : 0]);
        console.log(`[ADMIN] Bulk updated stock sites ${field} to ${value}`);
        res.json({ success: true });
    } catch (error) {
        console.error('Bulk update failed:', error);
        res.status(500).json({ error: error.message });
    }
});

// Reset User Stats
app.post('/api/user/reset-stats', requireAuth, async (req, res) => {
    const { uid } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        // 1. Fetch current stats to preserve limits/keys info
        const [uRows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        let currentStats = {};
        if (uRows.length > 0) {
            currentStats = typeof uRows[0].usage_stats === 'string'
                ? JSON.parse(uRows[0].usage_stats)
                : uRows[0].usage_stats || {};
        }

        // 2. Construct Reset Stats (Preserve API Credits/Limits)
        const resetStats = {
            ...currentStats,
            // Reset Content Counters
            imagesGenerated: 0,
            videosCreated: 0,
            trendSearches: 0,
            imagesReviewed: 0,
            metadataGenerated: 0,
            imagesEdited: 0,
            promptsGenerated: 0,

            // Clear Periodicals (but syncUserUsageStats will restore API calls)
            daily: {
                date: getPacificYMD(),
                apiCalls: currentStats.daily?.apiCalls || 0, // Preserve for safety, sync will validate
                // Clear daily content counters
            },
            monthly: {
                month: currentStats.monthly?.month || new Date().toISOString().slice(0, 7),
                apiCalls: currentStats.monthly?.apiCalls || 0, // Preserve
                // Clear monthly content counters
            },

            // Clear History/Charts
            history: {}
        };

        // Note: We intentionally do NOT clear apiCallsThisMonth or modelUsage here.
        // Even if we did, syncUserUsageStats would restore them from api_key_usage.
        // We preserve them to prevent UI flicker.

        const defaultPreferences = {
            theme: 'dark',
            generationConfig: { globalCooldown: 5, imageDelay: 0, videoDelay: 0, metadataDelay: 0 },
            uiConfig: { autoDownload: false, soundEffects: false, notifications: false, reducedMotion: false }
        };

        // Reset Stats AND Preferences
        await db.query('UPDATE users SET usage_stats = ?, preferences = ? WHERE uid = ?', [
            JSON.stringify(resetStats),
            JSON.stringify(defaultPreferences),
            uid
        ]);




        // 3. Clear Historical Data (Content Only)
        // REMOVED: await db.query('DELETE FROM api_key_usage WHERE user_uid = ?', [uid]); // Preserve Billing History!

        // Clear assets history (images, videos, reports, reviews)
        await db.query('DELETE FROM assets WHERE user_uid = ?', [uid]);
        console.log('[RESET STATS] Cleared assets history for user:', uid);

        // Clear activity logs
        await db.query('DELETE FROM activity_logs WHERE user_uid = ?', [uid]);
        console.log('[RESET STATS] Cleared activity logs for user:', uid);

        // 4. Force Sync to ensure everything is consistent
        await syncUserUsageStats(uid);

        // Return structured user with fresh stats
        const [rows] = await db.query('SELECT * FROM users WHERE uid = ?', [uid]);
        res.json({ success: true, user: toUserResponse(rows[0]) });
    } catch (error) {
        console.error("Reset Stats Error:", error);
        res.status(500).json({ error: error.message });
    }
});



function getSiteSettings() {
    return db.query('SELECT * FROM system_settings WHERE id = 1').then(([rows]) => {
        if (rows.length === 0) return {};
        let settings = rows[0].settings;
        if (typeof settings === 'string') settings = JSON.parse(settings);

        // Ensure modelIds exists and populate any missing defaults
        const defaultModelIds = {
            // Thinking
            'flash_lite': 'gemini-2.5-flash-lite-preview-09-2025',
            'flash': 'gemini-2.5-flash',
            'flash_3_0': 'gemini-3-flash-preview',
            'pro': 'gemini-3-pro-preview',
            'pro_2_5': 'gemini-2.5-pro',
            'gemini_1_5_pro': 'gemini-1.5-pro-latest',

            // Creative
            'imagen_fast': 'imagen-4.0-fast-generate-preview',
            'imagen_std': 'imagen-4.0-generate-001',
            'imagen_ultra': 'imagen-4.0-generate-ultra-preview',
            'gemini_2_0_flash_image': 'gemini-2.0-flash-image',
            'gemini_2_5_flash_image': 'gemini-2.5-flash-image',
            'gemini_3_pro_image': 'gemini-3-pro-image-preview',

            // Video
            'veo_3_1': 'veo-3.1-generate-preview',
            'veo_3_1_fast': 'veo-3.1-fast-generate-preview',
            'veo_3_0': 'veo-3.0-generate-preview',
            'veo_2_0': 'veo-2.0-generate-preview'
        };


        // Merge defaults with existing (existing values take precedence)
        console.log('🔧 Server: Before merge - existing modelIds keys:', Object.keys(settings.modelIds || {}));
        settings.modelIds = { ...defaultModelIds, ...(settings.modelIds || {}) };
        console.log('🔧 Server: After merge - total modelIds keys:', Object.keys(settings.modelIds).length);

        // Add branding fields with defaults
        settings.appIcon = settings.appIcon || '';
        settings.appLogo = settings.appLogo || '';
        settings.adobeStockInstruction = settings.adobeStockInstruction || '';

        return settings;
    });
}

// GET /api/user/keys - Fetch API keys with usage stats
app.get('/api/user/keys', requireAuth, async (req, res) => {
    console.log('[GET /api/user/keys] ====== REQUEST RECEIVED ======');
    console.log('[GET /api/user/keys] Query params:', req.query);
    console.log('[GET /api/user/keys] User from token:', req.user);
    const { uid } = req.query;

    if (!uid || (req.user.uid !== uid && req.user.role !== 'admin')) {
        console.log('[GET /api/user/keys] ❌ UNAUTHORIZED');
        return res.status(403).json({ error: 'Unauthorized' });
    }

    try {
        console.log('[GET /api/user/keys] Fetching keys for uid:', uid);
        // Fetch all API keys for the user
        const [keys] = await db.query(
            'SELECT id, key_value, name, status, tier, is_enabled, created_at FROM api_keys WHERE user_uid = ? AND is_deleted = 0',
            [uid]
        );

        // Get today's date in Pacific Time for usage calculation
        const today = getPacificYMD();

        // Calculate usage for each key
        const keysWithUsage = await Promise.all(keys.map(async (key) => {
            const keyHash = createHash('sha256').update(key.key_value).digest('hex');

            // Sum all API calls for this key (TODAY ONLY)
            const [usageRows] = await db.query(
                'SELECT SUM(api_calls) as total FROM api_key_usage WHERE key_hash = ? AND usage_date = ?',
                [keyHash, today]
            );

            const usage = Number(usageRows[0]?.total) || 0;

            return {
                id: key.id,
                key: `AIza••••••••••••••••••••••••${key.key_value.slice(-4)}`, // Masked
                fullKey: key.key_value, // Full key for validation
                status: key.status || 'unchecked',
                tier: key.tier || 'FREE',
                isEnabled: key.is_enabled === 1,
                usage: usage // All-time total usage for this key
            };
        }));

        console.log('[GET /api/user/keys] ✅ Returning keys:', JSON.stringify(keysWithUsage, null, 2));
        res.json({ keys: keysWithUsage });
    } catch (error) {
        console.error('[GET /api/user/keys] Error:', error);
        res.status(500).json({ error: 'Failed to fetch API keys' });
    }
});

app.post('/api/user/keys', requireAuth, async (req, res) => {
    const { uid, keys } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        // 1. Get all existing keys for user (Fetch key_value for usage continuity)
        let existingIds = [];
        let existingKeyMap = {}; // Map ID -> Real Key Value
        try {
            const [existing] = await db.query('SELECT id, key_value FROM api_keys WHERE user_uid = ? AND is_deleted = 0', [uid]);
            existingIds = existing.map(r => r.id);
            existing.forEach(r => { existingKeyMap[r.id] = r.key_value; });
        } catch (err) {
            console.error("Step 1 (Get existing) failed:", err);
            throw err;
        }

        // 2. Identify keys to delete (in DB but not in new list)
        // Check if input keys is undefined or not array just in case
        const safeKeys = Array.isArray(keys) ? keys : [];
        const incomingIds = safeKeys.map(k => k.id);
        const toDeleteIds = existingIds.filter(id => !incomingIds.includes(id));

        // 3. Delete removed keys
        if (toDeleteIds.length > 0) {
            try {
                for (const deadId of toDeleteIds) {
                    await db.query('UPDATE api_keys SET is_deleted = 1, is_enabled = 0 WHERE id = ?', [deadId]);
                }
            } catch (err) {
                console.error("Step 3 (Delete) failed:", err);
                throw err;
            }
        }

        // 4. Upsert new/updated keys
        try {
            for (const k of safeKeys) {
                const exists = existingIds.includes(k.id);

                // CRITICAL FIX: Use fullKey if available, otherwise fallback to key
                let valueToSave = k.fullKey || k.key;

                // If valueToSave is masked, do NOT update the DB with it.
                // We use the existing DB value if available.
                if (valueToSave && valueToSave.includes('••••')) {
                    if (exists && existingKeyMap[k.id]) {
                        // Keep existing value, do not overwrite with mask
                        valueToSave = existingKeyMap[k.id];
                    } else {
                        // New key but masked? Should not happen.
                        console.warn(`[Server] Skipping save of masked key ${k.id} (New but masked)`);
                        continue;
                    }
                }

                // Safe values
                const name = k.name || null;
                const status = k.status || 'unchecked';
                const isEnabled = k.isEnabled !== false ? 1 : 0;
                const network = k.network || 'google';
                const tier = k.tier || 'unknown'; // Preserve tier if provided
                const createdAt = k.createdAt || k.addedAt || Date.now();

                if (valueToSave && typeof valueToSave === 'string') {
                    valueToSave = valueToSave.trim();
                }

                if (exists) {
                    await db.query(
                        'UPDATE api_keys SET key_value = ?, name = ?, status = ?, tier = ?, is_enabled = ? WHERE id = ?',
                        [valueToSave, name, status, tier, isEnabled, k.id]
                    );
                } else {
                    await db.query(
                        'INSERT INTO api_keys (id, user_uid, key_value, name, network, tier, status, created_at, is_enabled) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
                        [k.id, uid, valueToSave, name, network, tier, status, createdAt, isEnabled]
                    );
                }

                // Update map for usage calc below
                existingKeyMap[k.id] = valueToSave;
            }
        } catch (err) {
            console.error("Step 4 (Upsert) failed:", err);
            throw err;
        }

        // Recalculate Limits Logic
        try {
            const settings = await getSiteSettings();

            const validKeys = safeKeys.filter(k => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
            const freeKeys = validKeys.filter(k => k.tier !== 'PAID');
            const paidKeys = validKeys.filter(k => k.tier === 'PAID');

            // Model Mapping
            const MODEL_LIMIT_MAPPING = {
                // Thinking
                'flash_lite': 'gemini-2.5-flash-lite-preview-09-2025',
                'flash': 'gemini-2.5-flash',
                'flash_3_0': 'gemini-3-flash-preview',
                'pro': 'gemini-3-pro-preview',
                'pro_2_5': 'gemini-2.5-pro',

                // Aliases (Thinking)
                'flash_lite_2_5': 'gemini-2.5-flash-lite-preview-09-2025',
                'flash_2_5': 'gemini-2.5-flash',
                'gemini_3_0_flash': 'gemini-3-flash-preview',
                'pro_3_0': 'gemini-3-pro-preview',

                // Creative
                'imagen_4_fast': 'imagen-4.0-fast-generate-preview',
                'imagen_4_std': 'imagen-4.0-generate-001',
                'imagen_4_ultra': 'imagen-4.0-generate-ultra-preview',
                'gemini_2_0_flash_image': 'gemini-2.0-flash-image',
                'gemini_2_5_flash_image': 'gemini-2.5-flash-image',
                'gemini_3_pro_image': 'gemini-3-pro-image-preview',

                // Aliases (Creative)
                'imagen_fast': 'imagen-4.0-fast-generate-preview',
                'imagen_std': 'imagen-4.0-generate-001',
                'imagen_ultra': 'imagen-4.0-generate-ultra-preview',
                'flash_image_2_0': 'gemini-2.0-flash-image',
                'flash_image_2_5': 'gemini-2.5-flash-image',
                'pro_image_3_0': 'gemini-3-pro-image-preview',

                // Video
                'veo_3_1_fast': 'veo-3.1-fast-generate-preview',
                'veo_3_1': 'veo-3.1-generate-preview',
                'veo_3_0': 'veo-3.0-generate-preview',
                'veo_2_0': 'veo-2.0-generate-preview',

                // Aliases (Video)
                'veo': 'veo-3.1-fast-generate-preview',
                'veo_hq': 'veo-3.1-generate-preview'
            };

            // Calculate totals and model limits
            let totalFreeCredits = 0;
            let totalPaidCredits = 0;
            const newModelLimits = {};

            // DEFAULT TIER LIMITS (Static Hardcoded)
            // DEFAULT TIER LIMITS (Static Hardcoded)
            const DEFAULT_TIER_LIMITS = {
                'pro_3_0': { free: 0, paid: 1000 },
                'pro_2_5': { free: 0, paid: 1000 },
                'flash_2_5': { free: 20, paid: 1000 },
                'flash_lite_2_5': { free: 20, paid: 1000 },
                'gemini_3_0_flash': { free: 20, paid: 1000 },
                'imagen_ultra': { free: 10, paid: 500 },
                'imagen_fast': { free: 10, paid: 500 },
                'imagen_std': { free: 10, paid: 500 },
                'gemini_3_pro_image': { free: 50, paid: 500 },
                'gemini_2_5_flash_image': { free: 100, paid: 500 },
                'gemini_2_0_flash_image': { free: 100, paid: 500 },
                'veo_3_1': { free: 20, paid: 500 },
                'veo_3_1_fast': { free: 50, paid: 500 },
                'veo_3_0': { free: 10, paid: 500 },
                'veo_2_0': { free: 5, paid: 500 },
                // Aliases mapped to same defaults
                'pro': { free: 0, paid: 1000 },
                'flash': { free: 20, paid: 1000 },
                'flash_3_0': { free: 20, paid: 1000 },
                'flash_lite': { free: 20, paid: 1000 },
                'imagen_4_fast': { free: 10, paid: 500 },
                'imagen_4_std': { free: 10, paid: 500 },
                'imagen_4_ultra': { free: 10, paid: 500 },
                'flash_image_2_0': { free: 100, paid: 500 },
                'flash_image_2_5': { free: 100, paid: 500 },
                'pro_image_3_0': { free: 50, paid: 500 },
                'veo': { free: 50, paid: 500 },
                'veo_hq': { free: 20, paid: 500 },
                'gemini_1_5_pro': { free: 0, paid: 1000 }
            };
            const tierSettings = settings.modelLimitsByTier || {};
            const accessSettings = settings.modelAccessTiers || {};

            // Helper to get limit for a key
            const getLimit = (key, type) => {
                // Return 0 if completely disabled in settings, but default is ENABLED if not set
                if (accessSettings[key]?.[type] === false) return 0;

                // Get value: Settings -> Default -> 0
                return tierSettings[key]?.[type] ?? DEFAULT_TIER_LIMITS[key]?.[type] ?? 0;
            };

            for (const [sKey, mId] of Object.entries(MODEL_LIMIT_MAPPING)) {
                // Calculate limits for this specific model based on keys
                const freeLimit = getLimit(sKey, 'free');
                const paidLimit = getLimit(sKey, 'paid');

                const calculated = (freeKeys.length * freeLimit) + (paidKeys.length * paidLimit);
                newModelLimits[mId] = Math.max(newModelLimits[mId] || 0, calculated);
            }

            // Calculate total credits for user (Sum of all models)
            // We iterate strictly over the keys in DEFAULT_TIER_LIMITS to avoid alias duplication issues if possible, 
            // OR we iterate over unique setting keys. 
            // Let's stick to the keys present in MODEL_LIMIT_MAPPING but be careful about duplicates.
            // Actually, MODEL_LIMIT_MAPPING has many aliases. We should likely only sum unique "Concept" keys to avoid ballooning totals.
            // Current "settings" structure usually has one key per concept (e.g. 'flash_2_5').
            // To match AdminDashboard, we should look at what keys are exposed there. 
            // The safest is to iterate the "Canonical" keys associated with the UI models.
            // For now, to match previous behavior (which might have been over-summing or just using what was in settings):
            // We will sum up based on the keys available in the defaults or settings.

            // Unique keys to sum for totals (Avoiding aliases)
            const UNIQUE_KEYS = [
                'pro_3_0', 'pro_2_5', 'flash_2_5', 'flash_lite_2_5', 'gemini_3_0_flash',
                'imagen_ultra', 'imagen_fast', 'imagen_std', 'pro_image_3_0', 'gemini_2_5_flash_image', 'gemini_2_0_flash_image',
                'veo_3_1', 'veo_3_1_fast', 'veo_3_0', 'veo_2_0'
            ];

            for (const key of UNIQUE_KEYS) {
                totalFreeCredits += getLimit(key, 'free');
                totalPaidCredits += getLimit(key, 'paid');
            }

            const newLimit = (freeKeys.length * totalFreeCredits) + (paidKeys.length * totalPaidCredits);

            // Fetch current stats to merge
            const [uRows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
            let stats = typeof uRows[0].usage_stats === 'string' ? JSON.parse(uRows[0].usage_stats) : uRows[0].usage_stats;
            if (!stats) stats = {};
            stats.apiCallsLimit = newLimit;
            stats.modelLimits = newModelLimits;

            // --- RECALCULATE USAGE FROM DATABASE (BY USER & HASHES) ---
            // We must query by user_uid (for new usage) AND key_hashes (for historical usage before user_uid was tracked)
            let calculatedUsage = 0;
            let dailyCalculatedUsage = 0;
            const dateStr = getPacificYMD();
            const recalculatedModelUsage = {};

            // Calculate hashes for all valid keys
            const keyHashes = [];
            console.log('[SAVE KEYS] DEBUG: Processing keys for usage calc...');
            safeKeys.forEach(k => {
                let val = k.fullKey || k.key;

                // If the key from frontend is masked/empty, try to use the one we just saved/updated/fetched from DB
                if (!val || val.includes('••••')) {
                    if (existingKeyMap[k.id]) {
                        val = existingKeyMap[k.id];
                    }
                }

                const isMasked = val && val.includes('••••');
                console.log(`[SAVE KEYS] Key ID: ${k.id}, HasVal: ${!!val}, Masked: ${isMasked}`);

                if (val && !isMasked) {
                    val = val.trim();
                    const hash = createHash('sha256').update(val).digest('hex');
                    keyHashes.push(hash);
                    console.log(`[SAVE KEYS] Added hash for ${k.id}: ${hash.substring(0, 8)}...`);
                }
            });
            console.log(`[SAVE KEYS] Total valid hashes: ${keyHashes.length}`);

            const monthStart = dateStr.substring(0, 7) + '-01'; // YYYY-MM-01

            if (keyHashes.length > 0) {
                // Query 1: Total API calls (Monthly Aggregation)
                // We want to sum all usage for the current month to match persisted stats

                // Build dynamic placeholders for the IN clause
                const placeholders = keyHashes.map(() => '?').join(',');

                const [usageRows] = await db.query(
                    `SELECT SUM(api_calls) as total FROM api_key_usage 
                     WHERE (user_uid = ? OR key_hash IN (${placeholders})) AND usage_date >= ?`,
                    [uid, ...keyHashes, monthStart]
                );
                calculatedUsage = usageRows[0].total || 0;

                // Query 1b: Daily API calls (Today Only)
                const [dailyUsageRows] = await db.query(
                    `SELECT SUM(api_calls) as total FROM api_key_usage 
                     WHERE (user_uid = ? OR key_hash IN (${placeholders})) AND usage_date = ?`,
                    [uid, ...keyHashes, dateStr]
                );
                dailyCalculatedUsage = dailyUsageRows[0].total || 0;

                console.log(`[SAVE KEYS] Calc usage (uid OR hash) since ${monthStart}: Monthly=${calculatedUsage}, Daily=${dailyCalculatedUsage}`);

                // Query 2: Per-model API calls (Daily Reset at Midnight PT)
                const [modelUsageRows] = await db.query(
                    `SELECT model_id, SUM(api_calls) as total 
                     FROM api_key_usage 
                     WHERE (user_uid = ? OR key_hash IN (${placeholders})) AND usage_date = ?
                     GROUP BY model_id`,
                    [uid, ...keyHashes, dateStr]
                );

                modelUsageRows.forEach(row => {
                    if (row.model_id) {
                        recalculatedModelUsage[row.model_id] = Number(row.total) || 0;
                    }
                });
            } else {
                // Fallback if no keys (should be rare if validKeys > 0)
                // Just query by uid
                const [usageRows] = await db.query(
                    `SELECT SUM(api_calls) as total FROM api_key_usage WHERE user_uid = ? AND usage_date >= ?`,
                    [uid, monthStart]
                );
                calculatedUsage = usageRows[0].total || 0;

                const [dailyUsageRows] = await db.query(
                    `SELECT SUM(api_calls) as total FROM api_key_usage WHERE user_uid = ? AND usage_date = ?`,
                    [uid, dateStr]
                );
                dailyCalculatedUsage = dailyUsageRows[0].total || 0;

                const [modelUsageRows] = await db.query(
                    `SELECT model_id, SUM(api_calls) as total 
                     FROM api_key_usage 
                     WHERE user_uid = ? AND usage_date = ?
                     GROUP BY model_id`,
                    [uid, dateStr]
                );
                modelUsageRows.forEach(row => {
                    if (row.model_id) recalculatedModelUsage[row.model_id] = Number(row.total) || 0;
                });
            }

            console.log('[SAVE KEYS] Recalculated by user_uid:', recalculatedModelUsage);

            // Update usage stats with recalculated values from database
            stats.apiCallsThisMonth = Number(calculatedUsage);
            stats.daily.apiCalls = Number(dailyCalculatedUsage); // Ensure daily is updated separately
            stats.modelUsage = recalculatedModelUsage; // Use recalculated values, not reset to {}

            await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), uid]);

            res.json({
                success: true,
                newLimit,
                newUsage: calculatedUsage,
                modelUsage: recalculatedModelUsage,  // Return model usage to frontend
                modelLimits: newModelLimits           // Return model limits to frontend
            });
        } catch (err) {
            console.error("Step 5 (Stats Update) failed:", err);
            throw err;
        }

    } catch (error) {
        console.error("Save Keys Global Error:", error);
        res.status(500).json({ error: error.message });
    }
});




// --- ASSET ROUTES ---

app.get('/api/assets/list', requireAuth, async (req, res) => {
    const { uid, type, isDeleted, limit, excludeUrl } = req.query;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        // Optimization: Allow excluding heavy URL field (base64) for stats/listing
        const columns = excludeUrl === 'true'
            ? 'id, user_uid, type, prompt, created_at, is_favorite, is_deleted, metadata, aspect_ratio, batch_id, model_used, generation_settings, title'
            : '*';

        let sql = `SELECT ${columns} FROM assets WHERE user_uid = ? AND is_deleted = ?`;
        const params = [uid, isDeleted === 'true' ? 1 : 0];

        if (type) {
            if (type.includes(',')) {
                // Support multiple types (e.g. "image,video")
                const types = type.split(',').map(t => t.trim());
                sql += ` AND type IN (${types.map(() => '?').join(',')})`;
                params.push(...types);
            } else {
                sql += ' AND type = ?';
                params.push(type);
            }
        }

        sql += ' ORDER BY created_at DESC';

        // Limit support
        if (limit) {
            const limitNum = parseInt(limit);
            if (!isNaN(limitNum) && limitNum > 0) {
                sql += ' LIMIT ?';
                params.push(limitNum);
            }
        }

        const [rows] = await db.query(sql, params);
        const assets = rows.map(r => ({
            id: r.id,
            type: r.type,
            url: r.url || '', // Handle missing URL if excluded
            prompt: r.prompt,
            createdAt: Number(r.created_at),
            isFavorite: !!r.is_favorite,
            isDeleted: !!r.is_deleted,
            metadata: typeof r.metadata === 'string' ? JSON.parse(r.metadata) : r.metadata,
            aspectRatio: r.aspect_ratio,
            batchId: r.batch_id,
            modelUsed: r.model_used,
            generationSettings: typeof r.generation_settings === 'string' ? JSON.parse(r.generation_settings) : r.generation_settings,
            title: r.title,
            userId: r.user_uid
        }));

        res.json({ assets });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/assets/create', requireAuth, async (req, res) => {
    const { asset, uid } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        const newId = asset.id || randomUUID();
        const now = Date.now();

        await db.query(
            `INSERT INTO assets 
            (id, user_uid, type, url, prompt, created_at, is_favorite, is_deleted, metadata, aspect_ratio, batch_id, model_used, generation_settings, title)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                newId, uid, asset.type, asset.url, asset.prompt || '', asset.createdAt || now,
                asset.isFavorite ? 1 : 0, 0, JSON.stringify(asset.metadata || {}),
                asset.aspectRatio || '', asset.batchId || '', asset.modelUsed || null,
                JSON.stringify(asset.generationSettings || null), asset.title || ''
            ]
        );

        // Update global stats
        await db.query(`UPDATE system_settings SET settings = JSON_SET(COALESCE(settings, '{}'), '$.stats.assets', COALESCE(JSON_EXTRACT(settings, '$.stats.assets'), 0) + 1) WHERE id = 1`);

        res.json({ success: true, id: newId });
    } catch (error) {
        console.error("Create Asset Error:", error);
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/assets/update', requireAuth, async (req, res) => {
    const { uid, assetId, updates } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        // Construct dynamic update
        const fields = [];
        const values = [];

        if (updates.isFavorite !== undefined) { fields.push('is_favorite = ?'); values.push(updates.isFavorite ? 1 : 0); }
        if (updates.isDeleted !== undefined) { fields.push('is_deleted = ?'); values.push(updates.isDeleted ? 1 : 0); }
        if (updates.metadata) { fields.push('metadata = ?'); values.push(JSON.stringify(updates.metadata)); }

        if (fields.length === 0) return res.json({ success: true });

        values.push(assetId);
        values.push(uid);

        await db.query(`UPDATE assets SET ${fields.join(', ')} WHERE id = ? AND user_uid = ?`, values);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/assets/delete', requireAuth, async (req, res) => {
    const { uid, assetId } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        await db.query('DELETE FROM assets WHERE id = ? AND user_uid = ?', [assetId, uid]);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// --- ADMIN ROUTES ---

app.get('/api/admin/stats', requireAuth, requireAdmin, async (req, res) => {
    try {
        const [userRows] = await db.query('SELECT COUNT(*) as count FROM users WHERE is_deleted = 0');
        const [assetRows] = await db.query('SELECT COUNT(*) as count FROM assets WHERE is_deleted = 0');

        // Calculate total API calls (from api_key_usage for accuracy, or system settings)
        const [usageRows] = await db.query('SELECT SUM(api_calls) as total FROM api_key_usage');
        const totalApiCalls = usageRows[0].total || 0;

        // Calculate total prompts (aggregated from user stats)
        // usage_stats is JSON, we need to extract promptsGenerated. 
        // Note: usage_stats might be string or object in DB return? In MySQL JSON type it's object, if TEXT it's string.
        // Assuming JSON column or parsing:
        // MySQL 5.7+ supports JSON_EXTRACT. 
        // If usage_stats is TEXT, we might need to fetch all and sum in JS (slow) or use regex in SQL.
        // Let's assume JSON column for now or fetch all is safer for "revert" logic if schema is unsure.
        // Actually, fetching all users is heavy.
        // Let's check schema/code usage. 
        // In increment: `UPDATE users SET usage_stats = ?` passing stringified JSON.
        // So it's likely a TEXT/JSON column storing stringified JSON.

        let totalPrompts = 0;
        const [allUsers] = await db.query('SELECT usage_stats FROM users');
        allUsers.forEach(u => {
            let s = u.usage_stats;
            if (typeof s === 'string') {
                try { s = JSON.parse(s); } catch (e) { s = {}; }
            }
            if (s && s.promptsGenerated) totalPrompts += Number(s.promptsGenerated);
        });

        // --- EXTENDED ANALYTICS ---

        // 1. Model Usage Distribution (Real Data)
        const [modelRows] = await db.query(`
            SELECT model_id, SUM(api_calls) as total 
            FROM api_key_usage 
            GROUP BY model_id 
            ORDER BY total DESC 
            LIMIT 6
        `);

        const MODEL_COLORS = {
            'gemini-2.5-flash': '#10b981', // Emerald
            'gemini-2.5-pro': '#8b5cf6',   // Violet
            'imagen-3.0-generate-001': '#f59e0b', // Amber
            'veo-2.0-generate-preview': '#3b82f6', // Blue
            'default': '#64748b' // Slate
        };

        const modelUsageDistribution = modelRows.map(row => {
            // Simplify name for display but keep version distinction
            let name = row.model_id;
            // Map specific technical IDs to cleaner readable names
            if (name === 'gemini-2.5-flash') name = 'Gemini 2.5 Flash';
            else if (name === 'gemini-2.5-flash-lite-preview-09-2025') name = 'Gemini 2.5 Flash Lite';

            else if (name === 'gemini-2.5-pro') name = 'Gemini 2.5 Pro';
            else if (name === 'gemini-2.0-pro') name = 'Gemini 2.0 Pro';
            else if (name.includes('imagen')) name = 'Imagen 4';
            else if (name.includes('veo')) name = 'Veo Video';
            else if (name === 'gemini-1.5-flash') name = 'Gemini 1.5 Flash';
            else if (name.includes('flash')) name = 'Gemini Flash';

            return {
                name: name,
                value: Number(row.total),
                color: MODEL_COLORS[row.model_id] || MODEL_COLORS.default
            };
        });

        if (modelUsageDistribution.length === 0) {
            // Fallback if no usage yet
            modelUsageDistribution.push({ name: 'No Usage', value: 1, color: '#e2e8f0' });
        }

        // 2. Top Countries (Real Data)
        const [countryRows] = await db.query(`
            SELECT country, COUNT(*) as count 
            FROM users 
            WHERE country IS NOT NULL AND country != 'Unknown' 
            GROUP BY country 
            ORDER BY count DESC 
            LIMIT 5
        `);

        const totalUsersCount = userRows[0].count || 1;
        const topCountries = countryRows.map(r => ({
            country: r.country,
            // Format: "5 users (20%)" to be explicit
            users: `${r.count} users (${Math.round((r.count / totalUsersCount) * 100)}%)`,
            trend: '+' + Math.floor(Math.random() * 5) + '%' // Mock trend for now
        }));

        // 3. Weekly Traffic (Real - from Activity Logs)
        // Get last 7 days dates
        const days = [];
        for (let i = 6; i >= 0; i--) {
            const d = new Date();
            d.setDate(d.getDate() - i);
            days.push(d.toISOString().slice(0, 10));
        }

        // Query activity logs for sessions (events) and users (unique)
        // Note: timestamp is ms
        const sevenDaysAgo = Date.now() - (7 * 24 * 60 * 60 * 1000);
        const [trafficRows] = await db.query(`
            SELECT 
                DATE(FROM_UNIXTIME(timestamp/1000)) as day, 
                COUNT(*) as sessions, 
                COUNT(DISTINCT user_uid) as users 
            FROM activity_logs 
            WHERE timestamp >= ? 
            GROUP BY day
        `, [sevenDaysAgo]);

        const weeklyTraffic = days.map(dayStr => {
            const row = trafficRows.find(r => {
                // Handle various SQL date return formats
                const rDate = typeof r.day === 'string' ? r.day : r.day.toISOString().slice(0, 10);
                return rDate === dayStr;
            });

            // map YYYY-MM-DD to "Mon", "Tue" etc
            // We use 'UTC' to avoid timezone shifts on the day string 'YYYY-MM-DD'
            const [y, m, d] = dayStr.split('-').map(Number);
            const dateObj = new Date(y, m - 1, d);
            const name = dateObj.toLocaleDateString('en-US', { weekday: 'short' });

            return {
                name,
                users: row ? Number(row.users) : 0,
                sessions: row ? Number(row.sessions) : 0
            };
        });

        // 4. Monthly Revenue (Estimated Real - from Subscriptions)
        const [subRows] = await db.query(`
            SELECT subscription_status, COUNT(*) as count 
            FROM users 
            GROUP BY subscription_status
        `);

        let currentMRR = 0;
        subRows.forEach(r => {
            // Normalize status just in case
            const status = (r.subscription_status || 'free').toLowerCase();
            const count = Number(r.count);
            // Pricing: Pro=29, Agency=99
            if (status.includes('pro')) currentMRR += (29 * count);
            else if (status.includes('agency')) currentMRR += (99 * count);
        });

        // Generate chart data: Past 5 months + Current
        const monthNames = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
        const dRev = new Date();
        const monthlyRevenue = [];
        for (let i = 5; i >= 0; i--) {
            const m = new Date(dRev.getFullYear(), dRev.getMonth() - i, 1);
            const name = monthNames[m.getMonth()];
            // Mock growth curve ending at real current MRR
            const factor = 1 - (i * 0.1);
            monthlyRevenue.push({
                name,
                revenue: Math.round(currentMRR * factor)
            });
        }

        // 5. System Load (Real - Hourly Activity Today)
        const startOfToday = new Date();
        startOfToday.setHours(0, 0, 0, 0);
        const startTs = startOfToday.getTime();

        const [hourlyRows] = await db.query(`
            SELECT HOUR(FROM_UNIXTIME(timestamp/1000)) as hr, COUNT(*) as count 
            FROM activity_logs 
            WHERE timestamp >= ? 
            GROUP BY hr
        `, [startTs]);

        // Map 00:00, 04:00, 08:00 etc.
        const timePoints = [0, 4, 8, 12, 16, 20, 23];
        const systemLoadData = timePoints.map(tp => {
            // Aggregate user usage around this hour
            let count = 0;
            for (let h = tp; h < (tp === 23 ? 24 : tp + 4); h++) {
                const r = hourlyRows.find(row => row.hr === h);
                if (r) count += Number(r.count);
            }
            // Normalize for display (cpu/mem are mocked based on load, api is real count)
            const loadFactor = Math.min((count / 10) + 10, 100);

            return {
                name: `${String(tp).padStart(2, '0')}:00`,
                cpu: Math.floor(loadFactor * 0.8),
                mem: Math.floor(loadFactor * 0.9),
                api: count
            };
        });

        res.json({
            users: userRows[0].count,
            totalGeneratedFiles: assetRows[0].count,
            totalApiCalls: Number(totalApiCalls),
            totalPromptsGenerated: totalPrompts,
            // Extended Data
            weeklyTraffic,
            modelUsageDistribution,
            monthlyRevenue,
            topCountries,
            systemLoadData
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// --- NEW: Reset All Stats Endpoint ---
app.post('/api/admin/stats/reset', requireAuth, async (req, res) => {
    if (req.user.role !== 'admin') return res.sendStatus(403);
    try {
        // 1. Clear Activity Logs (Weekly Traffic, System Load)
        await db.query('TRUNCATE TABLE activity_logs');

        // 2. Clear API Key Usage (Model Usage)
        await db.query('TRUNCATE TABLE api_key_usage');

        // 3. Reset User Usage Stats (Individual Counters)
        await db.query('UPDATE users SET usage_stats = "{}"');

        // 4. Clear Banned IPs (Optional but good for full reset)
        await db.query('TRUNCATE TABLE banned_ips');

        // 5. Clear Audit Logs (Clean slate)
        await db.query('TRUNCATE TABLE audit_logs');

        console.log(`[Admin] Stats reset by ${req.user.email}`);
        res.json({ success: true, message: 'All analytics stats reset successfully.' });
    } catch (error) {
        console.error('Reset stats failed:', error);
        res.status(500).json({ error: 'Failed to reset stats' });
    }
});

// --- BANNED IPS ENDPOINT ---
app.post('/api/admin/banned-ips', requireAuth, requireAdmin, async (req, res) => {
    const { ip, action, reason } = req.body;
    if (!ip) return res.status(400).json({ error: 'IP Required' });

    try {
        if (action === 'ban') {
            await db.query('INSERT IGNORE INTO banned_ips (ip_address, reason, banned_at) VALUES (?, ?, ?)', [ip, reason || 'Admin Ban', Date.now()]);
        } else if (action === 'unban') {
            await db.query('DELETE FROM banned_ips WHERE ip_address = ?', [ip]);
        }
        res.json({ success: true });
    } catch (error) {
        console.error("IP Ban/Unban Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// Generic Admin Data CRUD
const ADMIN_TABLES = ['tips', 'tutorial_steps', 'broadcasts', 'bug_reports', 'support_messages', 'audit_logs', 'users', 'system_settings', 'stock_sites'];

// Specific Permanent Delete for Users
app.post('/api/admin/users/permanent-delete', requireAuth, requireAdmin, async (req, res) => {
    const { uid } = req.body;
    if (!uid) return res.status(400).json({ error: 'User UID is required' });

    console.log(`[ADMIN] Permanently deleting user: ${uid}`);

    try {
        // 1. FILE CLEANUP
        // Get user assets to delete files
        const [assets] = await db.query('SELECT url FROM assets WHERE user_uid = ?', [uid]);
        const [userRows] = await db.query('SELECT photo_url FROM users WHERE uid = ?', [uid]);

        const filesToDelete = [];

        // Collect Asset Files
        assets.forEach(asset => {
            if (asset.url && asset.url.includes('/uploads/')) {
                const filename = asset.url.split('/uploads/')[1];
                if (filename) filesToDelete.push(path.join(__dirname, 'uploads', filename));
            }
        });

        // Collect Profile Photo
        if (userRows.length > 0 && userRows[0].photo_url && userRows[0].photo_url.includes('/uploads/')) {
            const filename = userRows[0].photo_url.split('/uploads/')[1];
            if (filename) filesToDelete.push(path.join(__dirname, 'uploads', filename));
        }

        // Delete Files
        console.log(`[CLEANUP] Deleting ${filesToDelete.length} files for user ${uid}`);
        for (const filePath of filesToDelete) {
            try {
                if (fs.existsSync(filePath)) {
                    fs.unlinkSync(filePath);
                    console.log(`Deleted file: ${filePath}`);
                }
            } catch (err) {
                console.error(`Failed to delete file ${filePath}:`, err);
            }
        }

        // 2. DATA CLEANUP
        // Manually delete non-cascading related data
        await db.query('DELETE FROM bug_reports WHERE user_uid = ?', [uid]);
        await db.query('DELETE FROM support_messages WHERE user_uid = ?', [uid]);

        // Key usage might have ON DELETE CASCADE if linked by user_uid, but let's be safe
        await db.query('DELETE FROM api_key_usage WHERE user_uid = ?', [uid]);

        // Finally delete the user (Will CASCADE assets, api_keys, activity_logs)
        await db.query('DELETE FROM users WHERE uid = ?', [uid]);

        res.json({ success: true });
    } catch (error) {
        console.error("Permanent Delete Error:", error);
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/increment-usage', async (req, res) => {
    try {
        // Fetch current settings
        const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 1');
        if (rows.length > 0) {
            let settings = rows[0].settings;
            if (typeof settings === 'string') settings = JSON.parse(settings);

            settings.systemApiUsage = (settings.systemApiUsage || 0) + 1;

            await db.query('UPDATE system_settings SET settings = ? WHERE id = 1', [JSON.stringify(settings)]);
        }
        res.json({ success: true });
    } catch (error) {
        console.error("Failed to increment usage:", error);
        res.json({ success: true }); // Fail silent to client
    }
});

app.get('/api/admin/data/:table', requireAuth, requireAdmin, async (req, res) => {
    const { table } = req.params;
    if (!ADMIN_TABLES.includes(table)) return res.status(400).json({ error: 'Invalid table' });

    try {
        const [rows] = await db.query(`SELECT * FROM ${table}`);
        res.json({ data: rows });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/admin/data/:table', requireAuth, requireAdmin, async (req, res) => {
    const { table } = req.params;
    const { item } = req.body;
    if (!ADMIN_TABLES.includes(table)) return res.status(400).json({ error: 'Invalid table' });

    try {
        const keys = Object.keys(item);
        const values = Object.values(item);
        const placeholders = keys.map(() => '?').join(',');
        const columns = keys.join(',');

        await db.query(`INSERT INTO ${table} (${columns}) VALUES (${placeholders})`, values);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/admin/data/:table/update', requireAuth, requireAdmin, async (req, res) => {
    const { table } = req.params;
    const { id, updates } = req.body;
    if (!ADMIN_TABLES.includes(table)) return res.status(400).json({ error: 'Invalid table' });

    try {
        const keys = Object.keys(updates);
        const values = Object.values(updates);
        const setClause = keys.map(k => `${k} = ?`).join(',');

        values.push(id);

        await db.query(`UPDATE ${table} SET ${setClause} WHERE id = ?`, values);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/admin/data/:table/delete', requireAuth, requireAdmin, async (req, res) => {
    const { table } = req.params;
    const { id } = req.body;
    if (!ADMIN_TABLES.includes(table)) return res.status(400).json({ error: 'Invalid table' });

    try {
        await db.query(`DELETE FROM ${table} WHERE id = ?`, [id]);
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Stats Increment Generic
// Stats Increment Generic (Enhanced for Daily/Monthly)
app.post('/api/user/stats/increment', requireAuth, async (req, res) => {
    console.log('\n\n========== INCREMENT API CALLED ==========');
    console.log('Body:', JSON.stringify(req.body, null, 2));

    const { uid, statName, amount } = req.body;

    try {
        // Step 1: Read current stats
        console.log('STEP 1: Reading from DB...');
        const [rows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) {
            console.error('ERROR: User not found');
            return res.status(404).json({ error: 'User not found' });
        }

        let stats = typeof rows[0].usage_stats === 'string' ? JSON.parse(rows[0].usage_stats) : rows[0].usage_stats;
        console.log('Current daily:', JSON.stringify(stats.daily || {}));

        // Step 2: Initialize if needed
        console.log('STEP 2: Initializing structure...');
        if (!stats) stats = {};
        if (!stats.daily) {
            stats.daily = { date: getPacificYMD() };
            console.log('Created daily with date:', stats.daily.date);
        }
        if (!stats.monthly) stats.monthly = { month: new Date().toISOString().slice(0, 7) };

        // Step 3: Check for date/month mismatch
        const today = getPacificYMD();
        const thisMonth = new Date().toISOString().slice(0, 7);

        console.log('STEP 3: Date/Month check', { today, thisMonth });

        if (stats.daily.date !== today) {
            console.log(`DATE MISMATCH - Resetting daily (${stats.daily.date} -> ${today})`);
            stats.history.lastDay = { ...stats.daily };
            // Reset helper inline or ensure fields are 0
            Object.keys(stats.daily).forEach(k => { if (typeof stats.daily[k] === 'number') stats.daily[k] = 0; });
            stats.daily.date = today;
        }

        if (stats.monthly.month !== thisMonth) {
            console.log(`MONTH MISMATCH - Resetting monthly (${stats.monthly.month} -> ${thisMonth})`);
            stats.history.lastMonth = { ...stats.monthly };
            Object.keys(stats.monthly).forEach(k => { if (typeof stats.monthly[k] === 'number') stats.monthly[k] = 0; });
            stats.monthly.month = thisMonth;
        }

        // Step 4: Increment (Daily AND Monthly)
        console.log('STEP 4: Incrementing', statName);
        const val = parseInt(amount) || 1;

        // Daily
        const dailyBefore = stats.daily[statName] || 0;
        stats.daily[statName] = dailyBefore + val;

        // Monthly
        const monthlyBefore = stats.monthly[statName] || 0;
        stats.monthly[statName] = monthlyBefore + val;

        console.log(`${statName} (Daily): ${dailyBefore} + ${val} = ${stats.daily[statName]}`);
        console.log(`${statName} (Monthly): ${monthlyBefore} + ${val} = ${stats.monthly[statName]}`);

        // Step 5: Persist
        console.log('STEP 5: Writing to DB...');
        const jsonString = JSON.stringify(stats);

        await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [jsonString, uid]);
        console.log('DB UPDATE COMPLETE');

        // Step 6: Verify
        console.log('STEP 6: Verifying write...');
        const [verify] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        const verified = typeof verify[0].usage_stats === 'string' ? JSON.parse(verify[0].usage_stats) : verify[0].usage_stats;
        console.log('Verified daily:', JSON.stringify(verified.daily));

        if (verified.daily[statName] === stats.daily[statName]) {
            console.log('✅ VERIFICATION SUCCESS');
        } else {
            console.error('❌ VERIFICATION FAILED - Expected:', stats.daily[statName], 'Got:', verified.daily[statName]);
        }

        console.log('========== INCREMENT API COMPLETE ==========\n\n');
        res.json({ success: true, stats });

    } catch (error) {
        console.error('\n\n========== INCREMENT ERROR ==========');
        console.error(error);
        console.error('========================================\n\n');
        res.status(500).json({ error: error.message });
    }
});

app.post('/api/user/deduct-credits', requireAuth, async (req, res) => {
    console.log('\n\n========== DEDUCT API CALLED ==========');
    console.log('Body:', JSON.stringify(req.body, null, 2));

    const { uid, amount, apiKey, modelId } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') {
        console.log('[DEDUCT] Auth failed');
        return res.status(403);
    }

    try {
        const val = parseInt(amount) || 1;
        const currentModel = modelId || 'unknown';
        const today = getPacificYMD();

        console.log(`[DEDUCT] User: ${uid}, Amount: ${val}, Model: ${currentModel}, Date: ${today}`);

        // If apiKey provided, track against it
        if (apiKey) {
            const hash = createHash('sha256').update(apiKey).digest('hex');

            // Insert or Update Usage (Now includes user_uid for historical data preservation)
            // usage_date is DATE type
            await db.query(`
                INSERT INTO api_key_usage (key_hash, user_uid, usage_date, api_calls, model_id)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE api_calls = api_calls + ?
            `, [hash, uid, today, val, currentModel, val]);
            console.log('[DEDUCT] API key usage logged with user_uid');
        }

        // Initialize stats (FIX: Fetch first!)
        console.log('[DEDUCT] Fetching user stats...');
        const [uRows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        let stats = {};
        if (uRows.length > 0) {
            stats = typeof uRows[0].usage_stats === 'string' ? JSON.parse(uRows[0].usage_stats) : uRows[0].usage_stats;
        }
        if (!stats) stats = {};

        if (!stats.daily) stats.daily = { date: getPacificYMD() };
        if (!stats.monthly) stats.monthly = { month: new Date().toISOString().slice(0, 7) };
        // Ensure modelUsage object exists
        if (!stats.modelUsage) stats.modelUsage = {};

        // Handle Rotations
        const todayReset = getPacificYMD();
        const monthReset = new Date().toISOString().slice(0, 7);

        if (stats.daily.date !== todayReset) {
            console.log(`[DEDUCT] Daily reset triggered (${stats.daily.date} → ${todayReset})`);
            if (stats.history) stats.history.lastDay = { ...stats.daily };
            resetStatsObj(stats.daily);
            stats.daily.date = todayReset;
        }
        if (stats.monthly.month !== monthReset) {
            console.log(`[DEDUCT] Monthly reset triggered (${stats.monthly.month} → ${monthReset})`);
            stats.monthly = { month: monthReset };
            // Optional: Reset monthly model usage if desired, but usually model limits are monthly?
            // If model limits are monthly, we should probably reset modelUsage here too or have a separate daily/monthly track.
            // For now, assuming modelUsage is cumulative or monthly. If monthly, reset here.
            // Use case implies "Remaining Credits" which usually reset monthly.
            stats.modelUsage = {};
        }

        const beforeDaily = stats.daily.apiCalls || 0;
        const beforeMonthly = stats.monthly.apiCalls || 0;

        // Increment apiCalls
        stats.apiCalls = (stats.apiCalls || 0) + val;
        // Ensure daily/monthly structure exists before adding
        if (!stats.daily.apiCalls) stats.daily.apiCalls = 0;
        if (!stats.monthly.apiCalls) stats.monthly.apiCalls = 0;

        stats.daily.apiCalls += val;
        stats.monthly.apiCalls += val;

        // Increment Model Specific Usage
        if (modelId) {
            // Initialize structures if missing
            if (!stats.modelUsage) stats.modelUsage = {};
            if (!stats.daily) stats.daily = { date: today, apiCalls: 0 }; // Ensure daily obj exists

            // Increment model usage
            stats.modelUsage[modelId] = (stats.modelUsage[modelId] || 0) + val;

            // Increment monthly usage (already handled above, but if this block was standalone, it would be here)
            // stats.apiCallsThisMonth = (stats.apiCallsThisMonth || 0) + val; // This is redundant with stats.monthly.apiCalls

            // Increment daily usage
            // Check if date matches first (already handled by rotation logic above)
            if (stats.daily.date !== today) { // This check is redundant due to rotation logic
                resetStatsObj(stats.daily);
                stats.daily.date = today;
                stats.daily.apiCalls = val;
            } else {
                // stats.daily.apiCalls = (stats.daily.apiCalls || 0) + val; // This is redundant with stats.daily.apiCalls += val;
            }
            console.log(`[DEDUCT] Model ${modelId} usage: ${stats.modelUsage[modelId]}`);
        }

        console.log(`[DEDUCT] apiCalls increment:`);
        console.log(`  Daily: ${beforeDaily} → ${stats.daily.apiCalls}`);
        console.log(`  Monthly: ${beforeMonthly} → ${stats.monthly.apiCalls}`);
        console.log(`  Total: ${stats.apiCalls}`);

        await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), uid]);
        console.log('[DEDUCT] ✅ Database updated successfully');
        console.log('========== DEDUCT API COMPLETE ==========\n\n');

        res.json({ success: true });
    } catch (error) {
        console.error('\n\n========== DEDUCT ERROR ==========');
        console.error("Deduct Credits Error:", error);
        console.error('======================================\n\n');
        res.status(500).json({ error: error.message });
    }
});

// Daily Reset Check Endpoint
app.post('/api/user/check-daily-reset', requireAuth, async (req, res) => {
    const { uid } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        const [rows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        if (rows.length === 0) return res.status(404).json({ error: 'User not found' });

        let stats = typeof rows[0].usage_stats === 'string' ? JSON.parse(rows[0].usage_stats) : rows[0].usage_stats;

        // Initialize if empty
        if (!stats) stats = {};

        // Track structural updates
        let updated = false;

        if (!stats.daily) { stats.daily = { date: getPacificYMD() }; updated = true; }
        if (!stats.monthly) { stats.monthly = { month: new Date().toISOString().slice(0, 7) }; updated = true; }
        if (!stats.history) { stats.history = { lastDay: {}, lastMonth: {} }; updated = true; }
        const today = getPacificYMD();
        const thisMonth = new Date().toISOString().slice(0, 7);

        // Daily Reset Check
        if (stats.daily.date !== today) {
            stats.history.lastDay = { ...stats.daily };
            resetStatsObj(stats.daily);
            stats.daily.date = today;
            updated = true;
        }

        // Monthly Reset Check
        if (stats.monthly.month !== thisMonth) {
            stats.history.lastMonth = { ...stats.monthly };
            resetStatsObj(stats.monthly);
            stats.monthly.month = thisMonth;
            updated = true;
        }

        if (updated) {
            await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), uid]);
        }

        res.json({ success: true, stats, updated });
    } catch (error) {
        console.error("Daily Reset Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// Get next PT midnight timestamp
app.get('/api/next-pt-midnight', (req, res) => {
    try {
        const ptDate = getPacificDate();
        const nextMidnight = new Date(ptDate);
        nextMidnight.setDate(nextMidnight.getDate() + 1);
        nextMidnight.setHours(0, 0, 0, 0);

        // Return timestamp for the frontend
        res.json({
            timestamp: nextMidnight.getTime(),
            formatted: nextMidnight.toISOString()
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// --- ACTIVITY LOGGING ---

app.post('/api/user/activity', requireAuth, async (req, res) => {
    const { uid, type, description, metadata } = req.body;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        const id = randomUUID();
        const timestamp = Date.now();
        await db.query(
            'INSERT INTO activity_logs (id, user_uid, type, text, timestamp) VALUES (?, ?, ?, ?, ?)',
            [id, uid, type, description, timestamp]
        );
        res.json({ success: true, id });
    } catch (error) {
        console.error("Log Activity Error:", error);
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/user/activity', requireAuth, async (req, res) => {
    const { uid, limit } = req.query;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        const limitVal = parseInt(limit) || 20;
        const [rows] = await db.query(
            'SELECT * FROM activity_logs WHERE user_uid = ? ORDER BY timestamp DESC LIMIT ?',
            [uid, limitVal]
        );

        // Map database fields to frontend expected format
        const activities = rows.map(r => ({
            id: r.id,
            type: r.type,
            text: r.text, // Frontend expects 'text' field
            timestamp: Number(r.timestamp)
        }));

        res.json({ activities });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// --- PUBLIC / SHARED DATA ENDPOINTS ---

app.get('/api/settings', async (req, res) => {
    try {
        const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 1');
        if (rows.length === 0) return res.json({ settings: "{}" });
        // Return parsed JSON or string? Frontend expects what?
        // adminService expects row with settings column.
        // Let's return the row structure to match what adminService expects or simplify.
        // adminService.getSiteSettings expects row.settings to be string or object.
        res.json({ data: rows });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/landing-content', async (req, res) => {
    try {
        const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 2');
        if (rows.length === 0) return res.json({ data: null });
        // Parse the settings column which contains landing content JSON
        const landingData = typeof rows[0].settings === 'string'
            ? JSON.parse(rows[0].settings)
            : rows[0].settings;
        res.json({ data: landingData });
    } catch (error) {
        console.error('[Landing Content API] Error:', error);
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/tips', async (req, res) => {
    try {
        const [rows] = await db.query("SELECT * FROM tips WHERE is_deleted = 0");
        res.json({ data: rows });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/broadcasts', requireAuth, async (req, res) => {
    try {
        const [rows] = await db.query("SELECT * FROM broadcasts WHERE is_deleted = 0");
        res.json({ data: rows });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Broadcast Dismissal / Read Receipt
app.post('/api/user/:uid/dismiss-broadcast', requireAuth, async (req, res) => {
    const { uid } = req.params;
    const { broadcastId } = req.body;

    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);

    try {
        // 1. Get current broadcast data
        const [rows] = await db.query("SELECT opened_by, read_count FROM broadcasts WHERE id = ?", [broadcastId]);

        if (rows.length === 0) return res.status(404).json({ error: 'Broadcast not found' });

        let openedBy = [];
        try {
            openedBy = typeof rows[0].opened_by === 'string' ? JSON.parse(rows[0].opened_by) : (rows[0].opened_by || []);
        } catch (e) {
            openedBy = [];
        }

        // 2. Check if already read
        if (!openedBy.includes(uid)) {
            openedBy.push(uid);

            // 3. Update DB
            await db.query(
                "UPDATE broadcasts SET read_count = read_count + 1, opened_by = ? WHERE id = ?",
                [JSON.stringify(openedBy), broadcastId]
            );
        }

        res.json({ success: true });
    } catch (error) {
        console.error("Dismiss Broadcast Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// --- UPLOAD ---
// Middleware already configured
app.post('/api/upload', upload.single('file'), (req, res) => {
    if (!req.file) return res.status(400).json({ error: 'No file' });
    const url = `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}`;
    res.json({ success: true, url, filename: req.file.filename });
});

// --- USER DATA ENDPOINTS (Dashboard/Sidebar) ---

app.get('/api/tutorial_steps', async (req, res) => {
    try {
        const [rows] = await db.query("SELECT * FROM tutorial_steps WHERE is_deleted = 0 ORDER BY step_order ASC");
        res.json({ data: rows });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/user/messages', requireAuth, async (req, res) => {
    try {
        const [rows] = await db.query("SELECT * FROM support_messages WHERE user_uid = ? ORDER BY timestamp DESC", [req.user.uid]);
        // Map to frontend expectation
        const messages = rows.map(r => ({
            id: r.id,
            text: r.message, // Map message -> text
            sender: r.sender,
            timestamp: Number(r.timestamp),
            status: r.status,
            subject: r.subject
        }));
        res.json({ data: messages });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Dashboard Stats Aggregation
app.get('/api/user/stats/dashboard', requireAuth, async (req, res) => {
    const { uid, days } = req.query;
    if (req.user.uid !== uid && req.user.role !== 'admin') return res.status(403);
    const limitDays = parseInt(days) || 30;

    try {
        // Initialize result object with date keys for last N days (USING PACIFIC TIME!)
        const result = {};
        const todayPT = getPacificDate(); // Get current Pacific Time
        for (let i = 0; i < limitDays; i++) {
            const d = new Date(todayPT);
            d.setDate(d.getDate() - i);
            const key = d.getFullYear() + '-' +
                String(d.getMonth() + 1).padStart(2, '0') + '-' +
                String(d.getDate()).padStart(2, '0');
            result[key] = {
                apiCalls: 0,
                trendSearches: 0,
                imagesGenerated: 0,
                videosCreated: 0,
                metadataGenerated: 0,
                imagesReviewed: 0,
                imagesEdited: 0,
                promptsGenerated: 0
            };
        }

        // 1. Get user's current usage_stats for today's data
        const [userRows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        if (userRows.length > 0) {
            const stats = typeof userRows[0].usage_stats === 'string'
                ? JSON.parse(userRows[0].usage_stats)
                : userRows[0].usage_stats;


            if (stats && stats.daily && stats.daily.date) {
                const todayKey = stats.daily.date;
                console.log('[Dashboard API] Today data check:', {
                    todayKey,
                    existsInResult: !!result[todayKey],
                    dailyStats: stats.daily,
                    resultKeys: Object.keys(result).slice(0, 5)
                });
                if (result[todayKey]) {
                    result[todayKey].apiCalls = stats.daily.apiCalls || 0;
                    result[todayKey].trendSearches = stats.daily.trendSearches || 0;
                    result[todayKey].imagesGenerated = stats.daily.imagesGenerated || 0;
                    result[todayKey].videosCreated = stats.daily.videosCreated || 0;
                    result[todayKey].metadataGenerated = stats.daily.metadataGenerated || 0;
                    result[todayKey].imagesReviewed = stats.daily.imagesReviewed || 0;
                    result[todayKey].imagesEdited = stats.daily.imagesEdited || 0;
                    result[todayKey].promptsGenerated = stats.daily.promptsGenerated || 0;
                } else {
                    console.log('[Dashboard API] ERROR: todayKey not in result!');
                }
            }
        }

        // 2. API Usage History from api_key_usage table (for past days)
        // Query by user_uid instead of key hashes to preserve historical data after keys are deleted
        // IMPORTANT: Calculate the cutoff date in Pacific Time, not MySQL server time
        const pacificNow = getPacificDate();
        const cutoffDate = new Date(pacificNow);
        cutoffDate.setDate(cutoffDate.getDate() - limitDays);
        const cutoffDateStr = cutoffDate.getFullYear() + '-' +
            String(cutoffDate.getMonth() + 1).padStart(2, '0') + '-' +
            String(cutoffDate.getDate()).padStart(2, '0');

        const [rows] = await db.query(`
            SELECT CAST(usage_date AS CHAR) as date_val, SUM(api_calls) as count 
            FROM api_key_usage 
            WHERE user_uid = ? AND usage_date >= ?
            GROUP BY usage_date
            ORDER BY usage_date ASC
        `, [uid, cutoffDateStr]);

        rows.forEach(row => {
            // row.date_val is already in YYYY-MM-DD format from the database
            const dateKey = row.date_val;
            if (result[dateKey]) {
                // Don't overwrite today's data from usage_stats
                const todayKey = getPacificYMD(); // Use Pacific Time instead of UTC!
                if (dateKey !== todayKey) {
                    result[dateKey].apiCalls = parseInt(row.count) || 0;
                }
            }
        });

        // 3. Asset Generation History from assets table
        // Calculate Pacific Time offset from UTC (in seconds)
        // Pacific Time is UTC-8 (PST) or UTC-7 (PDT)
        const pacificOffsetMs = getPacificDate().getTime() - new Date().getTime();
        const pacificOffsetSeconds = Math.floor(pacificOffsetMs / 1000);

        // Calculate cutoff timestamp in milliseconds
        const cutoffTimestamp = cutoffDate.getTime();

        const [assetRows] = await db.query(`
            SELECT CAST(DATE(FROM_UNIXTIME((created_at/1000) + ?)) AS CHAR) as date_val, type, COUNT(*) as count
            FROM assets
            WHERE user_uid = ? AND created_at >= ?
            GROUP BY date_val, type
            ORDER BY date_val ASC
        `, [pacificOffsetSeconds, uid, cutoffTimestamp]);

        assetRows.forEach(row => {
            // row.date_val is already in YYYY-MM-DD format from the database (Pacific Time)
            const dateKey = row.date_val;
            if (result[dateKey]) {
                const count = parseInt(row.count) || 0;
                // Map asset types to stat fields
                // OVERWRITE data from usage_stats with actual asset counts (Source of Truth)
                if (row.type === 'report') result[dateKey].trendSearches = count;
                else if (row.type === 'image') result[dateKey].imagesGenerated = count;
                else if (row.type === 'video') result[dateKey].videosCreated = count;
                else if (row.type === 'review') result[dateKey].imagesReviewed = count;
            }
        });

        res.json(result);
    } catch (error) {
        console.error("Dashboard Stats Error:", error);
        res.status(500).json({ error: error.message });
    }
});

// --- CRON JOBS ---

const resetDailyStats = async () => {
    console.log('[CRON] 🌙 Executing Pacific Midnight Reset...');
    const today = getPacificYMD();
    const newDaily = {
        date: today,
        apiCalls: 0,
        trendSearches: 0,
        imagesGenerated: 0,
        videosCreated: 0,
        metadataGenerated: 0,
        imagesReviewed: 0,
        imagesEdited: 0,
        promptsGenerated: 0
    };

    try {
        // Update ALL users: Reset daily stats in the JSON column
        // We use CAST(? AS JSON) to ensuring the stringified object is treated as a JSON object, not a string
        await db.query(`
            UPDATE users 
            SET usage_stats = JSON_SET(
                COALESCE(usage_stats, '{}'), 
                '$.daily', CAST(? AS JSON)
            )
        `, [JSON.stringify(newDaily)]);
        console.log(`[CRON] ✅ Reset daily credits for ALL users to date: ${today}`);
    } catch (e) {
        console.error('[CRON] ❌ Reset failed:', e);
    }
};

const scheduleDailyReset = () => {
    const now = new Date();
    // Calculate ms to next Midnight Pacific Time
    const ptNowStr = now.toLocaleString("en-US", { timeZone: "America/Los_Angeles" });
    const ptNow = new Date(ptNowStr);

    const ptMidnight = new Date(ptNow);
    ptMidnight.setHours(24, 0, 0, 0);

    let msUntil = ptMidnight.getTime() - ptNow.getTime();
    if (msUntil < 0) msUntil += 86400000; // Safety for edge cases

    console.log(`[CRON] 🕒 Next Daily Credit Reset in ${Math.round(msUntil / 1000 / 60)} minutes (Midnight PT)`);

    setTimeout(() => {
        resetDailyStats();
        scheduleDailyReset(); // Reschedule for tomorrow
    }, msUntil);
};

// Start the Scheduler
scheduleDailyReset();

// --- START ---
const PORT = process.env.PORT || 3001;
app.listen(PORT, () => {
    console.log(`Server running on port ${PORT} (MySQL Mode)`);
});
