
import pool from '../database/db.js';
import bcrypt from 'bcrypt';
import fetch from 'node-fetch'; // Fallback if global fetch missing, but usually global in recent Node

// Use global fetch if updated node, else might need import. 
// Assuming environment has it or running with node --experimental-fetch if older.

const API_URL = 'http://localhost:3001/api';

async function main() {
    console.log("Starting Verification...");

    const adminEmail = `admin_test_${Date.now()}@example.com`;
    const userEmail = `user_test_${Date.now()}@example.com`;
    const password = 'password123';
    const adminUid = `admin_${Date.now()}`;
    const userUid = `user_${Date.now()}`;

    try {
        // 1. Create Users directly in DB
        const hash = await bcrypt.hash(password, 10);

        await pool.query(
            'INSERT INTO users (uid, email, password_hash, role, is_deleted, is_banned) VALUES (?, ?, ?, ?, 0, 0)',
            [adminUid, adminEmail, hash, 'admin']
        );
        console.log("✅ Admin Created:", adminUid);

        await pool.query(
            'INSERT INTO users (uid, email, password_hash, role, is_deleted, is_banned) VALUES (?, ?, ?, ?, 0, 0)',
            [userUid, userEmail, hash, 'user']
        );
        console.log("✅ User Created:", userUid);

        // 2. Login as Admin
        const loginRes = await fetch(`${API_URL}/auth/login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email: adminEmail, password })
        });

        if (!loginRes.ok) throw new Error(`Login failed: ${loginRes.status}`);
        const loginData = await loginRes.json();
        const token = loginData.token;
        console.log("✅ Admin Logged In. Token acquired.");

        // 3. Test UPDATE (Change Role)
        console.log("Testing Update (Role)...");
        const updateRes = await fetch(`${API_URL}/admin/data/users/update`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}` },
            body: JSON.stringify({ id: userUid, updates: { role: 'admin' } }) // Client sends 'id', server should map to 'uid' if logic works
        });

        if (!updateRes.ok) {
            const err = await updateRes.json();
            throw new Error(`Update Failed: ${JSON.stringify(err)}`);
        }
        console.log("✅ Update API Call Success");

        // Verify DB
        const [rows] = await pool.query('SELECT role FROM users WHERE uid = ?', [userUid]);
        if (rows[0].role !== 'admin') throw new Error("DB verification failed: Role not updated!");
        console.log("✅ DB Verification Passed: Role is now admin");

        // 4. Test DELETE
        console.log("Testing Delete...");
        const deleteRes = await fetch(`${API_URL}/admin/data/users/delete`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}` },
            body: JSON.stringify({ id: userUid }) // Client sends 'id'
        });

        if (!deleteRes.ok) {
            const err = await deleteRes.json();
            throw new Error(`Delete Failed: ${JSON.stringify(err)}`);
        }
        console.log("✅ Delete API Call Success");

        // Verify DB
        // Check if row is gone
        const [delRows] = await pool.query('SELECT * FROM users WHERE uid = ?', [userUid]);
        if (delRows.length > 0) throw new Error("DB verification failed: User still exists!");
        console.log("✅ DB Verification Passed: User deleted");

    } catch (e) {
        console.error("❌ Verification Failed:", e);
    } finally {
        // Cleanup
        await pool.query('DELETE FROM users WHERE uid = ?', [adminUid]);
        await pool.query('DELETE FROM users WHERE uid = ?', [userUid]); // Just in case
        await pool.end();
        process.exit(0);
    }
}

main();
