
import db from '../database/db.js';

async function updateModelDefaults() {
    try {
        console.log("🔍 Fetching system settings...");
        const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 1');

        if (rows.length === 0) {
            console.error("❌ System settings not found!");
            process.exit(1);
        }

        let settings = rows[0].settings;
        if (typeof settings === 'string') settings = JSON.parse(settings);

        console.log("📊 Current Model Limits:", settings.modelLimits);

        // Define defaults for all new models
        const NEW_DEFAULTS = {
            // Thinking
            'flash_lite': 250,
            'flash_lite_2_0': 250,
            'flash_2_5': 250, // Alias
            'pro_2_5': 50,
            'gemini_2_5_deep_think': 10,

            // Creative
            'imagen_4_fast': 100,
            'imagen_4_std': 60,
            'imagen_4_ultra': 30,
            'gemini_2_0_flash_image': 100,
            'gemini_2_5_flash_image': 100,
            'gemini_3_pro_image': 20,

            // Video
            'veo_3_1_fast': 10,
            'veo_3_1': 5,
            'veo_3_0': 5,
            'veo': 10 // Alias
        };

        let updated = false;
        if (!settings.modelLimits) settings.modelLimits = {};

        for (const [key, val] of Object.entries(NEW_DEFAULTS)) {
            if (settings.modelLimits[key] === undefined) {
                console.log(`➕ Adding default for ${key}: ${val}`);
                settings.modelLimits[key] = val;
                updated = true;
            }
        }

        if (updated) {
            // Recalculate total credits per key based on these new limits?
            // Usually the admin dashboard saves 'creditsPerApiKey' as sum of limits.
            // Let's update that too for consistency.
            const total = Object.values(settings.modelLimits).reduce((a, b) => a + Number(b), 0);
            settings.creditsPerApiKey = total;
            console.log(`💰 New Credits Per API Key: ${total}`);

            await db.query('UPDATE system_settings SET settings = ? WHERE id = 1', [JSON.stringify(settings)]);
            console.log("✅ System settings updated successfully!");
        } else {
            console.log("✨ No updates needed (keys already exist).");
        }

    } catch (error) {
        console.error("Error:", error);
    } finally {
        process.exit();
    }
}

updateModelDefaults();
