import db, { checkConnection } from '../database/db.js';

async function setStaticModelLimits() {
    console.log("🔧 Setting Static Model Limits...");

    await checkConnection();

    try {
        // Fetch current settings
        const [rows] = await db.query('SELECT * FROM system_settings WHERE id = 1');
        if (rows.length === 0) {
            console.error("❌ No system settings found.");
            process.exit(1);
        }

        let settings = rows[0].settings;
        if (typeof settings === 'string') {
            settings = JSON.parse(settings);
        }

        // Static Model Limits - HARDCODED VALUES PER USER REQUEST
        const STATIC_MODEL_LIMITS = {
            // Thinking Models
            'flash_lite_2_5': 200,       // Gemini 2.5 Flash Lite
            'flash_2_5': 300,            // Gemini 2.5 Flash
            'pro_2_5': 50,               // Gemini 2.5 Pro
            'pro_3_0': 50,               // Gemini 3.0 Pro

            // Creative Models (keep existing or set defaults)
            'gemini_2_0_flash_image': 100,
            'gemini_2_5_flash_image': 100,
            'gemini_3_pro_image': 50,
            'imagen_4_fast': 300,
            'imagen_4_std': 200,
            'imagen_4_ultra': 50,

            // Video Models (keep existing or set defaults)
            'veo_3_1': 20,
            'veo_3_1_fast': 50,
            'veo_3_0': 10,
            'veo_2_0': 5
        };

        console.log("📊 Setting Model Limits:", STATIC_MODEL_LIMITS);

        // Set the static limits
        settings.modelLimits = STATIC_MODEL_LIMITS;

        // Save back to database
        await db.query(
            'UPDATE system_settings SET settings = ? WHERE id = 1',
            [JSON.stringify(settings)]
        );

        console.log("✅ Static Model Limits have been saved successfully!");
        console.log("\n📋 Updated Limits:");
        console.log("  Gemini 3.0 Pro: 50 credits");
        console.log("  Gemini 2.5 Pro: 50 credits");
        console.log("  Gemini 2.5 Flash: 300 credits");
        console.log("  Gemini 2.5 Flash Lite: 200 credits");

        process.exit(0);
    } catch (error) {
        console.error("❌ Error setting static limits:", error);
        process.exit(1);
    }
}

setStaticModelLimits();
