const fetch = require('node-fetch'); // You might need to install this or use built-in fetch if node 18+
const mysql = require('mysql2/promise');
require('dotenv').config();

// Config
const BASE_URL = 'http://localhost:3000'; // Adjust port if needed (user running on 3000?)
// Reading server.js output... running at ??
// User logs say "node server.js" running. Port is typically 3000 or 3001. 
// Frontend authService says API_URL = 'http://localhost:3001/api'.
const API_URL = 'http://localhost:3001/api';

async function main() {
    // 1. Database Connection to inspect directly
    const db = await mysql.createConnection({
        host: process.env.DB_HOST || 'localhost',
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || '',
        database: process.env.DB_NAME || 'stock_forge', // Adjust DB name
    });

    try {
        // 2. Get a User (admin/first user)
        const [users] = await db.query('SELECT uid, email FROM users LIMIT 1');
        if (users.length === 0) { console.log('No users.'); return; }
        const uid = users[0].uid;
        console.log(`Testing with user: ${uid} (${users[0].email})`);

        // Helper to check DB
        const checkDB = async (label) => {
            const [rows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
            const stats = typeof rows[0].usage_stats === 'string' ? JSON.parse(rows[0].usage_stats) : rows[0].usage_stats;
            console.log(`\n--- [${label}] DB State ---`);
            console.log('Daily:', JSON.stringify(stats.daily));
            console.log('Monthly:', JSON.stringify(stats.monthly));
            return stats;
        };

        // 3. Baseline
        await checkDB('BASELINE');

        // 4. Simulate Frontend "Review" Action
        // Action 1: Deduct Credits
        console.log('\n>>> Calling /deduct-credits...');
        // We need to bypass auth or use a token?
        // server.js checks `if (req.user.uid !== uid ...)`
        // We can't easily generate a valid token without the secret.
        // BUT we can use the `force_sync` approach of DIRECTLY modifying user stats via DB to simulate the "Write"?
        // NO, we want to test the ENDPOINT logic.

        // Options:
        // A. Login to get token (if we know password).
        // B. Temporarily disable auth in server.js (risky/slow).
        // C. Mock request using a local script that imports server.js app? (Hard with express).

        // Let's assume we can't call API easily without a token.
        // But wait! `authService.ts` says `localStorage.getItem('sf_token')`.
        // I can't read local storage from here.

        // Plan B: Write a script that USES THE DB connection to simulate the logic of the endpoint EXACTLY.
        // We can copy-paste the logic from server.js into this script and run it.
        // If the SCRIPT fails, it's the logic.
        // If the SCRIPT succeeds, then it's the Endpoint/Network/Race Condition.

        console.log("Simulating server logic directly...");

        // --- SIMULATED ENDPOINT LOGIC ---
        // Step 1: Read
        let [rows] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        let stats = typeof rows[0].usage_stats === 'string' ? JSON.parse(rows[0].usage_stats) : rows[0].usage_stats;

        // ... (Logic from server.js) ...
        const amount = 1;
        const statName = 'imagesReviewedTest';

        // ... Init ...
        if (!stats.monthly) stats.monthly = { month: new Date().toISOString().slice(0, 7) };
        const thisMonth = new Date().toISOString().slice(0, 7);
        if (stats.monthly.month !== thisMonth) {
            stats.monthly = { month: thisMonth }; // Simplified reset
        }

        // Increment
        stats.monthly[statName] = (stats.monthly[statName] || 0) + amount;

        console.log(`Simulated New Monthly State:`, stats.monthly);

        // Write
        const jsonString = JSON.stringify(stats);
        await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [jsonString, uid]);

        await checkDB('AFTER SIMULATION');

    } catch (e) {
        console.error(e);
    } finally {
        await db.end();
    }
}

main();
