
import db from '../database/db.js';

// Copy of the MAPPING from server.js (since we can't import it easily if it's not exported)
const MODEL_LIMIT_MAPPING = {
    // Thinking
    'flash_lite': 'gemini-2.5-flash-lite-preview-09-2025',
    'flash': 'gemini-2.5-flash',
    'pro': 'gemini-3-pro-preview',
    'pro_2_5': 'gemini-2.5-pro',

    // Aliases (Thinking)
    'flash_lite_2_5': 'gemini-2.5-flash-lite-preview-09-2025',
    'flash_2_5': 'gemini-2.5-flash',
    'pro_3_0': 'gemini-3-pro-preview',

    // Creative
    'imagen_4_fast': 'imagen-4.0-fast-generate-preview',
    'imagen_4_std': 'imagen-4.0-generate-001',
    'imagen_4_ultra': 'imagen-4.0-generate-ultra-preview',
    'gemini_2_0_flash_image': 'gemini-2.0-flash-image',
    'gemini_2_5_flash_image': 'gemini-2.5-flash-image',
    'gemini_3_pro_image': 'gemini-3-pro-image-preview',

    // Aliases (Creative)
    'imagen_fast': 'imagen-4.0-fast-generate-preview',
    'imagen_std': 'imagen-4.0-generate-001',
    'imagen_ultra': 'imagen-4.0-generate-ultra-preview',
    'flash_image_2_0': 'gemini-2.0-flash-image',
    'flash_image_2_5': 'gemini-2.5-flash-image',
    'pro_image_3_0': 'gemini-3-pro-image-preview',

    // Video
    'veo_3_1_fast': 'veo-3.1-fast-generate-preview',
    'veo_3_1': 'veo-3.1-generate-preview',
    'veo_3_0': 'veo-3.0-generate-preview',
    'veo_2_0': 'veo-2.0-generate-preview',

    // Aliases (Video)
    'veo': 'veo-3.1-fast-generate-preview',
    'veo_hq': 'veo-3.1-generate-preview'
};

async function recalculateLimits() {
    try {
        console.log("🔄 Starting Limit Recalculation...");

        // 1. Get Settings
        const [sRows] = await db.query('SELECT * FROM system_settings WHERE id = 1');
        let settings = (sRows[0] && sRows[0].settings) ? JSON.parse(sRows[0].settings) : {};
        if (typeof settings === 'string') settings = JSON.parse(settings);

        const modelLimits = settings.modelLimits || {};
        console.log("ℹ️  Using Model Settings:", modelLimits);

        // 2. Get All Users
        const [users] = await db.query('SELECT uid, email, usage_stats FROM users WHERE is_deleted = 0');
        console.log(`👥 Found ${users.length} users.`);

        for (const user of users) {
            // 3. Get Valid Keys for User
            const [keys] = await db.query('SELECT * FROM api_keys WHERE user_uid = ?', [user.uid]);
            const validKeys = keys.filter(k => (k.status === 'valid' || k.status === 'unchecked') && k.is_enabled !== 0);

            if (validKeys.length === 0) {
                console.log(`Skipping ${user.email} (0 valid keys)`);
                continue;
            }

            console.log(`Processing ${user.email} (${validKeys.length} keys)...`);

            // 4. Calculate New Limits
            const newLimits = {};
            for (const [sKey, mId] of Object.entries(MODEL_LIMIT_MAPPING)) {
                const calculated = (modelLimits[sKey] || 0) * validKeys.length;
                newLimits[mId] = Math.max(newLimits[mId] || 0, calculated);
            }

            // 5. Update User Stats
            let stats = typeof user.usage_stats === 'string' ? JSON.parse(user.usage_stats) : user.usage_stats;
            if (!stats) stats = {};

            stats.modelLimits = newLimits;
            // Also update apiCallsLimit just in case
            const creditsPerKey = settings.creditsPerApiKey || 2640;
            stats.apiCallsLimit = validKeys.length * creditsPerKey;

            await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), user.uid]);
            console.log(`✅ Updated ${user.email}`);
        }

        console.log("🎉 Recalculation Complete!");

    } catch (error) {
        console.error("Error:", error);
    } finally {
        process.exit();
    }
}

recalculateLimits();
