
import admin from 'firebase-admin';
import mysql from 'mysql2/promise';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';

// ESM compatibility
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Load environment variables from the root .env file
dotenv.config({ path: path.join(__dirname, '../.env') });

const serviceAccountPath = path.join(__dirname, '../serviceAccountKey.json');

if (!fs.existsSync(serviceAccountPath)) {
    console.error('Error: serviceAccountKey.json not found in root directory.');
    process.exit(1);
}

const serviceAccount = JSON.parse(fs.readFileSync(serviceAccountPath, 'utf8'));

admin.initializeApp({
    credential: admin.credential.cert(serviceAccount)
});

const firestore = admin.firestore();

// Database connection config
const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'stockforge'
};

async function migrate() {
    let connection;
    try {
        console.log('Connecting to MySQL...');
        connection = await mysql.createConnection(dbConfig);
        console.log('Connected.');

        // --- USERS ---
        console.log('Migrating Users...');
        const usersSnap = await firestore.collection('users').get();
        for (const doc of usersSnap.docs) {
            const u = doc.data();
            const uid = doc.id;

            // Handle timestamps safely
            const createdAt = u.createdAt || Date.now();
            const lastLogin = u.lastLogin || Date.now();

            // Prepare JSON fields
            const usageStats = JSON.stringify(u.usageStats || {});
            const preferences = JSON.stringify(u.preferences || {});
            const subscription = JSON.stringify(u.subscription || {}); // If object or string? 
            const assetsSnap = await firestore.collection(`users/${uid}/files`).get();
            if (!assetsSnap.empty) {
                // console.log(`  Migrating ${assetsSnap.size} assets for user ${uid}...`);
                for (const assetDoc of assetsSnap.docs) {
                    const a = assetDoc.data();
                    const navMetadata = JSON.stringify(a.navigationMetadata || {});
                    const aiMetadata = JSON.stringify(a.aiMetadata || {});

                    await connection.execute(`
                        INSERT INTO assets (id, user_uid, type, url, thumbnail_url, prompt, model, aspect_ratio, width, height, created_at, is_favorite, is_deleted, is_public, navigation_metadata, ai_metadata)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ON DUPLICATE KEY UPDATE is_deleted=VALUES(is_deleted)
                    `, [
                        assetDoc.id,
                        uid,
                        a.type || 'image',
                        a.url || '',
                        a.thumbnailUrl || '', // Note: db schema has thumbnail_url, Firestore usually thumbnailUrl
                        a.prompt || '',
                        a.model || '',
                        a.aspectRatio || '',
                        a.width || 0,
                        a.height || 0,
                        new Date(a.createdAt || Date.now()),
                        a.isFavorite ? 1 : 0,
                        a.isDeleted ? 1 : 0,
                        a.isPublic ? 1 : 0,
                        navMetadata,
                        aiMetadata
                    ]);
                }
            }
        }
        console.log(`Migrated ${usersSnap.size} users and their assets.`);

        // --- TIPS ---
        console.log('Migrating Tips...');
        const tipsSnap = await firestore.collection('admin_data/tips/items').get();
        for (const doc of tipsSnap.docs) {
            const t = doc.data();
            await connection.execute(`
                INSERT INTO tips (id, title, text, category, created_at, is_deleted)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE title=VALUES(title)
            `, [
                doc.id, t.title, t.text, t.category, new Date(t.createdAt || Date.now()), t.isDeleted ? 1 : 0
            ]);
        }

        // --- TUTORIAL STEPS ---
        console.log('Migrating Tutorial Steps...');
        const stepsSnap = await firestore.collection('admin_data/tutorial/steps').get();
        for (const doc of stepsSnap.docs) {
            const s = doc.data();
            await connection.execute(`
                INSERT INTO tutorial_steps (id, title, description, action_label, action_link, step_order, is_deleted)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE title=VALUES(title)
            `, [
                doc.id, s.title, s.description, s.actionLabel, s.actionLink, s.order || 0, s.isDeleted ? 1 : 0
            ]);
        }

        // --- BROADCASTS ---
        console.log('Migrating Broadcasts...');
        const broadcastsSnap = await firestore.collection('admin_data/broadcasts/items').get();
        for (const doc of broadcastsSnap.docs) {
            const b = doc.data();
            await connection.execute(`
                INSERT INTO broadcasts (id, title, message, type, target_audience, recipients, sent_at, read_count, open_rate, is_deleted)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE title=VALUES(title)
            `, [
                doc.id,
                b.title,
                b.message,
                b.type,
                b.targetAudience,
                b.recipients || 0,
                new Date(b.sentAt || Date.now()),
                b.readCount || 0,
                b.openRate || 0,
                b.isDeleted ? 1 : 0
            ]);
        }

        // --- BUG REPORTS ---
        console.log('Migrating Bug Reports...');
        const bugsSnap = await firestore.collection('bug_reports').get();
        for (const doc of bugsSnap.docs) {
            const b = doc.data();
            await connection.execute(`
                INSERT INTO bug_reports (id, user_uid, type, title, description, status, timestamp, is_deleted)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE status=VALUES(status)
            `, [
                doc.id,
                b.userId || b.user_uid || 'anonymous', // Handle field name variation
                b.type || 'bug',
                b.title || 'Untitled',
                b.description || '',
                b.status || 'open',
                new Date(b.timestamp || Date.now()),
                b.isDeleted ? 1 : 0
            ]);
        }

        // --- SYSTEM SETTINGS ---
        console.log('Migrating System Settings...');
        const settingsDoc = await firestore.doc('system/settings').get();
        if (settingsDoc.exists) {
            const s = settingsDoc.data();
            // Just insert as id=1? Or match schema expectations.
            // My schema: id INT PRIMARY KEY AUTO_INCREMENT, settings JSON, updated_at TIMESTAMP
            // I'll insert into row 1.
            await connection.execute(`
                INSERT INTO system_settings (id, settings)
                VALUES (1, ?)
                ON DUPLICATE KEY UPDATE settings=VALUES(settings)
            `, [JSON.stringify(s)]);
        }

        // --- LANDING CONTENT ---
        const landingDoc = await firestore.doc('system/landing').get();
        if (landingDoc.exists) {
            const l = landingDoc.data();
            // Insert as id=2?
            await connection.execute(`
                INSERT INTO system_settings (id, settings)
                VALUES (2, ?)
                ON DUPLICATE KEY UPDATE settings=VALUES(settings)
            `, [JSON.stringify(l)]);
        }

        console.log('Migration completed successfully!');

    } catch (error) {
        console.error('Migration failed:', error);
    } finally {
        if (connection) await connection.end();
        process.exit();
    }
}

migrate();
