import mysql from 'mysql2/promise';
import fs from 'fs/promises';
import path from 'path';
import dotenv from 'dotenv';
import { fileURLToPath } from 'url';

dotenv.config();

const __dirname = path.dirname(fileURLToPath(import.meta.url));

async function init() {
    console.log("Initializing database...");

    // Connect without database selected to create it
    const connection = await mysql.createConnection({
        host: process.env.DB_HOST || 'localhost',
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || ''
    });

    try {
        const schemaPath = path.join(__dirname, '../database/schema.sql');
        const sql = await fs.readFile(schemaPath, 'utf8');

        // Remove comments? Or just split.
        // Simple splitter for ;
        // Ideally we use a multi-statement query support but let's try manual split
        const statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0);

        for (const statement of statements) {
            try {
                await connection.query(statement);
                // console.log("Executed:", statement.substring(0, 50) + "...");
            } catch (err) {
                // Ignore "Database exists" or similar non-fatal if possible, but schema has IF NOT EXISTS
                console.warn("Statement warning:", err.message);
                // console.warn("Query:", statement);
            }
        }

        console.log("✅ Database initialized successfully!");
    } catch (error) {
        console.error("❌ Failed to init database:", error);
    } finally {
        await connection.end();
    }
}

init();
