const mysql = require('mysql2/promise');
const crypto = require('crypto');
const dotenv = require('dotenv');
dotenv.config();

const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'stockforge_v2'
};

async function sync() {
    console.log('Connecting to DB...');
    const conn = await mysql.createConnection(dbConfig);

    try {
        const [users] = await conn.query('SELECT uid, email, usage_stats FROM users LIMIT 1');
        if (!users.length) return;
        const user = users[0];
        const uid = user.uid;

        console.log(`Syncing user: ${user.email}`);

        // --- LOGIC FROM SERVER.JS ---
        let stats = typeof user.usage_stats === 'string' ? JSON.parse(user.usage_stats) : user.usage_stats;
        if (!stats) stats = {};

        const getPacificYMD = () => {
            const now = new Date();
            const ptString = now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' });
            const ptDate = new Date(ptString);
            const year = ptDate.getFullYear();
            const month = String(ptDate.getMonth() + 1).padStart(2, '0');
            const day = String(ptDate.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        };
        const pacificToday = getPacificYMD();

        // Ensure daily object exists
        if (!stats.daily) stats.daily = { date: pacificToday, apiCalls: 0 };

        // Get Active Keys
        const [kRows] = await conn.query('SELECT key_value FROM api_keys WHERE user_uid = ? AND is_enabled = 1', [uid]);

        let calculatedUsage = 0;
        if (kRows.length > 0) {
            const hashes = kRows.map(k => crypto.createHash('sha256').update(k.key_value).digest('hex'));
            const placeholders = hashes.map(() => '?').join(',');

            console.log('Querying usage for date:', pacificToday);
            const [usageRows] = await conn.query(
                `SELECT SUM(api_calls) as total FROM api_key_usage WHERE usage_date = ? AND key_hash IN (${placeholders})`,
                [pacificToday, ...hashes]
            );
            calculatedUsage = Number(usageRows[0].total) || 0;
            console.log('Calculated Usage:', calculatedUsage);
        }

        let updated = false;

        // 1. Sync Top Bar Stat
        if (stats.apiCallsThisMonth !== calculatedUsage) {
            console.log(`Fixing apiCallsThisMonth: ${stats.apiCallsThisMonth} -> ${calculatedUsage}`);
            stats.apiCallsThisMonth = calculatedUsage;
            updated = true;
        }

        // 2. Sync Dashboard Stat
        if (stats.daily.date !== pacificToday) {
            console.log(`Date Mismatch! Resetting daily stats to ${pacificToday}`);
            stats.history = stats.history || {};
            stats.history.lastDay = { ...stats.daily };

            // Reset helper
            const resetStatsObj = (obj) => {
                Object.keys(obj).forEach(k => {
                    if (typeof obj[k] === 'number') obj[k] = 0;
                });
            };
            resetStatsObj(stats.daily);
            stats.daily.date = pacificToday;
            stats.daily.apiCalls = calculatedUsage;
            updated = true;
        } else {
            if (stats.daily.apiCalls !== calculatedUsage) {
                console.log(`Fixing daily.apiCalls: ${stats.daily.apiCalls} -> ${calculatedUsage}`);
                stats.daily.apiCalls = calculatedUsage;
                updated = true;
            }
        }

        if (updated) {
            await conn.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), uid]);
            console.log('✅ Update saved to DB.');
        } else {
            console.log('No changes needed.');
        }

    } catch (e) {
        console.error(e);
    } finally {
        await conn.end();
    }
}

sync();
