
import db from '../database/db.js';

async function fixPhantomCredits() {
    try {
        console.log("🔍 Scanning for users with phantom credits...");

        // Find users with 0 keys but > 0 apiCallsLimit
        // This query requires joining or subqueries. 
        // Simplest: Get all users, check their key count.

        const [users] = await db.query("SELECT uid, email, usage_stats FROM users");

        for (const user of users) {
            const [keys] = await db.query("SELECT id FROM api_keys WHERE user_uid = ? AND is_enabled = 1 AND (status = 'valid' OR status = 'unchecked')", [user.uid]);

            let stats = typeof user.usage_stats === 'string' ? JSON.parse(user.usage_stats) : user.usage_stats;
            if (!stats) stats = {};

            const limit = stats.apiCallsLimit || 0;
            const expectedCount = keys.length;
            // We can't know exact expected limit without settings, but if count is 0, limit MUST be 0.

            if (expectedCount === 0 && limit > 0) {
                console.log(`⚠️ User ${user.email} (${user.uid}) has 0 keys but ${limit} credits. FIXING...`);

                stats.apiCallsLimit = 0;
                stats.modelLimits = {}; // Also clear model limits

                await db.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), user.uid]);
                console.log(`✅ Fixed user ${user.email}.`);
            }
        }

        console.log("🎉 Phantom credit scan complete.");
        process.exit();

    } catch (error) {
        console.error("Error:", error);
        process.exit(1);
    }
}

fixPhantomCredits();
