
import { createConnection } from 'mysql2/promise';
import dotenv from 'dotenv';
import { fileURLToPath } from 'url';
import { dirname, resolve } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// Load environment variables from parent directory
dotenv.config({ path: resolve(__dirname, '../.env') });

const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'stockforge_v5'
};

const fixStats = async () => {
    let connection;
    try {
        console.log('Connecting to database...');
        connection = await createConnection(dbConfig);
        console.log('Connected.');

        // 1. Get all users
        const [users] = await connection.query('SELECT uid, usage_stats FROM users WHERE is_deleted = 0');
        console.log(`Found ${users.length} users.`);

        const today = new Date().toISOString().split('T')[0];
        const thisMonth = new Date().toISOString().slice(0, 7);

        for (const user of users) {
            let stats = typeof user.usage_stats === 'string' ? JSON.parse(user.usage_stats) : user.usage_stats;
            if (!stats) stats = {};

            // Ensure Structure
            if (!stats.daily) stats.daily = { date: today };
            if (!stats.monthly) stats.monthly = { month: thisMonth };
            if (!stats.history) stats.history = { lastDay: {}, lastMonth: {} };

            // Reset daily/monthly if date mismatch (Safety)
            if (stats.daily.date !== today) {
                stats.daily = { date: today, trendSearches: 0, apiCalls: 0, imagesGenerated: 0 };
            }
            if (stats.monthly.month !== thisMonth) {
                stats.monthly = { month: thisMonth, trendSearches: 0, apiCalls: 0, imagesGenerated: 0 };
            }

            // --- RECALCULATE FROM SOURCE OF TRUTH ---

            // 1. Trend Searches (Assets type='report')
            // Daily
            const [dailyTrends] = await connection.query(
                `SELECT COUNT(*) as count FROM assets WHERE user_uid = ? AND type = 'report' AND DATE(FROM_UNIXTIME(created_at/1000)) = ?`,
                [user.uid, today]
            );

            // Monthly
            const [monthlyTrends] = await connection.query(
                `SELECT COUNT(*) as count FROM assets WHERE user_uid = ? AND type = 'report' AND DATE_FORMAT(FROM_UNIXTIME(created_at/1000), '%Y-%m') = ?`,
                [user.uid, thisMonth]
            );

            // 2. Images Generated (Assets type='image')
            const [dailyImages] = await connection.query(
                `SELECT COUNT(*) as count FROM assets WHERE user_uid = ? AND type = 'image' AND DATE(FROM_UNIXTIME(created_at/1000)) = ?`,
                [user.uid, today]
            );

            const [monthlyImages] = await connection.query(
                `SELECT COUNT(*) as count FROM assets WHERE user_uid = ? AND type = 'image' AND DATE_FORMAT(FROM_UNIXTIME(created_at/1000), '%Y-%m') = ?`,
                [user.uid, thisMonth]
            );

            // 3. API Calls (from api_key_usage)
            let apiCallsToday = 0;
            let apiCallsMonth = 0;

            const [keys] = await connection.query('SELECT key_value FROM api_keys WHERE user_uid = ?', [user.uid]);
            if (keys.length > 0) {
                const importCrypto = await import('crypto');
                const hashes = keys.map(k => k.key_value ? importCrypto.createHash('sha256').update(k.key_value.trim()).digest('hex') : null).filter(h => h);

                if (hashes.length > 0) {
                    const placeholders = hashes.map(() => '?').join(',');
                    const [dRows] = await connection.query(
                        `SELECT SUM(api_calls) as total FROM api_key_usage WHERE usage_date = ? AND key_hash IN (${placeholders})`,
                        [today, ...hashes]
                    );
                    apiCallsToday = Number(dRows[0].total || 0);

                    const [mRows] = await connection.query(
                        `SELECT SUM(api_calls) as total FROM api_key_usage WHERE usage_date LIKE ? AND key_hash IN (${placeholders})`,
                        [`${thisMonth}%`, ...hashes]
                    );
                    apiCallsMonth = Number(mRows[0].total || 0);
                }
            }

            // --- APPLY UPDATES ---
            stats.daily.trendSearches = dailyTrends[0].count;
            stats.monthly.trendSearches = monthlyTrends[0].count;

            stats.daily.imagesGenerated = dailyImages[0].count;
            stats.monthly.imagesGenerated = monthlyImages[0].count;

            if (apiCallsToday > 0) stats.daily.apiCalls = apiCallsToday;

            // Update Legacy & Monthly
            stats.apiCallsThisMonth = apiCallsMonth; // Or accumulate? Let's trust usage table as source of truth.
            if (!stats.monthly.apiCalls) stats.monthly.apiCalls = apiCallsMonth;
            else stats.monthly.apiCalls = Math.max(stats.monthly.apiCalls, apiCallsMonth);

            console.log(`User ${user.uid}: Trends(D:${stats.daily.trendSearches}), API(D:${stats.daily.apiCalls})`);

            await connection.query('UPDATE users SET usage_stats = ? WHERE uid = ?', [JSON.stringify(stats), user.uid]);
        }

        console.log('Finished recalculating stats.');

    } catch (e) {
        console.error('Error:', e);
    } finally {
        if (connection) await connection.end();
    }
};

fixStats();
