// Quick diagnostic script to test authentication
// Run with: node scripts/diagnose_auth.js

const token = process.argv[2];

if (!token) {
    console.log('\n❌ Usage: node scripts/diagnose_auth.js <YOUR_JWT_TOKEN>');
    console.log('\nTo get your token:');
    console.log('1. Open browser console (F12)');
    console.log('2. Run: localStorage.getItem("sf_token")');
    console.log('3. Copy the token (without quotes)');
    console.log('4. Run: node scripts/diagnose_auth.js YOUR_TOKEN_HERE\n');
    process.exit(1);
}

console.log('\n🔍 Testing Authentication...\n');

const testEndpoints = [
    '/api/auth/me',
    '/api/user/profile?uid=me',
    '/api/settings',
];

async function testAuth() {
    for (const endpoint of testEndpoints) {
        try {
            const response = await fetch(`http://localhost:3001${endpoint}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            });

            const status = response.status;
            const statusText = response.statusText;

            if (status === 200) {
                console.log(`✅ ${endpoint}`);
                console.log(`   Status: ${status} ${statusText}\n`);
            } else if (status === 401) {
                console.log(`❌ ${endpoint}`);
                console.log(`   Status: 401 Unauthorized`);
                console.log(`   → Token is invalid or expired\n`);
            } else if (status === 403) {
                console.log(`❌ ${endpoint}`);
                console.log(`   Status: 403 Forbidden`);
                console.log(`   → Token is valid but lacks permissions\n`);
            } else {
                console.log(`⚠️  ${endpoint}`);
                console.log(`   Status: ${status} ${statusText}\n`);
            }

            // Try to read response body
            const text = await response.text();
            if (text && status !== 200) {
                try {
                    const json = JSON.parse(text);
                    if (json.error) {
                        console.log(`   Error: ${json.error}\n`);
                    }
                } catch (e) {
                    // Not JSON, ignore
                }
            }

        } catch (error) {
            console.log(`❌ ${endpoint}`);
            console.log(`   Network Error: ${error.message}\n`);
        }
    }

    console.log('\n📊 Summary:');
    console.log('If you see 401 errors: Token is invalid/expired → Logout and login again');
    console.log('If you see 403 errors: Token is valid but lacks permissions → Check user role');
    console.log('If you see Network errors: Backend server is not running → Run: node server.js\n');
}

testAuth().catch(console.error);
