
const mysql = require('mysql2/promise');
const { createHash } = require('crypto');

// DB Config (Match your local setup)
const dbConfig = {
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'stockforge_db' // guessing, or check db.js
};

// Check db.js for real config
const fs = require('fs');
const path = require('path');
const dbPath = path.join(__dirname, 'database', 'db.js');

async function run() {
    console.log('Reading db config...');
    // We'll just define it or try to require db.js if exported
    // Simplest is to assume standard local xampp defaults or read from file

    // Connect
    const conn = await mysql.createConnection({
        host: 'localhost',
        user: 'root',
        password: '',
        database: 'stockforge' // Commonly used in this project
    });

    console.log('Connected to DB.');

    // 1. Get Users
    const [users] = await conn.query('SELECT uid, email, usage_stats FROM users');
    console.log(`Found ${users.length} users.`);

    for (const u of users) {
        console.log(`\nUser: ${u.email} (${u.uid})`);
        console.log('Stats (JSON):', typeof u.usage_stats === 'string' ? u.usage_stats : JSON.stringify(u.usage_stats));

        // 2. Get API Keys
        const [keys] = await conn.query('SELECT * FROM api_keys WHERE user_uid = ?', [u.uid]);
        console.log('Keys:', keys.map(k => `${k.id} (${k.key_value ? k.key_value.substring(0, 5) + '...' : 'NULL'})`));

        // 3. Get Usage
        const [usage] = await conn.query('SELECT * FROM api_key_usage WHERE user_uid = ?', [u.uid]);
        console.log('Usage (by UID):', usage);

        // 4. Check IDs by Hash
        for (const k of keys) {
            if (k.key_value) {
                // Untrimmed Hash
                const rawHash = createHash('sha256').update(k.key_value).digest('hex');
                // Trimmed Hash
                const trimHash = createHash('sha256').update(k.key_value.trim()).digest('hex');

                console.log(`Key ${k.id} RawHash: ${rawHash.substring(0, 8)}... TrimHash: ${trimHash.substring(0, 8)}...`);

                const [hashUsage] = await conn.query('SELECT * FROM api_key_usage WHERE key_hash IN (?)', [[rawHash, trimHash]]);
                console.log(`Usage for Key ${k.id} (by Hash):`, hashUsage);
            }
        }
    }

    await conn.end();
}

run().catch(console.error);
