const mysql = require('mysql2/promise');
const dotenv = require('dotenv');
dotenv.config();

const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'stockforge_v2'
};

async function check() {
    console.log('Connecting to DB...');
    const conn = await mysql.createConnection(dbConfig);

    try {
        // Get the first user (usually the one in the screenshot)
        const [users] = await conn.query('SELECT uid, email, usage_stats FROM users LIMIT 1');
        if (!users.length) {
            console.log('No users found.');
            return;
        }

        const user = users[0];
        console.log(`Checking user: ${user.email} (${user.uid})`);

        let stats = user.usage_stats;
        if (typeof stats === 'string') stats = JSON.parse(stats);

        console.log('--- USER STATS (JSON) ---');
        console.log('daily.apiCalls:', stats.daily?.apiCalls);
        console.log('apiCallsThisMonth:', stats.apiCallsThisMonth);
        console.log('Full Daily:', JSON.stringify(stats.daily, null, 2));

        // Get today's Pacific Date
        const getPacificYMD = () => {
            const now = new Date();
            const ptString = now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' });
            const ptDate = new Date(ptString);
            const year = ptDate.getFullYear();
            const month = String(ptDate.getMonth() + 1).padStart(2, '0');
            const day = String(ptDate.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        };
        const today = getPacificYMD();
        console.log('Pacific Date Today:', today);

        // Check api_key_usage
        console.log('--- API KEY USAGE (DB TABLE) ---');
        const [keys] = await conn.query('SELECT key_value FROM api_keys WHERE user_uid = ?', [user.uid]);
        const keyVal = keys[0]?.key_value; // Assuming first key

        // We need hash
        const crypto = require('crypto');
        const hashes = keys.map(k => crypto.createHash('sha256').update(k.key_value).digest('hex'));

        if (hashes.length > 0) {
            const placeholders = hashes.map(() => '?').join(',');
            const [usage] = await conn.query(
                `SELECT * FROM api_key_usage WHERE key_hash IN (${placeholders}) AND usage_date = ?`,
                [today]
            );
            console.log('Usage Rows for Today:', usage);

            const total = usage.reduce((acc, row) => acc + row.api_calls, 0);
            console.log('Total Calculated from Table:', total);
        } else {
            console.log('No API keys found for user.');
        }

    } catch (e) {
        console.error(e);
    } finally {
        await conn.end();
    }
}

check();
