
const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

// Load DB config from server.js or define directly if simple
// Assuming similar config as server.js or local dev defaults
const DB_CONFIG = {
    host: 'localhost',
    user: 'root', // Adjust if needed or prompt user? Assuming standard local dev
    password: '',
    database: 'stockforge_db' // Inferred from context? Need to verify.
};

// BETTER: Just rely on the existing db connection logic if possible, 
// or tell the user to run this with their DB credentials.
// Since I can't interactively prompt, I'll assume standard defaults or try to read from a config file.
// For now, I will create a script that uses the existing `database/db.js` if it exists, or just basic SQL.

// Let's look for db config. 
// I'll create a script that tries to run the ALTER TABLE command.

(async () => {
    try {
        // Try to connect using environment variables or defaults
        const connection = await mysql.createConnection({
            host: process.env.DB_HOST || 'localhost',
            user: process.env.DB_USER || 'root',
            password: process.env.DB_PASSWORD || '',
            database: process.env.DB_NAME || 'stockforge' // Most likely name
        });

        console.log('Connected to database.');

        // Add column if not exists
        try {
            await connection.query(`
                ALTER TABLE stock_sites 
                ADD COLUMN is_visible_market BOOLEAN DEFAULT 1;
            `);
            console.log("Successfully added 'is_visible_market' column.");
        } catch (e) {
            if (e.code === 'ER_DUP_FIELDNAME') {
                console.log("'is_visible_market' column already exists.");
            } else {
                throw e;
            }
        }

        // Populate it with current is_visible values to maintain state?
        // User said "standalone", but good defaults help.
        // Let's init it to 1 (Visible) as per default, or copy is_visible?
        // "make for every this 2 pages settings , standalone!"
        // Copying is_visible seems safest start.
        await connection.query('UPDATE stock_sites SET is_visible_market = is_visible');
        console.log("Initialized 'is_visible_market' with existing 'is_visible' values.");

        await connection.end();
        process.exit(0);
    } catch (error) {
        console.error('Migration failed:', error);
        process.exit(1);
    }
})();
