const jwt = require('jsonwebtoken');
const dotenv = require('dotenv');
const mysql = require('mysql2/promise');

dotenv.config();

const API_URL = 'http://localhost:3001/api/user/stats/increment';
const SECRET = process.env.JWT_SECRET || 'fallback_secret_dev_only_change_in_prod';
const UID = "34ad9630-ddfb-4773-a6d0-438b72c85c9e";

async function diagnoseIncrement() {
    console.log("=== DIAGNOSTIC TEST ===\n");

    // 1. Check DB Before
    const db = await mysql.createPool({
        host: process.env.DB_HOST || 'localhost',
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || '',
        database: process.env.DB_NAME || 'stockforge'
    });

    const [rows] = await db.query("SELECT usage_stats FROM users WHERE uid = ?", [UID]);
    const beforeStats = typeof rows[0].usage_stats === 'string' ? JSON.parse(rows[0].usage_stats) : rows[0].usage_stats;
    console.log("BEFORE (DB):", JSON.stringify(beforeStats.daily));

    // 2. Make API Call
    const token = jwt.sign({ uid: UID, role: 'admin' }, SECRET, { expiresIn: '1h' });
    console.log("\nCalling API...");

    const res = await fetch(API_URL, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
            uid: UID,
            statName: 'trendSearches',
            amount: 1,
            modelId: 'gemini-2.0-flash-thinking-exp'
        })
    });

    console.log("Status:", res.status);
    const responseText = await res.text();

    if (!res.ok) {
        console.error("ERROR Response:", responseText);
        return;
    }

    const responseData = JSON.parse(responseText);
    console.log("API returned daily:", JSON.stringify(responseData.stats.daily));

    // 3. Check DB After
    const [rows2] = await db.query("SELECT usage_stats FROM users WHERE uid = ?", [UID]);
    const afterStats = typeof rows2[0].usage_stats === 'string' ? JSON.parse(rows2[0].usage_stats) : rows2[0].usage_stats;
    console.log("\nAFTER (DB):", JSON.stringify(afterStats.daily));

    // 4. Compare
    const beforeTrend = beforeStats.daily?.trendSearches || 0;
    const afterTrend = afterStats.daily?.trendSearches || 0;

    console.log(`\n=== RESULT ===`);
    console.log(`Trend Searches: ${beforeTrend} → ${afterTrend}`);

    if (afterTrend > beforeTrend) {
        console.log("✅ SUCCESS: Stats incremented correctly!");
    } else {
        console.log("❌ FAIL: Stats DID NOT increment!");
        console.log("\nFull daily object comparison:");
        console.log("Before:", beforeStats.daily);
        console.log("After:", afterStats.daily);
    }

    process.exit();
}

diagnoseIncrement().catch(e => {
    console.error("FATAL ERROR:", e);
    process.exit(1);
});
