
import db from './database/db.js';

const getPacificDate = () => {
    const now = new Date();
    return new Date(now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' }));
};

const getPacificYMD = () => {
    const ptDate = getPacificDate();
    const year = ptDate.getFullYear();
    const month = String(ptDate.getMonth() + 1).padStart(2, '0');
    const day = String(ptDate.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
};

async function run() {
    try {
        console.log('--- FINDING ACTIVE USERS ---');
        const [users] = await db.query('SELECT uid, email, role, usage_stats FROM users');

        for (const u of users) {
            console.log(`\nUser: ${u.email} [${u.role}] (uid: ${u.uid})`);
            let stats = u.usage_stats;
            if (typeof stats === 'string') stats = JSON.parse(stats);
            console.log('Stats:', JSON.stringify(stats, null, 2));

            const [assets] = await db.query('SELECT type, COUNT(*) as count FROM assets WHERE user_uid = ? GROUP BY type', [u.uid]);
            console.log('Asset Counts:', assets);

            // Check for recent assets
            const [recent] = await db.query('SELECT id, type, created_at, LENGTH(metadata) as meta_len FROM assets WHERE user_uid = ? ORDER BY created_at DESC LIMIT 3', [u.uid]);
            if (recent.length > 0) {
                console.table(recent.map(r => ({ ...r, created_at: new Date(Number(r.created_at)).toLocaleString() })));
            }

            // CHECK AGGREGATION
            const pacificOffsetMs = getPacificDate().getTime() - new Date().getTime();
            const pacificOffsetSeconds = Math.floor(pacificOffsetMs / 1000);
            const cutoffDate = new Date(getPacificDate());
            cutoffDate.setDate(cutoffDate.getDate() - 30);
            const cutoffTimestamp = cutoffDate.getTime();

            const [aggRows] = await db.query(`
                SELECT CAST(DATE(FROM_UNIXTIME((created_at/1000) + ?)) AS CHAR) as date_val, type, COUNT(*) as count
                FROM assets
                WHERE user_uid = ? AND created_at >= ?
                GROUP BY date_val, type
            `, [pacificOffsetSeconds, u.uid, cutoffTimestamp]);
            console.log('Aggregation Results:', aggRows);
        }

    } catch (e) {
        console.error(e);
    } finally {
        process.exit();
    }
}

run();
