
import db from './database/db.js';

// Mock getPacificDate from server.js
const getPacificDate = () => {
    const now = new Date();
    return new Date(now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' }));
};

const getPacificYMD = () => {
    const ptDate = getPacificDate();
    const year = ptDate.getFullYear();
    const month = String(ptDate.getMonth() + 1).padStart(2, '0');
    const day = String(ptDate.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
};

async function run() {
    try {
        console.log('--- TIME DEBUG ---');
        const now = new Date();
        const pt = getPacificDate();
        console.log('Node System Time:', now.toString());
        console.log('Calculated PT:', pt.toString());

        const pacificOffsetMs = pt.getTime() - now.getTime();
        const pacificOffsetSeconds = Math.floor(pacificOffsetMs / 1000);
        console.log('Offset (ms):', pacificOffsetMs);
        console.log('Offset (seconds):', pacificOffsetSeconds);

        console.log('\n--- DB CONNECTION ---');
        // Fetch a user
        const [users] = await db.query('SELECT uid, email FROM users LIMIT 1');
        if (users.length === 0) {
            console.log('No users found');
            process.exit(0);
        }
        const uid = users[0].uid;
        console.log(`Using User: ${users[0].email} (${uid})`);

        console.log('\n--- RAW ASSETS (Last 5) ---');
        const [assets] = await db.query('SELECT id, type, created_at, FROM_UNIXTIME(created_at/1000) as sql_time FROM assets WHERE user_uid = ? ORDER BY created_at DESC LIMIT 5', [uid]);
        console.table(assets.map(a => ({
            ...a,
            created_at_date: new Date(Number(a.created_at)).toLocaleString()
        })));

        console.log('\n--- AGGREGATION QUERY ---');
        const cutoffDate = new Date(pt);
        cutoffDate.setDate(cutoffDate.getDate() - 30);
        const cutoffTimestamp = cutoffDate.getTime();

        console.log('Cutoff Timestamp:', cutoffTimestamp);
        console.log('Running Query with Offset:', pacificOffsetSeconds);

        const [assetRows] = await db.query(`
            SELECT DATE(FROM_UNIXTIME((created_at/1000) + ?)) as date_val, type, COUNT(*) as count
            FROM assets
            WHERE user_uid = ? AND created_at >= ?
            GROUP BY date_val, type
            ORDER BY date_val ASC
        `, [pacificOffsetSeconds, uid, cutoffTimestamp]);

        console.table(assetRows);

        // Check date match
        const todayKey = getPacificYMD();
        console.log('Today Key (JS):', todayKey);
        const match = assetRows.find(r => r.date_val === todayKey && r.type === 'report');
        console.log('Found Report for Today in Aggregation?', match);

        const [userStats] = await db.query('SELECT usage_stats FROM users WHERE uid = ?', [uid]);
        if (userStats.length > 0) {
            console.log('\n--- USAGE STATS (JSON) ---');
            let s = userStats[0].usage_stats;
            // Parse if string
            if (typeof s === 'string') s = JSON.parse(s);
            console.dir(s, { depth: null });
        }

    } catch (e) {
        console.error(e);
    } finally {
        process.exit();
    }
}

run();
