-- Database Creation
CREATE DATABASE IF NOT EXISTS stockforge;
USE stockforge;

-- Users Table
CREATE TABLE IF NOT EXISTS users (
    uid VARCHAR(128) PRIMARY KEY,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255), -- Nullable for social login users initially
    display_name VARCHAR(255),
    photo_url TEXT,
    role VARCHAR(50) DEFAULT 'user', -- 'user', 'admin'
    country VARCHAR(100) DEFAULT 'Unknown',
    is_verified BOOLEAN DEFAULT FALSE,
    is_deleted BOOLEAN DEFAULT FALSE,
    is_banned BOOLEAN DEFAULT FALSE,
    created_at BIGINT,
    last_login_at BIGINT,
    subscription_status VARCHAR(50) DEFAULT 'free',
    
    -- Storing complex objects as JSON for flexibility during migration
    usage_stats JSON, 
    preferences JSON,
    ip_addresses JSON -- Array of IPs
);

-- API Keys Table
CREATE TABLE IF NOT EXISTS api_keys (
    id VARCHAR(128) PRIMARY KEY,
    user_uid VARCHAR(128) NOT NULL,
    key_value VARCHAR(255) NOT NULL,
    name VARCHAR(255),
    network VARCHAR(50) DEFAULT 'google',
    tier VARCHAR(20) DEFAULT 'unknown',
    status VARCHAR(50) DEFAULT 'valid',
    created_at BIGINT,
    is_enabled BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (user_uid) REFERENCES users(uid) ON DELETE CASCADE
);

-- API Key Usage Stats
CREATE TABLE IF NOT EXISTS api_key_usage (
    id INT AUTO_INCREMENT PRIMARY KEY,
    key_hash VARCHAR(64) NOT NULL,
    date DATE NOT NULL,
    api_calls INT DEFAULT 0,
    model_id VARCHAR(100),
    UNIQUE KEY unique_usage (key_hash, date, model_id)
);

-- Assets Table
CREATE TABLE IF NOT EXISTS assets (
    id VARCHAR(128) PRIMARY KEY,
    user_uid VARCHAR(128) NOT NULL,
    type VARCHAR(50) NOT NULL, -- 'image', 'video', 'metadata', etc.
    url LONGTEXT NOT NULL,
    prompt TEXT,
    created_at BIGINT,
    is_favorite BOOLEAN DEFAULT FALSE,
    is_deleted BOOLEAN DEFAULT FALSE,
    metadata JSON, -- Flexible metadata storage
    aspect_ratio VARCHAR(20),
    batch_id VARCHAR(128),
    title TEXT,
    FOREIGN KEY (user_uid) REFERENCES users(uid) ON DELETE CASCADE
);

-- System Settings (Single Row usually)
CREATE TABLE IF NOT EXISTS system_settings (
    id INT PRIMARY KEY DEFAULT 1,
    settings JSON
);

-- Support Messages
CREATE TABLE IF NOT EXISTS support_messages (
    id VARCHAR(128) PRIMARY KEY,
    user_uid VARCHAR(128),
    email VARCHAR(255),
    subject VARCHAR(255),
    message TEXT,
    status VARCHAR(50) DEFAULT 'unread', -- 'unread', 'read', 'replied'
    timestamp BIGINT,
    sender VARCHAR(50) DEFAULT 'user' -- 'user' or 'admin'
);

-- Bug Reports
CREATE TABLE IF NOT EXISTS bug_reports (
    id VARCHAR(128) PRIMARY KEY,
    user_uid VARCHAR(128),
    title VARCHAR(255),
    description TEXT,
    steps TEXT,
    severity VARCHAR(50),
    status VARCHAR(50) DEFAULT 'open',
    timestamp BIGINT,
    is_deleted BOOLEAN DEFAULT FALSE
);

-- Audit Logs
CREATE TABLE IF NOT EXISTS audit_logs (
    id VARCHAR(128) PRIMARY KEY,
    action VARCHAR(255),
    details TEXT,
    admin_uid VARCHAR(128),
    ip VARCHAR(50),
    timestamp BIGINT
);

-- Tips (Admin Data)
CREATE TABLE IF NOT EXISTS tips (
    id VARCHAR(128) PRIMARY KEY,
    title VARCHAR(255),
    text TEXT,
    category VARCHAR(100),
    created_at BIGINT,
    is_deleted BOOLEAN DEFAULT FALSE
);

-- Tutorial Steps (Admin Data)
CREATE TABLE IF NOT EXISTS tutorial_steps (
    id VARCHAR(128) PRIMARY KEY,
    title VARCHAR(255),
    description TEXT,
    action_label VARCHAR(100),
    action_link VARCHAR(255),
    step_order INT,
    is_deleted BOOLEAN DEFAULT FALSE
);

-- Broadcasts (Admin Data)
CREATE TABLE IF NOT EXISTS broadcasts (
    id VARCHAR(128) PRIMARY KEY,
    title VARCHAR(255),
    message TEXT,
    type VARCHAR(50),
    sent_at BIGINT,
    read_count INT DEFAULT 0,
    open_rate FLOAT DEFAULT 0,
    opened_by JSON, -- List of user IDs
    target_audience VARCHAR(50) DEFAULT 'All',
    is_deleted BOOLEAN DEFAULT 0
);

-- Stock Sites (Market Intelligence)
CREATE TABLE IF NOT EXISTS stock_sites (
    id VARCHAR(128) PRIMARY KEY,
    name VARCHAR(255),
    url VARCHAR(255),
    category VARCHAR(100),
    description TEXT,
    contributors VARCHAR(100),
    library_size VARCHAR(100),
    commission VARCHAR(100),
    is_visible BOOLEAN DEFAULT TRUE,
    pros JSON,
    cons JSON,
    is_deleted BOOLEAN DEFAULT FALSE
);

-- Activity Logs (User Activity)
CREATE TABLE IF NOT EXISTS activity_logs (
    id VARCHAR(128) PRIMARY KEY,
    user_uid VARCHAR(128) NOT NULL,
    type VARCHAR(50), -- 'generate', 'system', 'setting'
    text VARCHAR(255),
    timestamp BIGINT,
    FOREIGN KEY (user_uid) REFERENCES users(uid) ON DELETE CASCADE
);

-- Initialize default settings row if not exists
INSERT IGNORE INTO system_settings (id, settings) VALUES (1, '{}');
