-- Safe migration: Add user_uid column to api_key_usage table
-- This allows querying historical data by user instead of by specific API keys
-- Handles case where column may already exist

-- Step 1: Add user_uid column (only if it doesn't exist)
SET @dbname = DATABASE();
SET @tablename = 'api_key_usage';
SET @columnname = 'user_uid';
SET @preparedStatement = (SELECT IF(
  (
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE
      (table_name = @tablename)
      AND (table_schema = @dbname)
      AND (column_name = @columnname)
  ) > 0,
  'SELECT 1', -- Column exists, do nothing
  CONCAT('ALTER TABLE ', @tablename, ' ADD COLUMN ', @columnname, ' VARCHAR(128) AFTER key_hash')
));
PREPARE alterIfNotExists FROM @preparedStatement;
EXECUTE alterIfNotExists;
DEALLOCATE PREPARE alterIfNotExists;

-- Step 2: Add index for better query performance (safe to run even if exists)
CREATE INDEX IF NOT EXISTS idx_user_date ON api_key_usage(user_uid, usage_date);

-- Step 3: Backfill existing records with user_uid by joining with api_keys table
-- This only updates rows where user_uid is NULL
UPDATE api_key_usage aku
INNER JOIN api_keys ak ON aku.key_hash = SHA2(ak.key_value, 256)
SET aku.user_uid = ak.user_uid
WHERE aku.user_uid IS NULL;

-- Check results
SELECT 
    COUNT(*) as total_rows,
    SUM(CASE WHEN user_uid IS NOT NULL THEN 1 ELSE 0 END) as rows_with_user_uid,
    SUM(CASE WHEN user_uid IS NULL THEN 1 ELSE 0 END) as rows_without_user_uid
FROM api_key_usage;

-- Optional: Clean up orphaned records (usage data for deleted keys with no user_uid)
-- Uncomment the following line if you want to remove historical data that can't be attributed to a user
-- DELETE FROM api_key_usage WHERE user_uid IS NULL;
