
import React, { createContext, useContext, useState, useEffect, useRef, useCallback } from 'react';
import { BatchItem, MetadataResult, MetadataSettings, User, StockSite, SiteSettings } from '../types';
import { generateMetadata, RateLimitExhaustedError, getPreferredModels } from '../services/geminiService';
import { extractVideoFrames, svgToPng } from '../utils/mediaUtils';
import { batchStorage } from '../utils/batchStorage';
import { authService } from '../services/authService';
import { dbService, Asset } from '../services/dbService';
import { playSound } from '../utils/soundEffects';
import { downloadItems } from '../utils/downloadUtils';
import { ADOBE_STOCK_CATEGORIES } from '../constants';

interface BatchContextType {
    batchItems: BatchItem[];
    isBatchProcessing: boolean;
    batchStatusInfo: { currentKey: string; currentModel: string; eta: string } | null;
    processedCount: number;
    totalCount: number;
    rateLimitError: boolean;
    setRateLimitError: (value: boolean) => void;
    // Actions
    addFiles: (files: File[]) => Promise<void>;
    removeBatchItem: (id: string) => Promise<void>;
    clearBatch: () => Promise<void>;
    startBatch: (settings: MetadataSettings, user: User, apiKeys: string[], availableSites: StockSite[], siteSettings: SiteSettings | null) => Promise<void>;
    pauseBatch: () => void;
    resumeBatch: (settings: MetadataSettings, user: User, apiKeys: string[], availableSites: StockSite[], siteSettings: SiteSettings | null) => Promise<void>;
    stopBatch: () => void;
    setBatchItems: React.Dispatch<React.SetStateAction<BatchItem[]>>; // Exposed for manual edits if needed
    isResuming: boolean; // Flag to show if we are recovering from a crash/restart
    batchStartTime: number | null;
}

const BatchContext = createContext<BatchContextType | undefined>(undefined);

export const useBatch = () => {
    const context = useContext(BatchContext);
    if (!context) throw new Error('useBatch must be used within a BatchProvider');
    return context;
};

export const BatchProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
    const [batchItems, setBatchItems] = useState<BatchItem[]>([]);
    const [isBatchProcessing, setIsBatchProcessing] = useState(false);
    const [batchStatusInfo, setBatchStatusInfo] = useState<{ currentKey: string; currentModel: string; eta: string } | null>(null);
    const [rateLimitError, setRateLimitError] = useState(false);
    const [isResuming, setIsResuming] = useState(true); // Start true to check DB
    const [batchStartTime, setBatchStartTime] = useState<number | null>(null);

    // Refs for processing loop
    const shouldStopRef = useRef(false);
    const itemControllersRef = useRef<Map<string, AbortController>>(new Map());
    const processingRef = useRef(false); // Ref for immediate loop control

    // Load persisted batch on mount
    useEffect(() => {
        const loadPersisted = async () => {
            try {
                const storedItems = await batchStorage.getAllItems();
                if (storedItems.length > 0) {
                    // Regenerate Blob URLs for images because they expire on refresh
                    const revivedItems = storedItems.map(item => {
                        let newPreview = item.preview;
                        if (item.file && item.file.type && item.file.type.startsWith('image/')) {
                            // Always regenerate for images to ensure validity
                            newPreview = URL.createObjectURL(item.file);
                        } else if (item.file && item.file.type && item.file.type.includes('svg')) {
                            // SVG might need conversion or just blob url
                            // svgToPng expects File, so we can't easily re-run it sync here.
                            // But createObjectURL works for displaying SVG too usually?
                            // Original logic used svgToPng (async).
                            // Ideally we stored the 'preview' base64 dataUrl if it was generated?
                            // If preview was base64, it persists fine. 
                            // If it was blob url, we must regen.
                            if (item.preview && item.preview.startsWith('blob:')) {
                                newPreview = URL.createObjectURL(item.file);
                            }
                        }
                        return { ...item, preview: newPreview };
                    });

                    // Sort by lastUpdated or id to maintain order (IndexedDB order is key-based usually)
                    // If we saved them in order, they come back in order if key is simple.
                    // BatchItem IDs are random strings, so order might be lost if we don't rely on array.
                    // Ideally we'd store the "Queue" as one object, but that's heavy.
                    // For now, simple list is fine.

                    // If any were 'processing', set them to 'pending'/paused
                    const fixedItems = revivedItems.map(i => i.status === 'processing' ? { ...i, status: 'pending' as const } : i);
                    setBatchItems(fixedItems);

                    // Update DB too to ensuring consistency (optional but good)
                    // await batchStorage.saveItems(fixedItems); // Avoid generic write loops if not needed
                    console.log('[BatchContext] Recovered interrupted batch from IndexedDB');
                }
            } catch (e) {
                console.error("Failed to load Batch DB", e);
            } finally {
                setIsResuming(false);
            }
        };
        loadPersisted();
    }, []);

    // Helper: Sync state to DB
    const updateBatchItems = (updater: (prev: BatchItem[]) => BatchItem[]) => {
        setBatchItems(prev => {
            const next = updater(prev);
            // Fire and forget DB update to avoid blocking UI
            // However, we should be careful about race conditions.
            // For now, saving specifically changed items is better, but full sync is safer for consistency.
            // We'll update changed items if we can identify them, or just all.
            // "saveItems" iterates and puts. efficient enough for <1000 items?
            // Maybe just update the ones that changed?
            // For simplicity in this turn: save all.
            batchStorage.saveItems(next).catch(e => console.error("DB Sync Error", e));
            return next;
        });
    };

    // Optimized update: only save specific item
    const updateHeaderItem = (id: string, updates: Partial<BatchItem>) => {
        setBatchItems(prev => {
            const next = prev.map(p => {
                if (p.id === id) {
                    const newItem = { ...p, ...updates };
                    batchStorage.saveItem(newItem);
                    return newItem;
                }
                return p;
            });
            return next;
        });
    };

    const addFiles = async (files: File[]) => {
        const newItems: BatchItem[] = await Promise.all(
            Array.from(files).map(async (file) => {
                let preview = '';
                if (file.type.startsWith('image/')) {
                    preview = URL.createObjectURL(file);
                } else if (file.type.startsWith('video/')) {
                    // Placeholder for video, real frame extracted during process? 
                    // Or extract waiting thumbnail now?
                    // Better to set a placeholder and extract during process or lazy load.
                    // For now, empty string or a default icon path.
                    // But MetadataGenerator expects a preview string.
                    // Let's use a generic video icon placeholder or try to extract frame 1 immediately?
                    // Immediate extraction might freeze if many videos.
                    // Lets use a dummy string and let UI handle it, OR quick extract.
                    // UI uses <VideoPreview> which extracts its own frames.
                    // So preview string is less critical for video unless used for thumb.
                    preview = '';
                }

                return {
                    id: Math.random().toString(36).substring(2, 10),
                    file,
                    preview, // Note: Blob URLs die on close, BUT we store File in IndexedDB
                    // On restore, we'll need to re-create URLs. 
                    // Wait, IndexedDB stores the File blob. 
                    // When we load from DB, 'preview' (blob url) is dead. 
                    // We should regenerate it on load? 
                    // Actually, 'preview' generated from URL.createURL(file) is fine.
                    // BUT if we restart, file is loaded from DB. We need to createURL again.
                    // We'll fix this in the helper below.
                    status: 'pending',
                    progress: 0
                };
            })
        );

        // Fix previews for Image persistence
        // Storing "blob:..." in DB is useless.
        // We should store empty in DB and re-create on load?
        // Or store base64? Base64 is heavy.
        // Let's rely on re-creating blob url from the File object we stored.

        updateBatchItems(prev => [...prev, ...newItems]);
    };

    const removeBatchItem = async (id: string) => {
        await batchStorage.removeItem(id);
        setBatchItems(prev => prev.filter(i => i.id !== id));
    };

    const clearBatch = async () => {
        await batchStorage.clear();
        setBatchItems([]);
        setIsBatchProcessing(false);
        setBatchStartTime(null); // Clear timer
        processingRef.current = false;
        shouldStopRef.current = true;
    };

    const stopBatch = () => {
        shouldStopRef.current = true;
        processingRef.current = false;
        setIsBatchProcessing(false);
        setBatchStartTime(null); // Clear timer
        // Cancel all
        itemControllersRef.current.forEach(c => c.abort());
        itemControllersRef.current.clear();
        setBatchItems(prev => prev.map(p => p.status === 'processing' ? { ...p, status: 'pending' } : p));
    };

    const pauseBatch = () => {
        // Same as stop but semantics differ (we might resume)
        stopBatch();
    };

    const processQueue = async (
        settings: MetadataSettings,
        user: User,
        apiKeys: string[],
        availableSites: StockSite[],
        passedSiteSettings: SiteSettings | null
    ) => {
        // Force refresh settings from LocalStorage to ensure we have the very latest Admin Panel updates
        // This fixes the bug where "Advanced Keyword Instructions" were ignored if the user didn't refresh the page.
        let siteSettings = passedSiteSettings;
        try {
            const stored = localStorage.getItem('site_settings');
            if (stored) {
                const parsed = JSON.parse(stored);
                // Merge passed (maybe generic defaults?) with stored (user overrides)
                siteSettings = { ...(passedSiteSettings || {}), ...parsed };
            }
        } catch (e) {
            console.error("Failed to refresh site settings from storage", e);
        }

        if (!user || apiKeys.length === 0) {
            alert("No user or API keys found");
            return;
        }

        setIsBatchProcessing(true);
        processingRef.current = true;
        shouldStopRef.current = false;
        setRateLimitError(false);

        let processedInSession = 0;
        const startTime = Date.now();
        setBatchStartTime(startTime); // Start timer

        // Get Valid Keys
        // ... (Logic from MetadataGenerator)
        // Assume keys passed are valid for now or we filter inside generator? 
        // MetadataGenerator logic:
        // const validKeys = apiKeys.filter ...
        // We will just use passed keys.
        const validKeys = apiKeys;

        // Preferred Model
        // We need to know preferred model if set? Or just rely on GeminiService defaults?
        // MetadataGenerator computed 'preferredModel'.
        // We'll compute it here:
        const prefModels = getPreferredModels('thinking'); // imported from geminiService
        // Actually getPreferredModels might need settings.
        const preferredModel = (prefModels && prefModels.length > 0) ? prefModels[0] : 'gemini-2.5-flash';

        // Context (Include/Exclude)
        // This is stored in local storage usually. 
        // We need to fetch it or pass it. 
        // passing it in `settings` if possible?
        // MetadataSettings has structure. 
        // Let's assume the caller merged Context Keywords into settings or we read them here.
        // Reading from localStorage here purely for backend context is safer.
        const contextKeywords = JSON.parse(localStorage.getItem(`metadata_include_keywords_${user.uid}`) || '[]');
        const excludeKeywords = JSON.parse(localStorage.getItem(`metadata_exclude_keywords_${user.uid}`) || '[]');
        // Context Desc?
        // It was local state. We might miss it if not passed.
        // Let's assume for now it's empty or the user must set it in a hypothetical "Global Context Settings".
        const contextDesc = ''; // Simplified for now.

        // Generate a single Batch ID for this session
        const currentBatchId = 'batch_' + Date.now();

        // Main Loop
        const queue = batchItems; // Initial snapshot
        // We need to iterate by index or ID to ensure we reference latest state?
        // Actually we should iterate through the *ids* of pending items.

        let pendingIds = batchItems.filter(i => i.status === 'pending' || i.status === 'error').map(i => i.id);

        for (let i = 0; i < pendingIds.length; i++) {
            if (shouldStopRef.current) break;

            // Re-read item from state to ensure it still exists and hasn't been removed
            const currentId = pendingIds[i];
            const storedItm = await batchStorage.getItem(currentId);
            if (!storedItm) continue; // Removed

            // Reconstruct item from stored
            let item: BatchItem = { ...storedItm, status: 'pending' };

            // Update UI to Processing
            updateHeaderItem(item.id, { status: 'processing', progress: 0 });

            // Avg Time Calculation
            const elapsed = Date.now() - startTime;
            const avgTimePerItem = processedInSession > 0 ? elapsed / processedInSession : 0;
            const remaining = pendingIds.length - i;
            let eta = 'Calculating...';
            if (processedInSession > 0) {
                const etaMs = avgTimePerItem * remaining;
                const etaSec = Math.ceil(etaMs / 1000);
                eta = etaSec > 60 ? `${Math.ceil(etaSec / 60)} min` : `${etaSec}s`;
            }
            setBatchStatusInfo({
                currentKey: validKeys[0]?.substring(0, 8) + '...',
                currentModel: preferredModel,
                eta
            });

            try {
                // Prepare File (Frames/SVG/Image)
                let base64: string | string[] = '';
                let mimeType = '';
                let dbThumbnail = '';

                if (item.file.type.startsWith('video/')) {
                    const frames = await extractVideoFrames(item.file);
                    // Selection logic (start/mid/end) - mimic MetadataGenerator
                    const opts = settings.videoFrameOptions || ['start'];
                    const selected: string[] = [];
                    if (opts.includes('start')) selected.push(frames[0]);
                    if (opts.includes('middle') && frames.length >= 2) selected.push(frames[Math.floor(frames.length / 2)]);
                    if (opts.includes('end') && frames.length > 0) selected.push(frames[frames.length - 1]);

                    const unique = [...new Set(selected)];
                    if (unique.length > 0) {
                        base64 = unique.map(p => p.split(',')[1]);
                        mimeType = 'image/jpeg';
                        dbThumbnail = frames[0];
                    } else {
                        // Fallback
                        base64 = frames[0].split(',')[1];
                        mimeType = 'image/jpeg';
                        dbThumbnail = frames[0];
                    }
                } else if (item.file.type.includes('svg')) {
                    const png = await svgToPng(item.file);
                    base64 = png.split(',')[1];
                    mimeType = 'image/png';
                    dbThumbnail = png;
                } else {
                    // Image
                    // Use FileReader
                    const fullDataUrl = await new Promise<string>((resolve) => {
                        const reader = new FileReader();
                        reader.onload = (e) => resolve(e.target?.result as string);
                        reader.readAsDataURL(item.file);
                    });
                    base64 = fullDataUrl.split(',')[1];
                    dbThumbnail = fullDataUrl;

                    // FIX: Set MIME Type from file or fallback to JPEG
                    mimeType = item.file.type || 'image/jpeg';

                    // FIX: Upload file to server for dbThumbnail (Avoids DB Size Limits)
                    try {
                        const uploadUrl = await dbService.uploadFile(item.file);
                        if (uploadUrl) {
                            dbThumbnail = uploadUrl;
                            console.log('[BatchContext] Uploaded thumbnail:', uploadUrl);
                        }
                    } catch (e) {
                        console.error("Failed to upload thumbnail, falling back to base64", e);
                    }
                }

                // Abort Controller
                const itemController = new AbortController();
                itemControllersRef.current.set(item.id, itemController);

                const metadata = await generateMetadata(
                    validKeys,
                    base64,
                    mimeType,
                    settings,
                    {
                        filename: item.file.name,
                        adobeStockInstruction: availableSites.find(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe')) ? siteSettings?.adobeStockInstruction : undefined,
                        titleInstruction: (() => {
                            const file = item.file;
                            if (file.type.startsWith('video/')) {
                                return siteSettings?.titleInstructionVideo || `You are an expert in stock video SEO.
Analyze the provided video and identify:
- moving elements
- camera angle or motion

Generate ONE SEO-friendly video title that:
- starts with the main visual subject
- includes motion words only if visible
- avoids cinematic or narrative language
- stays under 75 characters`;
                            } else if (file.type.includes('svg') || file.name.endsWith('.svg')) {
                                return siteSettings?.titleInstructionSvg || `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style
- context

Generate ONE SEO-optimized title that:
- starts with the main object name
- includes "vector", "icon", or "illustration" if relevant
- avoids artistic language
- remains under 65 characters`;
                            } else {
                                // Default to Image instruction
                                return siteSettings?.titleInstructionImage || siteSettings?.titleInstruction || `Act as a top-ranked Adobe Stock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (must be first in title)
2. Supporting visual elements

Create ONE SEO title that:
- matches buyer search intent
- is descriptive, not poetic
- starts with the main subject
- avoids adjectives that do not describe visuals
- is concise, professional, and under 70 characters`;
                            }
                        })(),
                        descriptionInstruction: (() => {
                            const file = item.file;
                            if (file.type.startsWith('video/')) {
                                return siteSettings?.descriptionInstructionVideo || "Write ONE SEO-friendly description for this stock video. Start with the main moving subject. Describe visible motion accurately. Avoid cinematic language. 1-2 concise sentences.";
                            } else if (file.type.includes('svg') || file.name.endsWith('.svg')) {
                                return siteSettings?.descriptionInstructionSvg || "Write ONE SEO-optimized description for this vector illustration. Start with main object. Include 'vector' or 'illustration'. Avoid artistic language.";
                            } else {
                                return siteSettings?.descriptionInstructionImage || siteSettings?.descriptionInstruction || "Write ONE factual, search-optimized description. Start with main subject in first 5 words. Describe only what is visible. 2-3 concise sentences.";
                            }
                        })(),
                        keywordInstruction: siteSettings?.keywordInstruction,
                        keywordGenerationPrompt: siteSettings?.keywordGenerationPrompt,
                        keywords: contextKeywords.join(', '),
                        excludeKeywords: excludeKeywords.join(', '),
                        description: contextDesc,
                    },
                    undefined,
                    itemController.signal
                );

                // Cleanup controller
                itemControllersRef.current.delete(item.id);

                if (shouldStopRef.current) break;

                const formattedMetadata = {
                    ...metadata,
                    filename: item.file.name,
                    category: (metadata.category && ADOBE_STOCK_CATEGORIES[metadata.category])
                        ? `${metadata.category} (${ADOBE_STOCK_CATEGORIES[metadata.category]})`
                        : metadata.category
                };

                // SUCCESS UPDATE
                updateHeaderItem(item.id, {
                    status: 'complete',
                    progress: 100,
                    result: formattedMetadata,
                    generationInfo: { model: metadata.modelUsed || preferredModel, apiKey: validKeys[0] }
                });

                // Save to History (DB)
                const asset: Asset = {
                    id: Math.random().toString(36).substring(2, 10),
                    type: 'metadata',
                    url: dbThumbnail,
                    prompt: metadata.title,
                    createdAt: Date.now(),
                    isFavorite: false,
                    isDeleted: false,
                    metadata: formattedMetadata,
                    modelUsed: metadata.modelUsed || preferredModel,
                    batchId: currentBatchId, // Use the CONSISTENT batch ID
                    generationSettings: {
                        enableGrounding: settings.enableGrounding,
                        descLength: settings.descLength,
                        keywordCount: settings.keywordCount,
                        titleLength: settings.titleLength,
                        exportFormat: settings.exportFormat || 'csv',
                        selectedStockSites: settings.selectedStockSites || []
                    }
                };
                await dbService.add(asset);

                // Deduct credits
                const cost = settings.enableGrounding ? 2 : 1;
                await authService.deductCredits(user.uid, cost, validKeys[0], metadata.modelUsed || preferredModel);

                processedInSession++;

            } catch (err: any) {
                console.error(`Error processing item ${item.id}`, err);
                updateHeaderItem(item.id, { status: 'error', progress: 0, error: err.message });

                if (err.name === 'RateLimitExhaustedError' || err.message?.includes('429')) {
                    setRateLimitError(true);
                    shouldStopRef.current = true;
                    // update status to waiting?
                }
            }

            // Apply Global Cooldown from User Preferences ONLY between generations
            if (i < pendingIds.length - 1) {
                const prefs = (user as any)?.preferences?.generationConfig;
                const globalCooldownSec = typeof prefs?.globalCooldown === 'number' ? prefs.globalCooldown : 5; // Default to 5s
                const metadataDelaySec = typeof prefs?.metadataDelay === 'number' ? prefs.metadataDelay : 0;

                const totalDelaySec = globalCooldownSec + metadataDelaySec;

                if (totalDelaySec > 0) {
                    const delayMs = totalDelaySec * 1000;
                    console.log(`[Batch] Waiting ${totalDelaySec}s (${globalCooldownSec}s Global + ${metadataDelaySec}s Metadata)...`);
                    await new Promise(r => setTimeout(r, delayMs));
                } else {
                    // Minimum safety delay between items
                    await new Promise(r => setTimeout(r, 1000));
                }
            }
        }

        setIsBatchProcessing(false);
        processingRef.current = false;
        setBatchStatusInfo(null);

        // Auto download if finished AND we processed something
        if (processedInSession > 0) {
            playSound('success');



            // Check auto download pref?
            // authService.getUserPreferences... logic.
        }
    };

    const startBatch = async (settings: MetadataSettings, user: User, apiKeys: string[], availableSites: StockSite[], siteSettings: SiteSettings | null) => {
        // Save settings for resume?
        // We'll rely on the caller to pass them for now, or assume they don't change often.
        processQueue(settings, user, apiKeys, availableSites, siteSettings);
    };

    // Alias
    const resumeBatch = startBatch;

    return (
        <BatchContext.Provider value={{
            batchItems,
            isBatchProcessing,
            batchStatusInfo,
            processedCount: batchItems.filter(i => i.status === 'complete').length,
            totalCount: batchItems.length,
            rateLimitError,
            setRateLimitError,
            addFiles,
            removeBatchItem,
            clearBatch,
            startBatch,
            pauseBatch,
            resumeBatch,
            stopBatch,
            setBatchItems, // Exposed
            isResuming,
            batchStartTime
        }}>
            {children}
        </BatchContext.Provider>
    );
};
