
import React, { useState, useEffect, useRef } from 'react';
import {
    Sparkles, Mail, Lock, User, Eye, EyeOff,
    ArrowRight, CheckCircle2, AlertCircle, Loader2,
    RefreshCw, ShieldCheck, X, ArrowLeft, Twitter, Facebook,
    RefreshCcw, Globe
} from 'lucide-react';

const COUNTRIES = [
    "United States", "United Kingdom", "Canada", "Germany", "France", "Australia", "Japan", "Brazil", "India", "China", "Russia", "Italy", "Spain", "Mexico", "South Korea", "Netherlands", "Turkey", "Switzerland", "Sweden", "Poland", "Other"
];
import { AuthView, User as UserType } from '../../types';
import { authService } from '../../services/authService';
import { adminService } from '../../services/adminService';

interface AuthPageProps {
    initialView: AuthView;
    onAuthSuccess: (user: UserType) => void;
    onBackToLanding: () => void;
}

const AuthPage: React.FC<AuthPageProps> = ({ initialView, onAuthSuccess, onBackToLanding }) => {
    const [view, setView] = useState<AuthView>(initialView);
    const [authContent, setAuthContent] = useState<any>({
        titlePrefix: "Forge Your",
        titleHighlight: "Digital Legacy",
        subtitle: "Join thousands of creators using Gemini AI to generate assets, analyze trends, and build the future of microstock.",
        feature1: "Access to Gemini 2.5 & 3.0 Models",
        feature2: "Unlimited Image Generation (Pro)",
        feature3: "Veo Cinematic Video Creation",
        feature4: "Deep Market Analysis Tools"
    });

    useEffect(() => {
        const savedTheme = localStorage.getItem('sf_theme');
        if (!savedTheme || savedTheme === 'dark') {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }

        adminService.getLandingContent().then(data => {
            if (data?.authPage) {
                setAuthContent(prev => ({ ...prev, ...data.authPage }));
            }
        });
    }, []);

    return (
        <div className="min-h-screen flex bg-slate-50 dark:bg-slate-950 text-slate-900 dark:text-slate-200 overflow-hidden font-sans transition-colors duration-300">
            {/* Left Side - Visual */}
            <div className="hidden lg:flex w-1/2 relative bg-slate-100 dark:bg-slate-900 overflow-hidden items-center justify-center transition-colors duration-300 border-r border-slate-200 dark:border-slate-800 min-h-screen">
                <div className="absolute inset-0 bg-[url('https://grainy-gradients.vercel.app/noise.svg')] opacity-10 dark:opacity-20 z-0" />
                <div className="absolute top-0 left-0 w-full h-full bg-gradient-to-br from-amber-500/5 to-purple-500/5 dark:from-amber-500/10 dark:to-purple-500/10 z-0" />
                <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-amber-500/10 dark:bg-amber-500/20 rounded-full blur-[100px] animate-pulse" />
                <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-indigo-500/10 dark:bg-indigo-500/20 rounded-full blur-[100px] animate-pulse delay-1000" />

                <div className="relative z-10 p-12 max-w-xl">
                    <div className="w-16 h-16 rounded-2xl bg-gradient-to-br from-amber-400 to-amber-600 flex items-center justify-center mb-8 shadow-2xl shadow-amber-500/30">
                        <Sparkles className="text-white w-8 h-8" />
                    </div>
                    <h1 className="text-5xl font-bold font-['Space_Grotesk'] text-slate-900 dark:text-white mb-6 leading-tight">
                        {authContent.titlePrefix} <br />
                        <span className="text-transparent bg-clip-text bg-gradient-to-r from-amber-500 to-orange-500 dark:from-amber-400 dark:to-orange-400">{authContent.titleHighlight}</span>
                    </h1>
                    <p className="text-lg text-slate-600 dark:text-slate-400 leading-relaxed mb-8">
                        {authContent.subtitle}
                    </p>

                    <div className="space-y-4">
                        {[authContent.feature1, authContent.feature2, authContent.feature3, authContent.feature4].filter(Boolean).map((item: string, i: number) => (
                            <div key={i} className="flex items-center gap-3 text-slate-700 dark:text-slate-300">
                                <div className="w-6 h-6 rounded-full bg-white dark:bg-slate-800 flex items-center justify-center border border-slate-200 dark:border-slate-700 shadow-sm">
                                    <CheckCircle2 size={14} className="text-amber-500 dark:text-amber-400" />
                                </div>
                                {item}
                            </div>
                        ))}
                    </div>
                </div>
            </div>

            {/* Right Side - Forms */}
            <div className="w-full lg:w-1/2 flex items-center justify-center p-6 relative bg-slate-50 dark:bg-slate-900 transition-colors duration-300 min-h-screen">
                <div className="w-full max-w-md py-10">
                    {view === AuthView.LOGIN && (
                        <LoginForm onSuccess={onAuthSuccess} switchToRegister={() => setView(AuthView.REGISTER)} switchToForgot={() => setView(AuthView.FORGOT_PASSWORD)} onBackToLanding={onBackToLanding} />
                    )}
                    {view === AuthView.REGISTER && (
                        <RegisterForm onSuccess={(user: UserType) => { setView(AuthView.VERIFY_EMAIL); }} switchToLogin={() => setView(AuthView.LOGIN)} onBackToLanding={onBackToLanding} onDirectSuccess={onAuthSuccess} />
                    )}
                    {view === AuthView.VERIFY_EMAIL && (
                        <VerifyEmailForm email="user@example.com" onSuccess={() => setView(AuthView.LOGIN)} onBack={() => setView(AuthView.REGISTER)} onBackToLanding={onBackToLanding} />
                    )}
                    {view === AuthView.FORGOT_PASSWORD && (
                        <ForgotPasswordForm onBackToLanding={onBackToLanding} setView={setView} />
                    )}
                </div>
            </div>
        </div>
    );
};

// --- Sub Components ---

const LoginForm = ({ onSuccess, switchToRegister, switchToForgot, onBackToLanding }: any) => {
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const [googleLoading, setGoogleLoading] = useState(false);
    const [twitterLoading, setTwitterLoading] = useState(false);
    const [facebookLoading, setFacebookLoading] = useState(false);
    const [showPass, setShowPass] = useState(false);
    const [error, setError] = useState('');

    // Settings State - Initialize as null to prevent flashing
    const [settings, setSettings] = useState<any>(null);
    const googleBtnRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        adminService.getSiteSettings().then(s => setSettings(s));
    }, []);

    useEffect(() => {
        // Google Real Auth Init - Only run if settings are loaded and valid
        if (settings && settings.enableGoogleLogin && !settings.enableSimulatedGoogleLogin && settings.googleClientId && (window as any).google && googleBtnRef.current) {
            try {
                (window as any).google.accounts.id.initialize({
                    client_id: settings.googleClientId,
                    callback: async (response: any) => {
                        try {
                            const user = await authService.handleGoogleCredential(response.credential);
                            onSuccess(user);
                        } catch (e) {
                            setError('Google sign-in failed.');
                        }
                    }
                });
                (window as any).google.accounts.id.renderButton(
                    googleBtnRef.current,
                    { theme: "outline", size: "large", width: "100%", text: "signin" }
                );
            } catch (e) {
                console.error("Google Auth Init Error", e);
            }
        }
    }, [settings]);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);
        try {
            const user = await authService.login(email, password);
            onSuccess(user);
        } catch (err: any) {
            setError(err.message || 'Login failed');
        } finally {
            setLoading(false);
        }
    };

    const handleSimulatedGoogle = async () => {
        setGoogleLoading(true);
        try {
            const user = await authService.loginWithGoogleSimulation();
            onSuccess(user);
        } catch (e) {
            setError("Google simulation failed.");
        } finally {
            setGoogleLoading(false);
        }
    };

    const handleTwitterLogin = async () => {
        if (!settings.enableSimulatedTwitterLogin && !settings.twitterClientId) {
            setError("Twitter login is not configured. Please check admin settings.");
            return;
        }
        setTwitterLoading(true);
        try {
            if (settings.enableSimulatedTwitterLogin) {
                const user = await authService.loginWithTwitterSimulation();
                onSuccess(user);
            } else {
                alert(`Redirecting to Twitter OAuth with Client ID: ${settings.twitterClientId}\n(This requires backend implementation)`);
            }
        } catch (e) {
            setError("Twitter login failed.");
        } finally {
            setTwitterLoading(false);
        }
    }

    const handleFacebookLogin = async () => {
        if (!settings.enableSimulatedFacebookLogin && !settings.facebookClientId) {
            setError("Facebook login is not configured. Please check admin settings.");
            return;
        }
        setFacebookLoading(true);
        try {
            if (settings.enableSimulatedFacebookLogin) {
                const user = await authService.loginWithFacebookSimulation();
                onSuccess(user);
            } else {
                alert(`Redirecting to Facebook OAuth with App ID: ${settings.facebookClientId}\n(This requires backend implementation)`);
            }
        } catch (e) {
            setError("Facebook login failed.");
        } finally {
            setFacebookLoading(false);
        }
    }

    // Determine if any social login is enabled. Check for null settings to prevent flash.
    const showSocial = settings && (settings.enableGoogleLogin || settings.enableTwitterLogin || settings.enableFacebookLogin);

    return (
        <div className="animate-fade-in">
            <button onClick={onBackToLanding} className="mb-8 flex items-center gap-2 text-sm font-bold text-slate-500 hover:text-slate-900 dark:hover:text-slate-300 transition-colors">
                <ArrowLeft size={16} /> Back to Home
            </button>

            <div className="mb-8">
                <h2 className="text-3xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk']">Welcome Back</h2>
                <p className="text-slate-600 dark:text-slate-400 mt-2 font-medium">Enter your credentials to access your workspace.</p>
            </div>

            {error && (
                <div className="mb-6 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-900/30 rounded-xl flex items-center gap-3 text-red-600 dark:text-red-400 text-sm font-bold animate-shake">
                    <AlertCircle size={18} /> {error}
                </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-5">
                <div className="space-y-2">
                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Email Address</label>
                    <div className="relative">
                        <Mail className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                        <input
                            type="email"
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                            placeholder="name@company.com"
                            required
                        />
                    </div>
                </div>

                <div className="space-y-2">
                    <div className="flex justify-between items-center">
                        <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Password</label>
                        <button type="button" onClick={switchToForgot} className="text-xs font-bold text-amber-600 dark:text-amber-500 hover:underline">Forgot?</button>
                    </div>
                    <div className="relative">
                        <Lock className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                        <input
                            type={showPass ? "text" : "password"}
                            value={password}
                            onChange={(e) => setPassword(e.target.value)}
                            className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-12 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                            placeholder="••••••••"
                            required
                        />
                        <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-4 top-3.5 text-slate-400 hover:text-slate-600 dark:text-slate-500 dark:hover:text-slate-300 transition-colors">
                            {showPass ? <EyeOff size={18} /> : <Eye size={18} />}
                        </button>
                    </div>
                </div>

                <button
                    type="submit"
                    disabled={loading}
                    className="w-full bg-slate-900 dark:bg-white hover:bg-slate-800 dark:hover:bg-slate-200 text-white dark:text-slate-950 font-bold py-3.5 rounded-xl transition-all shadow-lg hover:shadow-xl hover:-translate-y-0.5 disabled:opacity-70 disabled:hover:translate-y-0 flex items-center justify-center gap-2"
                >
                    {loading ? <Loader2 className="animate-spin" /> : "Sign In"}
                </button>
            </form>

            {/* Only render this block if settings are loaded AND at least one social option is enabled */}
            {showSocial && (
                <div className="animate-fade-in">
                    <div className="my-8 flex items-center gap-4">
                        <div className="h-px bg-slate-200 dark:bg-slate-800 flex-1"></div>
                        <span className="text-xs font-bold text-slate-400 uppercase">Or continue with</span>
                        <div className="h-px bg-slate-200 dark:bg-slate-800 flex-1"></div>
                    </div>

                    <div className="flex gap-3">
                        {/* GOOGLE BUTTON */}
                        {settings.enableGoogleLogin && (
                            <div className="flex-1">
                                {!settings.enableSimulatedGoogleLogin && settings.googleClientId ? (
                                    <div ref={googleBtnRef} className="h-[48px] w-full overflow-hidden rounded-xl"></div>
                                ) : (
                                    <button
                                        onClick={handleSimulatedGoogle}
                                        disabled={googleLoading}
                                        className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 hover:bg-slate-50 dark:hover:bg-slate-900 text-slate-700 dark:text-white font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 relative overflow-hidden group shadow-sm h-[48px]"
                                    >
                                        {googleLoading ? <Loader2 className="animate-spin text-slate-400 w-5 h-5" /> : (
                                            <>
                                                <svg className="w-5 h-5 shrink-0" viewBox="0 0 24 24"><path d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z" fill="#4285F4" /><path d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z" fill="#34A853" /><path d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z" fill="#FBBC05" /><path d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z" fill="#EA4335" /></svg>
                                                <span className="truncate">Google</span>
                                            </>
                                        )}
                                    </button>
                                )}
                            </div>
                        )}

                        {/* FACEBOOK BUTTON */}
                        {settings.enableFacebookLogin && (
                            <button
                                onClick={handleFacebookLogin}
                                disabled={facebookLoading}
                                className="flex-1 bg-[#1877F2] hover:bg-[#166fe5] text-white font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 shadow-sm h-[48px]"
                            >
                                {facebookLoading ? <Loader2 className="animate-spin w-5 h-5" /> : (
                                    <>
                                        <Facebook className="w-5 h-5 fill-current shrink-0" /> <span className="hidden sm:inline">Facebook</span>
                                    </>
                                )}
                            </button>
                        )}

                        {/* TWITTER BUTTON */}
                        {settings.enableTwitterLogin && (
                            <button
                                onClick={handleTwitterLogin}
                                disabled={twitterLoading}
                                className="flex-1 bg-black hover:bg-slate-900 text-white font-bold py-3 rounded-xl transition-all flex items-center justify-center gap-2 shadow-sm dark:border dark:border-slate-700 h-[48px]"
                            >
                                {twitterLoading ? <Loader2 className="animate-spin w-5 h-5" /> : (
                                    <>
                                        <Twitter className="w-5 h-5 fill-current shrink-0" /> <span className="hidden sm:inline">Twitter</span>
                                    </>
                                )}
                            </button>
                        )}
                    </div>
                </div>
            )}

            <p className="text-center mt-8 text-sm text-slate-600 dark:text-slate-400 font-medium">
                Don't have an account? <button onClick={switchToRegister} className="text-amber-600 dark:text-amber-500 font-bold hover:underline">Sign up</button>
            </p>
        </div>
    );
};

const RegisterForm = ({ onSuccess, switchToLogin, onBackToLanding, onDirectSuccess }: any) => {
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [confirmPassword, setConfirmPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');

    // Captcha State
    const [captchaChallenge, setCaptchaChallenge] = useState('');
    const [captchaInput, setCaptchaInput] = useState('');

    const generateCaptcha = () => {
        const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        let result = '';
        for (let i = 0; i < 5; i++) {
            result += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        setCaptchaChallenge(result);
        setCaptchaInput('');
    };

    useEffect(() => {
        generateCaptcha();
    }, []);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');

        // Validations
        if (password !== confirmPassword) {
            setError("Passwords do not match.");
            return;
        }
        if (captchaInput.toUpperCase() !== captchaChallenge) {
            setError("Incorrect captcha code. Please try again.");
            generateCaptcha();
            return;
        }

        setLoading(true);
        try {
            const user = await authService.register(name, email, password, false, null);
            if (user.isVerified && onDirectSuccess) {
                onDirectSuccess(user);
            } else {
                onSuccess(user);
            }
        } catch (err: any) {
            setError(err.message || 'Registration failed');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="animate-fade-in">
            <button onClick={onBackToLanding} className="mb-8 flex items-center gap-2 text-sm font-bold text-slate-500 hover:text-slate-900 dark:hover:text-slate-300 transition-colors">
                <ArrowLeft size={16} /> Back to Home
            </button>

            <div className="mb-8">
                <h2 className="text-3xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk']">Create Account</h2>
                <p className="text-slate-600 dark:text-slate-400 mt-2 font-medium">Join the future of content creation.</p>
            </div>

            {error && (
                <div className="mb-6 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-900/30 rounded-xl flex items-center gap-3 text-red-600 dark:text-red-400 text-sm font-bold animate-shake">
                    <AlertCircle size={18} /> {error}
                </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-5">
                <div className="space-y-2">
                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Full Name</label>
                    <div className="relative">
                        <User className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                        <input
                            type="text"
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                            className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                            placeholder="John Doe"
                            required
                        />
                    </div>
                </div>

                <div className="space-y-2">
                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Email Address</label>
                    <div className="relative">
                        <Mail className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                        <input
                            type="email"
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                            placeholder="name@company.com"
                            required
                        />
                    </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                        <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Password</label>
                        <div className="relative">
                            <Lock className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                            <input
                                type="password"
                                value={password}
                                onChange={(e) => setPassword(e.target.value)}
                                className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                                placeholder="••••••••"
                                required
                            />
                        </div>
                    </div>
                    <div className="space-y-2">
                        <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Confirm</label>
                        <div className="relative">
                            <Lock className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                            <input
                                type="password"
                                value={confirmPassword}
                                onChange={(e) => setConfirmPassword(e.target.value)}
                                className={`w-full bg-white dark:bg-slate-950 border rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 outline-none transition-all font-medium ${confirmPassword && confirmPassword !== password ? 'border-red-500 focus:ring-red-500' : 'border-slate-200 dark:border-slate-800 focus:ring-amber-500'
                                    }`}
                                placeholder="••••••••"
                                required
                            />
                        </div>
                    </div>
                </div>

                <div className="space-y-2">
                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Security Check</label>
                    <div className="flex items-center gap-4">
                        <div
                            className="bg-slate-200 dark:bg-slate-800 px-6 py-3 rounded-xl font-mono text-xl font-bold text-slate-700 dark:text-slate-300 tracking-widest select-none border border-slate-300 dark:border-slate-700"
                            style={{ textDecoration: 'line-through' }}
                        >
                            {captchaChallenge}
                        </div>
                        <button
                            type="button"
                            onClick={generateCaptcha}
                            className="p-3 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 rounded-xl text-slate-500 dark:text-slate-400 transition"
                        >
                            <RefreshCcw size={18} />
                        </button>
                        <input
                            type="text"
                            value={captchaInput}
                            onChange={(e) => setCaptchaInput(e.target.value)}
                            className="flex-1 bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 px-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium text-center uppercase"
                            placeholder="Enter Code"
                            maxLength={5}
                            required
                        />
                    </div>
                </div>

                <button
                    type="submit"
                    disabled={loading}
                    className="w-full bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold py-3.5 rounded-xl transition-all shadow-lg shadow-amber-500/20 hover:-translate-y-0.5 disabled:opacity-70 flex items-center justify-center gap-2"
                >
                    {loading ? <Loader2 className="animate-spin" /> : "Create Account"}
                </button>
            </form >

            <p className="text-center mt-8 text-sm text-slate-600 dark:text-slate-400 font-medium">
                Already have an account? <button onClick={switchToLogin} className="text-blue-600 dark:text-blue-400 font-bold hover:underline">Sign in</button>
            </p>
        </div >
    );
};

const VerifyEmailForm = ({ email, onSuccess, onBack, onBackToLanding }: any) => {
    const [code, setCode] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');

    const handleVerify = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);
        try {
            await authService.verifyEmail(code);
            onSuccess();
        } catch (e) {
            setError("Invalid code. Use '123456' for demo.");
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="animate-fade-in text-center">
            <div className="w-20 h-20 bg-blue-100 dark:bg-blue-900/20 rounded-full flex items-center justify-center mx-auto mb-6 text-blue-600 dark:text-blue-400 animate-bounce-slow">
                <Mail size={32} />
            </div>
            <h2 className="text-2xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk'] mb-2">Check your email</h2>
            <p className="text-slate-600 dark:text-slate-400 mb-8 max-w-xs mx-auto">
                We've sent a verification code to <span className="font-bold text-slate-900 dark:text-white">{email}</span>
            </p>

            {error && (
                <div className="mb-6 p-3 bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 text-sm font-bold rounded-lg animate-shake">
                    {error}
                </div>
            )}

            <form onSubmit={handleVerify} className="space-y-6">
                <input
                    type="text"
                    value={code}
                    onChange={(e) => setCode(e.target.value)}
                    className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 text-center text-2xl tracking-[0.5em] font-mono text-slate-900 dark:text-white focus:ring-2 focus:ring-amber-500 outline-none"
                    placeholder="000000"
                    maxLength={6}
                />
                <button
                    type="submit"
                    disabled={loading}
                    className="w-full bg-blue-600 hover:bg-blue-500 text-white font-bold py-3.5 rounded-xl transition-all shadow-lg shadow-blue-500/20 flex items-center justify-center gap-2"
                >
                    {loading ? <Loader2 className="animate-spin" /> : "Verify Email"}
                </button>
            </form>

            <div className="mt-8 space-y-4">
                <button onClick={() => authService.resendCode()} className="text-sm font-bold text-slate-500 hover:text-slate-900 dark:hover:text-white transition">Resend Code</button>
                <div>
                    <button onClick={onBack} className="text-xs text-slate-400 hover:underline">Back to Register</button>
                </div>
            </div>
        </div>
    );
};

const ForgotPasswordForm = ({ onBackToLanding, setView }: any) => {
    const [email, setEmail] = useState('');
    const [submitted, setSubmitted] = useState(false);

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        setTimeout(() => setSubmitted(true), 1000);
    }

    if (submitted) {
        return (
            <div className="text-center animate-fade-in">
                <div className="w-20 h-20 bg-green-100 dark:bg-green-900/20 rounded-full flex items-center justify-center mx-auto mb-6 text-green-600 dark:text-green-400">
                    <CheckCircle2 size={32} />
                </div>
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white mb-2">Email Sent</h2>
                <p className="text-slate-600 dark:text-slate-400 mb-8">Check your inbox for reset instructions.</p>
                <button onClick={() => setView(AuthView.LOGIN)} className="w-full bg-slate-900 dark:bg-white text-white dark:text-slate-950 font-bold py-3 rounded-xl">Back to Login</button>
            </div>
        )
    }

    return (
        <div className="animate-fade-in">
            <button onClick={onBackToLanding} className="mb-8 flex items-center gap-2 text-sm font-bold text-slate-500 hover:text-slate-900 dark:hover:text-slate-300 transition-colors">
                <ArrowLeft size={16} /> Back to Home
            </button>
            <h2 className="text-2xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk'] mb-2">Reset Password</h2>
            <p className="text-slate-600 dark:text-slate-400 mb-8">Enter your email to receive reset instructions.</p>

            <form onSubmit={handleSubmit} className="space-y-6">
                <div className="space-y-2">
                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Email Address</label>
                    <div className="relative">
                        <Mail className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" />
                        <input
                            type="email"
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            className="w-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-12 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium"
                            placeholder="name@company.com"
                            required
                        />
                    </div>
                </div>
                <button type="submit" className="w-full bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold py-3.5 rounded-xl transition-all shadow-lg shadow-amber-500/20">Send Reset Link</button>
            </form>
            <div className="mt-6 text-center">
                <button onClick={() => setView(AuthView.LOGIN)} className="text-sm font-bold text-slate-500 hover:text-slate-900 dark:hover:text-white">Back to Login</button>
            </div>
        </div>
    );
}

export default AuthPage;
