
import React, { useState, useRef, useEffect } from 'react';
import {
    Video, Upload, AlertCircle, Play, Film, MonitorPlay,
    Clock, Settings, Maximize, Download, RefreshCcw,
    Layers, History as HistoryIcon, Image as ImageIcon, Type, Zap,
    ChevronDown, ChevronUp, Wand2, Film as FilmIcon,
    Scissors, Share2, Heart, Trash2, Sparkles, Key, Cpu
} from 'lucide-react';
import { generateVideo, checkVeoKey, promptVeoKey, getPreferredModels } from '../services/geminiService';
import { dbService, Asset } from '../services/dbService';
import { authService } from '../services/authService';
import { AutoDownloadIndicator } from './AutoDownloadIndicator';
import { playSound } from '../utils/soundEffects';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';

const LOADING_MESSAGES = [
    "🎬 Patience is a virtue, especially with AI video...",
    "🎞️ Rendering high-quality frames takes time...",
    "🎨 Your video is being crafted by our digital artisans...",
    "🔧 Compositing scenes and generating motion vectors...",
    "✨ Applying cinematic effects and color grading...",
    "🎯 Optimizing keyframes and transitions...",
    "🚀 Finalizing the video render. Almost there!"
];

const VIDEO_MODELS = [
    { id: 'veo-3.1-generate-preview', name: 'Veo 3.1 (High Quality)' },
    { id: 'veo-3.1-fast-generate-preview', name: 'Veo 3.1 Fast (Speed)' }
];

const VideoGenerator: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'generator' | 'bulk' | 'history'>('generator');
    const [inputMode, setInputMode] = useState<'text' | 'image'>('text');
    const [hasVeoKey, setHasVeoKey] = useState(false);
    const [hasAppKey, setHasAppKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [rateLimitError, setRateLimitError] = useState(false); // Popup state

    // Inputs
    const [prompt, setPrompt] = useState('');
    const [imageFile, setImageFile] = useState<File | null>(null);
    const [imagePreview, setImagePreview] = useState<string | null>(null);

    // Settings
    const [aspectRatio, setAspectRatio] = useState<'16:9' | '9:16' | '1:1'>('16:9');
    const [resolution, setResolution] = useState<'720p' | '1080p'>('1080p');
    const [duration, setDuration] = useState(4);
    const [fps, setFps] = useState(24);
    const [cameraMotion, setCameraMotion] = useState('none');
    const [isAdvancedOpen, setIsAdvancedOpen] = useState(false);
    const [selectedModel, setSelectedModel] = useState('veo-3.1-fast-generate-preview');

    // Active Models (Enabled in Settings)
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // Generation State
    const [isGenerating, setIsGenerating] = useState(false);
    const [loadingMsgIndex, setLoadingMsgIndex] = useState(0);
    const [generatedVideo, setGeneratedVideo] = useState<string | null>(null);
    const [elapsedTime, setElapsedTime] = useState<number>(0);

    // Bulk State
    const [bulkPrompts, setBulkPrompts] = useState('');
    const [bulkStatus, setBulkStatus] = useState<'idle' | 'processing' | 'complete'>('idle');
    const [bulkProgress, setBulkProgress] = useState(0);
    const [bulkResults, setBulkResults] = useState<Asset[]>([]);

    // History
    const [history, setHistory] = useState<Asset[]>([]);

    const fileInputRef = useRef<HTMLInputElement>(null);

    const [availableModels, setAvailableModels] = useState(VIDEO_MODELS);

    useEffect(() => {
        refreshAccess();
        const handleFocus = () => refreshAccess();
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    const refreshAccess = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch specific Veo key check too if needed, but let's rely on filter
        checkVeoKey().then(setHasVeoKey);

        const [keys, settings] = await Promise.all([
            authService.getUserApiKeys(currentUser.uid),
            authService.getSiteSettings()
        ]);

        // Check App Key status
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        setHasAppKey(!!validKey);

        const filtered = VIDEO_MODELS.filter(m => authService.canUserAccessModel(m.id, keys, settings));
        setAvailableModels(filtered);

        // Update selected if needed
        if (filtered.length > 0 && !filtered.find(m => m.id === selectedModel)) {
            setSelectedModel(filtered[0].id);
        }

        // Update active stats display
        updateActiveModels(filtered.map(m => m.name));

        setIsCheckingKey(false);
    }

    const updateActiveModels = (names: string[]) => {
        setActiveModels(names);
    }

    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    // Loading Message Rotation & Timer
    useEffect(() => {
        let msgInterval: any;
        let timerInterval: any;

        if (isGenerating) {
            msgInterval = setInterval(() => {
                setLoadingMsgIndex(prev => (prev + 1) % LOADING_MESSAGES.length);
            }, 8000);

            const start = Date.now();
            timerInterval = setInterval(() => {
                setElapsedTime(Math.floor((Date.now() - start) / 1000));
            }, 1000);
        } else {
            setElapsedTime(0);
        }

        return () => {
            clearInterval(msgInterval);
            clearInterval(timerInterval);
        };
    }, [isGenerating]);

    const handleKeySelection = async () => {
        try {
            await promptVeoKey();
            setHasVeoKey(true);
        } catch (e) { console.error("Key selection failed", e); }
    }

    const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            const file = e.target.files[0];
            setImageFile(file);
            const reader = new FileReader();
            reader.onload = (ev) => setImagePreview(ev.target?.result as string);
            reader.readAsDataURL(file);
            setInputMode('image');
        }
    };

    const handleGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        // Clear history before generating new video
        await dbService.clearByType('video');

        setIsGenerating(true);
        setGeneratedVideo(null);

        try {
            let imageBytes = undefined;
            if (inputMode === 'image' && imagePreview) {
                imageBytes = imagePreview.split(',')[1];
            }
            const validRatio: any = aspectRatio === '1:1' ? '16:9' : aspectRatio;

            const videoUrl = await generateVideo(apiKeys, prompt, {
                aspectRatio: validRatio,
                resolution: resolution as '720p' | '1080p',
                modelId: selectedModel
            }, imageBytes);

            setGeneratedVideo(videoUrl);

            // Save to DB
            const asset: Asset = {
                id: Math.random().toString(36).substring(2, 10),
                type: 'video',
                url: videoUrl,
                prompt: prompt || 'Image to Video',
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: { resolution, duration, fps, model: selectedModel },
                batchId: Date.now().toString() // Add batchId
            };
            await dbService.add(asset);

            // Log Activity & Stats
            if (currentUser) {
                // Video costs 5 credits.
                authService.incrementUserStat('videosCreated', 1, selectedModel);
                // authService.incrementUserStat('apiCallsThisMonth', 4); // Removed manual update
                if (apiKeys[0]) {
                    authService.deductCredits(currentUser.uid, 5, apiKeys[0], selectedModel);
                }
                authService.logActivity(currentUser.uid, 'generate', `Generated video: "${prompt.substring(0, 30)}..."`);
            }

            // Redirect to History
            setActiveTab('history');

            // Play success sound
            await playSound('success');

        } catch (error) {
            alert('Video generation failed. Ensure you have a valid API key with Veo access.');

            // Play error sound
            await playSound('error');
        } finally {
            setIsGenerating(false);
        }
    };

    const handleBulkGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        const prompts = bulkPrompts.split('\n').filter(p => p.trim());
        if (prompts.length === 0) return;

        // Clear history before batch
        await dbService.clearByType('video');

        setBulkStatus('processing');
        setBulkProgress(0);
        setBulkResults([]);
        const results: Asset[] = [];
        const validRatio: any = aspectRatio === '1:1' ? '16:9' : aspectRatio;

        const batchId = Date.now().toString(); // Generate unique batch ID

        for (let i = 0; i < prompts.length; i++) {
            // Apply Video Delay / Global Cooldown
            const prefs = (currentUser as any)?.preferences?.generationConfig;
            const globalCooldownSec = Number(prefs?.globalCooldown || 0);
            const videoDelaySec = Number(prefs?.videoDelay || 0);

            // Additive Delay Logic: Global Cooldown + Video Delay
            const effectiveDelaySec = globalCooldownSec + videoDelaySec;

            if (effectiveDelaySec > 0) {
                const cooldownMs = effectiveDelaySec * 1000;
                console.log(`[VideoGen] Applying Total Delay: ${cooldownMs}ms (${globalCooldownSec}s Global + ${videoDelaySec}s Video)`);
                await new Promise(r => setTimeout(r, cooldownMs));
            }

            try {
                const p = prompts[i];
                const videoUrl = await generateVideo(apiKeys, p, {
                    aspectRatio: validRatio,
                    resolution: resolution as any,
                    modelId: selectedModel
                });

                const asset: Asset = {
                    id: Math.random().toString(36).substring(2, 10),
                    type: 'video',
                    url: videoUrl,
                    prompt: p,
                    createdAt: Date.now(),
                    isFavorite: false,
                    isDeleted: false,
                    metadata: { resolution, duration, model: selectedModel },
                    batchId: batchId // Add batchId
                };
                await dbService.add(asset);

                if (currentUser) {
                    await authService.deductCredits(currentUser.uid, 5, apiKeys[0], selectedModel); // Video costs 5 credits
                    authService.incrementUserStat('videosCreated', 1);
                    authService.incrementUserStat('modelUsage', 1, selectedModel);
                }

                results.push(asset);
            } catch (e) {
                console.error(`Failed bulk video: ${prompts[i]}`);
            }
            setBulkProgress(Math.round(((i + 1) / prompts.length) * 100));
        }

        if (currentUser) {
            authService.logActivity(currentUser.uid, 'generate', `Batch generated ${results.length} videos`);
        }

        setBulkResults(results);
        setBulkStatus('complete');
    };

    const formatTime = (seconds: number) => {
        const mins = Math.floor(seconds / 60);
        const secs = seconds % 60;
        return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }

    const handleDownloadAllHistory = () => {
        downloadItems(history.map(item => ({
            url: item.url,
            filename: `video_history_${item.id}.mp4`
        })), 'video_history');
    };

    const handleDownloadBatch = () => {
        downloadItems(bulkResults.map(item => ({
            url: item.url,
            filename: `video_batch_${item.id}.mp4`
        })), 'video_batch');
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">

            {/* NEW HEADER */}
            <div className="mb-6 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <Video className="text-blue-500 w-8 h-8" /> Cinematic Video Studio
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">Generate 1080p videos from text or animate images with Veo 3.1.</p>
            </div>

            {!hasAppKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Header & Tabs */}
            <div className="flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-6 shrink-0">
                <div className="flex w-full md:w-auto overflow-x-auto">
                    {[
                        { id: 'generator', label: 'Generator', icon: Video },
                        { id: 'bulk', label: 'Bulk', icon: Layers },
                        { id: 'history', label: 'History', icon: HistoryIcon },
                    ].map((tab) => (
                        <button
                            key={tab.id}
                            onClick={() => setActiveTab(tab.id as any)}
                            className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === tab.id ? 'text-blue-600 dark:text-blue-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                                }`}
                        >
                            <tab.icon size={16} />
                            {tab.label}
                            {activeTab === tab.id && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-blue-600 dark:bg-blue-400 animate-fade-in"></div>}
                        </button>
                    ))}
                </div>

                {/* Active Models Display & Auto-Download Indicator */}
                {activeTab !== 'history' && (
                    <div className="hidden md:flex items-center gap-2 my-2 mr-2">
                        <div className="flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg">
                            <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                                <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                            </span>
                            <div className="flex gap-1">
                                {activeModels.map((model, idx) => (
                                    <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                        {model}
                                    </span>
                                ))}
                                {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                            </div>
                        </div>
                        <AutoDownloadIndicator />
                    </div>
                )}
            </div>

            <div className={`flex-1 overflow-hidden transition-opacity ${!hasAppKey ? 'opacity-50 pointer-events-none' : ''}`}>
                {activeTab === 'generator' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20 custom-scrollbar">

                        {/* LEFT PANEL: INPUTS */}
                        <div className="lg:col-span-4 space-y-6 h-fit">
                            {/* Input Mode Toggle */}
                            <div className="bg-white dark:bg-slate-900 p-1 rounded-xl border border-slate-200 dark:border-slate-800 flex shadow-sm">
                                <button
                                    onClick={() => setInputMode('text')}
                                    className={`flex-1 py-3 rounded-lg text-sm font-bold transition-all flex items-center justify-center gap-2 ${inputMode === 'text' ? 'bg-amber-500 text-white dark:text-slate-950 shadow-md' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-200'}`}
                                >
                                    <Type size={16} /> Text-to-Video
                                </button>
                                <button
                                    onClick={() => setInputMode('image')}
                                    className={`flex-1 py-3 rounded-lg text-sm font-bold transition-all flex items-center justify-center gap-2 ${inputMode === 'image' ? 'bg-amber-500 text-white dark:text-slate-950 shadow-md' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-200'}`}
                                >
                                    <ImageIcon size={16} /> Image-to-Video
                                </button>
                            </div>

                            {/* Main Input Area */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-5 shadow-sm dark:shadow-none">
                                {inputMode === 'image' && (
                                    <div
                                        onClick={() => fileInputRef.current?.click()}
                                        className="border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl h-40 flex flex-col items-center justify-center cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-900/50 transition relative overflow-hidden group bg-slate-50 dark:bg-slate-950"
                                    >
                                        {imagePreview ? (
                                            <>
                                                <img src={imagePreview} className="w-full h-full object-contain opacity-80 group-hover:opacity-60 transition" />
                                                <div className="absolute inset-0 flex items-center justify-center">
                                                    <span className="bg-white/90 dark:bg-slate-950/90 px-3 py-1.5 rounded-lg text-xs font-bold text-slate-900 dark:text-white border border-slate-200 dark:border-slate-700 shadow-xl transform scale-95 group-hover:scale-100 transition">Replace Image</span>
                                                </div>
                                            </>
                                        ) : (
                                            <>
                                                <Upload className="text-slate-400 dark:text-slate-500 mb-2 w-8 h-8 group-hover:text-amber-500 dark:group-hover:text-amber-400 transition-colors" />
                                                <span className="text-sm font-bold text-slate-500 dark:text-slate-400">Upload image to animate</span>
                                            </>
                                        )}
                                        <input type="file" ref={fileInputRef} onChange={handleImageUpload} className="hidden" accept="image/*" />
                                    </div>
                                )}

                                <div className="space-y-2">
                                    <div className="flex justify-between items-center">
                                        <label className="text-sm font-bold text-slate-700 dark:text-slate-300">
                                            {inputMode === 'text' ? 'Describe your video' : 'Describe the motion'}
                                        </label>
                                        <span className="text-xs text-slate-500">{prompt.length}/4000</span>
                                    </div>
                                    <textarea
                                        value={prompt}
                                        onChange={(e) => setPrompt(e.target.value)}
                                        className="w-full h-32 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none resize-none placeholder:text-slate-400 dark:placeholder:text-slate-600 text-sm leading-relaxed"
                                        placeholder={inputMode === 'text' ? "A drone shot flying over a misty forest at sunrise, cinematic..." : "Camera slowly zooms in, gentle panning to the right..."}
                                    />
                                </div>
                            </div>

                            {/* Settings */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-6 shadow-sm dark:shadow-none">
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 flex items-center gap-2">
                                        <Cpu size={14} className="text-amber-500" /> Video Model
                                    </label>
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500">
                                        {availableModels.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                </div>

                                {/* Aspect Ratio */}
                                <div>
                                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-3 block">Aspect Ratio</label>
                                    <div className="flex gap-2">
                                        {[
                                            { id: '16:9', label: 'Landscape', icon: <div className="w-6 h-3.5 border-2 rounded-sm border-current" /> },
                                            { id: '9:16', label: 'Portrait', icon: <div className="w-3.5 h-6 border-2 rounded-sm border-current" /> },
                                            { id: '1:1', label: 'Square', icon: <div className="w-4 h-4 border-2 rounded-sm border-current" /> }
                                        ].map((ratio: any) => (
                                            <button
                                                key={ratio.id}
                                                onClick={() => setAspectRatio(ratio.id)}
                                                className={`flex-1 py-3 rounded-xl border flex flex-col items-center gap-2 transition-all ${aspectRatio === ratio.id ? 'bg-amber-500/10 border-amber-500 text-amber-600 dark:text-amber-400' : 'bg-slate-50 dark:bg-slate-950 border-slate-200 dark:border-slate-800 text-slate-500 hover:border-slate-300 dark:hover:border-slate-600'}`}
                                            >
                                                {ratio.icon}
                                                <span className="text-[10px] font-bold">{ratio.label}</span>
                                            </button>
                                        ))}
                                    </div>
                                </div>

                                {/* Res & Duration */}
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 block">Resolution</label>
                                        <select
                                            value={resolution} onChange={(e) => setResolution(e.target.value as any)}
                                            className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-2.5 text-sm font-bold text-slate-700 dark:text-slate-300 outline-none focus:border-amber-500"
                                        >
                                            <option value="720p">720p (HD)</option>
                                            <option value="1080p">1080p (FHD)</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 block">Frame Rate</label>
                                        <select
                                            value={fps} onChange={(e) => setFps(Number(e.target.value))}
                                            className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-2.5 text-sm font-bold text-slate-700 dark:text-slate-300 outline-none focus:border-amber-500"
                                        >
                                            <option value="24">24 fps (Cinema)</option>
                                            <option value="30">30 fps (Standard)</option>
                                            <option value="60">60 fps (Smooth)</option>
                                        </select>
                                    </div>
                                </div>

                                <div className="border-t border-slate-200 dark:border-slate-800 pt-4">
                                    <button
                                        onClick={() => setIsAdvancedOpen(!isAdvancedOpen)}
                                        className="flex items-center justify-between w-full text-xs font-bold text-slate-500 hover:text-slate-700 dark:hover:text-slate-300 uppercase"
                                    >
                                        <span className="flex items-center gap-2"><Settings size={12} /> Advanced Options</span>
                                        {isAdvancedOpen ? <ChevronUp size={14} /> : <ChevronDown size={14} />}
                                    </button>
                                    {isAdvancedOpen && (
                                        <div className="mt-4 space-y-4 animate-slide-up">
                                            <div>
                                                <label className="text-xs font-bold text-slate-500 dark:text-slate-400 mb-2 block">Camera Motion</label>
                                                <select
                                                    value={cameraMotion} onChange={(e) => setCameraMotion(e.target.value)}
                                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-2 text-sm text-slate-700 dark:text-slate-300 outline-none"
                                                >
                                                    <option value="none">AI Decides (None)</option>
                                                    <option value="pan_left">Pan Left</option>
                                                    <option value="pan_right">Pan Right</option>
                                                    <option value="zoom_in">Zoom In</option>
                                                    <option value="zoom_out">Zoom Out</option>
                                                    <option value="orbit">Orbit</option>
                                                </select>
                                            </div>
                                            <div className="flex items-center gap-2">
                                                <input type="checkbox" id="loop" className="rounded bg-slate-50 dark:bg-slate-950 border-slate-300 dark:border-slate-800 text-amber-500 focus:ring-0" />
                                                <label htmlFor="loop" className="text-xs text-slate-500 dark:text-slate-400">Seamless Loop</label>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>

                            <button
                                onClick={handleGenerate}
                                disabled={isGenerating || !prompt || !hasAppKey}
                                className="w-full py-4 bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-400 hover:to-amber-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-amber-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed group"
                            >
                                {isGenerating ? <span className="animate-pulse">Forging Video...</span> : <><Video className="fill-current group-hover:scale-110 transition-transform" /> Generate Video 🎬</>}
                                {!isGenerating && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">1 API Call</span>}
                            </button>
                        </div>

                        {/* RIGHT PANEL: PREVIEW / STATUS */}
                        <div className="lg:col-span-8 bg-slate-100 dark:bg-black rounded-2xl border border-slate-200 dark:border-slate-800 relative flex flex-col overflow-hidden min-h-[600px] shadow-inner dark:shadow-2xl">
                            {!generatedVideo && !isGenerating && (
                                <div className="flex-1 flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                    <div className="w-24 h-24 rounded-3xl bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-center mb-6 rotate-3 group hover:rotate-0 transition-transform duration-500 shadow-sm dark:shadow-none">
                                        <Film size={40} className="opacity-50 group-hover:opacity-100 text-amber-500 dark:text-amber-500/50 transition-opacity" />
                                    </div>
                                    <h3 className="text-2xl font-bold text-slate-700 dark:text-slate-500 mb-2">Veo Cinema Studio</h3>
                                    <p className="text-sm text-slate-600 dark:text-slate-700 mt-2 max-w-xs text-center">Configure your shot settings on the left to begin rendering in 1080p.</p>
                                </div>
                            )}

                            {isGenerating && (
                                <div className="flex-1 flex flex-col items-center justify-center bg-white/50 dark:bg-slate-950/80 backdrop-blur-sm relative">
                                    <div className="relative w-40 h-40 mb-12">
                                        <div className="absolute inset-0 border-4 border-slate-200 dark:border-slate-800 rounded-full"></div>
                                        <div className="absolute inset-0 border-4 border-amber-500 rounded-full border-t-transparent animate-spin"></div>
                                        <div className="absolute inset-0 flex items-center justify-center">
                                            <Film className="text-amber-500 animate-pulse" size={48} />
                                        </div>
                                    </div>
                                    <div className="h-8 mb-4 overflow-hidden relative w-full text-center px-4">
                                        <p className="text-amber-600 dark:text-amber-400 font-bold text-lg animate-fade-in key={loadingMsgIndex}">
                                            {LOADING_MESSAGES[loadingMsgIndex]}
                                        </p>
                                    </div>
                                    <div className="flex items-center gap-4 text-sm font-mono text-slate-500 dark:text-slate-400 mt-4 bg-white dark:bg-slate-900/50 px-4 py-2 rounded-full border border-slate-200 dark:border-slate-800">
                                        <span className="flex items-center gap-2 text-amber-600 dark:text-amber-500"><Clock size={14} /> {formatTime(elapsedTime)}</span>
                                    </div>
                                </div>
                            )}

                            {generatedVideo && !isGenerating && (
                                <div className="flex-1 flex flex-col bg-black animate-fade-in">
                                    <div className="flex-1 relative flex items-center justify-center bg-black group">
                                        <video
                                            src={generatedVideo}
                                            controls
                                            autoPlay
                                            loop
                                            className="max-w-full max-h-full shadow-2xl"
                                        />
                                    </div>
                                    <div className="p-6 border-t border-slate-800 bg-slate-900 flex justify-between items-center">
                                        <div className="text-xs text-slate-400 font-mono">
                                            {resolution} • {aspectRatio} • {fps}fps
                                        </div>
                                        <div className="flex gap-3">
                                            <a
                                                href={generatedVideo}
                                                download={`veo_video_${Date.now()}.mp4`}
                                                className="px-4 py-2 bg-slate-800 hover:bg-slate-700 text-white rounded-lg text-xs font-bold transition flex items-center gap-2 border border-slate-700"
                                            >
                                                <Download size={14} /> Download MP4
                                            </a>
                                            <button onClick={() => { navigator.clipboard.writeText(generatedVideo); alert("Link copied!") }} className="p-2 bg-slate-800 hover:bg-slate-700 text-white rounded-lg border border-slate-700 transition">
                                                <Share2 size={14} />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* BULK TAB */}
                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20">
                        <div className="lg:col-span-5 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <h3 className="font-bold text-slate-900 dark:text-slate-200 mb-4">Bulk Video Creator</h3>
                                <p className="text-xs text-slate-500 mb-4">Enter prompts, one per line. All videos will use selected settings.</p>
                                <textarea
                                    value={bulkPrompts}
                                    onChange={(e) => setBulkPrompts(e.target.value)}
                                    className="w-full h-64 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-4 text-slate-900 dark:text-slate-200 text-sm font-mono outline-none resize-none focus:border-amber-500"
                                    placeholder="A cinematic drone shot of a futuristic city&#10;A time-lapse of a flower blooming&#10;Slow motion water splash"
                                />
                                <div className="flex gap-4 mt-4">
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-700 dark:text-slate-300 outline-none focus:border-amber-500">
                                        {VIDEO_MODELS.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                    <button
                                        onClick={handleBulkGenerate}
                                        disabled={bulkStatus === 'processing' || !bulkPrompts || !hasAppKey}
                                        className="flex-1 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50"
                                    >
                                        {bulkStatus === 'processing' ? <RefreshCcw className="animate-spin" size={16} /> : <Play size={16} />}
                                        {bulkStatus === 'processing' ? 'Batching...' : 'Start'}
                                        {bulkStatus !== 'processing' && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">{bulkPrompts.split('\n').filter(p => p.trim()).length} API Calls</span>}
                                    </button>
                                </div>
                            </div>

                            {bulkStatus !== 'idle' && (
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                    <div className="flex justify-between text-sm mb-2 font-bold text-slate-700 dark:text-slate-300">
                                        <span>Batch Progress</span>
                                        <div className="flex items-center gap-2">
                                            {bulkStatus === 'complete' && bulkResults.length > 0 && (
                                                <button onClick={handleDownloadBatch} className="text-xs bg-amber-500/10 text-amber-600 dark:text-amber-400 hover:text-amber-700 dark:hover:text-amber-300 px-2 py-1 rounded transition-colors flex items-center gap-1">
                                                    <Download size={12} /> Download Batch
                                                </button>
                                            )}
                                            <span>{bulkProgress}%</span>
                                        </div>
                                    </div>
                                    <div className="w-full bg-slate-200 dark:bg-slate-800 h-2 rounded-full overflow-hidden">
                                        <div className="h-full bg-amber-500 transition-all duration-300" style={{ width: `${bulkProgress}%` }}></div>
                                    </div>
                                    <div className="mt-4 text-xs text-slate-500">
                                        {bulkResults.length} / {bulkPrompts.split('\n').filter(p => p.trim()).length} Videos Created
                                    </div>
                                </div>
                            )}
                        </div>

                        <div className="lg:col-span-7 bg-white dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 h-[600px] overflow-y-auto custom-scrollbar shadow-sm dark:shadow-none">
                            <div className="grid grid-cols-2 gap-4">
                                {bulkResults.map((asset, idx) => (
                                    <div key={idx} className="relative group rounded-xl overflow-hidden aspect-video bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700">
                                        <video src={asset.url} className="w-full h-full object-cover" controls />
                                        <div className="p-2 bg-white dark:bg-slate-900 border-t border-slate-200 dark:border-slate-700">
                                            <p className="text-xs truncate text-slate-700 dark:text-slate-300">{asset.prompt}</p>
                                        </div>
                                    </div>
                                ))}
                                {bulkResults.length === 0 && (
                                    <div className="col-span-full flex flex-col items-center justify-center text-slate-500 h-64">
                                        <Layers size={32} className="mb-2 opacity-50" />
                                        <p>Generated videos will appear here</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}

                {/* HISTORY TAB */}
                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="flex justify-between items-center mb-4 px-2">
                            <h3 className="font-bold text-slate-700 dark:text-slate-300">Video History ({history.length})</h3>
                            {history.length > 0 && (
                                <button onClick={handleDownloadAllHistory} className="flex items-center gap-2 px-3 py-1.5 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition shadow-lg shadow-amber-500/20">
                                    <Download size={14} /> Download All
                                </button>
                            )}
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-6">
                            {history.map((item) => (
                                <div key={item.id} className="group relative rounded-xl overflow-hidden border border-slate-200 dark:border-slate-800 bg-slate-100 dark:bg-slate-900 aspect-video">
                                    <video src={item.url} className="w-full h-full object-cover" />
                                    <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-all flex flex-col justify-end p-4">
                                        <p className="text-xs text-slate-300 line-clamp-2 mb-2">{item.prompt}</p>
                                        <div className="flex gap-2 justify-end">
                                            <button onClick={() => dbService.toggleFavorite(item.id)} className={`p-1.5 rounded-lg bg-white/10 backdrop-blur ${item.isFavorite ? 'text-red-500' : 'text-white'}`}><Heart size={14} /></button>
                                            <a href={item.url} download className="p-1.5 bg-white/10 backdrop-blur rounded-lg text-white"><Download size={14} /></a>
                                        </div>
                                    </div>
                                </div>
                            ))}
                            {history.length === 0 && (
                                <div className="col-span-full text-center py-12 text-slate-500">
                                    <HistoryIcon size={32} className="mx-auto mb-2 opacity-30" />
                                    No generation history found.
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </div>
            {/* Loading Overlay */}
            {/* Same overlay logic as ImageGenerator if needed, but VideoGenerator seems simpler */}
            {/* Rate Limit Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />
        </div>
    );
};

export default VideoGenerator;
