
import React, { useState, useEffect } from 'react';
import { ListChecks, Circle, ArrowRight, PartyPopper } from 'lucide-react';
import { adminService } from '../services/adminService';
import { TutorialStep, AppView } from '../types';

interface TutorialWidgetProps {
    onOpenSettings?: (tab?: string) => void;
    setView: (view: AppView) => void;
    className?: string;
}

export const TutorialWidget: React.FC<TutorialWidgetProps> = ({ onOpenSettings, setView, className = "" }) => {
    const [steps, setSteps] = useState<TutorialStep[]>([]);
    const [completedIds, setCompletedIds] = useState<Set<string>>(new Set());
    const [progress, setProgress] = useState(0);
    // Initialize visible state based on localStorage to prevent flash
    const [isVisible, setIsVisible] = useState(() => {
        return localStorage.getItem('sf_tutorial_dismissed') !== 'true';
    });

    useEffect(() => {
        // If already dismissed (via initial state check), don't load data
        if (!isVisible) return;

        loadSteps();
        loadProgress();

        const handleUpdate = () => loadSteps();
        window.addEventListener('sf_tutorial_updated', handleUpdate);
        return () => window.removeEventListener('sf_tutorial_updated', handleUpdate);
    }, [isVisible]);

    useEffect(() => {
        if (steps.length > 0) {
            const p = Math.round((completedIds.size / steps.length) * 100);
            setProgress(p);
        } else {
            setProgress(0);
        }
    }, [steps, completedIds]);

    const loadSteps = async () => {
        const data = await adminService.getTutorialSteps();
        // Sort by order to ensure sequential flow
        setSteps(data.sort((a, b) => a.order - b.order));
    };

    const loadProgress = () => {
        const saved = localStorage.getItem('sf_tutorial_progress');
        if (saved) {
            try {
                setCompletedIds(new Set(JSON.parse(saved)));
            } catch (e) {
                // ignore parsing error
            }
        }
    };

    const handleCompleteStep = (id: string) => {
        const newSet = new Set(completedIds);
        newSet.add(id);
        setCompletedIds(newSet);
        localStorage.setItem('sf_tutorial_progress', JSON.stringify(Array.from(newSet)));
    };

    const handleAction = (step: TutorialStep) => {
        if (step.actionLink) {
            if (step.actionLink.startsWith('SETTINGS:')) {
                const tab = step.actionLink.split(':')[1];
                if (onOpenSettings) onOpenSettings(tab);
            } else {
                // Assume it's an AppView enum string
                setView(step.actionLink as AppView);
            }
        }
    };

    // Logic to find the *first* incomplete step
    const activeStepIndex = steps.findIndex(s => !completedIds.has(s.id));
    const isAllDone = steps.length > 0 && activeStepIndex === -1;
    const activeStep = isAllDone ? null : steps[activeStepIndex];

    // Auto-dismiss effect when completed
    useEffect(() => {
        if (isAllDone && isVisible) {
            // Mark as dismissed permanently so it doesn't show on next reload
            localStorage.setItem('sf_tutorial_dismissed', 'true');
            
            // Fade out from current view after a short delay to celebrate completion
            const timer = setTimeout(() => {
                setIsVisible(false);
            }, 4000);
            return () => clearTimeout(timer);
        }
    }, [isAllDone, isVisible]);

    // Don't render if not visible or no steps
    if (!isVisible || steps.length === 0) return null;

    return (
        <div className={`relative bg-slate-900 dark:bg-slate-900/50 rounded-2xl p-5 border border-slate-800 dark:border-slate-800 shadow-sm transition-opacity duration-500 ${className}`}>
            {/* Header & Progress */}
            <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-2">
                    <div className={`p-1.5 rounded-lg ${isAllDone ? 'bg-green-500/20 text-green-400' : 'bg-emerald-500/20 text-emerald-400'}`}>
                        {isAllDone ? <PartyPopper size={16} /> : <ListChecks size={16} />}
                    </div>
                    <h4 className="font-bold text-white text-sm">Getting Started</h4>
                </div>
                <span className={`text-xl font-extrabold ${isAllDone ? 'text-green-500' : 'text-emerald-500'}`}>{progress}%</span>
            </div>

            {/* Progress Bar */}
            <div className="h-1.5 w-full bg-slate-800 rounded-full overflow-hidden mb-5">
                <div 
                    className={`h-full transition-all duration-500 ease-out ${isAllDone ? 'bg-green-500' : 'bg-emerald-500'}`}
                    style={{ width: `${progress}%` }}
                />
            </div>

            {/* Active Step Display */}
            <div className="min-h-[80px]">
                {isAllDone ? (
                    <div className="text-center py-2 animate-fade-in">
                        <p className="text-slate-200 font-bold text-sm mb-1">All steps completed!</p>
                        <p className="text-slate-500 text-xs">You are ready to create.</p>
                    </div>
                ) : (
                    activeStep && (
                        <div key={activeStep.id} className="animate-slide-up">
                            <div className="flex items-start gap-3">
                                <button 
                                    onClick={() => handleCompleteStep(activeStep.id)}
                                    className="mt-0.5 shrink-0 text-slate-500 hover:text-emerald-400 transition-colors group"
                                    title="Mark as complete"
                                >
                                    <Circle size={20} className="group-hover:scale-110 transition-transform" />
                                </button>
                                <div className="flex-1 min-w-0">
                                    <p className="text-sm font-bold text-white mb-1">
                                        {activeStep.title}
                                    </p>
                                    <p className="text-xs text-slate-400 leading-relaxed mb-3">
                                        {activeStep.description}
                                    </p>
                                    
                                    {activeStep.actionLabel && (
                                        <button 
                                            onClick={() => handleAction(activeStep)}
                                            className="text-xs font-bold text-emerald-400 hover:text-emerald-300 flex items-center gap-1 transition-colors bg-emerald-500/10 border border-emerald-500/20 px-3 py-1.5 rounded-lg w-fit hover:bg-emerald-500/20"
                                        >
                                            {activeStep.actionLabel} <ArrowRight size={12} />
                                        </button>
                                    )}
                                </div>
                            </div>
                        </div>
                    )
                )}
            </div>
        </div>
    );
};
