import React from 'react';
import {
    Globe, TrendingUp, Search, Activity, Users, DollarSign,
    Shield, Key, Sparkles, CheckCircle2, AlertTriangle, Zap, BarChart3
} from 'lucide-react';
import {
    AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
    BarChart, Bar, Legend
} from 'recharts';
import { TrendReport, StockSite } from '../types';

// --- MetricCard Component ---
const MetricCard = ({ title, value, icon: Icon, color, subtext, trend, isExport }: any) => {
    const colors: any = {
        blue: { bg: 'bg-blue-500/10', text: 'text-blue-600 dark:text-blue-400', border: 'border-blue-500/20' },
        amber: { bg: 'bg-amber-500/10', text: 'text-amber-600 dark:text-amber-400', border: 'border-amber-500/20' },
        red: { bg: 'bg-red-500/10', text: 'text-red-600 dark:text-red-400', border: 'border-red-500/20' },
        green: { bg: 'bg-emerald-500/10', text: 'text-emerald-600 dark:text-emerald-400', border: 'border-emerald-500/20' }
    };

    const theme = colors[color] || colors.blue;

    return (
        <div className={`relative overflow-hidden p-5 rounded-2xl transition-all duration-300 group ${isExport ? 'bg-white border border-slate-300 shadow-none' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm hover:shadow-lg'}`}>
            {/* Background Gradient Blob - Hidden in Export */}
            {!isExport && <div className={`absolute -top-10 -right-10 w-32 h-32 rounded-full ${theme.bg} blur-3xl opacity-0 group-hover:opacity-100 transition-opacity duration-500`} />}

            <div className="relative z-10">
                <div className="flex justify-between items-start mb-4">
                    <div className={`p-3 rounded-xl ${theme.bg} ${theme.text} border ${theme.border} shadow-sm group-hover:scale-110 transition-transform duration-300`}>
                        <Icon size={22} />
                    </div>
                    {trend && (
                        <div className={`flex items-center gap-1 text-[10px] font-bold ${trend > 0 ? 'text-emerald-500' : 'text-red-500'} bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded-full border border-slate-200 dark:border-slate-700`}>
                            <TrendingUp size={10} className={trend < 0 ? 'rotate-180' : ''} />
                            {Math.abs(trend)}%
                        </div>
                    )}
                </div>

                <div>
                    <h3 className="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1">{title}</h3>
                    <div className="flex items-baseline gap-2">
                        <p className={`text-2xl font-bold font-['Space_Grotesk'] tracking-tight ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>{value}</p>
                    </div>
                    {subtext && (
                        <p className="text-xs text-slate-400 mt-2 font-medium leading-relaxed">{subtext}</p>
                    )}
                </div>
            </div>
        </div>
    );
};

interface TrendReportViewProps {
    report: TrendReport;
    sitesList: StockSite[];
    children?: React.ReactNode; // For header buttons
    isExport?: boolean;
}

export const TrendReportView: React.FC<TrendReportViewProps> = ({ report, sitesList, children, isExport = false }) => {
    // Helper to calculate total reach
    const calculateTotalReach = () => {
        const idsToUse = report.selectedPlatforms || [];
        let total = 0;
        sitesList.filter((s) => idsToUse.includes(s.id)).forEach((s) => {
            const num = parseFloat(s.librarySize.replace(/[^0-9.]/g, ''));
            if (s.librarySize.includes('M')) total += num * 1000000;
            else if (s.librarySize.includes('k')) total += num * 1000;
            else total += num;
        });
        return total > 1000000 ? (total / 1000000).toFixed(1) + 'M+' : (total / 1000).toFixed(0) + 'k+';
    };

    // Helper to calculate avg commission
    const calculateAvgCommission = () => {
        const idsToUse = report.selectedPlatforms || [];
        const sites = sitesList.filter((s) => idsToUse.includes(s.id));
        if (sites.length === 0) return '0%';
        let total = 0;
        let count = 0;
        sites.forEach((s) => {
            const match = s.commission.match(/(\d+)/);
            if (match) {
                total += parseInt(match[0]);
                count++;
            }
        });
        return count > 0 ? Math.round(total / count) + '%' : 'N/A';
    };

    // Helper to calculate AI acceptance
    const calculateAiAcceptance = () => {
        const idsToUse = report.selectedPlatforms || [];
        const sites = sitesList.filter((s) => idsToUse.includes(s.id));
        if (sites.length === 0) return '0%';
        const aiSites = sites.filter((s) => s.isAiPowered).length;
        return Math.round((aiSites / sites.length) * 100) + '%';
    };

    return (
        <div className={`space-y-8 animate-slide-up p-8 rounded-3xl ${isExport ? 'bg-white text-slate-900' : 'bg-white dark:bg-slate-950'}`}>
            {/* Header */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <div>
                    <div className="flex items-center gap-3 mb-1">
                        <h2 className={`text-3xl font-bold capitalize ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>{report.query}</h2>
                        <span className="px-3 py-1 bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400 text-xs font-bold rounded-full border border-green-200 dark:border-green-500/30">
                            High Potential
                        </span>
                    </div>
                    <p className={`text-sm ${isExport ? 'text-slate-500' : 'text-slate-500 dark:text-slate-400'}`}>Analysis generated on {new Date(report.timestamp).toLocaleDateString()} based on {report.selectedPlatforms.length} sources.</p>
                </div>
                <div className="flex gap-3" data-html2canvas-ignore>
                    {children}
                </div>
            </div>

            {/* Platform & Keyword Intelligence */}
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
                {/* Platform Reach */}
                <div className={`rounded-2xl p-5 ${isExport ? 'bg-white border border-slate-300' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                    <h3 className="text-sm font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-4 flex items-center gap-2">
                        <Globe size={14} className="text-blue-500" /> Platform Intelligence
                    </h3>
                    <div className="space-y-4">
                        <div className="flex justify-between items-center">
                            <span className="text-sm text-slate-600 dark:text-slate-400">Selected Platforms</span>
                            <span className={`font-bold ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                                {report.selectedPlatforms.length}
                            </span>
                        </div>
                        <div className="flex justify-between items-center">
                            <span className="text-sm text-slate-600 dark:text-slate-400">Total Asset Reach</span>
                            <span className={`font-bold ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                                {calculateTotalReach()}
                            </span>
                        </div>
                        <div className="flex justify-between items-center">
                            <span className="text-sm text-slate-600 dark:text-slate-400">Avg. Commission</span>
                            <span className="font-bold text-emerald-600 dark:text-emerald-400">
                                {calculateAvgCommission()}
                            </span>
                        </div>
                        <div className="flex justify-between items-center">
                            <span className="text-sm text-slate-600 dark:text-slate-400">AI Acceptance Rate</span>
                            <span className="font-bold text-purple-600 dark:text-purple-400">
                                {calculateAiAcceptance()}
                            </span>
                        </div>
                    </div>
                </div>

                {/* Niche Keywords */}
                <div className={`lg:col-span-2 rounded-2xl p-5 ${isExport ? 'bg-white border border-slate-300' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                    <h3 className="text-sm font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-4 flex items-center gap-2">
                        <Key size={14} className="text-amber-500" /> Niche Keywords & Expansion
                    </h3>
                    <div className="flex flex-wrap gap-2">
                        {report.trend.subTrends.map((keyword: string, i: number) => (
                            <span key={i} className={`px-3 py-1.5 rounded-lg text-xs font-bold flex items-center gap-1 ${isExport ? 'bg-slate-100 text-slate-700 border border-slate-200' : 'bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 border border-slate-200 dark:border-slate-700'}`}>
                                <Search size={10} className="opacity-50" /> {keyword}
                            </span>
                        ))}
                        {report.insights.takeaways.slice(0, 2).map((t: string, i: number) => {
                            const words = t.match(/\b[A-Z][a-z]{3,}\b/g) || [];
                            return words.slice(0, 2).map((w, j) => (
                                <span key={`extra-${i}-${j}`} className={`px-3 py-1.5 rounded-lg text-xs font-bold flex items-center gap-1 ${isExport ? 'bg-amber-50 text-amber-700 border border-amber-100' : 'bg-amber-50 dark:bg-amber-900/10 text-amber-700 dark:text-amber-400 border border-amber-100 dark:border-amber-500/20'}`}>
                                    <Sparkles size={10} className="opacity-50" /> {w}
                                </span>
                            ));
                        })}
                    </div>
                    <div className="mt-4 pt-4 border-t border-slate-100 dark:border-slate-800 flex items-center justify-between text-xs text-slate-500">
                        <span>Based on analysis of {report.saturation.totalVolume.toLocaleString()} assets</span>
                        <span className="flex items-center gap-1"><Activity size={12} /> Volume: {report.saturation.level}</span>
                    </div>
                </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <MetricCard
                    title="Confidence Score"
                    value={`${report.insights.confidenceScore}/100`}
                    icon={Shield}
                    color="blue"
                    subtext="AI-calculated probability of success based on current market trends."
                    trend={report.insights.confidenceScore > 80 ? 12 : 5}
                    isExport={isExport}
                />
                <MetricCard
                    title="Saturation Level"
                    value={report.saturation.level}
                    icon={Activity}
                    color="amber"
                    subtext="Current density of similar content across major platforms."
                    isExport={isExport}
                />
                <MetricCard
                    title="Competition"
                    value={`${report.saturation.competitionLevel}/10`}
                    icon={Users}
                    color="red"
                    subtext="Intensity of active contributors in this specific niche."
                    isExport={isExport}
                />
                <MetricCard
                    title="Est. Monthly ROI"
                    value={report.pricing.potentialMonthly}
                    icon={DollarSign}
                    color="green"
                    subtext="Projected earnings for a strong, high-quality portfolio."
                    trend={15}
                    isExport={isExport}
                />
            </div>

            {/* Main Content Grid */}
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                {/* Left Column: Insights & Action Plan */}
                <div className="lg:col-span-2 space-y-8">
                    {/* Actionable Insights */}
                    <div className={`rounded-2xl p-6 ${isExport ? 'bg-white border border-slate-300' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                        <h3 className={`font-bold mb-6 flex items-center gap-2 ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                            <Sparkles className="text-purple-500" size={18} /> Strategic Insights
                        </h3>
                        <div className="grid gap-4">
                            {report.insights.create.map((item: any, i: number) => (
                                <div key={i} className={`flex gap-4 p-4 rounded-xl ${isExport ? 'bg-green-50 border border-green-100' : 'bg-green-50 dark:bg-green-900/10 border border-green-100 dark:border-green-500/20'}`}>
                                    <div className={`mt-1 p-1.5 rounded-full h-fit ${isExport ? 'bg-green-100' : 'bg-green-100 dark:bg-green-500/20'}`}>
                                        <CheckCircle2 size={18} className={isExport ? 'text-green-600' : 'text-green-600 dark:text-green-400'} />
                                    </div>
                                    <div>
                                        <h4 className={`font-bold text-sm mb-1 ${isExport ? 'text-green-800' : 'text-green-800 dark:text-green-400'}`}>Create: {item.title}</h4>
                                        <p className={`text-sm ${isExport ? 'text-slate-600' : 'text-slate-600 dark:text-slate-400'}`}>{item.reason}</p>
                                    </div>
                                </div>
                            ))}
                            {report.insights.caution.map((item: any, i: number) => (
                                <div key={i} className={`flex gap-4 p-4 rounded-xl ${isExport ? 'bg-amber-50 border border-amber-100' : 'bg-amber-50 dark:bg-amber-900/10 border border-amber-100 dark:border-amber-500/20'}`}>
                                    <div className={`mt-1 p-1.5 rounded-full h-fit ${isExport ? 'bg-amber-100' : 'bg-amber-100 dark:bg-amber-500/20'}`}>
                                        <AlertTriangle size={18} className={isExport ? 'text-amber-600' : 'text-amber-600 dark:text-amber-400'} />
                                    </div>
                                    <div>
                                        <h4 className={`font-bold text-sm mb-1 ${isExport ? 'text-amber-800' : 'text-amber-800 dark:text-amber-400'}`}>Caution: {item.title}</h4>
                                        <p className={`text-sm ${isExport ? 'text-slate-600' : 'text-slate-600 dark:text-slate-400'}`}>{item.reason}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>

                    {/* Trend Chart */}
                    <div id="volume-chart-container" className={`rounded-2xl p-6 h-80 relative overflow-hidden group ${isExport ? 'bg-white border border-slate-300' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                        {!isExport && <div className="absolute top-0 right-0 w-64 h-64 bg-blue-500/5 rounded-full blur-3xl -mr-16 -mt-16 pointer-events-none"></div>}
                        <h3 className={`font-bold mb-4 flex items-center gap-2 relative z-10 ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                            <TrendingUp className="text-blue-500" size={18} /> Volume Trend (Last 6 Weeks)
                        </h3>
                        <ResponsiveContainer width="100%" height="100%">
                            <AreaChart data={report.trend.data} margin={{ top: 10, right: 10, left: -20, bottom: 0 }}>
                                <defs>
                                    <linearGradient id="colorVal" x1="0" y1="0" x2="0" y2="1">
                                        <stop offset="5%" stopColor="#3b82f6" stopOpacity={0.4} />
                                        <stop offset="95%" stopColor="#3b82f6" stopOpacity={0} />
                                    </linearGradient>
                                </defs>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.1} vertical={false} />
                                <XAxis
                                    dataKey="name"
                                    stroke="#64748b"
                                    fontSize={10}
                                    tickLine={false}
                                    axisLine={false}
                                    dy={10}
                                />
                                <YAxis
                                    stroke="#64748b"
                                    fontSize={10}
                                    tickLine={false}
                                    axisLine={false}
                                />
                                <Tooltip
                                    contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc', boxShadow: '0 10px 15px -3px rgba(0, 0, 0, 0.5)' }}
                                    itemStyle={{ color: '#60a5fa' }}
                                    cursor={{ stroke: '#3b82f6', strokeWidth: 1, strokeDasharray: '4 4' }}
                                />
                                <Area
                                    type="monotone"
                                    dataKey="val"
                                    stroke="#3b82f6"
                                    strokeWidth={3}
                                    fillOpacity={1}
                                    fill="url(#colorVal)"
                                    activeDot={{ r: 6, strokeWidth: 0, fill: '#60a5fa' }}
                                    isAnimationActive={!isExport}
                                />
                            </AreaChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                {/* Right Column: Creative Brief & Stats */}
                <div className="space-y-8">
                    {/* Creative Direction */}
                    <div className={`rounded-2xl p-6 ${isExport ? 'bg-white border border-slate-300' : 'bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                        <h3 className={`font-bold mb-6 flex items-center gap-2 ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                            <Zap className="text-amber-500" size={18} /> Creative Direction
                        </h3>

                        <div className="mb-6">
                            <h4 className="text-xs font-bold text-slate-500 uppercase mb-3">Suggested Concepts</h4>
                            <div className="flex flex-wrap gap-2">
                                {report.creative.ideas.map((idea: string, i: number) => (
                                    <span key={i} className={`px-3 py-1.5 text-xs font-bold rounded-lg border ${isExport ? 'bg-slate-100 text-slate-700 border-slate-200' : 'bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 border-slate-200 dark:border-slate-700'}`}>
                                        {idea}
                                    </span>
                                ))}
                            </div>
                        </div>

                        <div>
                            <h4 className="text-xs font-bold text-slate-500 uppercase mb-3">Trending Palette</h4>
                            <div className={`flex h-12 rounded-xl overflow-hidden shadow-sm border ${isExport ? 'border-slate-200' : 'border-slate-200 dark:border-slate-700'}`}>
                                {report.creative.colorPalette.map((color: string, i: number) => (
                                    <div key={i} className="flex-1 h-full" style={{ backgroundColor: color }} title={color} />
                                ))}
                            </div>
                        </div>
                    </div>

                    {/* Supply vs Demand */}
                    <div id="supply-demand-chart-container" className={`rounded-2xl p-6 h-80 relative overflow-hidden ${isExport ? 'bg-white border border-slate-300' : 'bg-gradient-to-br from-white to-slate-50 dark:from-slate-900 dark:to-slate-900/50 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none'}`}>
                        <div className="flex justify-between items-start mb-6 relative z-10">
                            <div>
                                <h3 className={`font-bold flex items-center gap-2 ${isExport ? 'text-slate-900' : 'text-slate-900 dark:text-white'}`}>
                                    <BarChart3 className="text-emerald-500" size={18} /> Supply vs Demand
                                </h3>
                                <p className="text-xs text-slate-500 mt-1">Market saturation analysis over time</p>
                            </div>
                            {(() => {
                                const last = report.saturation.supplyData[report.saturation.supplyData.length - 1];
                                const gap = last ? last.demand - last.supply : 0;
                                const isPositive = gap > 0;
                                return (
                                    <div className={`text-right ${isPositive ? 'text-emerald-500' : 'text-red-500'}`}>
                                        <span className="text-[10px] font-bold uppercase tracking-wider block text-slate-400">Opportunity Gap</span>
                                        <span className="text-xl font-bold">{isPositive ? '+' : ''}{gap}%</span>
                                    </div>
                                );
                            })()}
                        </div>

                        <ResponsiveContainer width="100%" height="100%" className="relative z-10 -ml-2">
                            <BarChart data={report.saturation.supplyData} margin={{ top: 5, right: 5, left: 0, bottom: 25 }}>
                                <defs>
                                    <linearGradient id="supplyGradient" x1="0" y1="0" x2="0" y2="1">
                                        <stop offset="0%" stopColor="#ef4444" stopOpacity={0.8} />
                                        <stop offset="95%" stopColor="#ef4444" stopOpacity={0.4} />
                                    </linearGradient>
                                    <linearGradient id="demandGradient" x1="0" y1="0" x2="0" y2="1">
                                        <stop offset="0%" stopColor="#10b981" stopOpacity={0.8} />
                                        <stop offset="95%" stopColor="#10b981" stopOpacity={0.4} />
                                    </linearGradient>
                                </defs>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.1} vertical={false} />
                                <XAxis dataKey="name" stroke="#64748b" fontSize={10} tickLine={false} axisLine={false} dy={10} />
                                <YAxis stroke="#64748b" fontSize={10} tickLine={false} axisLine={false} />
                                <Tooltip
                                    cursor={{ fill: 'rgba(255,255,255,0.05)' }}
                                    contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc', boxShadow: '0 10px 15px -3px rgba(0, 0, 0, 0.5)' }}
                                />
                                <Bar dataKey="supply" fill="url(#supplyGradient)" radius={[4, 4, 0, 0]} name="Supply (Uploads)" barSize={20} isAnimationActive={!isExport} />
                                <Bar dataKey="demand" fill="url(#demandGradient)" radius={[4, 4, 0, 0]} name="Demand (Sales)" barSize={20} isAnimationActive={!isExport} />
                                <Legend wrapperStyle={{ paddingTop: '20px' }} iconType="circle" />
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>
            </div>
        </div>
    );
};
