import React, { useState, useEffect, useRef } from 'react';
import { User, AppView, SupportMessage } from '../types';
import {
    Menu, Search, Bell, ChevronRight, Settings, CreditCard,
    Book, Bug, LogOut, User as UserIcon, CheckCircle2, AlertTriangle,
    MessageSquare, Globe, Check, Image as ImageIcon, Video, Trash2,
    FileText, Loader2, Clock, X, CheckCheck, Calendar, Megaphone, Zap, Info
} from 'lucide-react';
import { dbService } from '../services/dbService';
import { adminService } from '../services/adminService';
import { UserDatabase, UserNotification } from '../services/USER-database';
import { authService } from '../services/authService';
import { MODELS, MODEL_TYPE_TO_SETTINGS } from '../constants';

interface TopBarProps {
    title: string;
    toggleMobileSidebar: () => void;
    user: User | null;
    onLogout: () => void;
    setView: (view: AppView) => void;
    onOpenDocumentation?: () => void;
    onOpenSettings?: (tab?: string) => void;
    appName?: string;
    unreadMessagesCount?: number;
}

const LANGUAGES = [
    { code: 'en', label: 'English', flag: '🇺🇸' },
    { code: 'es', label: 'Español', flag: '🇪🇸' },
    { code: 'fr', label: 'Français', flag: '🇫🇷' },
    { code: 'de', label: 'Deutsch', flag: '🇩🇪' },
    { code: 'it', label: 'Italiano', flag: '🇮🇹' },
    { code: 'pt', label: 'Português', flag: '🇵🇹' },
    { code: 'zh-CN', label: '中文 (简体)', flag: '🇨🇳' },
    { code: 'ja', label: '日本語', flag: '🇯🇵' },
    { code: 'ko', label: '한국어', flag: '🇰🇷' },
    { code: 'ru', label: 'Русский', flag: '🇷🇺' },
    { code: 'ar', label: 'العربية', flag: '🇸🇦' },
];

const TopBar: React.FC<TopBarProps> = ({ title, toggleMobileSidebar, user, onLogout, setView, onOpenDocumentation, onOpenSettings, appName = "StockForge", unreadMessagesCount = 0 }) => {
    const [isUserMenuOpen, setIsUserMenuOpen] = useState(false);
    const [isNotifOpen, setIsNotifOpen] = useState(false);
    const [isLangMenuOpen, setIsLangMenuOpen] = useState(false);
    const [currentLang, setCurrentLang] = useState('en');

    // Notifications State
    const [notifications, setNotifications] = useState<UserNotification[]>([]);

    // Modals State
    const [selectedNotification, setSelectedNotification] = useState<UserNotification | null>(null);
    const [isHistoryOpen, setIsHistoryOpen] = useState(false);

    // Search State
    const [searchQuery, setSearchQuery] = useState('');
    const [searchResults, setSearchResults] = useState<any[]>([]);
    const [isSearching, setIsSearching] = useState(false);
    const [showResults, setShowResults] = useState(false);
    const searchRef = useRef<HTMLDivElement>(null);

    // Timer State
    const [timeToReset, setTimeToReset] = useState('');
    const [enabledModelsLimit, setEnabledModelsLimit] = useState<number | null>(null);

    // Calculate credits from ONLY enabled models
    useEffect(() => {
        const fetchEnabledModelsLimit = async () => {
            if (!user?.uid) return;

            try {
                // Fetch settings and keys for dynamic calculation
                const siteSettings = await authService.getSiteSettings();
                const apiKeys = await authService.getUserApiKeys(user.uid);

                // DEFAULT TIER LIMITS (Must match Server/Admin)
                const DEFAULT_TIER_LIMITS: Record<string, { free: number, paid: number }> = {
                    'pro_3_0': { free: 50, paid: 500 },
                    'pro_2_5': { free: 50, paid: 500 },
                    'flash_2_5': { free: 300, paid: 1000 },
                    'flash_lite_2_5': { free: 0, paid: 1000 }, // Fixed: Was 200, causing discrepancy
                    'imagen_ultra': { free: 10, paid: 500 },
                    'imagen_fast': { free: 10, paid: 500 },
                    'imagen_std': { free: 10, paid: 500 },
                    'gemini_3_pro_image': { free: 50, paid: 500 },
                    'gemini_2_5_flash_image': { free: 100, paid: 500 },
                    'veo_3_1': { free: 20, paid: 500 },
                    'veo_3_1_fast': { free: 50, paid: 500 },
                    'veo_3_0': { free: 10, paid: 500 },
                    'veo_2_0': { free: 5, paid: 500 },
                    'gemini_1_5_pro': { free: 100, paid: 1000 },
                };

                // Sum up real dynamic limits
                let total = 0;
                let hasValidKeys = false;

                if (apiKeys && apiKeys.length > 0) {
                    apiKeys.forEach(key => {
                        // Allow 'valid' AND 'unchecked' to match authService logic
                        if ((key.status === 'valid' || key.status === 'unchecked') && key.isEnabled !== false) {
                            hasValidKeys = true;
                            const tier = (key.tier || 'FREE').toUpperCase();
                            const type = (tier === 'FREE') ? 'free' : 'paid';

                            MODELS.forEach(model => {
                                const settingKey = MODEL_TYPE_TO_SETTINGS[model.id];
                                if (!settingKey) return;

                                // Check if model is enabled for this tier
                                const isEnabled = siteSettings?.modelAccessTiers?.[model.id]?.[type] !== false;

                                if (isEnabled) {
                                    const limit = siteSettings?.modelLimitsByTier?.[settingKey]?.[type]
                                        ?? DEFAULT_TIER_LIMITS[settingKey]?.[type]
                                        ?? 0;
                                    total += limit;
                                }
                            });
                        }
                    });
                }

                // If we explicitly fetched keys and the list is empty, user has NO limits (0).
                // If the list is null/undefined (failed fetch), only then fallback to usageStats.
                if (apiKeys && apiKeys.length === 0) {
                    setEnabledModelsLimit(0);
                } else if (hasValidKeys) {
                    setEnabledModelsLimit(total);
                } else {
                    // Fallback only if we couldn't confirm empty keys or if logic failed
                    // setEnabledModelsLimit(-1); // Use -1 to indicate fallback needed
                    setEnabledModelsLimit(user.usageStats?.apiCallsLimit || 0);
                }

            } catch (error) {
                console.error('Failed to calculate enabled models limit:', error);
                // Fallback to apiCallsLimit if calculation fails
                setEnabledModelsLimit(user?.usageStats?.apiCallsLimit || 0);
            }
        };

        fetchEnabledModelsLimit();

        // Listen for live updates (e.g. key deletion)
        const handleUserUpdate = () => fetchEnabledModelsLimit();
        window.addEventListener('sf_user_updated', handleUserUpdate);

        return () => window.removeEventListener('sf_user_updated', handleUserUpdate);
    }, [user?.uid]); // REMOVED user.usageStats to fix infinite loop!

    // Calculate Remaining Credits for Display
    // Calculate Remaining Credits for Display
    // If enabledModelsLimit is NOT null (meaning we successfully calculated it, even if 0), use it.
    // Otherwise fallback to user.usageStats (legacy/server provided).
    const apiLimit = (enabledModelsLimit !== null) ? enabledModelsLimit : (user?.usageStats?.apiCallsLimit || 0);
    const apiUsed = user?.usageStats?.daily?.apiCalls || 0;
    const apiRemaining = Math.max(0, apiLimit - apiUsed);

    useEffect(() => {
        const checkTranslate = setInterval(() => {
            const select = document.querySelector('.goog-te-combo') as HTMLSelectElement;
            if (select) {
                if (select.value && select.value !== currentLang) {
                    setCurrentLang(select.value);
                }
            }
        }, 2000);

        const handleClickOutside = (event: MouseEvent) => {
            if (searchRef.current && !searchRef.current.contains(event.target as Node)) {
                setShowResults(false);
            }
        };

        document.addEventListener('mousedown', handleClickOutside);

        // Reset Timer Logic
        const resetTriggeredRef = { current: false };
        const lastKnownDayRef = { current: new Date().toLocaleString('en-US', { timeZone: 'America/Los_Angeles' }).split(',')[0] };

        const updateTimer = () => {
            const now = new Date();
            const ptNowString = now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' });

            // Check for Day Change (Midnight PT) based on string to reset the trigger lock
            const currentPtDay = ptNowString.split(',')[0];
            if (currentPtDay !== lastKnownDayRef.current) {
                console.log(`[TopBar] Day changed (String Check) from ${lastKnownDayRef.current} to ${currentPtDay}. Unlocking reset trigger.`);
                lastKnownDayRef.current = currentPtDay;
                resetTriggeredRef.current = false; // Allow trigger for the new day
            }

            // Create PT Midnight object for calculations
            // We use the string construction to avoid local timezone offset issues
            const ptDate = new Date(ptNowString);
            const ptMidnight = new Date(ptDate);
            ptMidnight.setDate(ptMidnight.getDate() + 1);
            ptMidnight.setHours(0, 0, 0, 0);

            // Current PT time in ms (approximation for diff)
            // We need to compare "now" vs "endpoint".
            // Since ptMidnight is a "Date" object created from a string, its .getTime() reflects the time *if* the server was in that timezone?
            // No, new Date(string) interprets based on browser/server local timezone if not ISO.
            // But toLocaleString gives "12/26/2025, 12:00:00 AM".
            // The safest way to get diff is to start from "now" (real UTC timestamp) and target timestamp.
            // But we don't know the exact offset easily without a library.
            // HOWEVER, we can just calculate the difference in seconds from the time components.

            const hours = ptDate.getHours();
            const mins = ptDate.getMinutes();
            const secs = ptDate.getSeconds();

            // Seconds until next midnight (24h clock)
            // Midnight is 24:00:00 aka 00:00:00 next day
            // Total seconds in a day = 86400
            const currentSeconds = (hours * 3600) + (mins * 60) + secs;
            const diffSeconds = 86400 - currentSeconds;

            if (diffSeconds <= 0) {
                setTimeToReset("00:00:00");
                // Prevent spamming
                if (!resetTriggeredRef.current && user?.uid) {
                    console.log("[TopBar] 🕛 MIDNIGHT PT REACHED! Triggering Daily Reset...");
                    resetTriggeredRef.current = true;
                    authService.checkDailyReset(user.uid).then(() => {
                        console.log("[TopBar] Daily reset check complete.");
                        // Force fetch profile to ensure UI updates immediately
                        authService.getUserProfile(user.uid).catch(console.error);
                    });
                }
                return;
            }

            // Trigger if close to zero (e.g. 1 second left) to catch it exactly
            if (diffSeconds <= 1 && !resetTriggeredRef.current && user?.uid) {
                console.log("[TopBar] 🕐 Almost Midnight PT! Pre-triggering reset...");
                resetTriggeredRef.current = true;
                authService.checkDailyReset(user.uid);
            }

            const h = Math.floor(diffSeconds / 3600);
            const m = Math.floor((diffSeconds % 3600) / 60);
            const s = Math.floor(diffSeconds % 60);
            setTimeToReset(`${h.toString().padStart(2, '0')}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`);
        };

        updateTimer();
        const interval = setInterval(updateTimer, 1000);

        return () => {
            clearInterval(interval);
            document.removeEventListener('mousedown', handleClickOutside);
        };
    }, [user?.uid]);

    // Broadcast Notification Sound - Play ONLY when unread count increases (red dot)
    const prevUnreadNotifRef = useRef(-1);
    useEffect(() => {
        const soundEnabled = user?.preferences?.uiConfig?.notifications ?? true;
        if (!soundEnabled) return;

        const currentUnread = notifications.filter(n => !n.read).length;

        // Play sound only if unread count increased (new notification)
        if (currentUnread > 0 &&
            currentUnread > prevUnreadNotifRef.current &&
            prevUnreadNotifRef.current >= 0) {
            try {
                const ctx = new (window.AudioContext || (window as any).webkitAudioContext)();
                const osc = ctx.createOscillator();
                const gain = ctx.createGain();

                osc.connect(gain);
                gain.connect(ctx.destination);

                // Longer, more distinct sound (2 seconds)
                // Sequence: High -> Low -> High (Ding-Dong-Ding style)
                const now = ctx.currentTime;

                osc.type = 'sine';

                // 0.0s - 0.5s: High pitch (880Hz)
                osc.frequency.setValueAtTime(880, now);
                gain.gain.setValueAtTime(0.1, now);

                // 0.5s - 1.0s: Drop to 440Hz
                osc.frequency.exponentialRampToValueAtTime(440, now + 0.5);

                // 1.0s - 1.5s: Back up to 660Hz
                osc.frequency.exponentialRampToValueAtTime(660, now + 1.0);

                // 1.5s - 2.0s: Fade out
                gain.gain.exponentialRampToValueAtTime(0.01, now + 2.0);

                osc.start(now);
                osc.stop(now + 2.0);
            } catch (e) {
                console.error("Audio play failed", e);
            }
        }

        prevUnreadNotifRef.current = currentUnread;
    }, [notifications, user?.preferences?.uiConfig?.notifications]);

    // Message Notification Sound - Play ONLY when red dot appears (unread > 0)
    const prevUnreadMessagesRef = useRef(-1);
    useEffect(() => {
        const soundEnabled = user?.preferences?.uiConfig?.notifications ?? true;

        // Play sound only if:
        // 1. Sound is enabled
        // 2. Unread count increased (new message received)
        // 3. Previous count was known (not initial load)
        if (soundEnabled &&
            unreadMessagesCount > 0 &&
            unreadMessagesCount > prevUnreadMessagesRef.current &&
            prevUnreadMessagesRef.current >= 0) {

            try {
                const ctx = new (window.AudioContext || (window as any).webkitAudioContext)();
                const osc = ctx.createOscillator();
                const gain = ctx.createGain();

                osc.connect(gain);
                gain.connect(ctx.destination);

                const now = ctx.currentTime;
                osc.type = 'sine';

                // Message notification sound: Higher pitched, quick
                osc.frequency.setValueAtTime(1000, now);
                gain.gain.setValueAtTime(0.15, now);

                // Quick chirp
                osc.frequency.exponentialRampToValueAtTime(800, now + 0.1);
                gain.gain.exponentialRampToValueAtTime(0.01, now + 0.2);

                osc.start(now);
                osc.stop(now + 0.2);
            } catch (e) {
                console.error("Message notification sound failed", e);
            }
        }

        // Update previous count for next comparison
        prevUnreadMessagesRef.current = unreadMessagesCount;
    }, [unreadMessagesCount, user?.preferences?.uiConfig?.notifications]);

    // Notification Logic: Poll and Merge
    useEffect(() => {
        const fetchNotifications = async () => {
            if (!user) return;

            // 1. Fetch existing specific user notifications from DB
            let currentNotifs = await UserDatabase.getNotifications(user.uid);


            // 2. Check for System Broadcasts & Add if New
            let hasUpdates = false;
            try {
                const broadcasts = await adminService.getBroadcasts();
                const prefs = await UserDatabase.getPreferences(user.uid);
                const dismissed = prefs.dismissedBroadcasts || [];

                broadcasts.forEach(b => {
                    const exists = currentNotifs.some(n => n.id === `broadcast_${b.id}`);
                    const isDismissed = dismissed.includes(b.id);

                    if (!exists && !isDismissed) {
                        // Only add if target matches
                        const target = (b.targetAudience || 'all').toLowerCase();
                        if (target.includes('all') ||
                            user.role === 'admin' || // Admins see all
                            (user.subscription && target.includes(user.subscription.toLowerCase()))) {

                            // Map broadcast content correctly
                            currentNotifs.push({
                                id: `broadcast_${b.id}`,
                                title: b.title,
                                text: b.content, // Using correct content field
                                type: b.type === 'urgent' ? 'urgent' : b.type === 'notice' ? 'warning' : 'info',
                                category: b.type === 'feature' ? 'Feature' : 'Announcement',
                                timestamp: b.sentAt,
                                read: false
                            });
                            hasUpdates = true;
                        }
                    }
                });
            } catch (error) {
                console.error("Failed to fetch broadcasts for notifications:", error);
            }

            // 3. Check for Support Messages & Add if New
            let messages: SupportMessage[] | undefined;
            try {
                if (user.role === 'admin') {
                    // Admin: Get all messages and filter for unread user messages
                    messages = await adminService.getSupportMessages();
                    messages = messages.filter(m => m.sender === 'user' && m.status !== 'read');
                } else {
                    // User: Get their own messages and filter for unread admin messages
                    messages = await adminService.getSupportMessages(user.uid);
                    messages = messages.filter(m => m.sender === 'admin' && m.status !== 'read');
                }

                // Get dismissed message IDs from preferences
                const prefs = await UserDatabase.getPreferences(user.uid);
                const dismissedMessages = prefs.dismissedMessages || [];

                messages.forEach(msg => {
                    const notifId = `message_${msg.id}`;
                    const exists = currentNotifs.some(n => n.id === notifId);
                    const isDismissed = dismissedMessages.includes(msg.id);

                    if (!exists && !isDismissed) {
                        currentNotifs.unshift({
                            id: notifId,
                            title: user.role === 'admin' ? `New message from ${msg.userName || msg.userEmail || 'User'}` : 'New message from Admin',
                            text: msg.text,
                            type: 'info',
                            category: 'Support',
                            timestamp: msg.timestamp,
                            read: false,
                            messageData: msg // Store the full message for later use
                        });
                        hasUpdates = true;
                    }
                });
            } catch (error) {
                console.error("Failed to fetch support messages for notifications:", error);
            }

            // 4. Check for Activity Logs (Generation Info) & Add if New
            try {
                const logs = await UserDatabase.getActivityLogs(user.uid);
                // Only show recent logs (last 24h) or important ones
                const recentLogs = logs.filter(l => Date.now() - l.timestamp < 24 * 60 * 60 * 1000);

                recentLogs.forEach(log => {
                    const notifId = `log_${log.id}`;
                    const exists = currentNotifs.some(n => n.id === notifId);

                    if (!exists) {
                        currentNotifs.unshift({
                            id: notifId,
                            title: log.type === 'generate' ? 'Generation Complete' : log.type === 'error' ? 'Generation Failed' : 'System Activity',
                            text: log.text,
                            type: log.type === 'error' ? 'urgent' : 'success',
                            category: 'System',
                            timestamp: log.timestamp,
                            read: false
                        });
                        hasUpdates = true;
                    }
                });
            } catch (error) {
                console.error("Failed to fetch activity logs for notifications:", error);
            }

            // 5. Check for Bug Reports (Admin Only) & Add if New
            if (user.role === 'admin') {
                try {
                    const reports = await adminService.getBugReports();
                    const openReports = reports.filter(r => r.status === 'open' && !r.isDeleted);

                    openReports.forEach(report => {
                        const notifId = `bug_${report.id}`;
                        const exists = currentNotifs.some(n => n.id === notifId);

                        if (!exists) {
                            currentNotifs.unshift({
                                id: notifId,
                                title: 'New Bug Report',
                                text: `${report.title} (${report.userEmail})`,
                                type: 'warning',
                                category: 'Support',
                                timestamp: report.timestamp,
                                read: false
                            });
                            hasUpdates = true;
                        }
                    });
                } catch (error) {
                    console.error("Failed to fetch bug reports for notifications:", error);
                }
            }

            // 3.5. Sync: Mark notifications as read if the message is no longer unread
            // This fixes the issue where badge persists after reading message in dashboard
            if (messages) {
                const unreadMessageIds = new Set(messages.map(m => m.id));
                currentNotifs.forEach(n => {
                    if (n.id.startsWith('message_') && !n.read) {
                        const msgId = n.id.replace('message_', '');
                        if (!unreadMessageIds.has(msgId)) {
                            n.read = true;
                            hasUpdates = true;
                        }
                    }
                });
            }

            // 4. Check for Low Credit Alerts (Dynamic)
            const limit = user?.usageStats?.apiCallsLimit || 0;
            const used = user?.usageStats?.apiCallsThisMonth || 0;
            if (limit > 0) {
                const percent = (used / limit) * 100;
                const today = new Date().toDateString();

                if (percent >= 100) {
                    const id = `limit_exceeded_${today}`;
                    if (!currentNotifs.some(n => n.id === id)) {
                        currentNotifs.unshift({ id, title: 'Credits Exhausted', text: 'You have reached your API limit.', type: 'urgent', category: 'System', timestamp: Date.now(), read: false });
                        hasUpdates = true;
                    }
                } else if (percent > 80) {
                    const id = `limit_warn_${today}`;
                    if (!currentNotifs.some(n => n.id === id)) {
                        currentNotifs.unshift({ id, title: 'Low Credits', text: `You have used ${Math.round(percent)}% of your credits.`, type: 'warning', category: 'System', timestamp: Date.now(), read: false });
                        hasUpdates = true;
                    }
                }
            }

            if (hasUpdates) {
                await UserDatabase.saveNotifications(user.uid, currentNotifs);
            }

            // Sort by date desc
            currentNotifs.sort((a, b) => b.timestamp - a.timestamp);
            setNotifications(currentNotifs);
        };

        if (user) fetchNotifications();
        const interval = setInterval(fetchNotifications, 30000); // Poll every 30s

        // Listen for manual update trigger from adminService (for instant notifications)
        const handleUpdate = () => fetchNotifications();
        window.addEventListener('sf_notifications_updated', handleUpdate);

        return () => {
            clearInterval(interval);
            window.removeEventListener('sf_notifications_updated', handleUpdate);
        };
    }, [user?.uid]); // Only re-run when user ID changes (login/logout), not on every user property change

    const unreadCount = notifications.filter(n => !n.read).length;

    const handleBellClick = () => {
        setIsNotifOpen(!isNotifOpen);
        setIsUserMenuOpen(false);
        setIsLangMenuOpen(false);
    };

    const handleMarkAllRead = async () => {
        if (user) {
            await UserDatabase.markAllNotificationsRead(user.uid);
            setNotifications(prev => prev.map(n => ({ ...n, read: true })));
        }
    };

    const navigateToMessageSource = () => {
        if (user?.role === 'admin') {
            // Admin: Navigate to Messages & Chat admin section
            window.dispatchEvent(new CustomEvent('sf_navigate_admin', { detail: 'MESSAGES' }));
            setView(AppView.ADMIN);
        } else {
            // User: Navigate to Support tab in Settings
            window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_support' }));
        }
    };

    const handleNotificationClick = async (notif: UserNotification) => {
        setSelectedNotification(notif);
        setIsNotifOpen(false);
        if (user && !notif.read) {
            await UserDatabase.markNotificationRead(user.uid, notif.id);

            // If it's a broadcast, mark it as read on the server too
            if (notif.id.startsWith('broadcast_')) {
                const broadcastId = notif.id.replace('broadcast_', '');
                // Fire and forget
                adminService.markBroadcastRead(broadcastId, user.uid).catch(console.error);
            }

            // If it's a message notification, handle navigation and mark message as read
            if (notif.id.startsWith('message_')) {
                const messageId = notif.id.replace('message_', '');

                navigateToMessageSource();

                // Mark message as read in Firestore (via server API)
                try {
                    await fetch('http://localhost:3001/api/support/mark-read', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ messageId })
                    });
                } catch (error) {
                    console.error('Failed to mark message as read:', error);
                }

                // Remove from notifications list (it's been read)
                setNotifications(prev => prev.filter(n => n.id !== notif.id));
                return; // Don't show the notification detail modal for messages
            }

            setNotifications(prev => prev.map(n => n.id === notif.id ? { ...n, read: true } : n));
        }
    };

    // Search Logic
    useEffect(() => {
        const timer = setTimeout(() => {
            if (searchQuery.trim()) {
                performSearch();
            } else {
                setSearchResults([]);
                setShowResults(false);
            }
        }, 300);
        return () => clearTimeout(timer);
    }, [searchQuery]);

    const performSearch = async () => {
        setIsSearching(true);
        setShowResults(true);
        const q = searchQuery.toLowerCase();
        const results: any[] = [];

        // 1. Search Assets (Library / History / Trash)
        const assets = await dbService.getAll();
        const matchedAssets = assets.filter(a => {
            const promptMatch = a.prompt.toLowerCase().includes(q);
            const typeMatch = a.type.toLowerCase().includes(q);
            const metaMatch = a.metadata ? JSON.stringify(a.metadata).toLowerCase().includes(q) : false;
            return promptMatch || typeMatch || metaMatch;
        }).slice(0, 8);

        matchedAssets.forEach(a => {
            results.push({
                id: a.id,
                type: 'asset',
                subtype: a.type,
                title: a.prompt,
                subtitle: a.isDeleted ? 'In Trash' : 'In Library',
                url: a.url,
                category: a.isDeleted ? 'Trash' : 'Library & History'
            });
        });

        setSearchResults(results);
        setIsSearching(false);
    };

    const handleLanguageChange = (langCode: string) => {
        setIsLangMenuOpen(false);
        setCurrentLang(langCode);

        const select = document.querySelector('.goog-te-combo') as HTMLSelectElement;
        if (select) {
            select.value = langCode;
            select.dispatchEvent(new Event('change', { bubbles: true }));
            const iframe = document.querySelector('.goog-te-menu-frame') as HTMLIFrameElement;
            if (!iframe) {
                select.click();
            }
        }
    };

    const getInitials = (name: string) => {
        if (!name) return 'U';
        return name.split(' ').map(n => n[0]).join('').substring(0, 2).toUpperCase();
    };

    const handleSettingsClick = (tab: string) => {
        if (onOpenSettings) {
            onOpenSettings(tab);
        } else {
            setView(AppView.SETTINGS);
        }
        setIsUserMenuOpen(false);
    };

    const handleResultClick = (item: any) => {
        setShowResults(false);
        setSearchQuery('');

        if (item.type === 'asset') {
            if (item.category === 'Trash') setView(AppView.TRASH);
            else setView(AppView.LIBRARY);
        } else if (item.type === 'user') {
            setView(AppView.ADMIN);
        } else if (item.type === 'bug') {
            setView(AppView.ADMIN);
        }
    };

    const formatTimeAgo = (timestamp: number) => {
        const diff = Date.now() - timestamp;
        const mins = Math.floor(diff / 60000);
        if (mins < 1) return 'Just now';
        if (mins < 60) return `${mins}m ago`;
        const hours = Math.floor(mins / 60);
        if (hours < 24) return `${hours}h ago`;
        return `${Math.floor(hours / 24)}d ago`;
    };

    // Admin Bug Notification
    const [adminBugCount, setAdminBugCount] = useState(0);

    useEffect(() => {
        const checkAdminNotifications = async () => {
            if (user?.role === 'admin') {
                const reports = await adminService.getBugReports();
                const open = reports.filter(r => r.status === 'open' && !r.isDeleted).length;
                setAdminBugCount(open);
            }
        };

        if (user?.role === 'admin') {
            checkAdminNotifications();
            const interval = setInterval(checkAdminNotifications, 60000); // Check every minute

            // Listen for immediate updates
            window.addEventListener('sf_bug_updated', checkAdminNotifications);

            return () => {
                clearInterval(interval);
                window.removeEventListener('sf_bug_updated', checkAdminNotifications);
            };
        }
    }, [user?.uid, user?.role]); // Only re-run when user ID or role changes

    return (
        <>
            <header className="h-20 bg-white/80 dark:bg-slate-950/80 backdrop-blur-md border-b border-slate-200 dark:border-slate-800 flex items-center justify-between px-4 lg:px-8 sticky top-0 z-30 transition-colors duration-300 gap-4">

                {/* Left: Mobile Toggle & Breadcrumbs */}
                <div className="flex items-center gap-4 shrink-0">
                    <button
                        onClick={toggleMobileSidebar}
                        className="lg:hidden p-2 text-slate-700 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg transition-colors"
                    >
                        <Menu size={20} />
                    </button>

                    <div className="flex items-center gap-2 text-sm text-slate-700 dark:text-slate-400 font-medium whitespace-nowrap">
                        <span className="hidden sm:inline hover:text-slate-900 dark:hover:text-slate-200 cursor-pointer font-['Space_Grotesk'] font-bold tracking-tight text-lg">{appName}</span>
                        <ChevronRight size={14} className="hidden sm:inline text-slate-300" />
                        <span className="text-slate-500 dark:text-slate-500">{title}</span>
                    </div>
                </div>

                {/* Center: Expanded Global Search - ALIGNED RIGHT */}
                <div className="flex-1 hidden md:flex justify-end px-6" ref={searchRef}>
                    <div className="relative group w-full max-w-[400px]">
                        <div className="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                            {isSearching ? <Loader2 className="animate-spin text-amber-500 h-5 w-5" /> : <Search className="text-slate-400 dark:text-slate-500 h-5 w-5 group-focus-within:text-amber-500 transition-colors" />}
                        </div>
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onFocus={() => { if (searchQuery) setShowResults(true); }}
                            placeholder="Search videos, images..."
                            className="block w-full pl-12 pr-4 py-3 bg-slate-100 dark:bg-slate-900 border border-transparent dark:border-slate-800 rounded-2xl leading-5 text-slate-900 dark:text-slate-100 placeholder-slate-500 focus:outline-none focus:bg-white dark:focus:bg-slate-950 focus:ring-2 focus:ring-amber-500/20 focus:border-amber-500 transition-all shadow-inner dark:shadow-none font-medium"
                        />
                        <div className="absolute inset-y-0 right-0 pr-4 flex items-center pointer-events-none">
                            <kbd className="inline-flex items-center border border-slate-300 dark:border-slate-700 rounded px-2 text-xs font-sans font-medium text-slate-400 dark:text-slate-500">⌘K</kbd>
                        </div>

                        {/* Search Results Dropdown */}
                        {showResults && (
                            <div className="absolute top-full left-0 w-full mt-2 bg-white dark:bg-slate-900 rounded-2xl shadow-2xl border border-slate-200 dark:border-slate-800 overflow-hidden z-50 animate-fade-in-up max-h-[60vh] overflow-y-auto custom-scrollbar">
                                {searchResults.length > 0 ? (
                                    <div className="py-2">
                                        {/* Group by Category */}
                                        {['Library & History', 'Trash'].map(category => {
                                            const items = searchResults.filter(r => r.category === category);
                                            if (items.length === 0) return null;

                                            return (
                                                <div key={category}>
                                                    <div className="px-4 py-2 bg-slate-50 dark:bg-slate-950/50 text-xs font-bold text-slate-500 uppercase tracking-wider sticky top-0 backdrop-blur-sm border-y border-slate-100 dark:border-slate-800/50 first:border-t-0">
                                                        {category}
                                                    </div>
                                                    {items.map((item, idx) => (
                                                        <button
                                                            key={`${item.type}-${item.id}-${idx}`}
                                                            onClick={() => handleResultClick(item)}
                                                            className="w-full text-left px-4 py-3 hover:bg-slate-50 dark:hover:bg-slate-800 flex items-center gap-4 transition-colors border-b border-slate-100 dark:border-slate-800/50 last:border-0 group"
                                                        >
                                                            <div className={`w-10 h-10 rounded-lg flex items-center justify-center shrink-0 ${item.type === 'asset' ? 'bg-slate-200 dark:bg-slate-800' :
                                                                item.type === 'user' ? 'bg-blue-100 dark:bg-blue-900/20 text-blue-600' :
                                                                    'bg-red-100 dark:bg-red-900/20 text-red-600'
                                                                }`}>
                                                                {item.type === 'asset' && (item.subtype === 'video' ? <Video size={18} /> : <ImageIcon size={18} />)}
                                                            </div>
                                                            <div className="flex-1 min-w-0">
                                                                <p className="text-sm font-bold text-slate-800 dark:text-slate-200 truncate group-hover:text-amber-600 dark:group-hover:text-amber-400 transition-colors">{item.title}</p>
                                                                <p className="text-xs text-slate-500 truncate">{item.subtitle}</p>
                                                            </div>
                                                            <ChevronRight size={14} className="text-slate-300 group-hover:text-amber-500 transition-colors" />
                                                        </button>
                                                    ))}
                                                </div>
                                            )
                                        })}
                                    </div>
                                ) : (
                                    <div className="p-8 text-center text-slate-500 dark:text-slate-400">
                                        {isSearching ? (
                                            <div className="flex flex-col items-center gap-2">
                                                <Loader2 className="animate-spin text-amber-500" size={24} />
                                                <span className="text-sm">Searching...</span>
                                            </div>
                                        ) : (
                                            <>
                                                <Search size={32} className="mx-auto mb-2 opacity-20" />
                                                <p className="text-sm">No results found for "{searchQuery}"</p>
                                            </>
                                        )}
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                </div>

                {/* Right: Actions */}
                <div className="flex items-center gap-3 lg:gap-5 shrink-0">

                    {/* Admin Bug Notification Icon */}
                    {user?.role === 'admin' && adminBugCount > 0 && (
                        <div className="relative animate-pulse">
                            <button
                                className="w-10 h-10 rounded-full bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 flex items-center justify-center transition-colors hover:bg-red-200 dark:hover:bg-red-900/50"
                                title={`${adminBugCount} Open Bug Reports`}
                                onClick={() => {
                                    setView(AppView.ADMIN);
                                    // Navigate to Bug Reports section
                                    window.dispatchEvent(new CustomEvent('sf_navigate_admin', { detail: 'BUG_REPORTS' }));
                                }}
                            >
                                <Bug size={20} />
                                <span className="absolute -top-1 -right-1 w-5 h-5 bg-red-600 text-white text-[10px] font-bold flex items-center justify-center rounded-full border-2 border-white dark:border-slate-950 shadow-sm">
                                    {adminBugCount}
                                </span>
                            </button>
                        </div>
                    )}

                    {/* Combined Stats Box */}
                    <div className="hidden lg:flex items-center bg-slate-100 dark:bg-slate-800/50 rounded-lg border border-slate-200 dark:border-slate-700 py-1">
                        <div className="px-3 flex flex-col items-end border-r border-slate-200 dark:border-slate-700/50">
                            <span className="text-[9px] font-bold text-slate-500 uppercase tracking-wider flex items-center gap-1">
                                <Clock size={10} /> RESETS IN
                            </span>
                            <span className="text-xs font-bold text-blue-600 dark:text-blue-400 font-mono">
                                {timeToReset}
                            </span>
                        </div>
                        <div className="px-3 flex flex-col items-end">
                            <span className="text-[9px] font-bold text-slate-500 uppercase tracking-wider">API CREDITS</span>
                            <span className="text-xs font-bold text-emerald-600 dark:text-emerald-400 font-mono">
                                {apiRemaining} / {apiLimit}
                            </span>
                        </div>
                    </div>

                    {/* Mobile Search Toggle */}
                    <button className="md:hidden p-2 text-slate-700 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white rounded-lg">
                        <Search size={20} />
                    </button>

                    {/* Language Selector */}
                    <div className="relative">
                        <button
                            onClick={() => { setIsLangMenuOpen(!isLangMenuOpen); setIsUserMenuOpen(false); setIsNotifOpen(false); }}
                            className="p-2.5 text-slate-700 dark:text-slate-400 hover:text-amber-600 dark:hover:text-amber-400 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-xl transition-all relative"
                            title="Change Language"
                        >
                            <Globe size={20} />
                        </button>

                        {isLangMenuOpen && (
                            <>
                                <div className="fixed inset-0 z-10" onClick={() => setIsLangMenuOpen(false)}></div>
                                <div className="absolute right-0 top-full mt-4 w-48 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl shadow-2xl z-20 overflow-hidden animate-fade-in-up">
                                    <div className="max-h-64 overflow-y-auto custom-scrollbar">
                                        {LANGUAGES.map((lang) => (
                                            <button
                                                key={lang.code}
                                                onClick={() => handleLanguageChange(lang.code)}
                                                className="w-full text-left px-4 py-3 text-sm text-slate-700 dark:text-slate-300 hover:bg-slate-100 dark:hover:bg-slate-800 flex items-center gap-3 transition-colors justify-between group font-medium"
                                            >
                                                <div className="flex items-center gap-2">
                                                    <span className="text-lg">{lang.flag}</span>
                                                    <span>{lang.label}</span>
                                                </div>
                                                {currentLang === lang.code && <Check size={14} className="text-amber-500" />}
                                            </button>
                                        ))}
                                    </div>
                                </div>
                            </>
                        )}
                    </div>

                    {/* Notifications */}
                    <div className="relative">
                        <button
                            onClick={handleBellClick}
                            className="p-2.5 text-slate-700 dark:text-slate-400 hover:text-amber-600 dark:hover:text-amber-400 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-xl transition-all relative"
                        >
                            <Bell size={20} />
                            {notifications.length > 0 && (
                                unreadCount > 0 ? (
                                    <>
                                        <span className="absolute top-2.5 right-2.5 w-2.5 h-2.5 bg-red-500 rounded-full border-2 border-white dark:border-slate-950 animate-pulse"></span>
                                        {/* Show number badge for total unread */}
                                        <span className="absolute -top-1 -right-1 min-w-[18px] h-[18px] bg-red-600 text-white text-[10px] font-bold flex items-center justify-center rounded-full border-2 border-white dark:border-slate-950 shadow-sm">
                                            {unreadCount}
                                        </span>
                                    </>
                                ) : (
                                    <span className="absolute top-2.5 right-2.5 w-2.5 h-2.5 bg-green-500 rounded-full border-2 border-white dark:border-slate-950"></span>
                                )
                            )}
                        </button>

                        {isNotifOpen && (
                            <>
                                <div className="fixed inset-0 z-10" onClick={() => setIsNotifOpen(false)}></div>
                                <div className="absolute right-0 top-full mt-4 w-80 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl shadow-2xl z-20 overflow-hidden animate-fade-in-up">
                                    <div className="p-4 border-b border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-950/50 flex justify-between items-center">
                                        <h3 className="font-bold text-slate-900 dark:text-slate-100 text-sm">Notifications</h3>
                                        <button onClick={handleMarkAllRead} className="text-xs text-amber-600 dark:text-amber-500 hover:text-amber-700 dark:hover:text-amber-400 font-bold flex items-center gap-1">
                                            <CheckCheck size={12} /> Mark all read
                                        </button>
                                    </div>
                                    <div className="max-h-[300px] overflow-y-auto custom-scrollbar">
                                        {notifications.length === 0 && (
                                            <div className="p-8 text-center text-slate-500 dark:text-slate-400 text-sm">
                                                All caught up! No new notifications.
                                            </div>
                                        )}
                                        {notifications.slice(0, 5).map((n) => (
                                            <div
                                                key={n.id}
                                                onClick={() => handleNotificationClick(n)}
                                                className={`p-4 border-b border-slate-100 dark:border-slate-800/50 hover:bg-slate-50 dark:hover:bg-slate-800/50 transition-colors cursor-pointer group ${!n.read ? 'bg-blue-50/50 dark:bg-blue-900/10' : ''}`}
                                            >
                                                <div className="flex gap-3">
                                                    <div className={`mt-1 w-8 h-8 rounded-lg flex items-center justify-center shrink-0 ${n.type === 'success' ? 'bg-green-100 text-green-600 dark:bg-green-900/20 dark:text-green-400' :
                                                        n.type === 'warning' ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/20 dark:text-amber-400' :
                                                            n.type === 'urgent' ? 'bg-red-100 text-red-600 dark:bg-red-900/20 dark:text-red-400' :
                                                                'bg-blue-100 text-blue-600 dark:bg-blue-900/20 dark:text-blue-400'
                                                        }`}>
                                                        {n.type === 'urgent' ? <AlertTriangle size={16} /> : n.category === 'Feature' ? <Zap size={16} /> : n.category === 'Announcement' ? <Megaphone size={16} /> : n.category === 'Support' ? <MessageSquare size={16} /> : n.type === 'success' ? <CheckCircle2 size={16} /> : <Info size={16} />}
                                                    </div>
                                                    <div className="flex-1 min-w-0">
                                                        <div className="flex justify-between items-start mb-0.5">
                                                            <h4 className={`font-bold text-sm truncate mr-2 ${!n.read ? 'text-slate-900 dark:text-white' : 'text-slate-700 dark:text-slate-300'}`}>{n.title}</h4>
                                                            {n.category && (
                                                                <span className={`text-[9px] font-bold uppercase tracking-wider px-1.5 py-0.5 rounded border ${n.category === 'Feature' ? 'bg-blue-100 dark:bg-blue-900/30 text-blue-600 border-blue-200 dark:border-blue-800' :
                                                                    n.type === 'urgent' ? 'bg-red-100 dark:bg-red-900/30 text-red-600 border-red-200 dark:border-red-800' :
                                                                        'bg-slate-100 dark:bg-slate-800 text-slate-500 border-slate-200 dark:border-slate-700'
                                                                    }`}>
                                                                    {n.category}
                                                                </span>
                                                            )}
                                                        </div>
                                                        <p className="text-xs text-slate-600 dark:text-slate-400 mt-0.5 font-medium line-clamp-2">{n.text}</p>
                                                        <p className="text-[10px] text-slate-400 dark:text-slate-500 mt-2 flex items-center gap-1"><Clock size={10} /> {formatTimeAgo(n.timestamp)}</p>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                    <div className="p-2 text-center bg-slate-50 dark:bg-slate-950/50 border-t border-slate-200 dark:border-slate-800">
                                        <button onClick={() => { setIsHistoryOpen(true); setIsNotifOpen(false); }} className="text-xs text-slate-700 hover:text-slate-900 dark:text-slate-500 dark:hover:text-slate-300 py-2 w-full font-bold">View All History</button>
                                    </div>
                                </div>
                            </>
                        )}
                    </div>

                    <div className="h-8 w-px bg-slate-200 dark:bg-slate-800 mx-1 hidden lg:block"></div>

                    {/* User Profile Dropdown */}
                    <div className="relative">
                        <button
                            onClick={() => { setIsUserMenuOpen(!isUserMenuOpen); setIsNotifOpen(false); setIsLangMenuOpen(false); }}
                            className="flex items-center gap-3 hover:bg-slate-100 dark:hover:bg-slate-800/50 rounded-full pr-4 pl-1 py-1 transition-all group"
                        >
                            {user?.photoURL ? (
                                <img
                                    src={user.photoURL}
                                    alt={user.displayName}
                                    className="w-9 h-9 rounded-full object-cover ring-2 ring-white dark:ring-slate-900 shadow-sm group-hover:ring-amber-500/30 transition-all"
                                />
                            ) : (
                                <div className="w-9 h-9 rounded-full bg-gradient-to-tr from-blue-500 to-purple-500 flex items-center justify-center text-xs font-bold text-white ring-2 ring-white dark:ring-slate-900 shadow-sm group-hover:ring-amber-500/30 transition-all">
                                    {user ? getInitials(user.displayName) : 'U'}
                                </div>
                            )}
                            <div className="hidden md:block text-left">
                                <p className="text-xs font-bold text-slate-900 dark:text-slate-200 group-hover:text-amber-600 dark:group-hover:text-amber-400 transition-colors">{user?.displayName}</p>
                                <p className="text-[10px] text-slate-600 dark:text-slate-500 uppercase tracking-wide font-bold">{user?.subscription} PLAN</p>
                            </div>
                        </button>

                        {isUserMenuOpen && (
                            <>
                                <div className="fixed inset-0 z-10" onClick={() => setIsUserMenuOpen(false)}></div>
                                <div className="absolute right-0 top-full mt-4 w-64 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl shadow-2xl z-20 overflow-hidden animate-fade-in-up">
                                    <div className="p-5 border-b border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-950/50">
                                        <p className="text-sm font-bold text-slate-900 dark:text-white">{user?.displayName}</p>
                                        <p className="text-xs text-slate-500 truncate">{user?.email}</p>
                                        <div className="mt-2 flex gap-2">
                                            <span className="px-2 py-0.5 rounded bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 text-[10px] font-bold uppercase text-slate-600 dark:text-slate-400">{user?.role}</span>
                                        </div>
                                    </div>
                                    <div className="p-2 space-y-0.5">
                                        <MenuLink icon={UserIcon} label="Profile" onClick={() => handleSettingsClick('profile')} />
                                        <MenuLink icon={Settings} label="Settings" onClick={() => handleSettingsClick('app')} />
                                        <MenuLink icon={CreditCard} label="Billing" onClick={() => handleSettingsClick('billing')} />
                                        <div className="my-1 border-t border-slate-200 dark:border-slate-800"></div>
                                        <MenuLink icon={Book} label="Documentation" onClick={() => { if (onOpenDocumentation) onOpenDocumentation(); setIsUserMenuOpen(false); }} />
                                        <MenuLink icon={Bug} label="Report Bug" onClick={() => handleSettingsClick('bug_report')} />
                                        <div className="my-1 border-t border-slate-200 dark:border-slate-800"></div>
                                        <button
                                            onClick={onLogout}
                                            className="w-full flex items-center gap-2 px-3 py-2.5 rounded-lg text-sm font-bold text-red-600 dark:text-red-400 hover:bg-red-50 dark:hover:bg-red-400/10 transition-colors"
                                        >
                                            <LogOut size={16} /> Sign Out
                                        </button>
                                    </div>
                                </div>
                            </>
                        )}
                    </div>

                </div>
            </header>

            {/* Notification Detail Modal */}
            {selectedNotification && (
                <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4 animate-fade-in" onClick={() => setSelectedNotification(null)}>
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-md p-6 shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}>
                        <div className="flex justify-between items-start mb-4">
                            <div className="flex items-center gap-3">
                                <div className={`p-2 rounded-full ${selectedNotification.type === 'success' ? 'bg-green-100 text-green-600' : selectedNotification.type === 'warning' ? 'bg-amber-100 text-amber-600' : selectedNotification.type === 'urgent' ? 'bg-red-100 text-red-600' : 'bg-blue-100 text-blue-600'
                                    }`}>
                                    {selectedNotification.type === 'urgent' ? <AlertTriangle size={20} /> : selectedNotification.category === 'Support' ? <MessageSquare size={20} /> : <Bell size={20} />}
                                </div>
                                <div>
                                    <h3 className="font-bold text-slate-900 dark:text-white text-lg">{selectedNotification.title}</h3>
                                    <p className="text-xs text-slate-500">{new Date(selectedNotification.timestamp).toLocaleString()}</p>
                                </div>
                            </div>
                            <button onClick={() => setSelectedNotification(null)} className="p-1 text-slate-400 hover:text-slate-900 dark:hover:text-white transition"><X size={20} /></button>
                        </div>
                        <div className="bg-slate-50 dark:bg-slate-950/50 rounded-xl p-4 text-sm text-slate-700 dark:text-slate-300 leading-relaxed mb-6">
                            {selectedNotification.text}
                        </div>
                        <div className="flex justify-end gap-2">
                            {selectedNotification.category === 'Support' && (
                                <button onClick={() => {
                                    navigateToMessageSource();
                                    setSelectedNotification(null);
                                }} className="px-4 py-2 bg-amber-500 text-slate-950 font-bold rounded-lg flex items-center gap-2">
                                    <MessageSquare size={18} /> Reply
                                </button>
                            )}
                            <button onClick={() => setSelectedNotification(null)} className="px-4 py-2 bg-slate-900 dark:bg-white text-white dark:text-slate-950 font-bold rounded-lg">Close</button>
                        </div>
                    </div>
                </div>
            )}

            {/* Notification History Modal */}
            {isHistoryOpen && (
                <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4 animate-fade-in" onClick={() => setIsHistoryOpen(false)}>
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-2xl h-[80vh] flex flex-col shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}>
                        <div className="p-6 border-b border-slate-200 dark:border-slate-800 flex justify-between items-center">
                            <h3 className="text-xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><Clock size={20} /> Notification History</h3>
                            <div className="flex items-center gap-2">
                                {notifications.length > 0 && (
                                    <button
                                        onClick={async () => {
                                            if (user) {
                                                await UserDatabase.clearNotifications(user.uid);
                                                setNotifications([]);
                                            }
                                        }}
                                        className="p-2 bg-red-50 text-red-600 hover:bg-red-100 dark:bg-red-900/20 dark:text-red-400 dark:hover:bg-red-900/30 rounded-full transition mr-2"
                                        title="Delete All History"
                                    >
                                        <Trash2 size={18} />
                                    </button>
                                )}
                                <button onClick={() => setIsHistoryOpen(false)} className="p-2 bg-slate-100 dark:bg-slate-800 rounded-full text-slate-500 hover:text-slate-900 dark:hover:text-white transition"><X size={18} /></button>
                            </div>
                        </div>
                        <div className="flex-1 overflow-y-auto custom-scrollbar p-4 space-y-2">
                            {notifications.length === 0 && (
                                <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-400">
                                    <Bell size={48} className="mb-4 opacity-20" />
                                    <p>No notification history found.</p>
                                </div>
                            )}
                            {notifications.map((n) => (
                                <div
                                    key={n.id}
                                    onClick={async () => {
                                        setSelectedNotification(n);
                                        await UserDatabase.markNotificationRead(user?.uid || '', n.id);
                                        setNotifications(prev => prev.map(prevN => prevN.id === n.id ? { ...prevN, read: true } : prevN));
                                    }}
                                    className={`flex gap-4 p-4 rounded-xl border border-slate-100 dark:border-slate-800 transition-all cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-800/50 ${!n.read ? 'bg-blue-50/30 dark:bg-blue-900/10 border-blue-100 dark:border-blue-900/20' : 'bg-white dark:bg-slate-900'}`}
                                >
                                    <div className={`mt-1 p-2 rounded-lg h-fit ${n.type === 'success' ? 'bg-green-100 text-green-600 dark:bg-green-900/20 dark:text-green-400' :
                                        n.type === 'warning' ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/20 dark:text-amber-400' :
                                            n.type === 'urgent' ? 'bg-red-100 text-red-600 dark:bg-red-900/20 dark:text-red-400' :
                                                'bg-blue-100 text-blue-600 dark:bg-blue-900/20 dark:text-blue-400'
                                        }`}>
                                        {n.type === 'urgent' ? <AlertTriangle size={16} /> : n.category === 'Feature' ? <Zap size={16} /> : n.category === 'Announcement' ? <Megaphone size={16} /> : n.category === 'Support' ? <MessageSquare size={16} /> : n.type === 'success' ? <CheckCircle2 size={16} /> : <Info size={16} />}
                                    </div>
                                    <div className="flex-1">
                                        <div className="flex justify-between items-start mb-1">
                                            <h4 className={`font-bold text-sm ${!n.read ? 'text-slate-900 dark:text-white' : 'text-slate-700 dark:text-slate-300'}`}>{n.title}</h4>
                                            <span className="text-xs text-slate-400 whitespace-nowrap flex items-center gap-1">
                                                <Calendar size={10} /> {new Date(n.timestamp).toLocaleDateString()} {new Date(n.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                                            </span>
                                        </div>
                                        <p className="text-sm text-slate-600 dark:text-slate-400">{n.text}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            )}
        </>
    );
};

const MenuLink = ({ icon: Icon, label, onClick }: any) => (
    <button
        onClick={onClick}
        className="w-full flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-bold text-slate-700 dark:text-slate-400 hover:text-slate-900 dark:hover:text-slate-100 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors group"
    >
        <Icon size={16} className="group-hover:text-amber-500 transition-colors" /> {label}
    </button>
);

export default TopBar;