import React, { useState, useRef, useEffect } from 'react';
import {
    Upload, X, Check, AlertTriangle, Download, RefreshCw, Eye, Trash2, Camera,
    History as HistoryIcon, Settings, Layers, Play, CheckCircle2, AlertCircle,
    Cpu, Key, AppWindow, Loader2, FileDown
} from 'lucide-react';
import { dbService, Asset } from '../services/dbService';
import { authService } from '../services/authService';
import { AutoDownloadIndicator } from './AutoDownloadIndicator';
import { playSound } from '../utils/soundEffects';
// import { auth } from '../firebase'; // Removed

import { EnhancedLoadingOverlay } from './EnhancedLoadingOverlay';
import JSZip from 'jszip';
import { getPreferredModels, performTwoStepReview, RateLimitExhaustedError } from '../services/geminiService';
import { TwoStepReviewResult } from '../types';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';

// Batch Item Type
interface ReviewBatchItem {
    id: string;
    file: File;
    preview: string;
    status: 'pending' | 'processing' | 'complete' | 'error';
    progress: number;
    result?: any;
    stats?: {
        model: string;
        apiKey: string;
        startTime: number;
        endTime?: number;
        duration?: string;
    };
}

const StockPhotoReviewer = () => {
    // Tab State
    const [activeTab, setActiveTab] = useState<'single' | 'bulk' | 'history'>('single');
    const [isDownloadingSeparate, setIsDownloadingSeparate] = useState(false);
    const [isDownloadingZip, setIsDownloadingZip] = useState(false);
    const [isDownloadingRejected, setIsDownloadingRejected] = useState(false);
    const [isDownloadingAccepted, setIsDownloadingAccepted] = useState(false);
    const [hasApiKey, setHasApiKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [rateLimitError, setRateLimitError] = useState(false); // Popup state
    const [generationInfo, setGenerationInfo] = useState<{ apiKey: string, model: string } | null>(null);

    // Single Mode State
    const [image, setImage] = useState<{ file: File; preview: string; base64: string; type: string } | null>(null);
    const [analysis, setAnalysis] = useState<any>(null);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [enableAnatomyCheck, setEnableAnatomyCheck] = useState(true);
    const [enableQualityCheck, setEnableQualityCheck] = useState(true);
    const [enableHandFix, setEnableHandFix] = useState(false);
    const [selectedFixModel, setSelectedFixModel] = useState<string>('imagen-3.0-generate-001');
    const [availableCreativeModels, setAvailableCreativeModels] = useState<string[]>([]);

    useEffect(() => {
        const loadModels = () => {
            const creative = getPreferredModels('creative');
            setAvailableCreativeModels(creative.length > 0 ? creative : ['imagen-3.0-generate-001']);
            if (creative.length > 0) setSelectedFixModel(creative[0]);
        };
        loadModels();
    }, []); // Default off
    const [enableStrictFaceCheck, setEnableStrictFaceCheck] = useState(false); // Default off (opt-in)
    const [enableAnimeStyle, setEnableAnimeStyle] = useState(false); // Default off (opt-in)
    const [hasError, setHasError] = useState(false);

    // Bulk Mode State
    const [batchItems, setBatchItems] = useState<ReviewBatchItem[]>([]);
    const [isBatchProcessing, setIsBatchProcessing] = useState(false);

    // History State
    const [history, setHistory] = useState<Asset[]>([]);
    const [selectedHistoryItem, setSelectedHistoryItem] = useState<Asset | null>(null);
    const [currentBatchId, setCurrentBatchId] = useState<string | null>(null);

    // Active Models Display
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // Clear All Confirmation State
    const [clearAllConfirming, setClearAllConfirming] = useState(false);
    const clearAllTimerRef = useRef<NodeJS.Timeout | null>(null);

    const fileInputRef = useRef<HTMLInputElement>(null);
    const bulkInputRef = useRef<HTMLInputElement>(null);

    // Initial load
    useEffect(() => {
        checkApiKey();
        const handleFocus = () => {
            checkApiKey();
            updateActiveModels();
        };
        window.addEventListener('focus', handleFocus);
        updateActiveModels();
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    // Load history when switching to history tab
    useEffect(() => {
        if (activeTab === 'history') {
            loadHistory();
        }
    }, [activeTab]);

    const checkApiKey = async () => {
        const keys = await getUserApiKeys();
        setHasApiKey(keys.length > 0);
        setIsCheckingKey(false);
    };

    const getUserApiKeys = async (): Promise<string[]> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return [];
        const keys = await authService.getUserApiKeys(currentUser.uid);
        return keys
            .filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false)
            .map((k: any) => k.fullKey);
    };

    const updateActiveModels = () => {
        // Get user's enabled thinking models
        const preferredModels = getPreferredModels('thinking');
        setActiveModels(preferredModels);
    };

    const loadHistory = async () => {
        try {
            const reviews = await dbService.getByType('review');

            if (reviews.length === 0) {
                setHistory([]);
                return;
            }

            // Sort by createdAt desc
            reviews.sort((a, b) => b.createdAt - a.createdAt);

            // Determine target batch ID
            let targetBatchId = currentBatchId;

            // If no current batch set, use the latest one from DB
            if (!targetBatchId) {
                targetBatchId = reviews[0].batchId || null;
            }

            if (targetBatchId) {
                const batchItems = reviews.filter(r => r.batchId === targetBatchId);
                setHistory(batchItems);
            } else {
                // Fallback for legacy items: just show the very last item
                setHistory([reviews[0]]);
            }
        } catch (err) {
            console.error("Failed to load history", err);
        }
    };

    // ===== SINGLE MODE HANDLERS =====

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            processFile(file);
        }
    };

    const processFile = (file: File) => {
        setAnalysis(null);
        setError(null);

        const reader = new FileReader();
        reader.onloadend = () => {
            if (typeof reader.result === 'string') {
                setImage({
                    file: file,
                    preview: reader.result,
                    base64: reader.result.split(',')[1],
                    type: file.type
                });
            }
        };
        reader.readAsDataURL(file);
    };

    const analyzeImage = async () => {
        if (!image) return;

        // HARD RESET all state before analysis to prevent corruption
        setAnalysis(null);
        setError(null);
        setHasError(false);
        setGenerationInfo(null);
        setCurrentBatchId(null);

        setLoading(true);

        // Generate new batch ID for this single analysis
        const newBatchId = Date.now().toString();
        setCurrentBatchId(newBatchId);

        try {
            const uid = authService.getCurrentUser()?.uid;

            if (!uid) throw new Error("You must be logged in to use this feature.");

            const apiKeys = await getUserApiKeys();
            if (apiKeys.length === 0) {
                throw new Error("API Key not found. Please configure it in Settings.");
            }

            // Get user's preferred thinking models
            const preferredModels = getPreferredModels('thinking');
            const selectedModel = preferredModels[0] || 'gemini-2.0-flash';

            setGenerationInfo({
                apiKey: apiKeys[0], // Show primary
                model: selectedModel
            });

            // Perform two-step review
            let result: TwoStepReviewResult = await performTwoStepReview(
                apiKeys,
                image.base64,
                image.type,
                selectedModel,
                undefined, // customPrompts
                { enableAnatomyCheck, enableQualityCheck, enableHandFix: enableHandFix, enableStrictFaceCheck, enableAnimeStyle } // options
            );

            // Step 2: Auto-Fix if Hands Detected
            if (enableHandFix && result.handDetection?.hasHands) {
                // Show "Fixing..." status via loading state or custom toast?
                // For now, we continue the loading state.
                console.log('👐 Hands Detected! Initiating Auto-Fix...');

                // Get creative model for fixing
                const creativeModel = selectedFixModel;

                const fixedImageBase64 = await import('../services/geminiService').then(m =>
                    m.fixImageHands(apiKeys, image.base64, image.type, creativeModel)
                );

                if (fixedImageBase64) {
                    console.log('✨ Image Fixed Successfully!');
                    // Update result to include fixed image
                    result = {
                        ...result,
                        handDetection: {
                            ...result.handDetection,
                            fixedImageBase64: fixedImageBase64
                        },
                        reason: "Hands detected and auto-fixed."
                    };
                } else {
                    console.warn('⚠️ Auto-Fix failed to generate image.');
                    result = {
                        ...result,
                        reason: "Hands detected but auto-fix failed."
                    };
                }
            }

            // Validate result before setting state
            if (!result || typeof result !== 'object') {
                throw new Error('Invalid analysis result received from API');
            }

            console.log('Setting analysis result:', result);
            setAnalysis(result);
            console.log('Analysis state updated successfully');

            // Auto-save to DB
            await saveResult(result, newBatchId);

            // Deduct Credits based on enabled checks (1 credit per enabled check)
            const creditsUsed = (enableAnatomyCheck ? 1 : 0) + (enableQualityCheck ? 1 : 0) + (enableHandFix ? 1 : 0);
            if (creditsUsed > 0) {
                await authService.deductCredits(uid, creditsUsed, apiKeys[0], selectedModel);
            }

            // Always track reviews, regardless of credits
            authService.incrementUserStat('imagesReviewed', 1);

            // Build specific activity description based on enabled checks
            const checks = [];
            if (enableAnatomyCheck) checks.push('Anatomy');
            if (enableQualityCheck) checks.push('Quality');
            if (enableHandFix) checks.push('Hand Fix');
            const checksText = checks.length > 0 ? ` (${checks.join(' + ')})` : '';

            await authService.logActivity(uid, 'generate', `Review AI Analysis${checksText}`, creditsUsed);

            // Auto-download if enabled and image is ACCEPTED
            try {
                const currentUser = authService.getCurrentUser();
                if (currentUser && result.verdict === 'ACCEPTED' && image?.file) {
                    // Fetch preferences using service
                    const prefs = await authService.getUserPreferences(currentUser.uid);
                    if (prefs?.uiConfig?.autoDownload) {
                        console.log('✅ Auto-downloading ACCEPTED image...');
                        await handleDownload(image.preview, image.file.name);
                    }
                }
            } catch (autoDownloadError) {
                console.error('Auto-download failed:', autoDownloadError);
                // Don't throw - auto-download is optional
            }

            // Play success sound
            await playSound('success');

        } catch (err: any) {
            console.error('=== REVIEWER AI ERROR ===');
            console.error('Error details:', err);
            console.error('Error message:', err.message);
            console.error('Error stack:', err.stack);

            // Safely set error state
            try {
                setError(err.message || "Failed to analyze image. Please try again.");
                if (err.name === 'RateLimitExhaustedError' || err.message?.includes('RateLimitExhaustedError')) {
                    setRateLimitError(true);
                }
                setHasError(true);
            } catch (stateError) {
                console.error('Failed to set error state:', stateError);
                // Force page reload if state is corrupted
                window.location.reload();
            }

            // Play error sound
            await playSound('error');
        } finally {
            try {
                setLoading(false);
            } catch (finallyError) {
                console.error('Failed to set loading state:', finallyError);
            }
        }
    };

    const uploadImage = async (file: File): Promise<string> => {
        const formData = new FormData();
        formData.append('file', file);

        try {
            const response = await fetch('http://localhost:3001/api/upload', {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error('Upload failed');
            }

            const data = await response.json();
            return data.url;
        } catch (error) {
            console.error("Upload error:", error);
            // Fallback to resizing if upload fails (e.g. server offline)
            return resizeImage(await new Promise(resolve => {
                const reader = new FileReader();
                reader.onload = (e) => resolve(e.target?.result as string);
                reader.readAsDataURL(file);
            }));
        }
    };

    const resizeImage = (base64Str: string, maxWidth = 300): Promise<string> => {
        return new Promise((resolve) => {
            const img = new Image();
            img.src = base64Str;
            img.onload = () => {
                const canvas = document.createElement('canvas');
                const scaleSize = maxWidth / img.width;
                canvas.width = maxWidth;
                canvas.height = img.height * scaleSize;
                const ctx = canvas.getContext('2d');
                ctx?.drawImage(img, 0, 0, canvas.width, canvas.height);
                resolve(canvas.toDataURL('image/jpeg', 0.7));
            };
        });
    };

    const saveResult = async (result: any, batchId?: string) => {
        if (!image) return;

        try {
            // Upload full size image
            const imageUrl = await uploadImage(image.file);

            await dbService.add({
                id: `review-${Date.now()}`,
                type: 'review',
                url: imageUrl,
                prompt: "Stock Photo Review",
                title: image.file.name,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: result,
                batchId: batchId
            });
            loadHistory();
        } catch (e) {
            console.error("Failed to save result", e);
        }
    };

    const handleReset = () => {
        setImage(null);
        setAnalysis(null);
        setError(null);
        if (fileInputRef.current) fileInputRef.current.value = '';
    };

    // ===== BULK MODE HANDLERS =====

    const handleBatchUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files) {
            const newItems: ReviewBatchItem[] = Array.from(e.target.files).slice(0, 50).map((file: File) => ({
                id: Math.random().toString(36).substring(7),
                file,
                preview: URL.createObjectURL(file),
                status: 'pending',
                progress: 0
            }));
            setBatchItems(prev => [...prev, ...newItems]);
        }
    };

    const processBatch = async () => {
        const apiKeys = await getUserApiKeys();
        if (apiKeys.length === 0) return;

        // Get user's preferred thinking models
        const preferredModels = getPreferredModels('thinking');
        const selectedModel = preferredModels[0] || 'gemini-2.0-flash-thinking-exp';

        setIsBatchProcessing(true);
        const uid = authService.getCurrentUser()?.uid;


        // Generate new batch ID
        const newBatchId = Date.now().toString();
        setCurrentBatchId(newBatchId);

        let processedCount = 0;

        // Sequential processing
        for (let i = 0; i < batchItems.length; i++) {
            const item = batchItems[i];
            if (item.status === 'complete') continue;

            // Apply Data Delay / Global Cooldown
            const currentUser = authService.getCurrentUser();
            const prefs = (currentUser as any)?.preferences?.generationConfig;
            const globalCooldownSec = Number(prefs?.globalCooldown || 0);
            const dataDelaySec = Number(prefs?.metadataDelay || 0);

            // Additive Delay Logic: Global Cooldown + Data Delay
            const effectiveDelaySec = globalCooldownSec + dataDelaySec;

            if (effectiveDelaySec > 0) {
                const cooldownMs = effectiveDelaySec * 1000;
                console.log(`[ReviewerAI] Applying Total Delay: ${cooldownMs}ms (${globalCooldownSec}s Global + ${dataDelaySec}s Data)`);
                await new Promise(r => setTimeout(r, cooldownMs));
            }

            setBatchItems(prev => prev.map(p => p.id === item.id ? {
                ...p,
                status: 'processing',
                progress: 10,
                stats: {
                    model: selectedModel,
                    apiKey: apiKeys[0].substring(0, 8) + '...',
                    startTime: Date.now()
                }
            } : p));

            try {
                const base64 = await fetch(item.preview)
                    .then(r => r.blob())
                    .then(blob => new Promise<string>((resolve) => {
                        const reader = new FileReader();
                        reader.onload = (e) => resolve((e.target?.result as string).split(',')[1]);
                        reader.readAsDataURL(blob);
                    }));

                // Perform two-step review
                let result: TwoStepReviewResult = await performTwoStepReview(
                    apiKeys,
                    base64,
                    item.file.type,
                    selectedModel,
                    undefined, // customPrompts
                    { enableAnatomyCheck, enableQualityCheck, enableHandFix: enableHandFix, enableStrictFaceCheck, enableAnimeStyle } // options
                );

                // Step 2: Auto-Fix if Hands Detected (Batch)
                if (enableHandFix && result.handDetection?.hasHands) {
                    // Get creative model for fixing - could optimize to fetch once
                    const creativeModel = selectedFixModel;

                    const fixedImageBase64 = await import('../services/geminiService').then(m =>
                        m.fixImageHands(apiKeys, base64, item.file.type, creativeModel)
                    );

                    if (fixedImageBase64) {
                        result = {
                            ...result,
                            handDetection: {
                                ...result.handDetection,
                                fixedImageBase64: fixedImageBase64
                            },
                            reason: "Hands detected and auto-fixed."
                        };
                    } else {
                        result = {
                            ...result,
                            reason: "Hands detected but auto-fix failed."
                        };
                    }
                }

                const endTime = Date.now();
                const startTime = item.stats?.startTime || Date.now(); // Should exist from previous update
                const duration = ((endTime - startTime) / 1000).toFixed(1) + 's';

                setBatchItems(prev => prev.map(p => p.id === item.id ? {
                    ...p,
                    status: 'complete',
                    progress: 100,
                    result,
                    stats: {
                        ...p.stats!,
                        endTime,
                        duration
                    }
                } : p));

                // Save to history
                const imageUrl = await uploadImage(item.file);
                await dbService.add({
                    id: `review-${Date.now()}-${i}`,
                    type: 'review',
                    url: imageUrl,
                    prompt: "Stock Photo Review (Two-Step)",
                    title: item.file.name, // Save filename
                    createdAt: Date.now(),
                    isFavorite: false,
                    isDeleted: false,
                    metadata: result,
                    batchId: newBatchId
                });

                processedCount++;

            } catch (e: any) {
                console.error("Batch item failed:", e);
                if (e.name === 'RateLimitExhaustedError' || e.message?.includes('RateLimitExhaustedError')) {
                    setRateLimitError(true);
                    setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, status: 'error', progress: 0 } : p));
                    setIsBatchProcessing(false);
                    return; // Stop batch
                }
                setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, status: 'error', progress: 0 } : p));
            }
        }

        if (uid && processedCount > 0) {
            // Deduct credits based on enabled checks
            const creditsPerImage = (enableAnatomyCheck ? 1 : 0) + (enableQualityCheck ? 1 : 0) + (enableHandFix ? 1 : 0);
            const totalCredits = processedCount * creditsPerImage;
            if (totalCredits > 0) {
                await authService.deductCredits(uid, totalCredits, apiKeys[0], selectedModel);
                authService.incrementUserStat('imagesReviewed', processedCount); // Track bulk reviews
            }
            authService.logActivity(uid, 'generate', `Batch processed ${processedCount} images in Reviewer AI (Two-Step)`);

            // Play success sound for batch completion
            await playSound('success');
        }

        setIsBatchProcessing(false);
    };

    // ===== HISTORY HANDLERS =====

    const deleteHistoryItem = async (id: string) => {
        await dbService.moveToTrash(id);
        setHistory(prev => prev.filter(i => i.id !== id));
    };

    const deleteAllHistory = async () => {
        if (history.length === 0) return;
        // In a real app, we might want a batch delete or server-side endpoint
        // For now, we'll loop through (or if dbService supports batch delete)
        // Assuming we just want to clear the view and move to trash one by one or clear local
        for (const item of history) {
            await dbService.moveToTrash(item.id);
        }
        setHistory([]);
    };

    const handleDownload = async (url: string, filename: string) => {
        try {
            const response = await fetch(url);
            const blob = await response.blob();
            const blobUrl = URL.createObjectURL(blob);

            const link = document.createElement('a');
            link.href = blobUrl;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            URL.revokeObjectURL(blobUrl);
        } catch (e) {
            console.error("Download failed, falling back to direct link", e);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            link.target = "_blank";
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    };

    const handleDownloadAllAccepted = async () => {
        const acceptedItems = history.filter(item => item.metadata?.verdict === 'ACCEPTED');
        if (acceptedItems.length === 0) return;

        setIsDownloadingAccepted(true);
        try {
            const itemsToDownload = acceptedItems.map(item => ({
                url: item.url,
                filename: item.title || `review-${item.id}.png`
            }));
            await downloadItems(itemsToDownload, "accepted_images");
        } finally {
            setIsDownloadingAccepted(false);
        }
    };

    const handleDownloadAllRejected = async () => {
        const rejectedItems = history.filter(item => item.metadata?.verdict !== 'ACCEPTED');
        if (rejectedItems.length === 0) return;

        setIsDownloadingRejected(true);
        try {
            const itemsToDownload = rejectedItems.map(item => ({
                url: item.url,
                filename: item.title || `review-${item.id}.png`
            }));
            await downloadItems(itemsToDownload, "rejected_images");
        } finally {
            setIsDownloadingRejected(false);
        }
    };

    const handleDownloadHistoryGeneric = async (forceSeparate: boolean) => {
        if (history.length === 0) return;

        if (forceSeparate) {
            setIsDownloadingSeparate(true);
        } else {
            setIsDownloadingZip(true);
        }

        try {
            const itemsToDownload = history.map(item => {
                let ext = 'png';
                if (item.url.startsWith('data:image/jpeg')) ext = 'jpg';
                else if (item.url.startsWith('data:image/webp')) ext = 'webp';

                const filename = item.title ? (item.title.endsWith('.' + ext) ? item.title : `${item.title}.${ext}`) : `review-${item.id}.${ext}`;
                return { url: item.url, filename };
            });

            const zipName = `reviewer_history_export_${Date.now()}`;
            await downloadItems(itemsToDownload, zipName, forceSeparate);

        } catch (e) {
            console.error("Download error", e);
        } finally {
            if (forceSeparate) {
                setIsDownloadingSeparate(false);
            } else {
                setIsDownloadingZip(false);
            }
        }
    };

    // ===== UI COMPONENTS =====

    const StatusBadge = ({ verdict, score }: { verdict: string, score: number }) => {
        const isAccepted = verdict === 'ACCEPTED';
        // Score is calculated as scoreA + scoreB (max 200), but we display as /100
        const displayScore = Math.round(score / 2);
        return (
            <div className={`flex items-center gap-2 px-4 py-2 rounded-full font-bold text-white shadow-lg ${isAccepted ? 'bg-green-500' : 'bg-red-500'}`}>
                {isAccepted ? <Check size={20} /> : <X size={20} />}
                <span>{verdict} ({displayScore}/100)</span>
            </div>
        );
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 min-h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">
            {/* Header */}
            <div className="mb-8 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <Camera className="text-blue-500 w-8 h-8" /> Reviewer AI
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">Pre-check your images for stock platform rejection reasons.</p>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Tabs */}
            <div className={`flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-8 shrink-0 transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>
                <div className="flex w-full md:w-auto overflow-x-auto">
                    {['single', 'bulk', 'history'].map((tab) => (
                        <button
                            key={tab}
                            onClick={() => setActiveTab(tab as any)}
                            className={`px-6 py-3 text-sm font-bold transition-all relative capitalize flex items-center gap-2 ${activeTab === tab ? 'text-blue-600 dark:text-blue-400' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                }`}
                        >
                            {tab === 'single' ? 'Single Image' : tab === 'bulk' ? 'Bulk Processing' : 'History'}
                            {activeTab === tab && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-blue-500 animate-fade-in"></div>}
                        </button>
                    ))}
                </div>

                {/* Right side controls - horizontal layout */}
                <div className="flex flex-row gap-2 items-center my-2">
                    {/* History Delete All Button */}
                    {activeTab === 'history' && history.length > 0 && (
                        <>
                            <button
                                onClick={() => handleDownloadHistoryGeneric(true)}
                                disabled={isDownloadingSeparate}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-slate-700 text-slate-400 hover:bg-slate-900 hover:text-white font-bold rounded-xl transition text-[10px] uppercase disabled:opacity-50 disabled:cursor-not-allowed"
                                title="Download all assets as separate files"
                            >
                                {isDownloadingSeparate ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                SEPARATE ALL
                            </button>
                            <button
                                onClick={() => handleDownloadHistoryGeneric(false)}
                                disabled={isDownloadingZip}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 font-bold rounded-xl transition text-[10px] uppercase disabled:opacity-50 disabled:cursor-not-allowed"
                                title="Download all assets in a single ZIP archive"
                            >
                                {isDownloadingZip ? <Loader2 size={14} className="animate-spin" /> : <FileDown size={14} />}
                                ZIP ALL
                            </button>
                            <button
                                onClick={handleDownloadAllRejected}
                                disabled={isDownloadingRejected}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 font-bold rounded-xl transition text-[10px] uppercase disabled:opacity-50 disabled:cursor-not-allowed"
                                title="Download all rejected images"
                            >
                                {isDownloadingRejected ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                ALL REJECTED
                            </button>
                            <button
                                onClick={handleDownloadAllAccepted}
                                disabled={isDownloadingAccepted}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-green-500 text-green-500 hover:bg-green-500/10 hover:shadow-[0_0_15px_rgba(34,197,94,0.3)] hover:text-green-400 font-bold rounded-xl transition text-[10px] uppercase disabled:opacity-50 disabled:cursor-not-allowed"
                                title="Download all accepted images"
                            >
                                {isDownloadingAccepted ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                ALL ACCEPTED
                            </button>
                        </>
                    )}



                    {/* Active Model Display & Auto-Download Indicator - on the right */}
                    {/* Active Model Display - REMOVED */}
                    {activeTab !== 'history' && (
                        <div className="hidden md:flex items-center gap-2">
                            <div className="flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg">
                                <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                                    <Cpu size={12} /> Active Model:
                                </span>
                                <div className="flex gap-1">
                                    {activeModels.map((model, idx) => (
                                        <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                            {model}
                                        </span>
                                    ))}
                                </div>
                            </div>
                            <AutoDownloadIndicator />
                        </div>
                    )}
                </div>
            </div>

            <div className={`flex-1 transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>

                {/* SINGLE MODE */}
                {activeTab === 'single' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 pb-20">
                        {/* Left: Upload \*/}
                        <div className="lg:col-span-5 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Upload Image</label>
                                <div
                                    onClick={() => fileInputRef.current?.click()}
                                    className="relative border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl h-64 flex flex-col items-center justify-center cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-900/50 transition overflow-hidden group bg-slate-50 dark:bg-slate-900"
                                >
                                    {image ? (
                                        <>
                                            <img src={image.preview} className="w-full h-full object-contain opacity-90" />
                                            <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition flex items-center justify-center">
                                                <span className="px-4 py-2 bg-white dark:bg-slate-900 rounded-lg text-xs font-bold text-slate-900 dark:text-white shadow-xl">Change Image</span>
                                            </div>
                                        </>
                                    ) : (
                                        <>
                                            <Upload className="text-slate-400 dark:text-slate-500 mb-3 w-10 h-10 group-hover:text-blue-500 dark:group-hover:text-blue-400 transition-colors" />
                                            <span className="text-sm font-bold text-slate-500 dark:text-slate-400">Click to Upload</span>
                                            <span className="text-[10px] text-slate-400 dark:text-slate-600 mt-1">JPG, PNG, WEBP supported</span>
                                        </>
                                    )}
                                    <input ref={fileInputRef} type="file" className="hidden" onChange={handleFileChange} accept="image/*" />
                                </div>
                            </div>



                            {/* Anatomy Check Toggle */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none">
                                <div className="flex items-start justify-between">
                                    <div className="flex-1 mr-4">
                                        <div className="flex items-center gap-2 mb-1">
                                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">Anatomy Check</h4>
                                            <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableAnatomyCheck ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                {enableAnatomyCheck ? 'ENABLED' : 'DISABLED'}
                                            </span>
                                        </div>
                                        <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                            Check only human face, eyes, artifacts AI, bugs, errors... 1 API call if enabled.
                                        </p>

                                    </div>
                                    <button
                                        onClick={() => setEnableAnatomyCheck(!enableAnatomyCheck)}
                                        className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableAnatomyCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                    >
                                        <span
                                            className={`${enableAnatomyCheck ? 'translate-x-6' : 'translate-x-1'} inline-block h-4 w-4 transform rounded-full bg-white transition-transform`}
                                        />
                                    </button>
                                </div>
                                {enableAnatomyCheck && (
                                    <div className="mt-3 pt-3 border-t border-slate-100 dark:border-slate-800 flex items-center justify-between animate-fade-in">
                                        <div className="flex items-center gap-2">
                                            <AlertCircle size={14} className="text-amber-500" />
                                            <div>
                                                <p className="text-xs font-bold text-slate-700 dark:text-slate-300">Strict Face Mode</p>
                                                <p className="text-[10px] text-slate-500">Aggressive check for "uncanny valley" & horror defects.</p>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => setEnableStrictFaceCheck(!enableStrictFaceCheck)}
                                            className={`relative inline-flex h-4 w-7 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableStrictFaceCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                        >
                                            <span
                                                className={`${enableStrictFaceCheck ? 'translate-x-3.5' : 'translate-x-0.5'} inline-block h-3 w-3 transform rounded-full bg-white transition-transform`}
                                            />
                                        </button>
                                    </div>
                                )}
                            </div>

                            {/* Quality Check Toggle */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none">
                                <div className="flex items-start justify-between">
                                    <div className="flex-1 mr-4">
                                        <div className="flex items-center gap-2 mb-1">
                                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">Quality Check</h4>
                                            <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableQualityCheck ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                {enableQualityCheck ? 'ENABLED' : 'DISABLED'}
                                            </span>
                                        </div>
                                        <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                            Analyzes technical quality (noise, blur, lighting, composition). 1 API call if enabled.
                                        </p>
                                    </div>
                                    <button
                                        onClick={() => setEnableQualityCheck(!enableQualityCheck)}
                                        className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableQualityCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                    >
                                        <span
                                            className={`${enableQualityCheck ? 'translate-x-6' : 'translate-x-1'} inline-block h-4 w-4 transform rounded-full bg-white transition-transform`}
                                        />
                                    </button>
                                </div>
                                {enableQualityCheck && (
                                    <div className="mt-3 pt-3 border-t border-slate-100 dark:border-slate-800 flex items-center justify-between animate-fade-in">
                                        <div className="flex items-center gap-2">
                                            <div className="p-1.5 bg-pink-500/10 rounded-full text-pink-500">
                                                <Layers size={14} />
                                            </div>
                                            <div>
                                                <p className="text-xs font-bold text-slate-700 dark:text-slate-300">Include Anime & Cartoon Style</p>
                                                <p className="text-[10px] text-slate-500">Allows plastic skin, cel-shading, and stylized non-photorealistic traits.</p>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => setEnableAnimeStyle(!enableAnimeStyle)}
                                            className={`relative inline-flex h-4 w-7 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-pink-500 focus:ring-offset-2 ${enableAnimeStyle ? 'bg-pink-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                        >
                                            <span
                                                className={`${enableAnimeStyle ? 'translate-x-3.5' : 'translate-x-0.5'} inline-block h-3 w-3 transform rounded-full bg-white transition-transform`}
                                            />
                                        </button>
                                    </div>
                                )}
                            </div>

                            {/* Hand Recognition & Fix Toggle */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none bg-indigo-50/50 dark:bg-indigo-900/5 border-indigo-100 dark:border-indigo-900/30">
                                <div className="flex items-start justify-between">
                                    <div className="flex-1 mr-4">
                                        <div className="flex items-center gap-2 mb-1">
                                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">Recognize Humans Hands & Fix It</h4>
                                            <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableHandFix ? 'bg-indigo-100 text-indigo-600 dark:bg-indigo-900/30 dark:text-indigo-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                {enableHandFix ? 'ENABLED' : 'DISABLED'}
                                            </span>
                                            <span className="text-[10px] bg-purple-500 text-white px-1.5 rounded-md font-bold">GEN</span>
                                        </div>
                                        <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                            Detects hands/arms. If found, automatically rejects & attempts to fix/regenerate the image without hands using AI.
                                        </p>

                                        {/* Model Selector for Fix - Only visible when enabled */}
                                        {enableHandFix && (
                                            <div className="mt-3 flex items-center gap-2">
                                                <span className="text-[10px] font-bold text-slate-500 uppercase">Fix Model:</span>
                                                <select
                                                    value={selectedFixModel}
                                                    onChange={(e) => setSelectedFixModel(e.target.value)}
                                                    onClick={(e) => e.stopPropagation()}
                                                    className="bg-white dark:bg-slate-800 text-xs px-2 py-1 rounded border border-slate-300 dark:border-slate-700 text-slate-700 dark:text-slate-300 focus:ring-2 focus:ring-indigo-500 outline-none"
                                                >
                                                    {availableCreativeModels.map(m => (
                                                        <option key={m} value={m}>{m}</option>
                                                    ))}
                                                </select>
                                            </div>
                                        )}
                                    </div>
                                    <button
                                        onClick={() => setEnableHandFix(!enableHandFix)}
                                        className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 ${enableHandFix ? 'bg-indigo-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                    >
                                        <span
                                            className={`${enableHandFix ? 'translate-x-6' : 'translate-x-1'} inline-block h-4 w-4 transform rounded-full bg-white transition-transform`}
                                        />
                                    </button>
                                </div>
                            </div>

                            <button
                                onClick={analyzeImage}
                                disabled={loading || !image || !hasApiKey || (!enableAnatomyCheck && !enableQualityCheck && !enableHandFix)}
                                className="w-full py-4 bg-gradient-to-r from-blue-500 to-blue-600 hover:from-blue-400 hover:to-blue-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-blue-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed group"
                            >
                                {loading ? <RefreshCw className="animate-spin" /> : <Eye className="group-hover:scale-110 transition-transform" />}
                                {loading ? 'Analyzing...' : 'Scan for Defects'}
                                {!loading && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">
                                    {(enableAnatomyCheck ? 1 : 0) + (enableQualityCheck ? 1 : 0) + (enableHandFix ? 1 : 0)} API Calls
                                </span>}
                            </button>
                        </div>

                        {/* Right: Results */}
                        {/* Right: Results */}


                        {/* Right: Results */}
                        <div className="lg:col-span-7 flex flex-col gap-6">
                            <div className="bg-white dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 min-h-[600px] flex flex-col relative shadow-sm dark:shadow-none">
                                {analysis ? (
                                    <div className="space-y-6">
                                        <div className="flex justify-between items-start">
                                            <h3 className="text-lg font-bold text-slate-900 dark:text-white flex items-center gap-2">
                                                <CheckCircle2 className="text-green-500" size={20} /> Analysis Complete
                                            </h3>
                                            <StatusBadge verdict={analysis.verdict} score={analysis.totalScore} />
                                        </div>

                                        {/* Kill Switch Warning */}
                                        {analysis.killSwitchActivated && (
                                            <div className="p-4 bg-red-50 dark:bg-red-900/10 border-2 border-red-500/50 rounded-xl flex items-start gap-3">
                                                <AlertTriangle className="text-red-500 shrink-0 mt-0.5" size={20} />
                                                <div>
                                                    <h4 className="text-red-700 dark:text-red-300 font-bold text-sm">Kill Switch Activated</h4>
                                                    <p className="text-red-600 dark:text-red-400 text-xs mt-1">Image rejected due to critical defects. Further checks were skipped to save costs.</p>
                                                </div>
                                            </div>
                                        )}

                                        {/* Score Breakdown */}
                                        <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                                            <div className="p-4 bg-blue-50 dark:bg-blue-900/10 border border-blue-200 dark:border-blue-900/30 rounded-xl">
                                                <div className="flex items-center justify-between mb-2">
                                                    <span className="text-xs font-bold text-blue-600 dark:text-blue-400 uppercase">Anatomy</span>
                                                    <span className={`text-2xl font-bold ${analysis.scoreA === 100 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                                        {analysis.scoreA}
                                                    </span>
                                                </div>
                                                <p className="text-[10px] text-blue-600/80 dark:text-blue-400/80">{analysis.anatomyAnalysis.hasHumans ? 'Humans detected' : 'No humans'}</p>
                                            </div>

                                            {enableQualityCheck && (
                                                <div className="p-4 bg-purple-50 dark:bg-purple-900/10 border border-purple-200 dark:border-purple-900/30 rounded-xl">
                                                    <div className="flex items-center justify-between mb-2">
                                                        <span className="text-xs font-bold text-purple-600 dark:text-purple-400 uppercase">Quality</span>
                                                        <span className={`text-2xl font-bold ${analysis.scoreB >= 70 ? 'text-green-600 dark:text-green-400' : analysis.scoreB >= 40 ? 'text-amber-600 dark:text-amber-400' : 'text-red-600 dark:text-red-400'}`}>
                                                            {analysis.scoreB}
                                                        </span>
                                                    </div>
                                                    <p className="text-[10px] text-purple-600/80 dark:text-purple-400/80">Tech Score</p>
                                                </div>
                                            )}

                                            {/* Hand Detection Result Box */}
                                            {analysis.handDetection && (
                                                <div className="p-4 bg-indigo-50 dark:bg-indigo-900/10 border border-indigo-200 dark:border-indigo-900/30 rounded-xl relative overflow-hidden">
                                                    <div className="flex items-center justify-between mb-2 relative z-10">
                                                        <span className="text-xs font-bold text-indigo-600 dark:text-indigo-400 uppercase">Detection</span>
                                                        <span className={`text-xl font-bold ${!analysis.handDetection.hasHands ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                                            {analysis.handDetection.hasHands ? 'HANDS FOUND' : 'CLEAN'}
                                                        </span>
                                                    </div>
                                                    <p className="text-[10px] text-indigo-600/80 dark:text-indigo-400/80 relative z-10">
                                                        {analysis.handDetection.details}
                                                    </p>

                                                    {analysis.handDetection.fixedImageBase64 && (
                                                        <div className="mt-3 relative z-10">
                                                            <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-between">
                                                                <span className="text-xs font-bold text-green-700 dark:text-green-300 flex items-center gap-1">
                                                                    <CheckCircle2 size={12} /> Auto-Fixed
                                                                </span>
                                                                <button
                                                                    onClick={() => handleDownload(`data:image/jpeg;base64,${analysis.handDetection.fixedImageBase64}`, `fixed-${image?.file.name}`)}
                                                                    className="px-2 py-1 bg-green-500 hover:bg-green-600 text-white rounded text-[10px] font-bold flex items-center gap-1 transition-colors"
                                                                >
                                                                    <Download size={10} /> Save Fixed
                                                                </button>
                                                            </div>
                                                        </div>
                                                    )}
                                                </div>
                                            )}
                                        </div>

                                        {/* Combined Reasoning */}
                                        <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800">
                                            <h4 className="text-xs font-bold text-slate-500 uppercase mb-2">Quality Analysis</h4>
                                            <p className="text-slate-700 dark:text-slate-300 italic text-sm leading-relaxed">
                                                "{typeof analysis.qualityAnalysis.reasoning === 'object'
                                                    ? JSON.stringify(analysis.qualityAnalysis.reasoning, null, 2)
                                                    : analysis.qualityAnalysis.reasoning}"
                                            </p>
                                        </div>


                                    </div>
                                ) : (
                                    <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                        <Camera size={48} className="mb-4 opacity-20" />
                                        <p className="text-sm font-medium">Ready to Analyze</p>
                                        <p className="text-xs mt-1">Upload an image to scan for defects.</p>
                                    </div>
                                )}

                                {error && (
                                    <div className="mt-4 bg-red-50 dark:bg-red-900/10 border border-red-200 dark:border-red-900/50 text-red-600 dark:text-red-200 p-4 rounded-xl flex items-start gap-3">
                                        <AlertTriangle className="shrink-0" />
                                        <p className="text-sm">{error}</p>
                                    </div>
                                )}
                            </div>

                            {/* Actions & Debug Info (Outside Box) */}
                            {analysis && (
                                <div className="space-y-6 animate-fade-in-up">
                                    {/* Detected Issues (Moved Outside) */}
                                    <div>
                                        <h4 className="text-xs uppercase tracking-wider text-slate-500 font-bold mb-4">Detected Issues</h4>
                                        {(analysis.anatomyAnalysis?.defects?.length > 0 || analysis.qualityAnalysis?.issues?.length > 0) ? (
                                            <div className="grid md:grid-cols-2 gap-3">
                                                {analysis.anatomyAnalysis?.defects?.map((defect: string, idx: number) => (
                                                    <div key={`anatomy-${idx}`} className="flex items-start gap-3 text-red-700 dark:text-red-300 bg-red-50 dark:bg-red-900/10 p-3 rounded-lg border border-red-100 dark:border-red-900/20">
                                                        <AlertTriangle size={18} className="mt-0.5 shrink-0 text-red-500" />
                                                        <span className="font-medium text-sm">{defect}</span>
                                                    </div>
                                                ))}
                                                {analysis.qualityAnalysis?.issues?.map((issue: string, idx: number) => (
                                                    <div key={`quality-${idx}`} className="flex items-start gap-3 text-amber-700 dark:text-amber-300 bg-amber-50 dark:bg-amber-900/10 p-3 rounded-lg border border-amber-100 dark:border-amber-900/20">
                                                        <AlertCircle size={18} className="mt-0.5 shrink-0 text-amber-500" />
                                                        <span className="font-medium text-sm">{issue}</span>
                                                    </div>
                                                ))}
                                            </div>
                                        ) : (
                                            <div className="flex items-center gap-3 text-green-700 dark:text-green-400 bg-green-50 dark:bg-green-900/10 p-4 rounded-lg border border-green-100 dark:border-green-900/20">
                                                <Check size={24} className="text-green-500" />
                                                <span className="font-medium text-base">No critical defects detected. Ready for submission!</span>
                                            </div>
                                        )}
                                    </div>

                                    <div className="flex gap-4">
                                        {analysis.verdict === 'ACCEPTED' ? (
                                            <a
                                                href={image?.preview}
                                                download={image?.file.name}
                                                className="flex-1 flex items-center justify-center gap-2 py-4 bg-green-600 hover:bg-green-500 text-white rounded-xl font-bold transition-all shadow-lg shadow-green-500/20 text-lg"
                                            >
                                                <Download size={24} />
                                                Download Image
                                            </a>
                                        ) : (
                                            <button
                                                onClick={handleReset}
                                                className="flex-1 flex items-center justify-center gap-2 py-4 bg-slate-200 dark:bg-slate-800 hover:bg-slate-300 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-200 rounded-xl font-bold transition-all text-lg"
                                            >
                                                <Trash2 size={24} />
                                                Discard & Try Another
                                            </button>
                                        )}
                                    </div>

                                    {/* Debug Info - Removed per user request
                                    <details className="group">
                                        <summary className="text-xs font-bold text-slate-400 uppercase cursor-pointer hover:text-slate-600 dark:hover:text-slate-300 select-none flex items-center gap-2">
                                            <span className="group-open:rotate-90 transition-transform">▶</span>
                                            Debug Information (Developer Mode)
                                        </summary>
                                        <div className="mt-3 space-y-4 text-xs font-mono text-slate-600 dark:text-slate-400 bg-slate-100 dark:bg-slate-900/50 p-4 rounded-xl border border-slate-200 dark:border-slate-800">
                                            <div>
                                                <span className="font-bold text-slate-800 dark:text-slate-200 block mb-1">Anatomy Analysis Raw:</span>
                                                <div className="p-2 bg-white dark:bg-slate-950 rounded border border-slate-200 dark:border-slate-800 whitespace-pre-wrap text-amber-600 dark:text-amber-400">
                                                    {analysis.anatomyAnalysis.analysis || "No analysis text provided"}
                                                </div>
                                            </div>
                                            <div>
                                                <span className="font-bold text-slate-800 dark:text-slate-200 block mb-1">Defects Array:</span>
                                                <div className="p-2 bg-white dark:bg-slate-950 rounded border border-slate-200 dark:border-slate-800 whitespace-pre-wrap">
                                                    {JSON.stringify(analysis.anatomyAnalysis.defects, null, 2)}
                                                </div>
                                            </div>
                                            <div>
                                                <span className="font-bold text-slate-800 dark:text-slate-200 block mb-1">Full Response Object:</span>
                                                <div className="p-2 bg-white dark:bg-slate-950 rounded border border-slate-200 dark:border-slate-800 whitespace-pre-wrap max-h-60 overflow-y-auto">
                                                    {JSON.stringify(analysis, null, 2)}
                                                </div>
                                            </div>
                                        </div>
                                    </details>
                                    */}
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* BULK MODE */}
                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 pb-20">
                        <div className="lg:col-span-4 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <h3 className="font-bold text-slate-900 dark:text-white mb-4">Batch Processing</h3>
                                <div
                                    onClick={() => bulkInputRef.current?.click()}
                                    className="relative border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl h-64 flex flex-col items-center justify-center cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-900/50 transition overflow-hidden group bg-slate-50 dark:bg-slate-900"
                                >
                                    <Layers className="text-slate-400 dark:text-slate-500 mb-3 w-10 h-10 group-hover:text-blue-500 dark:group-hover:text-blue-400 transition-colors" />
                                    <span className="text-sm font-bold text-slate-500 dark:text-slate-400">Upload Images</span>
                                    <span className="text-[10px] text-slate-400 dark:text-slate-600 mt-1">Up to 999 images • JPG, PNG, WEBP</span>
                                    {batchItems.length > 0 && (
                                        <span className="mt-2 px-3 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 rounded-full text-xs font-bold">
                                            {batchItems.length} Selected
                                        </span>
                                    )}
                                    <input ref={bulkInputRef} type="file" multiple onChange={handleBatchUpload} className="hidden" accept="image/*" />
                                </div>

                                {/* Anatomy Check Toggle */}
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none">
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1 mr-4">
                                            <div className="flex items-center gap-2 mb-1">
                                                <h4 className="font-bold text-slate-900 dark:text-white text-sm">Anatomy Check</h4>
                                                <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableAnatomyCheck ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                    {enableAnatomyCheck ? 'ENABLED' : 'DISABLED'}
                                                </span>
                                            </div>
                                            <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                                Check only human face, eyes, artifacts AI, bugs, errors...
                                            </p>
                                        </div>
                                        <button
                                            onClick={() => setEnableAnatomyCheck(!enableAnatomyCheck)}
                                            className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableAnatomyCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                        >
                                            <span
                                                className={`${enableAnatomyCheck ? 'translate-x-6' : 'translate-x-1'} inline-block h-4 w-4 transform rounded-full bg-white transition-transform`}
                                            />
                                        </button>
                                    </div>
                                    {enableAnatomyCheck && (
                                        <div className="mt-3 pt-3 border-t border-slate-100 dark:border-slate-800 flex items-center justify-between animate-fade-in">
                                            <div className="flex items-center gap-2">
                                                <AlertCircle size={14} className="text-amber-500" />
                                                <div>
                                                    <p className="text-xs font-bold text-slate-700 dark:text-slate-300">Strict Face Mode</p>
                                                    <p className="text-[10px] text-slate-500">Aggressive check for "uncanny valley" & horror defects.</p>
                                                </div>
                                            </div>
                                            <button
                                                onClick={() => setEnableStrictFaceCheck(!enableStrictFaceCheck)}
                                                className={`relative inline-flex h-4 w-7 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableStrictFaceCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                            >
                                                <span
                                                    className={`${enableStrictFaceCheck ? 'translate-x-3.5' : 'translate-x-0.5'} inline-block h-3 w-3 transform rounded-full bg-white transition-transform`}
                                                />
                                            </button>
                                        </div>
                                    )}
                                </div>

                                {/* Quality Check Toggle (Bulk) */}
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none mt-2">
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1 mr-4">
                                            <div className="flex items-center gap-2 mb-1">
                                                <h4 className="font-bold text-slate-900 dark:text-white text-sm">Quality</h4>
                                                <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableQualityCheck ? 'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                    {enableQualityCheck ? 'ON' : 'OFF'}
                                                </span>
                                            </div>
                                            <p className="text-[10px] text-slate-500 dark:text-slate-400">
                                                Technical check (+1 API).
                                            </p>
                                        </div>
                                        <button
                                            onClick={() => setEnableQualityCheck(!enableQualityCheck)}
                                            className={`relative inline-flex h-5 w-9 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-amber-500 focus:ring-offset-2 ${enableQualityCheck ? 'bg-amber-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                        >
                                            <span
                                                className={`${enableQualityCheck ? 'translate-x-4' : 'translate-x-1'} inline-block h-3.5 w-3.5 transform rounded-full bg-white transition-transform`}
                                            />
                                        </button>
                                    </div>
                                    {enableQualityCheck && (
                                        <div className="mt-3 pt-3 border-t border-slate-100 dark:border-slate-800 flex items-center justify-between animate-fade-in">
                                            <div className="flex items-center gap-2">
                                                <div className="p-1.5 bg-pink-500/10 rounded-full text-pink-500">
                                                    <Layers size={14} />
                                                </div>
                                                <div>
                                                    <p className="text-xs font-bold text-slate-700 dark:text-slate-300">Include Anime & Cartoon Style</p>
                                                    <p className="text-[10px] text-slate-500">Allows plastic skin, cel-shading, and stylized non-photorealistic traits.</p>
                                                </div>
                                            </div>
                                            <button
                                                onClick={() => setEnableAnimeStyle(!enableAnimeStyle)}
                                                className={`relative inline-flex h-4 w-7 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-pink-500 focus:ring-offset-2 ${enableAnimeStyle ? 'bg-pink-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                            >
                                                <span
                                                    className={`${enableAnimeStyle ? 'translate-x-3.5' : 'translate-x-0.5'} inline-block h-3 w-3 transform rounded-full bg-white transition-transform`}
                                                />
                                            </button>
                                        </div>
                                    )}
                                </div>

                                {/* Hand Recognition & Fix Toggle (Bulk) */}
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 shadow-sm dark:shadow-none mt-2 bg-indigo-50/50 dark:bg-indigo-900/5 border-indigo-100 dark:border-indigo-900/30">
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1 mr-4">
                                            <div className="flex items-center gap-2 mb-1">
                                                <h4 className="font-bold text-slate-900 dark:text-white text-sm">Recognize Humans Hands</h4>
                                                <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold ${enableHandFix ? 'bg-indigo-100 text-indigo-600 dark:bg-indigo-900/30 dark:text-indigo-400' : 'bg-slate-200 text-slate-600 dark:bg-slate-800 dark:text-slate-400'}`}>
                                                    {enableHandFix ? 'ON' : 'OFF'}
                                                </span>
                                                <span className="text-[10px] bg-purple-500 text-white px-1.5 rounded-md font-bold">GEN</span>
                                            </div>
                                            <p className="text-[10px] text-slate-500 dark:text-slate-400">
                                                Auto-reject & Fix if detected.
                                            </p>

                                            {/* Model Selector for Fix - Only visible when enabled */}
                                            {enableHandFix && (
                                                <div className="mt-3 flex items-center gap-2">
                                                    <span className="text-[10px] font-bold text-slate-500 uppercase">Fix Model:</span>
                                                    <select
                                                        value={selectedFixModel}
                                                        onChange={(e) => setSelectedFixModel(e.target.value)}
                                                        onClick={(e) => e.stopPropagation()}
                                                        className="bg-white dark:bg-slate-800 text-xs px-2 py-1 rounded border border-slate-300 dark:border-slate-700 text-slate-700 dark:text-slate-300 focus:ring-2 focus:ring-indigo-500 outline-none"
                                                    >
                                                        {availableCreativeModels.map(m => (
                                                            <option key={m} value={m}>{m}</option>
                                                        ))}
                                                    </select>
                                                </div>
                                            )}
                                        </div>
                                        <button
                                            onClick={() => setEnableHandFix(!enableHandFix)}
                                            className={`relative inline-flex h-5 w-9 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 ${enableHandFix ? 'bg-indigo-500' : 'bg-slate-200 dark:bg-slate-700'}`}
                                        >
                                            <span
                                                className={`${enableHandFix ? 'translate-x-4' : 'translate-x-1'} inline-block h-3.5 w-3.5 transform rounded-full bg-white transition-transform`}
                                            />
                                        </button>
                                    </div>
                                </div>

                                <button
                                    onClick={processBatch}
                                    disabled={isBatchProcessing || batchItems.length === 0 || !hasApiKey || (!enableAnatomyCheck && !enableQualityCheck && !enableHandFix)}
                                    className="w-full py-4 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-500 hover:to-indigo-500 text-white font-bold rounded-xl transition-all shadow-lg shadow-blue-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed group mt-4 mb-20"
                                >
                                    {isBatchProcessing ? <RefreshCw className="animate-spin" /> : <Play className="fill-current" />}
                                    {isBatchProcessing ? `Processing ${batchItems.filter(i => i.status === 'complete').length} / ${batchItems.length}` : 'Start Batch Analysis'}
                                    {!isBatchProcessing && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">
                                        {(enableAnatomyCheck ? 1 : 0) + (enableQualityCheck ? 1 : 0) + (enableHandFix ? 1 : 0)} API Calls/img
                                    </span>}
                                </button>
                            </div>

                            {batchItems.length > 0 && (
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                    <div className="flex justify-between text-sm mb-2 font-bold text-slate-700 dark:text-slate-300">
                                        <span>Progress</span>
                                        <span>{batchItems.filter(i => i.status === 'complete').length}/{batchItems.length}</span>
                                    </div>
                                    <div className="w-full bg-slate-200 dark:bg-slate-800 h-2 rounded-full overflow-hidden">
                                        <div className="h-full bg-blue-500 transition-all duration-300" style={{ width: `${(batchItems.filter(i => i.status === 'complete').length / batchItems.length) * 100}%` }}></div>
                                    </div>
                                </div>
                            )}
                        </div>

                        <div className="lg:col-span-8 bg-white dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800 min-h-[600px] flex flex-col shadow-sm dark:shadow-none">
                            <div className="flex-1 p-6">
                                {batchItems.length === 0 ? (
                                    <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                        <Layers size={48} className="mb-4 opacity-20" />
                                        <p className="text-sm">Batch queue is empty</p>
                                    </div>
                                ) : (
                                    <div className="space-y-3">
                                        {batchItems.map((item) => (
                                            <div key={item.id} className="flex items-center gap-4 p-3 bg-slate-50 dark:bg-slate-950/50 rounded-xl border border-slate-100 dark:border-slate-800/50">
                                                <img src={item.preview} className="w-12 h-12 rounded-lg object-cover bg-slate-200 dark:bg-slate-800" />
                                                <div className="flex-1 min-w-0">
                                                    <div className="flex items-center gap-2">
                                                        <p className="text-xs font-bold text-slate-700 dark:text-slate-300 truncate">{item.file.name}</p>
                                                        {item.stats && (
                                                            <div className="flex items-center gap-2 text-[10px] text-slate-400 font-mono bg-slate-100 dark:bg-slate-900 px-2 py-0.5 rounded border border-slate-200 dark:border-slate-800">
                                                                <span className="text-blue-500">{item.stats.model}</span>
                                                                <span className="w-px h-3 bg-slate-300 dark:bg-slate-700"></span>
                                                                <span>{item.stats.apiKey}</span>
                                                                <span className="w-px h-3 bg-slate-300 dark:bg-slate-700"></span>
                                                                <span className={item.status === 'processing' ? 'text-amber-500 animate-pulse' : 'text-green-500'}>
                                                                    {item.status === 'processing' ? 'Processing...' : item.stats.duration}
                                                                </span>
                                                            </div>
                                                        )}
                                                    </div>
                                                    <p className="text-[10px] text-slate-500 mt-1">
                                                        {item.status === 'complete' ? `${item.result?.verdict} (${item.result?.score}/100)` :
                                                            item.status === 'processing' ? 'Analyzing image...' :
                                                                item.status === 'error' ? 'Failed' : 'Queued'}
                                                    </p>
                                                </div>
                                                <div className="flex items-center gap-2">
                                                    {item.status === 'complete' && item.result?.verdict === 'ACCEPTED' && (
                                                        <a
                                                            href={item.preview}
                                                            download={item.file.name}
                                                            className="p-2 bg-green-500/10 hover:bg-green-500/20 text-green-600 dark:text-green-400 rounded-lg transition"
                                                            title="Download image"
                                                        >
                                                            <Download size={16} />
                                                        </a>
                                                    )}
                                                    {/* Show Fixed Image Download if available */}
                                                    {item.status === 'complete' && item.result?.handDetection?.fixedImageBase64 && (
                                                        <button
                                                            onClick={() => handleDownload(`data:image/jpeg;base64,${item.result.handDetection.fixedImageBase64}`, `fixed-${item.file.name}`)}
                                                            className="p-2 bg-purple-500/10 hover:bg-purple-500/20 text-purple-600 dark:text-purple-400 rounded-lg transition"
                                                            title="Download Fixed Image"
                                                        >
                                                            <Download size={16} />
                                                            <span className="text-[10px] ml-1 font-bold">FIXED</span>
                                                        </button>
                                                    )}
                                                    <button
                                                        onClick={() => setBatchItems(prev => prev.filter(i => i.id !== item.id))}
                                                        className="p-2 bg-red-500/10 hover:bg-red-500/20 text-red-600 dark:text-red-400 rounded-lg transition"
                                                        title="Remove from batch"
                                                    >
                                                        <Trash2 size={16} />
                                                    </button>
                                                    {item.status === 'complete' ? <CheckCircle2 size={18} className="text-green-500" /> :
                                                        item.status === 'error' ? <AlertCircle size={18} className="text-red-500" /> :
                                                            item.status === 'processing' ? <RefreshCw size={18} className="animate-spin text-blue-500" /> :
                                                                <RefreshCw size={18} className="text-slate-300 dark:text-slate-600" />}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}

                            </div>
                            {batchItems.length > 0 && batchItems.some(i => i.status === 'complete') && (
                                <div className="p-4 border-t border-slate-200 dark:border-slate-800 flex gap-4 bg-white dark:bg-slate-900">
                                    {batchItems.some(i => i.status === 'complete' && i.result?.verdict === 'ACCEPTED') && (
                                        <button
                                            onClick={() => {
                                                const acceptedItems = batchItems.filter(i => i.status === 'complete' && i.result?.verdict === 'ACCEPTED');
                                                if (acceptedItems.length === 0) {
                                                    alert('No accepted images to download!');
                                                    return;
                                                }
                                                acceptedItems.forEach(item => {
                                                    const a = document.createElement('a');
                                                    a.href = item.preview;
                                                    a.download = item.file.name;
                                                    a.click();
                                                });
                                            }}
                                            className="flex-1 flex items-center justify-center gap-2 py-4 bg-green-600 hover:bg-green-500 text-white rounded-xl font-bold transition-all shadow-lg shadow-green-500/20 text-lg"
                                        >
                                            <Download size={24} />
                                            Download All Accepted
                                        </button>
                                    )}
                                    <button
                                        onClick={() => {
                                            if (clearAllConfirming) {
                                                setBatchItems([]);
                                                setClearAllConfirming(false);
                                                if (clearAllTimerRef.current) {
                                                    clearTimeout(clearAllTimerRef.current);
                                                }
                                            } else {
                                                setClearAllConfirming(true);
                                                clearAllTimerRef.current = setTimeout(() => setClearAllConfirming(false), 3000);
                                            }
                                        }}
                                        className={`flex-1 flex items-center justify-center gap-2 py-4 rounded-xl font-bold transition-all text-lg ${clearAllConfirming
                                            ? 'bg-red-600 hover:bg-red-700 text-white animate-pulse ring-2 ring-red-500 shadow-lg shadow-red-500/20'
                                            : 'bg-slate-200 dark:bg-slate-800 hover:bg-slate-300 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-200'
                                            }`}
                                    >
                                        <Trash2 size={24} />
                                        {clearAllConfirming ? 'Click Again to Confirm' : 'DELETE ALL'}
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* HISTORY MODE */}
                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                            {history.map((item) => (
                                <div key={item.id} className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl overflow-hidden shadow-sm hover:border-blue-500/50 transition group flex flex-col">
                                    <div className="aspect-video bg-slate-100 dark:bg-slate-950 relative group cursor-pointer" onClick={() => setSelectedHistoryItem(item)}>
                                        <img src={item.url} className="w-full h-full object-cover" />
                                        <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-2">
                                            <button
                                                onClick={(e) => {
                                                    e.stopPropagation();
                                                    handleDownload(item.url, item.title || `review-${item.id}.png`);
                                                }}
                                                className="p-2 bg-white/20 backdrop-blur-sm text-white rounded-lg hover:bg-white/30 transition-colors"
                                                title="Download Image"
                                            >
                                                <Download size={20} />
                                            </button>
                                            <button
                                                onClick={() => setSelectedHistoryItem(item)}
                                                className="p-2 bg-blue-500/80 backdrop-blur-sm text-white rounded-lg hover:bg-blue-600/80 transition-colors"
                                                title="View Details"
                                            >
                                                <Eye size={20} />
                                            </button>
                                        </div>
                                    </div>
                                    <div className="p-4 flex flex-col flex-1">
                                        <div className="flex justify-between items-center mb-2">
                                            <span className={`text-[10px] font-bold px-2 py-0.5 rounded-full ${item.metadata?.verdict === 'ACCEPTED' ? 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400' : 'bg-red-100 dark:bg-red-900/30 text-red-700 dark:text-red-400'}`}>
                                                {item.metadata?.verdict || 'UNKNOWN'}
                                            </span>
                                            {/* Fix for Score Display: Use totalScore or fallback, divide by 2 for /100 scale */}
                                            <span className={`text-xs font-bold ${(item.metadata?.totalScore || item.metadata?.score || 0) / 2 >= 70 ? 'text-green-600' : 'text-slate-500'}`}>
                                                {Math.round((item.metadata?.totalScore || item.metadata?.score || 0) / 2)}/100
                                            </span>
                                        </div>
                                        <h4 className="font-bold text-slate-800 dark:text-slate-200 text-sm mb-1 truncate" title={item.title}>{item.title}</h4>
                                        <p className="text-xs text-slate-500 dark:text-slate-400 line-clamp-2 mb-3 flex-1">{item.metadata?.reasoning || item.metadata?.qualityAnalysis?.reasoning}</p>
                                        <div className="flex justify-between items-center pt-3 border-t border-slate-100 dark:border-slate-800 mt-auto">
                                            <span className="text-[10px] text-slate-400">{new Date(item.createdAt).toLocaleDateString()}</span>
                                            <button
                                                onClick={() => setSelectedHistoryItem(item)}
                                                className="text-[10px] font-bold text-blue-500 hover:text-blue-600 uppercase flex items-center gap-1"
                                            >
                                                More Info <AppWindow size={10} />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                            {history.length === 0 && (
                                <div className="col-span-full flex flex-col items-center justify-center h-64 text-slate-500">
                                    <HistoryIcon size={32} className="mb-2 opacity-30" />
                                    <p>No history found.</p>
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* History Details Modal */}
                {selectedHistoryItem && (
                    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80 backdrop-blur-sm animate-fade-in" onClick={() => setSelectedHistoryItem(null)}>
                        {/* WIDER MODAL: w-[95vw] max-w-[1600px] h-[90vh] */}
                        <div className="bg-white dark:bg-slate-900 w-[95vw] max-w-[1600px] h-[90vh] rounded-2xl shadow-2xl overflow-hidden flex flex-col md:flex-row animate-scale-in" onClick={e => e.stopPropagation()}>

                            {/* Image Side - Darker background for contrast */}
                            <div className="w-full md:w-[60%] bg-black flex items-center justify-center p-8 relative group">
                                <img src={selectedHistoryItem.url} className="w-full h-full object-contain shadow-2xl" />

                                {/* Status Badge - MOVED TO BOTTOM LEFT */}
                                <div className="absolute bottom-8 left-8 transition-transform hover:scale-105 origin-bottom-left scale-150">
                                    <StatusBadge verdict={selectedHistoryItem.metadata?.verdict} score={selectedHistoryItem.metadata?.totalScore || 0} />
                                </div>
                            </div>

                            {/* Info Side */}
                            <div className="w-full md:w-[40%] flex flex-col bg-white dark:bg-slate-900 border-l border-slate-200 dark:border-slate-800">
                                {/* Header - Fixed */}
                                <div className="p-6 border-b border-slate-100 dark:border-slate-800 flex justify-between items-start shrink-0">
                                    <div>
                                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-2 leading-tight">{selectedHistoryItem.title}</h3>
                                        <div className="flex items-center gap-3 text-slate-500 dark:text-slate-400 text-sm">
                                            <span className="font-mono">{new Date(selectedHistoryItem.createdAt).toLocaleString()}</span>
                                        </div>
                                    </div>
                                    <button onClick={() => setSelectedHistoryItem(null)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-full transition">
                                        <X size={24} className="text-slate-500" />
                                    </button>
                                </div>

                                {/* Scrollable Content */}
                                <div className="flex-1 overflow-y-auto custom-scrollbar p-6 space-y-8">

                                    {/* Scores - Large Cards */}
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="p-5 bg-blue-50 dark:bg-blue-900/10 rounded-xl border border-blue-100 dark:border-blue-900/20 flex flex-col items-center justify-center text-center">
                                            <span className="text-xs font-bold text-blue-600 dark:text-blue-400 uppercase tracking-widest mb-2">Anatomy Score</span>
                                            <span className="text-4xl font-black text-slate-800 dark:text-slate-200 font-['Space_Grotesk']">{selectedHistoryItem.metadata?.scoreA}/100</span>
                                        </div>
                                        <div className="p-5 bg-purple-50 dark:bg-purple-900/10 rounded-xl border border-purple-100 dark:border-purple-900/20 flex flex-col items-center justify-center text-center">
                                            <span className="text-xs font-bold text-purple-600 dark:text-purple-400 uppercase tracking-widest mb-2">Quality Score</span>
                                            <span className="text-4xl font-black text-slate-800 dark:text-slate-200 font-['Space_Grotesk']">{selectedHistoryItem.metadata?.scoreB}/100</span>
                                        </div>
                                    </div>

                                    {/* Issues List */}
                                    {((selectedHistoryItem.metadata?.anatomyAnalysis?.defects?.length > 0) || (selectedHistoryItem.metadata?.qualityAnalysis?.issues?.length > 0)) && (
                                        <div className="space-y-3">
                                            <h4 className="text-sm font-bold text-slate-500 uppercase tracking-widest">Detected Issues</h4>
                                            <div className="grid gap-2">
                                                {selectedHistoryItem.metadata?.anatomyAnalysis?.defects?.map((defect: string, i: number) => (
                                                    <div key={`a-${i}`} className="flex items-start gap-3 text-sm text-red-600 dark:text-red-400 bg-red-50 dark:bg-red-900/10 p-3 rounded-lg border border-red-100 dark:border-red-900/20">
                                                        <AlertTriangle size={16} className="mt-0.5 shrink-0" /> {defect}
                                                    </div>
                                                ))}
                                                {selectedHistoryItem.metadata?.qualityAnalysis?.issues?.map((issue: string, i: number) => (
                                                    <div key={`q-${i}`} className="flex items-start gap-3 text-sm text-amber-600 dark:text-amber-400 bg-amber-50 dark:bg-amber-900/10 p-3 rounded-lg border border-amber-100 dark:border-amber-900/20">
                                                        <AlertCircle size={16} className="mt-0.5 shrink-0" /> {issue}
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {/* Reasoning */}
                                    <div>
                                        <h4 className="text-sm font-bold text-slate-500 uppercase tracking-widest mb-3">Analysis Reasoning</h4>
                                        <div className="p-5 bg-slate-50 dark:bg-slate-800/50 rounded-xl text-base text-slate-700 dark:text-slate-300 italic leading-relaxed border border-slate-100 dark:border-slate-800">
                                            "{selectedHistoryItem.metadata?.reasoning || selectedHistoryItem.metadata?.qualityAnalysis?.reasoning}"
                                        </div>
                                    </div>
                                </div>

                                {/* Footer Actions - Fixed */}
                                <div className="p-6 border-t border-slate-100 dark:border-slate-800 flex gap-4 shrink-0 bg-white dark:bg-slate-900 z-10">
                                    <button
                                        onClick={() => handleDownload(selectedHistoryItem.url, selectedHistoryItem.title || 'image.png')}
                                        className="flex-1 py-4 bg-blue-600 hover:bg-blue-500 text-white rounded-xl font-bold flex items-center justify-center gap-2 transition shadow-lg shadow-blue-500/20 text-lg uppercase tracking-wide"
                                    >
                                        <Download size={20} /> Download Image
                                    </button>
                                    <button
                                        onClick={() => {
                                            deleteHistoryItem(selectedHistoryItem.id);
                                            setSelectedHistoryItem(null);
                                        }}
                                        className="px-6 py-4 bg-red-50 dark:bg-red-900/10 text-red-600 dark:text-red-400 hover:bg-red-100 dark:hover:bg-red-900/20 rounded-xl font-bold transition border border-red-200 dark:border-red-900/30"
                                    >
                                        <Trash2 size={24} />
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}



            </div>

            {/* Loading Overlay */}
            <EnhancedLoadingOverlay
                isVisible={loading}
                title="Analyzing Image"
                description="Scanning for technical defects and quality issues..."
                simulate={true}
                onCancel={() => setLoading(false)}
                apiKey={generationInfo?.apiKey}
                model={generationInfo?.model}
            />
            {/* Rate Limit Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />
        </div >
    );
};

export default StockPhotoReviewer;
