

import React, { useState, useEffect } from 'react';
import { AppView, User, AdminSection, TutorialStep, SidebarItemSetting } from '../types';
import {
    LayoutDashboard, Sparkles, Image as ImageIcon, Video, ScanEye, PencilRuler,
    LogOut, Settings, Folder, Star, Trash2, ChevronLeft, ChevronRight, Shield,
    Sun, Moon, HelpCircle, Users, Mail, Inbox, Sliders, BarChart3,
    FileText, LayoutGrid, Bug, LayoutTemplate, Lock, Lightbulb, Key,
    CheckCircle2, Circle, X, ListChecks, GraduationCap, Megaphone, BrainCircuit, History as HistoryIcon, TrendingUp, Wand2, SearchCheck
} from 'lucide-react';
import { adminService } from '../services/adminService';
import { TutorialWidget } from './TutorialWidget';

interface SidebarProps {
    currentView: AppView;
    setView: (view: AppView) => void;
    onLogout: () => void;
    user: User | null;
    isCollapsed: boolean;
    toggleCollapse: () => void;
    isMobileOpen: boolean;
    closeMobile: () => void;
    onOpenSettings: (tab?: string) => void;
    onOpenDocumentation: () => void;

    // Admin Props
    adminSection?: AdminSection;
    setAdminSection?: (section: AdminSection) => void;

    // Global Config
    appName?: string;
    appLogo?: string;

    // Notification Badge
    // Notification Badge
    unreadMessagesCount?: number;

    // Sidebar Settings
    sidebarSettings?: { [key: string]: SidebarItemSetting };
}

const Sidebar: React.FC<SidebarProps> = ({
    currentView,
    setView,
    onLogout,
    user,
    isCollapsed,
    toggleCollapse,
    isMobileOpen,
    closeMobile,
    onOpenSettings,
    onOpenDocumentation,
    adminSection,
    setAdminSection,
    appName = "StockForge",
    appLogo,
    unreadMessagesCount = 0,
    sidebarSettings
}) => {
    const [theme, setTheme] = useState<'dark' | 'light'>('dark');
    const isAdminMode = currentView === AppView.ADMIN;

    // Load Site Settings (Global Config) & Cache for Services
    useEffect(() => {
        const cacheSettings = async () => {
            // Clear old cache to ensure fresh fetch
            localStorage.removeItem('sf_site_settings');

            const settings = await adminService.getSiteSettings();
            if (settings) {
                console.log('🔧 Sidebar: Caching site settings', { hasModelIds: !!settings.modelIds, keys: settings.modelIds ? Object.keys(settings.modelIds).length : 0 });
                localStorage.setItem('sf_site_settings', JSON.stringify(settings));
                // Also dispatch event for other components to know settings are fresh
                window.dispatchEvent(new CustomEvent('sf_settings_refreshed', { detail: settings }));
            } else {
                console.warn('⚠️ Sidebar: No settings returned from getSiteSettings');
            }
        };
        cacheSettings();
    }, []);

    // Initialize Theme from LocalStorage
    useEffect(() => {
        const savedTheme = localStorage.getItem('sf_theme');
        if (savedTheme === 'light') {
            handleThemeChange('light');
        } else {
            handleThemeChange('dark'); // Default
        }
    }, []);

    const handleThemeChange = (t: 'dark' | 'light') => {
        setTheme(t);
        localStorage.setItem('sf_theme', t); // Persist

        const html = document.querySelector('html');
        if (t === 'dark') {
            html?.classList.add('dark');
            html?.classList.remove('light');
        } else {
            html?.classList.remove('dark');
            html?.classList.add('light');
        }
    };

    const handleAdminToggle = () => {
        if (isAdminMode) {
            setView(AppView.DASHBOARD);
        } else {
            setView(AppView.ADMIN);
        }
        if (window.innerWidth < 1024) closeMobile();
    };

    // --- Navigation Configs ---

    const userNavGroups = [
        {
            title: 'Main',
            items: [
                { id: AppView.DASHBOARD, label: 'Dashboard', icon: LayoutDashboard },
                { id: AppView.CHAT_INTELLIGENCE, label: 'Market Intelligence', icon: TrendingUp },
                { id: AppView.MEDIA_ANALYSIS, label: 'Metadata AI', icon: ScanEye },
                { id: AppView.REVIEWER_AI, label: 'Reviewer AI', icon: SearchCheck },
            ]
        },
        {
            title: 'Creation Tools',
            items: [
                { id: AppView.PROMPT_GENERATOR, label: 'Auto-Prompt Gen', icon: Sparkles },
                { id: AppView.IMAGE_GENERATION, label: 'Image Generation', icon: ImageIcon },
                { id: AppView.IMAGE_EDITOR, label: 'Image Editor', icon: Wand2 },
                { id: AppView.VIDEO_GENERATION, label: 'Video Generation', icon: Video },
                // Reviewer AI moved to Main
            ]
        },
        {
            title: 'Management',
            items: [
                { id: AppView.LIBRARY, label: 'My Library', icon: Folder },
                { id: AppView.FAVORITES, label: 'Favorites', icon: Star },
                { id: AppView.TRASH, label: 'Trash', icon: Trash2 },
            ]
        }
    ];

    // Admin Stats
    const [openBugCount, setOpenBugCount] = useState(0);

    useEffect(() => {
        if (isAdminMode) {
            const fetchStats = async () => {
                const reports = await adminService.getBugReports();
                const open = reports.filter(r => r.status === 'open' && !r.isDeleted).length;
                setOpenBugCount(open);
            };
            fetchStats();

            // Poll every minute
            const interval = setInterval(fetchStats, 60000);

            // Listen for immediate updates
            window.addEventListener('sf_bug_updated', fetchStats);

            return () => {
                clearInterval(interval);
                window.removeEventListener('sf_bug_updated', fetchStats);
            };
        }
    }, [isAdminMode]);

    const adminNavGroups = [
        {
            title: 'Administration',
            items: [
                { id: AdminSection.OVERVIEW, label: 'Overview', icon: LayoutGrid },
                { id: AdminSection.USERS, label: 'Users Management', icon: Users },
                { id: AdminSection.TIPS, label: 'Tips of Day', icon: Lightbulb },
                { id: AdminSection.TUTORIAL_STEPS, label: 'Tutorial Steps', icon: GraduationCap },
                { id: AdminSection.MESSAGES, label: 'Messages & Chat', icon: Mail, badge: unreadMessagesCount > 0 ? unreadMessagesCount : undefined },
                { id: AdminSection.BROADCASTS, label: 'Broadcasts', icon: Megaphone },
                { id: AdminSection.MARKET_INTELLIGENCE, label: 'Market Hub Admin', icon: BrainCircuit },
                { id: AdminSection.INBOX, label: 'Simulated Inbox', icon: Inbox },
                { id: AdminSection.BUG_REPORTS, label: 'Bug Reports', icon: Bug, badge: openBugCount > 0 ? openBugCount : undefined },
                { id: AdminSection.TRASH, label: 'Recycle Bin', icon: Trash2 },
            ]
        },
        {
            title: 'System',
            items: [
                { id: AdminSection.SITE_SETTINGS, label: 'App Settings', icon: Sliders },
                { id: AdminSection.AUTH_SETTINGS, label: 'Authentication', icon: Lock },
                { id: AdminSection.LANDING_PAGE, label: 'Landing Page', icon: LayoutTemplate },
                { id: AdminSection.EMAIL_SETTINGS, label: 'Email Config', icon: Mail },
                { id: AdminSection.AUDIT, label: 'Audit Log', icon: FileText },
            ]
        }
    ];

    const sidebarWidth = isCollapsed ? 'w-[80px]' : 'w-[280px]';
    const mobileClasses = isMobileOpen ? 'translate-x-0' : '-translate-x-full';

    // Theme accents based on mode
    const accentColor = isAdminMode ? 'blue' : 'amber';
    const bgGradient = isAdminMode ? 'from-blue-400 to-blue-600' : 'from-amber-400 to-amber-600';
    const textAccent = isAdminMode ? 'text-blue-600 dark:text-blue-400' : 'text-amber-700 dark:text-amber-400';
    const bgAccent = isAdminMode ? 'bg-blue-600 dark:bg-blue-500' : 'bg-amber-600 dark:bg-amber-500';

    const getInitials = (name: string) => {
        if (!name) return 'U';
        return name.split(' ').map(n => n[0]).join('').substring(0, 2).toUpperCase();
    };

    return (
        <>
            {/* Mobile Overlay */}
            {isMobileOpen && (
                <div
                    className="fixed inset-0 bg-black/60 backdrop-blur-sm z-40 lg:hidden transition-opacity"
                    onClick={closeMobile}
                />
            )}

            <aside className={`
        fixed lg:static inset-y-0 left-0 z-50 bg-white/90 dark:bg-slate-950/90 backdrop-blur-xl border-r border-slate-200 dark:border-slate-800 
        flex flex-col transition-all duration-300 ease-in-out
        ${sidebarWidth} ${mobileClasses} lg:translate-x-0
      `}>

                {/* Header & Toggle */}
                <div className="h-20 flex items-center px-6 border-b border-slate-200 dark:border-slate-800/50 justify-between shrink-0">
                    <div className={`flex items-center gap-3 ${isCollapsed ? 'justify-center w-full' : ''}`}>
                        {appLogo ? (
                            <img src={appLogo} alt="Logo" className="h-9 w-9 rounded-xl shadow-lg shrink-0 object-cover" />
                        ) : (
                            <div className={`h-9 w-9 rounded-xl bg-gradient-to-br ${bgGradient} flex items-center justify-center shadow-lg shrink-0`}>
                                {isAdminMode ? <Shield className="text-white w-5 h-5" /> : <Sparkles className="text-white w-5 h-5" />}
                            </div>
                        )}

                        {!isCollapsed && (
                            <div className="flex flex-col">
                                <span className="text-xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk'] tracking-tight whitespace-nowrap overflow-hidden text-ellipsis max-w-[160px]">
                                    {appName}
                                </span>
                                {isAdminMode && <span className="text-[10px] font-bold text-blue-600 dark:text-blue-400 uppercase tracking-wider">Admin Console</span>}
                            </div>
                        )}
                    </div>
                    {!isCollapsed && (
                        <button
                            onClick={toggleCollapse}
                            className="hidden lg:flex h-6 w-6 rounded-full bg-slate-100 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 items-center justify-center text-slate-600 hover:text-slate-900 dark:text-slate-400 dark:hover:text-white hover:bg-slate-200 dark:hover:bg-slate-800 transition-colors absolute -right-3 top-7 shadow-lg"
                        >
                            <ChevronLeft size={14} />
                        </button>
                    )}
                </div>

                {/* Expand Button */}
                {isCollapsed && (
                    <button
                        onClick={toggleCollapse}
                        className="hidden lg:flex h-6 w-6 rounded-full bg-slate-100 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 items-center justify-center text-slate-600 hover:text-slate-900 dark:text-slate-400 dark:hover:text-white hover:bg-slate-200 dark:hover:bg-slate-800 transition-colors absolute -right-3 top-7 z-50 shadow-lg"
                    >
                        <ChevronRight size={14} />
                    </button>
                )}

                {/* User Profile Card */}
                {!isCollapsed && user && (
                    <div className="px-4 py-6 shrink-0">
                        <div className="p-4 rounded-2xl bg-gradient-to-b from-slate-50/80 to-white/80 dark:from-slate-800/50 dark:to-slate-900/50 border border-slate-200 dark:border-slate-700/50 group hover:border-slate-300 dark:hover:border-slate-600 transition-all shadow-sm">
                            <div className="flex items-center gap-3">
                                <div className="relative">
                                    {user.photoURL ? (
                                        <img
                                            src={user.photoURL}
                                            alt={user.displayName}
                                            className="w-10 h-10 rounded-full object-cover border-2 border-white dark:border-slate-900 shadow-sm"
                                        />
                                    ) : (
                                        <div className="w-10 h-10 rounded-full bg-gradient-to-tr from-blue-500 to-purple-500 flex items-center justify-center text-xs font-bold text-white shrink-0 border-2 border-white dark:border-slate-900 shadow-sm">
                                            {getInitials(user.displayName)}
                                        </div>
                                    )}
                                    <div className="absolute bottom-0 right-0 w-3 h-3 rounded-full bg-green-500 border-2 border-white dark:border-slate-900"></div>
                                </div>
                                <div className="flex-1 min-w-0">
                                    <p className="text-sm font-bold text-slate-900 dark:text-slate-200 truncate">{user.displayName}</p>
                                    <p className="text-xs text-slate-600 dark:text-slate-500 truncate">{user.email}</p>
                                </div>
                                <button
                                    onClick={() => onOpenSettings('profile')}
                                    className={`text-slate-500 dark:text-slate-500 hover:${textAccent} transition-colors`}
                                >
                                    <Settings size={16} />
                                </button>
                            </div>
                        </div>
                    </div>
                )}

                {/* Admin Toggle Switch */}
                {!isCollapsed && user?.role === 'admin' && (
                    <div className="px-4 mb-2 shrink-0">
                        <button
                            onClick={handleAdminToggle}
                            className={`w-full flex items-center justify-between p-3 rounded-xl border transition-all ${isAdminMode
                                ? 'bg-blue-50 dark:bg-blue-500/10 border-blue-200 dark:border-blue-500/30 text-blue-700 dark:text-blue-400 shadow-[0_0_15px_rgba(59,130,246,0.1)]'
                                : 'bg-slate-50 dark:bg-slate-900/30 border-slate-200 dark:border-slate-800 text-slate-700 dark:text-slate-400 hover:border-slate-300 dark:hover:border-slate-700 hover:text-slate-900 dark:hover:text-slate-300'
                                }`}
                        >
                            <div className="flex items-center gap-2">
                                <Shield size={16} />
                                <span className="text-sm font-medium">{isAdminMode ? 'Admin Panel' : 'Switch to Admin'}</span>
                            </div>
                            <div className={`w-8 h-4 rounded-full p-0.5 flex transition-colors ${isAdminMode ? 'bg-blue-500 justify-end' : 'bg-slate-300 dark:bg-slate-700 justify-start'}`}>
                                <div className="w-3 h-3 bg-white rounded-full shadow-sm" />
                            </div>
                        </button>
                    </div>
                )}

                {/* Navigation */}
                <div className="flex-1 overflow-y-auto custom-scrollbar px-3 space-y-6 py-2">
                    {(isAdminMode ? adminNavGroups : userNavGroups).map((group, idx) => (
                        <div key={idx}>
                            {!isCollapsed && (
                                <h3 className="px-3 text-xs font-bold text-slate-600 dark:text-slate-500 uppercase tracking-wider mb-2 font-['Space_Grotesk']">
                                    {group.title}
                                </h3>
                            )}
                            <div className="space-y-1">
                                {group.items.map((item: any) => {
                                    const setting = sidebarSettings?.[item.id];
                                    if (setting && !setting.enabled) return null;
                                    const isComingSoon = setting?.comingSoon;

                                    const isActive = isAdminMode
                                        ? adminSection === item.id
                                        : currentView === item.id;

                                    return (
                                        <button
                                            key={item.id}
                                            onClick={() => {
                                                if (isComingSoon) return;

                                                if (isAdminMode && setAdminSection) {
                                                    setAdminSection(item.id as AdminSection);
                                                } else {
                                                    setView(item.id as AppView);
                                                }
                                                if (window.innerWidth < 1024) closeMobile();
                                            }}
                                            className={`
                                         w-full flex items-center relative group
                                         ${isCollapsed ? 'justify-center p-3' : 'px-3 py-2.5'}
                                         rounded-xl transition-all duration-200
                                         ${isActive
                                                    ? `bg-gradient-to-r from-${accentColor}-50 dark:from-${accentColor}-500/10 to-${accentColor}-100/50 dark:to-${accentColor}-500/5 ${textAccent} shadow-sm`
                                                    : isComingSoon
                                                        ? 'text-slate-400 dark:text-slate-600 cursor-not-allowed opacity-70'
                                                        : 'text-slate-600 dark:text-slate-400 hover:bg-slate-100 dark:hover:bg-slate-800/50 hover:text-slate-900 dark:hover:text-slate-100'
                                                }
                                     `}
                                            title={isCollapsed ? item.label : ''}
                                        >
                                            {isActive && (
                                                <div className={`absolute left-0 w-1 rounded-r-full ${bgAccent} ${isCollapsed ? 'h-1.5 top-1/2 -translate-y-1/2 left-1' : 'h-6'}`} />
                                            )}

                                            <div className="relative">
                                                <item.icon
                                                    className={`
                                            ${isCollapsed ? 'w-6 h-6' : 'w-5 h-5'} 
                                            ${isActive ? textAccent : 'text-slate-500 dark:text-slate-400 group-hover:text-slate-700 dark:group-hover:text-slate-100'}
                                            transition-colors
                                         `}
                                                />
                                                {item.badge && isCollapsed && (
                                                    <div className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full border-2 border-white dark:border-slate-900"></div>
                                                )}
                                            </div>

                                            {!isCollapsed && (
                                                <span className="ml-3 text-sm font-medium flex-1 text-left flex items-center gap-2">
                                                    {item.label}
                                                    {isComingSoon && <span className="px-1.5 py-0.5 rounded text-[9px] font-bold bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400 uppercase tracking-wide">Soon</span>}
                                                </span>
                                            )}

                                            {!isCollapsed && item.badge && (
                                                <span className="px-2 py-0.5 rounded-full bg-red-500 text-white text-[10px] font-bold shadow-sm">
                                                    {item.badge}
                                                </span>
                                            )}

                                            {!isCollapsed && isActive && !item.badge && (
                                                <div className={`ml-auto w-1.5 h-1.5 rounded-full ${bgAccent}`} />
                                            )}
                                        </button>
                                    )
                                })}
                            </div>
                        </div>
                    ))}
                </div>

                {/* TUTORIAL WIDGET (Pinned Bottom) */}
                {!isAdminMode && !isCollapsed && (
                    <div className="px-4 pb-4 shrink-0">
                        <TutorialWidget
                            onOpenSettings={onOpenSettings}
                            setView={setView}
                        />
                    </div>
                )}

                {/* Footer */}
                <div className="p-4 border-t border-slate-200 dark:border-slate-800/50 bg-slate-50/50 dark:bg-slate-950/50 shrink-0">
                    <div className={`flex items-center justify-between ${isCollapsed ? 'flex-col gap-4' : ''}`}>

                        {/* Theme Switcher */}
                        <div className={`flex items-center bg-white dark:bg-slate-900/80 rounded-lg p-1 border border-slate-200 dark:border-slate-800 ${isCollapsed ? 'flex-col' : ''}`}>
                            {[
                                { id: 'light', icon: Sun },
                                { id: 'dark', icon: Moon }
                            ].map((t) => (
                                <button
                                    key={t.id}
                                    onClick={() => handleThemeChange(t.id as 'light' | 'dark')}
                                    className={`
                                p-1.5 rounded-md flex items-center justify-center transition-all
                                ${theme === t.id ? `bg-slate-100 dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm` : 'text-slate-500 dark:text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}
                            `}
                                    title={`Theme: ${t.id}`}
                                >
                                    <t.icon size={14} />
                                </button>
                            ))}
                        </div>

                        {/* Divider for expanded view */}
                        {!isCollapsed && <div className="h-6 w-px bg-slate-200 dark:bg-slate-800 mx-1"></div>}

                        {/* App Actions */}
                        <div className={`flex items-center gap-1 ${isCollapsed ? 'flex-col' : ''}`}>
                            <button
                                onClick={() => onOpenSettings('api')}
                                className="p-2 rounded-lg text-slate-500 hover:text-amber-600 dark:hover:text-amber-400 hover:bg-amber-50 dark:hover:bg-amber-900/20 transition-colors"
                                title="Gemini API Keys"
                            >
                                <Key size={18} />
                            </button>
                            <button
                                onClick={() => onOpenSettings('api:models')}
                                className="p-2 rounded-lg text-slate-500 hover:text-purple-600 dark:hover:text-purple-400 hover:bg-purple-50 dark:hover:bg-purple-900/20 transition-colors"
                                title="AI Models"
                            >
                                <BrainCircuit size={18} />
                            </button>
                            <button
                                onClick={() => onOpenSettings('app')}
                                className={`p-2 rounded-lg transition-colors ${currentView === AppView.SETTINGS ? `${textAccent} bg-slate-100 dark:bg-slate-800` : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-100 hover:bg-slate-100 dark:hover:bg-slate-800/50'}`}
                                title="App Settings"
                            >
                                <Settings size={18} />
                            </button>

                            <button
                                onClick={onLogout}
                                className="p-2 rounded-lg text-slate-500 hover:text-red-600 dark:hover:text-red-400 hover:bg-red-50 dark:hover:bg-red-500/10 transition-colors"
                                title="Logout"
                            >
                                <LogOut size={18} />
                            </button>
                        </div>
                    </div>
                </div>

            </aside>
        </>
    );
};

export default Sidebar;