import React, { useState, useEffect, useRef } from 'react';
import {
    User as UserIcon, Key, MessageSquare, Sliders, Camera,
    Mail, Shield, ChevronRight, Edit2, Save, Trash2, Plus,
    CheckCircle2, AlertCircle, Loader2, Copy, RefreshCw,
    Terminal, Info, Check, BookOpen, Eye, EyeOff, Lock,
    Activity, BrainCircuit, Palette, ExternalLink, CreditCard,
    Bug, Send, FileText, RotateCcw, Zap, Layers, AlertTriangle,
    Thermometer, Gauge, ShieldAlert, Fingerprint, Video,
    Monitor, Moon, Sun, Smartphone, Clock, ZapOff, MousePointer2,
    Volume2, Bell, Download, HardDrive, RefreshCcw, Paperclip, Laptop, Globe,
    Rocket, Upload, Wand2, TrendingUp, HelpCircle, BarChart2,
    ArrowRight, Sparkles, Image as ImageIcon
} from 'lucide-react';

const COUNTRIES = [
    "United States", "United Kingdom", "Canada", "Germany", "France", "Australia", "Japan", "Brazil", "India", "China", "Russia", "Italy", "Spain", "Mexico", "South Korea", "Netherlands", "Turkey", "Switzerland", "Sweden", "Poland", "Other"
];
import { User, ApiKeyData, ValidationLog, SupportMessage, ModelType, SiteSettings } from '../types';
import { adminService } from '../services/adminService';
import { authService } from '../services/authService';

import { MODELS, MODEL_TYPE_TO_SETTINGS } from '../constants';

const Toggle = ({ checked, onChange }: { checked: boolean, onChange: () => void }) => (
    <button
        onClick={onChange}
        className={`w-12 h-6 rounded-full transition-colors relative ${checked ? 'bg-green-500' : 'bg-slate-200 dark:bg-slate-700'}`}
    >
        <div className={`absolute top-1 left-1 w-4 h-4 bg-white rounded-full transition-transform ${checked ? 'translate-x-6' : 'translate-x-0'}`} />
    </button>
);

type Model = typeof MODELS[number];

export type SettingsTab = 'profile' | 'api' | 'support' | 'app' | 'documentation' | 'billing' | 'bug_report';

interface SettingsPageProps {
    user: User | null;
    activeTab: string;
    onTabChange: (tab: string) => void;
    unreadMessagesCount?: number;
}

// Extended preference interface for local usage within this component
interface ExtendedUserPreferences {
    theme: 'dark' | 'light';
    generationConfig: {
        globalCooldown: number; // ms
        imageDelay: number; // ms
        videoDelay: number; // ms
        metadataDelay: number; // ms
    };
    uiConfig: {
        autoDownload: boolean;
        soundEffects: boolean;
        notifications: boolean;
        reducedMotion: boolean;
    };
}

const SettingsPage: React.FC<SettingsPageProps> = ({ user, activeTab, onTabChange, unreadMessagesCount = 0 }) => {
    const [toast, setToast] = useState<{ type: 'success' | 'error', msg: string } | null>(null);

    // Auto-dismiss toast after 3 seconds
    useEffect(() => {
        if (toast) {
            const timer = setTimeout(() => setToast(null), 3000);
            return () => clearTimeout(timer);
        }
    }, [toast]);

    const [mainTab, subTab] = activeTab.split(':');
    const effectiveTab = mainTab || activeTab;

    const renderTabContent = () => {
        switch (effectiveTab) {
            case 'profile': return <ProfileTab user={user} />;
            case 'api': return <GeminiApiTab key={user?.uid} user={user} setToast={setToast} activeSubTab={subTab} onSubTabChange={(t) => onTabChange(`api:${t}`)} />;
            case 'support': return <SupportTab currentUser={user} />;
            case 'app': return <AppSettingsTab user={user} />;
            case 'documentation': return <DocumentationTab />;
            case 'billing': return <BillingTab />;
            case 'bug_report': return <ReportBugTab user={user} />;
            default: return <ProfileTab user={user} />;
        }
    };

    return (
        <div className="max-w-7xl mx-auto p-4 lg:p-8 min-h-screen font-sans animate-fade-in text-slate-900 dark:text-slate-200">
            <div className="mb-8">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk'] mb-2">Settings & Administration</h1>
                <p className="text-slate-600 dark:text-slate-400 font-medium">Manage your account, configure AI models, and get support.</p>
            </div>

            <div className="flex border-b border-slate-200 dark:border-slate-800 mb-8 overflow-x-auto hide-scrollbar">
                {[
                    { id: 'profile', label: 'Profile', icon: UserIcon },
                    { id: 'api', label: 'Gemini API', icon: Key },
                    { id: 'app', label: 'App Settings', icon: Sliders },
                    { id: 'billing', label: 'Billing', icon: CreditCard },
                    { id: 'documentation', label: 'Documentation', icon: BookOpen },
                    { id: 'support', label: 'Support', icon: MessageSquare },
                    { id: 'bug_report', label: 'Report Bug', icon: Bug },
                ].map((tab) => (
                    <button
                        key={tab.id}
                        onClick={() => onTabChange(tab.id)}
                        className={`flex items-center gap-2 px-6 py-4 text-sm font-bold transition-all relative whitespace-nowrap ${activeTab === tab.id || effectiveTab === tab.id
                            ? 'text-amber-600 dark:text-amber-400'
                            : 'text-slate-600 dark:text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <tab.icon size={18} />
                        {tab.label}
                        {/* Show badge for Support tab when there are unread messages */}
                        {tab.id === 'support' && unreadMessagesCount > 0 && (
                            <span className="px-2 py-0.5 rounded-full bg-red-500 text-white text-[10px] font-bold shadow-sm">
                                {unreadMessagesCount}
                            </span>
                        )}
                        {activeTab === tab.id || (effectiveTab === tab.id) ? <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in"></div> : null}
                    </button>
                ))}
            </div>

            <div className="animate-fade-in">
                {renderTabContent()}
            </div>

            {/* Toast Notification */}
            {toast && (
                <div className={`fixed bottom-4 right-4 px-6 py-4 rounded-xl shadow-2xl z-50 animate-fade-in flex items-center gap-3 ${toast.type === 'success'
                    ? 'bg-green-500 text-white'
                    : 'bg-red-500 text-white'
                    }`}>
                    {toast.type === 'success' ? (
                        <CheckCircle2 size={20} className="shrink-0" />
                    ) : (
                        <AlertCircle size={20} className="shrink-0" />
                    )}
                    <span className="font-bold text-sm">{toast.msg}</span>
                    <button
                        onClick={() => setToast(null)}
                        className="ml-2 hover:opacity-70 transition"
                    >
                        ✕
                    </button>
                </div>
            )}
        </div>
    );
};

const ProfileTab = ({ user }: { user: User | null }) => {
    const displayUser = user || { uid: 'guest', displayName: 'Guest', email: 'guest@example.com', photoURL: null, role: 'user', isVerified: false, createdAt: Date.now(), lastLoginAt: Date.now(), subscription: 'free', usageStats: { imagesGenerated: 0, videosCreated: 0, apiCallsThisMonth: 0, apiCallsLimit: 100, modelUsage: {}, modelLimits: {} } };
    const [displayName, setDisplayName] = useState(displayUser.displayName || '');
    const [photoPreview, setPhotoPreview] = useState<string | null>(displayUser.photoURL);
    const [passwordData, setPasswordData] = useState({ current: '', new: '', confirm: '' });
    const [showPass, setShowPass] = useState({ current: false, new: false });
    const [isChangePassOpen, setIsChangePassOpen] = useState(false);
    const [isSaving, setIsSaving] = useState(false);
    const [toast, setToast] = useState<{ type: 'success' | 'error', msg: string } | null>(null);
    const fileInputRef = useRef<HTMLInputElement>(null);
    const [selectedFile, setSelectedFile] = useState<Blob | null>(null);
    useEffect(() => { if (user) { setDisplayName(user.displayName || ''); setPhotoPreview(user.photoURL); } }, [user?.uid, user?.displayName, user?.photoURL]);
    const getPassStrength = (pass: string) => { let score = 0; if (pass.length >= 8) score++; if (/[A-Z]/.test(pass)) score++; if (/[0-9]/.test(pass)) score++; if (/[^A-Za-z0-9]/.test(pass)) score++; return score; };
    const strength = getPassStrength(passwordData.new);
    const strengthLabels = ['Very Weak', 'Weak', 'Fair', 'Good', 'Strong'];
    const strengthColors = ['bg-red-500', 'bg-red-400', 'bg-amber-500', 'bg-blue-500', 'bg-green-500'];
    const textColors = ['text-red-600 dark:text-red-500', 'text-red-400', 'text-amber-500', 'text-blue-500', 'text-green-500'];
    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            const file = e.target.files[0];
            if (file.size > 5 * 1024 * 1024) {
                setToast({ type: 'error', msg: 'File size too large (Max 5MB)' });
                return;
            }

            const reader = new FileReader();
            reader.onload = (ev) => {
                const img = new Image();
                img.onload = () => {
                    const canvas = document.createElement('canvas');
                    let width = img.width;
                    let height = img.height;
                    const MAX_SIZE = 500;

                    if (width > height) {
                        if (width > MAX_SIZE) {
                            height *= MAX_SIZE / width;
                            width = MAX_SIZE;
                        }
                    } else {
                        if (height > MAX_SIZE) {
                            width *= MAX_SIZE / height;
                            height = MAX_SIZE;
                        }
                    }

                    canvas.width = width;
                    canvas.height = height;
                    const ctx = canvas.getContext('2d');
                    ctx?.drawImage(img, 0, 0, width, height);

                    // Compress to JPEG for preview AND upload
                    const dataUrl = canvas.toDataURL('image/jpeg', 0.7);
                    setPhotoPreview(dataUrl);

                    canvas.toBlob((blob) => {
                        if (blob) setSelectedFile(blob);
                    }, 'image/jpeg', 0.7);
                };
                img.src = ev.target?.result as string;
            };
            reader.readAsDataURL(file);
        }
    };
    const handleRemovePhoto = () => { setPhotoPreview(null); setSelectedFile(null); if (fileInputRef.current) fileInputRef.current.value = ''; };
    const handleSave = async () => {
        if (!user) return;
        setIsSaving(true);
        try {
            let finalPhotoURL = photoPreview;

            if (selectedFile) {
                // Upload newly selected file
                // Convert Blob to File
                const fileToUpload = new File([selectedFile], "avatar.jpg", { type: "image/jpeg" });
                finalPhotoURL = await authService.uploadFile(fileToUpload);
                setSelectedFile(null); // Clear pending upload
            }

            const updatedUser = await authService.updateProfile(user.uid, { displayName, photoURL: finalPhotoURL });

            // Log activity for profile changes
            const changes = [];
            if (displayName !== user.displayName) changes.push('name');
            if (finalPhotoURL !== user.photoURL) changes.push('photo');
            if (changes.length > 0) {
                await authService.logActivity(user.uid, 'setting', `Updated profile (${changes.join(', ')})`);
            }

            // Dispatch event to update global user state (TopBar, etc.)
            window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));

            if (passwordData.new) {
                if (passwordData.new !== passwordData.confirm) {
                    throw new Error("New passwords do not match.");
                }
                if (passwordData.new.length < 6) {
                    throw new Error("Password must be at least 6 characters.");
                }
                await authService.changePassword(user.uid, passwordData.current, passwordData.new);
                await authService.logActivity(user.uid, 'setting', 'Changed password');
                setPasswordData({ current: '', new: '', confirm: '' });
                setIsChangePassOpen(false);
            }
            setToast({ type: 'success', msg: 'Profile updated successfully!' });
        } catch (e: any) {
            setToast({ type: 'error', msg: e.message || 'Update failed' });
        } finally {
            setIsSaving(false);
            setTimeout(() => setToast(null), 4000);
        }
    };
    const [deleteConfirm, setDeleteConfirm] = useState(false);

    const handleDeleteAccount = async () => {
        if (!user) return;

        if (deleteConfirm) {
            try {
                await authService.deleteAccount(user.uid);
                // Optionally, redirect or show success message
            } catch (e: any) {
                alert("Failed to delete account: " + e.message);
            } finally {
                setDeleteConfirm(false); // Reset confirmation state
            }
        } else {
            setDeleteConfirm(true);
        }
    };

    useEffect(() => {
        let timer: NodeJS.Timeout;
        if (deleteConfirm) {
            timer = setTimeout(() => {
                setDeleteConfirm(false);
            }, 5000); // Reset after 5 seconds
        }
        return () => clearTimeout(timer);
    }, [deleteConfirm]);

    useEffect(() => {
        let timer: NodeJS.Timeout;
        if (deleteConfirm) {
            timer = setTimeout(() => {
                setDeleteConfirm(false);
            }, 5000); // Reset after 5 seconds
        }
        return () => clearTimeout(timer);
    }, [deleteConfirm]);

    return (
        <div className="max-w-4xl space-y-8 text-slate-900 dark:text-slate-200">
            <div className="flex flex-col md:flex-row items-center gap-8 p-8 bg-white dark:bg-slate-900 rounded-3xl border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none"><div className="relative group"><div className="w-32 h-32 rounded-full bg-gradient-to-br from-amber-400 to-purple-600 flex items-center justify-center text-4xl font-bold text-white shadow-2xl shadow-purple-500/20 overflow-hidden border-4 border-white dark:border-slate-800">{photoPreview ? (<img src={photoPreview} alt="avatar" className="w-full h-full object-cover" />) : ((displayUser.displayName || 'U').charAt(0).toUpperCase())}</div><button onClick={() => fileInputRef.current?.click()} className="absolute inset-0 rounded-full bg-black/60 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all cursor-pointer backdrop-blur-sm"><div className="flex flex-col items-center"><Camera className="text-white mb-1" size={24} /><span className="text-xs text-white font-bold">Change</span></div></button><input type="file" ref={fileInputRef} onChange={handleFileChange} accept="image/png, image/jpeg, image/webp" className="hidden" /></div><div className="text-center md:text-left flex-1"><h3 className="text-2xl font-bold mb-1 text-slate-900 dark:text-slate-100">Profile Picture</h3><p className="text-sm text-slate-600 dark:text-slate-400 mb-6 font-medium">Supports JPG, PNG (max 5MB)</p><div className="flex flex-wrap gap-3 justify-center md:justify-start"><button onClick={() => fileInputRef.current?.click()} className="px-4 py-2 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 rounded-xl text-sm font-bold text-slate-700 dark:text-slate-300 transition border border-slate-200 dark:border-slate-700">Upload New</button><button onClick={handleRemovePhoto} className="px-4 py-2 bg-slate-100 dark:bg-slate-800 hover:bg-red-100 dark:hover:bg-red-900/20 hover:text-red-600 dark:hover:text-red-400 hover:border-red-200 dark:hover:border-red-500/30 rounded-xl text-sm font-bold text-slate-700 dark:text-slate-300 transition border border-slate-200 dark:border-slate-700">Remove</button></div></div></div>
            <div className="p-8 bg-white dark:bg-slate-900 rounded-3xl border border-slate-200 dark:border-slate-800 space-y-8 shadow-sm dark:shadow-none"><div className="flex items-center gap-2 border-b border-slate-200 dark:border-slate-800 pb-4 mb-2"><UserIcon className="text-amber-600 dark:text-amber-400" size={20} /><h3 className="text-lg font-bold text-slate-900 dark:text-slate-100">Account Information</h3></div><div className="grid grid-cols-1 md:grid-cols-2 gap-8"><div className="space-y-3"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Full Name</label><div className="relative"><input type="text" value={displayName} onChange={e => setDisplayName(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-3.5 pl-4 pr-10 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none transition-all focus:border-amber-500/50 font-medium" placeholder="Enter your full name" /><Edit2 className="absolute right-4 top-3.5 text-slate-400 dark:text-slate-600 w-4 h-4" /></div></div><div className="space-y-3"><div className="flex justify-between items-center"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Email Address</label>{displayUser.isVerified ? (<span className="text-xs font-bold text-green-700 dark:text-green-500 bg-green-100 dark:bg-green-500/10 px-2 py-0.5 rounded flex items-center gap-1 border border-green-200 dark:border-green-500/20"><CheckCircle2 size={12} /> Verified</span>) : (<span className="text-xs font-bold text-amber-700 dark:text-amber-500 bg-amber-100 dark:bg-amber-500/10 px-2 py-0.5 rounded flex items-center gap-1 border border-amber-200 dark:border-amber-500/20"><AlertCircle size={12} /> Unverified</span>)}</div><div className="relative"><Mail className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" /><input type="text" value={displayUser.email} disabled className="w-full bg-slate-50 dark:bg-slate-950/50 border border-slate-200 dark:border-slate-800 rounded-xl py-3.5 pl-12 pr-4 text-slate-500 cursor-not-allowed font-medium" /></div></div></div><div className="flex items-center gap-4 pt-2"><span className="text-sm font-bold text-slate-600 dark:text-slate-400">Account Role:</span><span className={`px-3 py-1 rounded-lg text-xs font-bold flex items-center gap-1 uppercase tracking-wider ${displayUser.role === 'admin' ? 'bg-purple-100 dark:bg-purple-500/10 text-purple-700 dark:text-purple-400 border border-purple-200 dark:border-purple-500/20' : 'bg-blue-100 dark:bg-blue-500/10 text-blue-700 dark:text-blue-400 border border-blue-200 dark:border-blue-500/20'}`}><Shield size={12} /> {displayUser.role}</span></div></div>
            <div className="bg-white dark:bg-slate-900 rounded-3xl border border-slate-200 dark:border-slate-800 overflow-hidden shadow-sm dark:shadow-none"><button onClick={() => setIsChangePassOpen(!isChangePassOpen)} className="w-full flex items-center justify-between p-8 hover:bg-slate-50 dark:hover:bg-slate-800/50 transition"><h3 className="font-bold text-slate-900 dark:text-slate-200 flex items-center gap-3 text-lg"><Lock size={20} className="text-amber-600 dark:text-amber-500" /> Change Password</h3><ChevronRight className={`text-slate-400 dark:text-slate-500 transition-transform duration-300 ${isChangePassOpen ? 'rotate-90' : ''}`} /></button>{isChangePassOpen && (<div className="px-8 pb-8 pt-0 space-y-6 animate-slide-up"><div className="w-full h-px bg-slate-200 dark:bg-slate-800 mb-6"></div><div className="space-y-3"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Current Password</label><div className="relative"><input type={showPass.current ? "text" : "password"} value={passwordData.current} onChange={e => setPasswordData({ ...passwordData, current: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-3.5 px-4 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium" placeholder="Enter current password to confirm" /><button onClick={() => setShowPass({ ...showPass, current: !showPass.current })} className="absolute right-4 top-3.5 text-slate-400 hover:text-slate-600 dark:text-slate-500 dark:hover:text-slate-300">{showPass.current ? <EyeOff size={18} /> : <Eye size={18} />}</button></div></div><div className="grid grid-cols-1 md:grid-cols-2 gap-8"><div className="space-y-3"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">New Password</label><div className="relative"><input type={showPass.new ? "text" : "password"} value={passwordData.new} onChange={e => setPasswordData({ ...passwordData, new: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-3.5 px-4 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none transition-all font-medium" placeholder="Minimum 6 characters" /></div><div className="space-y-2 pt-2"><div className="flex justify-between text-[10px] font-bold"><span className="text-slate-500 uppercase">Security Strength</span><span className={`${textColors[strength]}`}>{strengthLabels[strength]}</span></div><div className="h-1.5 bg-slate-200 dark:bg-slate-800 rounded-full overflow-hidden"><div className={`h-full transition-all duration-500 ease-out ${strengthColors[strength]}`} style={{ width: `${(strength / 4) * 100}%` }}></div></div></div></div><div className="space-y-3"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Confirm Password</label><input type="password" value={passwordData.confirm} onChange={e => setPasswordData({ ...passwordData, confirm: e.target.value })} className={`w-full bg-slate-50 dark:bg-slate-900 border rounded-xl py-3.5 px-4 text-slate-900 dark:text-slate-200 focus:ring-1 outline-none transition-all font-medium ${passwordData.confirm && passwordData.confirm !== passwordData.new ? 'border-red-500/50 focus:border-red-500 focus:ring-red-500' : 'border-slate-200 dark:border-slate-800 focus:ring-amber-500'}`} placeholder="Re-type new password" /></div></div></div>)}</div>
            <div className="flex flex-col-reverse sm:flex-row items-center justify-between pt-8 border-t border-slate-200 dark:border-slate-800 gap-4">
                <div className="w-full sm:w-auto">
                    <button
                        onClick={handleDeleteAccount}
                        className={`text-sm font-bold border px-6 py-3 rounded-xl transition flex items-center gap-2 w-full justify-center ${deleteConfirm
                            ? 'bg-red-500 text-white border-red-500 hover:bg-red-600 animate-pulse'
                            : 'text-red-600 dark:text-red-400 hover:text-red-700 dark:hover:text-red-300 border-red-200 dark:border-red-500/20 hover:bg-red-50 dark:hover:bg-red-500/10'
                            }`}
                    >
                        {deleteConfirm ? <AlertTriangle size={16} /> : <Trash2 size={16} />}
                        {deleteConfirm ? 'Are you sure? Click to Confirm' : 'Delete Account'}
                    </button>
                    {deleteConfirm && (
                        <p className="text-[10px] text-red-500 font-bold mt-2 text-center animate-fade-in">
                            This action is permanent and cannot be undone.
                        </p>
                    )}
                </div>
                <button onClick={handleSave} disabled={isSaving} className="px-10 py-3 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl shadow-lg shadow-amber-500/20 transition flex items-center gap-2 disabled:opacity-70 transform hover:-translate-y-1 active:translate-y-0 w-full sm:w-auto justify-center">{isSaving ? <Loader2 className="animate-spin" /> : <Save size={18} />}{isSaving ? 'Saving...' : 'Save Changes'}</button>
            </div>
            {toast && (<div className={`fixed bottom-8 right-8 px-6 py-4 rounded-xl shadow-2xl animate-slide-in-right flex items-center gap-3 font-bold z-50 ${toast.type === 'success' ? 'bg-green-500 text-slate-950' : 'bg-red-500 text-white'}`}>{toast.type === 'success' ? <CheckCircle2 size={20} /> : <AlertCircle size={20} />}{toast.msg}</div>)}
        </div>
    );
};

const ModelCard: React.FC<any> = ({ model, user, selectedModels, onToggle, apiKeys, siteSettings }) => {
    // Debug Usage Logic
    // console.log(`[ModelCard Debug] ${model.id}`, { usage: user?.usageStats?.modelUsage?.[model.id], limit: user?.usageStats?.modelLimits?.[model.id] });

    // DEFAULT TIER LIMITS (Static Hardcoded - Must match AdminDashboard/Server)
    const DEFAULT_TIER_LIMITS: Record<string, { free: number, paid: number }> = {
        'pro_3_0': { free: 0, paid: 1000 },
        'pro_2_5': { free: 0, paid: 1000 },
        'flash_2_5': { free: 20, paid: 1000 },
        'flash_lite_2_5': { free: 20, paid: 1000 },
        'gemini_3_0_flash': { free: 20, paid: 1000 },

        'imagen_ultra': { free: 10, paid: 500 },
        'imagen_fast': { free: 10, paid: 500 },
        'imagen_std': { free: 10, paid: 500 },
        'gemini_3_pro_image': { free: 50, paid: 500 },
        'gemini_2_5_flash_image': { free: 100, paid: 500 },
        'veo_3_1': { free: 20, paid: 500 },
        'veo_3_1_fast': { free: 50, paid: 500 },
        'veo_3_0': { free: 10, paid: 500 },
        'veo_2_0': { free: 5, paid: 500 },
        'gemini_1_5_pro': { free: 0, paid: 1000 },
    };

    // Calculate limit dynamically based on current site settings and user's keys
    const modelLimit = React.useMemo(() => {
        // If no API keys, fallback to user stats or 0
        if (!apiKeys) return user?.usageStats?.modelLimits?.[model.id] || 0;

        const settingKey = MODEL_TYPE_TO_SETTINGS[model.id];
        if (!settingKey) return 0;

        let total = 0;
        let keysFound = false;

        apiKeys.forEach((key: any) => {
            // Allow 'valid' AND 'unchecked' to match TopBar/Server logic
            if ((key.status === 'valid' || key.status === 'unchecked') && key.isEnabled !== false) {
                keysFound = true;
                const tier = (key.tier || 'FREE').toUpperCase();
                const type = (tier === 'FREE') ? 'free' : 'paid';

                // Check access - if siteSettings is null, default to enabled
                const isEnabled = siteSettings ? (siteSettings.modelAccessTiers?.[model.id]?.[type] !== false) : true;

                if (isEnabled) {
                    // Use siteSettings limits if available AND higher than default (to allow overrides but prevent accidental 0/low), 
                    // OR just prefer defaults if we suspect DB is stale.
                    // Given the user report, we will prioritize Default > SiteSettings for now to ensure visibility of correct credits.
                    const defaultLimit = DEFAULT_TIER_LIMITS[settingKey]?.[type] ?? 0;
                    const siteLimit = siteSettings?.modelLimitsByTier?.[settingKey]?.[type];

                    // Logic: Use Site Setting if it exists and is > default (upgrades), otherwise stick to default/codebase truth.
                    // Actually, simpler: Just use Default ?? Site ?? 0 to force fix.
                    const limit = defaultLimit > 0 ? defaultLimit : (siteLimit ?? 0);
                    total += limit;
                }
            }
        });

        // If we processed keys, return the calculated total. 
        // If NO keys, return 0 (User should not have credits without keys)
        return keysFound ? total : 0;

    }, [apiKeys, siteSettings, model.id, user?.usageStats?.modelLimits]);

    const modelUsed = user?.usageStats?.modelUsage?.[model.id] || 0;
    const remaining = Math.max(0, modelLimit - modelUsed);
    const percentage = modelLimit > 0 ? (remaining / modelLimit) * 100 : 0;
    const isSelected = selectedModels.includes(model.id);

    return (
        <div
            onClick={() => onToggle(model.id)}
            className={`
                relative p-6 rounded-3xl border-2 cursor-pointer transition-all duration-300 group overflow-hidden
                ${isSelected
                    ? 'bg-white dark:bg-slate-900 border-amber-500 shadow-lg'
                    : 'bg-white dark:bg-slate-900 border-slate-200 dark:border-slate-800 hover:border-slate-300 dark:hover:border-slate-600 hover:bg-slate-50 dark:hover:bg-slate-900'
                }
            `}
        >
            {isSelected && <div className="absolute top-0 right-0 w-64 h-64 bg-amber-500/5 rounded-full blur-3xl -translate-y-1/2 translate-x-1/2"></div>}

            <div className="relative z-10">
                <div className="flex justify-between items-start mb-4">
                    <div>
                        <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-2 flex items-center gap-2">
                            {model.name}
                            {model.recommended && <span className="px-2 py-0.5 bg-amber-500 text-white dark:text-slate-950 text-[10px] font-bold rounded-full shadow-lg shadow-amber-500/20">RECOMMENDED</span>}
                        </h3>
                        <span className={`inline-block px-2 py-1 rounded-lg text-[10px] font-bold border ${model.tagColor}`}>
                            {model.tag}
                        </span>
                    </div>
                    <div className={`w-6 h-6 rounded-full flex items-center justify-center transition-all ${isSelected ? 'bg-amber-500 text-white dark:text-slate-950 scale-110 shadow-lg' : 'border-2 border-slate-300 dark:border-slate-700'}`}>
                        {isSelected && <Check size={14} strokeWidth={4} />}
                    </div>
                </div>

                <p className="text-sm text-slate-600 dark:text-slate-400 mb-6 h-10 leading-relaxed font-medium">{model.desc}</p>

                <div className="bg-slate-100 dark:bg-slate-800 rounded-xl p-3 border border-slate-200 dark:border-slate-700 mt-4">
                    <div className="flex justify-between items-center mb-2">
                        <span className="text-xs font-bold text-slate-500 uppercase">Remaining Credits</span>
                        <span className={`text-xs font-bold ${remaining === 0 ? 'text-red-500' : 'text-slate-700 dark:text-slate-300'}`}>
                            {remaining} / {modelLimit}
                        </span>
                    </div>
                    <div className="h-2 bg-slate-200 dark:bg-slate-700 rounded-full overflow-hidden">
                        <div
                            className={`h-full transition-all duration-500 ${remaining === 0 ? 'bg-red-500' : 'bg-amber-500'}`}
                            style={{ width: `${Math.min(100, percentage)}%` }}
                        ></div>
                    </div>
                    {remaining === 0 && (
                        <p className="text-[10px] text-red-500 mt-1 font-bold">Limit reached! Add API keys or switch models.</p>
                    )}
                </div>
            </div>
        </div>
    );
};

const DoubleConfirmButton: React.FC<{
    onConfirm: () => void;
    className?: string;
    title?: string;
    children?: React.ReactNode;
    disabled?: boolean;
}> = ({ onConfirm, className, title, children, disabled }) => {
    const [confirming, setConfirming] = useState(false);
    const timeoutRef = useRef<NodeJS.Timeout | null>(null);

    const handleClick = (e: React.MouseEvent) => {
        e.preventDefault();
        e.stopPropagation();

        if (disabled) return;

        if (confirming) {
            onConfirm();
            setConfirming(false);
            if (timeoutRef.current) clearTimeout(timeoutRef.current);
        } else {
            setConfirming(true);
            timeoutRef.current = setTimeout(() => setConfirming(false), 3000);
        }
    };

    useEffect(() => {
        return () => {
            if (timeoutRef.current) clearTimeout(timeoutRef.current);
        };
    }, []);

    return (
        <button
            onClick={handleClick}
            className={`${className} ${confirming ? 'animate-pulse ring-2 ring-red-500 ring-offset-2 dark:ring-offset-slate-900' : ''}`}
            title={confirming ? "Click again to confirm" : title}
            disabled={disabled}
        >
            {confirming ? (
                <span className="flex items-center gap-1 text-red-600 dark:text-red-400 font-bold">
                    <AlertTriangle size={14} />
                    Confirm?
                </span>
            ) : (
                children
            )}
        </button>
    );
};

const GeminiApiTab: React.FC<{
    user: User | null;
    setToast: (toast: { type: 'success' | 'error', msg: string } | null) => void;
    activeSubTab?: string;
    onSubTabChange: (tab: string) => void;
}> = ({ user, setToast, activeSubTab, onSubTabChange }) => {
    // ... (state definitions remain same)

    // ...



    // ...

    const handleTestDB = async () => {
        if (!user?.uid) return;
        setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `🔧 Testing Database Connection...`, type: 'process' }]);
        try {
            // Attempt to write a dummy key and delete it
            const testKey: ApiKeyData = {
                id: 'test_db_connection',
                key: 'TEST_KEY',
                fullKey: 'TEST_KEY',
                status: 'unchecked',
                addedAt: Date.now(),
                usageCount: 0,
                validationAttempts: 0
            };
            await authService.saveUserApiKeys(user.uid, [...apiKeys, testKey]);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `✅ Database Write Successful`, type: 'success' }]);

            // Cleanup
            await authService.saveUserApiKeys(user.uid, apiKeys);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `✅ Database Cleanup Successful`, type: 'success' }]);
            setToast({ type: 'success', msg: "Database connection is working correctly!" });
        } catch (e: any) {
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `❌ Database Test Failed: ${e.message}`, type: 'error' }]);
            setToast({ type: 'error', msg: `Database Test Failed: ${e.message}` });
        }
    };

    // ... (delete handlers already updated)

    // ...

    // In JSX map:
    // <button onClick={() => { navigator.clipboard.writeText(key.fullKey); setToast({ type: 'success', msg: "Copied API Key!" }); }} ...
    const subTab = activeSubTab || 'keys';
    const [apiKeys, setApiKeys] = useState<ApiKeyData[]>([]);
    const [enableRandomKeyRotation, setEnableRandomKeyRotation] = useState(() => {
        const stored = localStorage.getItem('sf_random_key_rotation');
        return stored !== null ? stored === 'true' : true; // Default to TRUE
    });
    const [newKey, setNewKey] = useState('');
    const [logs, setLogs] = useState<ValidationLog[]>([]);

    const [selectedThinkingModels, setSelectedThinkingModels] = useState<string[]>(() => {
        const stored = localStorage.getItem('sf_thinking_models');
        return stored ? JSON.parse(stored) : [ModelType.FLASH];
    });

    const [selectedCreativeModels, setSelectedCreativeModels] = useState<string[]>(() => {
        const stored = localStorage.getItem('sf_creative_models');
        return stored ? JSON.parse(stored) : [ModelType.IMAGEN_STD];
    });

    const [selectedVideoModels, setSelectedVideoModels] = useState<string[]>(() => {
        const stored = localStorage.getItem('sf_video_models');
        return stored ? JSON.parse(stored) : ['veo-3.1-fast-generate-preview'];
    });

    // Advanced Settings State
    const [apiSettings, setApiSettings] = useState({
        temperature: 0.7,
        topK: 40,
        topP: 0.95,
        maxOutputTokens: 8192,
        safetyFilter: 'block_some'
    });

    const [isSavingModel, setIsSavingModel] = useState(false);
    const [modelSaved, setModelSaved] = useState(false);
    const [isValidatingAll, setIsValidatingAll] = useState(false);
    const [siteSettings, setSiteSettings] = useState<SiteSettings | null>(null);

    const logsEndRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        if (!user?.uid) {
            console.log('[GeminiApiTab] No user UID, skipping key fetch');
            return;
        }

        console.log(`[GeminiApiTab] Fetching API keys for user ${user.uid}...`);
        const fetchData = async () => {
            try {
                // Refresh user profile to get latest stats (usage/credits) from server
                // This ensures we aren't showing stale data if an optimistic update was missed
                await authService.getUserProfile(user.uid, true);

                const keys = await authService.getUserApiKeys(user.uid);
                console.log(`[GeminiApiTab] Successfully fetched ${keys.length} API keys:`, keys.map(k => ({ id: k.id, status: k.status, tier: k.tier, key: k.key })));
                setApiKeys(keys);

                // Fetch site settings for model visibility and access tiers
                const settings = await authService.getSiteSettings();
                setSiteSettings(settings);
            } catch (error) {
                console.error("[GeminiApiTab] Failed to fetch API keys or settings:", error);
                setApiKeys([]); // Reset to empty array on error
            }
        };
        fetchData();

        const storedSettings = localStorage.getItem('sf_model_config_defaults');
        if (storedSettings) {
            setApiSettings(JSON.parse(storedSettings));
        }
    }, [user?.uid]);

    const persistKeys = async (keys: ApiKeyData[]) => {
        if (!user?.uid) throw new Error("User context missing.");

        // INFO: Saving 0 keys is expected behavior when deleting all API keys
        if (keys.length === 0) {
            console.log('[persistKeys] INFO: Saving 0 keys (all keys deleted)');
        }

        try {
            await authService.saveUserApiKeys(user.uid, keys);

            // CRITICAL: Fetch updated keys from server to get latest USAGE stats
            // The local keys have usage: 0 (or old data), but the server calculates real usage from DB.
            const freshKeys = await authService.getUserApiKeys(user.uid);
            if (freshKeys && freshKeys.length > 0) {
                // Preserve 'validating' status if we are in the middle of validation?
                // Actually, if we just saved, we might want the server's truth.
                // But validation runs: setApiKeys(validating) -> validate -> setApiKeys(valid/tier) -> persistKeys
                // If persistKeys overwrites state with server data (which might say 'unchecked' if not synced yet?), we might lose 'valid' status 
                // UNLESS server persists status? Yes, server saves status.

                // However, there is a race:
                // 1. validateKey sets status='valid', calls persistKeys matches
                // 2. persistKeys calls API save (saves 'valid')
                // 3. persistKeys calls API diff (gets 'valid' + real usage)
                // 4. We update state.
                // This seems safe.
                setApiKeys(freshKeys);
            }

            // Fetch updated user profile to reflect new credit limits immediately
            // Use silent=true to prevent getUserProfile from dispatching (we do it manually below)
            const updatedUser = await authService.getUserProfile(user.uid, true);
            if (updatedUser) {
                window.dispatchEvent(new CustomEvent('sf_user_updated', { detail: updatedUser }));
            }
        } catch (e: any) {
            console.error("Persist failed:", e);
            throw e;
        }
    };

    const validateKey = async (key: string, id: string) => {
        let detectedTier: 'FREE' | 'PAID' | 'unknown' = 'FREE';
        const maskedKey = `...${key.slice(-4)}`;

        const log = (msg: string, type: ValidationLog['type']) => {
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: msg, type }]);
        };

        try {
            setApiKeys(prev => prev.map(k => k.id === id ? { ...k, status: 'validating' } : k));

            log(`🔄 Validating API key ${maskedKey}`, 'process');
            await new Promise(r => setTimeout(r, 300));

            let status: 'valid' | 'invalid' = 'invalid';

            // 1. Format Check
            if (!key.startsWith('AIza')) {
                log(`❌ Key format: Invalid (must start with AIza)`, 'error');
                setApiKeys(prev => prev.map(k => k.id === id ? { ...k, status: 'invalid', validationAttempts: k.validationAttempts + 1 } : k));
                return;
            }
            log(`ℹ️ Key format: Valid (starts with AIza)`, 'info');

            // 2. Length Check
            if (key.length === 39) {
                log(`✅ Length: 39 characters (Standard)`, 'success');
            } else if (key.length === 49) {
                log(`✅ Length: 49 characters (Extended)`, 'success');
            } else {
                log(`⚠️ Length: ${key.length} characters (Expected 39 or 49)`, 'warning');
            }

            // 3. Test API connection with Gemini 2.5 Flash Lite
            log(`📡 Testing API key with Gemini 2.5 Flash Lite...`, 'network');

            try {
                const testResponse = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-lite-preview-09-2025:generateContent?key=${key}`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        contents: [{ parts: [{ text: "test" }] }]
                    })
                });

                if (testResponse.ok) {
                    log(`✅ API Connection Successful`, 'success');
                    status = 'valid';

                    // Check rate limit headers for tier detection
                    const rpmLimit = testResponse.headers.get('x-ratelimit-limit-rpm');
                    if (rpmLimit) {
                        const rpm = parseInt(rpmLimit);
                        // Free: 15 RPM, Paid tiers: 1000+ RPM
                        if (rpm >= 1000) {
                            detectedTier = 'PAID';
                            log(`💎 PAID Tier Detected (${rpm} RPM)`, 'success');
                        } else {
                            detectedTier = 'FREE';
                            log(`🆓 FREE Tier Detected (${rpm} RPM)`, 'success');
                        }
                    } else {
                        // No rate limit header - default to FREE
                        detectedTier = 'FREE';
                        log(`🆓 FREE Tier (Default)`, 'success');
                    }
                } else if (testResponse.status === 429) {
                    // Rate limited but valid key
                    log(`✅ API Key Valid (Rate Limited)`, 'success');
                    status = 'valid';

                    const rpmLimit = testResponse.headers.get('x-ratelimit-limit-rpm');
                    if (rpmLimit) {
                        const rpm = parseInt(rpmLimit);
                        detectedTier = rpm >= 1000 ? 'PAID' : 'FREE';
                        log(`${detectedTier === 'PAID' ? '💎 PAID' : '🆓 FREE'} Tier Detected (${rpm} RPM)`, 'success');
                    } else {
                        detectedTier = 'FREE';
                        log(`🆓 FREE Tier (Default)`, 'success');
                    }
                } else if (testResponse.status === 400) {
                    // Model might not be available for this key, try alternative method
                    log(`⚠️ Gemini 2.5 Flash Lite unavailable, using fallback detection`, 'warning');

                    // Try listing models as fallback
                    const listResponse = await fetch(`https://generativelanguage.googleapis.com/v1beta/models?key=${key}`);
                    if (listResponse.ok) {
                        status = 'valid';
                        log(`✅ API Key Valid`, 'success');
                        detectedTier = 'FREE'; // Default to FREE if we can't determine
                        log(`🆓 FREE Tier (Default)`, 'success');
                    } else {
                        const errorData = await listResponse.json();
                        log(`❌ API Error: ${errorData.error?.message || 'Unknown error'}`, 'error');
                    }
                } else {
                    const errorData = await testResponse.json();
                    log(`❌ API Error: ${errorData.error?.message || testResponse.statusText}`, 'error');
                }
            } catch (error: any) {
                log(`❌ Network Error: ${error.message}`, 'error');
            }

            if (status === 'valid') {
                log(`🎉 VALID API KEY - Ready to use!`, 'success');

                // Log activity for Recent Activity box (without deduction)
                if (user?.uid) {
                    try {
                        // NEW: Deduct 1 Credit for the validation call (Gemini 2.5 Flash Lite)
                        // This ensures "2 api calls" are taken if validated twice (or 1 per check)
                        await authService.deductCredits(
                            user.uid,
                            1,
                            key, // Charge specifically to this key if possible (though backend might use user pool if key new)
                            'gemini-2.5-flash-lite-preview-09-2025' // Specific model used for the test
                        );
                        log(`💰 Deducted 1 credit (Flash Lite)`, 'info');

                        await authService.logActivity(user.uid, 'setting', `API Key Validated (${detectedTier} tier)`);
                    } catch (error: any) {
                        console.error('Failed to log validation activity/deduct:', error);
                    }
                }
            }

            // Update state AND persist to database
            setApiKeys(prev => {
                const updatedKeys = prev.map(k =>
                    k.id === id
                        ? { ...k, status, tier: detectedTier, validationAttempts: k.validationAttempts + 1 }
                        : k
                );
                // Persist the tier to database
                if (detectedTier) {
                    persistKeys(updatedKeys).catch(err => console.error('Failed to persist tier:', err));
                }
                return updatedKeys;
            });
        } catch (error: any) {
            log(`❌ Validation Error: ${error.message}`, 'error');
            setApiKeys(prev => prev.map(k =>
                k.id === id
                    ? { ...k, status: 'invalid', validationAttempts: k.validationAttempts + 1 }
                    : k
            ));
        }
    };

    const validateAllKeys = async () => {
        if (apiKeys.length === 0) return;
        if (isValidatingAll) return; // Prevent multiple simultaneous validations

        setIsValidatingAll(true);
        setLogs([]); // Clear logs
        setToast({ type: 'success', msg: `Validating ${apiKeys.length} key${apiKeys.length > 1 ? 's' : ''}...` });

        try {
            for (const keyData of apiKeys) {
                await validateKey(keyData.fullKey, keyData.id);
                await new Promise(r => setTimeout(r, 500)); // Delay between checks
            }
            // Add completion feedback
            setToast({ type: 'success', msg: `Validation complete! Checked all ${apiKeys.length} key${apiKeys.length > 1 ? 's' : ''}.` });
            setLogs(prev => [...prev, {
                id: Math.random().toString(),
                timestamp: new Date().toLocaleTimeString(),
                message: `✅ Validation Complete - All ${apiKeys.length} key${apiKeys.length > 1 ? 's have' : ' has'} been checked`,
                type: 'success'
            }]);
        } catch (error: any) {
            setToast({ type: 'error', msg: `Validation error: ${error.message}` });
            setLogs(prev => [...prev, {
                id: Math.random().toString(),
                timestamp: new Date().toLocaleTimeString(),
                message: `❌ Validation Error: ${error.message}`,
                type: 'error'
            }]);
        } finally {
            setIsValidatingAll(false);
        }
    };

    const handleAddKey = async () => {
        if (!newKey.trim()) return;
        const id = Math.random().toString(36).substr(2, 9);
        const keyData: ApiKeyData = {
            id,
            key: `AIza•••••••••••••••••••••••••${newKey.slice(-4)}`,
            fullKey: newKey,
            status: 'unchecked',
            addedAt: Date.now(),
            usageCount: 0,
            validationAttempts: 0,
            usage: 0 // Initialize usage to 0 so the "Used: 0" badge appears immediately
        };

        const nextKeys = [...apiKeys, keyData];
        setApiKeys(nextKeys);

        try {
            await persistKeys(nextKeys);
        } catch (e: any) {
            console.error("Failed to save key:", e);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `❌ Database Save Failed: ${e.message}`, type: 'error' }]);
            setToast({ type: 'error', msg: `Failed to save API key: ${e.message}` });
        }

        setNewKey('');
        validateKey(newKey, id);
    };



    const handleDeleteKey = async (id: string) => {
        try {
            if (!user?.uid) throw new Error("User ID is missing. Session may be invalid.");
            const nextKeys = apiKeys.filter(k => k.id !== id);
            setApiKeys(nextKeys);
            await persistKeys(nextKeys);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `🗑️ Deleted Key ID: ${id.substring(0, 4)}...`, type: 'success' }]);
        } catch (error: any) {
            const errorMsg = error?.message || "Unknown error";
            console.error("Delete API Key Error:", error);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `❌ Delete Failed: ${errorMsg}`, type: 'error' }]);
        }
    };

    const handleDeleteAllKeys = async () => {
        if (apiKeys.length === 0) return;
        try {
            if (!user?.uid) throw new Error("User ID is missing.");
            setApiKeys([]);
            await persistKeys([]);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `🗑️🗑️ DELETED ALL API KEYS`, type: 'success' }]);
        } catch (error: any) {
            console.error("Delete All Failed:", error);
            setLogs(prev => [...prev, { id: Math.random().toString(), timestamp: new Date().toLocaleTimeString(), message: `❌ Delete All Failed: ${error.message}`, type: 'error' }]);
        }
    };

    const handleToggleKey = async (id: string) => {
        const targetKey = apiKeys.find(k => k.id === id);
        if (!targetKey) return;

        // Default to true if undefined, so we know current state
        const currentState = targetKey.isEnabled ?? true;
        const newState = !currentState;

        // Validation for Mixed Tiers (Only check when enabling)
        if (newState === true) {
            const targetTier = (targetKey.tier || 'FREE').toUpperCase();
            const isTargetFree = targetTier === 'FREE';

            const hasConflict = apiKeys.some(k => {
                if (k.id === id) return false; // Skip self
                if (k.isEnabled === false) return false; // Skip disabled keys

                const otherTier = (k.tier || 'FREE').toUpperCase();
                const isOtherFree = otherTier === 'FREE';

                // Conflict if one is Free and other is Paid
                return isTargetFree !== isOtherFree;
            });

            if (hasConflict) {
                setToast({
                    type: 'error',
                    msg: `Cannot enable ${isTargetFree ? 'Free' : 'Paid'} key while ${isTargetFree ? 'Paid' : 'Free'} keys are active. Please disable them first.`
                });
                return;
            }
        }

        const nextKeys = apiKeys.map(k => {
            if (k.id === id) {
                return { ...k, isEnabled: newState };
            }
            return k;
        });
        setApiKeys(nextKeys);
        await persistKeys(nextKeys);
    };

    const toggleThinkingModel = (id: string) => {
        if (selectedThinkingModels.includes(id) && selectedThinkingModels.length === 1) {
            setToast({ type: 'error', msg: 'At least one Thinking model must be enabled!' });
            return;
        }
        setSelectedThinkingModels(prev => prev.includes(id) ? prev.filter(m => m !== id) : [...prev, id]);
    };
    const toggleCreativeModel = (id: string) => {
        if (selectedCreativeModels.includes(id) && selectedCreativeModels.length === 1) {
            setToast({ type: 'error', msg: 'At least one Creative model must be enabled!' });
            return;
        }
        setSelectedCreativeModels(prev => prev.includes(id) ? prev.filter(m => m !== id) : [...prev, id]);
    };
    const toggleVideoModel = (id: string) => {
        if (selectedVideoModels.includes(id) && selectedVideoModels.length === 1) {
            setToast({ type: 'error', msg: 'At least one Video model must be enabled!' });
            return;
        }
        setSelectedVideoModels(prev => prev.includes(id) ? prev.filter(m => m !== id) : [...prev, id]);
    };

    const handleSaveModels = () => {
        setIsSavingModel(true);

        // Save immediately to prevent data loss on refresh
        localStorage.setItem('sf_thinking_models', JSON.stringify(selectedThinkingModels));
        localStorage.setItem('sf_creative_models', JSON.stringify(selectedCreativeModels));
        localStorage.setItem('sf_video_models', JSON.stringify(selectedVideoModels));
        localStorage.setItem('sf_model_config_defaults', JSON.stringify(apiSettings));

        setTimeout(() => {
            setIsSavingModel(false);
            setModelSaved(true);
            setTimeout(() => setModelSaved(false), 3000);
        }, 500);
    };

    useEffect(() => {
        logsEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    }, [logs]);

    return (
        <div className="max-w-[1800px] mx-auto">
            <div className="flex gap-2 mb-8 p-1 bg-slate-100 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 w-fit">
                {[
                    { id: 'keys', label: 'API Keys' },
                    { id: 'models', label: 'Models' },
                    { id: 'settings', label: 'API Settings' }
                ].map(t => {
                    const isLocked = t.id === 'models' && apiKeys.length === 0;
                    return (
                        <button
                            key={t.id}
                            onClick={() => onSubTabChange(t.id)}
                            className={`px-6 py-2.5 rounded-lg text-sm font-bold transition-all flex items-center gap-2 ${subTab === t.id
                                ? 'bg-amber-500 text-white dark:text-slate-950 shadow-lg'
                                : isLocked
                                    ? 'text-slate-500 dark:text-slate-500 bg-slate-100 dark:bg-slate-900/50 hover:text-slate-700 dark:hover:text-slate-300'
                                    : 'text-slate-600 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white'
                                }`}
                            title={isLocked ? "Add an API Key to configure models" : t.label}
                        >
                            {isLocked && <Lock size={14} />}
                            {t.label}
                        </button>
                    );
                })}
            </div>

            {subTab === 'keys' && (
                <div className="grid grid-cols-1 lg:grid-cols-10 gap-8 animate-fade-in">
                    <div className="lg:col-span-7 space-y-8">
                        <div className="bg-blue-50 dark:bg-blue-500/10 border border-blue-200 dark:border-blue-500/20 rounded-2xl p-6 flex gap-4"><div className="p-2 bg-blue-100 dark:bg-blue-500/20 rounded-lg h-fit"><Info className="text-blue-600 dark:text-blue-400" size={24} /></div><div><h4 className="text-blue-700 dark:text-blue-400 font-bold text-lg mb-2">How API Keys Work</h4><p className="text-blue-600/80 dark:text-blue-200/70 text-sm leading-relaxed mb-3 font-medium">Add multiple Gemini API keys to increase your usage limits. The system automatically rotates between valid keys to distribute API calls evenly. Invalid keys are temporarily disabled and retried after 1 hour.</p><a href="https://aistudio.google.com/app/apikey" target="_blank" rel="noreferrer" className="text-blue-700 dark:text-blue-400 text-sm font-bold hover:underline inline-flex items-center gap-1">Get free API key from Google AI Studio <ChevronRight size={12} /></a></div></div>
                        <div className="space-y-3"><label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Add New Key</label><div className="flex gap-3"><div className="relative flex-1"><Key className="absolute left-4 top-3.5 text-slate-400 dark:text-slate-500 w-5 h-5" /><input type="text" value={newKey} onChange={e => setNewKey(e.target.value)} placeholder="Paste your Gemini API key here (AIza...)" className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-3.5 pl-12 pr-4 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none font-mono text-sm transition-all font-medium" onKeyDown={e => e.key === 'Enter' && handleAddKey()} /></div><button onClick={handleAddKey} disabled={!newKey} className="px-6 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl transition flex items-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg shadow-amber-500/20"><Plus size={18} /> Add Key</button></div></div>
                        <div className="space-y-4">
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-4">
                                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Active Keys</label>
                                    <div className="flex items-center gap-2">
                                        <div className="group relative flex items-center">
                                            <Info size={14} className="text-slate-400 hover:text-blue-500 cursor-help transition-colors" />
                                            <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 w-64 p-3 bg-slate-900 text-white text-xs rounded-lg shadow-xl opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none z-50">
                                                <p className="font-bold mb-1">Random Key Rotation</p>
                                                <p className="leading-relaxed opacity-90">
                                                    When enabled, the system picks a random API key for each request instead of using them in order. This helps distribute load evenly across all your keys and prevents hitting rate limits on a single key.
                                                </p>
                                                <div className="absolute bottom-[-4px] left-1/2 -translate-x-1/2 w-2 h-2 bg-slate-900 rotate-45"></div>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => {
                                                const newState = !enableRandomKeyRotation;
                                                setEnableRandomKeyRotation(newState);
                                                localStorage.setItem('sf_random_key_rotation', String(newState));
                                            }}
                                            className={`w-8 h-4 rounded-full relative transition-colors ${enableRandomKeyRotation ? 'bg-amber-500' : 'bg-slate-300 dark:bg-slate-700'}`}
                                            title="Random API Key Rotation: Randomly picks a key for each request to distribute load."
                                        >
                                            <div className={`absolute top-0.5 left-0.5 w-3 h-3 bg-white rounded-full transition-transform ${enableRandomKeyRotation ? 'translate-x-4' : 'translate-x-0'}`} />
                                        </button>
                                        <span className="text-[10px] font-bold text-slate-500 dark:text-slate-400 uppercase">Random Rotation</span>
                                    </div>
                                </div>
                                {apiKeys.length > 0 && (
                                    <div className="flex items-center gap-2">
                                        <button
                                            onClick={handleTestDB}
                                            className="p-2 bg-slate-100 dark:bg-slate-800 hover:bg-purple-100 dark:hover:bg-purple-900/30 text-slate-600 dark:text-slate-400 hover:text-purple-600 dark:hover:text-purple-400 rounded-lg transition"
                                            title="Test Database Connection"
                                        >
                                            <HardDrive size={16} />
                                        </button>
                                        <button
                                            onClick={validateAllKeys}
                                            disabled={isValidatingAll || apiKeys.length === 0}
                                            className={`p-2 bg-slate-100 dark:bg-slate-800 rounded-lg transition ${isValidatingAll
                                                ? 'opacity-50 cursor-not-allowed'
                                                : 'hover:bg-blue-100 dark:hover:bg-blue-900/30 text-slate-600 dark:text-slate-400 hover:text-blue-600 dark:hover:text-blue-400'
                                                }`}
                                            title={isValidatingAll ? 'Validating...' : 'Refresh / Validate All Keys'}
                                        >
                                            <RefreshCw size={16} className={isValidatingAll ? 'animate-spin' : ''} />
                                        </button>
                                        <DoubleConfirmButton
                                            onConfirm={handleDeleteAllKeys}
                                            className="p-2 bg-slate-100 dark:bg-slate-800 hover:bg-red-100 dark:hover:bg-red-900/30 text-slate-600 dark:text-slate-400 hover:text-red-600 dark:hover:text-red-400 rounded-lg transition"
                                            title="Delete All Keys"
                                        >
                                            <Trash2 size={16} />
                                        </DoubleConfirmButton>
                                    </div>
                                )}
                            </div>
                            {apiKeys.length === 0 && (<div className="text-center p-12 border-2 border-dashed border-slate-200 dark:border-slate-800 rounded-3xl text-slate-400 dark:text-slate-500 bg-slate-50 dark:bg-slate-900/30"><Key size={32} className="mx-auto mb-3 opacity-50" />No API keys added yet. Add one to get started.</div>)}
                            {apiKeys.map(key => (
                                <div key={key.id} className={`bg-white dark:bg-slate-900 border ${key.isEnabled === false ? 'border-slate-100 dark:border-slate-800 opacity-60' : 'border-slate-200 dark:border-slate-800'} rounded-2xl p-5 flex flex-col sm:flex-row sm:items-center justify-between group hover:border-slate-300 dark:hover:border-slate-600 transition gap-4 shadow-sm dark:shadow-none`}>
                                    <div className="space-y-3">
                                        <div className="flex items-center gap-3">
                                            <div className="font-mono text-slate-700 dark:text-slate-300 font-bold bg-slate-100 dark:bg-slate-950 px-3 py-1.5 rounded-lg border border-slate-200 dark:border-slate-800 text-sm tracking-wide">{key.key}</div>
                                            {key.status === 'valid' && <span className="text-[10px] font-bold bg-green-100 dark:bg-green-500/10 text-green-700 dark:text-green-400 px-2.5 py-1 rounded-full border border-green-200 dark:border-green-500/20 flex items-center gap-1 shadow-sm"><CheckCircle2 size={12} /> Valid</span>}
                                            {key.status === 'invalid' && <span className="text-[10px] font-bold bg-red-100 dark:bg-red-500/10 text-red-700 dark:text-red-400 px-2.5 py-1 rounded-full border border-red-200 dark:border-red-500/20 flex items-center gap-1 shadow-sm"><AlertCircle size={12} /> Invalid</span>}
                                            {key.status === 'validating' && <span className="text-[10px] font-bold bg-blue-100 dark:bg-blue-500/10 text-blue-700 dark:text-blue-400 px-2.5 py-1 rounded-full border border-blue-200 dark:border-blue-500/20 flex items-center gap-1 shadow-sm"><Loader2 size={12} className="animate-spin" /> Checking</span>}

                                            {/* Tier Badge */}
                                            {key.tier === 'FREE' && <span className="text-[10px] font-bold bg-blue-100 dark:bg-blue-900/20 text-blue-700 dark:text-blue-400 px-2.5 py-1 rounded-full border border-blue-200 dark:border-blue-500/20 flex items-center gap-1 shadow-sm">🆓 FREE</span>}
                                            {key.tier === 'PAID' && <span className="text-[10px] font-bold bg-amber-100 dark:bg-amber-900/20 text-amber-700 dark:text-amber-400 px-2.5 py-1 rounded-full border border-amber-200 dark:border-amber-500/20 flex items-center gap-1 shadow-sm">💎 PAID</span>}

                                            {/* Usage Badge */}
                                            {key.usage !== undefined && (
                                                <span className="text-[10px] font-bold bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-400 px-2.5 py-1 rounded-full border border-slate-200 dark:border-slate-700 flex items-center gap-1 shadow-sm" title="Calls made using this key today">
                                                    <Zap size={10} className="text-amber-500" /> Used: {key.usage}
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                    <div className="flex gap-2 items-center">
                                        {/* Enable/Disable Toggle */}
                                        <button
                                            onClick={() => handleToggleKey(key.id)}
                                            className={`w-10 h-5 rounded-full relative transition-colors ${key.isEnabled !== false ? 'bg-green-500' : 'bg-slate-300 dark:bg-slate-700'}`}
                                            title={key.isEnabled !== false ? "Disable Key" : "Enable Key"}
                                        >
                                            <div className={`absolute top-1 left-1 w-3 h-3 bg-white rounded-full transition-transform ${key.isEnabled !== false ? 'translate-x-5' : 'translate-x-0'}`} />
                                        </button>

                                        <button onClick={() => validateKey(key.fullKey, key.id)} className="p-2.5 bg-slate-100 dark:bg-slate-800 rounded-xl hover:text-blue-600 dark:hover:text-blue-500 transition" title="Re-validate"><RefreshCw size={18} /></button>
                                        <button onClick={() => { navigator.clipboard.writeText(key.fullKey); setToast({ type: 'success', msg: "Copied API Key!" }); }} className="p-2.5 bg-slate-100 dark:bg-slate-800 rounded-xl hover:text-slate-900 dark:hover:text-white transition" title="Copy"><Copy size={18} /></button>
                                        <DoubleConfirmButton onConfirm={() => handleDeleteKey(key.id)} className="px-3 py-2.5 bg-red-500/10 dark:bg-red-900/20 rounded-xl text-red-600 dark:text-red-400 hover:bg-red-500/20 dark:hover:bg-red-900/40 transition border border-red-500/20 dark:border-red-500/30 flex items-center gap-2" title="Delete API Key"><Trash2 size={18} /><span className="text-xs font-bold hidden xl:block">Delete</span></DoubleConfirmButton>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                    <div className="lg:col-span-3 bg-[#0a0f1e] border border-slate-800 rounded-3xl overflow-hidden flex flex-col h-[600px] shadow-2xl ring-1 ring-slate-800"><div className="p-4 border-b border-slate-800 bg-[#0f1623] flex justify-between items-center"><span className="text-xs font-bold text-slate-400 flex items-center gap-2 uppercase tracking-wider"><Terminal size={14} /> Validation Log</span><button onClick={() => setLogs([])} className="text-[10px] text-slate-500 hover:text-white uppercase font-bold bg-slate-800/50 px-2 py-1 rounded hover:bg-slate-800 transition">Clear Log</button></div><div className="flex-1 overflow-y-auto p-5 font-mono text-[11px] leading-relaxed space-y-1.5 custom-scrollbar text-slate-300">{logs.length === 0 && (<div className="h-full flex flex-col items-center justify-center text-slate-700 italic"><Activity size={32} className="mb-2 opacity-20" /><p>// Waiting for validation events...</p></div>)}{logs.map(log => (<div key={log.id} className="flex flex-col gap-1 hover:bg-white/5 p-1 rounded mb-2 border-b border-white/5 pb-2 last:border-0"><span className="text-slate-500 font-bold tracking-wide text-[10px] opacity-70">[{log.timestamp}]</span><span className={`${log.type === 'success' ? 'text-emerald-400' : log.type === 'error' ? 'text-red-400' : log.type === 'warning' ? 'text-amber-400' : log.type === 'network' ? 'text-cyan-400' : log.type === 'process' ? 'text-purple-400' : 'text-blue-400'} break-all pl-2 border-l-2 ${log.type === 'success' ? 'border-emerald-500/30' : log.type === 'error' ? 'border-red-500/30' : 'border-slate-700'}`}>{log.type === 'success' ? '✅' : log.type === 'error' ? '❌' : log.type === 'network' ? '📡' : log.type === 'process' ? '🔧' : 'ℹ️'} {log.message}</span></div>))}<div ref={logsEndRef} /></div></div>
                </div>
            )}

            {subTab === 'models' && (
                apiKeys.length === 0 ? (
                    <div className="flex flex-col items-center justify-center p-20 text-center border-2 border-dashed border-slate-200 dark:border-slate-800 rounded-3xl bg-slate-50 dark:bg-slate-900/30 animate-fade-in my-8">
                        <div className="w-20 h-20 bg-slate-200 dark:bg-slate-800 rounded-full flex items-center justify-center mb-6 text-slate-400 dark:text-slate-500">
                            <Lock size={40} />
                        </div>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-2 font-['Space_Grotesk']">Models Configuration Locked</h3>
                        <p className="text-slate-500 dark:text-slate-400 max-w-md mb-8 leading-relaxed font-medium">
                            To access model configuration and credits, you must first add at least one valid Gemini API Key.
                        </p>
                        <button
                            onClick={() => onSubTabChange('keys')}
                            className="px-8 py-3 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl transition shadow-lg shadow-amber-500/20 flex items-center gap-2 transform hover:-translate-y-0.5"
                        >
                            <Key size={18} />
                            Go to API Keys
                        </button>
                    </div>
                ) : (
                    <div className="animate-fade-in">
                        <div className="p-4 mb-8 bg-blue-50 dark:bg-blue-900/10 border border-blue-100 dark:border-blue-900/30 rounded-2xl flex items-start gap-3">
                            <Info className="text-blue-600 dark:text-blue-400 shrink-0 mt-0.5" size={20} />
                            <div>
                                <h4 className="font-bold text-blue-700 dark:text-blue-300 text-sm mb-1">Multi-Model Selection</h4>
                                <p className="text-sm text-blue-600/80 dark:text-blue-400/80">You can select multiple models for each category. If your first choice hits a rate limit or fails, the system will automatically fallback to the next selected model in the list.</p>
                            </div>
                        </div>

                        <div className="mb-8">
                            <div className="flex flex-col md:flex-row md:items-center justify-between gap-4 mb-4">
                                <h3 className="text-lg font-bold text-slate-900 dark:text-white flex items-center gap-2">
                                    <BrainCircuit className="text-blue-500" size={20} /> Thinking Models
                                    <span className="text-xs font-normal text-slate-500 ml-2 bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded">For Metadata & Trends</span>
                                </h3>

                                {/* Save Action Bar - Aligned with Header */}
                                <div className="flex items-center gap-4">
                                    {modelSaved && (
                                        <span className="text-green-600 dark:text-green-400 font-bold text-sm flex items-center gap-1 animate-fade-in">
                                            <CheckCircle2 size={16} /> Saved
                                        </span>
                                    )}
                                    <button
                                        onClick={handleSaveModels}
                                        disabled={isSavingModel}
                                        className="px-6 py-2.5 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl shadow-lg shadow-amber-500/20 transition flex items-center gap-2 transform hover:-translate-y-0.5 active:translate-y-0 disabled:opacity-70 disabled:transform-none text-sm"
                                    >
                                        {isSavingModel ? <Loader2 className="animate-spin" size={16} /> : <Save size={16} />}
                                        {isSavingModel ? 'Saving...' : 'Save Preferences'}
                                    </button>
                                </div>
                            </div>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">{MODELS.filter(m => m.category === 'thinking' && authService.canUserAccessModel(m.id, apiKeys, siteSettings || {})).map(model => (<ModelCard key={model.id} model={model} user={user} selectedModels={selectedThinkingModels} onToggle={toggleThinkingModel} apiKeys={apiKeys} siteSettings={siteSettings} />))}</div>
                        </div>
                        <div className="mb-8">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-white mb-4 flex items-center gap-2"><Palette className="text-pink-500" size={20} /> Creative Models<span className="text-xs font-normal text-slate-500 ml-2 bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded">For Images & Editing</span></h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">{MODELS.filter(m => m.category === 'creative' && authService.canUserAccessModel(m.id, apiKeys, siteSettings || {})).map(model => (<ModelCard key={model.id} model={model} user={user} selectedModels={selectedCreativeModels} onToggle={toggleCreativeModel} apiKeys={apiKeys} siteSettings={siteSettings} />))}</div>
                        </div>
                        <div className="mb-8">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-white mb-4 flex items-center gap-2"><Video className="text-purple-500" size={20} /> Video Models<span className="text-xs font-normal text-slate-500 ml-2 bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded">For Video Generation</span></h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">{MODELS.filter(m => m.category === 'video' && authService.canUserAccessModel(m.id, apiKeys, siteSettings || {})).map(model => (<ModelCard key={model.id} model={model} user={user} selectedModels={selectedVideoModels} onToggle={toggleVideoModel} apiKeys={apiKeys} siteSettings={siteSettings} />))}</div>
                        </div>
                    </div>
                )
            )}

            {subTab === 'settings' && (
                <div className="animate-fade-in max-w-3xl">
                    <div className="mb-6">
                        <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-2 flex items-center gap-2">
                            <Sliders className="text-purple-500" /> Global Model Configuration
                        </h3>
                        <p className="text-slate-600 dark:text-slate-400 text-sm">
                            Set default parameters for all Gemini API calls. These settings control how the AI "thinks" and responds.
                        </p>
                    </div>

                    <div className="space-y-8 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-8 shadow-sm">

                        {/* Temperature */}
                        <div>
                            <div className="flex justify-between items-center mb-2">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 flex items-center gap-2">
                                    <Thermometer size={16} className="text-amber-500" /> Temperature
                                </label>
                                <span className="text-xs font-bold text-slate-900 dark:text-white bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded">
                                    {apiSettings.temperature}
                                </span>
                            </div>
                            <input
                                type="range" min="0" max="1" step="0.1"
                                value={apiSettings.temperature}
                                onChange={e => setApiSettings({ ...apiSettings, temperature: parseFloat(e.target.value) })}
                                className="w-full h-2 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500"
                            />
                            <div className="flex justify-between text-[10px] text-slate-500 mt-1 font-bold uppercase">
                                <span>Precise (0.0)</span>
                                <span>Balanced (0.5)</span>
                                <span>Creative (1.0)</span>
                            </div>
                            <p className="text-xs text-slate-500 dark:text-slate-400 mt-3 leading-relaxed">
                                Controls the randomness of the output. Lower values are more deterministic and precise, while higher values are more creative and varied.
                            </p>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                            {/* Top P */}
                            <div>
                                <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 block">Top P (Nucleus)</label>
                                <input
                                    type="number" step="0.05" min="0" max="1"
                                    value={apiSettings.topP}
                                    onChange={e => setApiSettings({ ...apiSettings, topP: parseFloat(e.target.value) })}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm font-bold text-slate-900 dark:text-white outline-none focus:border-blue-500 transition"
                                />
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-2 leading-relaxed">
                                    Limits the token pool to the top percentage of probability mass. Lower values make text less random.
                                </p>
                            </div>

                            {/* Top K */}
                            <div>
                                <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 block">Top K</label>
                                <input
                                    type="number" step="1" min="1" max="100"
                                    value={apiSettings.topK}
                                    onChange={e => setApiSettings({ ...apiSettings, topK: parseInt(e.target.value) })}
                                    className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm font-bold text-slate-900 dark:text-white outline-none focus:border-blue-500 transition"
                                />
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-2 leading-relaxed">
                                    Limits the token pool to the top K most likely tokens. Lower values reduce the chance of nonsense words.
                                </p>
                            </div>
                        </div>

                        {/* Max Output Tokens */}
                        <div>
                            <div className="flex justify-between items-center mb-2">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 flex items-center gap-2">
                                    <Gauge size={16} className="text-blue-500" /> Max Output Tokens
                                </label>
                                <span className="text-xs font-bold text-slate-900 dark:text-white bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded">
                                    {apiSettings.maxOutputTokens}
                                </span>
                            </div>
                            <input
                                type="range" min="1024" max="32768" step="1024"
                                value={apiSettings.maxOutputTokens}
                                onChange={e => setApiSettings({ ...apiSettings, maxOutputTokens: parseInt(e.target.value) })}
                                className="w-full h-2 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-blue-500"
                            />
                            <p className="text-xs text-slate-500 dark:text-slate-400 mt-2 leading-relaxed">
                                Sets the maximum number of tokens (words/pieces) the model can generate in a single response. Higher limits allow for longer content but may take longer to generate.
                            </p>
                        </div>

                        {/* Safety Filter */}
                        <div className="pt-6 border-t border-slate-200 dark:border-slate-800">
                            <div className="flex items-center justify-between mb-4">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 flex items-center gap-2">
                                    <ShieldAlert size={16} className="text-red-500" /> Safety Filter Strength
                                </label>
                                <select
                                    value={apiSettings.safetyFilter}
                                    onChange={e => setApiSettings({ ...apiSettings, safetyFilter: e.target.value })}
                                    className="bg-slate-100 dark:bg-slate-800 border-none rounded-lg text-xs font-bold px-3 py-1.5 text-slate-900 dark:text-white outline-none cursor-pointer hover:bg-slate-200 dark:hover:bg-slate-700 transition"
                                >
                                    <option value="block_none">Block None (Dangerous)</option>
                                    <option value="block_few">Block Few</option>
                                    <option value="block_some">Block Some (Default)</option>
                                    <option value="block_most">Block Most (Strict)</option>
                                </select>
                            </div>
                            <div className="p-4 bg-slate-50 dark:bg-slate-900 rounded-xl border border-slate-200 dark:border-slate-800 text-xs text-slate-600 dark:text-slate-400 leading-relaxed">
                                <p className="mb-2">Controls how aggressively the AI filters potentially harmful content.</p>
                                <p className="flex items-start gap-2">
                                    <AlertTriangle size={14} className="text-amber-500 shrink-0 mt-0.5" />
                                    <span><strong>Caution:</strong> Setting this to "Block None" disables most safety guardrails and may result in the generation of offensive, harmful, or sexually explicit content. Use responsibly.</span>
                                </p>
                            </div>
                        </div>

                        <div className="flex justify-end pt-2 items-center gap-4">
                            {modelSaved && (
                                <span className="text-green-600 dark:text-green-400 font-bold text-sm flex items-center gap-1 animate-fade-in">
                                    <CheckCircle2 size={16} /> Saved
                                </span>
                            )}
                            <button
                                onClick={handleSaveModels}
                                disabled={isSavingModel}
                                className="px-8 py-3 bg-slate-900 dark:bg-white hover:bg-slate-800 dark:hover:bg-slate-200 text-white dark:text-slate-950 font-bold rounded-xl transition shadow-lg hover:-translate-y-0.5 active:translate-y-0 flex items-center gap-2 disabled:opacity-70"
                            >
                                {isSavingModel ? <Loader2 className="animate-spin" size={18} /> : <Save size={18} />}
                                {isSavingModel ? 'Saving...' : 'Save Configuration'}
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

const SupportTab = ({ currentUser }: { currentUser: User | null }) => {
    const [messages, setMessages] = useState<SupportMessage[]>([]);
    const [input, setInput] = useState('');
    const scrollRef = useRef<HTMLDivElement>(null);
    // Track previous messages to detect new ones
    const [prevMessages, setPrevMessages] = useState<SupportMessage[]>([]);

    useEffect(() => {
        const fetchMessages = async () => {
            if (!currentUser) return;
            const msgs = await adminService.getSupportMessages(currentUser.uid);
            setMessages(msgs);
        };
        fetchMessages();
        const interval = setInterval(fetchMessages, 5000);
        return () => clearInterval(interval);
    }, [currentUser]);

    // Handle notifications for new messages
    useEffect(() => {
        if (messages.length > prevMessages.length && prevMessages.length > 0) {
            const newMsgs = messages.filter(m => !prevMessages.find(pm => pm.id === m.id));
            const newAdminMsg = newMsgs.find(m => m.sender === 'admin');

            if (newAdminMsg) {
                // Play notification sound
                const audio = new Audio('https://assets.mixkit.co/active_storage/sfx/2869/2869-preview.mp3');
                audio.volume = 0.5;
                audio.play().catch(() => { });

                // Dispatch event for toast/notification
                window.dispatchEvent(new CustomEvent('sf_notification', {
                    detail: { type: 'info', message: 'New support message from Admin' }
                }));
            }
        }
        setPrevMessages(messages);
    }, [messages]);

    // Mark admin messages as read when user views the chat
    useEffect(() => {
        const markAdminMessagesRead = async () => {
            if (!currentUser) return;

            // Find all unread admin messages
            const unreadAdminMessages = messages.filter(m => m.sender === 'admin' && m.status !== 'read');

            if (unreadAdminMessages.length === 0) return;

            // Immediately update local state to show messages as read
            setMessages(prev => prev.map(m =>
                m.sender === 'admin' && m.status !== 'read'
                    ? { ...m, status: 'read' as const }
                    : m
            ));

            // Mark each as read in Firestore (in background)
            const markReadPromises = unreadAdminMessages.map(msg =>
                adminService.markMessageAsRead(msg.id)
                    .catch(error => console.error('Failed to mark message as read:', error))
            );

            await Promise.all(markReadPromises);

            // Optimistically update App.tsx state with IDs
            window.dispatchEvent(new CustomEvent('sf_messages_read', { detail: unreadAdminMessages.map(m => m.id) }));

            // Trigger a notification refresh to update badge counts
            window.dispatchEvent(new Event('sf_notifications_updated'));
        };

        markAdminMessagesRead();
    }, [messages, currentUser]); // Run whenever messages change
    useEffect(() => { scrollRef.current?.scrollIntoView({ behavior: 'smooth' }); }, [messages]);
    const handleSend = async () => {
        if (!currentUser || !input.trim()) return;
        const newMessage: SupportMessage = {
            id: Date.now().toString(),
            sender: 'user',
            text: input,
            timestamp: Date.now(),
            status: 'sent',
            userId: currentUser.uid,
            userEmail: currentUser.email,
            userName: currentUser.displayName
        };

        try {
            // Clear input immediately for better UX
            const messageText = input;
            setInput('');

            // Send to API first
            await adminService.sendSupportMessage(newMessage);

            // Only add to local state after successful API call
            // The polling will pick it up anyway, but this makes it instant
            setMessages(prev => [...prev, newMessage]);
        } catch (error) {
            console.error('Failed to send message:', error);
            // Restore input on error
            setInput(newMessage.text);
            alert('Failed to send message. Please try again.');
        }
    };
    return (<div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden flex flex-col h-[600px] shadow-sm"><div className="p-4 border-b border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-950/50 flex justify-between items-center"><h3 className="font-bold text-slate-900 dark:text-white">Support Chat</h3><span className="text-xs text-slate-500">Typical reply time: 2 hours</span></div><div className="flex-1 overflow-y-auto p-4 space-y-4 bg-slate-50 dark:bg-slate-950 custom-scrollbar">{messages.length === 0 && (<div className="text-center text-slate-500 mt-10 text-sm"><p>No messages yet.</p><p>Send a message to start a conversation with support.</p></div>)}{messages.map(msg => (<div key={msg.id} className={`flex flex-col ${msg.sender === 'user' ? 'items-end' : 'items-start'}`}><span className={`text-[10px] font-bold uppercase tracking-wider mb-1 px-2 ${msg.sender === 'user' ? 'text-blue-600 dark:text-blue-400' : 'text-slate-600 dark:text-slate-400'}`}>{msg.sender === 'user' ? (msg.userName || 'You') : 'Admin'}</span><div className={`max-w-[80%] p-3 rounded-2xl text-sm ${msg.sender === 'user' ? 'bg-blue-600 text-white rounded-br-none' : 'bg-white dark:bg-slate-800 text-slate-800 dark:text-slate-200 border border-slate-200 dark:border-slate-700 rounded-bl-none'}`}><p>{msg.text}</p><p className={`text-[10px] mt-1 opacity-70 ${msg.sender === 'user' ? 'text-blue-100' : 'text-slate-400'}`}>{new Date(msg.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}</p></div></div>))}<div ref={scrollRef}></div></div><div className="p-4 bg-white dark:bg-slate-900 border-t border-slate-200 dark:border-slate-800"><div className="flex gap-2"><input type="text" value={input} onChange={e => setInput(e.target.value)} onKeyDown={e => e.key === 'Enter' && handleSend()} className="flex-1 bg-slate-100 dark:bg-slate-800 border-none rounded-xl px-4 py-3 text-sm focus:ring-2 focus:ring-blue-500 outline-none text-slate-900 dark:text-white" placeholder="Type a message..." /><button onClick={handleSend} disabled={!input.trim()} className="p-3 bg-blue-600 hover:bg-blue-500 text-white rounded-xl transition disabled:opacity-50"><Send size={18} /></button></div></div></div>);
};

const AppSettingsTab = ({ user }: { user: User | null }) => {
    const [prefs, setPrefs] = useState<ExtendedUserPreferences>({
        theme: (localStorage.getItem('sf_theme') as any) || 'dark',
        // Default Global Cooldown: 5s (Recommended)
        generationConfig: { globalCooldown: 5, imageDelay: 0, videoDelay: 0, metadataDelay: 0 },
        // UPDATED DEFAULTS: Auto-Download, Sound Effects, and Notifications OFF by default for new users.
        uiConfig: { autoDownload: false, soundEffects: false, notifications: false, reducedMotion: false }
    });
    const [isSaving, setIsSaving] = useState(false);
    const [saved, setSaved] = useState(false);
    const [cacheSize, setCacheSize] = useState('Checking...');

    useEffect(() => {
        const loadPrefs = async () => {
            if (user?.uid) {
                try {
                    const stored = await authService.getUserPreferences(user.uid);
                    setPrefs(prev => ({
                        ...prev,
                        ...stored,
                        generationConfig: { ...prev.generationConfig, ...((stored as any).generationConfig || {}) },
                        uiConfig: { ...prev.uiConfig, ...((stored as any).uiConfig || {}) }
                    }));
                } catch (error) {
                    console.error("Failed to load preferences:", error);
                }
            }
        };
        loadPrefs();
        calculateCacheSize();
    }, [user?.uid]);

    useEffect(() => {
        if (prefs.uiConfig.reducedMotion) {
            document.documentElement.classList.add('motion-reduce');
        } else {
            document.documentElement.classList.remove('motion-reduce');
        }
    }, [prefs.uiConfig.reducedMotion]);

    const calculateCacheSize = () => {
        let bytes = 0;
        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            if (key && (key.startsWith('sf_trends_cache') || key.startsWith('sf_temp') || key.includes('cache') || key.startsWith('sf_daily_stats'))) {
                bytes += (localStorage.getItem(key)?.length || 0) * 2;
            }
        }

        if (bytes === 0) setCacheSize('0 KB');
        else if (bytes < 1024 * 1024) setCacheSize((bytes / 1024).toFixed(1) + ' KB');
        else setCacheSize((bytes / (1024 * 1024)).toFixed(1) + ' MB');
    };

    const [clearConfirm, setClearConfirm] = useState(false);
    const [clearSuccess, setClearSuccess] = useState<string | null>(null);

    const handleClearCache = () => {
        if (!clearConfirm) {
            setClearConfirm(true);
            setTimeout(() => setClearConfirm(false), 3000);
            return;
        }

        let cleared = 0;
        let bytesCleared = 0;
        const keysToRemove: string[] = [];

        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            if (key && (
                key.startsWith('sf_trends_cache') ||
                key.startsWith('sf_temp') ||
                key.includes('cache') ||
                key.startsWith('sf_daily_stats')
            )) {
                keysToRemove.push(key);
            }
        }

        keysToRemove.forEach(key => {
            bytesCleared += (localStorage.getItem(key)?.length || 0) * 2;
            localStorage.removeItem(key);
            cleared++;
        });

        calculateCacheSize();

        const sizeStr = bytesCleared > 1024 * 1024
            ? `${(bytesCleared / (1024 * 1024)).toFixed(2)} MB`
            : `${(bytesCleared / 1024).toFixed(2)} KB`;

        setClearSuccess(`Success! Cleared ${cleared} files (${sizeStr}).`);
        setClearConfirm(false);
        setTimeout(() => setClearSuccess(null), 4000);
    };

    const [resetConfirm, setResetConfirm] = useState(false);
    const [resetCountdown, setResetCountdown] = useState(3);
    const [isResetting, setIsResetting] = useState(false);

    useEffect(() => {
        let timer: NodeJS.Timeout;
        if (resetConfirm && resetCountdown > 0) {
            timer = setTimeout(() => setResetCountdown(prev => prev - 1), 1000);
        } else if (resetCountdown === 0) {
            setResetConfirm(false);
            setResetCountdown(3);
        }
        return () => clearTimeout(timer);
    }, [resetConfirm, resetCountdown]);

    const handleResetStats = async (e: React.MouseEvent) => {
        e.preventDefault();

        if (!resetConfirm) {
            setResetConfirm(true);
            setResetCountdown(3);
            return;
        }

        if (!user?.uid) return;
        setIsResetting(true);

        try {
            await authService.resetUserStats(user.uid);
            setSaved(true);

            // Show Success UI or refresh
            window.location.reload(); // Simplest way to ensure all stats effectively clear from all components
        } catch (error) {
            console.error("Failed to reset stats:", error);
        } finally {
            setIsResetting(false);
            setResetConfirm(false);
            setResetCountdown(3);
            setTimeout(() => setSaved(false), 2000);
        }
    };

    const handleSave = () => {
        if (!user?.uid) return;
        setIsSaving(true);

        // Persist theme
        localStorage.setItem('sf_theme', prefs.theme);

        // Persist all preferences to Mock DB
        authService.updateUserPreferences(user.uid, prefs);

        setTimeout(() => {
            setIsSaving(false);
            setSaved(true);
            setTimeout(() => setSaved(false), 2000);
        }, 600);
    };

    const updateGen = (key: keyof ExtendedUserPreferences['generationConfig'], val: number) => {
        setPrefs(prev => ({ ...prev, generationConfig: { ...prev.generationConfig, [key]: val } }));
    };

    const updateUI = (key: keyof ExtendedUserPreferences['uiConfig'], val: boolean) => {
        const newUiConfig = { ...prefs.uiConfig, [key]: val };
        setPrefs(prev => ({ ...prev, uiConfig: newUiConfig }));

        // Save to backend
        if (user) {
            fetch('http://localhost:3001/api/user/preferences', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    ...authService.getHeaders()
                },
                body: JSON.stringify({
                    uid: user.uid,
                    data: { uiConfig: newUiConfig }
                })
            }).catch(err => console.error('Failed to save uiConfig to backend:', err));
        }

        if (key === 'soundEffects' && val) {
            const audio = new Audio('https://assets.mixkit.co/active_storage/sfx/2568/2568-preview.mp3');
            audio.volume = 0.2;
            audio.play().catch(() => { });
        }
    };

    const setLiveTheme = (theme: 'dark' | 'light') => {
        setPrefs(prev => ({ ...prev, theme }));
        localStorage.setItem('sf_theme', theme);
        if (theme === 'dark') {
            document.documentElement.classList.add('dark');
            document.documentElement.classList.remove('light');
        } else {
            document.documentElement.classList.remove('dark');
            document.documentElement.classList.add('light');
        }
    };

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 animate-fade-in">
            {prefs.uiConfig.reducedMotion && (
                <style>{`
                    .motion-reduce *, .motion-reduce *::before, .motion-reduce *::after {
                        animation-duration: 0.001s !important;
                        animation-iteration-count: 1 !important;
                        transition-duration: 0.001s !important;
                        scroll-behavior: auto !important;
                    }
                `}</style>
            )}

            {/* Visual & Interface Card */}
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 shadow-sm hover:border-slate-300 dark:hover:border-slate-700 transition-colors">
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-3 bg-blue-100 dark:bg-blue-500/10 rounded-xl text-blue-600 dark:text-blue-400">
                        <Monitor size={24} />
                    </div>
                    <div>
                        <h3 className="font-bold text-lg text-slate-900 dark:text-white">Visual Interface</h3>
                        <p className="text-xs text-slate-500">Customize app appearance and accessibility.</p>
                    </div>
                </div>

                <div className="space-y-4">
                    <div>
                        <label className="text-xs font-bold text-slate-500 uppercase mb-3 block">Theme Mode</label>
                        <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-xl">
                            <button
                                onClick={() => setLiveTheme('light')}
                                className={`flex-1 py-2 rounded-lg text-sm font-bold flex items-center justify-center gap-2 transition-all ${prefs.theme === 'light' ? 'bg-white text-slate-900 shadow-md' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                            >
                                <Sun size={16} /> Light
                            </button>
                            <button
                                onClick={() => setLiveTheme('dark')}
                                className={`flex-1 py-2 rounded-lg text-sm font-bold flex items-center justify-center gap-2 transition-all ${prefs.theme === 'dark' ? 'bg-slate-700 text-white shadow-md' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                            >
                                <Moon size={16} /> Dark
                            </button>
                        </div>
                    </div>

                    <div className="flex items-center justify-between p-3 bg-slate-50 dark:bg-slate-950/50 rounded-xl border border-slate-100 dark:border-slate-800">
                        <div>
                            <span className="text-sm font-bold text-slate-700 dark:text-slate-300 flex items-center gap-2 mb-1">
                                <ZapOff size={16} className="text-slate-400" /> Reduced Motion
                            </span>
                            <p className="text-xs text-slate-500 pl-6">Disables all animations and transitions.</p>
                        </div>
                        <button onClick={() => updateUI('reducedMotion', !prefs.uiConfig.reducedMotion)} className={`w-10 h-5 rounded-full relative transition-colors ${prefs.uiConfig.reducedMotion ? 'bg-amber-500' : 'bg-slate-300 dark:bg-slate-700'}`}>
                            <div className={`absolute top-1 left-1 w-3 h-3 bg-white rounded-full transition-transform ${prefs.uiConfig.reducedMotion ? 'translate-x-5' : 'translate-x-0'}`} />
                        </button>
                    </div>
                </div>
            </div>

            {/* Workflow Automation Card */}
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 shadow-sm hover:border-slate-300 dark:hover:border-slate-700 transition-colors">
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-3 bg-amber-100 dark:bg-amber-500/10 rounded-xl text-amber-600 dark:text-amber-400">
                        <Clock size={24} />
                    </div>
                    <div>
                        <h3 className="font-bold text-lg text-slate-900 dark:text-white">Generation Flow</h3>
                        <p className="text-xs text-slate-500">Manage pacing and throttle generation requests.</p>
                    </div>
                </div>

                <div className="space-y-5">
                    <div>
                        <div className="flex justify-between mb-1">
                            <label className="text-xs font-bold text-slate-500 uppercase flex items-center gap-2">Global Cooldown</label>
                            <span className={`text-xs font-mono font-bold ${prefs.generationConfig.globalCooldown < 5 ? 'text-red-500' : 'text-amber-600 dark:text-amber-400'}`}>{prefs.generationConfig.globalCooldown}s</span>
                        </div>
                        <input type="range" min="0" max="60" step="1" value={prefs.generationConfig.globalCooldown} onChange={(e) => updateGen('globalCooldown', parseInt(e.target.value))} className={`w-full h-2 rounded-lg appearance-none cursor-pointer ${prefs.generationConfig.globalCooldown < 5 ? 'bg-red-200 dark:bg-red-900/30 accent-red-500' : 'bg-slate-200 dark:bg-slate-700 accent-amber-500'}`} />
                        <div className="relative w-full h-5 mt-1 text-[10px] font-mono text-slate-400">
                            <span className="absolute left-0">0s</span>
                            {/* Position 5s at approx 8.33% (5/60). Using left-[8.33%] and pulling back slightly to align '5s' with thumb */}
                            <span className="absolute left-[8.33%] -translate-x-[10%] text-green-600 dark:text-green-400 font-bold flex items-center gap-1 whitespace-nowrap transform">
                                <span className="flex flex-col items-center">
                                    <span>5s</span>
                                    {/* Tick mark helper if needed, but text is enough */}
                                </span>
                                <span className="text-[9px] bg-green-100 dark:bg-green-500/10 px-1.5 py-0.5 rounded uppercase tracking-wider">Recommended</span>
                            </span>
                            <span className="absolute right-0">60s</span>
                        </div>
                        {prefs.generationConfig.globalCooldown < 5 && (
                            <p className="text-[10px] text-red-500 font-bold mt-2 flex items-center gap-1 animate-fade-in">
                                <AlertTriangle size={12} />
                                Warning: Intervals lower than 5s may cause 429 Rate Limit errors.
                            </p>
                        )}
                    </div>

                    <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
                        {/* Image Delay */}
                        <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800 text-center group hover:border-purple-500/30 transition-colors">
                            <Camera size={24} className="mx-auto mb-2 text-purple-500" />
                            <p className="text-[10px] font-bold text-slate-500 uppercase mb-3">Image Delay</p>
                            <div className="text-2xl font-bold text-slate-900 dark:text-white mb-1 font-mono">
                                {prefs.generationConfig.imageDelay}<span className="text-xs text-slate-400 ml-1">s</span>
                            </div>
                            <input
                                type="range"
                                min="0"
                                max="60"
                                step="1"
                                value={prefs.generationConfig.imageDelay}
                                onChange={e => updateGen('imageDelay', parseInt(e.target.value))}
                                className="w-full h-1.5 bg-slate-200 dark:bg-slate-800 rounded-lg appearance-none cursor-pointer accent-purple-500"
                            />
                        </div>

                        {/* Video Delay */}
                        <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800 text-center group hover:border-blue-500/30 transition-colors">
                            <Video size={24} className="mx-auto mb-2 text-blue-500" />
                            <p className="text-[10px] font-bold text-slate-500 uppercase mb-3">Video Delay</p>
                            <div className="text-2xl font-bold text-slate-900 dark:text-white mb-1 font-mono">
                                {prefs.generationConfig.videoDelay}<span className="text-xs text-slate-400 ml-1">s</span>
                            </div>
                            <input
                                type="range"
                                min="0"
                                max="60"
                                step="1"
                                value={prefs.generationConfig.videoDelay}
                                onChange={e => updateGen('videoDelay', parseInt(e.target.value))}
                                className="w-full h-1.5 bg-slate-200 dark:bg-slate-800 rounded-lg appearance-none cursor-pointer accent-blue-500"
                            />
                        </div>

                        {/* Data Delay */}
                        <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800 text-center group hover:border-emerald-500/30 transition-colors">
                            <BrainCircuit size={24} className="mx-auto mb-2 text-emerald-500" />
                            <p className="text-[10px] font-bold text-slate-500 uppercase mb-3">Data Delay</p>
                            <div className="text-2xl font-bold text-slate-900 dark:text-white mb-1 font-mono">
                                {prefs.generationConfig.metadataDelay}<span className="text-xs text-slate-400 ml-1">s</span>
                            </div>
                            <input
                                type="range"
                                min="0"
                                max="60"
                                step="1"
                                value={prefs.generationConfig.metadataDelay}
                                onChange={e => updateGen('metadataDelay', parseInt(e.target.value))}
                                className="w-full h-1.5 bg-slate-200 dark:bg-slate-800 rounded-lg appearance-none cursor-pointer accent-emerald-500"
                            />
                        </div>
                    </div>
                </div>
            </div>

            {/* Automation & Notifications */}
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 shadow-sm hover:border-slate-300 dark:hover:border-slate-700 transition-colors">
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-3 bg-green-100 dark:bg-green-500/10 rounded-xl text-green-600 dark:text-green-400">
                        <Zap size={24} />
                    </div>
                    <div>
                        <h3 className="font-bold text-lg text-slate-900 dark:text-white">Automation</h3>
                        <p className="text-xs text-slate-500">Streamline your workflow with auto-actions.</p>
                    </div>
                </div>

                <div className="space-y-3">
                    <div className="flex items-center justify-between p-3 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-100 dark:border-slate-800 transition-colors hover:border-slate-200 dark:hover:border-slate-700">
                        <div className="flex items-center gap-3">
                            <Download size={18} className="text-slate-400" />
                            <div>
                                <p className="text-sm font-bold text-slate-800 dark:text-slate-200">Auto-Download</p>
                                <p className="text-[10px] text-slate-500">Save assets immediately upon generation</p>
                            </div>
                        </div>
                        <button onClick={() => updateUI('autoDownload', !prefs.uiConfig.autoDownload)} className={`w-10 h-5 rounded-full relative transition-colors ${prefs.uiConfig.autoDownload ? 'bg-green-500' : 'bg-slate-300 dark:bg-slate-700'}`}>
                            <div className={`absolute top-1 left-1 w-3 h-3 bg-white rounded-full transition-transform ${prefs.uiConfig.autoDownload ? 'translate-x-5' : 'translate-x-0'}`} />
                        </button>
                    </div>

                    {/* Sound Effects */}
                    <div className="flex items-center justify-between p-4 bg-slate-50 dark:bg-slate-900 rounded-xl border border-slate-200 dark:border-slate-800">
                        <div className="flex items-center gap-3">
                            <div className="p-2 bg-purple-100 dark:bg-purple-900/30 text-purple-600 dark:text-purple-400 rounded-lg">
                                <Volume2 size={18} />
                            </div>
                            <div>
                                <h4 className="font-bold text-slate-900 dark:text-white text-sm">Sound Effects</h4>
                                <p className="text-xs text-slate-500">Play sound on completion or error</p>
                            </div>
                        </div>
                        <Toggle
                            checked={prefs.uiConfig.soundEffects}
                            onChange={() => setPrefs({ ...prefs, uiConfig: { ...prefs.uiConfig, soundEffects: !prefs.uiConfig.soundEffects } })}
                        />
                    </div>

                    {/* Notification Sound */}
                    <div className="flex items-center justify-between p-4 bg-slate-50 dark:bg-slate-900 rounded-xl border border-slate-200 dark:border-slate-800">
                        <div className="flex items-center gap-3">
                            <div className="p-2 bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 rounded-lg">
                                <Bell size={18} />
                            </div>
                            <div>
                                <h4 className="font-bold text-slate-900 dark:text-white text-sm">Notification Sound</h4>
                                <p className="text-xs text-slate-500">Play sound when new notifications arrive</p>
                            </div>
                        </div>
                        <Toggle
                            checked={prefs.uiConfig.notifications} // Reusing 'notifications' flag for sound, or should I add a specific one? 
                            // User asked for "ENABLE OR DISABLE in USER APP SETTINGS PAGE". 
                            // Existing 'notifications' flag likely controls the toast alerts.
                            // I will use a new property if possible, but 'notifications' is already there.
                            // Let's check the type definition.
                            // ExtendedUserPreferences has uiConfig: { ... notifications: boolean }.
                            // I'll assume 'notifications' controls the sound too, OR add a specific 'notificationSound' if I can edit the type.
                            // For now, I'll use 'notifications' as the master switch for notification sounds.
                            onChange={() => setPrefs({ ...prefs, uiConfig: { ...prefs.uiConfig, notifications: !prefs.uiConfig.notifications } })}
                        />
                    </div>

                    {/* Notifications (Toasts) - Renaming to "Toast Alerts" to distinguish? Or just keep as is. */}
                    {/* Actually, let's check the existing code. It had "Sound Effects" and "Notifications". */}
                    {/* I'll modify the existing "Notifications" toggle to be "Toast Alerts" and add "Notification Sound" separately if I can, or just use one. */}
                    {/* User said "make sound of notifications... ENABLE OR DISABLE". */}
                    {/* I will add a specific toggle for the sound. I'll need to cast it if the type doesn't exist, or just use the existing 'notifications' for sound and 'soundEffects' for app sounds. */}
                    {/* Better: I'll use `prefs.uiConfig.notifications` for the sound. */}
                    {/* I will use the `notifications` toggle to control the notification sound. */}
                    {/* AND I will rename the label to "Notification Sound" to be clear, or add a sub-option. */}
                    {/* Let's just add a specific "Notification Sound" toggle and store it in local storage directly if not in type. */}
                    {/* Wait, I can't easily change the type in a `replace` call without seeing `types.ts`. */}
                    {/* I will use `soundEffects` for general app sounds and `notifications` for the notification sound. */}
                    {/* Actually, `soundEffects` is usually for clicks/actions. `notifications` is for incoming messages. */}
                    {/* I will use the `notifications` toggle to control the notification sound. */}
                    {/* AND I will rename the label to "Notification Sound" to be clear, or add a sub-option. */}
                    {/* Let's just add a specific "Notification Sound" toggle and store it in local storage directly if not in type. */}
                    {/* Better: I'll use `prefs.uiConfig.notifications` for the sound. */}
                </div>
            </div>

            {/* System Actions */}
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 shadow-sm hover:border-slate-300 dark:hover:border-slate-700 transition-colors">
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-3 bg-red-100 dark:bg-red-500/10 rounded-xl text-red-600 dark:text-red-400">
                        <HardDrive size={24} />
                    </div>
                    <div>
                        <h3 className="font-bold text-lg text-slate-900 dark:text-white">System & Data</h3>
                        <p className="text-xs text-slate-500">Manage local data and reset configurations.</p>
                    </div>
                </div>

                <div className="space-y-3">
                    <button onClick={handleClearCache} className="w-full py-3 px-4 rounded-xl bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 hover:bg-slate-100 dark:hover:bg-slate-900 text-slate-700 dark:text-slate-300 font-bold text-sm transition flex items-center justify-between group">
                        <span className="flex items-center gap-2"><RefreshCcw size={16} className="group-hover:rotate-180 transition-transform duration-500" /> Clear Local Cache</span>
                        <span className="text-xs bg-slate-200 dark:bg-slate-800 px-2 py-0.5 rounded">{cacheSize}</span>
                    </button>

                    <button
                        type="button"
                        onClick={handleResetStats}
                        disabled={isResetting}
                        className={`w-full py-3 px-4 rounded-xl font-bold text-sm transition flex items-center gap-2 group ${resetConfirm
                            ? 'bg-red-500 text-white hover:bg-red-600 border border-red-600 animate-pulse'
                            : 'bg-red-50 dark:bg-red-900/10 border border-red-100 dark:border-red-900/30 hover:bg-red-100 dark:hover:bg-red-900/20 text-red-600 dark:text-red-400'
                            }`}
                    >
                        {isResetting ? <Loader2 className="animate-spin" size={16} /> : <AlertTriangle size={16} />}
                        {resetConfirm ? `Confirm Reset Stats (${resetCountdown}s)` : 'Reset All Stats'}
                    </button>
                </div>
            </div>

            {/* Save Action Bar */}
            <div className="md:col-span-2 flex justify-end pt-4 border-t border-slate-200 dark:border-slate-800 mt-2">
                <div className="flex items-center gap-4">
                    {saved && (
                        <span className="text-green-600 dark:text-green-400 font-bold text-sm flex items-center gap-1 animate-fade-in">
                            <CheckCircle2 size={18} /> Settings Saved
                        </span>
                    )}
                    <button
                        onClick={handleSave}
                        disabled={isSaving}
                        className="px-10 py-4 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-2xl shadow-lg shadow-amber-500/20 transition-all transform hover:-translate-y-1 active:translate-y-0 flex items-center gap-2 disabled:opacity-70 disabled:transform-none"
                    >
                        {isSaving ? <Loader2 className="animate-spin" size={20} /> : <Save size={20} />}
                        {isSaving ? 'Saving...' : ' Save Configuration'}
                    </button>
                </div>
            </div>
        </div>
    );
};

const DocumentationTab = () => {
    const [activeSection, setActiveSection] = useState('getting_started');

    const sections = [
        {
            id: 'getting_started',
            title: 'Getting Started',
            icon: Rocket,
            color: 'text-blue-500',
            bg: 'bg-blue-500/10',
            border: 'border-blue-500/20',
            content: (
                <div className="space-y-6">
                    <div className="bg-gradient-to-br from-blue-600 to-indigo-700 rounded-2xl p-8 text-white shadow-xl relative overflow-hidden">
                        <div className="absolute top-0 right-0 w-64 h-64 bg-white/10 rounded-full blur-3xl -translate-y-1/2 translate-x-1/2"></div>
                        <div className="relative z-10">
                            <h3 className="text-2xl font-bold mb-4">Welcome to StockForge AI</h3>
                            <p className="text-blue-100 mb-6 max-w-xl leading-relaxed">
                                Your all-in-one toolkit for stock photography dominance. Generate metadata, create visuals, analyze trends, and streamline your workflow with the power of advanced AI.
                            </p>
                            <div className="flex flex-wrap gap-4">
                                <a href="https://aistudio.google.com/app/apikey" target="_blank" rel="noreferrer" className="px-6 py-3 bg-white text-blue-700 font-bold rounded-xl shadow-lg hover:bg-blue-50 transition flex items-center gap-2">
                                    <Key size={18} /> Get Free API Key
                                </a>
                                <button onClick={() => document.getElementById('api-setup')?.scrollIntoView({ behavior: 'smooth' })} className="px-6 py-3 bg-blue-800/30 text-white font-bold rounded-xl border border-white/20 hover:bg-blue-800/50 transition flex items-center gap-2">
                                    <ArrowRight size={18} /> Setup Guide
                                </button>
                            </div>
                        </div>
                    </div>

                    <div id="api-setup" className="scroll-mt-20">
                        <h4 className="text-xl font-bold text-slate-900 dark:text-white mb-4 flex items-center gap-2">
                            <span className="flex items-center justify-center w-8 h-8 rounded-full bg-slate-200 dark:bg-slate-800 text-sm font-bold">1</span>
                            Setting Up Your API Key
                        </h4>
                        <div className="bg-slate-50 dark:bg-slate-900/50 rounded-2xl p-6 border border-slate-200 dark:border-slate-800">
                            <ol className="space-y-4 text-sm text-slate-600 dark:text-slate-400">
                                <li className="flex gap-4">
                                    <div className="shrink-0 mt-1"><CheckCircle2 size={18} className="text-green-500" /></div>
                                    <div>
                                        <strong className="text-slate-900 dark:text-slate-200 block mb-1">Get your Key</strong>
                                        Visit Google AI Studio to generate a free API key. This key powers all the AI models in StockForge.
                                    </div>
                                </li>
                                <li className="flex gap-4">
                                    <div className="shrink-0 mt-1"><CheckCircle2 size={18} className="text-green-500" /></div>
                                    <div>
                                        <strong className="text-slate-900 dark:text-slate-200 block mb-1">Add to Settings</strong>
                                        Go to the <strong>Gemini API</strong> tab in Settings, paste your key, and click "Add Key".
                                    </div>
                                </li>
                                <li className="flex gap-4">
                                    <div className="shrink-0 mt-1"><CheckCircle2 size={18} className="text-green-500" /></div>
                                    <div>
                                        <strong className="text-slate-900 dark:text-slate-200 block mb-1">Automatic Validation</strong>
                                        The system will automatically test your key and determine your tier (Free vs Paid) to optimize limits.
                                    </div>
                                </li>
                            </ol>
                        </div>
                    </div>
                </div>
            )
        },
        {
            id: 'metadata',
            title: 'Metadata Generator',
            icon: FileText,
            color: 'text-amber-500',
            bg: 'bg-amber-500/10',
            border: 'border-amber-500/20',
            content: (
                <div className="space-y-8">
                    <div>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-4">Mastering Metadata</h3>
                        <p className="text-slate-600 dark:text-slate-400 mb-6 leading-relaxed">
                            Generate SEO-optimized titles, descriptions, and keywords for your stock photos in seconds. StockForge uses vision AI to analyze your images and produce relevant, high-ranking metadata.
                        </p>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                            <div className="bg-white dark:bg-slate-900 p-5 rounded-2xl border border-slate-200 dark:border-slate-800 shadow-sm">
                                <h4 className="font-bold text-slate-900 dark:text-white mb-2 flex items-center gap-2">
                                    <Upload size={18} className="text-blue-500" /> Single Upload
                                </h4>
                                <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                    Perfect for fine-tuning individual images. Drag & drop an image, review the analysis, and generate metadata. You can manually edit any field before exporting.
                                </p>
                            </div>
                            <div className="bg-white dark:bg-slate-900 p-5 rounded-2xl border border-slate-200 dark:border-slate-800 shadow-sm">
                                <h4 className="font-bold text-slate-900 dark:text-white mb-2 flex items-center gap-2">
                                    <Layers size={18} className="text-purple-500" /> Bulk Processing
                                </h4>
                                <p className="text-xs text-slate-500 dark:text-slate-400 leading-relaxed">
                                    Process hundreds of images at once. Use the "Bulk" tab to queue up folders of images. The AI will process them in the background while you work on other things.
                                </p>
                            </div>
                        </div>

                        <div className="bg-amber-50 dark:bg-amber-900/10 border border-amber-200 dark:border-amber-500/20 rounded-xl p-6">
                            <h4 className="font-bold text-amber-800 dark:text-amber-400 mb-3 flex items-center gap-2">
                                <Sparkles size={18} /> Pro Tip: Custom Instructions
                            </h4>
                            <p className="text-sm text-amber-700/80 dark:text-amber-300/80 leading-relaxed">
                                In the Metadata Generator, use the <strong>"Custom Prompt"</strong> toggle to give specific instructions to the AI. For example: <em>"Focus on the emotional aspect of the image"</em> or <em>"Include technical photography terms in keywords"</em>.
                            </p>
                        </div>
                    </div>
                </div>
            )
        },
        {
            id: 'creation',
            title: 'Content Creation',
            icon: Palette,
            color: 'text-purple-500',
            bg: 'bg-purple-500/10',
            border: 'border-purple-500/20',
            content: (
                <div className="space-y-8">
                    <div>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-4">AI Creation Tools</h3>
                        <div className="grid grid-cols-1 gap-6">
                            <div className="flex gap-4 items-start p-6 bg-slate-50 dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 hover:border-purple-500/30 transition-colors">
                                <div className="p-3 bg-purple-100 dark:bg-purple-500/20 text-purple-600 dark:text-purple-400 rounded-xl shrink-0">
                                    <ImageIcon size={24} />
                                </div>
                                <div>
                                    <h4 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Image Generator</h4>
                                    <p className="text-sm text-slate-600 dark:text-slate-400 leading-relaxed mb-3">
                                        Create stunning stock-ready images using models like Imagen 3 and Gemini.
                                    </p>
                                    <ul className="space-y-1 text-xs text-slate-500 dark:text-slate-500">
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-purple-500 rounded-full"></div> Select aspect ratios (16:9, 1:1, 4:5)</li>
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-purple-500 rounded-full"></div> Choose styles (Photorealistic, Cinematic, 3D)</li>
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-purple-500 rounded-full"></div> Auto-enhance prompts for better results</li>
                                    </ul>
                                </div>
                            </div>

                            <div className="flex gap-4 items-start p-6 bg-slate-50 dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 hover:border-blue-500/30 transition-colors">
                                <div className="p-3 bg-blue-100 dark:bg-blue-500/20 text-blue-600 dark:text-blue-400 rounded-xl shrink-0">
                                    <Video size={24} />
                                </div>
                                <div>
                                    <h4 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Video Generator</h4>
                                    <p className="text-sm text-slate-600 dark:text-slate-400 leading-relaxed mb-3">
                                        Turn text or images into high-quality video clips using Google's Veo models.
                                    </p>
                                    <ul className="space-y-1 text-xs text-slate-500 dark:text-slate-500">
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-blue-500 rounded-full"></div> Text-to-Video: Describe a scene to animate it</li>
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-blue-500 rounded-full"></div> Image-to-Video: Bring static photos to life</li>
                                        <li className="flex items-center gap-2"><div className="w-1 h-1 bg-blue-500 rounded-full"></div> 1080p resolution support</li>
                                    </ul>
                                </div>
                            </div>

                            <div className="flex gap-4 items-start p-6 bg-slate-50 dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 hover:border-emerald-500/30 transition-colors">
                                <div className="p-3 bg-emerald-100 dark:bg-emerald-500/20 text-emerald-600 dark:text-emerald-400 rounded-xl shrink-0">
                                    <Wand2 size={24} />
                                </div>
                                <div>
                                    <h4 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Auto Prompt Gen</h4>
                                    <p className="text-sm text-slate-600 dark:text-slate-400 leading-relaxed mb-3">
                                        Never struggle with writer's block. Enter a simple concept like "business meeting", and the AI will generate comprehensive, detailed prompts optimized for image generation models.
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )
        },
        {
            id: 'tools',
            title: 'Market & Review',
            icon: BarChart2,
            color: 'text-pink-500',
            bg: 'bg-pink-500/10',
            border: 'border-pink-500/20',
            content: (
                <div className="space-y-8">
                    <div>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-6">Strategy & Quality Control</h3>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div className="group relative bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 overflow-hidden hover:shadow-lg transition-all">
                                <div className="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                                    <TrendingUp size={100} className="text-pink-500" />
                                </div>
                                <h4 className="font-bold text-lg text-slate-900 dark:text-white mb-2 relative z-10">Market Intelligence</h4>
                                <p className="text-sm text-slate-600 dark:text-slate-400 mb-4 relative z-10 leading-relaxed h-20">
                                    Make data-driven decisions. Support for analyzing current stock photography trends, keyword popularity, and niche opportunities.
                                </p>
                                <div className="flex flex-wrap gap-2 relative z-10">
                                    <span className="px-2 py-1 bg-pink-50 dark:bg-pink-900/30 text-pink-600 dark:text-pink-400 text-[10px] uppercase font-bold rounded">Trend Reports</span>
                                    <span className="px-2 py-1 bg-pink-50 dark:bg-pink-900/30 text-pink-600 dark:text-pink-400 text-[10px] uppercase font-bold rounded">Keyword Analysis</span>
                                </div>
                            </div>

                            <div className="group relative bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-6 overflow-hidden hover:shadow-lg transition-all">
                                <div className="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                                    <CheckCircle2 size={100} className="text-emerald-500" />
                                </div>
                                <h4 className="font-bold text-lg text-slate-900 dark:text-white mb-2 relative z-10">Reviewer AI</h4>
                                <p className="text-sm text-slate-600 dark:text-slate-400 mb-4 relative z-10 leading-relaxed h-20">
                                    Pre-screen your images before uploading. The AI Reviewer simulates a stock agency inspector, checking for noise, focus issues, and trademark violations.
                                </p>
                                <div className="flex flex-wrap gap-2 relative z-10">
                                    <span className="px-2 py-1 bg-emerald-50 dark:bg-emerald-900/30 text-emerald-600 dark:text-emerald-400 text-[10px] uppercase font-bold rounded">Technical check</span>
                                    <span className="px-2 py-1 bg-emerald-50 dark:bg-emerald-900/30 text-emerald-600 dark:text-emerald-400 text-[10px] uppercase font-bold rounded">Rejection Reasons</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )
        },
        {
            id: 'support',
            title: 'Help & Support',
            icon: HelpCircle,
            color: 'text-cyan-500',
            bg: 'bg-cyan-500/10',
            border: 'border-cyan-500/20',
            content: (
                <div className="space-y-6">
                    <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-4">Troubleshooting & Support</h3>

                    <div className="space-y-4">
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-5 cursor-pointer hover:border-cyan-500/50 transition">
                            <h5 className="font-bold text-slate-800 dark:text-slate-200 mb-2 flex items-center gap-2">
                                <AlertTriangle size={16} className="text-amber-500" /> Why am I getting "429 Rate Limit" errors?
                            </h5>
                            <p className="text-sm text-slate-600 dark:text-slate-400 leading-relaxed">
                                This means you are sending too many requests too quickly for your current API tier. To fix this:
                                <br />1. Add more API keys (the app rotates them automatically).
                                <br />2. Increase the "Global Cooldown" in App Settings.
                                <br />3. Switch to a paid tier key if producing in high volume.
                            </p>
                        </div>

                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-5 cursor-pointer hover:border-cyan-500/50 transition">
                            <h5 className="font-bold text-slate-800 dark:text-slate-200 mb-2 flex items-center gap-2">
                                <Monitor size={16} className="text-blue-500" /> The app is running slow?
                            </h5>
                            <p className="text-sm text-slate-600 dark:text-slate-400 leading-relaxed">
                                StockForge stores data locally in your browser. If you have generated thousands of items, it might slow down. Go to <strong>Settings &gt; App Settings</strong> and use "Clear Local Cache" to free up memory.
                            </p>
                        </div>
                    </div>

                    <div className="bg-slate-100 dark:bg-slate-800 rounded-2xl p-6 text-center mt-8">
                        <p className="text-slate-500 dark:text-slate-400 font-medium mb-4">Still need help?</p>
                        <button onClick={() => window.dispatchEvent(new CustomEvent('sf_switch_tab', { detail: 'support' }))} className="px-8 py-3 bg-slate-900 dark:bg-white text-white dark:text-slate-950 font-bold rounded-xl shadow-lg hover:-translate-y-1 transition transform">
                            Contact Support
                        </button>
                    </div>
                </div>
            )
        }
    ];

    return (
        <div className="max-w-[1600px] mx-auto min-h-[800px] animate-fade-in">
            <div className="grid grid-cols-1 xl:grid-cols-4 gap-8">
                {/* Navigation Sidebar */}
                <div className="xl:col-span-1 space-y-3">
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl p-4 shadow-sm sticky top-4">
                        <h3 className="px-4 py-2 text-xs font-bold text-slate-500 uppercase tracking-wider mb-2">Documentation</h3>
                        {sections.map(section => (
                            <button
                                key={section.id}
                                onClick={() => setActiveSection(section.id)}
                                className={`w-full flex items-center gap-3 px-4 py-3 rounded-xl text-left font-bold transition-all duration-300 ${activeSection === section.id
                                    ? `${section.bg} ${section.color} shadow-sm ring-1 ring-inset ${section.border}`
                                    : 'text-slate-600 dark:text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-800'
                                    }`}
                            >
                                <section.icon size={20} className={activeSection === section.id ? 'scale-110' : ''} />
                                {section.title}
                                {activeSection === section.id && <ChevronRight size={16} className="ml-auto opacity-50" />}
                            </button>
                        ))}

                        <div className="mt-8 px-4 pb-4">
                            <div className="p-4 bg-gradient-to-br from-indigo-500 to-purple-600 rounded-2xl text-white shadow-lg text-center">
                                <BookOpen size={24} className="mx-auto mb-2 opacity-80" />
                                <p className="text-xs font-bold opacity-90 mb-2">Need personal help?</p>
                                <p className="text-[10px] opacity-70 leading-tight">Our support team is available 24/7 for Pro users.</p>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Content Area */}
                <div className="xl:col-span-3">
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-[2.5rem] p-8 min-h-[600px] shadow-sm relative overflow-hidden">
                        {/* Background Decoration */}
                        <div className="absolute top-0 right-0 w-[500px] h-[500px] bg-slate-50 dark:bg-slate-800/20 rounded-full blur-3xl -translate-y-1/2 translate-x-1/2 pointer-events-none"></div>

                        <div key={activeSection} className="relative z-10 animate-fade-in">
                            {sections.find(s => s.id === activeSection)?.content}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

const BillingTab = () => {
    return (<div className="max-w-3xl space-y-6"><div className="bg-gradient-to-r from-amber-500 to-orange-600 rounded-2xl p-8 text-white shadow-lg"><div className="flex justify-between items-start"><div><p className="text-white/80 font-bold uppercase tracking-wider text-sm mb-1">Current Plan</p><h2 className="text-3xl font-bold mb-4">Pro Creator</h2><p className="text-white/90 text-sm max-w-md">You have access to all advanced features.</p></div><div className="bg-white/20 backdrop-blur rounded-xl p-4"><CreditCard size={32} className="text-white mb-2" /><p className="font-mono text-sm">•••• 4242</p></div></div></div></div>);
};

const ReportBugTab = ({ user }: { user: User | null }) => {
    const [title, setTitle] = useState('');
    const [desc, setDesc] = useState('');
    const [steps, setSteps] = useState('');
    const [category, setCategory] = useState('UI');
    const [severity, setSeverity] = useState<'low' | 'medium' | 'high' | 'critical'>('low');
    const [submitted, setSubmitted] = useState(false);
    const [systemInfo, setSystemInfo] = useState({ browser: '', os: '' });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [submitError, setSubmitError] = useState<string | null>(null);

    useEffect(() => {
        // Simple detection
        const browser = navigator.userAgent.includes("Chrome") ? "Chrome" : navigator.userAgent.includes("Firefox") ? "Firefox" : navigator.userAgent.includes("Safari") ? "Safari" : "Other";
        const os = navigator.userAgent.includes("Mac") ? "MacOS" : navigator.userAgent.includes("Win") ? "Windows" : "Linux/Other";
        setSystemInfo({ browser, os });
    }, []);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!user) {
            alert("You must be logged in to submit a bug report.");
            return;
        }
        if (!title || !desc) {
            setSubmitError("Please fill in the issue summary and detailed description.");
            return;
        }

        setIsSubmitting(true);
        setSubmitError(null); // Clear previous errors

        try {
            const fullDesc = `User Agent: ${navigator.userAgent}\nBrowser: ${systemInfo.browser}\nOS: ${systemInfo.os}\n\n${desc}`;

            await adminService.submitBugReport({
                userId: user.uid,
                userEmail: user.email,
                title,
                description: fullDesc,
                steps,
                severity,
                category
            });
            setTitle('');
            setDesc('');
            setSteps('');
            setSeverity('low');
            setCategory('Other');
            setSubmitted(true);
        } catch (error: any) {
            console.error("Error submitting bug report:", error);
            setSubmitError(`Failed to submit: ${error.message || 'Unknown error'}`);
        } finally {
            setIsSubmitting(false);
        }
    };


    if (submitted) return (
        <div className="flex flex-col items-center justify-center p-16 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-3xl shadow-sm animate-fade-in">
            <div className="w-20 h-20 bg-green-100 dark:bg-green-900/30 text-green-600 dark:text-green-400 rounded-full flex items-center justify-center mb-6 shadow-lg">
                <CheckCircle2 size={40} />
            </div>
            <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-2">Report Submitted</h3>
            <p className="text-slate-500 dark:text-slate-400 mb-8 text-center">Thank you for helping us improve StockForge.</p>
            <button onClick={() => { setSubmitted(false); setTitle(''); setDesc(''); setSteps(''); }} className="px-8 py-3 bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 font-bold rounded-xl hover:bg-slate-200 dark:hover:bg-slate-700 transition">
                Submit Another
            </button>
        </div>
    );

    return (
        <div className="max-w-4xl mx-auto">
            <div className="flex items-center gap-4 mb-8">
                <div className="p-3 bg-red-100 dark:bg-red-900/20 rounded-2xl text-red-600 dark:text-red-400">
                    <Bug size={32} />
                </div>
                <div>
                    <h2 className="text-2xl font-bold text-slate-900 dark:text-white">Report a Bug</h2>
                    <p className="text-slate-600 dark:text-slate-400 text-sm">Found an issue? Let our engineering team know.</p>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div className="lg:col-span-2 space-y-6">
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm">
                        <div className="space-y-6">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Issue Summary</label>
                                <input
                                    type="text"
                                    value={title} onChange={e => setTitle(e.target.value)}
                                    className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3.5 text-sm outline-none focus:border-red-500 focus:ring-1 focus:ring-red-500 transition-all"
                                    placeholder="Briefly describe the bug..."
                                />
                            </div>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Category</label>
                                <select
                                    value={category}
                                    onChange={e => setCategory(e.target.value)}
                                    className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3.5 text-sm outline-none focus:border-red-500 transition-all"
                                >
                                    <option value="UI">User Interface</option>
                                    <option value="Functional">Functional Issue</option>
                                    <option value="Performance">Performance / Speed</option>
                                    <option value="Security">Security</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Severity Level</label>
                                <div className="grid grid-cols-4 gap-2">
                                    {['low', 'medium', 'high', 'critical'].map(lev => (
                                        <button
                                            key={lev}
                                            onClick={() => setSeverity(lev as any)}
                                            className={`py-2 rounded-lg text-xs font-bold uppercase border transition-all ${severity === lev
                                                ? (lev === 'critical' ? 'bg-red-500 text-white border-red-600' : lev === 'high' ? 'bg-orange-500 text-white border-orange-600' : 'bg-blue-500 text-white border-blue-600')
                                                : 'bg-slate-50 dark:bg-slate-950 text-slate-600 dark:text-slate-400 border-slate-200 dark:border-slate-700 hover:border-slate-400'
                                                }`}
                                        >
                                            {lev}
                                        </button>
                                    ))}
                                </div>
                            </div>
                        </div>
                        <div>
                            <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Detailed Description</label>
                            <textarea
                                value={desc} onChange={e => setDesc(e.target.value)}
                                className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3.5 text-sm outline-none h-32 resize-none focus:border-red-500 transition-all"
                                placeholder="What happened? What did you expect to happen?"
                            />
                        </div>
                        <div>
                            <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Steps to Reproduce</label>
                            <textarea
                                value={steps} onChange={e => setSteps(e.target.value)}
                                className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3.5 text-sm outline-none h-24 resize-none focus:border-red-500 transition-all"
                                placeholder="1. Go to... 2. Click on... 3. See error..."
                            />
                        </div>
                    </div>
                    {submitError && (
                        <div className="bg-red-50 dark:bg-red-900/10 border border-red-200 dark:border-red-900/30 rounded-xl p-4 text-red-600 dark:text-red-400 text-sm flex items-center gap-2">
                            <AlertCircle size={16} />
                            {submitError}
                        </div>
                    )}
                    <button
                        onClick={handleSubmit}
                        disabled={!title || !desc || isSubmitting}
                        className="w-full py-4 bg-red-600 hover:bg-red-700 text-white font-bold rounded-xl transition disabled:opacity-50 disabled:cursor-not-allowed shadow-lg shadow-red-600/20 mt-6 flex items-center justify-center gap-2"
                    >
                        {isSubmitting ? (
                            <>
                                <Loader2 className="animate-spin" size={18} />
                                Submitting...
                            </>
                        ) : (
                            'Submit Bug Report'
                        )}
                    </button>
                </div>

                {/* Side Panel: System Info */}
                <div className="space-y-6">
                    <div className="bg-slate-100 dark:bg-slate-800/50 rounded-2xl p-6 border border-slate-200 dark:border-slate-700">
                        <h3 className="font-bold text-slate-900 dark:text-white mb-4 flex items-center gap-2">
                            <Laptop size={18} /> System Info
                        </h3>
                        <div className="space-y-3 text-sm">
                            <div className="flex justify-between">
                                <span className="text-slate-500">Browser</span>
                                <span className="font-mono font-bold text-slate-700 dark:text-slate-300">{systemInfo.browser}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-slate-500">OS</span>
                                <span className="font-mono font-bold text-slate-700 dark:text-slate-300">{systemInfo.os}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-slate-500">Screen</span>
                                <span className="font-mono font-bold text-slate-700 dark:text-slate-300">{window.innerWidth}x{window.innerHeight}</span>
                            </div>
                            <div className="pt-3 border-t border-slate-200 dark:border-slate-700 text-xs text-slate-500 leading-relaxed">
                                This info will be automatically included in your report to help us debug faster.
                            </div>
                        </div>
                    </div>

                    <div className="bg-amber-50 dark:bg-amber-900/10 rounded-2xl p-6 border border-amber-200 dark:border-amber-500/20">
                        <h3 className="font-bold text-amber-800 dark:text-amber-400 mb-2 text-sm">Note on Critical Bugs</h3>
                        <p className="text-xs text-amber-700 dark:text-amber-300 leading-relaxed">
                            For security vulnerabilities or critical outages, please contact us directly via the Support chat for immediate assistance.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default SettingsPage;