import React, { useState, useEffect } from 'react';
import {
    Sparkles, Download, RefreshCw, Copy, Trash2,
    History as HistoryIcon, ChevronDown, ChevronUp,
    Heart, Lock, Shuffle, CheckCircle2, AlertCircle,
    Plus, Palette, Play, Key, Cpu, Layers, Lightbulb,
    Zap, TrendingUp, Camera, Wand2
} from 'lucide-react';
import { generateStockPrompts, getPreferredModels } from '../services/geminiService';
import { dbService, Asset } from '../services/dbService';
import { authService } from '../services/authService';
import { ModelType } from '../types';
import { EnhancedLoadingOverlay } from './EnhancedLoadingOverlay';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';

const STYLE_PRESETS = [
    { id: 'photorealistic', label: 'Photorealistic', icon: Camera },
    { id: 'artistic', label: 'Artistic', icon: Palette },
    { id: 'cinematic', label: 'Cinematic', icon: TrendingUp },
    { id: 'commercial', label: 'Commercial', icon: Zap },
    { id: 'editorial', label: 'Editorial', icon: Lightbulb },
    { id: 'creative', label: 'Creative', icon: Wand2 },
];

const THINKING_MODELS = [
    { id: ModelType.FLASH, name: 'Gemini 2.0 Flash (Fast)' },
    { id: ModelType.FLASH, name: 'Gemini 2.5 Flash (Balanced)' },
    { id: ModelType.PRO, name: 'Gemini 3 Pro (Quality)' },
    { id: ModelType.PRO_2_5, name: 'Gemini 2.5 Pro (Advanced)' },
];

const PromptGenerator: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'generator' | 'bulk' | 'history'>('generator');
    const [hasApiKey, setHasApiKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [generationInfo, setGenerationInfo] = useState<{ apiKey: string, model: string } | null>(null);
    const [rateLimitError, setRateLimitError] = useState(false); // Popup state

    // Generator State
    const [topic, setTopic] = useState('');
    const [style, setStyle] = useState('photorealistic');
    const [keywords, setKeywords] = useState('');
    const [promptCount, setPromptCount] = useState(5);
    const [selectedModel, setSelectedModel] = useState(ModelType.FLASH);
    const [includeNegative, setIncludeNegative] = useState(false);

    // Active Models (Enabled in Settings)
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // Processing State
    const [isGenerating, setIsGenerating] = useState(false);
    const [generatedPrompts, setGeneratedPrompts] = useState<string[]>([]);
    const [copiedIndex, setCopiedIndex] = useState<number | null>(null);

    // Bulk State
    const [bulkTopics, setBulkTopics] = useState('');
    const [bulkStatus, setBulkStatus] = useState<'idle' | 'processing' | 'complete'>('idle');
    const [bulkProgress, setBulkProgress] = useState(0);
    const [bulkResults, setBulkResults] = useState<Array<{ topic: string, prompts: string[] }>>([]);

    // History
    const [history, setHistory] = useState<Asset[]>([]);

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('prompt').then(setHistory);
        }
    }, [activeTab]);

    useEffect(() => {
        checkApiKey();
        updateActiveModels();
        const handleFocus = () => {
            checkApiKey();
            updateActiveModels();
        }
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    const checkApiKey = async () => {
        const key = await getUserApiKey();
        setHasApiKey(!!key);
        setIsCheckingKey(false);
    };

    const updateActiveModels = () => {
        const models = getPreferredModels('thinking');
        setActiveModels(models);
    };

    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;

        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    const handleGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!topic || !currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        setIsGenerating(true);
        setGenerationInfo({ apiKey: apiKeys[0], model: selectedModel });

        try {
            const keywordsList = keywords.split(',').map(k => k.trim()).filter(k => k);
            const prompts = await generateStockPrompts(apiKeys, topic, style, keywordsList, promptCount, selectedModel);

            setGeneratedPrompts(prompts);

            // Save to DB
            const asset: Asset = {
                id: Math.random().toString(36).substring(2, 10),
                type: 'prompt',
                url: '', // Not applicable for prompts
                prompt: topic,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: {
                    style,
                    keywords: keywordsList,
                    prompts,
                    count: prompts.length,
                    model: selectedModel
                }
            };
            await dbService.add(asset);

            // Log Activity & Stats
            if (currentUser) {
                authService.incrementUserStat('promptsGenerated', prompts.length, selectedModel);
                authService.logActivity(currentUser.uid, 'generate', `Generated ${prompts.length} prompts for: "${topic}"`);

                if (apiKeys[0]) {
                    authService.deductCredits(currentUser.uid, 1, apiKeys[0], selectedModel);
                }
            }

        } catch (error: any) {
            console.error('Prompt generation failed:', error);
            if (error.name === 'RateLimitExhaustedError' || error.message?.includes('RateLimitExhaustedError')) {
                setRateLimitError(true);
            } else {
                alert('Failed to generate prompts. Please check your API key and quota.');
            }
        } finally {
            setIsGenerating(false);
        }
    };

    const handleBulkGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        const topics = bulkTopics.split('\n').filter(t => t.trim());
        if (topics.length === 0) return;

        setBulkStatus('processing');
        setBulkProgress(0);
        setBulkResults([]);

        const results: Array<{ topic: string, prompts: string[] }> = [];

        for (let i = 0; i < topics.length; i++) {
            // Apply Data Delay / Global Cooldown
            const prefs = (currentUser as any)?.preferences?.generationConfig;
            const globalCooldownSec = Number(prefs?.globalCooldown || 0);
            const dataDelaySec = Number(prefs?.metadataDelay || 0);

            // Additive Delay Logic: Global Cooldown + Data Delay
            const effectiveDelaySec = globalCooldownSec + dataDelaySec;

            if (effectiveDelaySec > 0) {
                const cooldownMs = effectiveDelaySec * 1000;
                console.log(`[PromptGen] Applying Total Delay: ${cooldownMs}ms (${globalCooldownSec}s Global + ${dataDelaySec}s Data)`);
                await new Promise(r => setTimeout(r, cooldownMs));
            }

            try {
                const t = topics[i];
                const keywordsList = keywords.split(',').map(k => k.trim()).filter(k => k);
                const prompts = await generateStockPrompts(apiKeys, t, style, keywordsList, 3, selectedModel);

                // Save to DB
                const asset: Asset = {
                    id: Math.random().toString(36).substring(2, 10),
                    type: 'prompt',
                    url: '',
                    prompt: t,
                    createdAt: Date.now(),
                    isFavorite: false,
                    isDeleted: false,
                    metadata: { style, prompts, model: selectedModel }
                };
                await dbService.add(asset);

                if (currentUser) {
                    authService.incrementUserStat('promptsGenerated', prompts.length);
                    // authService.incrementUserStat('modelUsage', 1, selectedModel); // Handled by deductCredits
                    if (apiKeys[0]) {
                        await authService.deductCredits(currentUser.uid, 1, apiKeys[0], selectedModel);
                    }
                }

                results.push({ topic: t, prompts });
            } catch (e) {
                console.error(`Failed topic: ${topics[i]}`);
            }

            setBulkProgress(Math.round(((i + 1) / topics.length) * 100));
        }

        if (currentUser) {
            authService.logActivity(currentUser.uid, 'generate', `Batch generated prompts for ${results.length} topics`);
        }

        setBulkResults(results);
        setBulkStatus('complete');
    };

    const handleCopyPrompt = (prompt: string, index: number) => {
        navigator.clipboard.writeText(prompt);
        setCopiedIndex(index);
        setTimeout(() => setCopiedIndex(null), 2000);
    };

    const handleUseInGenerator = (prompt: string) => {
        // Navigate to Image Generator with the prompt
        localStorage.setItem('sf_preset_prompt', prompt);
        window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'IMAGE_GENERATION' }));
    };

    const handleDownloadAllHistory = () => {
        downloadItems(history.map(item => {
            const content = `Topic: ${item.prompt}\nStyle: ${item.metadata?.style}\n\nPrompts:\n${item.metadata?.prompts?.join('\n\n')}`;
            const blob = new Blob([content], { type: 'text/plain' });
            return {
                url: URL.createObjectURL(blob),
                filename: `prompts_${item.id}.txt`
            };
        }), 'prompt_history');
    };

    const handleDownloadBatch = () => {
        downloadItems(bulkResults.map((item, idx) => {
            const content = `Topic: ${item.topic}\n\nPrompts:\n${item.prompts.join('\n\n')}`;
            const blob = new Blob([content], { type: 'text/plain' });
            return {
                url: URL.createObjectURL(blob),
                filename: `batch_prompts_${idx}.txt`
            };
        }), 'batch_prompts');
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">

            {/* Header */}
            <div className="mb-6 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <Sparkles className="text-orange-500 w-8 h-8" /> Auto-Prompt Generator
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">AI-powered prompt generation for stock photography and creative content.</p>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Tab Navigation */}
            <div className="flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-6 shrink-0">
                <div className="flex w-full md:w-auto overflow-x-auto">
                    {[
                        { id: 'generator', label: 'Generator', icon: Sparkles },
                        { id: 'bulk', label: 'Bulk', icon: Layers },
                        { id: 'history', label: 'History', icon: HistoryIcon },
                    ].map((tab) => (
                        <button
                            key={tab.id}
                            onClick={() => setActiveTab(tab.id as any)}
                            className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === tab.id ? 'text-orange-600 dark:text-orange-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                                }`}
                        >
                            <tab.icon size={16} />
                            {tab.label}
                            {activeTab === tab.id && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-orange-500 animate-fade-in"></div>}
                        </button>
                    ))}
                </div>

                {/* Active Models Display */}
                {activeTab !== 'history' && (
                    <div className="hidden md:flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg my-2 mr-2">
                        <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                            <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                        </span>
                        <div className="flex gap-1">
                            {activeModels.map((model, idx) => (
                                <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                    {model}
                                </span>
                            ))}
                            {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                        </div>
                    </div>
                )}
            </div>

            {/* Content Area */}
            <div className={`flex-1 overflow-hidden transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>

                {/* GENERATOR TAB */}
                {activeTab === 'generator' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20 custom-scrollbar">
                        {/* LEFT PANEL: CONTROLS */}
                        <div className="lg:col-span-4 space-y-6 h-fit">
                            {/* Input Form */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">Topic / Subject</label>
                                    <input
                                        type="text"
                                        value={topic}
                                        onChange={(e) => setTopic(e.target.value)}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-orange-500 outline-none placeholder:text-slate-400 dark:placeholder:text-slate-600 text-sm"
                                        placeholder="e.g., Business meeting, Nature landscape, Technology concept"
                                    />
                                </div>

                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">Style / Mood</label>
                                    <div className="grid grid-cols-3 gap-2">
                                        {STYLE_PRESETS.map((preset) => (
                                            <button
                                                key={preset.id}
                                                onClick={() => setStyle(preset.id)}
                                                className={`p-3 rounded-xl border transition-all flex flex-col items-center gap-1.5 ${style === preset.id
                                                    ? 'bg-orange-500/10 border-orange-500 text-orange-600 dark:text-orange-400'
                                                    : 'bg-slate-50 dark:bg-slate-950 border-slate-200 dark:border-slate-800 text-slate-500 hover:border-orange-300'
                                                    }`}
                                            >
                                                <preset.icon size={16} />
                                                <span className="text-[10px] font-medium">{preset.label}</span>
                                            </button>
                                        ))}
                                    </div>
                                </div>

                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">Keywords (optional)</label>
                                    <input
                                        type="text"
                                        value={keywords}
                                        onChange={(e) => setKeywords(e.target.value)}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 text-sm outline-none focus:border-orange-500 placeholder:text-slate-400 dark:placeholder:text-slate-600"
                                        placeholder="professional, modern, minimal (comma-separated)"
                                    />
                                </div>
                            </div>

                            {/* Generation Settings */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 flex items-center gap-2">
                                        <Cpu size={14} className="text-orange-500" /> AI Model
                                    </label>
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value as ModelType)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-orange-500">
                                        {THINKING_MODELS.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                </div>

                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">Number of Prompts</label>
                                    <select value={promptCount} onChange={(e) => setPromptCount(Number(e.target.value))} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-orange-500">
                                        {[3, 5, 7, 10].map(n => <option key={n} value={n}>{n} Prompts</option>)}
                                    </select>
                                </div>

                                <div className="flex items-center justify-between pt-2 border-t border-slate-200 dark:border-slate-800">
                                    <label className="text-xs font-bold text-slate-600 dark:text-slate-400">Include Negative Prompts</label>
                                    <button
                                        onClick={() => setIncludeNegative(!includeNegative)}
                                        className={`w-10 h-5 rounded-full p-0.5 flex transition-colors ${includeNegative ? 'bg-orange-500 justify-end' : 'bg-slate-300 dark:bg-slate-700 justify-start'}`}
                                    >
                                        <div className="w-4 h-4 bg-white rounded-full shadow-sm" />
                                    </button>
                                </div>
                            </div>

                            <button
                                onClick={handleGenerate}
                                disabled={isGenerating || !topic || !hasApiKey}
                                className="w-full py-4 bg-gradient-to-r from-orange-500 to-orange-600 hover:from-orange-400 hover:to-orange-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-orange-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {isGenerating ? <><RefreshCw className="animate-spin" /> Generating...</> : <><Sparkles /> Generate Prompts</>}
                                {!isGenerating && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">1 API Call</span>}
                            </button>
                        </div>

                        {/* RIGHT PANEL: RESULTS */}
                        <div className="lg:col-span-8 bg-white dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 flex flex-col relative min-h-[600px] shadow-sm dark:shadow-none">
                            {generatedPrompts.length === 0 && !isGenerating && (
                                <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                    <div className="w-24 h-24 bg-slate-100 dark:bg-slate-900 rounded-3xl flex items-center justify-center mb-6 border border-slate-200 dark:border-slate-800 rotate-12">
                                        <Sparkles size={48} className="opacity-50" />
                                    </div>
                                    <h3 className="text-2xl font-bold text-slate-700 dark:text-slate-500 mb-2">Ready to Create Prompts?</h3>
                                    <p className="text-sm text-slate-600 dark:text-slate-600 max-w-xs text-center">Configure your settings and hit generate to create amazing stock photography prompts.</p>
                                </div>
                            )}

                            {generatedPrompts.length > 0 && (
                                <div className="space-y-3 overflow-y-auto custom-scrollbar h-full">
                                    {generatedPrompts.map((prompt, idx) => (
                                        <div key={idx} className="group relative rounded-xl border border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-900 p-4 hover:border-orange-500/30 transition-all">
                                            <div className="flex items-start gap-3">
                                                <div className="flex-shrink-0 w-8 h-8 bg-orange-500/10 rounded-lg flex items-center justify-center text-orange-600 dark:text-orange-400 font-bold text-sm">
                                                    {idx + 1}
                                                </div>
                                                <div className="flex-1">
                                                    <p className="text-sm text-slate-700 dark:text-slate-300 leading-relaxed">{prompt}</p>
                                                </div>
                                            </div>
                                            <div className="flex gap-2 mt-3 pt-3 border-t border-slate-200 dark:border-slate-800">
                                                <button
                                                    onClick={() => handleCopyPrompt(prompt, idx)}
                                                    className="flex-1 py-2 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-300 font-bold rounded-lg text-xs flex items-center justify-center gap-2 transition"
                                                >
                                                    {copiedIndex === idx ? <><CheckCircle2 size={14} /> Copied!</> : <><Copy size={14} /> Copy</>}
                                                </button>
                                                <button
                                                    onClick={() => handleUseInGenerator(prompt)}
                                                    className="flex-1 py-2 bg-orange-500 hover:bg-orange-600 text-white dark:text-slate-950 font-bold rounded-lg text-xs flex items-center justify-center gap-2 transition"
                                                >
                                                    <Wand2 size={14} /> Use in Generator
                                                </button>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* BULK TAB */}
                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20">
                        <div className="lg:col-span-5 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <h3 className="font-bold text-slate-900 dark:text-slate-200 mb-4">Bulk Generator</h3>
                                <p className="text-xs text-slate-500 mb-4">Enter multiple topics, one per line.</p>
                                <textarea
                                    value={bulkTopics}
                                    onChange={(e) => setBulkTopics(e.target.value)}
                                    className="w-full h-64 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-4 text-slate-900 dark:text-slate-200 text-sm font-mono outline-none resize-none focus:border-orange-500"
                                    placeholder="Business meeting&#10;Nature landscape&#10;Technology concept"
                                />
                                <div className="flex flex-col gap-4 mt-4">
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value as ModelType)} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-700 dark:text-slate-300 outline-none focus:border-orange-500">
                                        {THINKING_MODELS.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                    <button
                                        onClick={handleBulkGenerate}
                                        disabled={bulkStatus === 'processing' || !bulkTopics || !hasApiKey}
                                        className="w-full py-2 bg-orange-500 hover:bg-orange-600 text-white dark:text-slate-950 font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50"
                                    >
                                        {bulkStatus === 'processing' ? <RefreshCw className="animate-spin" size={16} /> : <Play size={16} />}
                                        {bulkStatus === 'processing' ? 'Processing...' : 'Start Batch'}
                                        {bulkStatus !== 'processing' && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">{bulkTopics.split('\n').filter(t => t.trim()).length} API Calls</span>}
                                    </button>
                                </div>
                            </div>

                            {bulkStatus !== 'idle' && (
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                    <div className="flex justify-between text-sm mb-2 font-bold text-slate-700 dark:text-slate-300">
                                        <span>Batch Progress</span>
                                        <div className="flex items-center gap-2">
                                            {bulkStatus === 'complete' && bulkResults.length > 0 && (
                                                <button onClick={handleDownloadBatch} className="text-xs bg-orange-500/10 text-orange-600 dark:text-orange-400 hover:text-orange-700 dark:hover:text-orange-300 px-2 py-1 rounded transition-colors flex items-center gap-1">
                                                    <Download size={12} /> Download Batch
                                                </button>
                                            )}
                                            <span>{bulkProgress}%</span>
                                        </div>
                                    </div>
                                    <div className="w-full bg-slate-200 dark:bg-slate-800 h-2 rounded-full overflow-hidden">
                                        <div className="h-full bg-orange-500 transition-all duration-300" style={{ width: `${bulkProgress}%` }}></div>
                                    </div>
                                </div>
                            )}
                        </div>

                        <div className="lg:col-span-7 bg-white dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 h-[600px] overflow-y-auto custom-scrollbar shadow-sm dark:shadow-none">
                            <div className="space-y-4">
                                {bulkResults.map((result, idx) => (
                                    <div key={idx} className="border border-slate-200 dark:border-slate-800 rounded-xl p-4 bg-slate-50 dark:bg-slate-950">
                                        <h4 className="font-bold text-slate-900 dark:text-white mb-2 text-sm">{result.topic}</h4>
                                        <div className="space-y-2">
                                            {result.prompts.map((p, pIdx) => (
                                                <p key={pIdx} className="text-xs text-slate-600 dark:text-slate-400 pl-3 border-l-2 border-orange-500">{p}</p>
                                            ))}
                                        </div>
                                    </div>
                                ))}
                                {bulkResults.length === 0 && (
                                    <div className="flex flex-col items-center justify-center text-slate-500 h-64">
                                        <Layers size={32} className="mb-2 opacity-50" />
                                        <p>Generated prompts will appear here</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}

                {/* HISTORY TAB */}
                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="flex justify-between items-center mb-4 px-2">
                            <h3 className="font-bold text-slate-700 dark:text-slate-300">Prompt History ({history.length})</h3>
                            {history.length > 0 && (
                                <button onClick={handleDownloadAllHistory} className="flex items-center gap-2 px-3 py-1.5 bg-orange-500 hover:bg-orange-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition shadow-lg shadow-orange-500/20">
                                    <Download size={14} /> Download All
                                </button>
                            )}
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                            {history.map((item) => (
                                <div key={item.id} className="group relative rounded-xl border border-slate-200 dark:border-slate-800 p-4 bg-white dark:bg-slate-900 hover:border-orange-500/30 transition">
                                    <div className="flex items-start justify-between mb-2">
                                        <h4 className="font-bold text-slate-900 dark:text-white text-sm">{item.prompt}</h4>
                                        <button onClick={() => dbService.toggleFavorite(item.id)} className={`p-1.5 rounded-lg ${item.isFavorite ? 'text-red-500' : 'text-slate-400'}`}>
                                            <Heart size={14} />
                                        </button>
                                    </div>
                                    <p className="text-xs text-slate-500 mb-3">{item.metadata?.prompts?.length || 0} prompts • {new Date(item.createdAt).toLocaleDateString()}</p>
                                    <div className="space-y-1 max-h-32 overflow-y-auto custom-scrollbar">
                                        {item.metadata?.prompts?.map((p: string, idx: number) => (
                                            <p key={idx} className="text-xs text-slate-600 dark:text-slate-400 line-clamp-2">{p}</p>
                                        ))}
                                    </div>
                                </div>
                            ))}
                            {history.length === 0 && (
                                <div className="col-span-full text-center py-12 text-slate-500">
                                    <HistoryIcon size={32} className="mx-auto mb-2 opacity-30" />
                                    No generation history found.
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </div>

            {/* Loading Overlay */}
            <EnhancedLoadingOverlay
                isVisible={isGenerating}
                title="Crafting Prompts"
                description={`Generating ${promptCount} creative prompts with ${selectedModel}...`}
                simulate={true}
                onCancel={() => setIsGenerating(false)}
                apiKey={generationInfo?.apiKey}
                model={generationInfo?.model}
            />
            {/* Rate Limit Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />
        </div>
    );
};

export default PromptGenerator;
