
import React, { useState, useRef, useEffect } from 'react';
import {
    ScanEye, Upload, X, Image as ImageIcon, Settings,
    RefreshCw, Copy, Download, Save, FileText, Sparkles,
    Globe, Check, CheckCircle2, AlertCircle, Info, ChevronDown,
    ChevronUp, Hash, Layers, Play, Pause, Square, FolderArchive, Edit2, RotateCcw,
    MoreVertical, FileJson, FileType, Archive, Key, Cpu,
    History as HistoryIcon, Trash2, PlusCircle, MinusCircle, Clock, AlertTriangle, Loader2, FileDown
} from 'lucide-react';
import { generateMetadata, regenerateTitle, getPreferredModels, RateLimitExhaustedError } from '../services/geminiService';
import { MetadataResult, MetadataSettings, BatchItem, ModelType, User } from '../types';
import { authService } from '../services/authService';
import { dbService, Asset } from '../services/dbService';
import { adminService } from '../services/adminService';
import { StockSite, SiteSettings } from '../types';
import { ADOBE_STOCK_CATEGORIES } from '../constants';
import { EnhancedLoadingOverlay } from './EnhancedLoadingOverlay';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';
import { playSound } from '../utils/soundEffects';
import { Lightbox } from './Lightbox';
import { useBatch } from '../contexts/BatchContext';
import { extractVideoFrames, svgToPng } from '../utils/mediaUtils';

// --- Default Settings ---
const DEFAULT_SETTINGS: MetadataSettings = {
    titleLength: 60,
    descLength: 250, // Changed from words to chars default
    keywordCount: 25,
    keywordType: 'single',
    enableGrounding: false,
    exportFormat: 'csv',
    useFilenameMode: false,
    videoFrameOptions: ['start'], // Default to 3rd frame (Start)
    selectedStockSites: []
};

// --- Helper Components ---

interface TagInputProps {
    placeholder?: string;
    tags: string[];
    onTagsChange: (tags: string[]) => void;
    color?: 'blue' | 'red';
}

const TagInput: React.FC<TagInputProps> = ({ placeholder, tags, onTagsChange, color = 'blue' }) => {
    const [input, setInput] = useState('');

    const handleKeyDown = (e: React.KeyboardEvent) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const val = input.trim();
            if (val && !tags.includes(val)) {
                onTagsChange([...tags, val]);
                setInput('');
            }
        } else if (e.key === 'Backspace' && !input && tags.length > 0) {
            onTagsChange(tags.slice(0, -1));
        }
    };

    const removeTag = (tagToRemove: string) => {
        onTagsChange(tags.filter(tag => tag !== tagToRemove));
    };

    const colorClasses = color === 'blue'
        ? 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300'
        : 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-300';

    return (
        <div className="flex flex-wrap items-center gap-2 p-2 bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg focus-within:ring-2 focus-within:ring-amber-500/20 focus-within:border-amber-500 transition-all min-h-[42px]">
            {tags.map(tag => (
                <span key={tag} className={`flex items-center gap-1 px-2 py-0.5 rounded text-xs font-medium animate-fade-in ${colorClasses}`}>
                    {tag}
                    <button onClick={() => removeTag(tag)} className="hover:opacity-70"><X size={12} /></button>
                </span>
            ))}
            <input
                type="text"
                value={input}
                onChange={e => setInput(e.target.value)}
                onKeyDown={handleKeyDown}
                placeholder={tags.length === 0 ? placeholder : ''}
                className="flex-1 bg-transparent border-none outline-none text-sm min-w-[60px] text-slate-900 dark:text-slate-200 placeholder:text-slate-400"
            />
        </div>
    );
};
// Video Preview Component (Shows Start/End Frames) moved below
// Helpers moved to utils/mediaUtils

// Helper to resize image for DB storage (Thumbnail)
const resizeImage = (base64Str: string, maxWidth = 1024): Promise<string> => {
    return new Promise((resolve) => {
        const img = new Image();
        img.src = base64Str;
        img.onload = () => {
            let width = img.width;
            let height = img.height;

            if (width > maxWidth) {
                height = Math.round((height * maxWidth) / width);
                width = maxWidth;
            }

            const canvas = document.createElement('canvas');
            canvas.width = width;
            canvas.height = height;
            const ctx = canvas.getContext('2d');
            ctx?.drawImage(img, 0, 0, width, height);
            resolve(canvas.toDataURL('image/jpeg', 0.8)); // Compress nicely
        };
        img.onerror = (e) => {
            console.error("Resize failed", e);
            resolve(base64Str); // Fallback to original if fail
        };
    });
};

const InfoTooltip = ({ text, size = 12 }: { text: string, size?: number }) => (
    <div className="relative group/info inline-block ml-1 align-middle z-50">
        <Info size={size} className="text-slate-400 hover:text-amber-500 cursor-help" />
        <div className="absolute left-1/2 -translate-x-1/2 bottom-full mb-2 w-48 p-2 bg-slate-800 text-white text-[10px] rounded-lg opacity-0 group-hover/info:opacity-100 transition pointer-events-none shadow-xl border border-slate-700 text-center leading-relaxed">
            {text}
            <div className="absolute left-1/2 -translate-x-1/2 top-full w-0 h-0 border-l-[6px] border-l-transparent border-r-[6px] border-r-transparent border-t-[6px] border-t-slate-800"></div>
        </div>
    </div>
);

// Video Preview Component (Shows Start/End Frames)
const VideoPreview: React.FC<{ file: File, onImageClick?: (src: string) => void }> = ({ file, onImageClick }) => {
    const [frames, setFrames] = useState<string[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        let mounted = true;
        extractVideoFrames(file)
            .then(f => {
                if (mounted) {
                    setFrames(f);
                    setLoading(false);
                }
            })
            .catch(() => {
                if (mounted) setLoading(false);
            });
        return () => { mounted = false; };
    }, [file]);

    if (loading || frames.length === 0) {
        return (
            <div className="w-12 h-12 rounded-lg bg-slate-200 dark:bg-slate-800 flex items-center justify-center shrink-0">
                <Play size={24} className="text-slate-400 animate-pulse" />
            </div>
        );
    }

    // Show all available frames (Start, Middle, End)
    const displayFrames = frames;

    return (
        <div className="flex gap-1 shrink-0">
            {displayFrames.map((src, i) => {
                let label = 'MID';
                if (i === 0) label = 'START';
                else if (i === displayFrames.length - 1) label = 'END';

                return (
                    <div key={i} className="relative group cursor-pointer" onClick={() => onImageClick?.(src)}>
                        <img
                            src={src}
                            className="h-12 w-20 rounded-md object-cover bg-black border border-slate-700 hover:border-amber-500 transition-colors"
                            title={`Frame: ${label} (Click to View)`}
                        />
                        <div className="absolute inset-0 bg-black/20 group-hover:bg-transparent transition rounded-md" />
                        <div className="absolute bottom-0.5 right-0.5 px-1 py-0.5 bg-black/60 rounded text-[8px] text-white font-mono pointer-events-none">
                            {label}
                        </div>
                    </div>
                );
            })}
        </div>
    );
};

const MetadataGenerator: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'single' | 'bulk' | 'history'>(() => {
        return (localStorage.getItem('sf_metadata_active_tab') as 'single' | 'bulk' | 'history') || 'single';
    });

    useEffect(() => {
        localStorage.setItem('sf_metadata_active_tab', activeTab);
    }, [activeTab]);
    const [hasApiKey, setHasApiKey] = useState(false);
    const [generationInfo, setGenerationInfo] = useState<{ apiKey: string, model: string } | null>(null);
    const [availableSites, setAvailableSites] = useState<StockSite[]>([]);
    const [siteSettings, setSiteSettings] = useState<SiteSettings | null>(null);


    // --- Batch Control State (Context Handles Refs) ---
    const [isBatchPaused, setIsBatchPaused] = useState(false);

    const togglePause = () => {
        if (isBatchProcessing) {
            pauseBatch();
            setIsBatchPaused(true);
        } else if (isBatchPaused) {
            resumeBatch(settings, currentUser!, [], availableSites, siteSettings); // Resume needs args
            setIsBatchPaused(false);
        }
    };

    // --- Settings (Shared) ---
    // --- Settings (Shared) ---
    // --- User Context & Persistence ---
    const [currentUser, setCurrentUser] = useState<User | null>(authService.getCurrentUser());

    useEffect(() => {
        return authService.onAuthStateChanged(setCurrentUser);
    }, []);

    // Helper functions for loading user-specific data
    const loadUserSettings = (uid: string): MetadataSettings => {
        try {
            const saved = localStorage.getItem(`metadata_settings_${uid}`);
            if (saved) return { ...DEFAULT_SETTINGS, ...JSON.parse(saved) };
        } catch (e) { console.error(e); }
        return DEFAULT_SETTINGS;
    };

    const loadUserKeywords = (uid: string, type: 'include' | 'exclude'): string[] => {
        try {
            const key = type === 'include' ? `metadata_include_keywords_${uid}` : `metadata_exclude_keywords_${uid}`;
            const saved = localStorage.getItem(key);
            return saved ? JSON.parse(saved) : [];
        } catch { return []; }
    };

    // --- Settings State (User-Specific) ---
    const [settings, setSettings] = useState<MetadataSettings>(() => {
        const u = authService.getCurrentUser();
        return u ? loadUserSettings(u.uid) : DEFAULT_SETTINGS;
    });

    // --- Single Image State (Moved) ---
    const [singleFile, setSingleFile] = useState<File | null>(null);
    const [singlePreview, setSinglePreview] = useState<string | null>(null);
    const [videoFrames, setVideoFrames] = useState<string[]>([]); // Store extracted video frames
    const [isDragging, setIsDragging] = useState(false);
    const [fileDimensions, setFileDimensions] = useState<{ width: number; height: number } | null>(null);

    // --- Persistent Keywords (User-Specific) ---
    const [contextKeywords, setContextKeywords] = useState<string[]>(() => {
        const u = authService.getCurrentUser();
        return u ? loadUserKeywords(u.uid, 'include') : [];
    });

    const [excludeKeywords, setExcludeKeywords] = useState<string[]>(() => {
        const u = authService.getCurrentUser();
        return u ? loadUserKeywords(u.uid, 'exclude') : [];
    });

    // --- State & Effects for User Switching and Persistence ---

    // 1. Reload when user changes
    useEffect(() => {
        if (currentUser) {
            setSettings(loadUserSettings(currentUser.uid));
            setContextKeywords(loadUserKeywords(currentUser.uid, 'include'));
            setExcludeKeywords(loadUserKeywords(currentUser.uid, 'exclude'));
        } else {
            // Reset to defaults if no user (logged out)
            setSettings(DEFAULT_SETTINGS);
            setContextKeywords([]);
            setExcludeKeywords([]);
        }
    }, [currentUser?.uid]);

    // 2. Save on change (only if user logged in)
    useEffect(() => {
        if (currentUser?.uid) {
            console.log('[DEBUG] Saving settings to localStorage:', settings);
            console.log('[DEBUG] exportFormat being saved:', settings.exportFormat);
            localStorage.setItem(`metadata_settings_${currentUser.uid}`, JSON.stringify(settings));
        }
    }, [settings, currentUser?.uid]);

    useEffect(() => {
        if (currentUser?.uid) {
            localStorage.setItem(`metadata_include_keywords_${currentUser.uid}`, JSON.stringify(contextKeywords));
        }
    }, [contextKeywords, currentUser?.uid]);

    useEffect(() => {
        if (currentUser?.uid) {
            localStorage.setItem(`metadata_exclude_keywords_${currentUser.uid}`, JSON.stringify(excludeKeywords));
        }
    }, [excludeKeywords, currentUser?.uid]);

    const [contextDesc, setContextDesc] = useState('');
    const [isGenerating, setIsGenerating] = useState(false);
    const [result, setResult] = useState<MetadataResult | null>(null);
    // batchStatusInfo moved to context

    // Lightbox State
    const [lightboxSrc, setLightboxSrc] = useState<string | null>(null);

    // --- Effects ---
    const [isEditingDesc, setIsEditingDesc] = useState(false);
    const [isRegeneratingTitle, setIsRegeneratingTitle] = useState(false);
    const [showExportMenu, setShowExportMenu] = useState(false);
    const [showSources, setShowSources] = useState(false);

    // --- Bulk Image State (Managed by Context) ---
    const {
        batchItems,
        isBatchProcessing,
        batchStatusInfo,
        rateLimitError,
        setRateLimitError,
        addFiles,
        startBatch,
        stopBatch,
        pauseBatch,
        resumeBatch,
        clearBatch,
        removeBatchItem,
        setBatchItems, // For inline edits
        isResuming,
        batchStartTime
    } = useBatch();

    const [totalElapsedTime, setTotalElapsedTime] = useState<string | null>(null);

    // Calculate Total Elapsed Time when batch completes
    useEffect(() => {
        if (!batchStartTime) {
            setTotalElapsedTime(null);
            return;
        }

        // If batch is complete, calculate final elapsed time
        if (!isBatchProcessing && batchItems.length > 0) {
            const allDone = batchItems.every(i => i.status === 'complete' || i.status === 'error');
            if (allDone) {
                const endTime = Date.now();
                const elapsed = Math.floor((endTime - batchStartTime) / 1000);
                const minutes = Math.floor(elapsed / 60);
                const seconds = elapsed % 60;
                setTotalElapsedTime(`${minutes}m ${seconds}s`);
            } else {
                setTotalElapsedTime(null);
            }
        } else {
            setTotalElapsedTime(null);
        }
    }, [isBatchProcessing, batchStartTime, batchItems]);

    const [editingBatchItemId, setEditingBatchItemId] = useState<string | null>(null);

    // --- History State ---
    const [historyItems, setHistoryItems] = useState<Asset[]>([]);

    // --- Auto-resize Textareas ---
    useEffect(() => {
        if (result) {
            // slight delay to allow render
            const timer = setTimeout(() => {
                const textareas = document.querySelectorAll('textarea[data-autosize="true"]');
                textareas.forEach((el: any) => {
                    el.style.height = 'auto';
                    el.style.height = el.scrollHeight + 'px';
                });
            }, 0);
            return () => clearTimeout(timer);
        }
    }, [result, activeTab]);

    // --- Active Models Display ---
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // --- API Key State ---
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    // rateLimitError moved to context

    // --- Single Mode Ref ---
    const abortControllerRef = useRef<AbortController | null>(null);

    // Refresh history when tab becomes active
    useEffect(() => {
        if (activeTab === 'history') {
            const loadHistory = async () => {
                const items = await dbService.getByType('metadata');
                setHistoryItems(items);
            };
            loadHistory();
        }
    }, [activeTab]);

    useEffect(() => {
        checkApiKey();
        const handleFocus = () => {
            checkApiKey();
            updateActiveModels();
        };
        window.addEventListener('focus', handleFocus);
        updateActiveModels(); // Initial fetch

        // Fetch sites and settings
        adminService.getMicrostockSites().then(data => {
            const visible = data.filter((s: StockSite) => s.isVisible);
            setAvailableSites(visible);

            // Auto-select first site if none selected (Default to first available, usually Adobe Stock)
            setSettings(prev => {
                if (!prev.selectedStockSites || prev.selectedStockSites.length === 0) {
                    if (visible.length > 0) {
                        return { ...prev, selectedStockSites: [visible[0].id] };
                    }
                }
                return prev;
            });
        });
        adminService.getSiteSettings().then(setSiteSettings);

        return () => window.removeEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    // Derived State
    const isAdobeEnabled = availableSites.some(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe'));

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('metadata').then(setHistoryItems);
        }
    }, [activeTab]);

    const updateActiveModels = () => {
        setActiveModels(getPreferredModels('thinking'));
    };

    const checkApiKey = async () => {
        const key = await getUserApiKey();
        setHasApiKey(!!key);
        setIsCheckingKey(false);
    };

    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    // --- Handlers ---

    const processSingleFile = async (file: File) => {
        if (!file) return;

        setSingleFile(file);
        setVideoFrames([]);
        setResult(null);
        setFileDimensions(null);

        if (file.type.startsWith('image/') || file.type === 'image/svg+xml') {
            const dataUrl = await new Promise<string>((resolve) => {
                const reader = new FileReader();
                reader.onload = (e) => resolve(e.target?.result as string);
                reader.readAsDataURL(file);
            });
            setSinglePreview(dataUrl);

            // Get Dimensions
            const img = new Image();
            img.onload = () => setFileDimensions({ width: img.width, height: img.height });
            img.src = dataUrl;

        } else if (file.type.startsWith('video/')) {
            const url = URL.createObjectURL(file);
            setSinglePreview(url);

            // Get Dimensions
            const video = document.createElement('video');
            video.onloadedmetadata = () => {
                setFileDimensions({ width: video.videoWidth, height: video.videoHeight });
            };
            video.src = url;

            try {
                const frames = await extractVideoFrames(file);
                setVideoFrames(frames);
            } catch (e) { console.error("Video frame error", e); }

        } else if (file.type.includes('svg') || file.name.endsWith('.svg')) {
            try {
                const pngData = await svgToPng(file);
                setSinglePreview(pngData);
                // Get Dimensions (from png or parse svg?) - pngData works
                const img = new Image();
                img.onload = () => setFileDimensions({ width: img.width, height: img.height });
                img.src = pngData;
            } catch (e) {
                console.error("SVG Error:", e);
                alert("Failed to process SVG.");
            }
        }
    };

    const handleSingleUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            processSingleFile(e.target.files[0]);
        }
    };

    const handleDragOver = (e: React.DragEvent) => {
        e.preventDefault();
        setIsDragging(true);
    };

    const handleDragLeave = (e: React.DragEvent) => {
        e.preventDefault();
        setIsDragging(false);
    };

    const handleDrop = (e: React.DragEvent) => {
        e.preventDefault();
        setIsDragging(false);
        if (e.dataTransfer.files && e.dataTransfer.files[0]) {
            processSingleFile(e.dataTransfer.files[0]);
        }
    };

    const generateSingle = async () => {
        // Fetch ALL valid keys
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        const allKeys = await authService.getUserApiKeys(currentUser.uid);
        const validKeys = allKeys
            .filter(k => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false)
            .map(k => k.fullKey);

        let apiKeysToUse = [...validKeys];
        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeysToUse = apiKeysToUse.sort(() => Math.random() - 0.5);
        }

        if (!singleFile || !singlePreview || apiKeysToUse.length === 0) {
            if (apiKeysToUse.length === 0) setHasApiKey(false);
            return;
        }

        // REMOVED: Clear metadata history for a clean slate on new single request
        await dbService.clearByType('metadata', true);

        setIsGenerating(true);
        // Create new AbortController
        const abortController = new AbortController();
        abortControllerRef.current = abortController;

        const preferredModel = getPreferredModels('thinking')[0] || ModelType.FLASH;

        setGenerationInfo({
            apiKey: apiKeysToUse[0], // Display the first one as "primary"
            model: preferredModel
        });
        try {
            let payload: string | string[] = singlePreview.split(',')[1];
            let mimeType = singleFile.type;

            if (singleFile.type.startsWith('video/') && videoFrames.length > 0) {
                // Select frames based on user preference (Multiple)
                const selectedPayloads: string[] = [];
                const opts = settings.videoFrameOptions || ['start'];

                if (opts.includes('start')) selectedPayloads.push(videoFrames[0]);
                if (opts.includes('middle') && videoFrames.length >= 2) selectedPayloads.push(videoFrames[Math.floor(videoFrames.length / 2)]);
                if (opts.includes('end') && videoFrames.length > 0) selectedPayloads.push(videoFrames[videoFrames.length - 1]);

                // Dedupe just in case
                const uniquePayloads = [...new Set(selectedPayloads)];

                if (uniquePayloads.length > 0) {
                    payload = uniquePayloads.map(p => p.split(',')[1]);
                    mimeType = 'image/jpeg';
                }
            }

            console.log('[DEBUG] BEFORE GENERATE - settings.exportFormat:', settings.exportFormat);
            console.log('[DEBUG] BEFORE GENERATE - full settings:', settings);

            const metadata = await generateMetadata(
                apiKeysToUse,
                payload,
                mimeType,
                settings,
                {
                    keywords: contextKeywords.join(', '),
                    excludeKeywords: excludeKeywords.join(', '),
                    description: contextDesc,
                    filename: singleFile?.name,
                    adobeStockInstruction: availableSites.find(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe')) ? siteSettings?.adobeStockInstruction : undefined,
                    titleInstruction: (() => {
                        if (singleFile?.type.startsWith('video/')) {
                            return siteSettings?.titleInstructionVideo || `You are an expert in stock video SEO.
Analyze the provided video and identify:
- moving elements
- camera angle or motion

Generate ONE SEO-friendly video title that:
- starts with the main visual subject
- includes motion words only if visible
- avoids cinematic or narrative language
- stays under 75 characters`;
                        } else if (singleFile?.type.includes('svg') || singleFile?.name.endsWith('.svg')) {
                            return siteSettings?.titleInstructionSvg || `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style
- context

Generate ONE SEO-optimized title that:
- starts with the main object name
- includes "vector", "icon", or "illustration" if relevant
- avoids artistic language
- remains under 65 characters`;
                        } else {
                            // Default to Image instruction
                            return siteSettings?.titleInstructionImage || siteSettings?.titleInstruction || `Act as a top-ranked Adobe Stock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (must be first in title)
2. Supporting visual elements

Create ONE SEO title that:
- matches buyer search intent
- is descriptive, not poetic
- starts with the main subject
- avoids adjectives that do not describe visuals
- is concise, professional, and under 70 characters`;
                        }
                    })(),
                    descriptionInstruction: (() => {
                        if (singleFile?.type.startsWith('video/')) {
                            return siteSettings?.descriptionInstructionVideo || `You are an expert in stock video SEO.
Analyze the video and identify:
- moving subjects
- camera motion or angle

Write ONE SEO-friendly description that:
- starts with the main moving subject
- describes visible motion accurately
- avoids cinematic or narrative language
- uses 1–2 concise sentences`;
                        } else if (singleFile?.type.includes('svg') || singleFile?.name.endsWith('.svg')) {
                            return siteSettings?.descriptionInstructionSvg || `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style
- context

Write ONE SEO-optimized description that:
- starts with the main object
- includes "vector", "icon", or "illustration" if relevant
- focuses on usability and clarity
- avoids artistic or emotional language`;
                        } else {
                            // Default to Image instruction
                            return siteSettings?.descriptionInstructionImage || siteSettings?.descriptionInstruction || `Act as a top microstock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject
2. Supporting objects or details

Write ONE description that:
- clearly describes only what is visible
- opens with the main subject in the first 5 words
- naturally includes common buyer search terms
- avoids emotional adjectives unless visually clear
- remains factual and professional
- is between 2–3 concise sentences`;
                        }
                    })(),
                    keywordInstruction: siteSettings?.keywordInstruction,
                    keywordGenerationPrompt: siteSettings?.keywordGenerationPrompt
                },
            );
            const finalMetadata = {
                ...metadata,
                filename: singleFile?.name || "image.jpg",
                category: (metadata.category && ADOBE_STOCK_CATEGORIES[metadata.category])
                    ? `${metadata.category} (${ADOBE_STOCK_CATEGORIES[metadata.category]})`
                    : metadata.category
            };
            setResult(finalMetadata);

            // Resize thumbnail for DB to avoid max packet errors (and ensure persistent URL for videos)
            let dbThumbnail = singlePreview;

            if (singleFile.type.startsWith('video/') && videoFrames.length > 0) {
                // Use the first extracted frame as the thumbnail for videos (Persistent Data URL)
                dbThumbnail = videoFrames[0];
            } else {
                try {
                    if (singlePreview && singlePreview.length > 500000 && !singlePreview.startsWith('blob:')) { // Only resize if base64 and large
                        dbThumbnail = await resizeImage(singlePreview);
                    } else if (singlePreview.startsWith('blob:')) {
                        // Even for images, if it somehow is a blob (rare here but possible), try to convert or it will fail
                        // Usually images are read as DataURL in processSingleFile so this is safe for them.
                    }
                } catch (err) { console.warn("Thumbnail resize failed", err); }
            }

            // Save to History with enhanced tracking
            console.log('[DEBUG] Saving asset with exportFormat:', settings.exportFormat);
            const asset: Asset = {
                id: Math.random().toString(36).substring(2, 10),
                type: 'metadata',
                url: dbThumbnail, // Save optimized thumbnail
                prompt: metadata.title,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: finalMetadata, // Store full result WITH filename
                batchId: Date.now().toString(), // Add batchId
                modelUsed: metadata.modelUsed || preferredModel, // Track which model generated this
                generationSettings: { // Store generation settings snapshot
                    enableGrounding: settings.enableGrounding,
                    descLength: settings.descLength,
                    keywordCount: settings.keywordCount,
                    titleLength: settings.titleLength,
                    exportFormat: settings.exportFormat || 'csv', // Track chosen export format with fallback
                    selectedStockSites: settings.selectedStockSites || [] // Track targeted microstock platforms
                }
            };
            console.log('[DEBUG] Asset generationSettings:', asset.generationSettings);
            await dbService.add(asset);
            // Refresh history immediately so new item shows up
            console.log('[DEBUG] Refreshing history after asset creation...');
            const updatedHistory = await dbService.getByType('metadata');
            console.log('[DEBUG] History refreshed, total items:', updatedHistory.length);
            console.log('[DEBUG] Newest item exportFormat:', updatedHistory[updatedHistory.length - 1]?.generationSettings?.exportFormat);
            setHistoryItems(updatedHistory);

            // Determine specific model used (first preferred)
            // const preferredModel = getPreferredModels('thinking')[0] || ModelType.FLASH; // Already defined above

            const currentUser = authService.getCurrentUser();
            if (currentUser) {
                authService.incrementUserStat('metadataGenerated', 1, preferredModel);
                authService.logActivity(currentUser.uid, 'generate', `Generated metadata for: "${singleFile.name}"`);

                if (apiKeysToUse.length > 0) {
                    const cost = settings.enableGrounding ? 2 : 1;
                    // Use ACTUAL model used, or fallback
                    const finalModel = metadata.modelUsed || preferredModel;
                    console.log(`[Single] Generation complete. Model used: ${finalModel}`);
                    authService.deductCredits(currentUser.uid, cost, apiKeysToUse[0], finalModel);
                }

                // Automation: Sound & Auto-Download
                try {
                    const prefs = await authService.getUserPreferences(currentUser.uid);

                    // Sound
                    await playSound('success');

                    // Auto-Download
                    if (prefs?.uiConfig?.autoDownload) {
                        console.log('✅ Auto-downloading generated metadata...');
                        handleExport(settings.exportFormat || 'csv'); // Uses state result, might need delay/check
                        // Actually handleExport uses 'data' argument or 'result' state. 
                        // Since setResult(finalMetadata) is async, we should pass finalMetadata directly.
                        handleExport(settings.exportFormat || 'csv', finalMetadata);
                    }
                } catch (autoErr) {
                    console.error("Automation error:", autoErr);
                }
            }

        } catch (e: any) {
            if (e.message === 'Aborted' || abortControllerRef.current?.signal.aborted) {
                console.log('Single generation cancelled by user.');
                return; // Silent exit
            }
            console.error("Single generation error:", e);
            if (e.name === 'RateLimitExhaustedError' || e.message?.includes('RateLimitExhaustedError')) {
                setRateLimitError(true);
            } else {
                alert("Generation failed. Please try again.");
            }
        } finally {
            setIsGenerating(false);
        }
    };

    const handleRegenerateTitle = async () => {
        const currentUser = authService.getCurrentUser();
        if (!result || !currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        setIsRegeneratingTitle(true);
        try {
            const newTitle = await regenerateTitle(apiKeys, result.title, settings);
            setResult({ ...result, title: newTitle });

            // Deduct Credits & Log
            // For regeneration, we default to FLASH if not specified, but ideally regenerateTitle should also return modelUsed.
            // For now, let's keep regeneration simple or we'd need to update regenerateTitle too.
            // Assuming regeneration uses preferred 'thinking' model or flash.
            const modelUsed = getPreferredModels('thinking')[0] || ModelType.FLASH;

            if (currentUser) {
                await authService.deductCredits(currentUser.uid, 1, apiKeys[0], modelUsed);
                authService.logActivity(currentUser.uid, 'generate', `Regenerated title for asset`);
            }
        } catch (e) { console.error(e); }
        finally { setIsRegeneratingTitle(false); }
    }

    const handleExport = (format: 'txt' | 'json' | 'csv', data?: MetadataResult) => {
        const target = data || result;
        if (!target) return;

        let content = '';
        let type = 'text/plain';
        let ext = 'txt';

        const isAdobeEnabled = availableSites.some(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe'));

        if (format === 'json') {
            content = JSON.stringify(target, null, 2);
            type = 'application/json';
            ext = 'json';
        } else if (format === 'csv') {
            if (isAdobeEnabled) {
                // Adobe Stock Specific Format: Filename,Title,Keywords,Category
                // Note: Single file export usually doesn't need filename inside CSV content as much as batch, but per spec "Filename,Title,Keywords,Category"
                const filename = target.filename || singleFile?.name || "image.jpg";
                // Map category Name to ID
                let catId = target.category || '';
                if (target.category) {
                    const match = target.category.match(/\((\d+)\)/);
                    if (match) {
                        catId = match[1];
                    } else {
                        // Case-insensitive lookup
                        const catKey = Object.keys(ADOBE_STOCK_CATEGORIES).find(k => k.toLowerCase() === target.category!.toLowerCase());
                        if (catKey) {
                            catId = ADOBE_STOCK_CATEGORIES[catKey].toString();
                        }
                    }
                }
                content = `Filename,Title,Keywords,Category\n"${filename}","${target.title.replace(/"/g, '""')}","${target.commonKeywords.join(',')}",${catId ? `"${catId}"` : '""'}`;
            } else {
                content = `Title,Description,Keywords\n"${target.title.replace(/"/g, '""')}","${target.description.replace(/"/g, '""')}","${target.commonKeywords.join(',')}"`;
            }
            type = 'text/csv';
            ext = 'csv';
        } else {
            content = `Title: ${target.title}\n\nDescription:\n${target.description}\n\nKeywords:\n${target.commonKeywords.join(', ')}`;
            if (target.category) content += `\n\nCategory:\n${target.category}`;
        }

        const blob = new Blob([content], { type });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `metadata_${Date.now()}.${ext}`;
        link.click();
        setShowExportMenu(false);
    };

    const validatePlatformSelection = (): boolean => {
        if (!settings.selectedStockSites || settings.selectedStockSites.length === 0) {
            alert("Please select at least one target platform (e.g., Adobe Stock) to continue generation.");
            return false;
        }
        return true;
    };

    // --- Bulk Logic ---


    // --- Bulk Logic ---

    const handleBatchUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files) {
            addFiles(Array.from(e.target.files));
        }
    };

    const processBatch = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Validate Settings
        if (!validatePlatformSelection()) return;

        const allKeys = await authService.getUserApiKeys(currentUser.uid);
        const validKeys = allKeys
            .filter(k => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false)
            .map(k => k.fullKey);

        if (validKeys.length === 0) return; // Hook handles alert? Or we should alert here.

        // Start Context Batch
        await dbService.clearByType('metadata', true);
        await startBatch(settings, currentUser, validKeys, availableSites, siteSettings);
    };

    // stopBatch is imported from context

    const downloadBatchResults = () => {
        const completedItems = batchItems.filter(i => i.status === 'complete' && i.result);
        if (completedItems.length === 0) return;

        const format = settings.exportFormat || 'csv';
        const isAdobeEnabled = availableSites.some(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe'));

        let content = '';
        let mimeType = 'text/plain';
        let extension = 'txt';

        if (format === 'json') {
            const data = completedItems.map(i => ({
                filename: i.file.name,
                ...i.result
            }));
            content = JSON.stringify(data, null, 2);
            mimeType = 'application/json';
            extension = 'json';
        } else if (format === 'csv') {
            // Headers
            let headers = ['File Name', 'Title', 'Description', 'Keywords', 'iStock Tags'];
            if (isAdobeEnabled) headers = ['Filename', 'Title', 'Keywords', 'Category'];

            const rows = completedItems.map(i => {
                const res = i.result!;
                if (isAdobeEnabled) {
                    const row = [
                        `"${i.file.name}"`,
                        `"${res.title.replace(/"/g, '""')}"`,
                        `"${res.commonKeywords.join(', ')}"`
                    ];

                    // Adobe Category ID Logic
                    let catId = res.category || '';
                    if (res.category) {
                        const match = res.category.match(/\((\d+)\)/);
                        if (match) {
                            catId = match[1];
                        } else {
                            // Case-insensitive lookup
                            const catKey = Object.keys(ADOBE_STOCK_CATEGORIES).find(k => k.toLowerCase() === res.category!.toLowerCase());
                            if (catKey) {
                                catId = ADOBE_STOCK_CATEGORIES[catKey].toString();
                            }
                        }
                    }
                    row.push(`"${catId}"`);
                    return row;
                }
                return [
                    i.file.name,
                    `"${res.title.replace(/"/g, '""')}"`,
                    `"${res.description.replace(/"/g, '""')}"`,
                    `"${res.commonKeywords.join(', ')}"`,
                    `"${res.iStockTags.join(', ')}"`
                ];
            });

            content = [headers.join(','), ...rows.map(r => r.join(','))].join('\n');
            mimeType = 'text/csv;charset=utf-8;';
            extension = 'csv';
        } else {
            // TXT Format - Concatenated with Separators
            content = completedItems.map(i => {
                const res = i.result!;
                let itemText = `File: ${i.file.name}\nTitle: ${res.title}\nDescription: ${res.description}\nKeywords: ${res.commonKeywords.join(', ')}`;
                if (res.category) itemText += `\nCategory: ${res.category}`;
                return itemText;
            }).join('\n\n----------------------------------------\n\n');
            extension = 'txt';
        }

        const blob = new Blob([content], { type: mimeType });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = `stockforge_batch_${Date.now()}.${extension}`;
        link.click();
    };

    const resetBatch = async () => {
        // Stop any active processing
        stopBatch();
        // Clear all persistent storage
        await clearBatch();
    };

    const deleteBatchItem = (id: string) => {
        setBatchItems(prev => prev.filter(i => i.id !== id));
    };

    const downloadSingleBatchItem = (item: BatchItem) => {
        if (!item.result) return;

        // Reuse generic export logic but for single item
        handleExport(settings.exportFormat || 'csv', item.result);
    };

    const [regeneratingBatchItemId, setRegeneratingBatchItemId] = useState<string | null>(null);

    const handleRegenerateBatchItemTitle = async (item: BatchItem) => {
        if (!item.result) return;
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) {
            alert("No valid API keys available.");
            return;
        }

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        setRegeneratingBatchItemId(item.id);
        try {
            const newTitle = await regenerateTitle(apiKeys, item.result.title, settings);

            setBatchItems(prev => prev.map(p => p.id === item.id ? {
                ...p,
                result: { ...p.result!, title: newTitle }
            } : p));

            // Deduct Credits
            const modelUsed = getPreferredModels('thinking')[0] || ModelType.FLASH;
            await authService.deductCredits(currentUser.uid, 1, apiKeys[0], modelUsed);
            authService.logActivity(currentUser.uid, 'generate', `Regenerated title for batch item`);

        } catch (e) {
            console.error("Batch title regeneration failed", e);
            alert("Failed to regenerate title. Please try again.");
        } finally {
            setRegeneratingBatchItemId(null);
        }
    };



    // Helper to generate content string based on format
    const getMetadataContent = (asset: Asset): { content: string, format: string, extension: string } => {
        const format = asset.generationSettings?.exportFormat || settings.exportFormat || 'json';
        const isAdobeEnabled = availableSites.some(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe')); // Check current settings or asset settings?
        // Asset might store which sites it was for, but 'availableSites' check is runtime.
        // Let's use the current 'isAdobeEnabled' logic if possible, or fallback.
        // The asset.generationSettings is best.

        const m = asset.metadata;
        if (!m) return { content: '', format: 'json', extension: 'json' };

        if (asset.metadata?.csvContent) {
            return { content: asset.metadata.csvContent, format: 'csv', extension: 'csv' };
        }

        if (format === 'csv') {
            let csvContent = '';

            // Keyword Aggregation logic
            let allKeywords: string[] = [];
            if (Array.isArray(m.commonKeywords)) allKeywords.push(...m.commonKeywords);
            if (Array.isArray(m.longTailKeywords)) allKeywords.push(...m.longTailKeywords);
            if (allKeywords.length === 0) {
                if (Array.isArray(m.keywords)) allKeywords.push(...m.keywords);
                else if (typeof m.keywords === 'string') allKeywords.push(...m.keywords.split(',').map((k: string) => k.trim()));
            }
            const keywordsString = allKeywords.join(', ');
            const keywords = `"${keywordsString.replace(/"/g, '""')}"`;

            if (isAdobeEnabled) {
                const filename = `"${(m.filename || asset.title || 'asset')}"`;
                const title = `"${(m.title || '').replace(/"/g, '""')}"`;
                let catName = m.category || '';
                if (catName.includes('(')) catName = catName.split('(')[0].trim();
                const catId = ADOBE_STOCK_CATEGORIES[catName] || catName;
                const category = `"${catId}"`;

                csvContent = "Filename,Title,Keywords,Category\n";
                csvContent += `${filename},${title},${keywords},${category}\n`;
            } else {
                const filename = `"${(m.filename || asset.title || 'asset')}"`;
                const title = `"${(m.title || '').replace(/"/g, '""')}"`;
                const desc = `"${(m.description || '').replace(/"/g, '""')}"`;
                // Add iStock Tags if available? Library didn't but maybe we should?
                // Library just did standard.
                csvContent = "Filename,Title,Description,Keywords\n";
                csvContent += `${filename},${title},${desc},${keywords}\n`;
            }

            return { content: csvContent, format: 'csv', extension: 'csv' };
        }

        if (format === 'txt') {
            const txtContent = `Title: ${m.title}\n\nDescription:\n${m.description}\n\nKeywords:\n${m.commonKeywords?.join(', ')}, ${m.longTailKeywords?.join(', ') || ''}`;
            return { content: txtContent, format: 'txt', extension: 'txt' };
        }

        // Default JSON
        return { content: JSON.stringify(asset.metadata, null, 2), format: 'json', extension: 'json' };
    };

    // Helper to generate download payload
    const getDownloadPayload = (asset: Asset) => {
        const { content, format, extension } = getMetadataContent(asset);
        const mimeType = format === 'csv' ? 'text/csv;charset=utf-8;' : format === 'txt' ? 'text/plain' : 'application/json';
        return {
            url: URL.createObjectURL(new Blob([content], { type: mimeType })),
            filename: `metadata_${asset.id}.${extension}`
        };
    };

    const [isZipping, setIsZipping] = useState(false);

    const handleDownloadHistoryGeneric = async (forceSeparate: boolean) => {
        if (historyItems.length === 0) return;

        setIsZipping(true);
        try {
            const itemsToDownload: { url: string, filename: string }[] = [];

            for (const asset of historyItems) {
                // If asset is metadata type (which they should be in Metadata Generator history)
                if (asset.type === 'metadata') {
                    itemsToDownload.push(getDownloadPayload(asset));
                } else {
                    // Fallback if mixed types (review items shouldn't be here)
                    const ext = asset.type === 'video' ? 'mp4' : 'png';
                    const filename = asset.title ? `${asset.title}.${ext}` : `${asset.type}_${asset.id}.${ext}`;
                    itemsToDownload.push({ url: asset.url, filename });
                }
            }

            const zipName = `history_export_${Date.now()}`;
            await downloadItems(itemsToDownload, zipName, forceSeparate);

        } catch (e) {
            console.error("Download error", e);
        } finally {
            setIsZipping(false);
        }
    };



    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">
            {/* Header */}
            <div className="mb-8 shrink-0">
                <div className="flex justify-between items-start">
                    <div>
                        <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                            <ScanEye className="text-amber-500 w-8 h-8" /> Metadata AI Generator
                        </h1>
                        <p className="text-slate-600 dark:text-slate-400 mt-2">Generate optimized titles, descriptions, and keywords for microstock platforms using Gemini Vision.</p>
                    </div>
                    {/* Buttons removed from header */}
                </div>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Tabs */}
            <div className={`flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-8 shrink-0 transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>
                <div className="flex w-full md:w-auto overflow-x-auto">
                    {['single', 'bulk', 'history'].map((tab) => (
                        <button
                            key={tab}
                            onClick={() => setActiveTab(tab as any)}
                            className={`px-6 py-3 text-sm font-bold transition-all relative capitalize flex items-center gap-2 ${activeTab === tab ? 'text-amber-600 dark:text-amber-400' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                }`}
                        >
                            {tab === 'single' ? 'Single Asset' : tab === 'bulk' ? 'Bulk Processing' : 'History'}
                            {activeTab === tab && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in"></div>}
                        </button>
                    ))}
                </div>

                {/* Right side controls - horizontal layout */}
                <div className="flex flex-row gap-2 items-center my-2">
                    {activeTab === 'history' && historyItems.length > 0 && (
                        <>
                            <button
                                onClick={() => handleDownloadHistoryGeneric(true)}
                                disabled={isZipping}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-slate-700 text-slate-400 hover:bg-slate-900 hover:text-white font-bold rounded-xl transition text-[10px] uppercase"
                                title="Download all assets as separate files"
                            >
                                {isZipping ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                SEPARATE ALL
                            </button>
                            <button
                                onClick={() => handleDownloadHistoryGeneric(false)}
                                disabled={isZipping}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 font-bold rounded-xl transition text-[10px] uppercase"
                                title="Download all assets in a single ZIP archive"
                            >
                                {isZipping ? <Loader2 size={14} className="animate-spin" /> : <FileDown size={14} />}
                                ZIP ALL
                            </button>
                        </>
                    )}

                    {/* Active Model Display */}
                    {activeTab !== 'history' && (
                        <div className="hidden md:flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg my-2 mr-2">
                            <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                                <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                            </span>
                            <div className="flex gap-1">
                                {activeModels.map((model, idx) => (
                                    <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                        {model}
                                    </span>
                                ))}
                                {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                            </div>
                        </div>
                    )}
                </div>
            </div>

            <div className={`flex-1 transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>

                {/* SINGLE MODE */}
                {activeTab === 'single' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full pb-20">
                        {/* Left: Input */}
                        <div className="lg:col-span-5 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Upload Asset</label>
                                <div
                                    onClick={() => document.getElementById('single-upload')?.click()}
                                    onDragOver={handleDragOver}
                                    onDragLeave={handleDragLeave}
                                    onDrop={handleDrop}
                                    className={`relative border-2 border-dashed rounded-xl h-64 flex flex-col items-center justify-center cursor-pointer transition overflow-hidden group 
                                    ${isDragging
                                            ? 'border-amber-500 bg-amber-50 dark:bg-amber-900/20 scale-[1.02]'
                                            : 'border-slate-300 dark:border-slate-700 bg-slate-50 dark:bg-slate-900 hover:bg-slate-100 dark:hover:bg-slate-900/80'
                                        }`}
                                >
                                    {singlePreview ? (
                                        <>
                                            <div className="relative w-full h-full group flex items-center justify-center bg-slate-100 dark:bg-slate-900 overflow-hidden rounded-xl">
                                                {singleFile?.type.startsWith('video/') && videoFrames.length > 0 ? (
                                                    <div className="flex w-full h-full">
                                                        {videoFrames.map((frame, idx) => {
                                                            let label = 'Middle';
                                                            if (idx === 0) label = 'Start';
                                                            else if (idx === videoFrames.length - 1) label = 'End';

                                                            return (
                                                                <div key={idx} className="flex-1 h-full border-r border-slate-300 dark:border-slate-700 last:border-none relative min-w-0">
                                                                    <img src={frame} className="w-full h-full object-cover" alt={`Frame ${idx + 1}`} />
                                                                    <span className="absolute bottom-2 left-2 px-2 py-0.5 bg-black/60 text-white text-[10px] rounded backdrop-blur-sm font-bold uppercase">
                                                                        {label}
                                                                    </span>
                                                                </div>
                                                            );
                                                        })}
                                                    </div>
                                                ) : (
                                                    <img src={singlePreview} className="w-full h-full object-contain opacity-90" />
                                                )}

                                                <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition flex items-center justify-center pointer-events-none">
                                                    <span className="px-4 py-2 bg-white dark:bg-slate-900 rounded-lg text-xs font-bold text-slate-900 dark:text-white shadow-xl pointer-events-auto">
                                                        {singleFile?.type.startsWith('video/') ? 'Change Video' : 'Change Image'}
                                                    </span>
                                                </div>
                                            </div>
                                        </>
                                    ) : (
                                        <>
                                            <Upload className={`mb-3 w-10 h-10 transition-colors ${isDragging ? 'text-amber-500 scale-110' : 'text-slate-400 dark:text-slate-500 group-hover:text-amber-500 dark:group-hover:text-amber-400'}`} />
                                            <span className={`text-sm font-bold ${isDragging ? 'text-amber-600 dark:text-amber-400' : 'text-slate-500 dark:text-slate-400'}`}>
                                                {isDragging ? 'Drop to Upload!' : 'Click or Drag to Upload'}
                                            </span>
                                            <span className="text-[10px] text-slate-400 dark:text-slate-600 mt-1">JPG, PNG, SVG, WEBP, MP4, MOV supported</span>
                                        </>
                                    )}
                                    <input id="single-upload" type="file" className="hidden" onChange={handleSingleUpload} accept="image/*,video/*,.svg" />
                                </div>
                            </div>

                            <div className="flex gap-3 mt-4">
                                <button
                                    onClick={generateSingle}
                                    disabled={isGenerating || !singleFile || !hasApiKey}
                                    className="flex-1 py-4 bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-400 hover:to-amber-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-amber-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed group"
                                >
                                    {isGenerating ? <RefreshCw className="animate-spin" size={20} /> : <Sparkles className="group-hover:scale-110 transition-transform" size={20} />}
                                    {isGenerating ? (settings.enableGrounding ? 'Analyzing...' : 'Analyzing...') : 'Start Generation'}
                                    {!isGenerating && (
                                        <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">
                                            {settings.enableGrounding ? '2 API Calls' : '1 API Call'}
                                        </span>
                                    )}
                                </button>

                                {/* Stop Button for Single Generation */}
                                {isGenerating && (
                                    <button
                                        onClick={() => {
                                            if (abortControllerRef.current) {
                                                abortControllerRef.current.abort();
                                                abortControllerRef.current = null;
                                            }
                                            setIsGenerating(false);
                                        }}
                                        title="Stop Generation"
                                        className="px-4 py-4 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 hover:bg-red-200 dark:hover:bg-red-800/40 font-bold rounded-xl transition flex items-center justify-center"
                                    >
                                        <Square size={20} className="fill-current" />
                                    </button>
                                )}

                                <button
                                    onClick={() => {
                                        setSingleFile(null);
                                        setSinglePreview(null);
                                        setResult(null);
                                        setVideoFrames([]);
                                        if (document.getElementById('single-upload') as HTMLInputElement) {
                                            (document.getElementById('single-upload') as HTMLInputElement).value = '';
                                        }
                                    }}
                                    title="Reset / Clear"
                                    className="px-4 py-4 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-300 font-bold rounded-xl transition flex items-center justify-center"
                                >
                                    <RotateCcw size={20} />
                                </button>

                                <button
                                    onClick={() => handleExport(settings.exportFormat)}
                                    disabled={!result}
                                    title="Download Result"
                                    className="px-6 py-4 bg-slate-800 dark:bg-slate-700 hover:bg-slate-700 dark:hover:bg-slate-600 text-white font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed uppercase text-xs tracking-wide"
                                >
                                    <Download size={18} />
                                    <span className="hidden sm:inline">Download CSV</span>
                                </button>
                            </div>

                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div className="flex justify-between items-center">
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300">Context (Optional)</label>
                                    <div className="flex items-center gap-2">
                                        <div className="flex items-center gap-1.5 mr-2 group relative">
                                            <input type="checkbox" checked={settings.enableGrounding} onChange={e => setSettings({ ...settings, enableGrounding: e.target.checked })} className="rounded bg-slate-200 dark:bg-slate-700 border-transparent text-amber-500 focus:ring-0 w-3.5 h-3.5" />
                                            <span className="text-xs font-bold text-slate-600 dark:text-slate-400">Search Grounding</span>
                                            <div className="relative">
                                                <Info size={12} className="text-slate-400 hover:text-amber-500 cursor-help" />
                                                <div className="absolute right-0 top-full mt-2 w-80 p-2 bg-slate-800 text-white text-[10px] rounded-lg opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl border border-slate-700">
                                                    <p className="mb-2"><span className="text-amber-400 font-bold">IF ENABLED:</span> Costs you 1 API Credit more!</p>
                                                    <p className="text-slate-300 text-[9px] leading-relaxed">When enabled, the AI searches the web for real-time information about your image subject before generating metadata. This provides more accurate, factual descriptions and keywords based on current data, making your content more discoverable and relevant.</p>
                                                </div>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => setSettings({ ...settings, useFilenameMode: !settings.useFilenameMode })}
                                            className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-xs font-bold transition border ${settings.useFilenameMode
                                                ? 'bg-amber-100 text-amber-700 dark:bg-amber-900/30 dark:text-amber-400 border-amber-200 dark:border-amber-800'
                                                : 'bg-slate-100 text-slate-600 dark:bg-slate-900 dark:text-slate-400 border-slate-200 dark:border-slate-800 hover:bg-slate-200 dark:hover:bg-slate-800'
                                                }`}
                                        >
                                            <FileText size={14} />
                                            <span>Filename Mode: {settings.useFilenameMode ? 'ON' : 'OFF'}</span>
                                        </button>
                                        <div className="relative group">
                                            <Info size={14} className="text-slate-400 hover:text-amber-500 cursor-help" />
                                            <div className="absolute right-0 top-full mt-2 w-64 p-3 bg-slate-800 text-white text-xs rounded-xl opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl border border-slate-700">
                                                <p className="font-bold mb-1 text-amber-400">Filename-Only Mode</p>
                                                <p>When enabled, the AI prioritizes the filename (e.g., "sunset_beach.jpg") as the primary source of truth. Useful if visual analysis misidentifies the subject.</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Target Platforms (Moved to Top & Highlighted) */}
                                <div className="p-3 rounded-xl border border-amber-200 dark:border-amber-900/50 bg-amber-50 dark:bg-amber-900/10 shadow-sm relative group">
                                    <div className="absolute top-0 right-0 w-16 h-16 bg-gradient-to-br from-amber-200/20 to-transparent rounded-bl-full rounded-tr-xl pointer-events-none transition-opacity group-hover:opacity-100 opacity-50"></div>

                                    <div className="flex justify-between items-center text-xs mb-2 relative z-10">
                                        <div className="flex items-center gap-2">
                                            <span className="font-bold text-amber-800 dark:text-amber-400 flex items-center gap-1.5">
                                                <Globe size={14} className="text-amber-600 dark:text-amber-500" /> Target Platforms
                                                <InfoTooltip text="Select which stock platforms you are uploading to. Formats metadata specifically for them." />
                                            </span>
                                            <div className="flex gap-2">
                                                <span className="px-1.5 py-0.5 rounded bg-white/50 dark:bg-black/20 text-[10px] text-amber-700/70 dark:text-amber-400/70 font-mono border border-amber-100 dark:border-amber-800/30">
                                                    Selected: {settings.selectedStockSites?.length || 0}/{availableSites.length}
                                                </span>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => {
                                                const allIds = availableSites.map(s => s.id);
                                                const isAllSelected = allIds.every(id => settings.selectedStockSites?.includes(id));
                                                setSettings({ ...settings, selectedStockSites: isAllSelected ? [allIds[0]] : allIds });
                                            }}
                                            className="text-[10px] font-bold text-amber-600 dark:text-amber-400 hover:text-amber-700 dark:hover:text-amber-300 uppercase underline decoration-dashed underline-offset-2 transition-colors"
                                        >
                                            {availableSites.length > 0 && availableSites.every(s => settings.selectedStockSites?.includes(s.id)) ? 'Uncheck All' : 'Check All'}
                                        </button>
                                    </div>
                                    <div className="grid grid-cols-5 gap-2 max-h-32 overflow-y-auto custom-scrollbar p-1 relative z-10">
                                        {availableSites.length === 0 && <p className="text-[10px] text-slate-400 italic col-span-5">No sites configured.</p>}
                                        {availableSites.map(site => (
                                            <div key={site.id} className="flex items-center gap-2 overflow-hidden bg-white/50 dark:bg-slate-900/50 p-1.5 rounded-lg border border-amber-100 dark:border-amber-900/30 hover:border-amber-300 dark:hover:border-amber-700 transition-colors cursor-pointer" onClick={() => {
                                                const current = settings.selectedStockSites || [];
                                                const isSelected = current.includes(site.id);

                                                if (isSelected && current.length <= 1) return;

                                                const newer = !isSelected
                                                    ? [...current, site.id]
                                                    : current.filter(id => id !== site.id);
                                                setSettings({ ...settings, selectedStockSites: newer });
                                            }}>
                                                <div className={`w-3 h-3 rounded-full border flex items-center justify-center transition-colors ${settings.selectedStockSites?.includes(site.id) ? 'bg-amber-500 border-amber-500' : 'border-slate-300 dark:border-slate-600 bg-white dark:bg-slate-800'}`}>
                                                    {settings.selectedStockSites?.includes(site.id) && <Check size={8} className="text-white" strokeWidth={4} />}
                                                </div>
                                                <span className={`text-[10px] truncate font-medium ${settings.selectedStockSites?.includes(site.id) ? 'text-amber-900 dark:text-amber-200' : 'text-slate-600 dark:text-slate-400'}`} title={site.name}>{site.name}</span>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                                <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-700 space-y-3">
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Title Length
                                                <InfoTooltip text="Sets the target character length for the generated title." />
                                            </span>
                                            <span className="font-bold">{settings.titleLength}</span>
                                        </div>
                                        <input type="range" min="40" max="150" value={settings.titleLength} onChange={e => setSettings({ ...settings, titleLength: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Description Length
                                                <InfoTooltip text="Sets the target character length for the generated description." />
                                            </span>
                                            <span className="font-bold">{settings.descLength}</span>
                                        </div>
                                        <input type="range" min="100" max="300" step="10" value={settings.descLength} onChange={e => setSettings({ ...settings, descLength: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Keywords
                                                <InfoTooltip text="Number of keywords to generate (10-50)." />
                                            </span>
                                            <span className="font-bold">{settings.keywordCount}</span>
                                        </div>
                                        <input type="range" min="10" max="50" value={settings.keywordCount} onChange={e => setSettings({ ...settings, keywordCount: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>







                                    {/* Keyword Format Selection */}
                                    <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                        <div className="flex justify-between text-xs mb-2">
                                            <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                                Keyword Format
                                                <InfoTooltip text="Single (e.g., 'sunset'), Double (e.g., 'sunset beach'), or Mixed." />
                                            </span>
                                        </div>
                                        <div className="flex bg-slate-200 dark:bg-slate-800 p-1 rounded-lg">
                                            {(['single', 'double', 'mixed'] as const).map(type => (
                                                <button
                                                    key={type}
                                                    onClick={() => setSettings({ ...settings, keywordType: type })}
                                                    className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition whitespace-nowrap ${settings.keywordType === type
                                                        ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                        : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                        }`}
                                                >
                                                    {type === 'single' ? 'Single' : type === 'double' ? 'Double' : 'Mixed'}
                                                </button>
                                            ))}
                                        </div>
                                    </div>

                                    {/* Video Frame Source (Visible Only if Video) */}
                                    {singleFile?.type.startsWith('video/') && (
                                        <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                            <span className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1 mb-1">
                                                Video Frame Source <InfoTooltip text="Select which frame from the video to use for visual analysis." />
                                            </span>
                                            {videoFrames.length >= 3 ? (
                                                <div className="grid grid-cols-3 gap-4 bg-slate-200 dark:bg-slate-800 p-2 rounded-lg">
                                                    {(['start', 'middle', 'end'] as const).map((opt, idx) => {
                                                        const frameIndex = opt === 'start' ? 0 : opt === 'middle' ? Math.floor(videoFrames.length / 2) : videoFrames.length - 1;
                                                        const frameSrc = videoFrames[frameIndex];
                                                        const isSelected = (settings.videoFrameOptions || ['start']).includes(opt);
                                                        return (
                                                            <div
                                                                key={opt}
                                                                onClick={() => {
                                                                    const current = settings.videoFrameOptions || ['start'];
                                                                    let next = [];
                                                                    if (current.includes(opt)) {
                                                                        next = current.filter(o => o !== opt);
                                                                        if (next.length === 0) next = [opt]; // Prevent empty
                                                                    } else {
                                                                        next = [...current, opt];
                                                                    }
                                                                    setSettings({ ...settings, videoFrameOptions: next });
                                                                }}
                                                                className={`cursor-pointer relative group rounded-md overflow-hidden border-2 transition-all ${isSelected
                                                                    ? 'border-amber-500 shadow-md scale-105 z-10'
                                                                    : 'border-transparent opacity-70 hover:opacity-100 hover:border-slate-400'
                                                                    }`}
                                                                title={`Toggle ${opt} frame`}
                                                            >
                                                                <img src={frameSrc} alt={opt} className="w-full h-16 object-cover bg-black" />
                                                                <div className={`absolute bottom-0 inset-x-0 py-0.5 text-[8px] font-bold text-center uppercase ${isSelected ? 'bg-amber-500 text-white' : 'bg-black/60 text-white'}`}>
                                                                    {opt === 'start' ? 'Start' : opt === 'middle' ? 'Middle' : 'End'}
                                                                </div>
                                                                {isSelected && <div className="absolute top-1 right-1 bg-amber-500 rounded-full p-0.5"><Check size={8} className="text-white" /></div>}
                                                            </div>
                                                        );
                                                    })}
                                                </div>
                                            ) : (
                                                <div className="flex Gap-1 bg-slate-200 dark:bg-slate-800 p-1 rounded-lg gap-1">
                                                    {(['start', 'middle', 'end'] as const).map(opt => {
                                                        const isSelected = (settings.videoFrameOptions || ['start']).includes(opt);
                                                        return (
                                                            <button
                                                                key={opt}
                                                                onClick={() => {
                                                                    const current = settings.videoFrameOptions || ['start'];
                                                                    let next = [];
                                                                    if (current.includes(opt)) {
                                                                        next = current.filter(o => o !== opt);
                                                                        if (next.length === 0) next = [opt];
                                                                    } else {
                                                                        next = [...current, opt];
                                                                    }
                                                                    setSettings({ ...settings, videoFrameOptions: next });
                                                                }}
                                                                className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition-all shadow-sm ${isSelected
                                                                    ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                                    : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                                    }`}
                                                                title={opt === 'start' ? 'Use 3rd Frame (Start)' : opt === 'middle' ? 'Use Middle Frame' : 'Use Last Frame'}
                                                            >
                                                                {opt === 'start' ? 'Start' : opt === 'middle' ? 'Middle' : 'End'}
                                                            </button>
                                                        );
                                                    })}
                                                </div>
                                            )}
                                        </div>
                                    )}

                                    {/* Export Format Selection */}
                                    <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                        <div className="flex justify-between text-xs mb-2">
                                            <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                                Export Format
                                                <InfoTooltip text="Choose the file format for your metadata download." />
                                            </span>
                                        </div>
                                        <div className="flex bg-slate-200 dark:bg-slate-800 p-1 rounded-lg">
                                            {(['csv', 'json', 'txt'] as const).map(fmt => (
                                                <button
                                                    key={fmt}
                                                    onClick={() => setSettings({ ...settings, exportFormat: fmt })}
                                                    className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition whitespace-nowrap flex items-center justify-center gap-1 ${settings.exportFormat === fmt
                                                        ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                        : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                        }`}
                                                >
                                                    {fmt === 'txt' && <FileText size={10} />}
                                                    {fmt === 'json' && <FileJson size={10} />}
                                                    {fmt === 'csv' && <FileType size={10} />}
                                                    {fmt.toUpperCase()}
                                                    {fmt === 'csv' && (
                                                        <div className="relative group ml-1">
                                                            <Info size={10} className="text-red-500" />
                                                            <div className="absolute top-full mt-2 left-1/2 -translate-x-1/2 w-max max-w-[300px] p-2 bg-slate-800 text-white text-[10px] rounded-lg opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl normal-case text-center">
                                                                Required option for Microstock websites
                                                            </div>
                                                        </div>
                                                    )}
                                                </button>
                                            ))}
                                        </div>
                                    </div>
                                </div>

                                <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-700 space-y-2">
                                    <div className="flex justify-between text-xs">
                                        <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                            Include in Description
                                            <InfoTooltip text="Add specific details, locations, or subject names that must be included in the description." />
                                        </span>
                                    </div>
                                    <input
                                        type="text"
                                        placeholder="Location, specific event, or subject names..."
                                        value={contextDesc} onChange={e => setContextDesc(e.target.value)}
                                        className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500 transition-colors"
                                    />
                                </div>
                                <div className="grid grid-cols-2 gap-3">
                                    <div className="space-y-1">
                                        <label className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1">
                                            <PlusCircle size={12} className="text-blue-500" /> Include Keywords
                                            <InfoTooltip text="Force these keywords to be included in the result." />
                                        </label>
                                        <TagInput
                                            placeholder="Type & Enter..."
                                            tags={contextKeywords}
                                            onTagsChange={setContextKeywords}
                                            color="blue"
                                        />
                                    </div>
                                    <div className="space-y-1">
                                        <label className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1">
                                            <MinusCircle size={12} className="text-red-500" /> Exclude Keywords
                                            <InfoTooltip text="Ensure these words never appear in the result." />
                                        </label>
                                        <TagInput
                                            placeholder="Type & Enter..."
                                            tags={excludeKeywords}
                                            onTagsChange={setExcludeKeywords}
                                            color="red"
                                        />
                                    </div>
                                </div>
                            </div>

                            {/* Buttons moved to top */}
                        </div>

                        {/* Right: Results */}
                        <div className="lg:col-span-7 bg-white dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 min-h-[600px] flex flex-col relative shadow-sm dark:shadow-none">
                            {result ? (
                                <div className="space-y-6 pr-2">
                                    <div className="flex justify-between items-start">
                                        <h3 className="text-lg font-bold text-slate-900 dark:text-white flex items-center gap-2">
                                            <CheckCircle2 className="text-green-500" size={20} /> Analysis Complete
                                        </h3>
                                        <div className="relative">
                                            <button
                                                onClick={() => handleExport(settings.exportFormat)}
                                                className="p-2 bg-slate-100 dark:bg-slate-900 hover:bg-slate-200 dark:hover:bg-slate-800 rounded-lg text-slate-700 dark:text-slate-300 transition border border-slate-200 dark:border-slate-800"
                                                title={`Download as ${settings.exportFormat.toUpperCase()}`}
                                            >
                                                <Download size={16} />
                                            </button>
                                        </div>
                                    </div>

                                    {/* Filename (Adobe Only) */}
                                    {isAdobeEnabled && (
                                        <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 group relative mb-4">
                                            <div className="flex justify-between items-center mb-2">
                                                <span className="text-xs font-bold text-green-500 uppercase">Filename</span>
                                                <button onClick={() => navigator.clipboard.writeText(result.filename || '')} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                            </div>
                                            <textarea
                                                value={result.filename || ''}
                                                onChange={(e) => setResult({ ...result, filename: e.target.value })}
                                                onInput={(e) => { const el = e.target as HTMLTextAreaElement; el.style.height = 'auto'; el.style.height = el.scrollHeight + 'px'; }}
                                                data-autosize="true"
                                                className="w-full bg-transparent text-sm font-bold text-slate-800 dark:text-slate-200 leading-relaxed outline-none resize-none border-none p-0 focus:ring-0 overflow-hidden"
                                                rows={1}
                                            />
                                        </div>
                                    )}

                                    {/* Title */}
                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 group relative">
                                        <div className="flex justify-between items-center mb-2">
                                            <span className="text-xs font-bold text-slate-500 uppercase">Title ({result.title.length} chars)</span>
                                            <div className="flex gap-1">
                                                <button
                                                    onClick={handleRegenerateTitle}
                                                    disabled={isRegeneratingTitle}
                                                    className={`p-1.5 text-slate-400 hover:text-amber-500 transition rounded-md ${isRegeneratingTitle ? 'animate-spin text-amber-500 cursor-not-allowed' : ''}`}
                                                    title="Regenerate Title (1 API Call)"
                                                >
                                                    <RefreshCw size={12} />
                                                </button>
                                                <button onClick={() => navigator.clipboard.writeText(result.title)} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                            </div>
                                        </div>
                                        <textarea
                                            value={result.title}
                                            onChange={(e) => setResult({ ...result, title: e.target.value })}
                                            onInput={(e) => { const el = e.target as HTMLTextAreaElement; el.style.height = 'auto'; el.style.height = el.scrollHeight + 'px'; }}
                                            data-autosize="true"
                                            className="w-full bg-transparent text-sm font-bold text-slate-800 dark:text-slate-200 leading-relaxed outline-none resize-none border-none p-0 focus:ring-0 overflow-hidden"
                                            rows={1}
                                        />
                                    </div>

                                    {/* Description */}
                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800">
                                        <div className="flex justify-between items-center mb-2">
                                            <span className="text-xs font-bold text-slate-500 uppercase">Description ({result.description.length} chars)</span>
                                            <button onClick={() => navigator.clipboard.writeText(result.description)} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                        </div>
                                        <textarea
                                            value={result.description}
                                            onChange={(e) => setResult({ ...result, description: e.target.value })}
                                            onInput={(e) => { const el = e.target as HTMLTextAreaElement; el.style.height = 'auto'; el.style.height = el.scrollHeight + 'px'; }}
                                            data-autosize="true"
                                            className="w-full bg-transparent text-sm text-slate-700 dark:text-slate-300 leading-relaxed outline-none resize-none border-none p-0 focus:ring-0 overflow-hidden"
                                            rows={1}
                                        />
                                    </div>

                                    {/* Keywords */}
                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800">
                                        <div className="flex justify-between items-center mb-3">
                                            <span className="text-xs font-bold text-slate-500 uppercase">Keywords ({result.commonKeywords.length + result.longTailKeywords.length})</span>
                                            <button onClick={() => navigator.clipboard.writeText([...result.commonKeywords, ...result.longTailKeywords].join(', '))} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy All"><Copy size={12} /></button>
                                        </div>
                                        <textarea
                                            value={[...result.commonKeywords, ...result.longTailKeywords].join(', ')}
                                            onChange={(e) => {
                                                const keywords = e.target.value.split(',').map(k => k.trim()).filter(k => k);
                                                setResult({
                                                    ...result,
                                                    commonKeywords: keywords,
                                                    longTailKeywords: [] // Merge all into common for simplicity when editing
                                                });
                                            }}
                                            onInput={(e) => { const el = e.target as HTMLTextAreaElement; el.style.height = 'auto'; el.style.height = el.scrollHeight + 'px'; }}
                                            data-autosize="true"
                                            className="w-full bg-transparent text-xs font-medium text-slate-700 dark:text-slate-300 leading-relaxed outline-none resize-none border-none p-0 focus:ring-0 overflow-hidden"
                                            rows={1}
                                        />

                                    </div>

                                    {/* Category (Adobe Only) */}
                                    {isAdobeEnabled && (
                                        <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 mt-4">
                                            <div className="flex justify-between items-center mb-2">
                                                <span className="text-xs font-bold text-slate-500 uppercase">Category</span>
                                                <button onClick={() => navigator.clipboard.writeText(result.category || '')} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                            </div>
                                            <textarea
                                                value={result.category || ''}
                                                onChange={(e) => setResult({ ...result, category: e.target.value })}
                                                onInput={(e) => { const el = e.target as HTMLTextAreaElement; el.style.height = 'auto'; el.style.height = el.scrollHeight + 'px'; }}
                                                data-autosize="true"
                                                className="w-full bg-transparent text-sm font-bold text-slate-800 dark:text-slate-200 leading-relaxed outline-none resize-none border-none p-0 focus:ring-0 overflow-hidden"
                                                rows={1}
                                            />
                                        </div>
                                    )}

                                    {/* Sources (If Grounding Enabled) */}
                                    {result.sources && (
                                        <div className="p-3 bg-amber-50 dark:bg-amber-900/10 border border-amber-100 dark:border-amber-500/20 rounded-xl">
                                            <button onClick={() => setShowSources(!showSources)} className="flex items-center justify-between w-full text-xs font-bold text-amber-700 dark:text-amber-500 uppercase">
                                                <span className="flex items-center gap-2"><Globe size={12} /> Grounding Sources ({result.sources.length})</span>
                                                {showSources ? <ChevronUp size={12} /> : <ChevronDown size={12} />}
                                            </button>
                                            {showSources && (
                                                <ul className="mt-2 space-y-1">
                                                    {result.sources.map((s, i) => (
                                                        <li key={i} className="text-[10px] text-amber-900/70 dark:text-amber-200/70 truncate">
                                                            <a href={s.uri} target="_blank" rel="noreferrer" className="hover:underline">• {s.title}</a>
                                                        </li>
                                                    ))}
                                                </ul>
                                            )}
                                        </div>
                                    )}
                                </div>
                            ) : (
                                <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                    <ScanEye size={48} className="mb-4 opacity-20" />
                                    <p className="text-sm font-medium">Ready to Analyze</p>
                                    <p className="text-xs mt-1">Upload an image to generate metadata.</p>
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* BULK MODE */}
                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full pb-20">
                        {/* Left: Input & Settings */}
                        <div className="lg:col-span-5 space-y-6 pr-2">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Upload Assets (Max 1000)</label>
                                <div className="p-8 border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl text-center hover:bg-slate-50 dark:hover:bg-slate-900/50 transition cursor-pointer relative group">
                                    <div className="absolute inset-0 bg-amber-500/5 opacity-0 group-hover:opacity-100 transition"></div>
                                    <Layers size={32} className="mx-auto text-slate-400 mb-2 group-hover:scale-110 transition-transform duration-300 group-hover:text-amber-500" />
                                    <p className="text-sm font-bold text-slate-600 dark:text-slate-400 group-hover:text-amber-600 dark:group-hover:text-amber-400 transition-colors">Click or Drag to Upload Bulk Assets</p>
                                    <p className="text-xs text-slate-500 mt-1">Up to 1000 images, videos, or SVGs</p>
                                    <input type="file" multiple onChange={handleBatchUpload} className="absolute inset-0 opacity-0 cursor-pointer" accept="image/*,video/*,.svg" />
                                </div>

                                <div className="flex gap-3 mt-4">
                                    <button
                                        onClick={processBatch}
                                        disabled={isBatchProcessing || batchItems.length === 0 || !hasApiKey}
                                        className="flex-1 py-4 bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-400 hover:to-amber-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-amber-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed group"
                                    >
                                        {isBatchProcessing ? <RefreshCw className="animate-spin" size={20} /> : <Sparkles className="group-hover:scale-110 transition-transform" size={20} />}
                                        Start Batch
                                        {!isBatchProcessing && batchItems.length > 0 && (
                                            <>
                                                <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">
                                                    {batchItems.length} {batchItems.length === 1 ? 'Item' : 'Items'}
                                                </span>
                                                <span className="px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">
                                                    {(settings.enableGrounding ? 2 : 1) * batchItems.length} {((settings.enableGrounding ? 2 : 1) * batchItems.length) === 1 ? 'API Call' : 'API Calls'}
                                                </span>
                                            </>
                                        )}
                                    </button>

                                    {/* Pause/Resume Button */}
                                    {isBatchProcessing && (
                                        <button
                                            onClick={togglePause}
                                            title={isBatchPaused ? "Resume Batch" : "Pause Batch"}
                                            className={`px-4 py-4 ${isBatchPaused ? 'bg-green-500 hover:bg-green-600 text-white' : 'bg-amber-100 dark:bg-amber-900/30 text-amber-600 dark:text-amber-400 hover:bg-amber-200 dark:hover:bg-amber-800/40'} font-bold rounded-xl transition flex items-center justify-center`}
                                        >
                                            {isBatchPaused ? <Play size={20} className="fill-current" /> : <Pause size={20} className="fill-current" />}
                                        </button>
                                    )}

                                    {/* Stop Button */}
                                    {isBatchProcessing && (
                                        <button
                                            onClick={stopBatch}
                                            title="Stop Generation"
                                            className="px-4 py-4 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 hover:bg-red-200 dark:hover:bg-red-800/40 font-bold rounded-xl transition flex items-center justify-center"
                                        >
                                            <Square size={20} className="fill-current" />
                                        </button>
                                    )}
                                    <button
                                        onClick={resetBatch}
                                        title="Start New Batch (Reset)"
                                        className="px-4 py-4 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-300 font-bold rounded-xl transition flex items-center justify-center"
                                    >
                                        <RotateCcw size={20} />
                                    </button>
                                    <button
                                        onClick={downloadBatchResults}
                                        disabled={batchItems.filter(i => i.status === 'complete').length === 0}
                                        className="px-4 py-4 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-300 font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50"
                                    >
                                        <Download size={20} /> <span className="uppercase">Download All ({settings.exportFormat || 'csv'})</span>
                                    </button>
                                </div>
                            </div>

                            {/* Progress Bar moved here */}
                            {batchItems.length > 0 && (
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none animate-fade-in">
                                    <div className="flex justify-between text-sm mb-2 font-bold text-slate-700 dark:text-slate-300">
                                        <div className="flex flex-col gap-1">
                                            <span>Progress</span>
                                            {batchStatusInfo && (
                                                <div className="flex items-center gap-2 text-[10px] font-mono font-medium text-slate-500">
                                                    <span className="px-1.5 py-0.5 bg-red-100 dark:bg-red-900/20 text-red-600 dark:text-red-400 rounded border border-red-200 dark:border-red-800/30 flex items-center gap-1">
                                                        <Key size={10} /> {batchStatusInfo.currentKey.substring(0, 8)}...
                                                    </span>
                                                    <span className="px-1.5 py-0.5 bg-emerald-100 dark:bg-emerald-900/20 text-emerald-600 dark:text-emerald-400 rounded border border-emerald-200 dark:border-emerald-800/30 flex items-center gap-1">
                                                        <Cpu size={10} /> {batchStatusInfo.currentModel}
                                                    </span>
                                                </div>
                                            )}
                                        </div>
                                        <div className="text-right">
                                            <div className="flex items-center justify-end gap-3 mb-1">
                                                {totalElapsedTime && (
                                                    <span className="text-[10px] text-amber-600 dark:text-amber-500 font-bold">
                                                        Total Time: {totalElapsedTime}
                                                    </span>
                                                )}
                                                <span className="font-bold text-slate-700 dark:text-slate-200 text-sm">
                                                    {batchItems.filter(i => i.status === 'complete').length}/{batchItems.length}
                                                </span>
                                            </div>
                                            {batchStatusInfo && (
                                                <span className="text-[10px] text-amber-600 dark:text-amber-500 font-bold block">
                                                    ETA: {batchStatusInfo.eta}
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                    <div className="w-full bg-slate-200 dark:bg-slate-800 h-2 rounded-full overflow-hidden">
                                        <div className="h-full bg-amber-500 transition-all duration-300" style={{ width: `${(batchItems.filter(i => i.status === 'complete').length / batchItems.length) * 100}%` }}></div>
                                    </div>
                                    {isBatchProcessing && (
                                        <p className="text-xs text-center mt-2 text-slate-500 animate-pulse">Processing items sequentially...</p>
                                    )}
                                </div>
                            )}

                            {/* Settings Panel (Duplicated from Single) */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div className="flex justify-between items-center">
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300">Context & Settings</label>
                                    <div className="flex items-center gap-2">
                                        <div className="flex items-center gap-1.5 mr-2 group relative">
                                            <input type="checkbox" checked={settings.enableGrounding} onChange={e => setSettings({ ...settings, enableGrounding: e.target.checked })} className="rounded bg-slate-200 dark:bg-slate-700 border-transparent text-amber-500 focus:ring-0 w-3.5 h-3.5" />
                                            <span className="text-xs font-bold text-slate-600 dark:text-slate-400">Search Grounding</span>
                                            <div className="relative">
                                                <Info size={12} className="text-slate-400 hover:text-amber-500 cursor-help" />
                                                <div className="absolute right-0 top-full mt-2 w-80 p-2 bg-slate-800 text-white text-[10px] rounded-lg opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl border border-slate-700">
                                                    <p className="mb-2"><span className="text-amber-400 font-bold">IF ENABLED:</span> Costs you 1 API Credit more!</p>
                                                    <p className="text-slate-300 text-[9px] leading-relaxed">When enabled, the AI searches the web for real-time information about your image subject before generating metadata. This provides more accurate, factual descriptions and keywords based on current data, making your content more discoverable and relevant.</p>
                                                </div>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => setSettings({ ...settings, useFilenameMode: !settings.useFilenameMode })}
                                            className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-xs font-bold transition border ${settings.useFilenameMode
                                                ? 'bg-amber-100 text-amber-700 dark:bg-amber-900/30 dark:text-amber-400 border-amber-200 dark:border-amber-800'
                                                : 'bg-slate-100 text-slate-600 dark:bg-slate-900 dark:text-slate-400 border-slate-200 dark:border-slate-800 hover:bg-slate-200 dark:hover:bg-slate-800'
                                                }`}
                                        >
                                            <FileText size={14} />
                                            <span>Filename Mode: {settings.useFilenameMode ? 'ON' : 'OFF'}</span>
                                        </button>
                                        <div className="relative group">
                                            <Info size={14} className="text-slate-400 hover:text-amber-500 cursor-help" />
                                            <div className="absolute right-0 top-full mt-2 w-64 p-3 bg-slate-800 text-white text-xs rounded-xl opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl border border-slate-700">
                                                <p className="font-bold mb-1 text-amber-400">Filename-Only Mode</p>
                                                <p>When enabled, the AI prioritizes the filename (e.g., "sunset_beach.jpg") as the primary source of truth. Useful if visual analysis misidentifies the subject.</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div className="p-3 rounded-xl border border-amber-200 dark:border-amber-900/50 bg-amber-50 dark:bg-amber-900/10 shadow-sm relative group mb-3">
                                    <div className="absolute top-0 right-0 w-16 h-16 bg-gradient-to-br from-amber-200/20 to-transparent rounded-bl-full rounded-tr-xl pointer-events-none transition-opacity group-hover:opacity-100 opacity-50"></div>

                                    <div className="flex justify-between items-center text-xs mb-2 relative z-10">
                                        <div className="flex items-center gap-2">
                                            <span className="font-bold text-amber-800 dark:text-amber-400 flex items-center gap-1.5">
                                                <Globe size={14} className="text-amber-600 dark:text-amber-500" /> Target Platforms
                                                <InfoTooltip text="Select which stock platforms you are uploading to. Formats metadata specifically for them." />
                                            </span>
                                            <div className="flex gap-2">
                                                <span className="px-1.5 py-0.5 rounded bg-white/50 dark:bg-black/20 text-[10px] text-amber-700/70 dark:text-amber-400/70 font-mono border border-amber-100 dark:border-amber-800/30">
                                                    Selected: {settings.selectedStockSites?.length || 0}/{availableSites.length}
                                                </span>
                                            </div>
                                        </div>
                                        <button
                                            onClick={() => {
                                                const allIds = availableSites.map(s => s.id);
                                                const isAllSelected = allIds.every(id => settings.selectedStockSites?.includes(id));
                                                setSettings({ ...settings, selectedStockSites: isAllSelected ? [allIds[0]] : allIds });
                                            }}
                                            className="text-[10px] font-bold text-amber-600 dark:text-amber-400 hover:text-amber-700 dark:hover:text-amber-300 uppercase underline decoration-dashed underline-offset-2 transition-colors"
                                        >
                                            {availableSites.length > 0 && availableSites.every(s => settings.selectedStockSites?.includes(s.id)) ? 'Uncheck All' : 'Check All'}
                                        </button>
                                    </div>
                                    <div className="grid grid-cols-5 gap-2 max-h-32 overflow-y-auto custom-scrollbar p-1 relative z-10">
                                        {availableSites.length === 0 && <p className="text-[10px] text-slate-400 italic col-span-5">No sites configured.</p>}
                                        {availableSites.map(site => (
                                            <div key={site.id} className="flex items-center gap-2 overflow-hidden bg-white/50 dark:bg-slate-900/50 p-1.5 rounded-lg border border-amber-100 dark:border-amber-900/30 hover:border-amber-300 dark:hover:border-amber-700 transition-colors cursor-pointer" onClick={() => {
                                                const current = settings.selectedStockSites || [];
                                                const isSelected = current.includes(site.id);

                                                if (isSelected && current.length <= 1) return;

                                                const newer = !isSelected
                                                    ? [...current, site.id]
                                                    : current.filter(id => id !== site.id);
                                                setSettings({ ...settings, selectedStockSites: newer });
                                            }}>
                                                <div className={`w-3 h-3 rounded-full border flex items-center justify-center transition-colors ${settings.selectedStockSites?.includes(site.id) ? 'bg-amber-500 border-amber-500' : 'border-slate-300 dark:border-slate-600 bg-white dark:bg-slate-800'}`}>
                                                    {settings.selectedStockSites?.includes(site.id) && <Check size={8} className="text-white" strokeWidth={4} />}
                                                </div>
                                                <span className={`text-[10px] truncate font-medium ${settings.selectedStockSites?.includes(site.id) ? 'text-amber-900 dark:text-amber-200' : 'text-slate-600 dark:text-slate-400'}`} title={site.name}>{site.name}</span>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                                <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-700 space-y-3">
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Title Length
                                                <InfoTooltip text="Sets the target character length for the generated title." />
                                            </span>
                                            <span className="font-bold">{settings.titleLength}</span>
                                        </div>
                                        <input type="range" min="40" max="150" value={settings.titleLength} onChange={e => setSettings({ ...settings, titleLength: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Description Length
                                                <InfoTooltip text="Sets the target character length for the generated description." />
                                            </span>
                                            <span className="font-bold">{settings.descLength}</span>
                                        </div>
                                        <input type="range" min="100" max="300" step="10" value={settings.descLength} onChange={e => setSettings({ ...settings, descLength: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>
                                    <div>
                                        <div className="flex justify-between text-xs mb-1">
                                            <span className="flex items-center">
                                                Keywords
                                                <InfoTooltip text="Number of keywords to generate (10-50)." />
                                            </span>
                                            <span className="font-bold">{settings.keywordCount}</span>
                                        </div>
                                        <input type="range" min="10" max="50" value={settings.keywordCount} onChange={e => setSettings({ ...settings, keywordCount: Number(e.target.value) })} className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-amber-500" />
                                    </div>





                                    {/* Keyword Format Selection */}
                                    <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                        <div className="flex justify-between text-xs mb-2">
                                            <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                                Keyword Format
                                                <InfoTooltip text="Single (e.g., 'sunset'), Double (e.g., 'sunset beach'), or Mixed." />
                                            </span>
                                        </div>
                                        <div className="flex bg-slate-200 dark:bg-slate-800 p-1 rounded-lg">
                                            {(['single', 'double', 'mixed'] as const).map(type => (
                                                <button
                                                    key={type}
                                                    onClick={() => setSettings({ ...settings, keywordType: type })}
                                                    className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition whitespace-nowrap ${settings.keywordType === type
                                                        ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                        : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                        }`}
                                                >
                                                    {type === 'single' ? 'Single' : type === 'double' ? 'Double' : 'Mixed'}
                                                </button>
                                            ))}
                                        </div>
                                    </div>
                                    {/* Video Frame Source (Visible Only if Batch has Video) */}
                                    {batchItems.some(item => item.file.type.startsWith('video/')) && (
                                        <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                            <span className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1 mb-1">
                                                Video Frame Source <InfoTooltip text="Select which frame from the video to use for visual analysis." />
                                            </span>
                                            <div className="flex bg-slate-200 dark:bg-slate-800 p-1 rounded-lg gap-1">
                                                {(['start', 'middle', 'end'] as const).map(opt => {
                                                    const isSelected = (settings.videoFrameOptions || ['start']).includes(opt);
                                                    return (
                                                        <button
                                                            key={opt}
                                                            onClick={() => {
                                                                const current = settings.videoFrameOptions || ['start'];
                                                                let next = [];
                                                                if (current.includes(opt)) {
                                                                    next = current.filter(o => o !== opt);
                                                                    if (next.length === 0) next = [opt];
                                                                } else {
                                                                    next = [...current, opt];
                                                                }
                                                                setSettings({ ...settings, videoFrameOptions: next });
                                                            }}
                                                            className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition-all shadow-sm ${isSelected
                                                                ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                                : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                                }`}
                                                            title={opt === 'start' ? 'Use 3rd Frame (Start)' : opt === 'middle' ? 'Use Middle Frame' : 'Use Last Frame'}
                                                        >
                                                            {opt === 'start' ? 'Start' : opt === 'middle' ? 'Middle' : 'End'}
                                                        </button>
                                                    );
                                                })}
                                            </div>
                                        </div>
                                    )}


                                    {/* Export Format Selection */}
                                    <div className="pt-2 border-t border-slate-200 dark:border-slate-800 mt-2">
                                        <div className="flex justify-between text-xs mb-2">
                                            <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                                Export Format
                                                <InfoTooltip text="Choose the file format for your metadata download." />
                                            </span>
                                        </div>
                                        <div className="flex bg-slate-200 dark:bg-slate-800 p-1 rounded-lg">
                                            {(['csv', 'json', 'txt'] as const).map(fmt => (
                                                <button
                                                    key={fmt}
                                                    onClick={() => setSettings({ ...settings, exportFormat: fmt })}
                                                    className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-md transition whitespace-nowrap flex items-center justify-center gap-1 ${settings.exportFormat === fmt
                                                        ? 'bg-white dark:bg-slate-700 text-amber-600 dark:text-amber-400 shadow-sm'
                                                        : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                                        }`}
                                                >
                                                    {fmt === 'txt' && <FileText size={10} />}
                                                    {fmt === 'json' && <FileJson size={10} />}
                                                    {fmt === 'csv' && <FileType size={10} />}
                                                    {fmt.toUpperCase()}
                                                    {fmt === 'csv' && (
                                                        <div className="relative group ml-1">
                                                            <Info size={10} className="text-red-500" />
                                                            <div className="absolute top-full mt-2 left-1/2 -translate-x-1/2 w-max max-w-[300px] p-2 bg-slate-800 text-white text-[10px] rounded-lg opacity-0 group-hover:opacity-100 transition pointer-events-none z-50 shadow-xl normal-case text-center">
                                                                Required option for Microstock websites
                                                            </div>
                                                        </div>
                                                    )}
                                                </button>
                                            ))}
                                        </div>
                                    </div>
                                </div>

                                <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-700 space-y-2">
                                    <div className="flex justify-between text-xs">
                                        <span className="font-bold text-slate-600 dark:text-slate-400 flex items-center">
                                            Include in Description
                                            <InfoTooltip text="Add specific details, locations, or subject names that must be included in the description." />
                                        </span>
                                    </div>
                                    <input
                                        type="text"
                                        placeholder="Location, specific event, or subject names..."
                                        value={contextDesc} onChange={e => setContextDesc(e.target.value)}
                                        className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500 transition-colors"
                                    />
                                </div>
                                <div className="grid grid-cols-2 gap-3">
                                    <div className="space-y-1">
                                        <label className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1">
                                            <PlusCircle size={12} className="text-blue-500" /> Include Keywords
                                            <InfoTooltip text="Force these keywords to be included in the result." />
                                        </label>
                                        <TagInput
                                            placeholder="Type & Enter..."
                                            tags={contextKeywords}
                                            onTagsChange={setContextKeywords}
                                            color="blue"
                                        />
                                    </div>
                                    <div className="space-y-1">
                                        <label className="text-xs font-bold text-slate-600 dark:text-slate-400 flex items-center gap-1">
                                            <MinusCircle size={12} className="text-red-500" /> Exclude Keywords
                                            <InfoTooltip text="Ensure these words never appear in the result." />
                                        </label>
                                        <TagInput
                                            placeholder="Type & Enter..."
                                            tags={excludeKeywords}
                                            onTagsChange={setExcludeKeywords}
                                            color="red"
                                        />
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Right: Batch Queue / Result List */}
                        <div className="lg:col-span-7 bg-white dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 h-[calc(100vh-200px)] overflow-y-auto custom-scrollbar shadow-sm dark:shadow-none">

                            {batchItems.length === 0 ? (
                                <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                    <FolderArchive size={48} className="mb-4 opacity-20" />
                                    <p className="text-sm">Batch queue is empty</p>
                                </div>
                            ) : (
                                <div className="space-y-3">
                                    {batchItems.map((item) => (
                                        <div key={item.id} className="p-3 bg-slate-50 dark:bg-slate-950/50 rounded-xl border border-slate-100 dark:border-slate-800/50 transition-all">
                                            <div className="flex items-center gap-4">
                                                {item.file.type.startsWith('video/') ? (
                                                    <VideoPreview file={item.file} onImageClick={setLightboxSrc} />
                                                ) : (
                                                    <div
                                                        onClick={() => setLightboxSrc(item.preview)}
                                                        className="relative group cursor-pointer shrink-0"
                                                    >
                                                        <img src={item.preview} className="w-12 h-12 rounded-lg object-cover bg-slate-200 dark:bg-slate-800 transition-all group-hover:scale-105 group-hover:shadow-md" />
                                                        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/10 rounded-lg transition-colors flex items-center justify-center opacity-0 group-hover:opacity-100">
                                                            <ScanEye size={16} className="text-white drop-shadow-md" />
                                                        </div>
                                                    </div>
                                                )}

                                                <div className="flex-1 min-w-0">
                                                    <div className="flex justify-between items-start">
                                                        <p className="text-xs font-bold text-slate-700 dark:text-slate-300 truncate pr-2" title={item.file.name}>{item.file.name}</p>
                                                    </div>
                                                    <p className="text-[10px] text-slate-500">{item.status === 'complete' ? 'Analysis Done' : item.status === 'processing' ? 'Processing...' : item.status === 'waiting' ? 'Waiting (Delay)...' : item.status === 'error' ? 'Failed' : 'Queued'}</p>
                                                </div>

                                                <div className="flex items-center gap-3 shrink-0">
                                                    <div className="flex items-center gap-1">
                                                        {item.status === 'complete' && (
                                                            <>
                                                                <button
                                                                    onClick={() => downloadSingleBatchItem(item)}
                                                                    className="p-1 rounded-md transition hover:bg-slate-200 dark:hover:bg-slate-800 text-slate-400 hover:text-amber-500"
                                                                    title="Download"
                                                                >
                                                                    <Download size={14} />
                                                                </button>
                                                                <button
                                                                    onClick={() => setEditingBatchItemId(editingBatchItemId === item.id ? null : item.id)}
                                                                    className={`p-1 rounded-md transition ${editingBatchItemId === item.id ? 'bg-amber-100 text-amber-600' : 'hover:bg-slate-200 dark:hover:bg-slate-800 text-slate-400'}`}
                                                                    title="Edit"
                                                                >
                                                                    <Edit2 size={14} />
                                                                </button>
                                                                {/* Generation Info Button */}
                                                                <div className="relative group">
                                                                    <button className="p-1 rounded-md transition hover:bg-slate-200 dark:hover:bg-slate-800 text-slate-400 hover:text-blue-500">
                                                                        <Info size={14} />
                                                                    </button>
                                                                    <div className="absolute top-full right-0 mt-2 w-[260px] p-3 bg-slate-900 text-white text-[10px] rounded-xl opacity-0 group-hover:opacity-100 transition pointer-events-none z-[999] shadow-2xl border border-slate-700 text-left">
                                                                        <p className="font-bold text-slate-400 uppercase text-[9px] mb-2 border-b border-slate-700 pb-1">Generation Info</p>
                                                                        {item.generationInfo ? (
                                                                            <div className="space-y-1.5">
                                                                                <div className="flex justify-between items-center gap-4">
                                                                                    <span className="text-slate-400">Model:</span>
                                                                                    <span className="font-mono text-emerald-400 bg-emerald-400/10 px-1.5 py-0.5 rounded">{item.generationInfo.model}</span>
                                                                                </div>
                                                                                <div className="flex justify-between items-center gap-4">
                                                                                    <span className="text-slate-400">Key:</span>
                                                                                    <span className="font-mono text-amber-400 bg-amber-400/10 px-1.5 py-0.5 rounded text-[9px] truncate max-w-[120px]" title={item.generationInfo.apiKey}>{item.generationInfo.apiKey.substring(0, 16)}...</span>
                                                                                </div>
                                                                            </div>
                                                                        ) : (
                                                                            <span className="italic text-slate-500">No info available</span>
                                                                        )}
                                                                    </div>
                                                                </div>
                                                            </>
                                                        )}
                                                        <button
                                                            onClick={() => removeBatchItem(item.id)}
                                                            className="p-1 rounded-md transition hover:bg-red-100 dark:hover:bg-red-900/40 text-slate-400 hover:text-red-500"
                                                            title="Delete"
                                                        >
                                                            <Trash2 size={14} />
                                                        </button>
                                                    </div>
                                                    {item.status === 'complete' ? <CheckCircle2 size={18} className="text-green-500" /> : item.status === 'error' ? <AlertCircle size={18} className="text-red-500" /> : item.status === 'processing' ? <RefreshCw size={18} className="animate-spin text-amber-500" /> : item.status === 'waiting' ? <Loader2 size={18} className="animate-spin text-blue-500" /> : <Clock size={18} className="text-slate-400" />}
                                                </div>
                                            </div>

                                            {/* EDIT MODE */}
                                            {editingBatchItemId === item.id && item.status === 'complete' && item.result && (
                                                <div className="mt-4 pt-4 border-t border-slate-200 dark:border-slate-800 space-y-3 animate-fade-in">
                                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 group relative">
                                                        <div className="flex justify-between items-center mb-2">
                                                            <span className="text-[10px] font-bold text-slate-500 uppercase">Title ({item.result.title.length} chars)</span>
                                                            <div className="flex items-center gap-1">
                                                                <button
                                                                    onClick={() => handleRegenerateBatchItemTitle(item)}
                                                                    disabled={regeneratingBatchItemId === item.id}
                                                                    className={`p-1.5 transition rounded-md ${regeneratingBatchItemId === item.id ? 'text-amber-500 cursor-not-allowed' : 'text-slate-400 hover:text-amber-500'}`}
                                                                    title="Regenerate Title"
                                                                >
                                                                    <RefreshCw size={12} className={regeneratingBatchItemId === item.id ? 'animate-spin' : ''} />
                                                                </button>
                                                                <button onClick={() => navigator.clipboard.writeText(item.result!.title)} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                                            </div>
                                                        </div>
                                                        <input
                                                            type="text"
                                                            value={item.result.title}
                                                            onChange={(e) => {
                                                                const newVal = e.target.value;
                                                                setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, result: { ...p.result!, title: newVal } } : p));
                                                            }}
                                                            className="w-full text-xs font-medium bg-transparent border-none focus:ring-0 outline-none p-0 text-slate-800 dark:text-slate-200"
                                                        />
                                                    </div>
                                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800">
                                                        <div className="flex justify-between items-center mb-2">
                                                            <span className="text-[10px] font-bold text-slate-500 uppercase">Description ({item.result.description.length} chars)</span>
                                                            <button onClick={() => navigator.clipboard.writeText(item.result!.description)} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                                        </div>
                                                        <textarea
                                                            rows={3}
                                                            value={item.result.description}
                                                            onChange={(e) => {
                                                                const newVal = e.target.value;
                                                                setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, result: { ...p.result!, description: newVal } } : p));
                                                            }}
                                                            className="w-full text-xs bg-transparent border-none focus:ring-0 outline-none resize-none p-0 text-slate-700 dark:text-slate-300 leading-relaxed"
                                                        />
                                                    </div>
                                                    <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800">
                                                        <div className="flex justify-between items-center mb-2">
                                                            <span className="text-[10px] font-bold text-slate-500 uppercase">Keywords ({item.result.commonKeywords.length})</span>
                                                            <button onClick={() => navigator.clipboard.writeText(item.result!.commonKeywords.join(', '))} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy All"><Copy size={12} /></button>
                                                        </div>
                                                        <textarea
                                                            rows={2}
                                                            value={item.result.commonKeywords.join(', ')}
                                                            onChange={(e) => {
                                                                const newVal = e.target.value.split(',').map(s => s.trim()).filter(Boolean);
                                                                setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, result: { ...p.result!, commonKeywords: newVal } } : p));
                                                            }}
                                                            className="w-full text-xs bg-transparent border-none focus:ring-0 outline-none resize-none p-0 text-slate-700 dark:text-slate-300 leading-relaxed"
                                                        />
                                                    </div>
                                                    {(availableSites.some(s => settings.selectedStockSites?.includes(s.id) && s.name.toLowerCase().includes('adobe')) || item.result.category) && (
                                                        <div className="p-4 bg-slate-50 dark:bg-slate-900/50 rounded-xl border border-slate-200 dark:border-slate-800 mt-2">
                                                            <div className="flex justify-between items-center mb-2">
                                                                <span className="text-[10px] font-bold text-slate-500 uppercase">Category</span>
                                                                <button onClick={() => navigator.clipboard.writeText(item.result!.category || '')} className="p-1.5 text-slate-400 hover:text-blue-500 transition rounded-md" title="Copy"><Copy size={12} /></button>
                                                            </div>
                                                            <input
                                                                type="text"
                                                                value={item.result.category || ''}
                                                                onChange={(e) => {
                                                                    const newVal = e.target.value;
                                                                    setBatchItems(prev => prev.map(p => p.id === item.id ? { ...p, result: { ...p.result!, category: newVal } } : p));
                                                                }}
                                                                className="w-full text-xs bg-transparent border-none focus:ring-0 outline-none p-0 text-slate-800 dark:text-slate-200 font-bold"
                                                            />
                                                        </div>
                                                    )}
                                                    <button onClick={() => setEditingBatchItemId(null)} className="w-full py-2 bg-amber-500 hover:bg-amber-600 text-white text-xs font-bold rounded-lg transition">
                                                        Done Editing
                                                    </button>
                                                </div>
                                            )}
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* HISTORY MODE */}
                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {historyItems.map((item) => {
                                const formatDateTime = (timestamp: number) => {
                                    const date = new Date(timestamp);
                                    return date.toLocaleString('en-US', {
                                        month: 'short',
                                        day: 'numeric',
                                        year: 'numeric',
                                        hour: 'numeric',
                                        minute: '2-digit',
                                        hour12: true
                                    });
                                };

                                const handleEditRerun = () => {
                                    // Load this metadata back into single tab for editing
                                    if (item.url) {
                                        // Convert data URL back to file-like preview
                                        setSinglePreview(item.url);
                                        // Recreate a minimal file object with correct filename
                                        const filename = item.metadata?.filename || 'image.jpg';
                                        setSingleFile(new File([item.prompt], filename, { type: 'image/jpeg' }));
                                    }
                                    if (item.metadata) {
                                        // Ensure filename is set in result even if missing from stored metadata
                                        const filename = item.metadata.filename || singleFile?.name || 'image.jpg';
                                        setResult({
                                            ...item.metadata,
                                            filename: filename
                                        });
                                    }
                                    // Switch to single tab
                                    setActiveTab('single');
                                };

                                const handleDownloadItem = (format: 'txt' | 'json' | 'csv') => {
                                    handleExport(format, item.metadata);
                                };

                                return (
                                    <div key={item.id} className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl overflow-hidden shadow-sm hover:border-amber-500/50 hover:shadow-md transition group">
                                        {/* Image Preview */}
                                        <div className="aspect-video bg-slate-100 dark:bg-slate-950 relative">
                                            <img src={item.url} className="w-full h-full object-cover" alt={item.metadata?.title || 'Generated metadata'} />

                                            {/* Model Badge Overlay - REMOVED */}
                                        </div>

                                        {/* Content */}
                                        <div className="p-4">
                                            {/* Title */}
                                            <h4 className="text-sm font-bold text-slate-900 dark:text-white truncate mb-1" title={item.metadata?.title}>
                                                {item.metadata?.title || 'Untitled'}
                                            </h4>

                                            {/* Description Preview */}
                                            <p className="text-xs text-slate-500 dark:text-slate-400 line-clamp-2 mb-3">
                                                {item.metadata?.description}
                                            </p>

                                            {/* Extra Info - Generation Settings */}
                                            {item.generationSettings && (
                                                <div className="mb-3 p-2 bg-slate-50 dark:bg-slate-950 rounded-lg border border-slate-100 dark:border-slate-800">
                                                    <div className="flex flex-wrap gap-1.5">
                                                        {item.generationSettings.enableGrounding && (
                                                            <span className="px-1.5 py-0.5 bg-amber-100 dark:bg-amber-900/30 text-amber-700 dark:text-amber-400 text-[9px] font-bold rounded border border-amber-200 dark:border-amber-800">
                                                                <Globe size={8} className="inline mr-0.5" />GROUNDING
                                                            </span>
                                                        )}
                                                        {item.generationSettings.titleLength && (
                                                            <span className="px-1.5 py-0.5 bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400 text-[9px] font-bold rounded border border-green-200 dark:border-green-800">
                                                                Title: {item.generationSettings.titleLength} Characters
                                                            </span>
                                                        )}
                                                        <span className="px-1.5 py-0.5 bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-400 text-[9px] font-bold rounded border border-blue-200 dark:border-blue-800">
                                                            {item.generationSettings.keywordCount || 25} Keywords
                                                        </span>
                                                        <span className="px-1.5 py-0.5 bg-purple-100 dark:bg-purple-900/30 text-purple-700 dark:text-purple-400 text-[9px] font-bold rounded border border-purple-200 dark:border-purple-800">
                                                            Description: {item.generationSettings.descLength || 250} Characters
                                                        </span>
                                                        {/* Category Badge */}
                                                        {item.metadata?.category && (
                                                            <span className="px-1.5 py-0.5 bg-pink-100 dark:bg-pink-900/30 text-pink-700 dark:text-pink-400 text-[9px] font-bold rounded border border-pink-200 dark:border-pink-800">
                                                                Category: {item.metadata.category}
                                                            </span>
                                                        )}
                                                    </div>
                                                </div>
                                            )}

                                            {/* Timestamp & Actions */}
                                            <div className="flex flex-col gap-2 pt-3 border-t border-slate-100 dark:border-slate-800">
                                                {/* Action Buttons */}
                                                <div className="flex gap-2">
                                                    {/* EDIT Button with Icon */}
                                                    <button
                                                        onClick={handleEditRerun}
                                                        className="flex-1 flex items-center justify-center gap-1.5 px-3 py-2 bg-slate-100 dark:bg-slate-800 hover:bg-amber-100 dark:hover:bg-amber-900/30 hover:text-amber-600 dark:hover:text-amber-400 text-slate-600 dark:text-slate-400 rounded-lg text-xs font-bold transition border border-slate-200 dark:border-slate-700 hover:border-amber-300 dark:hover:border-amber-800"
                                                        title="Load into editor for editing or re-running"
                                                    >
                                                        <Edit2 size={14} />
                                                        EDIT
                                                    </button>

                                                    {/* Download Button - Format based on original export setting */}
                                                    {(() => {
                                                        const savedFormat = item.generationSettings?.exportFormat || 'csv'; // Default to CSV not TXT
                                                        const formatIcon = savedFormat === 'json' ? <FileJson size={10} /> : savedFormat === 'csv' ? <FileType size={10} /> : <FileText size={10} />;
                                                        const formatLabel = savedFormat.toUpperCase();

                                                        return (
                                                            <button
                                                                onClick={() => handleDownloadItem(savedFormat as 'txt' | 'json' | 'csv')}
                                                                className="flex-1 flex items-center justify-center gap-1 px-2 py-1.5 bg-blue-500 hover:bg-blue-600 text-white text-[10px] font-bold rounded-lg transition"
                                                                title={`Download as ${formatLabel}`}
                                                            >
                                                                <Download size={10} />
                                                                {formatLabel}
                                                            </button>
                                                        );
                                                    })()}
                                                </div>

                                                {/* Timestamp */}
                                                <div className="flex items-center justify-center gap-1.5 text-[10px] text-slate-400">
                                                    <Clock size={10} />
                                                    <span>{formatDateTime(item.createdAt)}</span>
                                                </div>

                                                {/* Targeted Microstock Platforms */}
                                                {item.generationSettings?.selectedStockSites && item.generationSettings.selectedStockSites.length > 0 && (
                                                    <div className="pt-2 border-t border-slate-100 dark:border-slate-800">
                                                        <div className="flex items-center justify-center gap-1.5 flex-wrap">
                                                            <span className="text-[9px] text-slate-500 dark:text-slate-400 font-medium">Target:</span>
                                                            {item.generationSettings.selectedStockSites.map((siteId: string) => {
                                                                const site = availableSites.find(s => s.id === siteId);
                                                                if (!site) return null;
                                                                return (
                                                                    <div
                                                                        key={siteId}
                                                                        className="px-2 py-1 bg-gradient-to-r from-blue-500/10 to-purple-500/10 border border-blue-300/30 dark:border-blue-700/30 rounded-md flex items-center gap-1"
                                                                        title={site.name}
                                                                    >
                                                                        {site.logo && (
                                                                            <img src={site.logo} alt={site.name} className="w-3 h-3 object-contain" />
                                                                        )}
                                                                        <span className="text-[8px] font-bold text-blue-700 dark:text-blue-400">{site.name}</span>
                                                                    </div>
                                                                );
                                                            })}
                                                        </div>
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                    </div>
                                );
                            })}
                            {historyItems.length === 0 && (
                                <div className="col-span-full flex flex-col items-center justify-center h-64 text-slate-500">
                                    <HistoryIcon size={32} className="mb-2 opacity-30" />
                                    <p>No history found.</p>
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </div>
            {/* Loading Overlay */}
            <EnhancedLoadingOverlay
                isVisible={isGenerating}
                title="Generating Metadata"
                description="Analyzing image content and generating SEO tags..."
                simulate={true}
                onCancel={() => {
                    if (abortControllerRef.current) abortControllerRef.current.abort();
                    setIsGenerating(false);
                }}
                apiKey={generationInfo?.apiKey}
                model={generationInfo?.model}
            />
            {/* Rate Limit Exhausted Popup */}
            {/* Rate Limit Exhausted Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />

            {/* Lightbox for Previewing Images */}
            <Lightbox
                isOpen={!!lightboxSrc}
                src={lightboxSrc}
                onClose={() => setLightboxSrc(null)}
            />
        </div>
    );
};

export default MetadataGenerator;
