import React, { useState } from 'react';
import { ScanEye, Upload, FileText, Loader2 } from 'lucide-react';
import { generateChatResponse } from '../services/geminiService';
import { Attachment, ModelType } from '../types';

const MediaAnalyzer: React.FC = () => {
  const [file, setFile] = useState<File | null>(null);
  const [preview, setPreview] = useState<string | null>(null);
  const [analysis, setAnalysis] = useState('');
  const [isAnalyzing, setIsAnalyzing] = useState(false);

  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const f = e.target.files[0];
      setFile(f);
      
      const reader = new FileReader();
      reader.onload = (ev) => {
        setPreview(ev.target?.result as string);
      };
      reader.readAsDataURL(f);
    }
  };

  const analyze = async () => {
    if (!preview || !file) return;
    setIsAnalyzing(true);
    
    try {
      const base64 = preview.split(',')[1];
      const attachment: Attachment = {
        mimeType: file.type,
        data: base64
      };
      
      // Determine prompt based on type
      const isVideo = file.type.startsWith('video');
      const prompt = isVideo 
        ? "Analyze this video in detail. Describe key events, visual style, and potential tags for a stock library." 
        : "Analyze this image. Describe the subject, composition, lighting, and suggest title and keywords for stock photography.";

      // Use Pro model for deep understanding
      const response = await generateChatResponse(
          prompt, 
          ModelType.PRO, 
          [], 
          [attachment], 
          true // Enable thinking for deep analysis
      );
      
      setAnalysis(response.text);
    } catch (e) {
      setAnalysis("Failed to analyze media.");
    } finally {
      setIsAnalyzing(false);
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6 h-full flex flex-col">
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-slate-100 flex items-center gap-2">
            <ScanEye className="text-emerald-400" /> Deep Media Analysis
        </h1>
        <p className="text-slate-400 text-sm">Powered by Gemini 3.0 Pro Vision</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6 flex-1">
        {/* Input Side */}
        <div className="flex flex-col gap-4">
            <div className="flex-1 border-2 border-dashed border-slate-700 rounded-2xl bg-slate-900/50 hover:bg-slate-900 transition-colors relative flex items-center justify-center overflow-hidden group">
                {preview ? (
                    file?.type.startsWith('video') ? (
                         <video src={preview} className="max-h-full max-w-full object-contain" controls />
                    ) : (
                         <img src={preview} alt="Analysis Target" className="max-h-full max-w-full object-contain" />
                    )
                ) : (
                    <div className="text-center p-8">
                        <Upload className="w-12 h-12 text-slate-600 mx-auto mb-4 group-hover:text-slate-400 transition-colors" />
                        <p className="text-slate-400 font-medium">Drop image or video here</p>
                    </div>
                )}
                <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" onChange={handleFileUpload} accept="image/*,video/*" />
            </div>
            <button 
                onClick={analyze}
                disabled={!file || isAnalyzing}
                className="py-3 bg-emerald-600 hover:bg-emerald-500 text-white font-bold rounded-xl transition shadow-lg shadow-emerald-600/20 disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
            >
                {isAnalyzing ? <Loader2 className="animate-spin" /> : <ScanEye />} 
                Analyze Asset
            </button>
        </div>

        {/* Output Side */}
        <div className="bg-slate-900 rounded-2xl border border-slate-800 p-6 overflow-y-auto custom-scrollbar">
            <h3 className="text-sm font-bold text-slate-400 uppercase tracking-wider mb-4 flex items-center gap-2">
                <FileText size={16} /> Report
            </h3>
            {analysis ? (
                <div className="prose prose-invert prose-sm max-w-none">
                    <p className="whitespace-pre-wrap leading-relaxed text-slate-300">{analysis}</p>
                </div>
            ) : (
                <div className="h-full flex items-center justify-center text-slate-600 text-sm italic">
                    Analysis results will appear here...
                </div>
            )}
        </div>
      </div>
    </div>
  );
};

export default MediaAnalyzer;
