
import React, { useState, useEffect, useRef } from 'react';
import {
    Globe, TrendingUp, Search, Filter, ArrowRight,
    Shield, Activity, LayoutGrid, Info, ExternalLink,
    CheckCircle2, AlertTriangle, X, BrainCircuit,
    Clock, ListFilter, Mic, Calendar, DollarSign,
    MessageSquare, Share2, Download, Bookmark,
    ChevronDown, ChevronUp, Sparkles, Zap,
    BarChart3, Plus, Check, ThumbsUp, ThumbsDown,
    Users, Key, Cpu, History, Trash2, Loader2, FileArchive, Copy
} from 'lucide-react';
import { jsPDF } from 'jspdf';
import html2canvas from 'html2canvas';
import { StockSite, TrendReport, ModelType } from '../types';

import {
    AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
    BarChart, Bar, Legend
} from 'recharts';
import { authService } from '../services/authService';
import { adminService } from '../services/adminService';
import { getPreferredModels } from '../services/geminiService';
import { dbService, Asset } from '../services/dbService';
import { AutoDownloadIndicator } from './AutoDownloadIndicator';
import { playSound } from '../utils/soundEffects';
import { downloadItems } from '../utils/downloadUtils';

// --- Constants & Data ---

const CATEGORIES = [
    { id: 'all', label: 'All Categories', color: 'bg-slate-800 text-slate-300' },
    { id: 'Major', label: 'Major Platforms', color: 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20' },
    { id: 'Video', label: 'Video Marketplaces', color: 'bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20' },
    { id: 'Vector', label: 'Vector/Illustration', color: 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20' },
    { id: 'POD', label: 'Print-on-Demand', color: 'bg-pink-500/10 text-pink-600 dark:text-pink-400 border-pink-500/20' },
    { id: 'AI', label: 'AI Content', color: 'bg-cyan-500/10 text-cyan-600 dark:text-cyan-400 border-cyan-500/20' },
    { id: 'Free', label: 'Free Stock', color: 'bg-green-500/10 text-green-600 dark:text-green-400 border-green-500/20' },
    { id: 'Specialty', label: 'Specialty/Niche', color: 'bg-amber-500/10 text-amber-600 dark:text-amber-400 border-amber-500/20' },
    { id: 'Design', label: 'Design Assets', color: 'bg-indigo-500/10 text-indigo-600 dark:text-indigo-400 border-indigo-500/20' },
    { id: 'Audio', label: 'Audio', color: 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20' },
];

// --- Helper Components ---

import { EnhancedLoadingOverlay } from './EnhancedLoadingOverlay';
import { TrendReportView } from './TrendReportView';
import { RateLimitPopup } from './RateLimitPopup';

// --- Helper Components ---

const MetricCard = ({ title, value, icon: Icon, color, subtext, trend }: any) => {
    const colors: any = {
        blue: { bg: 'bg-blue-500/10', text: 'text-blue-600 dark:text-blue-400', border: 'border-blue-500/20' },
        amber: { bg: 'bg-amber-500/10', text: 'text-amber-600 dark:text-amber-400', border: 'border-amber-500/20' },
        red: { bg: 'bg-red-500/10', text: 'text-red-600 dark:text-red-400', border: 'border-red-500/20' },
        green: { bg: 'bg-emerald-500/10', text: 'text-emerald-600 dark:text-emerald-400', border: 'border-emerald-500/20' }
    };

    const theme = colors[color] || colors.blue;

    return (
        <div className="relative overflow-hidden bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 p-5 rounded-2xl shadow-sm hover:shadow-lg transition-all duration-300 group">
            {/* Background Gradient Blob */}
            <div className={`absolute -top-10 -right-10 w-32 h-32 rounded-full ${theme.bg} blur-3xl opacity-0 group-hover:opacity-100 transition-opacity duration-500`} />

            <div className="relative z-10">
                <div className="flex justify-between items-start mb-4">
                    <div className={`p-3 rounded-xl ${theme.bg} ${theme.text} border ${theme.border} shadow-sm group-hover:scale-110 transition-transform duration-300`}>
                        <Icon size={22} />
                    </div>
                    {trend && (
                        <div className={`flex items-center gap-1 text-[10px] font-bold ${trend > 0 ? 'text-emerald-500' : 'text-red-500'} bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded-full border border-slate-200 dark:border-slate-700`}>
                            <TrendingUp size={10} className={trend < 0 ? 'rotate-180' : ''} />
                            {Math.abs(trend)}%
                        </div>
                    )}
                </div>

                <div>
                    <h3 className="text-[10px] font-bold text-slate-500 uppercase tracking-widest mb-1">{title}</h3>
                    <div className="flex items-baseline gap-2">
                        <p className="text-2xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk'] tracking-tight">{value}</p>
                    </div>
                    {subtext && (
                        <p className="text-xs text-slate-400 mt-2 font-medium leading-relaxed">{subtext}</p>
                    )}
                </div>
            </div>
        </div>
    );
};



// --- Tab 1: Websites Directory ---

const WebsitesTab = ({ searchFilter, selectedSites, setSelectedSites, sitesList }: any) => {
    const [categoryFilter, setCategoryFilter] = useState('all');
    const [sortOption, setSortOption] = useState('popular');
    const [internalSearch, setInternalSearch] = useState('');
    const [viewingSite, setViewingSite] = useState<StockSite | null>(null);

    const activeSearch = searchFilter || internalSearch;

    // Filter Logic
    const filteredSites = sitesList
        .filter(site => {
            // Search Filter
            if (!site.name.toLowerCase().includes(activeSearch.toLowerCase())) return false;

            // Category Filter
            if (categoryFilter === 'all') return true;
            if (categoryFilter === 'AI') {
                return site.category === 'AI' || site.isAiPowered;
            }
            return site.category === categoryFilter;
        });

    // Helper to parse "15-40%", "2M+", "500k" into numbers
    const parseMetric = (val: string): number => {
        if (!val) return 0;
        // Remove symbols
        const clean = val.replace(/[%>+]/g, '').trim();

        // Handle Range (e.g., "15-40" -> take max 40 for 'best')
        if (clean.includes('-')) {
            const parts = clean.split('-').map(p => parseFloat(p));
            return Math.max(...parts) || 0;
        }

        let num = parseFloat(clean);
        if (isNaN(num)) return 0;

        if (val.includes('M')) num *= 1000000;
        else if (val.includes('k')) num *= 1000;

        return num;
    };

    // Sorting Logic
    const sortedSites = [...filteredSites].sort((a, b) => {
        if (sortOption === 'alpha') return a.name.localeCompare(b.name);
        if (sortOption === 'category') return a.category.localeCompare(b.category);
        if (sortOption === 'recent') return parseInt(b.id) - parseInt(a.id);
        if (sortOption === 'commission') return parseMetric(b.commission) - parseMetric(a.commission);
        if (sortOption === 'users') return parseMetric(b.contributors) - parseMetric(a.contributors);
        if (sortOption === 'library') return parseMetric(b.librarySize) - parseMetric(a.librarySize);
        return 0; // Popular (default order in array or by ID ascending)
    });

    const toggleAll = () => {
        const filteredIds = new Set(sortedSites.map(s => s.id));
        const allFilteredSelected = sortedSites.every(s => selectedSites.has(s.id));

        const newSet = new Set(selectedSites);
        if (allFilteredSelected) {
            // Deselect all currently filtered
            filteredIds.forEach(id => newSet.delete(id));
        } else {
            // Select all currently filtered
            filteredIds.forEach(id => newSet.add(id));
        }
        setSelectedSites(newSet);
    };

    const toggleSite = (id: string) => {
        const newSet = new Set(selectedSites);
        if (newSet.has(id)) newSet.delete(id);
        else newSet.add(id);
        setSelectedSites(newSet);
    };

    return (
        <div className="animate-fade-in space-y-8">
            {/* Header Actions */}
            <div className="flex flex-col gap-4">
                {/* Combined Row for Categories and Tools - Align Items to Center */}
                <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4">
                    {/* Categories List - Takes remaining space */}
                    <div className="flex flex-wrap gap-2 flex-1">
                        {CATEGORIES.map(cat => (
                            <button
                                key={cat.id}
                                onClick={() => setCategoryFilter(cat.id)}
                                className={`px-3 py-1.5 rounded-full text-xs font-bold transition-all border whitespace-nowrap ${categoryFilter === cat.id
                                    ? 'bg-slate-200 dark:bg-slate-100 text-slate-900 border-slate-300 dark:border-white'
                                    : `${cat.color} bg-opacity-5 hover:bg-opacity-20 border border-transparent opacity-80 hover:opacity-100`
                                    }`}
                            >
                                {cat.label}
                            </button>
                        ))}
                    </div>

                    {/* Tools (Sort, Bulk Select, Count) - Right Aligned */}
                    <div className="flex items-center gap-3 shrink-0 ml-auto lg:ml-0">
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-1 flex items-center">
                            <select
                                value={sortOption} onChange={(e) => setSortOption(e.target.value)}
                                className="bg-transparent text-xs font-bold text-slate-700 dark:text-slate-400 outline-none px-2 py-1 cursor-pointer hover:text-slate-900 dark:hover:text-white"
                            >
                                <option value="popular" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Most Popular</option>
                                <option value="alpha" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Alphabetical</option>
                                <option value="category" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">By Category</option>
                                <option value="recent" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Recently Added</option>
                                <option value="commission" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Best Commission</option>
                                <option value="users" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Most Contributors</option>
                                <option value="library" className="text-slate-900 dark:text-slate-300 bg-white dark:bg-slate-900">Biggest Library</option>
                            </select>
                        </div>

                        <button
                            onClick={toggleAll}
                            className="px-3 py-1.5 bg-white dark:bg-slate-800 hover:bg-slate-100 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-300 text-xs font-bold rounded-lg transition border border-slate-200 dark:border-slate-700"
                        >
                            {sortedSites.every(s => selectedSites.has(s.id)) && sortedSites.length > 0 ? 'Deselect All' : 'Select All'}
                        </button>

                        <div className="bg-amber-500/10 border border-amber-500/20 px-3 py-1.5 rounded-lg text-amber-600 dark:text-amber-400 text-xs font-bold">
                            {selectedSites.size} selected
                        </div>
                    </div>
                </div>

                {/* Internal Search */}
                {!searchFilter && (
                    <div className="relative">
                        <Search className="absolute left-3 top-2.5 text-slate-400 dark:text-slate-500 w-4 h-4" />
                        <input
                            value={internalSearch} onChange={e => setInternalSearch(e.target.value)}
                            type="text" placeholder="Filter sites..."
                            className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-2 pl-10 pr-4 text-sm focus:ring-1 focus:ring-amber-500 outline-none text-slate-900 dark:text-slate-200"
                        />
                    </div>
                )}
            </div>

            {/* Grid */}
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-4 gap-4">
                {sortedSites.map(site => {
                    const CategoryStyle = CATEGORIES.find(c => c.id === site.category) || CATEGORIES[0];
                    return (
                        <div
                            key={site.id}
                            className={`
                                relative p-6 rounded-2xl border transition-all group hover:-translate-y-1 duration-300
                                ${selectedSites.has(site.id)
                                    ? 'bg-amber-500/5 border-amber-500/50 shadow-[0_0_30px_rgba(245,158,11,0.05)]'
                                    : 'bg-white dark:bg-slate-900/40 border-slate-200 dark:border-slate-800 hover:border-slate-300 dark:hover:border-slate-600 hover:bg-slate-50 dark:hover:bg-slate-900'}
                            `}
                        >
                            <div className="flex justify-between items-start mb-4">
                                <div className="flex items-center gap-3">
                                    {/* Logo Placeholder */}
                                    <div className="w-10 h-10 rounded-lg bg-gradient-to-br from-slate-100 to-slate-200 dark:from-slate-800 dark:to-slate-700 flex items-center justify-center text-lg font-bold text-slate-600 dark:text-slate-300 shadow-inner">
                                        {site.name[0]}
                                    </div>
                                    <div>
                                        <h3 className="font-bold text-slate-900 dark:text-slate-100 leading-tight group-hover:text-amber-600 dark:group-hover:text-amber-400 transition-colors">{site.name}</h3>
                                        <div className="flex gap-1 mt-1">
                                            {site.category !== 'AI' && (
                                                <span className={`text-[10px] font-bold px-1.5 py-0.5 rounded border ${CategoryStyle.color.replace('bg-', 'border-').replace('text-', 'text-')} bg-transparent`}>
                                                    {CategoryStyle.label}
                                                </span>
                                            )}
                                            {site.isAiPowered && (
                                                <span className="text-[10px] font-bold px-1.5 py-0.5 rounded border bg-cyan-500/10 text-cyan-600 dark:text-cyan-400 border-cyan-500/20">
                                                    AI Content
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Checkbox with visible Dot */}
                                <label className="relative flex items-center justify-center w-6 h-6 cursor-pointer group/check" title="Include in Trend Search">
                                    <input
                                        type="checkbox"
                                        checked={selectedSites.has(site.id)}
                                        onChange={() => toggleSite(site.id)}
                                        className="hidden"
                                    />
                                    <div className={`w-5 h-5 rounded-md border-2 flex items-center justify-center transition-all duration-200 ${selectedSites.has(site.id)
                                        ? 'bg-amber-500 border-amber-500'
                                        : 'bg-white dark:bg-slate-950 border-slate-300 dark:border-slate-600 group-hover/check:border-amber-400'
                                        }`}>
                                        {selectedSites.has(site.id) && <div className="w-2.5 h-2.5 bg-white rounded-full shadow-sm" />}
                                    </div>
                                </label>
                            </div>

                            <p className="text-sm text-slate-700 dark:text-slate-400 mb-6 line-clamp-2 min-h-[40px]">
                                {site.description}
                            </p>

                            <div className="grid grid-cols-3 gap-2 mb-4">
                                <StatBox label="CONTRIBS" value={site.contributors} />
                                <StatBox label="LIBRARY" value={site.librarySize} />
                                <StatBox label="COMM." value={site.commission} />
                            </div>

                            {/* Pros & Cons */}
                            <div className="grid grid-cols-2 gap-2 mb-4">
                                <div className="p-2 bg-green-50 dark:bg-green-900/10 border border-green-100 dark:border-green-500/20 rounded-lg">
                                    <div className="flex items-center gap-1 text-green-700 dark:text-green-400 font-bold text-[10px] mb-1 uppercase tracking-wider">
                                        <ThumbsUp size={10} /> Pros
                                    </div>
                                    <ul className="text-[10px] text-slate-600 dark:text-slate-400 space-y-0.5 leading-tight">
                                        {site.pros && site.pros.slice(0, 3).map((pro: string, i: number) => (
                                            <li key={i} className="flex gap-1">
                                                <span className="text-green-500">•</span> {pro}
                                            </li>
                                        ))}
                                    </ul>
                                </div>
                                <div className="p-2 bg-red-50 dark:bg-red-900/10 border border-red-100 dark:border-red-500/20 rounded-lg">
                                    <div className="flex items-center gap-1 text-red-700 dark:text-red-400 font-bold text-[10px] mb-1 uppercase tracking-wider">
                                        <ThumbsDown size={10} /> Cons
                                    </div>
                                    <ul className="text-[10px] text-slate-600 dark:text-slate-400 space-y-0.5 leading-tight">
                                        {site.cons && site.cons.slice(0, 3).map((con: string, i: number) => (
                                            <li key={i} className="flex gap-1">
                                                <span className="text-red-500">•</span> {con}
                                            </li>
                                        ))}
                                    </ul>
                                </div>
                            </div>

                            <div className="flex items-center justify-between mt-auto">
                                {site.isAiPowered && (
                                    <div className="flex items-center gap-1 text-[10px] font-bold text-cyan-600 dark:text-cyan-400 px-2 py-1 bg-cyan-100 dark:bg-cyan-500/10 rounded-full border border-cyan-200 dark:border-cyan-500/20">
                                        <BrainCircuit size={12} /> AI-Generative
                                    </div>
                                )}
                                <div className={`flex gap-2 ${!site.isAiPowered ? 'ml-auto' : ''}`}>
                                    <a
                                        href={site.url}
                                        target="_blank"
                                        rel="noreferrer"
                                        className="flex items-center gap-2 px-3 py-2 bg-white dark:bg-slate-950 border border-amber-500/30 hover:bg-amber-500 hover:text-white dark:hover:text-slate-950 text-amber-600 dark:text-amber-500 text-xs font-bold rounded-lg transition shadow-lg shadow-amber-500/5"
                                    >
                                        Visit <ExternalLink size={12} />
                                    </a>
                                </div>
                            </div>
                        </div>
                    );
                })}
            </div>
            {viewingSite && <SiteDetailsModal site={viewingSite} onClose={() => setViewingSite(null)} />}
        </div>
    );
};

const SiteDetailsModal = ({ site, onClose }: { site: StockSite, onClose: () => void }) => (
    <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4" onClick={onClose}>
        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-2xl overflow-hidden shadow-2xl relative animate-fade-in-up" onClick={e => e.stopPropagation()}>
            <button onClick={onClose} className="absolute top-4 right-4 p-2 bg-slate-100 dark:bg-slate-800 rounded-full text-slate-600 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white transition">
                <X size={18} />
            </button>
            <div className="p-8">
                <div className="flex items-center gap-4 mb-6">
                    <div className="w-16 h-16 rounded-2xl bg-gradient-to-br from-slate-100 to-slate-200 dark:from-slate-800 dark:to-slate-700 flex items-center justify-center text-2xl font-bold text-slate-600 dark:text-slate-300">
                        {site.name[0]}
                    </div>
                    <div>
                        <h2 className="text-2xl font-bold text-slate-900 dark:text-white">{site.name}</h2>
                        <div className="flex items-center gap-2 mt-1">
                            <span className="px-2 py-0.5 rounded bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700 text-xs font-bold text-slate-700 dark:text-slate-300">
                                {site.category}
                            </span>
                            {site.isAiPowered && <span className="px-2 py-0.5 rounded bg-cyan-100 dark:bg-cyan-900/30 border border-cyan-200 dark:border-cyan-500/30 text-xs font-bold text-cyan-600 dark:text-cyan-400">AI Content</span>}
                        </div>
                    </div>
                </div>
                <p className="text-slate-700 dark:text-slate-300 leading-relaxed mb-6">{site.description}</p>

                <div className="grid grid-cols-3 gap-4 mb-8">
                    <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800">
                        <p className="text-xs text-slate-500 uppercase font-bold mb-1">Contributors</p>
                        <p className="text-xl font-bold text-slate-900 dark:text-white">{site.contributors}</p>
                    </div>
                    <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800">
                        <p className="text-xs text-slate-500 uppercase font-bold mb-1">Asset Count</p>
                        <p className="text-xl font-bold text-slate-900 dark:text-white">{site.librarySize}</p>
                    </div>
                    <div className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800">
                        <p className="text-xs text-slate-500 uppercase font-bold mb-1">Commission</p>
                        <p className="text-xl font-bold text-emerald-600 dark:text-emerald-400">{site.commission}</p>
                    </div>
                </div>

                <div className="space-y-3 mb-8">
                    <h3 className="text-sm font-bold text-slate-900 dark:text-white uppercase">Pros & Cons</h3>
                    <div className="flex gap-4">
                        <div className="flex-1 p-3 bg-green-100 dark:bg-green-900/10 border border-green-200 dark:border-green-500/20 rounded-lg">
                            <div className="flex items-center gap-2 text-green-600 dark:text-green-400 font-bold text-xs mb-2"><ThumbsUp size={12} /> Pros</div>
                            <ul className="text-xs text-slate-700 dark:text-slate-400 space-y-1 list-disc list-inside">
                                {site.pros && site.pros.length > 0 ? (
                                    site.pros.map((pro: string, i: number) => (
                                        <li key={i}>{pro}</li>
                                    ))
                                ) : (
                                    <li>No specific pros listed.</li>
                                )}
                            </ul>
                        </div>
                        <div className="flex-1 p-3 bg-red-100 dark:bg-red-900/10 border border-red-200 dark:border-red-500/20 rounded-lg">
                            <div className="flex items-center gap-2 text-red-600 dark:text-red-400 font-bold text-xs mb-2"><ThumbsDown size={12} /> Cons</div>
                            <ul className="text-xs text-slate-700 dark:text-slate-400 space-y-1 list-disc list-inside">
                                {site.cons && site.cons.length > 0 ? (
                                    site.cons.map((con: string, i: number) => (
                                        <li key={i}>{con}</li>
                                    ))
                                ) : (
                                    <li>No specific cons listed.</li>
                                )}
                            </ul>
                        </div>
                    </div>
                </div>

                <a
                    href={site.url}
                    target="_blank"
                    rel="noreferrer"
                    className="w-full py-3 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl flex items-center justify-center gap-2 transition shadow-lg shadow-amber-500/20"
                >
                    Visit Website <ExternalLink size={16} />
                </a>
            </div>
        </div>
    </div>
);

const StatBox = ({ label, value }: any) => (
    <div className="flex flex-col items-center justify-center py-3 px-2 rounded-xl border border-slate-100 dark:border-slate-800 bg-slate-50/50 dark:bg-slate-900/50 transition-all hover:border-slate-200 dark:hover:border-slate-700 shadow-sm dark:shadow-none">
        <p className="text-[10px] font-bold text-slate-400 dark:text-slate-500 uppercase tracking-widest mb-1.5">{label}</p>
        <p className="text-sm font-bold text-slate-900 dark:text-slate-200 truncate w-full text-center font-['Space_Grotesk']">{value}</p>
    </div>
);

// --- Tab 2: Trend Analysis ---

const TrendsTab = ({ initialQuery, status, setStatus, report, setReport, selectedSiteCount, selectedSiteIds, setGenerationInfo, setParentActiveTab, sitesList, onExportPDF, setActiveModels }: any) => {
    const [searchQuery, setSearchQuery] = useState(initialQuery || '');
    const [history, setHistory] = useState(['AI Technology', 'Sustainable Living', 'Remote Work', 'Cybersecurity']);
    const [hasApiKey, setHasApiKey] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [rateLimitError, setRateLimitError] = useState(false);

    useEffect(() => {
        if (initialQuery) setSearchQuery(initialQuery);
    }, [initialQuery]);

    useEffect(() => {
        const checkKey = async () => {
            const currentUser = authService.getCurrentUser();
            if (!currentUser) {
                setHasApiKey(false);
                return;
            }
            const keys = await authService.getUserApiKeys(currentUser.uid);
            const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
            setHasApiKey(!!validKey);
        };
        checkKey();
        // Listen for focus to re-check key status in case user added it in another tab
        const handleFocus = () => {
            checkKey();
        }
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    const generateReport = async () => {
        setError(null);
        if (!searchQuery) return;
        if (selectedSiteIds.length === 0) {
            setError("Please select at least one website from the 'Microstock Websites' tab to analyze.");
            return;
        }

        // 1. Check for User API Key
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        const keys = await authService.getUserApiKeys(currentUser.uid);
        // Get all valid/unchecked enabled keys
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);

        if (enabledKeys.length === 0) {
            window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }));
            return;
        }

        // Random Rotation Logic
        let keysToUse = enabledKeys.map(k => k.fullKey.trim());
        const isRandomRotation = localStorage.getItem('sf_random_key_rotation') === 'true';
        if (isRandomRotation) {
            console.log("🔀 [MarketIntelligence] Randomized Key Rotation Enabled. Shuffling keys...");
            keysToUse = keysToUse.sort(() => Math.random() - 0.5);
        }

        setStatus('loading');

        // Apply Data Delay / Global Cooldown
        const prefs = (currentUser as any)?.preferences?.generationConfig;
        const globalCooldownSec = Number(prefs?.globalCooldown || 0);
        const dataDelaySec = Number(prefs?.metadataDelay || 0);

        // Additive Delay Logic: Global Cooldown + Data Delay
        const effectiveDelaySec = globalCooldownSec + dataDelaySec;

        if (effectiveDelaySec > 0) {
            const cooldownMs = effectiveDelaySec * 1000;
            console.log(`[MarketIntelligence] Applying Total Delay: ${cooldownMs}ms (${globalCooldownSec}s Global + ${dataDelaySec}s Data)`);
            await new Promise(r => setTimeout(r, cooldownMs));
        }

        try {
            const usedModel = getPreferredModels('thinking')[0] || 'gemini-2.0-flash-thinking-exp';
            setGenerationInfo({
                apiKey: keysToUse[0], // Display the primary key designated for this run
                model: usedModel
            });

            // Pass ALL keys to the service for robust failover
            // Map IDs to Names for the Prompt (so the LLM generates readable text like "Adobe Stock" instead of "site_176...")
            const selectedSiteNames = selectedSiteIds.map((id: string) => sitesList.find((s: any) => s.id === id)?.name || id);

            const { report: data, modelUsed: actualModel } = await import('../services/geminiService').then(m => m.generateTrendAnalysis(keysToUse, searchQuery, selectedSiteNames));

            // Update Active Model UI to reflect fallback
            console.log("DEBUG: Setting Active Model UI to fallback:", actualModel);
            setActiveModels([actualModel]);

            // Inject selected platforms for persistence
            data.selectedPlatforms = selectedSiteIds;
            setReport(data);

            // Save to History
            const asset: Asset = {
                id: Math.random().toString(36).substring(2, 10),
                type: 'report',
                url: '', // Placeholder
                prompt: searchQuery,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: data,
                batchId: Date.now().toString() // Add batchId
            };
            await dbService.add(asset);

            // --- DEDUCT CREDITS & INCREMENT STATS (Independent Ops) ---
            if (currentUser) {
                // Op 1: Deduct Credits using ACTUAL model (raw ID) so optimistic update matches ModelCard expectation
                try {
                    console.log(`[MarketIntelligence] Deducting credits for ACTUAL model: "${actualModel}"`);
                    await authService.deductCredits(currentUser.uid, 1, keysToUse[0], actualModel);
                    console.log('[MarketIntelligence] Credits deducted successfully.');
                } catch (err) {
                    console.error('[MarketIntelligence] Failed to deduct credits:', err);
                }

                // Op 2: Increment Stats using ACTUAL model that was used
                try {
                    console.log(`[MarketIntelligence] Incrementing stats for model: "${actualModel}"`);
                    await authService.incrementUserStat('trendSearches', 1, actualModel);
                    console.log('[MarketIntelligence] Trend searches incremented successfully.');

                    // 3. Log Activity
                    await authService.logActivity(currentUser.uid, 'generate', `Trend Analysis: "${searchQuery}"`, 1);

                    // No need to call getUserProfile - incrementUserStat and deductCredits already update
                } catch (err) {
                    console.error('[MarketIntelligence] Failed to update stats:', err);
                }
            }

            setStatus('complete');

            // Play success sound
            await playSound('success');

            // Auto-download PDF if enabled
            try {
                console.log('[AUTO-DOWNLOAD] Starting auto-download check...');
                console.log('[AUTO-DOWNLOAD] currentUser:', currentUser);
                console.log('[AUTO-DOWNLOAD] onExportPDF:', onExportPDF);

                if (currentUser && onExportPDF) {
                    // Fetch preferences from backend
                    console.log('[AUTO-DOWNLOAD] Fetching preferences from backend...');
                    const prefsUrl = `http://localhost:3001/api/user/preferences?uid=${currentUser.uid}`;
                    console.log('[AUTO-DOWNLOAD] URL:', prefsUrl);

                    const prefsResponse = await fetch(prefsUrl, {
                        headers: authService.getHeaders()
                    });

                    console.log('[AUTO-DOWNLOAD] Response status:', prefsResponse.status);

                    if (prefsResponse.ok) {
                        const prefsData = await prefsResponse.json();
                        console.log('[AUTO-DOWNLOAD] Preferences data:', prefsData);
                        console.log('[AUTO-DOWNLOAD] uiConfig:', prefsData?.data?.uiConfig);
                        console.log('[AUTO-DOWNLOAD] autoDownload value:', prefsData?.data?.uiConfig?.autoDownload);

                        if (prefsData?.data?.uiConfig?.autoDownload) {
                            console.log('✅ [AUTO-DOWNLOAD] Auto-download is ENABLED, triggering PDF generation...');
                            // Small delay to ensure report is fully rendered
                            setTimeout(() => {
                                console.log('[AUTO-DOWNLOAD] Calling onExportPDF with data...');
                                onExportPDF(data);
                            }, 1000);
                        } else {
                            console.log('⚠️ [AUTO-DOWNLOAD] Auto-download is DISABLED in preferences');
                        }
                    } else {
                        console.error('[AUTO-DOWNLOAD] Failed to fetch preferences:', prefsResponse.status);
                    }
                } else {
                    console.log('[AUTO-DOWNLOAD] Skipped - missing currentUser or onExportPDF');
                }
            } catch (autoDownloadError) {
                console.error('[AUTO-DOWNLOAD] Error:', autoDownloadError);
                // Don't throw - auto-download is optional
            }

            // Redirect to History - REMOVED per user request
            // if (setParentActiveTab) {
            //    setParentActiveTab('history');
            // }
        } catch (e) {
            console.error(e);
            setRateLimitError(true);
            setStatus('idle');

            // Play error sound
            await playSound('error');
        }
    };

    const reportRef = useRef<HTMLDivElement>(null);
    const [isExporting, setIsExporting] = useState(false);

    const handleExportPDF = async () => {
        if (onExportPDF) {
            setIsExporting(true);
            await onExportPDF(report);
            setIsExporting(false);
        }
    };

    return (
        <div className="animate-fade-in">
            {status === 'idle' && (
                <div className="max-w-2xl mx-auto text-center pt-10">

                    <div className="mb-8">
                        <h2 className="text-3xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk'] mb-3">Analyze Market Trends</h2>
                        <p className="text-slate-600 dark:text-slate-400 text-lg">Discover high-demand, low-competition niches across {selectedSiteCount > 0 ? selectedSiteCount : 'all'} platforms.</p>
                    </div>

                    {error && (
                        <div className="mb-6 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-500/30 rounded-xl flex items-center gap-3 text-left animate-shake">
                            <div className="p-2 bg-red-100 dark:bg-red-500/20 rounded-lg text-red-600 dark:text-red-400 shrink-0">
                                <AlertTriangle size={20} />
                            </div>
                            <div>
                                <h4 className="font-bold text-red-700 dark:text-red-400 text-sm">Action Required</h4>
                                <p className="text-xs text-red-600 dark:text-red-300">{error}</p>
                            </div>
                        </div>
                    )}

                    <RateLimitPopup isOpen={rateLimitError} onClose={() => setRateLimitError(false)} />

                    <div className={`bg-white dark:bg-slate-900 p-2 rounded-2xl shadow-lg border border-slate-200 dark:border-slate-800 mb-8 transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>
                        <div className="flex items-center bg-slate-50 dark:bg-slate-900 rounded-xl px-4 py-2 border border-slate-200 dark:border-slate-800 focus-within:ring-2 focus-within:ring-amber-500 transition-all">
                            <Search className="text-slate-400" size={20} />
                            <input
                                type="text"
                                value={searchQuery}
                                onChange={(e) => { setSearchQuery(e.target.value); if (error) setError(null); }}
                                placeholder="Enter a topic, keyword, or niche (e.g. 'Sustainable Living')..."
                                className="flex-1 bg-transparent border-none outline-none px-4 py-3 text-slate-900 dark:text-white placeholder-slate-500 font-medium"
                                onKeyDown={(e) => e.key === 'Enter' && generateReport()}
                                disabled={!hasApiKey}
                            />
                            <button
                                onClick={generateReport}
                                disabled={!hasApiKey}
                                className="bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold py-2 px-6 rounded-lg transition flex items-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                Analyze <ArrowRight size={18} />
                                <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">1 API Call</span>
                            </button>
                        </div>
                    </div>

                    {/* Quick Chips */}
                    <div className={`flex flex-wrap justify-center gap-2 mb-12 ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>
                        {history.map((item, i) => (
                            <button key={i} onClick={() => { setSearchQuery(item); if (error) setError(null); }} className="px-4 py-2 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-full text-sm font-medium text-slate-600 dark:text-slate-400 hover:border-amber-500 hover:text-amber-600 dark:hover:text-amber-500 transition flex items-center gap-2">
                                <TrendingUp size={14} /> {item}
                            </button>
                        ))}
                    </div>
                </div>
            )}

            {status === 'complete' && report && (
                <div ref={reportRef}>
                    <TrendReportView report={report} sitesList={sitesList}>
                        <button onClick={() => setStatus('idle')} className="px-4 py-2 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 rounded-xl text-slate-600 dark:text-slate-300 font-bold text-sm transition">New Search</button>
                        <button
                            onClick={handleExportPDF}
                            disabled={isExporting}
                            className={`px-4 py-2.5 bg-transparent hover:bg-amber-50 dark:hover:bg-amber-900/10 text-amber-600 dark:text-amber-500 font-bold rounded-lg transition border border-amber-200 dark:border-amber-900/30 flex items-center justify-center gap-2 ${isExporting ? 'opacity-70 cursor-wait' : ''}`}
                        >
                            {isExporting ? (
                                <>
                                    <Loader2 size={16} className="animate-spin" /> Exporting...
                                </>
                            ) : (
                                <>
                                    <Download size={16} /> DOWNLOAD PDF
                                </>
                            )}
                        </button>
                    </TrendReportView>
                </div>
            )}
        </div>
    );
};

// --- Main Component ---

const MarketIntelligence: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'sites' | 'trends' | 'history'>('sites');
    const [selectedSites, setSelectedSites] = useState<Set<string>>(new Set());

    const [reportStatus, setReportStatus] = useState<'idle' | 'loading' | 'complete'>('idle');
    const [report, setReport] = useState<TrendReport | null>(null);

    // History State
    const [historyReports, setHistoryReports] = useState<Asset[]>([]);

    // Active Models State for Header
    const [activeModels, setActiveModels] = useState<string[]>([]);
    const [hasApiKey, setHasApiKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [sitesList, setSitesList] = useState<StockSite[]>([]);

    useEffect(() => {
        // Fetch sites from adminService which handles localStorage
        const loadSites = async () => {
            const data = await adminService.getMicrostockSites();
            console.log('[MarketIntelligence] Loaded sites:', data.length, data);
            console.log('[MarketIntelligence] Sites with isVisible=true:', data.filter((s: any) => s.isVisible).length);
            setSitesList(data.filter((s: any) => s.isVisibleMarket)); // Filter by new Market visibility toggle
        };
        loadSites();
    }, []);

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('report').then(setHistoryReports);
        }
    }, [activeTab]);
    const [generationInfo, setGenerationInfo] = useState<{ apiKey: string, model: string } | null>(null);

    useEffect(() => {
        // Initial load
        setActiveModels(getPreferredModels('thinking'));

        const checkKey = async () => {
            const currentUser = authService.getCurrentUser();
            if (!currentUser) {
                setHasApiKey(false);
                return;
            }
            const keys = await authService.getUserApiKeys(currentUser.uid);
            const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
            setHasApiKey(!!validKey);
        };
        checkKey();

        const handleFocus = () => {
            checkKey();
        };
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('report').then(setHistoryReports);
        }
    }, [activeTab]);

    const handleLoadReport = (savedReport: Asset) => {
        setReport(savedReport.metadata);
        setReportStatus('complete');
        setActiveTab('trends');
    };

    // --- PDF Export Logic for History ---
    const hiddenReportRef = useRef<HTMLDivElement>(null);
    const [pdfExportReport, setPdfExportReport] = useState<TrendReport | null>(null);
    const [isExportingHistory, setIsExportingHistory] = useState(false);
    const [downloadingReportId, setDownloadingReportId] = useState<string | null>(null);
    const pdfResolveRef = useRef<(() => void) | null>(null);

    // Batch State
    const [batchQueue, setBatchQueue] = useState<TrendReport[]>([]);
    const [pdfBlobs, setPdfBlobs] = useState<{ url: string, filename: string }[]>([]);
    const [isBatchMode, setIsBatchMode] = useState(false);
    const [batchModeType, setBatchModeType] = useState<'zip' | 'separate'>('zip');

    useEffect(() => {
        if (pdfExportReport && hiddenReportRef.current) {
            const generatePDF = async () => {
                setIsExportingHistory(true);
                // Small delay to ensure render
                await new Promise(resolve => setTimeout(resolve, 500));

                try {
                    if (!hiddenReportRef.current) return;

                    const canvas = await html2canvas(hiddenReportRef.current, {
                        scale: 2,
                        useCORS: true,
                        logging: false,
                        backgroundColor: '#0f172a'
                    });

                    const imgData = canvas.toDataURL('image/png');
                    const imgWidth = 210;
                    const imgHeight = (canvas.height * imgWidth) / canvas.width;

                    const pdf = new jsPDF({
                        orientation: 'portrait',
                        unit: 'mm',
                        format: [imgWidth, imgHeight]
                    });

                    pdf.addImage(imgData, 'PNG', 0, 0, imgWidth, imgHeight);
                    const filename = `Trend_Report_${pdfExportReport.query.replace(/\s+/g, '_')}_${Date.now()}.pdf`;

                    if (isBatchMode) {
                        const blob = pdf.output('blob');
                        const url = URL.createObjectURL(blob);
                        setPdfBlobs(prev => [...prev, { url, filename }]);

                        // Proceed to next in queue
                        if (batchQueue.length > 0) {
                            const nextReport = batchQueue[0];
                            setBatchQueue(prev => prev.slice(1));
                            setPdfExportReport(nextReport);
                        } else {
                            // Done!
                            setPdfExportReport(null);
                        }
                    } else {
                        pdf.save(filename);
                        setPdfExportReport(null);
                    }

                } catch (err) {
                    console.error("History PDF Export Failed:", err);
                    // If batch, maybe continue?
                    if (isBatchMode && batchQueue.length > 0) {
                        const nextReport = batchQueue[0];
                        setBatchQueue(prev => prev.slice(1));
                        setPdfExportReport(nextReport);
                    } else {
                        setPdfExportReport(null);
                    }
                } finally {
                    setDownloadingReportId(null);
                    setIsExportingHistory(false);
                    if (pdfResolveRef.current) {
                        pdfResolveRef.current();
                        pdfResolveRef.current = null;
                    }
                }
            };
            generatePDF();
        } else if (isBatchMode && batchQueue.length === 0 && !pdfExportReport && pdfBlobs.length > 0) {
            // Batch complete, trigger download
            console.log(`[BatchComplete] Mode: ${batchModeType}, Items: ${pdfBlobs.length}`);
            downloadItems(pdfBlobs, `trend_reports_batch_${Date.now()}`, batchModeType === 'separate');
            setIsBatchMode(false);
            setPdfBlobs([]);
        }
    }, [pdfExportReport, batchQueue.length, isBatchMode, batchModeType]); // Added dependencies

    const handleDownloadAllHistory = (mode: 'zip' | 'separate') => {
        if (historyReports.length === 0) return;

        const reports = historyReports.map(h => h.metadata).filter(Boolean);
        if (reports.length === 0) return;

        setBatchModeType(mode);
        setIsBatchMode(true);
        setPdfBlobs([]);
        // Start with first
        setPdfExportReport(reports[0]);
        // Queue rest
        setBatchQueue(reports.slice(1));
    };

    const handleDownloadHistoryPDF = (report: TrendReport) => {
        return new Promise<void>((resolve) => {
            pdfResolveRef.current = resolve;
            setDownloadingReportId(report.id);
            setPdfExportReport(report);
        });
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 min-h-screen flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">

            {/* Hidden div for PDF generation */}
            {pdfExportReport && (
                <div ref={hiddenReportRef} style={{ position: 'absolute', left: '-9999px', top: '-9999px', width: '1000px', height: 'auto', zIndex: -1 }}>
                    <TrendReportView report={pdfExportReport} sitesList={sitesList} />
                </div>
            )}

            {/* Header */}
            <div className="mb-6 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <BrainCircuit className="text-amber-500 w-8 h-8" /> Market Intelligence Hub
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">Analyze microstock trends, discover profitable niches, and explore top agencies.</p>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in text-left shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">Market intelligence requires deep reasoning. Please add your Gemini API key.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add Key
                    </button>
                </div>
            )}

            {/* Tab Navigation */}
            <div className="flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-6 shrink-0">
                <div className="flex w-full md:w-auto overflow-x-auto">
                    <button
                        onClick={() => setActiveTab('sites')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'sites'
                            ? 'text-amber-600 dark:text-amber-400'
                            : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <Globe size={16} />
                        Microstock Websites
                        {activeTab === 'sites' && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in" />}
                    </button>
                    <button
                        onClick={() => setActiveTab('trends')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'trends'
                            ? 'text-amber-600 dark:text-amber-400'
                            : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <TrendingUp size={16} />
                        Trend Analysis
                        {activeTab === 'trends' && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in" />}
                    </button>
                    <button
                        onClick={() => setActiveTab('history')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'history'
                            ? 'text-amber-600 dark:text-amber-400'
                            : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <History size={16} />
                        History
                        {activeTab === 'history' && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in" />}
                    </button>
                </div>

                {/* Right Side Actions: Reviewer History Download & Active Models */}
                <div className="hidden md:flex items-center gap-4 my-2 mr-2">
                    {/* "DOWNLOAD ALL" Button - Only visible in History tab */}
                    {activeTab === 'history' && historyReports.length > 0 && (
                        <div className="flex items-center gap-2">
                            <button
                                onClick={() => handleDownloadAllHistory('separate')}
                                disabled={isBatchMode}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-slate-700 text-slate-400 hover:bg-slate-900 hover:text-white font-bold rounded-xl transition text-[10px] uppercase"
                                title="Download each report as separate PDF files"
                            >
                                {isBatchMode && batchModeType === 'separate' ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                Separate All
                            </button>
                            <button
                                onClick={() => handleDownloadAllHistory('zip')}
                                disabled={isBatchMode}
                                className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 font-bold rounded-xl transition text-[10px] uppercase"
                                title="Download all reports in a single ZIP archive"
                            >
                                {isBatchMode && batchModeType === 'zip' ? <Loader2 size={14} className="animate-spin" /> : <FileArchive size={14} />}
                                Zip All
                            </button>
                        </div>
                    )}

                    {/* Active Models Display & Auto-Download Indicator (Only if NOT in history, or generally available) */}
                    {activeTab !== 'history' && (
                        <>
                            <div className="flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg">
                                <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                                    <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                                </span>
                                <div className="flex gap-1">
                                    {activeModels.map((model, idx) => (
                                        <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                            {model}
                                        </span>
                                    ))}
                                    {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                                </div>
                            </div>
                            <AutoDownloadIndicator />
                        </>
                    )}
                </div>
            </div>

            <div className="flex-1 w-full">
                {activeTab === 'sites' && (
                    <WebsitesTab
                        searchFilter=""
                        selectedSites={selectedSites}
                        setSelectedSites={setSelectedSites}
                        sitesList={sitesList}
                    />
                )}

                {activeTab === 'trends' && (
                    <TrendsTab
                        initialQuery=""
                        status={reportStatus}
                        setStatus={setReportStatus}
                        report={report}
                        setReport={setReport}
                        selectedSiteCount={selectedSites.size}
                        selectedSiteIds={Array.from(selectedSites)}
                        setGenerationInfo={setGenerationInfo}
                        setParentActiveTab={setActiveTab}
                        sitesList={sitesList}
                        onExportPDF={handleDownloadHistoryPDF}
                        setActiveModels={setActiveModels}
                    />
                )}

                {/* Tab 3: History */}
                {activeTab === 'history' && (
                    <div className="animate-fade-in">
                        {historyReports.length === 0 ? (
                            <div className="flex flex-col items-center justify-center py-20 text-slate-500 dark:text-slate-400">
                                <History size={48} className="mb-4 opacity-50" />
                                <p className="text-lg font-medium">No saved reports.</p>
                                <p className="text-sm">Generate a trend analysis to see it here.</p>
                            </div>
                        ) : (
                            <div className="flex flex-col space-y-4 max-w-5xl mx-auto">
                                {historyReports.slice(0, 15).map((item) => (
                                    <div key={item.id} className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl overflow-hidden shadow-sm hover:shadow-md transition group flex flex-col md:flex-row items-center p-4 gap-4">

                                        {/* Icon & Date */}
                                        <div className="flex items-center gap-4 w-full md:w-auto shrink-0">
                                            <div className="p-3 bg-amber-100 dark:bg-amber-900/20 rounded-xl text-amber-600 dark:text-amber-400">
                                                <TrendingUp size={24} />
                                            </div>
                                            <div className="md:hidden">
                                                <h3 className="font-bold text-slate-900 dark:text-white capitalize text-lg">{item.prompt}</h3>
                                                <span className="text-xs text-slate-400 font-mono">{new Date(item.createdAt).toLocaleDateString()}</span>
                                            </div>
                                        </div>

                                        {/* Content */}
                                        <div className="flex-1 w-full">
                                            <div className="hidden md:block mb-1">
                                                <h3 className="font-bold text-slate-900 dark:text-white capitalize text-lg">{item.prompt}</h3>
                                            </div>

                                            <div className="flex flex-wrap items-center gap-4 text-sm text-slate-500 dark:text-slate-400">
                                                <span className="hidden md:inline font-mono text-xs">{new Date(item.createdAt).toLocaleDateString()}</span>
                                                <div className="flex items-center gap-1.5 px-2 py-0.5 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold">
                                                    <Shield size={12} className="text-blue-500" />
                                                    <span>Score: {item.metadata?.insights?.confidenceScore || 0}/100</span>
                                                </div>
                                                <div className="flex items-center gap-1.5 px-2 py-0.5 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold">
                                                    <Activity size={12} className="text-emerald-500" />
                                                    <span>Vol: {item.metadata?.saturation?.level || 'N/A'}</span>
                                                </div>
                                                <div className="flex items-center gap-1.5 px-2 py-0.5 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold">
                                                    <Users size={12} className="text-purple-500" />
                                                    <span>Comp: {item.metadata?.saturation?.competitionLevel || 0}/10</span>
                                                </div>
                                            </div>
                                        </div>

                                        {/* Action */}
                                        <div className="flex flex-col md:flex-row gap-2 w-full md:w-auto shrink-0">
                                            <button
                                                onClick={async (e) => {
                                                    e.stopPropagation();
                                                    await handleDownloadHistoryPDF(item.metadata);
                                                }}
                                                disabled={downloadingReportId === item.metadata.id}
                                                className={`w-full md:w-auto px-4 py-2.5 bg-transparent hover:bg-amber-50 dark:hover:bg-amber-900/10 text-amber-600 dark:text-amber-500 font-bold rounded-lg transition border border-amber-200 dark:border-amber-900/30 flex items-center justify-center gap-2 ${downloadingReportId === item.metadata.id ? 'opacity-70 cursor-wait' : ''}`}
                                            >
                                                {downloadingReportId === item.metadata.id ? (
                                                    <>
                                                        <Loader2 size={16} className="animate-spin" /> DOWNLOADING...
                                                    </>
                                                ) : (
                                                    <>
                                                        <Download size={16} /> DOWNLOAD PDF
                                                    </>
                                                )}
                                            </button>
                                            <button
                                                onClick={() => {
                                                    setReport(item.metadata);
                                                    setReportStatus('complete');
                                                    setActiveTab('trends');
                                                }}
                                                className="w-full md:w-auto px-6 py-2.5 bg-slate-900 dark:bg-white hover:bg-slate-800 dark:hover:bg-slate-200 text-white dark:text-slate-900 font-bold rounded-lg transition shadow-lg shadow-slate-900/10 flex items-center justify-center gap-2 shrink-0"
                                            >
                                                View Report <ArrowRight size={16} />
                                            </button>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                )}

                {/* Loading Overlay */}
                <EnhancedLoadingOverlay
                    isVisible={reportStatus === 'loading'}
                    title="Analyzing Market Trends"
                    description={`Scanning ${selectedSites.size} sources for volume and competition metrics...`}
                    simulate={true}
                    onCancel={() => setReportStatus('idle')}
                    apiKey={generationInfo?.apiKey}
                    model={generationInfo?.model}
                />
            </div>
        </div>
    );
};

export default MarketIntelligence;
