import React, { useEffect } from 'react';
import { X, ZoomIn, ZoomOut, Download, ExternalLink } from 'lucide-react';

interface LightboxProps {
    isOpen: boolean;
    onClose: () => void;
    src: string | null;
    alt?: string;
}

export const Lightbox: React.FC<LightboxProps> = ({ isOpen, onClose, src, alt = 'Preview' }) => {
    // Prevent scrolling when lightbox is open
    useEffect(() => {
        if (isOpen) {
            document.body.style.overflow = 'hidden';
        } else {
            document.body.style.overflow = 'unset';
        }
        return () => {
            document.body.style.overflow = 'unset';
        };
    }, [isOpen]);

    // Handle escape key
    useEffect(() => {
        const handleKeyDown = (e: KeyboardEvent) => {
            if (e.key === 'Escape') onClose();
        };
        window.addEventListener('keydown', handleKeyDown);
        return () => window.removeEventListener('keydown', handleKeyDown);
    }, [onClose]);

    if (!isOpen || !src) return null;

    return (
        <div
            className="fixed inset-0 z-[100] flex items-center justify-center bg-black/90 backdrop-blur-md animate-in fade-in duration-200"
            onClick={onClose}
        >
            {/* Toolbar */}
            <div className="absolute top-4 right-4 flex gap-2 z-[110]" onClick={e => e.stopPropagation()}>
                <button
                    onClick={() => {
                        const link = document.createElement('a');
                        link.href = src;
                        link.download = `preview-${Date.now()}.png`;
                        link.click();
                    }}
                    className="p-2 rounded-full bg-white/10 hover:bg-white/20 text-white transition backdrop-blur-sm"
                    title="Download Image"
                >
                    <Download size={20} />
                </button>
                <button
                    onClick={() => window.open(src, '_blank')}
                    className="p-2 rounded-full bg-white/10 hover:bg-white/20 text-white transition backdrop-blur-sm"
                    title="Open Original"
                >
                    <ExternalLink size={20} />
                </button>
                <button
                    onClick={onClose}
                    className="p-2 rounded-full bg-white/10 hover:bg-red-500/80 text-white transition backdrop-blur-sm"
                    title="Close (Esc)"
                >
                    <X size={20} />
                </button>
            </div>

            {/* Image Container */}
            <div
                className="relative max-w-[90vw] max-h-[90vh] flex items-center justify-center p-2"
                onClick={e => e.stopPropagation()}
            >
                <img
                    src={src}
                    alt={alt}
                    className="max-w-full max-h-[90vh] object-contain rounded-lg shadow-2xl animate-in zoom-in-95 duration-300"
                />
            </div>

            <div className="absolute bottom-6 left-1/2 -translate-x-1/2 text-white/50 text-xs font-mono px-4 py-1 bg-black/40 rounded-full backdrop-blur-sm border border-white/10">
                Click anywhere to close
            </div>
        </div>
    );
};
