import React, { useState, useEffect, useRef } from 'react';
import {
    Folder, Heart, Trash2, Image as ImageIcon, Video,
    Search, Grid, List, Filter, MoreVertical, Download,
    RefreshCcw, CheckCircle2, Undo2, AlertCircle,
    ScanEye, TrendingUp, FileText, BrainCircuit,
    FileJson, Clock, Settings, ChevronDown, ChevronUp, History, Eye, X, Loader2, FileDown
} from 'lucide-react';
import { dbService, Asset, TrashConfig } from '../services/dbService';
import { AppView } from '../types';
import { DoubleConfirmDeleteButton } from './DoubleConfirmDeleteButton';
import { TrendReportView } from './TrendReportView';
import { adminService } from '../services/adminService';
import { authService } from '../services/authService';
import { StockSite, TrendReport } from '../types';
import { ADOBE_STOCK_CATEGORIES } from '../constants';
import { jsPDF } from 'jspdf';
import html2canvas from 'html2canvas';
import { downloadItems } from '../utils/downloadUtils';


interface LibraryProps {
    view: AppView; // LIBRARY, FAVORITES, TRASH, HISTORY
}

const Library: React.FC<LibraryProps> = ({ view }) => {
    const [assets, setAssets] = useState<Asset[]>([]);
    const [search, setSearch] = useState('');
    const [typeFilter, setTypeFilter] = useState<'all' | 'image' | 'video' | 'metadata' | 'report' | 'review'>('all');
    const [layout, setLayout] = useState<'grid' | 'list'>('grid');
    const [selected, setSelected] = useState<Set<string>>(new Set());
    const [previewAsset, setPreviewAsset] = useState<Asset | null>(null);
    const [downloadingAssetId, setDownloadingAssetId] = useState<string | null>(null);

    // PDF Generation State
    const [sitesList, setSitesList] = useState<StockSite[]>([]);
    const [siteSettings, setSiteSettings] = useState<any | null>(null); // Load generic settings
    const [reportToDownload, setReportToDownload] = useState<TrendReport | null>(null);
    const [downloadQueue, setDownloadQueue] = useState<TrendReport[]>([]);
    const [isZipMode, setIsZipMode] = useState(false);
    const [zipItems, setZipItems] = useState<{ url: string, filename: string }[]>([]);
    const reportRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        const loadSites = async () => {
            const data = await adminService.getMicrostockSites();
            setSitesList(data);
            const settings = await adminService.getSiteSettings();
            setSiteSettings(settings);
        };
        loadSites();
    }, []);

    // Visual PDF Generation Effect
    useEffect(() => {
        if (reportToDownload && reportRef.current) {
            const generate = async () => {
                try {
                    // Wait for render (Increased to 3s to ensure Recharts/images load)
                    await new Promise(resolve => setTimeout(resolve, 3000));

                    if (!reportRef.current) return;

                    const canvas = await html2canvas(reportRef.current, {
                        scale: 2,
                        useCORS: true,
                        allowTaint: true, // Allow cross-origin images if CORS fails
                        logging: false,
                        backgroundColor: '#ffffff', // White BG for PDF
                        windowWidth: 1600 // Ensure wide capture
                    });

                    const imgData = canvas.toDataURL('image/png');
                    const imgWidth = 210;
                    const imgHeight = (canvas.height * imgWidth) / canvas.width;

                    const doc = new jsPDF({
                        orientation: 'portrait',
                        unit: 'mm',
                        format: [imgWidth, imgHeight]
                    });

                    doc.addImage(imgData, 'PNG', 0, 0, imgWidth, imgHeight);

                    if (isZipMode) {
                        const blob = doc.output('blob');
                        const url = URL.createObjectURL(blob);
                        setZipItems(prev => [...prev, { url, filename: `${reportToDownload.query.replace(/\s+/g, '_')}_Report.pdf` }]);
                    } else {
                        doc.save(`${reportToDownload.query.replace(/\s+/g, '_')}_Report.pdf`);
                    }
                } catch (e) {
                    console.error("PDF Gen Error", e);
                } finally {
                    setReportToDownload(null);
                    setDownloadingAssetId(null); // Clear download indicator when PDF generation completes
                }
            };
            generate();
        }
    }, [reportToDownload, isZipMode]);

    // ... (rest of code) ...

    {/* Hidden Report Renderer for PDF Generation */ }
    <div style={{ position: 'fixed', left: '-10000px', top: 0, width: '1200px', zIndex: -100, visibility: 'visible' }}>
        {reportToDownload && (
            <div ref={reportRef} className="bg-white text-slate-900 p-8">
                <TrendReportView report={reportToDownload} sitesList={sitesList} isExport={true} />
            </div>
        )}
    </div>

    // Queue Processing Effect
    useEffect(() => {
        if (!reportToDownload && downloadQueue.length > 0) {
            const nextReport = downloadQueue[0];
            setDownloadQueue(prev => prev.slice(1));
            // Trigger next report
            setReportToDownload(nextReport);
        } else if (isZipMode && downloadQueue.length === 0 && !reportToDownload && zipItems.length > 0) {
            // Zip Mode Complete
            const finalZipName = view === AppView.LIBRARY ? `library_export_${Date.now()}` : `selected_assets_${Date.now()}`;
            downloadItems(zipItems, finalZipName);

            // Allow time for logic to clear before resetting
            setTimeout(() => {
                setIsZipMode(false);
                setZipItems([]);
            }, 1000);
        }
    }, [reportToDownload, downloadQueue.length, isZipMode, zipItems.length, view]);

    // Trash Logic
    const [trashTimer, setTrashTimer] = useState<string>('');
    const [trashConfig, setTrashConfig] = useState<TrashConfig>({ enabled: true, frequency: 'hourly' });
    const [isTrashSettingsOpen, setIsTrashSettingsOpen] = useState(false);
    const trashSettingsRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        setAssets([]); // Clear assets immediately to prevent showing wrong view's data during transition
        refreshData();
        const loadTrashConfig = async () => {
            if (view === AppView.TRASH) {
                setTrashConfig(await dbService.getTrashConfig());
            }
        };
        loadTrashConfig();
    }, [view]);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (trashSettingsRef.current && !trashSettingsRef.current.contains(event.target as Node)) {
                setIsTrashSettingsOpen(false);
            }
        };
        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    // Timer Logic for Trash Page
    const trashConfigRef = useRef(trashConfig);
    useEffect(() => {
        trashConfigRef.current = trashConfig;
    }, [trashConfig]);

    useEffect(() => {
        if (view !== AppView.TRASH) return;

        const updateTimer = async () => {
            const config = trashConfigRef.current;
            if (!config.enabled) {
                setTrashTimer('Disabled');
                return;
            }

            // Check if it's time to delete
            const didClear = await dbService.checkAutoClearTrash();
            if (didClear) {
                refreshData();
            }

            const nextTime = await dbService.getNextTrashClearTime(config); // Pass config from ref!
            const diff = Math.max(0, nextTime - Date.now());

            const days = Math.floor(diff / (1000 * 60 * 60 * 24));
            const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const mins = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
            const secs = Math.floor((diff % (1000 * 60)) / 1000);

            if (days > 0) {
                setTrashTimer(`${days}d ${hours}h`);
            } else if (hours > 0) {
                setTrashTimer(`${hours}h ${mins}m`);
            } else {
                setTrashTimer(`${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`);
            }
        };

        updateTimer(); // Initial call
        const interval = setInterval(updateTimer, 1000);

        return () => clearInterval(interval);
    }, [view]); // Only depend on view, not trashConfig

    const handleTrashConfigChange = async (updates: Partial<TrashConfig>) => {
        const previousConfig = { ...trashConfig };
        const newConfig = { ...trashConfig, ...updates };

        // Optimistic update
        setTrashConfig(newConfig);

        // Update ref synchronously to prevent race condition with timer
        trashConfigRef.current = newConfig;

        try {
            const newConfigToSave = { ...trashConfig, ...updates };
            // Update DB first to ensure localStorage is ready for the effect
            await dbService.setTrashConfig(newConfigToSave);

            // Force immediate timer update to show new countdown
            if (view === AppView.TRASH && newConfig.enabled) {
                const nextTime = await dbService.getNextTrashClearTime(newConfig); // Pass new config directly
                const diff = Math.max(0, nextTime - Date.now());

                const days = Math.floor(diff / (1000 * 60 * 60 * 24));
                const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                const mins = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                const secs = Math.floor((diff % (1000 * 60)) / 1000);

                if (days > 0) {
                    setTrashTimer(`${days}d ${hours}h`);
                } else if (hours > 0) {
                    setTrashTimer(`${hours}h ${mins}m`);
                } else {
                    setTrashTimer(`${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`);
                }
            }
        } catch (error) {
            console.error("Failed to update trash config:", error);
            // Revert on failure
            setTrashConfig(previousConfig);
            trashConfigRef.current = previousConfig;
            alert("Failed to save settings. Please check your connection.");
        }
    };

    const refreshData = async () => {
        try {
            let fetchedAssets: Asset[] = [];
            if (view === AppView.FAVORITES) {
                fetchedAssets = await dbService.getFavorites();
                console.log(`[Library] Fetched ${fetchedAssets.length} favorites`);
            } else if (view === AppView.TRASH) {
                fetchedAssets = await dbService.getTrash();
            } else {
                fetchedAssets = await dbService.getLibrary();
            }
            setAssets(fetchedAssets);
            setSelected(new Set());
        } catch (error) {
            console.error("[Library] Failed to refresh data:", error);
            // Optionally show a toast or error state here
        }
    };

    const handleToggleSelect = (id: string) => {
        const newSet = new Set(selected);
        if (newSet.has(id)) newSet.delete(id);
        else newSet.add(id);
        setSelected(newSet);
    };

    const handleBulkAction = async (action: 'delete' | 'restore' | 'favorite' | 'permanentDelete') => {
        if (selected.size === 0) return;

        const promises: Promise<any>[] = [];
        if (action === 'delete') {
            selected.forEach(id => promises.push(dbService.moveToTrash(id)));
        } else if (action === 'restore') {
            selected.forEach(id => promises.push(dbService.restore(id)));
        } else if (action === 'favorite') {
            selected.forEach(id => {
                const asset = assets.find(a => a.id === id);
                if (asset) {
                    promises.push(dbService.toggleFavorite(id, asset.isFavorite));
                }
            });
        } else if (action === 'permanentDelete') {
            // Confirmation handled by DoubleConfirmDeleteButton
            selected.forEach(id => promises.push(dbService.permanentDelete(id)));
        }

        await Promise.all(promises);
        refreshData();
    };

    const handleEmptyTrash = async () => {
        // Confirmation handled by DoubleConfirmDeleteButton
        await dbService.emptyTrash();
        refreshData();
    }

    const [isDeleting, setIsDeleting] = useState(false);

    const handleDeleteAllVisible = async () => {
        // Delete all currently visible assets (based on filter)
        const idsToDelete = filteredAssets.map(a => a.id);
        if (idsToDelete.length === 0) return;

        setIsDeleting(true);
        try {
            const promises = idsToDelete.map(id => dbService.moveToTrash(id));
            await Promise.all(promises);
            await new Promise(resolve => setTimeout(resolve, 500)); // Min delay for visual feedback
            refreshData();
        } catch (error) {
            console.error("Failed to delete all:", error);
        } finally {
            setIsDeleting(false);
        }
    };

    const downloadFile = async (url: string, filename: string) => {
        try {
            const response = await fetch(url);
            const blob = await response.blob();
            const blobUrl = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = blobUrl;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            URL.revokeObjectURL(blobUrl);
        } catch (e) {
            console.error("Download failed, falling back to direct link", e);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            link.target = "_blank";
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    };

    const handlePreview = (e: React.MouseEvent, asset: Asset) => {
        e.stopPropagation();
        setPreviewAsset(asset);
    };

    // Helper to generate content string based on format
    const getMetadataContent = (asset: Asset): { content: string, format: string, extension: string } => {
        const format = asset.generationSettings?.exportFormat || 'json';
        const isAdobeEnabled = siteSettings?.adobeStockInstruction && siteSettings.adobeStockInstruction.length > 0;
        const m = asset.metadata;

        if (asset.metadata?.csvContent) {
            return { content: asset.metadata.csvContent, format: 'csv', extension: 'csv' };
        }

        if (format === 'csv') {
            let csvContent = '';

            // Keyword Aggregation logic
            let allKeywords: string[] = [];
            if (Array.isArray(m.commonKeywords)) allKeywords.push(...m.commonKeywords);
            if (Array.isArray(m.longTailKeywords)) allKeywords.push(...m.longTailKeywords);
            if (allKeywords.length === 0) {
                if (Array.isArray(m.keywords)) allKeywords.push(...m.keywords);
                else if (typeof m.keywords === 'string') allKeywords.push(...m.keywords.split(',').map((k: string) => k.trim()));
            }
            const keywordsString = allKeywords.join(', ');
            const keywords = `"${keywordsString.replace(/"/g, '""')}"`;

            if (isAdobeEnabled) {
                const filename = `"${(m.filename || asset.title || 'asset')}"`;
                const title = `"${(m.title || '').replace(/"/g, '""')}"`;
                let catName = m.category || '';
                if (catName.includes('(')) catName = catName.split('(')[0].trim();
                const catId = ADOBE_STOCK_CATEGORIES[catName] || catName;
                const category = `"${catId}"`;

                csvContent = "Filename,Title,Keywords,Category\n";
                csvContent += `${filename},${title},${keywords},${category}\n`;
            } else {
                const filename = `"${(m.filename || asset.title || 'asset')}"`;
                const title = `"${(m.title || '').replace(/"/g, '""')}"`;
                const desc = `"${(m.description || '').replace(/"/g, '""')}"`;
                csvContent = "Filename,Title,Description,Keywords\n";
                csvContent += `${filename},${title},${desc},${keywords}\n`;
            }

            return { content: csvContent, format: 'csv', extension: 'csv' };
        }

        if (format === 'txt') {
            const txtContent = `Title: ${m.title}\n\nDescription:\n${m.description}\n\nKeywords:\n${m.commonKeywords?.join(', ')}, ${m.longTailKeywords?.join(', ') || ''}`;
            return { content: txtContent, format: 'txt', extension: 'txt' };
        }

        // Default JSON
        return { content: JSON.stringify(asset.metadata, null, 2), format: 'json', extension: 'json' };
    };

    // Helper to generate download payload
    const getDownloadPayload = (asset: Asset) => {
        const { content, format, extension } = getMetadataContent(asset);
        const mimeType = format === 'csv' ? 'text/csv;charset=utf-8;' : format === 'txt' ? 'text/plain' : 'application/json';
        return {
            url: URL.createObjectURL(new Blob([content], { type: mimeType })),
            filename: `metadata_${asset.id}.${extension}`
        };
    };

    const handleDownload = async (e: React.MouseEvent, asset: Asset) => {
        e.stopPropagation();

        setDownloadingAssetId(asset.id);
        try {
            if (asset.type === 'metadata') {
                const { url, filename } = getDownloadPayload(asset);
                const link = document.createElement('a');
                link.href = url;
                link.download = filename;
                link.click();
                setTimeout(() => URL.revokeObjectURL(url), 100);
            } else if (asset.type === 'report') {
                // Trigger Visual PDF Download
                setReportToDownload(asset.metadata as TrendReport);
                // Note: downloadingAssetId will be cleared when reportToDownload is cleared
            } else {
                const ext = asset.type === 'video' ? 'mp4' : 'png';
                const filename = asset.title || `${asset.type}_${asset.id}.${ext}`;
                await downloadFile(asset.url, filename);
            }
        } finally {
            // For non-report types, clear immediately. For reports, it will be cleared when PDF generation completes
            if (asset.type !== 'report') {
                setTimeout(() => setDownloadingAssetId(null), 500);
            }
        }
    };

    const handleDownloadAllVisible = async (forceSeparate: boolean = false) => {
        if (filteredAssets.length === 0) return;

        const reportsToQueue: TrendReport[] = [];
        const itemsToZip: { url: string, filename: string }[] = [];

        for (const asset of filteredAssets) {
            if (asset.type === 'report') {
                reportsToQueue.push(asset.metadata as TrendReport);
                continue;
            }

            if (asset.type === 'metadata') {
                itemsToZip.push(getDownloadPayload(asset));
                continue;
            }

            const ext = asset.type === 'video' ? 'mp4' : 'png';
            const filename = asset.title ? `${asset.title}.${ext}` : `${asset.type}_${asset.id}.${ext}`;
            itemsToZip.push({ url: asset.url, filename });
        }

        const totalItems = reportsToQueue.length + itemsToZip.length;

        if (forceSeparate) {
            // SEPARATE DOWNLOADS - Always download individually
            if (itemsToZip.length > 0) {
                downloadItems(itemsToZip, `library_export_${Date.now()}`, true); // Pass true to force separate
            }
            if (reportsToQueue.length > 0) {
                setDownloadQueue(prev => [...prev, ...reportsToQueue]);
            }
        } else {
            // ZIP MODE - Always create a zip
            setIsZipMode(true);
            setZipItems(itemsToZip); // Initialize with static items
            setDownloadQueue(reportsToQueue); // Start queuing reports
        }
    };

    const handleDownloadSelectedZip = async () => {
        if (selected.size === 0) return;

        const selectedAssets = assets.filter(a => selected.has(a.id));
        const reportsToQueue: TrendReport[] = [];
        const itemsToZip: { url: string, filename: string }[] = [];

        for (const asset of selectedAssets) {
            if (asset.type === 'report') {
                reportsToQueue.push(asset.metadata as TrendReport);
                continue;
            }

            if (asset.type === 'metadata') {
                itemsToZip.push(getDownloadPayload(asset));
                continue;
            }

            const ext = asset.type === 'video' ? 'mp4' : 'png';
            const filename = asset.title ? `${asset.title}.${ext}` : `${asset.type}_${asset.id}.${ext}`;
            itemsToZip.push({ url: asset.url, filename });
        }

        // UNIFIED ZIP MODE (Forced)
        setIsZipMode(true);
        setZipItems(itemsToZip);
        setDownloadQueue(reportsToQueue);
    };

    const handleDownloadSelectedIndividual = async () => {
        if (selected.size === 0) return;

        const selectedAssets = assets.filter(a => selected.has(a.id));
        const reportsToQueue: TrendReport[] = [];
        const itemsToDownload: { url: string, filename: string }[] = [];

        for (const asset of selectedAssets) {
            if (asset.type === 'report') {
                reportsToQueue.push(asset.metadata as TrendReport);
                continue;
            }

            if (asset.type === 'metadata') {
                itemsToDownload.push(getDownloadPayload(asset));
                continue;
            }

            const ext = asset.type === 'video' ? 'mp4' : 'png';
            const filename = asset.title ? `${asset.title}.${ext}` : `${asset.type}_${asset.id}.${ext}`;
            itemsToDownload.push({ url: asset.url, filename });
        }

        // SEPARATE DOWNLOADS
        if (itemsToDownload.length > 0) {
            // downloadItems utility usually handles lists, but we want to ensure individual downloads if browser allows.
            // Assuming downloadItems takes care of it, or we loop here.
            // To ensure "one by one", let's use the utility but with zip disabled context if relevant
            // Actually existing logic used downloadItems() for non-zip too.
            downloadItems(itemsToDownload, `selected_assets_${Date.now()}`);
        }

        if (reportsToQueue.length > 0) {
            setDownloadQueue(prev => [...prev, ...reportsToQueue]);
        }
    };

    const filteredAssets = assets.filter(a => {
        if (view === AppView.TRASH) {
            if (!a.isDeleted) return false;
        } else if (view === AppView.FAVORITES) {
            // Hide deleted favorites so "delete" action works visually
            if (a.isDeleted) return false;
        } else {
            if (a.isDeleted) return false;
        }

        const q = search.toLowerCase();
        const matchesSearch =
            a.prompt.toLowerCase().includes(q) ||
            a.type.toLowerCase().includes(q) ||
            (a.metadata && JSON.stringify(a.metadata).toLowerCase().includes(q)) ||
            (a.title && a.title.toLowerCase().includes(q));

        const matchesType = typeFilter === 'all' || a.type === typeFilter;
        return matchesSearch && matchesType;
    });

    const [activeFolderId, setActiveFolderId] = useState<string | null>(null);

    // Group assets into folders based on batchId
    // Only applies if view is LIBRARY, METADATA, or REVIEW (favorites/trash/history kept flat usually, or could group history)
    // For now, allow grouping in Library and History
    const groupedAssets = React.useMemo(() => {
        if (view === AppView.TRASH || view === AppView.FAVORITES) {
            // Keep Trash and Favorites distinct items for easier management
            return { folders: [], files: filteredAssets };
        }

        const folders: { id: string, items: Asset[], timestamp: number }[] = [];
        const files: Asset[] = [];
        const batchMap = new Map<string, Asset[]>();

        filteredAssets.forEach(asset => {
            if (asset.batchId) {
                if (!batchMap.has(asset.batchId)) {
                    batchMap.set(asset.batchId, []);
                }
                batchMap.get(asset.batchId)?.push(asset);
            } else {
                files.push(asset);
            }
        });

        // Determine if batch should be a folder (count > 1) or single file
        batchMap.forEach((items, batchId) => {
            if (items.length > 1) {
                // It's a folder
                folders.push({
                    id: batchId,
                    items: items,
                    timestamp: items[0].createdAt
                });
            } else {
                // Single item batch -> straight to files
                files.push(...items);
            }
        });

        // Sort folders by newest first
        folders.sort((a, b) => b.timestamp - a.timestamp);

        // Return structured data
        return { folders, files };
    }, [filteredAssets, view]);

    const currentViewAssets = React.useMemo(() => {
        if (activeFolderId) {
            const folder = groupedAssets.folders.find(f => f.id === activeFolderId);
            return folder ? folder.items : [];
        }
        return groupedAssets.files;
    }, [groupedAssets, activeFolderId]);


    // Handle Folder Actions
    const handleFolderDelete = async (batchId: string, e: React.MouseEvent) => {
        e.stopPropagation();
        if (confirm("Are you sure you want to move this entire folder to Trash?")) {
            const folder = groupedAssets.folders.find(f => f.id === batchId);
            if (!folder) return;

            const promises = folder.items.map(item => dbService.moveToTrash(item.id));
            await Promise.all(promises);
            refreshData();
        }
    }

    const handleFolderDownload = async (batchId: string, e: React.MouseEvent) => {
        e.stopPropagation();
        const folder = groupedAssets.folders.find(f => f.id === batchId);
        if (!folder) return;

        // Fetch settings for CSV format
        const siteSettings = await authService.getSiteSettings();
        const isAdobeEnabled = siteSettings.adobeStockInstruction && siteSettings.adobeStockInstruction.length > 0;

        const itemsToZip: { url: string, filename: string }[] = [];
        const metadataItems: Asset[] = [];

        for (const asset of folder.items) {
            // Special handling for metadata type download in folder
            if (asset.type === 'metadata') {
                // Use reusable helper to respect CSV/TXT/JSON preference
                itemsToZip.push(getDownloadPayload(asset));
                metadataItems.push(asset);
            } else {
                const ext = asset.type === 'video' ? 'mp4' : 'png';
                const filename = asset.title ? `${asset.title}.${ext}` : `${asset.type}_${asset.id}.${ext}`;
                itemsToZip.push({ url: asset.url, filename });
            }
        }

        // Generate Consolidated CSV if metadata exists
        if (metadataItems.length > 0) {
            let csvContent = '';

            if (isAdobeEnabled) {
                // Adobe Stock Format
                csvContent = "Filename,Title,Keywords,Category\n";
                metadataItems.forEach(item => {
                    const m = item.metadata;
                    // Escape CSV fields
                    const filename = `"${(m.filename || item.title || 'asset')}"`; // Original filename
                    const title = `"${(m.title || '').replace(/"/g, '""')}"`;

                    // Robust Keyword Handling: Aggregate common + longTail
                    let allKeywords: string[] = [];
                    if (Array.isArray(m.commonKeywords)) allKeywords.push(...m.commonKeywords);
                    if (Array.isArray(m.longTailKeywords)) allKeywords.push(...m.longTailKeywords);

                    // Fallback to 'keywords' if main arrays are empty (legacy data)
                    if (allKeywords.length === 0) {
                        if (Array.isArray(m.keywords)) {
                            allKeywords.push(...m.keywords);
                        } else if (typeof m.keywords === 'string') {
                            allKeywords.push(...m.keywords.split(',').map((k: string) => k.trim()));
                        }
                    }

                    // Join and clean
                    const keywordsString = allKeywords.join(', ');
                    const keywords = `"${keywordsString.replace(/"/g, '""')}"`;

                    // Category Logic: Convert Name -> ID
                    let catName = m.category || '';
                    if (catName.includes('(')) catName = catName.split('(')[0].trim();

                    // Lookup ID from constant
                    // constants.ts exports ADOBE_STOCK_CATEGORIES as Record<string, number>
                    // We need to match the name.
                    const catId = ADOBE_STOCK_CATEGORIES[catName] || catName; // Fallback to name if not found
                    const category = `"${catId}"`;

                    csvContent += `${filename},${title},${keywords},${category}\n`;
                });
            } else {
                // Standard Format
                csvContent = "Filename,Title,Description,Keywords\n";
                metadataItems.forEach(item => {
                    const m = item.metadata;
                    // Escape CSV fields
                    const filename = `"${(m.filename || item.title || 'asset')}"`;
                    const title = `"${(m.title || '').replace(/"/g, '""')}"`;
                    const desc = `"${(m.description || '').replace(/"/g, '""')}"`;

                    // Robust Keyword Handling: Aggregate common + longTail
                    let allKeywords: string[] = [];
                    if (Array.isArray(m.commonKeywords)) allKeywords.push(...m.commonKeywords);
                    if (Array.isArray(m.longTailKeywords)) allKeywords.push(...m.longTailKeywords);

                    // Fallback to 'keywords' if main arrays are empty (legacy data)
                    if (allKeywords.length === 0) {
                        if (Array.isArray(m.keywords)) {
                            allKeywords.push(...m.keywords);
                        } else if (typeof m.keywords === 'string') {
                            allKeywords.push(...m.keywords.split(',').map((k: string) => k.trim()));
                        }
                    }

                    // Join and clean
                    const keywordsString = allKeywords.join(', ');
                    const keywords = `"${keywordsString.replace(/"/g, '""')}"`;

                    csvContent += `${filename},${title},${desc},${keywords}\n`;
                });
            }

            const csvBlob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            itemsToZip.push({
                url: URL.createObjectURL(csvBlob),
                filename: 'Batch-Summary.csv'
            });
        }

        downloadItems(itemsToZip, `batch_${batchId}`);
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in">
            <div className="flex flex-col md:flex-row md:items-center justify-between gap-4 mb-6 relative z-20">
                <div className="flex items-center gap-3">
                    {activeFolderId && (
                        <button
                            onClick={() => setActiveFolderId(null)}
                            className="p-2 -ml-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-full transition text-slate-500"
                        >
                            <Undo2 size={20} />
                        </button>
                    )}
                    <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk'] flex items-center gap-3">
                        {view === AppView.LIBRARY && <Folder className="text-blue-500" />}
                        {view === AppView.FAVORITES && <Heart className="text-red-500" />}
                        {view === AppView.TRASH && <Trash2 className="text-slate-500" />}
                        {view === AppView.HISTORY && <History className="text-purple-500" />}

                        {activeFolderId ? 'Folder View' : (view === AppView.LIBRARY ? 'Asset Library' : view === AppView.FAVORITES ? 'Favorites' : view === AppView.TRASH ? 'Trash' : 'History')}

                        <span className="text-sm font-normal text-slate-500 ml-2">
                            ({activeFolderId ? currentViewAssets.length : (groupedAssets.folders.length + groupedAssets.files.length)})
                        </span>
                    </h1>
                </div>

                <div className="flex gap-3 items-center">
                    {/* Folder View Download All Button */}
                    {activeFolderId && (
                        <button
                            onClick={(e) => handleFolderDownload(activeFolderId, e)}
                            className="px-4 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 rounded-xl text-xs font-bold flex items-center gap-2 transition uppercase"
                            title="Download All In Folder"
                        >
                            <Download size={14} /> DOWNLOAD ALL
                        </button>
                    )}

                    {/* Global Delete All Button for Library/Favorites */}
                    {(view === AppView.LIBRARY || view === AppView.FAVORITES) && filteredAssets.length > 0 && !activeFolderId && (
                        <>
                            <div className="flex items-center gap-2">
                                <button
                                    onClick={() => handleDownloadAllVisible(true)}
                                    disabled={isZipMode || downloadQueue.length > 0}
                                    className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-slate-700 text-slate-400 hover:bg-slate-900 hover:text-white font-bold rounded-xl transition text-[10px] uppercase"
                                    title="Download all assets as separate files"
                                >
                                    {isZipMode && downloadQueue.length > 0 ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                    Separate All
                                </button>
                                <button
                                    onClick={() => handleDownloadAllVisible(false)}
                                    disabled={isZipMode || downloadQueue.length > 0}
                                    className="flex items-center gap-2 px-3 py-2 bg-slate-950 border border-amber-500 text-amber-500 hover:bg-amber-500/10 hover:shadow-[0_0_15px_rgba(245,158,11,0.3)] hover:text-amber-400 font-bold rounded-xl transition text-[10px] uppercase"
                                    title="Download all assets in a single ZIP archive"
                                >
                                    {isZipMode && downloadQueue.length === 0 ? <Loader2 size={14} className="animate-spin" /> : <FileDown size={14} />}
                                    Zip All
                                </button>
                            </div>
                            <DoubleConfirmDeleteButton
                                onDelete={handleDeleteAllVisible}
                                className="px-3 py-2 bg-slate-950 border border-red-500 text-red-500 hover:bg-red-500/10 hover:shadow-[0_0_15px_rgba(239,68,68,0.3)] hover:text-red-400 rounded-xl text-[10px] font-bold flex items-center gap-2 transition uppercase"
                                title={`Move All ${view === AppView.FAVORITES ? 'Favorites' : 'Assets'} to Trash`}
                                label="TRASH ALL"
                                isLoading={isDeleting}
                            />
                        </>
                    )}

                    {view === AppView.TRASH && (
                        <div className="relative" ref={trashSettingsRef}>
                            <div
                                onClick={() => setIsTrashSettingsOpen(!isTrashSettingsOpen)}
                                className={`flex items-center gap-2 px-3 py-2 border rounded-xl text-xs font-bold cursor-pointer transition select-none
                                    ${trashConfig.enabled
                                        ? 'bg-amber-50 dark:bg-amber-900/20 border-amber-200 dark:border-amber-500/30 text-amber-700 dark:text-amber-400'
                                        : 'bg-slate-100 dark:bg-slate-900 border-slate-200 dark:border-slate-800 text-slate-600 dark:text-slate-400'}
                                `}
                            >
                                <Clock size={14} className={trashConfig.enabled ? "text-amber-500" : "text-slate-400"} />
                                <span>{trashConfig.enabled ? `Auto-delete: ${trashTimer}` : 'Auto-delete: Off'}</span>
                                <Settings size={12} className="ml-1 opacity-70" />
                            </div>

                            {/* Trash Settings Dropdown */}
                            {isTrashSettingsOpen && (
                                <div
                                    onMouseDown={(e) => e.stopPropagation()} // Prevent closing when clicking inside
                                    className="absolute right-0 top-full mt-2 w-64 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl shadow-xl z-50 overflow-hidden animate-fade-in-up p-4"
                                >
                                    <div className="flex justify-between items-center mb-4">
                                        <span className="text-xs font-bold text-slate-900 dark:text-white uppercase tracking-wider">Auto-Delete</span>
                                        <button
                                            onClick={() => handleTrashConfigChange({ enabled: !trashConfig.enabled })}
                                            className={`w-10 h-5 rounded-full relative transition-colors cursor-pointer ${trashConfig.enabled ? 'bg-green-500' : 'bg-slate-300 dark:bg-slate-700'}`}
                                        >
                                            <div className={`absolute top-1 left-1 w-3 h-3 bg-white rounded-full transition-transform ${trashConfig.enabled ? 'translate-x-5' : 'translate-x-0'}`} />
                                        </button>
                                    </div>

                                    <div className={`space-y-3 transition-opacity ${!trashConfig.enabled ? 'opacity-50' : ''}`}>
                                        <label className="text-xs font-bold text-slate-500">Frequency</label>
                                        <div className="grid grid-cols-2 gap-2">
                                            {['minute', 'hourly', 'daily', 'weekly', 'monthly'].map(freq => (
                                                <button
                                                    key={freq}
                                                    onClick={() => handleTrashConfigChange({ frequency: freq as any })}
                                                    className={`px-2 py-2 text-xs font-bold rounded-lg border capitalize transition-all cursor-pointer
                                                        ${trashConfig.frequency === freq
                                                            ? 'bg-blue-50 dark:bg-blue-900/30 border-blue-500 text-blue-600 dark:text-blue-400'
                                                            : 'bg-slate-50 dark:bg-slate-800 border-slate-200 dark:border-slate-700 text-slate-600 dark:text-slate-400 hover:border-slate-300'}
                                                        ${freq === 'monthly' ? 'col-span-2' : ''}
                                                    `}
                                                >
                                                    {freq === 'minute' ? '1 Minute' : freq}
                                                </button>
                                            ))}
                                        </div>
                                        <p className="text-[10px] text-slate-400 mt-2 leading-tight">
                                            Items in trash will be permanently removed {trashConfig.frequency === 'minute' ? 'every minute' : trashConfig.frequency}.
                                        </p>
                                    </div>
                                </div>
                            )}
                        </div>
                    )}

                    {
                        view === AppView.TRASH && assets.length > 0 && (
                            <DoubleConfirmDeleteButton
                                onDelete={handleEmptyTrash}
                                className="px-4 py-2 bg-slate-950 border border-red-500 text-red-500 hover:bg-red-500/10 hover:shadow-[0_0_15px_rgba(239,68,68,0.3)] hover:text-red-400 rounded-xl text-xs font-bold flex items-center gap-2 transition uppercase"
                                title="Empty Trash"
                            >
                                <Trash2 size={14} /> EMPTY TRASH
                            </DoubleConfirmDeleteButton>
                        )
                    }
                    {
                        selected.size > 0 && (
                            <div className="flex gap-2 bg-white dark:bg-slate-900 p-1 rounded-xl border border-slate-200 dark:border-slate-800 animate-fade-in shadow-sm dark:shadow-none">
                                {view === AppView.TRASH ? (
                                    <>
                                        <button onClick={() => handleBulkAction('restore')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-green-500" title="Restore"><Undo2 size={18} /></button>
                                        <DoubleConfirmDeleteButton
                                            onDelete={() => handleBulkAction('permanentDelete')}
                                            className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-red-500"
                                            title="Delete Forever"
                                            iconSize={18}
                                        >
                                            <AlertCircle size={18} />
                                        </DoubleConfirmDeleteButton>
                                    </>
                                ) : (
                                    <>
                                        {/* Blue Zip Download Button */}
                                        <div className="relative group">
                                            <button
                                                onClick={handleDownloadSelectedZip}
                                                disabled={isZipMode}
                                                className={`p-2 rounded-lg text-white transition flex items-center justify-center gap-2 ${isZipMode ? 'bg-blue-400 cursor-wait' : 'bg-blue-500 hover:bg-blue-600 shadow-md hover:shadow-lg'}`}
                                            >
                                                {isZipMode ? <Loader2 size={18} className="animate-spin" /> : <Download size={18} />}
                                            </button>
                                            <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded pointer-events-none opacity-0 group-hover:opacity-100 transition whitespace-nowrap">
                                                Download as ZIP
                                            </div>
                                        </div>

                                        {/* Individual Download Button */}
                                        <div className="relative group">
                                            <button
                                                onClick={handleDownloadSelectedIndividual}
                                                disabled={isZipMode}
                                                className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-slate-600 dark:text-slate-400 hover:text-blue-500 transition"
                                            >
                                                <FileDown size={18} />
                                            </button>
                                            <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded pointer-events-none opacity-0 group-hover:opacity-100 transition whitespace-nowrap">
                                                Download Separately
                                            </div>
                                        </div>

                                        {/* Separator */}
                                        <div className="w-px h-6 bg-slate-200 dark:bg-slate-800 mx-1"></div>

                                        {/* Favorite */}
                                        <div className="relative group">
                                            <button onClick={() => handleBulkAction('favorite')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-red-500 transition">
                                                <Heart size={18} />
                                            </button>
                                            <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded pointer-events-none opacity-0 group-hover:opacity-100 transition whitespace-nowrap">
                                                Favorite
                                            </div>
                                        </div>

                                        {/* Trash */}
                                        <div className="relative group">
                                            <button onClick={() => handleBulkAction('delete')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-slate-500 dark:text-slate-400 hover:text-red-500 transition">
                                                <Trash2 size={18} />
                                            </button>
                                            <div className="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded pointer-events-none opacity-0 group-hover:opacity-100 transition whitespace-nowrap">
                                                Move to Trash
                                            </div>
                                        </div>
                                    </>
                                )}
                                <span className="px-2 flex items-center text-xs font-bold text-slate-500 border-l border-slate-200 dark:border-slate-800">{selected.size} selected</span>
                            </div>
                        )
                    }
                </div >
            </div >

            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-4 flex flex-col xl:flex-row justify-between gap-4 items-center mb-6 shadow-sm dark:shadow-none">
                <div className="relative w-full xl:w-[400px]">
                    <Search className="absolute left-3 top-2.5 text-slate-400 dark:text-slate-500 w-4 h-4" />
                    <input
                        type="text"
                        placeholder="Search assets, keywords..."
                        value={search} onChange={e => setSearch(e.target.value)}
                        className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-2 pl-10 pr-4 text-sm text-slate-900 dark:text-slate-200 focus:border-blue-500 outline-none transition-colors"
                    />
                </div>
                {!activeFolderId && (
                    <div className="flex flex-wrap gap-2 justify-center">
                        <div className="flex flex-wrap bg-slate-50 dark:bg-slate-900 rounded-xl border border-slate-200 dark:border-slate-800 p-1">
                            <button onClick={() => setTypeFilter('all')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'all' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>All</button>
                            <button onClick={() => setTypeFilter('image')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'image' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>Images</button>
                            <button onClick={() => setTypeFilter('video')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'video' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>Videos</button>
                            <button onClick={() => setTypeFilter('metadata')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'metadata' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>Metadata</button>
                            <button onClick={() => setTypeFilter('report')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'report' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>Reports</button>
                            <button onClick={() => setTypeFilter('review')} className={`px-3 py-1.5 rounded-lg text-xs font-bold transition ${typeFilter === 'review' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}>Reviews</button>
                        </div>
                        <div className="w-px h-8 bg-slate-200 dark:bg-slate-800 mx-1 hidden md:block"></div>
                        <div className="flex items-center gap-2">
                            <button onClick={() => setLayout('grid')} className={`p-2 rounded-xl border transition ${layout === 'grid' ? 'bg-white dark:bg-slate-800 border-slate-300 dark:border-slate-700 text-slate-900 dark:text-white' : 'border-transparent text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}><Grid size={18} /></button>
                            <button onClick={() => setLayout('list')} className={`p-2 rounded-xl border transition ${layout === 'list' ? 'bg-white dark:bg-slate-800 border-slate-300 dark:border-slate-700 text-slate-900 dark:text-white' : 'border-transparent text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}><List size={18} /></button>
                        </div>
                    </div>
                )}
            </div>

            <div className="flex-1 overflow-y-auto custom-scrollbar pb-20">
                {currentViewAssets.length === 0 && !activeFolderId && groupedAssets.folders.length === 0 ? (
                    <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                        <Folder size={64} className="mb-4 opacity-20" />
                        <p className="text-lg font-medium">No assets found</p>
                        <p className="text-sm">Generate content to populate your library.</p>
                    </div>
                ) : (
                    <div className={`grid gap-6 ${layout === 'grid' ? 'grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5' : 'grid-cols-1'}`}>
                        {/* RENDER FOLDERS (Only at root level) */}
                        {!activeFolderId && groupedAssets.folders.map(folder => (
                            <div
                                key={folder.id}
                                onClick={() => setActiveFolderId(folder.id)}
                                className={`
                                    group relative bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden cursor-pointer transition-all duration-200 shadow-sm dark:shadow-none hover:border-blue-500 dark:hover:border-blue-500
                                    ${layout === 'list' ? 'flex h-24 items-center px-4' : 'aspect-square flex flex-col items-center justify-center p-6'}
                                `}
                            >
                                <div className="absolute top-2 right-2 flex gap-1 opacity-0 group-hover:opacity-100 transition">
                                    <button
                                        onClick={(e) => handleFolderDownload(folder.id, e)}
                                        className="p-1.5 bg-white dark:bg-slate-800 rounded-lg text-slate-400 hover:text-blue-500 shadow-sm"
                                        title="Download All in Folder"
                                    >
                                        <Download size={14} />
                                    </button>
                                    <button
                                        onClick={(e) => handleFolderDelete(folder.id, e)}
                                        className="p-1.5 bg-white dark:bg-slate-800 rounded-lg text-slate-400 hover:text-red-500 shadow-sm"
                                        title="Delete Folder"
                                    >
                                        <Trash2 size={14} />
                                    </button>
                                </div>

                                <Folder size={layout === 'list' ? 40 : 64} className="text-amber-500 mb-2 fill-amber-500/10" />
                                <div className={`${layout === 'list' ? 'ml-4 flex-1' : 'text-center'}`}>
                                    <h4 className="font-bold text-slate-900 dark:text-white text-sm">Batch Folder</h4>
                                    <p className="text-xs text-slate-500 dark:text-slate-400">{folder.items.length} items</p>
                                    <p className="text-[10px] text-slate-400 mt-1">{new Date(folder.timestamp).toLocaleDateString()}</p>
                                </div>
                                <div className="absolute top-2 left-2 px-2 py-0.5 bg-amber-100 dark:bg-amber-900/30 text-amber-600 dark:text-amber-400 text-[10px] font-bold rounded uppercase">
                                    {folder.items[0].type}
                                </div>
                            </div>
                        ))}

                        {/* RENDER ASSETS (Files + Orphans) */}
                        {currentViewAssets.map(asset => (
                            <div
                                key={asset.id}
                                onClick={() => handleToggleSelect(asset.id)}
                                className={`
                                    group relative bg-white dark:bg-slate-900 border rounded-2xl overflow-hidden cursor-pointer transition-all duration-200 shadow-sm dark:shadow-none
                                    ${selected.has(asset.id) ? 'border-blue-500 ring-1 ring-blue-500 shadow-lg shadow-blue-500/20' : 'border-slate-200 dark:border-slate-800 hover:border-slate-300 dark:hover:border-slate-600'}
                                    ${layout === 'list' ? 'flex h-32' : 'aspect-square'}
                                `}
                            >
                                {/* Asset Rendering */}
                                <div className={`relative overflow-hidden ${layout === 'list' ? 'w-48 shrink-0' : 'w-full h-full'}`}>
                                    {asset.type === 'video' && asset.url && (
                                        <video src={asset.url} className="w-full h-full object-cover bg-black" />
                                    )}

                                    {asset.type === 'image' && asset.url && (
                                        <img src={asset.url} className="w-full h-full object-cover bg-slate-100 dark:bg-slate-800" loading="lazy" />
                                    )}

                                    {asset.type === 'metadata' && (
                                        <div className="w-full h-full relative flex items-center justify-center bg-slate-100 dark:bg-slate-800">
                                            {asset.url ? (
                                                <img src={asset.url} className="w-full h-full object-cover opacity-50 grayscale transition group-hover:grayscale-0 group-hover:opacity-100" />
                                            ) : (
                                                <FileText size={40} className="text-slate-400 opacity-50" />
                                            )}
                                            <div className="absolute inset-0 bg-amber-500/10 mix-blend-overlay"></div>
                                        </div>
                                    )}

                                    {asset.type === 'report' && (
                                        <div className="w-full h-full bg-gradient-to-br from-slate-800 to-slate-950 flex flex-col items-center justify-center p-4 text-center border-b border-slate-800">
                                            <TrendingUp className="text-emerald-500 mb-2" size={32} />
                                            <span className="text-[10px] font-bold text-slate-500 uppercase tracking-wider">Trend Report</span>
                                        </div>
                                    )}

                                    {asset.type === 'review' && asset.url && (
                                        <img src={asset.url} className="w-full h-full object-cover bg-slate-100 dark:bg-slate-800" loading="lazy" />
                                    )}

                                    {/* Overlay Type Icon */}
                                    <div className="absolute top-2 left-2 px-2 py-1 bg-black/60 backdrop-blur rounded-md text-[10px] font-bold text-white uppercase flex items-center gap-1.5 shadow-lg">
                                        {asset.type === 'video' && <Video size={10} />}
                                        {asset.type === 'image' && <ImageIcon size={10} />}
                                        {asset.type === 'metadata' && <ScanEye size={10} className="text-amber-400" />}
                                        {asset.type === 'report' && <FileText size={10} className="text-emerald-400" />}
                                        {asset.type === 'review' && <CheckCircle2 size={10} className="text-blue-400" />}
                                        {asset.type}
                                    </div>

                                    {/* Selection Check */}
                                    {selected.has(asset.id) && (
                                        <div className="absolute top-2 right-2 w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center shadow-lg animate-bounce-slow">
                                            <CheckCircle2 size={14} className="text-white" />
                                        </div>
                                    )}
                                </div>

                                {/* Details Panel */}
                                <div className={`p-4 flex flex-col ${layout === 'list' ? 'flex-1 justify-center' : 'absolute inset-0 bg-gradient-to-t from-black/90 dark:from-slate-950 via-black/40 to-transparent opacity-0 group-hover:opacity-100 transition-opacity justify-end'}`}>
                                    {asset.title && (
                                        <p className={`text-xs font-bold text-amber-400 mb-1 truncate ${layout === 'list' ? 'text-slate-500' : ''}`}>
                                            {asset.title}
                                        </p>
                                    )}
                                    <p className={`text-sm font-bold text-white line-clamp-2 mb-1 ${layout === 'list' ? 'text-slate-900 dark:text-white' : 'text-shadow-lg'}`}>
                                        {asset.prompt || (asset.type === 'report' ? 'Market Analysis' : 'Untitled Asset')}
                                    </p>
                                    <p className={`text-xs ${layout === 'list' ? 'text-slate-500' : 'text-slate-300 font-medium'}`}>
                                        {new Date(asset.createdAt).toLocaleDateString()}
                                    </p>

                                    {/* Verdict Display */}
                                    {asset.type === 'review' && asset.metadata?.verdict && (
                                        <div className={`mt-2 font-bold text-[10px] uppercase tracking-wider ${asset.metadata.verdict === 'ACCEPTED' ? 'text-green-400' : 'text-red-500'}`}>
                                            {asset.metadata.verdict}
                                        </div>
                                    )}

                                    {/* Quick Actions */}
                                    <div className={`flex gap-2 mt-3 ${layout === 'list' ? '' : 'justify-end'}`}>
                                        <button
                                            onClick={(e) => handlePreview(e, asset)}
                                            className={`p-1.5 rounded-lg backdrop-blur transition ${layout === 'list' ? 'bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 hover:bg-slate-200' : 'bg-white/10 text-white hover:bg-white/20'}`}
                                            title="Open Preview"
                                        >
                                            <Eye size={14} />
                                        </button>
                                        <button
                                            onClick={(e) => handleDownload(e, asset)}
                                            disabled={downloadingAssetId === asset.id}
                                            className={`p-1.5 rounded-lg backdrop-blur transition ${layout === 'list' ? 'bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 hover:bg-slate-200' : 'bg-white/10 text-white hover:bg-white/20'} ${downloadingAssetId === asset.id ? 'opacity-70 cursor-wait' : ''}`}
                                            title="Download"
                                        >
                                            {downloadingAssetId === asset.id ? <Loader2 size={14} className="animate-spin" /> : <Download size={14} />}
                                        </button>
                                        <button
                                            onClick={async (e) => { e.stopPropagation(); await dbService.toggleFavorite(asset.id, asset.isFavorite); refreshData(); }}
                                            className={`p-1.5 rounded-lg backdrop-blur transition ${layout === 'list' ? 'hover:bg-slate-100 dark:hover:bg-slate-800' : 'bg-white/10 hover:bg-white/20'} ${asset.isFavorite ? 'text-red-500' : layout === 'list' ? 'text-slate-400' : 'text-white'}`}
                                        >
                                            <Heart size={14} className={asset.isFavorite ? "fill-current" : ""} />
                                        </button>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>
            {/* Preview Modal */}
            {previewAsset && (
                <div className="fixed inset-0 z-[100] flex items-center justify-center bg-black/90 backdrop-blur-sm p-4 animate-fade-in" onClick={() => setPreviewAsset(null)}>
                    <button className="absolute top-4 right-4 p-2 bg-white/10 hover:bg-white/20 rounded-full text-white transition z-[101]" onClick={() => setPreviewAsset(null)}>
                        <X size={24} />
                    </button>
                    <div className="max-w-6xl w-full max-h-[90vh] overflow-y-auto rounded-3xl shadow-2xl animate-scale-up" onClick={e => e.stopPropagation()}>
                        {/* Type specific render */}
                        {(previewAsset.type === 'image' || previewAsset.type === 'review') && (
                            <img src={previewAsset.url} className="w-full h-auto rounded-3xl" />
                        )}
                        {previewAsset.type === 'video' && (
                            <div className="aspect-video bg-black rounded-3xl overflow-hidden">
                                <video src={previewAsset.url} controls autoPlay className="w-full h-full" />
                            </div>
                        )}
                        {previewAsset.type === 'report' && (
                            <div className="bg-slate-950 p-4 md:p-8 rounded-3xl">
                                <TrendReportView report={previewAsset.metadata} sitesList={sitesList} />
                            </div>
                        )}
                        {previewAsset.type === 'metadata' && (
                            <div className="bg-slate-900 border border-slate-800 p-8 rounded-3xl text-left overflow-x-auto">
                                <h3 className="text-xl font-bold text-amber-500 mb-4 flex items-center gap-2">
                                    {previewAsset.generationSettings?.exportFormat === 'csv' ? <FileText /> : previewAsset.generationSettings?.exportFormat === 'txt' ? <FileText /> : <FileJson />}
                                    Metadata Preview ({previewAsset.generationSettings?.exportFormat?.toUpperCase() || 'JSON'})
                                </h3>
                                <pre className="text-xs font-mono text-slate-300 whitespace-pre-wrap">
                                    {getMetadataContent(previewAsset).content}
                                </pre>
                            </div>
                        )}
                    </div>
                </div>
            )}

            {/* Hidden Report Renderer for PDF Generation */}
            <div className="dark" style={{ position: 'absolute', left: '-9999px', top: 0, width: '1200px' }}>
                {reportToDownload && (
                    <div ref={reportRef} className="bg-slate-900 text-slate-200 p-8">
                        <TrendReportView report={reportToDownload} sitesList={sitesList} />
                    </div>
                )}
            </div>
        </div >
    );
};
export default Library;