
import React, { useState, useEffect, useRef } from 'react';
import {
    Sparkles, ArrowRight, Video, Image as ImageIcon,
    ScanEye, PencilRuler, Globe, BrainCircuit, CheckCircle2,
    Play, LayoutGrid, ShieldCheck, Zap, Cpu, Layers, Users, Heart, Lightbulb,
    Sun, Moon, ArrowUp, Check, ChevronLeft, ChevronRight, ThumbsUp, ThumbsDown, AlertTriangle, Clock, Lock
} from 'lucide-react';
import { adminService } from '../services/adminService';
import { dbService } from '../services/dbService';

interface LandingPageProps {
    onStart: () => void;
    onSignIn: () => void;
}

const LANGUAGES = [
    { code: 'en', label: 'English', flag: '🇺🇸' },
    { code: 'es', label: 'Español', flag: '🇪🇸' },
    { code: 'fr', label: 'Français', flag: '🇫🇷' },
    { code: 'de', label: 'Deutsch', flag: '🇩🇪' },
    { code: 'it', label: 'Italiano', flag: '🇮🇹' },
    { code: 'pt', label: 'Português', flag: '🇵🇹' },
    { code: 'zh-CN', label: '中文 (简体)', flag: '🇨🇳' },
    { code: 'ja', label: '日本語', flag: '🇯🇵' },
    { code: 'ko', label: '한국어', flag: '🇰🇷' },
    { code: 'ru', label: 'Русский', flag: '🇷🇺' },
    { code: 'ar', label: 'العربية', flag: '🇸🇦' },
];

const Counter = ({ end, label, suffix = '+' }: { end: number, label: string, suffix?: string }) => {
    const [count, setCount] = useState(0);

    useEffect(() => {
        let startTimestamp: number | null = null;
        const duration = 2500; // 2.5 seconds animation

        const step = (timestamp: number) => {
            if (!startTimestamp) startTimestamp = timestamp;
            const progress = Math.min((timestamp - startTimestamp) / duration, 1);
            // Cubic ease-out function for smooth deceleration
            const easeProgress = 1 - Math.pow(1 - progress, 3);

            setCount(Math.floor(easeProgress * end));

            if (progress < 1) {
                window.requestAnimationFrame(step);
            }
        };

        window.requestAnimationFrame(step);
    }, [end]);

    const format = (n: number) => {
        if (n >= 1000000) return (n / 1000000).toFixed(1) + 'M';
        if (n >= 1000) return (n / 1000).toFixed(0) + 'k';
        return n.toString();
    };

    return (
        <div className="space-y-1">
            <h4 className="text-3xl font-bold text-slate-900 dark:text-white tabular-nums">
                {format(count)}{suffix}
            </h4>
            <p className="text-sm text-slate-800 dark:text-slate-50 uppercase tracking-wider font-bold">{label}</p>
        </div>
    );
};

const LandingPage: React.FC<LandingPageProps> = ({ onStart, onSignIn }) => {
    const [isVideoOpen, setIsVideoOpen] = useState(false);
    const [theme, setTheme] = useState<'dark' | 'light'>('dark');
    const [showScrollTop, setShowScrollTop] = useState(false);
    const [content, setContent] = useState<any>({});
    const [showLangMenu, setShowLangMenu] = useState(false);
    const [currentLang, setCurrentLang] = useState('en');

    // Maintenance State
    const [maintenance, setMaintenance] = useState<{ enabled: boolean, message: string, preCode: string, endTime?: number }>({ enabled: false, message: '', preCode: '', endTime: undefined });
    const [showPreCodeModal, setShowPreCodeModal] = useState(false);
    const [preCodeInput, setPreCodeInput] = useState('');
    const [timeLeft, setTimeLeft] = useState<{ d: number, h: number, m: number, s: number } | null>(null);
    const [liveStats, setLiveStats] = useState<{ users: number, assets: number } | null>(null);

    // Slider State
    const [slideIndex, setSlideIndex] = useState(0);
    const [itemsPerView, setItemsPerView] = useState(3);
    const [impactMediaType, setImpactMediaType] = useState<'image' | 'video'>('image');
    const sliderRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        // Initialize theme
        const savedTheme = localStorage.getItem('sf_theme') || 'dark';
        setTheme(savedTheme as 'dark' | 'light');
        if (savedTheme === 'dark') document.documentElement.classList.add('dark');
        else document.documentElement.classList.remove('dark');

        // Scroll listener
        const handleScroll = () => {
            if (window.scrollY > 300) {
                setShowScrollTop(true);
            } else {
                setShowScrollTop(false);
            }
        };

        // Responsive slider items
        const handleResize = () => {
            if (window.innerWidth < 768) setItemsPerView(1);
            else if (window.innerWidth < 1024) setItemsPerView(2);
            else setItemsPerView(3);
        }
        handleResize();
        window.addEventListener('resize', handleResize);
        window.addEventListener('scroll', handleScroll);

        // Load content & settings
        adminService.getLandingContent().then(setContent);
        adminService.getSiteSettings().then(settings => {
            if (settings.maintenanceMode) {
                setMaintenance({
                    enabled: true,
                    message: settings.maintenanceMessage,
                    preCode: settings.maintenancePreCode || 'admin123',
                    endTime: settings.maintenanceEndTime
                });
            }
        });

        // Subscribe to live stats
        const unsubscribeStats = dbService.subscribeToGlobalStats((stats) => {
            setLiveStats(stats);
        });

        // Check Google Translate init state
        const checkTranslate = setInterval(() => {
            const select = document.querySelector('.goog-te-combo') as HTMLSelectElement;
            if (select) {
                // If value is set by google, sync state
                if (select.value && select.value !== currentLang) {
                    setCurrentLang(select.value);
                }
            }
        }, 2000);

        return () => {
            window.removeEventListener('scroll', handleScroll);
            window.removeEventListener('resize', handleResize);
            clearInterval(checkTranslate);
        }
    }, [currentLang]);

    // Countdown Timer Logic
    useEffect(() => {
        if (!maintenance.enabled || !maintenance.endTime) return;

        const interval = setInterval(() => {
            const now = Date.now();
            const dist = maintenance.endTime! - now;

            if (dist < 0) {
                setTimeLeft(null);
                clearInterval(interval);
            } else {
                setTimeLeft({
                    d: Math.floor(dist / (1000 * 60 * 60 * 24)),
                    h: Math.floor((dist % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
                    m: Math.floor((dist % (1000 * 60 * 60)) / (1000 * 60)),
                    s: Math.floor((dist % (1000 * 60)) / 1000)
                });
            }
        }, 1000);

        return () => clearInterval(interval);
    }, [maintenance]);

    // Auto-scroll slider logic
    const allSlides = content.globalImpact?.slides || [];
    const filteredSlides = allSlides.filter((s: any) => s.type === impactMediaType);

    useEffect(() => {
        if (filteredSlides.length <= itemsPerView) return;

        const interval = setInterval(() => {
            setSlideIndex((prev) => {
                const maxIndex = filteredSlides.length - itemsPerView;
                // "revert auto when app show last image" - when we hit the end, go back to 0
                if (prev >= maxIndex) return 0;
                return prev + 1;
            });
        }, 3500);
        return () => clearInterval(interval);
    }, [filteredSlides.length, itemsPerView, impactMediaType]);

    const handleMediaTypeChange = (type: 'image' | 'video') => {
        setImpactMediaType(type);
        setSlideIndex(0); // Reset slider to start
    };

    const toggleTheme = () => {
        const newTheme = theme === 'dark' ? 'light' : 'dark';
        setTheme(newTheme);
        localStorage.setItem('sf_theme', newTheme);
        if (newTheme === 'dark') document.documentElement.classList.add('dark');
        else document.documentElement.classList.remove('dark');
    };

    const scrollToSection = (id: string) => {
        const element = document.getElementById(id);
        if (element) {
            element.scrollIntoView({ behavior: 'smooth' });
        }
    };

    const handleLanguageChange = (langCode: string) => {
        setCurrentLang(langCode);
        setShowLangMenu(false);

        const select = document.querySelector('.goog-te-combo') as HTMLSelectElement;
        if (select) {
            select.value = langCode;
            select.dispatchEvent(new Event('change', { bubbles: true }));
            // Force immediate update
            const iframe = document.querySelector('.goog-te-menu-frame') as HTMLIFrameElement;
            if (!iframe) {
                // Fallback for some modes
                select.click();
            }
        }
    };

    // Default content structure
    const general = content.general || {};
    const navigation = content.navigation || {};
    const hero = content.hero || {};
    const features = content.features || {};
    const about = content.about || {};
    const pricing = content.pricing || {};
    const cta = content.cta || {};
    const poweredBy = content.poweredBy || {};
    const globalImpact = content.globalImpact || {};

    // Logic for pricing grid
    const visiblePlans = [
        pricing.starterVisible !== false,
        pricing.proVisible !== false,
        pricing.agencyVisible !== false // Now respects the config which defaults to false
    ].filter(Boolean).length;

    const gridClass = visiblePlans === 1
        ? "grid-cols-1 max-w-md mx-auto"
        : visiblePlans === 2
            ? "grid-cols-1 md:grid-cols-2 max-w-4xl mx-auto justify-items-center" // Center logic for 2 items
            : "grid-cols-1 md:grid-cols-3";

    if (maintenance.enabled) {
        return (
            <div className="min-h-screen flex flex-col items-center justify-center bg-slate-50 dark:bg-slate-950 p-6 text-center relative overflow-hidden">
                {/* Abstract Background */}
                <div className="absolute inset-0 overflow-hidden pointer-events-none">
                    <div className="absolute top-0 left-1/2 -translate-x-1/2 w-[800px] h-[800px] bg-amber-500/5 rounded-full blur-[120px]"></div>
                </div>

                <div className="relative z-10 flex flex-col items-center max-w-2xl w-full">
                    <div className="w-24 h-24 bg-amber-100 dark:bg-amber-500/20 rounded-3xl flex items-center justify-center mb-8 animate-pulse">
                        <AlertTriangle size={48} className="text-amber-600 dark:text-amber-500" />
                    </div>
                    <h1 className="text-4xl font-bold text-slate-900 dark:text-white mb-6 font-['Space_Grotesk']">System Maintenance</h1>
                    <p className="text-slate-600 dark:text-slate-400 text-lg leading-relaxed mb-10 font-medium">
                        {maintenance.message || "We are currently performing scheduled maintenance. Please check back later."}
                    </p>

                    {/* Countdown Timer */}
                    {timeLeft && (
                        <div className="grid grid-cols-4 gap-4 mb-12">
                            {[
                                { label: 'Days', value: timeLeft.d },
                                { label: 'Hours', value: timeLeft.h },
                                { label: 'Minutes', value: timeLeft.m },
                                { label: 'Seconds', value: timeLeft.s }
                            ].map((item, i) => (
                                <div key={i} className="flex flex-col items-center p-4 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl shadow-sm min-w-[80px]">
                                    <span className="text-3xl font-bold text-slate-900 dark:text-white font-mono">
                                        {item.value.toString().padStart(2, '0')}
                                    </span>
                                    <span className="text-[10px] uppercase font-bold text-slate-500 tracking-wider mt-1">{item.label}</span>
                                </div>
                            ))}
                        </div>
                    )}

                    {/* Admin Access Trigger */}
                    <button
                        onClick={() => setShowPreCodeModal(true)}
                        className="mt-4 px-6 py-2 text-slate-500 dark:text-slate-400 text-sm font-bold hover:text-slate-800 dark:hover:text-slate-200 transition flex items-center gap-2 border border-transparent hover:border-slate-200 dark:hover:border-slate-800 rounded-full"
                    >
                        <Lock size={14} /> Admin Access
                    </button>
                </div>

                {/* Pre-Code Modal */}
                {showPreCodeModal && (
                    <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4 animate-fade-in" onClick={() => setShowPreCodeModal(false)}>
                        <div className="bg-white dark:bg-slate-900 p-8 rounded-3xl max-w-sm w-full shadow-2xl border border-slate-200 dark:border-slate-800 transform scale-100 transition-all" onClick={e => e.stopPropagation()}>
                            <div className="flex items-center gap-3 mb-6">
                                <div className="p-3 bg-slate-100 dark:bg-slate-800 rounded-xl">
                                    <Lock size={20} className="text-slate-900 dark:text-white" />
                                </div>
                                <h3 className="text-xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk']">Admin Verification</h3>
                            </div>

                            <p className="text-sm text-slate-500 mb-4">Enter the pre-shared code to bypass maintenance mode.</p>

                            <input
                                type="password"
                                value={preCodeInput}
                                onChange={e => setPreCodeInput(e.target.value)}
                                placeholder="Enter Access Code"
                                className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-4 mb-4 focus:ring-2 focus:ring-amber-500 outline-none text-slate-900 dark:text-white font-medium transition-all"
                                autoFocus
                                onKeyDown={(e) => {
                                    if (e.key === 'Enter') {
                                        if (preCodeInput === maintenance.preCode) onSignIn();
                                        else alert("Invalid Access Code");
                                    }
                                }}
                            />
                            <div className="flex gap-3">
                                <button
                                    onClick={() => setShowPreCodeModal(false)}
                                    className="flex-1 py-3 bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 font-bold rounded-xl hover:bg-slate-200 dark:hover:bg-slate-700 transition"
                                >
                                    Cancel
                                </button>
                                <button
                                    onClick={() => {
                                        if (preCodeInput === maintenance.preCode) onSignIn();
                                        else alert("Invalid Access Code");
                                    }}
                                    className="flex-1 py-3 bg-slate-900 dark:bg-white text-white dark:text-slate-950 font-bold rounded-xl hover:opacity-90 transition shadow-lg"
                                >
                                    Unlock
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-slate-50 dark:bg-slate-950 text-slate-900 dark:text-slate-50 font-sans selection:bg-amber-500/30 overflow-x-hidden transition-colors duration-300">
            {/* Navigation */}
            <nav className="fixed top-0 w-full z-50 border-b border-slate-200 dark:border-white/5 bg-white/80 dark:bg-slate-950/80 backdrop-blur-xl transition-colors duration-300">
                <div className="max-w-7xl mx-auto px-6 h-20 flex items-center justify-between">
                    <div className="flex items-center gap-2">
                        {general.appLogo ? (
                            <img src={general.appLogo} alt="Logo" className="w-10 h-10 rounded-xl shadow-lg shadow-amber-500/20 object-cover" />
                        ) : (
                            <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-amber-400 to-amber-600 flex items-center justify-center shadow-lg shadow-amber-500/20">
                                <Sparkles className="text-white w-6 h-6" />
                            </div>
                        )}
                        <span className="text-xl font-bold font-['Space_Grotesk'] tracking-tight text-slate-900 dark:text-white">{general.appName || "StockForge AI"}</span>
                    </div>
                    <div className="hidden md:flex items-center gap-8 text-sm font-bold text-slate-700 dark:text-slate-400">
                        <button
                            onClick={() => scrollToSection('features')}
                            className="hover:text-slate-900 dark:hover:text-slate-100 transition-colors cursor-pointer bg-transparent border-none focus:outline-none"
                        >
                            {navigation.featuresLabel || "Features"}
                        </button>
                        {pricing.isVisible !== false && visiblePlans > 0 && (
                            <button
                                onClick={() => scrollToSection('pricing')}
                                className="hover:text-slate-900 dark:hover:text-slate-100 transition-colors cursor-pointer bg-transparent border-none focus:outline-none"
                            >
                                {navigation.pricingLabel || "Pricing"}
                            </button>
                        )}
                        <button
                            onClick={() => scrollToSection('about')}
                            className="hover:text-slate-900 dark:hover:text-slate-100 transition-colors cursor-pointer bg-transparent border-none focus:outline-none"
                        >
                            {navigation.aboutLabel || "About"}
                        </button>
                    </div>
                    <div className="flex items-center gap-4">
                        {/* Language Selector */}
                        <div className="relative">
                            {showLangMenu && (
                                <>
                                    <div className="fixed inset-0 z-10" onClick={() => setShowLangMenu(false)}></div>
                                    <div className="absolute top-full right-0 mt-2 w-48 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl shadow-xl overflow-hidden z-50 animate-fade-in">
                                        <div className="max-h-64 overflow-y-auto custom-scrollbar">
                                            {LANGUAGES.map((lang) => (
                                                <button
                                                    key={lang.code}
                                                    onClick={() => handleLanguageChange(lang.code)}
                                                    className="w-full text-left px-4 py-3 text-sm text-slate-700 dark:text-slate-300 hover:bg-slate-100 dark:hover:bg-slate-800 flex items-center gap-3 transition-colors justify-between group"
                                                >
                                                    <div className="flex items-center gap-2">
                                                        <span className="text-lg">{lang.flag}</span>
                                                        <span className="font-medium">{lang.label}</span>
                                                    </div>
                                                    {currentLang === lang.code && <Check size={14} className="text-amber-500" />}
                                                </button>
                                            ))}
                                        </div>
                                    </div>
                                </>
                            )}
                            <button
                                onClick={() => setShowLangMenu(!showLangMenu)}
                                className="p-2 rounded-full text-slate-600 dark:text-slate-400 hover:text-amber-600 dark:hover:text-amber-400 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors relative z-20"
                                title="Choose Language"
                            >
                                <Globe size={20} />
                            </button>
                        </div>

                        <button onClick={toggleTheme} className="p-2 rounded-full text-slate-600 dark:text-slate-400 hover:text-amber-600 dark:hover:text-amber-400 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors" title="Toggle Theme">
                            {theme === 'dark' ? <Sun size={20} /> : <Moon size={20} />}
                        </button>
                        <button onClick={onSignIn} className="text-sm font-bold text-slate-800 hover:text-slate-900 dark:text-slate-300 dark:hover:text-white hidden md:block">Sign In</button>
                        <button
                            onClick={onStart}
                            className="px-5 py-2.5 bg-slate-900 dark:bg-slate-100 text-white dark:text-slate-950 rounded-lg font-bold text-sm hover:bg-slate-800 dark:hover:bg-white hover:scale-105 transition-all"
                        >
                            Get Started
                        </button>
                    </div>
                </div>
            </nav>

            {/* Hero Section */}
            <section className="relative pt-32 pb-20 lg:pt-48 lg:pb-32 px-6 overflow-hidden">
                {/* Background Effects */}
                <div className="absolute top-0 left-1/2 -translate-x-1/2 w-[1000px] h-[600px] bg-indigo-600/10 dark:bg-indigo-600/20 rounded-full blur-[120px] -z-10" />
                <div className="absolute bottom-0 right-0 w-[800px] h-[600px] bg-amber-600/5 dark:bg-amber-600/10 rounded-full blur-[120px] -z-10" />
                <div className="absolute inset-0 bg-[url('https://grainy-gradients.vercel.app/noise.svg')] opacity-5 dark:opacity-20 -z-10" />

                <div className="max-w-7xl mx-auto text-center relative z-10">
                    <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/80 dark:bg-slate-900/50 border border-slate-300 dark:border-slate-800 mb-8 backdrop-blur-sm animate-fade-in-up shadow-sm dark:shadow-none">
                        <span className="flex h-2 w-2 rounded-full bg-green-500 animate-pulse" />
                        <span className="text-xs font-bold text-slate-800 dark:text-slate-300 tracking-wide">{hero.badge || "AI POWERED"}</span>
                    </div>

                    <h1 className="text-5xl lg:text-7xl font-bold mb-6 tracking-tight font-['Space_Grotesk'] leading-tight text-slate-900 dark:text-white">
                        {hero.titlePrefix || "Forge Your"} <span className="text-transparent bg-clip-text bg-gradient-to-r from-amber-500 to-orange-600 dark:from-amber-400 dark:to-orange-500">{hero.titleHighlight || "Creative Future"}</span> <br />
                        {hero.titleSuffix || "with AI Intelligence"}
                    </h1>

                    <p className="text-lg lg:text-xl text-slate-800 dark:text-slate-400 max-w-2xl mx-auto mb-10 leading-relaxed font-medium">
                        {hero.subtitle || "The ultimate microstock creator toolkit."}
                    </p>

                    <div className="flex flex-col sm:flex-row items-center justify-center gap-4">
                        <button
                            onClick={onStart}
                            className="w-full sm:w-auto px-8 py-4 bg-gradient-to-r from-amber-500 to-orange-600 rounded-xl font-bold text-white dark:text-slate-950 text-lg hover:shadow-lg hover:shadow-amber-500/25 hover:-translate-y-1 transition-all flex items-center justify-center gap-2"
                        >
                            {hero.buttonPrimary || "Start Creating"} <ArrowRight className="w-5 h-5" />
                        </button>
                        <button
                            onClick={() => setIsVideoOpen(true)}
                            className="w-full sm:w-auto px-8 py-4 bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-700 rounded-xl font-bold text-slate-800 dark:text-slate-200 text-lg hover:bg-slate-50 dark:hover:bg-slate-800 transition-all flex items-center justify-center gap-2 group shadow-sm dark:shadow-none"
                        >
                            <Play className="w-5 h-5 fill-current group-hover:text-amber-600 dark:group-hover:text-amber-400 transition-colors" /> {hero.buttonSecondary || "Watch Demo"}
                        </button>
                    </div>

                    {/* Stats/Social Proof */}
                    <div className="mt-16 pt-8 border-t border-slate-200 dark:border-white/5 flex flex-wrap justify-center gap-8 lg:gap-16 text-center">
                        <Counter end={liveStats && liveStats.users > 0 ? liveStats.users : (Number(hero.stat1Value) || 13000)} label={hero.stat1Label || "Creators"} />
                        <Counter end={liveStats && liveStats.assets > 0 ? liveStats.assets : (Number(hero.stat2Value) || 1500000)} label={hero.stat2Label || "Assets Generated"} />

                        <div className="space-y-1">
                            <h4 className="text-3xl font-bold text-slate-900 dark:text-white">{hero.stat3Value || "4.9/5"}</h4>
                            <p className="text-sm text-slate-800 dark:text-slate-50 uppercase tracking-wider font-bold">{hero.stat3Label || "Rating"}</p>
                        </div>
                    </div>
                </div>
            </section>

            {/* Features Grid (Bento) */}
            <section id="features" className="py-24 px-6 bg-white dark:bg-slate-950 relative transition-colors duration-300">
                <div className="max-w-7xl mx-auto">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl lg:text-4xl font-bold mb-4 font-['Space_Grotesk'] text-slate-900 dark:text-white">{features.title || "Features"}</h2>
                        <p className="text-slate-700 dark:text-slate-400 max-w-2xl mx-auto font-medium">{features.subtitle}</p>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-6 auto-rows-[200px]">
                        {/* Cards - Updated with sharper rounded-xl and stronger borders for light mode */}
                        <div className="md:col-span-2 lg:col-span-2 row-span-2 p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-purple-500/50 transition-all group relative overflow-hidden shadow-xl dark:shadow-none">
                            <div className="absolute top-0 right-0 p-20 bg-purple-500/10 blur-3xl rounded-full group-hover:bg-purple-500/20 transition-all" />
                            <div className="relative z-10 h-full flex flex-col">
                                <div className="w-12 h-12 rounded-xl bg-purple-500/20 flex items-center justify-center mb-6 text-purple-600 dark:text-purple-400">
                                    <ImageIcon className="w-6 h-6" />
                                </div>
                                <h3 className="text-2xl font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card1Title}</h3>
                                <p className="text-slate-700 dark:text-slate-400 mb-6 font-medium">{features.card1Text}</p>
                            </div>
                        </div>
                        <div className="md:col-span-1 lg:col-span-2 p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-blue-500/50 transition-all group relative overflow-hidden shadow-xl dark:shadow-none">
                            <div className="relative z-10">
                                <div className="flex items-center justify-between mb-4">
                                    <div className="w-12 h-12 rounded-xl bg-blue-500/20 flex items-center justify-center text-blue-600 dark:text-blue-400">
                                        <Video className="w-6 h-6" />
                                    </div>
                                    <span className="text-xs font-bold bg-blue-500/20 text-blue-600 dark:text-blue-300 px-2 py-1 rounded">VEO</span>
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card2Title}</h3>
                                <p className="text-slate-700 dark:text-slate-400 text-sm font-medium">{features.card2Text}</p>
                            </div>
                        </div>
                        <div className="p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-emerald-500/50 transition-all group shadow-xl dark:shadow-none">
                            <div className="w-12 h-12 rounded-xl bg-emerald-500/20 flex items-center justify-center mb-4 text-emerald-600 dark:text-emerald-400">
                                <ScanEye className="w-6 h-6" />
                            </div>
                            <h3 className="text-lg font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card3Title}</h3>
                            <p className="text-slate-700 dark:text-slate-400 text-sm font-medium">{features.card3Text}</p>
                        </div>
                        <div className="p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-pink-500/50 transition-all group shadow-xl dark:shadow-none">
                            <div className="w-12 h-12 rounded-xl bg-pink-500/20 flex items-center justify-center mb-4 text-pink-600 dark:text-pink-400">
                                <PencilRuler className="w-6 h-6" />
                            </div>
                            <h3 className="text-lg font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card4Title}</h3>
                            <p className="text-slate-700 dark:text-slate-400 text-sm font-medium">{features.card4Text}</p>
                        </div>
                        <div className="md:col-span-2 p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-amber-500/50 transition-all group flex flex-col md:flex-row gap-6 items-center shadow-xl dark:shadow-none">
                            <div className="flex-1">
                                <div className="w-12 h-12 rounded-xl bg-amber-500/20 flex items-center justify-center mb-4 text-amber-600 dark:text-amber-400">
                                    <Globe className="w-6 h-6" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card5Title}</h3>
                                <p className="text-slate-700 dark:text-slate-400 text-sm font-medium">{features.card5Text}</p>
                            </div>
                        </div>
                        <div className="md:col-span-1 lg:col-span-2 p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 hover:border-cyan-500/50 transition-all group shadow-xl dark:shadow-none">
                            <div className="flex items-start gap-4">
                                <div className="w-12 h-12 rounded-xl bg-cyan-500/20 flex items-center justify-center flex-shrink-0 text-cyan-600 dark:text-cyan-400">
                                    <BrainCircuit className="w-6 h-6" />
                                </div>
                                <div>
                                    <h3 className="text-xl font-bold mb-2 text-slate-900 dark:text-slate-100">{features.card6Title}</h3>
                                    <p className="text-slate-700 dark:text-slate-400 text-sm font-medium">{features.card6Text}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* Pricing - Conditionally Rendered */}
            {pricing.isVisible !== false && visiblePlans > 0 && (
                <section id="pricing" className="py-24 px-6 bg-slate-50 dark:bg-slate-950 relative transition-colors duration-300">
                    <div className="max-w-7xl mx-auto">
                        <div className="text-center mb-16">
                            <h2 className="text-3xl lg:text-4xl font-bold mb-4 font-['Space_Grotesk'] text-slate-900 dark:text-white">{pricing.title || "Pricing"}</h2>
                            <p className="text-slate-700 dark:text-slate-400 font-medium">{pricing.subtitle}</p>
                        </div>

                        {/* Dynamic Grid Logic */}
                        <div className={`grid gap-8 ${gridClass}`}>
                            {/* Starter - Updated corners and contrast */}
                            {pricing.starterVisible !== false && (
                                <div className="p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 shadow-2xl dark:shadow-none flex flex-col w-full">
                                    <h3 className="text-lg font-bold text-slate-700 dark:text-slate-400 mb-2">{pricing.starterName}</h3>
                                    <div className="flex items-baseline gap-1 mb-6">
                                        <span className="text-4xl font-bold text-slate-900 dark:text-white">{pricing.starterPrice}</span>
                                        <span className="text-slate-600 dark:text-slate-500 font-medium">/mo</span>
                                    </div>
                                    <ul className="space-y-4 mb-8 flex-1">
                                        {(pricing.starterFeatures || "").split('\n').map((feat: string, i: number) => (
                                            <li key={i} className="flex items-center gap-3 text-slate-800 dark:text-slate-300 text-sm font-medium"><CheckCircle2 size={16} className="text-slate-500 dark:text-slate-600" /> {feat}</li>
                                        ))}
                                    </ul>
                                    <button onClick={onStart} className="w-full py-3 rounded-xl border border-slate-300 dark:border-slate-700 text-slate-800 dark:text-slate-300 font-bold hover:bg-slate-100 dark:hover:bg-slate-800 transition">{pricing.starterButton}</button>
                                </div>
                            )}

                            {/* Pro - Updated corners and contrast */}
                            {pricing.proVisible !== false && (
                                <div className="p-8 rounded-xl bg-white dark:bg-slate-900 border border-amber-500/50 relative shadow-2xl shadow-amber-500/10 flex flex-col w-full ring-1 ring-amber-500/20 dark:ring-0">
                                    <div className="absolute top-0 right-0 bg-amber-500 text-white dark:text-slate-950 text-xs font-bold px-3 py-1 rounded-bl-xl rounded-tr-xl">{pricing.proTag}</div>
                                    <h3 className="text-lg font-bold text-amber-600 dark:text-amber-400 mb-2">{pricing.proName}</h3>
                                    <div className="flex items-baseline gap-1 mb-6">
                                        <span className="text-4xl font-bold text-slate-900 dark:text-white">{pricing.proPrice}</span>
                                        <span className="text-slate-600 dark:text-slate-500 font-medium">/mo</span>
                                    </div>
                                    <ul className="space-y-4 mb-8 flex-1">
                                        {(pricing.proFeatures || "").split('\n').map((feat: string, i: number) => (
                                            <li key={i} className="flex items-center gap-3 text-slate-800 dark:text-slate-300 text-sm font-medium"><CheckCircle2 size={16} className="text-amber-500" /> {feat}</li>
                                        ))}
                                    </ul>
                                    <button onClick={onStart} className="w-full py-3 rounded-xl bg-gradient-to-r from-amber-500 to-amber-600 text-white dark:text-slate-950 font-bold hover:shadow-lg hover:shadow-amber-500/20 transition">{pricing.proButton}</button>
                                </div>
                            )}

                            {/* Enterprise - Updated corners and contrast */}
                            {pricing.agencyVisible !== false && (
                                <div className="p-8 rounded-xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 shadow-2xl dark:shadow-none flex flex-col w-full">
                                    <h3 className="text-lg font-bold text-slate-700 dark:text-slate-400 mb-2">{pricing.agencyName}</h3>
                                    <div className="flex items-baseline gap-1 mb-6">
                                        <span className="text-4xl font-bold text-slate-900 dark:text-white">{pricing.agencyPrice}</span>
                                        <span className="text-slate-600 dark:text-slate-500 font-medium">/mo</span>
                                    </div>
                                    <ul className="space-y-4 mb-8 flex-1">
                                        {(pricing.agencyFeatures || "").split('\n').map((feat: string, i: number) => (
                                            <li key={i} className="flex items-center gap-3 text-slate-800 dark:text-slate-300 text-sm font-medium"><CheckCircle2 size={16} className="text-slate-500 dark:text-slate-600" /> {feat}</li>
                                        ))}
                                    </ul>
                                    <button className="w-full py-3 rounded-xl border border-slate-300 dark:border-slate-700 text-slate-800 dark:text-slate-300 font-bold hover:bg-slate-100 dark:hover:bg-slate-800 transition">{pricing.agencyButton}</button>
                                </div>
                            )}
                        </div>
                    </div>
                </section>
            )}

            {/* ABOUT SECTION */}
            <section id="about" className="py-32 px-6 bg-white dark:bg-slate-950 relative overflow-hidden transition-colors duration-300">
                <div className="max-w-7xl mx-auto relative z-10">
                    <div className="text-center mb-24">
                        <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-amber-500/10 border border-amber-500/20 text-amber-600 dark:text-amber-400 text-xs font-bold uppercase tracking-wider mb-6">
                            Our Mission
                        </div>
                        <h2 className="text-4xl lg:text-6xl font-bold font-['Space_Grotesk'] mb-8 text-slate-900 dark:text-white" dangerouslySetInnerHTML={{ __html: about.title }} />
                        <p className="text-slate-700 dark:text-slate-400 max-w-2xl mx-auto text-xl leading-relaxed font-medium">
                            {about.description}
                        </p>
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-24 items-center mb-32">
                        {/* Abstract Visual */}
                        <div className="relative group order-2 lg:order-1">
                            <div className="absolute -inset-2 bg-gradient-to-r from-amber-500 to-purple-600 rounded-3xl blur-2xl opacity-20 group-hover:opacity-40 transition duration-1000"></div>
                            <div className="relative rounded-3xl overflow-hidden border border-slate-200 dark:border-slate-800 bg-slate-100 dark:bg-slate-900 h-[500px]">
                                <img
                                    src="https://images.unsplash.com/photo-1618005182384-a83a8bd57fbe?q=80&w=2564&auto=format&fit=crop"
                                    alt="Abstract Future Tech"
                                    className="w-full h-full object-cover opacity-80 dark:opacity-60 group-hover:scale-105 transition duration-700"
                                />
                                <div className="absolute inset-0 bg-gradient-to-t from-white dark:from-slate-950 via-transparent to-transparent"></div>
                            </div>
                        </div>

                        {/* Mission Text */}
                        <div className="space-y-10 order-1 lg:order-2">
                            <h3 className="text-3xl font-bold text-slate-900 dark:text-white">{about.whyTitle}</h3>
                            <div className="space-y-8">
                                <div className="flex gap-5">
                                    <div className="w-14 h-14 rounded-2xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 flex items-center justify-center shrink-0 shadow-lg">
                                        <Zap className="text-amber-500 dark:text-amber-400 w-6 h-6" />
                                    </div>
                                    <div>
                                        <h4 className="text-xl font-bold text-slate-800 dark:text-slate-200 mb-2">{about.box1Title}</h4>
                                        <p className="text-slate-700 dark:text-slate-400 leading-relaxed font-medium">{about.box1Text}</p>
                                    </div>
                                </div>
                                <div className="flex gap-5">
                                    <div className="w-14 h-14 rounded-2xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 flex items-center justify-center shrink-0 shadow-lg">
                                        <ShieldCheck className="text-emerald-600 dark:text-emerald-400 w-6 h-6" />
                                    </div>
                                    <div>
                                        <h4 className="text-xl font-bold text-slate-800 dark:text-slate-200 mb-2">{about.box2Title}</h4>
                                        <p className="text-slate-700 dark:text-slate-400 leading-relaxed font-medium">{about.box2Text}</p>
                                    </div>
                                </div>
                                <div className="flex gap-5">
                                    <div className="w-14 h-14 rounded-2xl bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-800 flex items-center justify-center shrink-0 shadow-lg">
                                        <Cpu className="text-blue-600 dark:text-blue-400 w-6 h-6" />
                                    </div>
                                    <div>
                                        <h4 className="text-xl font-bold text-slate-800 dark:text-slate-200 mb-2">{about.box3Title}</h4>
                                        <p className="text-slate-700 dark:text-slate-400 leading-relaxed font-medium">{about.box3Text}</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Global Impact Slider (Auto-Scroller) */}
                    <div className="mb-32">
                        <div className="flex items-center justify-between mb-10">
                            <h3 className="text-2xl font-bold text-slate-900 dark:text-white">{globalImpact.title || "Global Impact"}</h3>
                            <div className="flex gap-2">
                                <button
                                    onClick={() => handleMediaTypeChange('image')}
                                    className={`px-4 py-1.5 rounded-full text-sm font-bold border transition-all ${impactMediaType === 'image' ? 'bg-amber-500 text-slate-950 border-amber-500' : 'bg-slate-100 dark:bg-slate-900 border-slate-200 dark:border-slate-800 text-slate-700 dark:text-slate-400 hover:border-slate-400'}`}
                                >
                                    Images
                                </button>
                                <button
                                    onClick={() => handleMediaTypeChange('video')}
                                    className={`px-4 py-1.5 rounded-full text-sm font-bold border transition-all ${impactMediaType === 'video' ? 'bg-blue-500 text-white border-blue-500' : 'bg-slate-100 dark:bg-slate-900 border-slate-200 dark:border-slate-800 text-slate-700 dark:text-slate-400 hover:border-slate-400'}`}
                                >
                                    Videos
                                </button>
                            </div>
                        </div>

                        {/* Auto Slider Container */}
                        {filteredSlides.length > 0 ? (
                            <div className="relative overflow-hidden rounded-2xl" ref={sliderRef}>
                                <div
                                    className="flex transition-transform duration-700 ease-in-out"
                                    style={{ transform: `translateX(-${slideIndex * (100 / itemsPerView)}%)` }}
                                >
                                    {filteredSlides.map((slide: any, index: number) => (
                                        <div key={`${slide.id}-${index}`} className="flex-shrink-0 px-3 box-border" style={{ width: `${100 / itemsPerView}%` }}>
                                            <div className="relative group overflow-hidden rounded-2xl border border-slate-200 dark:border-slate-800 h-80 bg-slate-100 dark:bg-slate-900">
                                                {slide.type === 'video' ? (
                                                    <video src={slide.url} autoPlay muted loop playsInline className="w-full h-full object-cover opacity-90 dark:opacity-70 group-hover:opacity-100 group-hover:scale-105 transition duration-700" />
                                                ) : (
                                                    <img src={slide.url} className="w-full h-full object-cover opacity-90 dark:opacity-70 group-hover:opacity-100 group-hover:scale-105 transition duration-700" alt={slide.title} />
                                                )}
                                                <div className="absolute bottom-0 left-0 p-6 bg-gradient-to-t from-white/90 dark:from-slate-950 to-transparent w-full">
                                                    <p className="text-slate-900 dark:text-white font-bold text-lg">{slide.title}</p>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        ) : (
                            <div className="h-80 flex items-center justify-center bg-slate-100 dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800">
                                <p className="text-slate-500 dark:text-slate-400">No content available for this category.</p>
                            </div>
                        )}
                    </div>

                    {/* Tech Stack Strip - UPDATED VISIBILITY */}
                    <div className="pt-16 text-center group">
                        <div className="inline-block mb-10 px-6 py-2 rounded-full bg-blue-50 dark:bg-blue-900/10 border border-blue-100 dark:border-blue-900/30">
                            <p className="text-sm uppercase tracking-[0.2em] font-bold text-blue-600 dark:text-blue-400">
                                {poweredBy.title || "Powered By"}
                            </p>
                        </div>

                        {/* Inverted Hover Logic: Colored by default, Grey on container hover */}
                        <div className="flex flex-wrap justify-center gap-6 lg:gap-12 transition-all duration-500 group-hover:opacity-60">
                            <div className="flex items-center gap-3 px-6 py-4 bg-white dark:bg-slate-900/50 border border-slate-300 dark:border-slate-800 rounded-xl shadow-lg dark:shadow-none hover:scale-105 transition-all duration-300 group-hover:grayscale hover:!grayscale-0 hover:!opacity-100">
                                <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
                                <span className="text-slate-900 dark:text-slate-200 font-bold text-lg">Gemini</span>
                                <span className="text-xs bg-blue-100 dark:bg-blue-500/10 text-blue-700 dark:text-blue-400 px-2 py-1 rounded font-mono border border-blue-200 dark:border-blue-500/20">{poweredBy.chip1}</span>
                            </div>
                            <div className="flex items-center gap-3 px-6 py-4 bg-white dark:bg-slate-900/50 border border-slate-300 dark:border-slate-800 rounded-xl shadow-lg dark:shadow-none hover:scale-105 transition-all duration-300 group-hover:grayscale hover:!grayscale-0 hover:!opacity-100">
                                <div className="w-2 h-2 bg-purple-500 rounded-full animate-pulse"></div>
                                <span className="text-slate-900 dark:text-slate-200 font-bold text-lg">Veo</span>
                                <span className="text-xs bg-purple-100 dark:bg-purple-500/10 text-purple-700 dark:text-purple-400 px-2 py-1 rounded font-mono border border-purple-200 dark:border-purple-500/20">{poweredBy.chip2}</span>
                            </div>
                            <div className="flex items-center gap-3 px-6 py-4 bg-white dark:bg-slate-900/50 border border-slate-300 dark:border-slate-800 rounded-xl shadow-lg dark:shadow-none hover:scale-105 transition-all duration-300 group-hover:grayscale hover:!grayscale-0 hover:!opacity-100">
                                <div className="w-2 h-2 bg-amber-500 rounded-full animate-pulse"></div>
                                <span className="text-slate-900 dark:text-slate-200 font-bold text-lg">Imagen</span>
                                <span className="text-xs bg-amber-100 dark:bg-amber-500/10 text-amber-700 dark:text-amber-400 px-2 py-1 rounded font-mono border border-amber-200 dark:border-amber-500/20">{poweredBy.chip3}</span>
                            </div>
                        </div>
                    </div>

                    <div className="mt-32 text-center bg-gradient-to-b from-slate-100 to-white dark:from-slate-900 dark:to-slate-950 border border-slate-200 dark:border-slate-800 rounded-3xl p-12 relative overflow-hidden shadow-lg dark:shadow-none">
                        <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-amber-500 via-purple-500 to-blue-500"></div>
                        <h2 className="text-3xl font-bold text-slate-900 dark:text-white mb-6">{cta.title || "Ready to Start?"}</h2>
                        <button
                            onClick={onStart}
                            className="px-8 py-4 bg-slate-900 dark:bg-white text-white dark:text-slate-950 font-bold rounded-xl hover:bg-slate-800 dark:hover:bg-slate-200 transition-colors inline-flex items-center gap-2"
                        >
                            {cta.buttonText || "Join Now"} <ArrowRight size={20} />
                        </button>
                    </div>
                </div>
            </section>

            {/* Footer */}
            <footer className="py-12 px-6 bg-white dark:bg-slate-950 relative z-10 transition-colors duration-300">
                <div className="max-w-7xl mx-auto flex flex-col md:flex-row justify-between items-center gap-6">
                    <div className="flex items-center gap-2">
                        {general.appLogo ? (
                            <img src={general.appLogo} alt="Logo" className="w-8 h-8 rounded-lg object-cover" />
                        ) : (
                            <div className="w-8 h-8 rounded-lg bg-gradient-to-br from-amber-400 to-amber-600 flex items-center justify-center">
                                <Sparkles className="text-white w-4 h-4" />
                            </div>
                        )}
                        <span className="text-lg font-bold text-slate-900 dark:text-slate-200">{general.appName || "StockForge AI"}</span>
                    </div>
                    <div className="text-slate-800 dark:text-slate-500 text-sm font-medium">
                        {general.footerText || "© 2024 StockForge AI. Made with 💛 and Gemini."}
                    </div>
                    <div className="flex gap-6 items-center">
                        <ShieldCheck className="w-5 h-5 text-slate-600 hover:text-slate-900 dark:text-slate-500 dark:hover:text-slate-300 cursor-pointer" />
                    </div>
                </div>
            </footer>

            {isVideoOpen && (
                <div className="fixed inset-0 z-[60] bg-black/80 backdrop-blur-md flex items-center justify-center p-6" onClick={() => setIsVideoOpen(false)}>
                    <div className="relative w-full max-w-4xl aspect-video bg-slate-900 rounded-2xl border border-slate-800 shadow-2xl flex items-center justify-center">
                        <p className="text-slate-500 flex items-center gap-2"><Play size={32} /> Demo Video Placeholder</p>
                    </div>
                </div>
            )}

            <button
                onClick={() => window.scrollTo({ top: 0, behavior: 'smooth' })}
                className={`fixed bottom-8 right-8 p-2 bg-gradient-to-r from-amber-400 to-orange-500 hover:from-amber-300 hover:to-orange-400 text-slate-900 rounded-full shadow-lg shadow-orange-500/20 z-50 transition-all duration-300 transform hover:scale-110 ${showScrollTop ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10 pointer-events-none'
                    }`}
            >
                <ArrowUp size={20} strokeWidth={3} />
            </button>
        </div>
    );
};

export default LandingPage;
