
import React, { useState, useEffect } from 'react';
import {
    Wand2, Download, RefreshCw, Copy, Trash2,
    Sparkles, Sliders, Image as ImageIcon, Layers,
    History as HistoryIcon, ChevronDown, ChevronUp,
    Maximize2, Share2, Heart, Lock, Shuffle,
    CheckCircle2, AlertCircle, Plus, Palette, Play, Key, Cpu
} from 'lucide-react';
import { generateImage, enhancePrompt, getPreferredModels } from '../services/geminiService';
import { dbService, Asset } from '../services/dbService';
import { authService } from '../services/authService';
import { ModelType } from '../types';
import { EnhancedLoadingOverlay } from './EnhancedLoadingOverlay';
import { AutoDownloadIndicator } from './AutoDownloadIndicator';
import { playSound } from '../utils/soundEffects';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';

const STYLE_PRESETS = [
    { id: 'none', label: 'No Preset (Raw)', preview: 'bg-slate-800' },
    { id: 'photorealistic', label: 'Photorealistic', preview: 'bg-emerald-500' },
    { id: 'digital_art', label: 'Digital Art', preview: 'bg-blue-500' },
    { id: 'oil_painting', label: 'Oil Painting', preview: 'bg-amber-500' },
    { id: 'watercolor', label: 'Watercolor', preview: 'bg-pink-500' },
    { id: 'anime', label: 'Anime / Cartoon', preview: 'bg-purple-500' },
    { id: 'cyberpunk', label: 'Cyberpunk', preview: 'bg-cyan-500' },
    { id: 'fantasy', label: 'Fantasy Art', preview: 'bg-indigo-500' },
    { id: '3d_render', label: '3D Render', preview: 'bg-red-500' },
    { id: 'vintage', label: 'Vintage', preview: 'bg-orange-500' },
];

const ASPECT_RATIOS = [
    { id: '1:1', label: 'Square', desc: 'Instagram', w: 'w-8', h: 'h-8' },
    { id: '16:9', label: 'Landscape', desc: 'YouTube', w: 'w-10', h: 'h-6' },
    { id: '9:16', label: 'Portrait', desc: 'Stories', w: 'w-6', h: 'h-10' },
    { id: '4:3', label: 'Classic', desc: 'Photo', w: 'w-9', h: 'h-7' },
    { id: '3:4', label: 'Print', desc: 'Magazine', w: 'w-7', h: 'h-9' },
];

const CREATIVE_MODELS = [
    { id: ModelType.IMAGEN_STD, name: 'Imagen 4 Generate (Standard)' },
    { id: ModelType.IMAGEN_FAST, name: 'Imagen 4 Fast (Speed)' },
    { id: ModelType.IMAGEN_ULTRA, name: 'Imagen 4 Ultra (Quality)' },
    { id: ModelType.PRO_IMAGE_3_0, name: 'Gemini 3 Pro Image (Raw)' },
    { id: ModelType.FLASH_IMAGE_2_5, name: 'Gemini 2.5 Flash Image (Balanced)' }
];

const ImageGenerator: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'generator' | 'bulk' | 'history'>('generator');
    const [hasApiKey, setHasApiKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [generationInfo, setGenerationInfo] = useState<{ apiKey: string, model: string } | null>(null);

    // Generator State
    const [prompt, setPrompt] = useState('');
    const [negativePrompt, setNegativePrompt] = useState('');
    const [aspectRatio, setAspectRatio] = useState('1:1');
    const [imageCount, setImageCount] = useState(1);
    const [stylePreset, setStylePreset] = useState('photorealistic');
    const [quality, setQuality] = useState(80);
    const [seed, setSeed] = useState<number | string>('');
    const [isAdvancedOpen, setIsAdvancedOpen] = useState(false);
    const [isNegativeOpen, setIsNegativeOpen] = useState(false);
    const [selectedModel, setSelectedModel] = useState(ModelType.IMAGEN_STD);

    // Active Models (Enabled in Settings)
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // Advanced Options State
    const [useWatermark, setUseWatermark] = useState(false);
    const [watermarkText, setWatermarkText] = useState('');
    const [colorGrading, setColorGrading] = useState('none');

    // Processing State
    const [isGenerating, setIsGenerating] = useState(false);
    const [generatedImages, setGeneratedImages] = useState<Asset[]>([]);
    const [rateLimitError, setRateLimitError] = useState(false); // Popup state

    // Bulk State
    const [bulkPrompts, setBulkPrompts] = useState('');
    const [bulkStatus, setBulkStatus] = useState<'idle' | 'processing' | 'complete'>('idle');
    const [bulkProgress, setBulkProgress] = useState(0);
    const [bulkResults, setBulkResults] = useState<Asset[]>([]);

    // History
    const [history, setHistory] = useState<Asset[]>([]);

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('image').then(setHistory);
        }
    }, [activeTab]);

    const [availableModels, setAvailableModels] = useState(CREATIVE_MODELS);

    useEffect(() => {
        refreshAccess();
        const handleFocus = () => refreshAccess();
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    const refreshAccess = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        const [keys, settings] = await Promise.all([
            authService.getUserApiKeys(currentUser.uid),
            authService.getSiteSettings()
        ]);

        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        setHasApiKey(!!validKey);

        const filtered = CREATIVE_MODELS.filter(m => authService.canUserAccessModel(m.id, keys, settings));
        setAvailableModels(filtered);

        // Update selected if needed
        if (filtered.length > 0 && !filtered.find(m => m.id === selectedModel)) {
            setSelectedModel(filtered[0].id as ModelType);
        }

        // Update active stats display
        updateActiveModels(filtered.map(m => m.name));

        setIsCheckingKey(false);
    };

    const updateActiveModels = (names: string[]) => {
        // Just used for display of 'Active Models'
        setActiveModels(names);
    };

    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    const handleGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!prompt || !currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        // Clear history for this type before starting new batch
        await dbService.clearByType('image');

        setIsGenerating(true);
        setGenerationInfo({ apiKey: apiKeys[0], model: selectedModel });
        try {
            let finalPrompt = stylePreset !== 'none' ? `${prompt}, ${stylePreset} style` : prompt;
            if (colorGrading !== 'none') finalPrompt += `, ${colorGrading} color grading`;

            const urls = await generateImage(apiKeys, finalPrompt, {
                aspectRatio,
                numberOfImages: imageCount,
                negativePrompt,
                modelId: selectedModel
            });

            const batchId = Date.now().toString(); // Generate unique batch ID

            const newAssets: Asset[] = urls.map(url => ({
                id: Math.random().toString(36).substring(2, 10),
                type: 'image',
                url,
                prompt: finalPrompt,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                aspectRatio,
                metadata: { style: stylePreset, seed: seed || 'random', model: selectedModel },
                batchId: batchId // Add batchId
            }));

            // Save to DB (add all from this batch)
            for (const a of newAssets) {
                await dbService.add(a);
            }

            // Log Activity & Stats
            const currentUser = authService.getCurrentUser();
            if (currentUser) {
                authService.incrementUserStat('imagesGenerated', imageCount, selectedModel);
                authService.logActivity(currentUser.uid, 'generate', `Generated ${imageCount} images: "${prompt.substring(0, 30)}..."`);

                // Track Usage per API Key
                // Using first key as proxy for billing
                if (apiKeys[0]) {
                    authService.deductCredits(currentUser.uid, imageCount, apiKeys[0], selectedModel);
                }
            }

            // Redirect to History
            setActiveTab('history');

            setGeneratedImages(newAssets);

            // Play success sound
            await playSound('success');
        } catch (error) {
            alert('Failed to generate image. Please check your API key and quota.');

            // Play error sound
            await playSound('error');
        } finally {
            setIsGenerating(false);
        }
    };

    const handleBulkGenerate = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        const prompts = bulkPrompts.split('\n').filter(p => p.trim());
        if (prompts.length === 0) return;

        // Clear history for this type before starting new batch
        await dbService.clearByType('image');

        setBulkStatus('processing');
        setBulkProgress(0);
        setBulkResults([]);

        const results: Asset[] = [];

        for (let i = 0; i < prompts.length; i++) {
            // Apply Image Delay / Global Cooldown
            const prefs = (currentUser as any)?.preferences?.generationConfig;
            const globalCooldownSec = Number(prefs?.globalCooldown || 0);
            const imageDelaySec = Number(prefs?.imageDelay || 0);

            // Additive Delay Logic: Global Cooldown + Image Delay
            const effectiveDelaySec = globalCooldownSec + imageDelaySec;

            if (effectiveDelaySec > 0) {
                const cooldownMs = effectiveDelaySec * 1000;
                console.log(`[ImageGen] Applying Total Delay: ${cooldownMs}ms (${globalCooldownSec}s Global + ${imageDelaySec}s Image)`);
                await new Promise(r => setTimeout(r, cooldownMs));
            }

            try {
                const p = prompts[i];
                let finalPrompt = stylePreset !== 'none' ? `${p}, ${stylePreset} style` : p;

                // Generate with Robust Failover
                const urls = await generateImage(apiKeys, finalPrompt, {
                    aspectRatio,
                    numberOfImages: 1,
                    modelId: selectedModel
                });

                if (urls.length > 0) {
                    const asset: Asset = {
                        id: Math.random().toString(36).substring(2, 10),
                        type: 'image',
                        url: urls[0],
                        prompt: p,
                        createdAt: Date.now(),
                        isFavorite: false,
                        isDeleted: false,
                        aspectRatio,
                        metadata: { model: selectedModel }
                    };
                    await dbService.add(asset);

                    if (currentUser) {
                        authService.incrementUserStat('imagesGenerated', 1);
                        authService.incrementUserStat('modelUsage', 1, selectedModel);

                        // Track Usage per API Key (Single)
                        if (apiKeys[0]) {
                            authService.deductCredits(currentUser.uid, 1, apiKeys[0], selectedModel);
                        }
                    }
                    results.push(asset);
                }
            } catch (e) {
                console.error(`Failed prompt: ${prompts[i]}`);
            }

            // Update progress
            setBulkProgress(Math.round(((i + 1) / prompts.length) * 100));
        }

        if (currentUser) {
            authService.logActivity(currentUser.uid, 'generate', `Batch generated ${results.length} images`);
        }

        setBulkResults(results);
        setBulkStatus('complete');
    }

    const handleEnhancePrompt = async () => {
        const currentUser = authService.getCurrentUser();
        if (!prompt || !currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        const enhanced = await enhancePrompt(apiKeys, prompt);
        setPrompt(enhanced);
    };

    const handleRandomSeed = () => setSeed(Math.floor(Math.random() * 1000000));

    const handleDownloadAllHistory = () => {
        downloadItems(history.map(item => ({
            url: item.url,
            filename: `image_history_${item.id}.png`
        })), 'image_history');
    };

    const handleDownloadBatch = () => {
        downloadItems(bulkResults.map(item => ({
            url: item.url,
            filename: `batch_${item.id}.png`
        })), 'image_batch');
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans animate-fade-in text-slate-900 dark:text-slate-200">

            {/* Header */}
            <div className="mb-6 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <ImageIcon className="text-purple-500 w-8 h-8" /> Image Generation Studio
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">Create stunning photorealistic images and art using Imagen 4.0 and Gemini.</p>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Tab Navigation */}
            <div className="flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-6 shrink-0">
                <div className="flex w-full md:w-auto overflow-x-auto">
                    {[
                        { id: 'generator', label: 'Generator', icon: Wand2 },
                        { id: 'bulk', label: 'Bulk', icon: Layers },
                        { id: 'history', label: 'History', icon: HistoryIcon },
                    ].map((tab) => (
                        <button
                            key={tab.id}
                            onClick={() => setActiveTab(tab.id as any)}
                            className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === tab.id ? 'text-amber-600 dark:text-amber-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                                }`}
                        >
                            <tab.icon size={16} />
                            {tab.label}
                            {activeTab === tab.id && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-amber-500 animate-fade-in"></div>}
                        </button>
                    ))}
                </div>

                {/* Active Models Display & Auto-Download Indicator */}
                {activeTab !== 'history' && (
                    <div className="hidden md:flex items-center gap-2 my-2 mr-2">
                        <div className="flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg">
                            <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                                <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                            </span>
                            <div className="flex gap-1">
                                {activeModels.map((model, idx) => (
                                    <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                        {model}
                                    </span>
                                ))}
                                {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                            </div>
                        </div>
                        <AutoDownloadIndicator />
                    </div>
                )}
            </div>

            {/* Content Area */}
            <div className={`flex-1 overflow-hidden transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>

                {/* GENERATOR TAB */}
                {activeTab === 'generator' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20 custom-scrollbar">
                        {/* LEFT PANEL: CONTROLS */}
                        <div className="lg:col-span-4 space-y-6 h-fit">
                            {/* Prompt Input */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div className="flex justify-between items-center">
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300">Describe your image</label>
                                    <span className="text-xs text-slate-500">{prompt.length}/4000</span>
                                </div>
                                <textarea
                                    value={prompt}
                                    onChange={(e) => setPrompt(e.target.value)}
                                    className="w-full h-32 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 focus:ring-1 focus:ring-amber-500 outline-none resize-none placeholder:text-slate-400 dark:placeholder:text-slate-600 text-sm"
                                    placeholder="A serene mountain landscape at sunset with a lake reflection, photorealistic, high detail"
                                />
                                <div className="flex justify-between items-center">
                                    <button onClick={handleEnhancePrompt} disabled={!hasApiKey} className="text-xs flex items-center gap-1 text-amber-600 dark:text-amber-500 hover:text-amber-700 dark:hover:text-amber-400 font-bold transition-colors bg-amber-500/10 px-3 py-1.5 rounded-lg border border-amber-500/20 disabled:opacity-50">
                                        <Sparkles size={12} /> Enhance Prompt
                                    </button>
                                </div>
                                <div className="border-t border-slate-200 dark:border-slate-800/50 pt-3">
                                    <button onClick={() => setIsNegativeOpen(!isNegativeOpen)} className="flex items-center gap-2 text-xs font-bold text-slate-500 hover:text-slate-700 dark:hover:text-slate-300 uppercase tracking-wider w-full">
                                        {isNegativeOpen ? <ChevronUp size={12} /> : <ChevronDown size={12} />} Negative Prompt (Optional)
                                    </button>
                                    {isNegativeOpen && (
                                        <textarea
                                            value={negativePrompt}
                                            onChange={(e) => setNegativePrompt(e.target.value)}
                                            className="w-full h-20 mt-2 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 text-sm focus:border-red-500/50 outline-none resize-none"
                                            placeholder="blurry, low quality..."
                                        />
                                    )}
                                </div>
                            </div>

                            {/* Generation Settings */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-6 shadow-sm dark:shadow-none">
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 flex items-center gap-2">
                                        <Cpu size={14} className="text-amber-500" /> AI Model
                                    </label>
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value as ModelType)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500">
                                        {availableModels.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Image Count</label>
                                    <select value={imageCount} onChange={(e) => setImageCount(Number(e.target.value))} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-2 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500">
                                        <option value={1}>1 Image</option>
                                        <option value={2}>2 Images</option>
                                        <option value={3}>3 Images</option>
                                        <option value={4}>4 Images</option>
                                    </select>
                                </div>
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Aspect Ratio</label>
                                    <div className="grid grid-cols-5 gap-2">
                                        {ASPECT_RATIOS.map((ratio) => (
                                            <button key={ratio.id} onClick={() => setAspectRatio(ratio.id)} className={`group flex flex-col items-center gap-2 p-2 rounded-xl border transition-all ${aspectRatio === ratio.id ? 'bg-amber-500/10 border-amber-500 text-amber-600 dark:text-amber-400' : 'bg-slate-50 dark:bg-slate-950 border-slate-200 dark:border-slate-800 text-slate-500'}`}>
                                                <div className={`${ratio.w} ${ratio.h} border-2 rounded-sm transition-colors ${aspectRatio === ratio.id ? 'border-amber-500 bg-amber-500/20' : 'border-slate-400 dark:border-slate-600'}`}></div>
                                                <span className="text-[10px] font-medium">{ratio.id}</span>
                                            </button>
                                        ))}
                                    </div>
                                </div>
                                <div>
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Style Preset</label>
                                    <select value={stylePreset} onChange={(e) => setStylePreset(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-sm text-slate-900 dark:text-slate-200 outline-none focus:border-amber-500">
                                        {STYLE_PRESETS.map(s => <option key={s.id} value={s.id}>{s.label}</option>)}
                                    </select>
                                </div>
                                <div className="border-t border-slate-200 dark:border-slate-800 pt-4">
                                    <button onClick={() => setIsAdvancedOpen(!isAdvancedOpen)} className="flex items-center justify-between w-full text-xs font-bold text-slate-500 hover:text-slate-700 dark:hover:text-slate-300 uppercase">
                                        <span className="flex items-center gap-2"><Sliders size={12} /> Advanced Options</span>
                                        {isAdvancedOpen ? <ChevronUp size={14} /> : <ChevronDown size={14} />}
                                    </button>
                                    {isAdvancedOpen && (
                                        <div className="mt-4 space-y-4 animate-slide-up">
                                            <div className="space-y-2">
                                                <label className="text-xs font-bold text-slate-500 dark:text-slate-400">Seed</label>
                                                <div className="flex gap-2">
                                                    <input type="number" value={seed} onChange={(e) => setSeed(e.target.value)} placeholder="Random" className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none" />
                                                    <button onClick={handleRandomSeed} className="p-2 bg-slate-100 dark:bg-slate-800 rounded-lg text-slate-500 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white border border-slate-200 dark:border-slate-700"><Shuffle size={16} /></button>
                                                </div>
                                            </div>
                                            <div>
                                                <label className="text-xs font-bold text-slate-500 dark:text-slate-400 block mb-2">Color Grading</label>
                                                <select value={colorGrading} onChange={(e) => setColorGrading(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none">
                                                    <option value="none">None</option>
                                                    <option value="warm">Warm Tone</option>
                                                    <option value="cool">Cool Tone</option>
                                                    <option value="vibrant">Vibrant</option>
                                                    <option value="bw">Black & White</option>
                                                </select>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>

                            <button onClick={handleGenerate} disabled={isGenerating || !prompt || !hasApiKey} className="w-full py-4 bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-400 hover:to-amber-500 text-white dark:text-slate-950 font-bold rounded-xl transition-all shadow-lg shadow-amber-500/20 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed">
                                {isGenerating ? <><RefreshCw className="animate-spin" /> Generating...</> : <><Wand2 /> Generate Images</>}
                                {!isGenerating && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">{imageCount} API Call{imageCount > 1 ? 's' : ''}</span>}
                            </button>
                        </div>

                        {/* RIGHT PANEL: PREVIEW */}
                        <div className="lg:col-span-8 bg-white dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 flex flex-col relative min-h-[600px] shadow-sm dark:shadow-none">
                            {generatedImages.length === 0 && !isGenerating && (
                                <div className="h-full flex flex-col items-center justify-center text-slate-500 dark:text-slate-600">
                                    <div className="w-24 h-24 bg-slate-100 dark:bg-slate-900 rounded-3xl flex items-center justify-center mb-6 border border-slate-200 dark:border-slate-800 rotate-12">
                                        <ImageIcon size={48} className="opacity-50" />
                                    </div>
                                    <h3 className="text-2xl font-bold text-slate-700 dark:text-slate-500 mb-2">Ready to Create?</h3>
                                    <p className="text-sm text-slate-600 dark:text-slate-600 max-w-xs text-center">Configure your settings on the left and hit generate to see Imagen 4.0 magic happen.</p>
                                </div>
                            )}



                            {generatedImages.length > 0 && (
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 overflow-y-auto custom-scrollbar h-full">
                                    {generatedImages.map((img) => (
                                        <div key={img.id} className="group relative rounded-xl overflow-hidden border border-slate-200 dark:border-slate-800 bg-slate-100 dark:bg-slate-900 aspect-square">
                                            <img src={img.url} alt={img.prompt} className="w-full h-full object-cover" />
                                            <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-all flex flex-col justify-end p-4">
                                                <p className="text-xs text-slate-300 line-clamp-2 mb-3">{img.prompt}</p>
                                                <div className="flex gap-2">
                                                    <a href={img.url} download="image.png" className="flex-1 py-2 bg-white text-black font-bold rounded-lg text-xs flex items-center justify-center gap-2 hover:bg-slate-200">
                                                        <Download size={14} /> Download
                                                    </a>
                                                    <button className="p-2 bg-white/10 backdrop-blur rounded-lg text-white hover:bg-white/20">
                                                        <Maximize2 size={16} />
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {/* BULK TAB */}
                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20">
                        <div className="lg:col-span-5 space-y-6">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <h3 className="font-bold text-slate-900 dark:text-slate-200 mb-4">Bulk Generator</h3>
                                <p className="text-xs text-slate-500 mb-4">Enter multiple prompts, one per line.</p>
                                <textarea
                                    value={bulkPrompts}
                                    onChange={(e) => setBulkPrompts(e.target.value)}
                                    className="w-full h-64 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-4 text-slate-900 dark:text-slate-200 text-sm font-mono outline-none resize-none focus:border-amber-500"
                                    placeholder="A cyberpunk city in neon rain&#10;A peaceful zen garden with cherry blossoms&#10;A futuristic robot playing chess"
                                />
                                <div className="flex flex-col gap-4 mt-4">
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value as ModelType)} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-700 dark:text-slate-300 outline-none focus:border-amber-500">
                                        {CREATIVE_MODELS.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                    <div className="flex gap-4">
                                        <select value={aspectRatio} onChange={(e) => setAspectRatio(e.target.value)} className="bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-700 dark:text-slate-300 outline-none">
                                            {ASPECT_RATIOS.map(r => <option key={r.id} value={r.id}>{r.label}</option>)}
                                        </select>
                                        <button
                                            onClick={handleBulkGenerate}
                                            disabled={bulkStatus === 'processing' || !bulkPrompts || !hasApiKey}
                                            className="flex-1 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50"
                                        >
                                            {bulkStatus === 'processing' ? <RefreshCw className="animate-spin" size={16} /> : <Play size={16} />}
                                            {bulkStatus === 'processing' ? 'Processing...' : 'Start Batch'}
                                            {bulkStatus !== 'processing' && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">{bulkPrompts.split('\n').filter(p => p.trim()).length} API Calls</span>}
                                        </button>
                                    </div>
                                </div>
                            </div>

                            {bulkStatus !== 'idle' && (
                                <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                    <div className="flex justify-between text-sm mb-2 font-bold text-slate-700 dark:text-slate-300">
                                        <span>Batch Progress</span>
                                        <div className="flex items-center gap-2">
                                            {bulkStatus === 'complete' && bulkResults.length > 0 && (
                                                <button onClick={handleDownloadBatch} className="text-xs bg-amber-500/10 text-amber-600 dark:text-amber-400 hover:text-amber-700 dark:hover:text-amber-300 px-2 py-1 rounded transition-colors flex items-center gap-1">
                                                    <Download size={12} /> Download Batch
                                                </button>
                                            )}
                                            <span>{bulkProgress}%</span>
                                        </div>
                                    </div>
                                    <div className="w-full bg-slate-200 dark:bg-slate-800 h-2 rounded-full overflow-hidden">
                                        <div className="h-full bg-amber-500 transition-all duration-300" style={{ width: `${bulkProgress}%` }}></div>
                                    </div>
                                    <div className="mt-4 text-xs text-slate-500">
                                        {bulkResults.length} / {bulkPrompts.split('\n').filter(p => p.trim()).length} Completed
                                    </div>
                                </div>
                            )}
                        </div>

                        <div className="lg:col-span-7 bg-white dark:bg-slate-900 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 h-[600px] overflow-y-auto custom-scrollbar shadow-sm dark:shadow-none">
                            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                                {bulkResults.map((asset, idx) => (
                                    <div key={idx} className="relative group rounded-xl overflow-hidden aspect-square bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700">
                                        <img src={asset.url} alt={asset.prompt} className="w-full h-full object-cover" />
                                        <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition flex items-center justify-center">
                                            <a href={asset.url} download className="p-2 bg-white text-black rounded-full"><Download size={16} /></a>
                                        </div>
                                    </div>
                                ))}
                                {bulkResults.length === 0 && (
                                    <div className="col-span-full flex flex-col items-center justify-center text-slate-500 h-64">
                                        <Layers size={32} className="mb-2 opacity-50" />
                                        <p>Generated images will appear here</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}

                {/* HISTORY TAB */}
                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="flex justify-between items-center mb-4 px-2">
                            <h3 className="font-bold text-slate-700 dark:text-slate-300">Generation History ({history.length})</h3>
                            {history.length > 0 && (
                                <button onClick={handleDownloadAllHistory} className="flex items-center gap-2 px-3 py-1.5 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition shadow-lg shadow-amber-500/20">
                                    <Download size={14} /> Download All
                                </button>
                            )}
                        </div>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                            {history.map((item) => (
                                <div key={item.id} className="group relative rounded-xl overflow-hidden border border-slate-200 dark:border-slate-800 aspect-square bg-slate-100 dark:bg-slate-900">
                                    <img src={item.url} alt={item.prompt} className="w-full h-full object-cover" />
                                    <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-all flex flex-col justify-end p-4">
                                        <p className="text-xs text-slate-300 line-clamp-2 mb-2">{item.prompt}</p>
                                        <div className="flex gap-2 justify-end">
                                            <button onClick={() => dbService.toggleFavorite(item.id)} className={`p-1.5 rounded-lg bg-white/10 backdrop-blur ${item.isFavorite ? 'text-red-500' : 'text-white'}`}><Heart size={14} /></button>
                                            <a href={item.url} download className="p-1.5 bg-white/10 backdrop-blur rounded-lg text-white"><Download size={14} /></a>
                                        </div>
                                    </div>
                                </div>
                            ))}
                            {history.length === 0 && (
                                <div className="col-span-full text-center py-12 text-slate-500">
                                    <HistoryIcon size={32} className="mx-auto mb-2 opacity-30" />
                                    No generation history found.
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </div>
            {/* Loading Overlay */}
            <EnhancedLoadingOverlay
                isVisible={isGenerating}
                title="Forging Pixels"
                description={`Generating ${imageCount} high-quality image${imageCount > 1 ? 's' : ''} with ${selectedModel}...`}
                simulate={true}
                onCancel={() => setIsGenerating(false)}
                apiKey={generationInfo?.apiKey}
                model={generationInfo?.model}
            />
            {/* Rate Limit Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />
        </div>
    );
};

export default ImageGenerator;
