
import React, { useState, useRef, useEffect } from 'react';
import {
    PencilRuler, Upload, RefreshCw, Download,
    History, ArrowRight, Eraser, Image as ImageIcon,
    Maximize2, Undo2, Save, Sparkles, ChevronRight,
    Trash2, Heart, Copy, Sliders, Key, Cpu, Layers,
    Play, CheckCircle2
} from 'lucide-react';
import { editImage, getPreferredModels } from '../services/geminiService';
import { authService } from '../services/authService';
import { dbService, Asset } from '../services/dbService';
import { ModelType } from '../types';
import { downloadItems } from '../utils/downloadUtils';
import { RateLimitPopup } from './RateLimitPopup';

const EDIT_MODELS = [
    { id: ModelType.FLASH_IMAGE_2_5, name: 'Gemini 2.5 Flash Image (Balanced)' },
    { id: ModelType.PRO_IMAGE_3_0, name: 'Gemini 3 Pro Image (Precise)' },
    { id: ModelType.FLASH_IMAGE_2_0, name: 'Gemini 2.0 Flash Image (Fast)' }
];

const ImageEditor: React.FC = () => {
    const [activeTab, setActiveTab] = useState<'editor' | 'bulk' | 'history'>('editor');
    const [hasApiKey, setHasApiKey] = useState(false);
    const [isCheckingKey, setIsCheckingKey] = useState(true);
    const [rateLimitError, setRateLimitError] = useState(false); // Popup state

    // Editor State
    const [file, setFile] = useState<File | null>(null);
    const [preview, setPreview] = useState<string | null>(null);
    const [result, setResult] = useState<string | null>(null);
    const [prompt, setPrompt] = useState('');
    const [strength, setStrength] = useState(0.6);
    const [loading, setLoading] = useState(false);
    const [selectedModel, setSelectedModel] = useState(ModelType.FLASH_IMAGE_2_5);
    const [isSaved, setIsSaved] = useState(false);

    // Bulk State
    const [bulkFiles, setBulkFiles] = useState<File[]>([]);
    const [bulkPreviews, setBulkPreviews] = useState<{ id: string, src: string }[]>([]);
    const [bulkResults, setBulkResults] = useState<{ id: string, original: string, edited: string, status: 'pending' | 'processing' | 'complete' | 'error' }[]>([]);
    const [isBulkProcessing, setIsBulkProcessing] = useState(false);

    // Active Models (Enabled in Settings)
    const [activeModels, setActiveModels] = useState<string[]>([]);

    // Comparison Slider State
    const [comparePos, setComparePos] = useState(50);
    const compareRef = useRef<HTMLDivElement>(null);

    // Session History (Single)
    const [sessionHistory, setSessionHistory] = useState<string[]>([]);

    // Global History Tab
    const [history, setHistory] = useState<Asset[]>([]);

    useEffect(() => {
        checkApiKey();
        updateActiveModels();
        const handleFocus = () => {
            checkApiKey();
            updateActiveModels();
        }
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    useEffect(() => {
        if (activeTab === 'history') {
            dbService.getByType('image').then(assets => {
                setHistory(assets.filter(a => a.metadata?.isEdit));
            });
        }
    }, [activeTab]);

    const updateActiveModels = () => {
        const models = getPreferredModels('creative');
        setActiveModels(models);
    };

    const checkApiKey = async () => {
        const key = await getUserApiKey();
        setHasApiKey(!!key);
        setIsCheckingKey(false);
    };

    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    const handleUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            const f = e.target.files[0];
            setFile(f);
            const reader = new FileReader();
            reader.onload = (ev) => {
                setPreview(ev.target?.result as string);
                setResult(null);
                setSessionHistory([]);
            };
            reader.readAsDataURL(f);
        }
    };

    const handleBulkUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files) {
            const newFiles = Array.from(e.target.files).slice(0, 10) as File[]; // Limit 10
            setBulkFiles(newFiles);

            const newPreviews: any[] = [];
            const newResults: any[] = [];

            newFiles.forEach(f => {
                const id = Math.random().toString(36).substring(7);
                const reader = new FileReader();
                reader.onload = (ev) => {
                    setBulkPreviews(prev => [...prev, { id, src: ev.target?.result as string }]);
                };
                reader.readAsDataURL(f);
                newResults.push({ id, original: '', edited: '', status: 'pending' });
            });

            setBulkResults(newResults);
        }
    };

    const handleEdit = async () => {
        const currentUser = authService.getCurrentUser();
        if (!preview || !prompt || !currentUser) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        // Clear history before editing to keep "last batch" logic consistent,
        // even for a single edit.
        await dbService.clearByType('image');

        setLoading(true);
        try {
            const base64 = preview.split(',')[1];
            const editedDataUrl = await editImage(apiKeys, base64, prompt, strength, selectedModel);
            setResult(editedDataUrl);
            setSessionHistory(prev => [...prev, editedDataUrl]);

            // Save to Library automatically
            const asset: Asset = {
                id: Math.random().toString(36).substring(2, 10),
                type: 'image',
                url: editedDataUrl,
                prompt: `Edited: ${prompt}`,
                createdAt: Date.now(),
                isFavorite: false,
                isDeleted: false,
                metadata: { isEdit: true, model: selectedModel, strength }
            };
            await dbService.add(asset);

            if (currentUser) {
                authService.incrementUserStat('imagesGenerated', 1); // Count as generation
                authService.incrementUserStat('imagesEdited', 1); // Track edits
                authService.incrementUserStat('modelUsage', 1, selectedModel);
                if (apiKeys[0]) {
                    // Costs 1 credit per edit
                    authService.deductCredits(currentUser.uid, 1, apiKeys[0], selectedModel);
                }
                authService.logActivity(currentUser.uid, 'generate', `Edited image: "${prompt.substring(0, 30)}..."`);
            }
        } catch (e: any) {
            if (e.name === 'RateLimitExhaustedError' || e.message?.includes('RateLimitExhaustedError')) {
                setRateLimitError(true);
            } else {
                alert("Editing failed.");
            }
        } finally {
            setLoading(false);
        }
    }

    const handleSaveToLibrary = async () => {
        if (!result) return;
        // Result is already saved to DB in handleEdit, but we can animate the save button
        // Or if we want to support saving duplicates/manual saves if auto-save failed?
        // For now, just show the success indicator as feedback for the user intention
        setIsSaved(true);
        setTimeout(() => setIsSaved(false), 2000);
        // Optional: Trigger a real save if needed, but handleEdit already saved it. 
        // We could just re-verify or do nothing but feedback.
    };

    const processBulk = async () => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser || !prompt) return;

        // Fetch API Keys
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        let apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (apiKeys.length === 0) return;

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
            apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }

        // Clear history before processing batch
        await dbService.clearByType('image');

        setIsBulkProcessing(true);

        for (let i = 0; i < bulkResults.length; i++) {
            const item = bulkResults[i];
            const previewItem = bulkPreviews.find(p => p.id === item.id);
            if (!previewItem) continue;

            setBulkResults(prev => prev.map(r => r.id === item.id ? { ...r, status: 'processing', original: previewItem.src } : r));

            try {
                const base64 = previewItem.src.split(',')[1];
                const editedUrl = await editImage(apiKeys, base64, prompt, strength, selectedModel);

                // Save to DB
                const asset: Asset = {
                    id: Math.random().toString(36).substring(2, 10),
                    type: 'image',
                    url: editedUrl,
                    prompt: `Bulk Edit: ${prompt}`,
                    createdAt: Date.now(),
                    isFavorite: false,
                    isDeleted: false,
                    metadata: { isEdit: true, model: selectedModel, batch: true }
                };
                await dbService.add(asset);

                setBulkResults(prev => prev.map(r => r.id === item.id ? { ...r, status: 'complete', edited: editedUrl } : r));

                if (currentUser) {
                    authService.incrementUserStat('imagesGenerated', 1);
                    authService.incrementUserStat('imagesEdited', 1); // Track bulk edits
                    authService.incrementUserStat('modelUsage', 1, selectedModel);
                    if (apiKeys[0]) {
                        await authService.deductCredits(currentUser.uid, 1, apiKeys[0], selectedModel);
                    }
                    authService.logActivity(currentUser.uid, 'generate', `Batch edited image: "${prompt.substring(0, 30)}..."`);
                }

            } catch (e) {
                setBulkResults(prev => prev.map(r => r.id === item.id ? { ...r, status: 'error' } : r));
            }
        }
        setIsBulkProcessing(false);
    };

    const handleMouseMove = (e: React.MouseEvent) => {
        if (compareRef.current && result) {
            const rect = compareRef.current.getBoundingClientRect();
            const x = Math.max(0, Math.min(e.clientX - rect.left, rect.width));
            setComparePos((x / rect.width) * 100);
        }
    }

    const suggestions = [
        "Remove background", "Make it sunny", "Add a vintage filter",
        "Turn into a sketch", "Cyberpunk style", "Add snow"
    ];

    const handleDownloadAllHistory = () => {
        downloadItems(history.map(item => ({
            url: item.url,
            filename: `edited_image_${item.id}.png`
        })), 'edited_history');
    };

    const handleDownloadBatch = () => {
        const completedItems = bulkResults.filter(item => item.status === 'complete');
        downloadItems(completedItems.map(item => ({
            url: item.edited,
            filename: `batch_edit_${item.id}.png`
        })), 'batch_edits');
    };

    return (
        <div className="max-w-[1800px] mx-auto p-4 lg:p-8 h-[calc(100vh-4rem)] flex flex-col font-sans text-slate-900 dark:text-slate-200 animate-fade-in">

            {/* NEW HEADER */}
            <div className="mb-6 shrink-0">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 flex items-center gap-3 font-['Space_Grotesk']">
                    <PencilRuler className="text-emerald-500 w-8 h-8" /> Magic Image Editor
                </h1>
                <p className="text-slate-600 dark:text-slate-400 mt-2">Edit, repair, and transform images using natural language instructions.</p>
            </div>

            {!hasApiKey && !isCheckingKey && (
                <div className="bg-amber-500/10 border border-amber-500/20 rounded-xl p-4 mb-6 flex flex-col sm:flex-row items-center justify-between gap-4 animate-fade-in shrink-0">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-amber-500/20 rounded-lg text-amber-600 dark:text-amber-500 shrink-0">
                            <Key size={20} />
                        </div>
                        <div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-sm">User API Key Required</h4>
                            <p className="text-xs text-slate-600 dark:text-slate-400">This feature requires your own Gemini API key. Please add it in Settings.</p>
                        </div>
                    </div>
                    <button
                        onClick={() => window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }))}
                        className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition whitespace-nowrap shadow-lg shadow-amber-500/20"
                    >
                        Add API Key
                    </button>
                </div>
            )}

            {/* Tabs */}
            <div className="flex flex-col md:flex-row items-center justify-between border-b border-slate-200 dark:border-slate-800 mb-6 shrink-0">
                <div className="flex w-full md:w-auto overflow-x-auto">
                    <button
                        onClick={() => setActiveTab('editor')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'editor' ? 'text-purple-600 dark:text-purple-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}
                    >
                        <PencilRuler size={16} /> Editor
                        {activeTab === 'editor' && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-purple-600 dark:bg-purple-400 animate-fade-in"></div>}
                    </button>
                    <button
                        onClick={() => setActiveTab('bulk')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'bulk' ? 'text-purple-600 dark:text-purple-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}
                    >
                        <Layers size={16} /> Bulk
                        {activeTab === 'bulk' && <div className="absolute bottom-0 left-0 w-full h-0.5 bg-purple-600 dark:bg-purple-400 animate-fade-in"></div>}
                    </button>
                    <button
                        onClick={() => setActiveTab('history')}
                        className={`px-6 py-3 text-sm font-bold transition-all relative flex items-center gap-2 ${activeTab === 'history' ? 'text-purple-600 dark:text-purple-400' : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'}`}
                    >
                        <History size={16} /> History
                    </button>
                </div>

                {/* Active Models Display */}
                {activeTab !== 'history' && (
                    <div className="hidden md:flex items-center gap-2 px-3 py-1.5 bg-emerald-50 dark:bg-emerald-900/10 border border-emerald-100 dark:border-emerald-500/20 rounded-lg my-2 mr-2">
                        <span className="text-[10px] font-bold text-emerald-700 dark:text-emerald-400 uppercase flex items-center gap-1.5">
                            <Cpu size={12} /> Active Model{activeModels.length > 1 ? 's' : ''}:
                        </span>
                        <div className="flex gap-1">
                            {activeModels.map((model, idx) => (
                                <span key={idx} className="px-2 py-0.5 bg-white dark:bg-emerald-900/20 text-emerald-700 dark:text-emerald-300 border border-emerald-200 dark:border-emerald-800 rounded text-[10px] font-mono font-bold shadow-sm">
                                    {model}
                                </span>
                            ))}
                            {activeModels.length === 0 && <span className="text-[10px] text-emerald-600 italic">Default</span>}
                        </div>
                    </div>
                )}
            </div>

            <div className={`flex-1 overflow-hidden transition-opacity ${!hasApiKey ? 'opacity-50 pointer-events-none' : ''}`}>
                {activeTab === 'editor' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20 custom-scrollbar">
                        {/* LEFT: CONTROLS */}
                        <div className="lg:col-span-4 space-y-6 h-fit">
                            {/* Upload */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Source Image</label>
                                <div
                                    onClick={() => document.getElementById('edit-upload')?.click()}
                                    className="relative border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl h-48 flex flex-col items-center justify-center cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-900/50 transition overflow-hidden group bg-slate-50 dark:bg-slate-950"
                                >
                                    {preview ? (
                                        <>
                                            <img src={preview} className="w-full h-full object-contain opacity-50 group-hover:opacity-30 transition" />
                                            <div className="absolute inset-0 flex items-center justify-center">
                                                <span className="bg-white dark:bg-slate-900 px-3 py-1 rounded-full text-xs font-bold text-slate-900 dark:text-white border border-slate-200 dark:border-slate-700 shadow-xl">Replace Image</span>
                                            </div>
                                            <div className="absolute bottom-2 left-2 text-[10px] text-slate-500 bg-white/90 dark:bg-slate-950/80 px-2 py-1 rounded backdrop-blur">
                                                {file?.name} ({(file?.size || 0 / 1024 / 1024).toFixed(1)} MB)
                                            </div>
                                        </>
                                    ) : (
                                        <>
                                            <Upload className="text-slate-400 dark:text-slate-500 mb-2 w-8 h-8 group-hover:text-purple-600 dark:group-hover:text-purple-400 transition-colors" />
                                            <span className="text-sm font-bold text-slate-500 dark:text-slate-400">Upload or Drop Image</span>
                                            <span className="text-[10px] text-slate-400 dark:text-slate-600 mt-1">JPG, PNG, WebP (Max 10MB)</span>
                                        </>
                                    )}
                                    <input id="edit-upload" type="file" className="hidden" onChange={handleUpload} accept="image/*" />
                                </div>
                            </div>

                            {/* Instructions */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <div className="flex justify-between items-center">
                                    <label className="text-sm font-bold text-slate-700 dark:text-slate-300">What would you like to change?</label>
                                    <span className="text-xs text-slate-500">{prompt.length}/2000</span>
                                </div>
                                <textarea
                                    value={prompt} onChange={(e) => setPrompt(e.target.value)}
                                    className="w-full h-32 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 focus:border-purple-500 focus:ring-1 focus:ring-purple-500 outline-none resize-none placeholder:text-slate-400 dark:placeholder:text-slate-600 text-sm leading-relaxed"
                                    placeholder="Remove the background, change the sky to sunset, add snow..."
                                />

                                <div className="space-y-2">
                                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider">Quick Edits</label>
                                    <div className="flex flex-wrap gap-2">
                                        {suggestions.map(s => (
                                            <button key={s} onClick={() => setPrompt(s)} className="px-2 py-1 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 rounded-lg text-[10px] text-slate-600 dark:text-slate-300 transition border border-slate-200 dark:border-slate-700 flex items-center gap-1">
                                                <Sparkles size={10} className="text-purple-500" /> {s}
                                            </button>
                                        ))}
                                    </div>
                                </div>
                            </div>

                            {/* Strength & Model */}
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-6 shadow-sm dark:shadow-none">
                                <div>
                                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 flex items-center gap-2">
                                        <Cpu size={14} className="text-purple-500" /> Editing Model
                                    </label>
                                    <select value={selectedModel} onChange={(e) => setSelectedModel(e.target.value as ModelType)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-700 dark:text-slate-300 outline-none focus:border-purple-500">
                                        {EDIT_MODELS.map(m => (
                                            <option key={m.id} value={m.id}>{m.name}</option>
                                        ))}
                                    </select>
                                </div>

                                <div>
                                    <div className="flex justify-between text-xs text-slate-500 dark:text-slate-400 mb-2">
                                        <span className="uppercase font-bold">Edit Strength</span>
                                        <span className="text-purple-600 dark:text-purple-400 font-bold">{strength}</span>
                                    </div>
                                    <input
                                        type="range" min="0" max="1" step="0.1" value={strength} onChange={(e) => setStrength(Number(e.target.value))}
                                        className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-purple-500"
                                    />
                                    <div className="flex justify-between text-[10px] text-slate-500 dark:text-slate-600 mt-1">
                                        <span>Subtle (0.1)</span><span>Balanced (0.5)</span><span>Strong (0.9)</span>
                                    </div>
                                </div>
                            </div>

                            <button
                                onClick={handleEdit}
                                disabled={loading || !preview || !prompt || !hasApiKey}
                                className="w-full py-4 bg-purple-600 hover:bg-purple-500 text-white font-bold rounded-xl transition-all shadow-lg shadow-purple-600/20 flex items-center justify-center gap-2 disabled:opacity-50"
                            >
                                {loading ? <RefreshCw className="animate-spin" /> : <PencilRuler />} Apply Edit ✨
                                {!loading && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">1 API Call</span>}
                            </button>
                        </div>

                        {/* RIGHT: COMPARISON VIEWER */}
                        <div className="lg:col-span-8 flex flex-col gap-4">
                            <div className="bg-slate-100 dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 relative flex items-center justify-center overflow-hidden flex-1 min-h-[500px] shadow-inner dark:shadow-2xl">
                                {!preview && (
                                    <div className="text-center text-slate-500 dark:text-slate-600">
                                        <div className="w-24 h-24 bg-white dark:bg-slate-900 rounded-full flex items-center justify-center mx-auto mb-6 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none">
                                            <Eraser size={40} className="opacity-50" />
                                        </div>
                                        <h3 className="text-xl font-bold text-slate-700 dark:text-slate-500 mb-2">Magic Editor Canvas</h3>
                                        <p className="text-sm text-slate-600 dark:text-slate-500">Upload an image to start editing</p>
                                    </div>
                                )}

                                {preview && !result && (
                                    <div className="relative w-full h-full flex items-center justify-center bg-black/5 dark:bg-black/20">
                                        <img src={preview} className="max-w-full max-h-full object-contain" />
                                        {loading && (
                                            <div className="absolute inset-0 bg-white/50 dark:bg-black/50 backdrop-blur-sm flex flex-col items-center justify-center">
                                                <div className="w-16 h-16 border-4 border-purple-500 border-t-transparent rounded-full animate-spin mb-4"></div>
                                                <p className="text-purple-600 dark:text-purple-400 font-bold animate-pulse">Applying Magic...</p>
                                            </div>
                                        )}
                                    </div>
                                )}

                                {result && preview && (
                                    <div
                                        ref={compareRef}
                                        onMouseMove={handleMouseMove}
                                        className="relative w-full h-full cursor-ew-resize group select-none"
                                    >
                                        {/* Background Image (Edited - After) */}
                                        <div
                                            className="absolute inset-0 w-full h-full bg-contain bg-center bg-no-repeat"
                                            style={{ backgroundImage: `url(${result})` }}
                                        />

                                        {/* Foreground Image (Original - Before) - Clip Path */}
                                        <div
                                            className="absolute inset-0 w-full h-full bg-contain bg-center bg-no-repeat border-r-2 border-white/50 shadow-[10px_0_20px_rgba(0,0,0,0.5)]"
                                            style={{
                                                backgroundImage: `url(${preview})`,
                                                width: `${comparePos}%`,
                                                transition: 'none'
                                            }}
                                        >
                                            <div className="absolute top-4 left-4 bg-black/60 text-white text-xs font-bold px-2 py-1 rounded backdrop-blur">BEFORE</div>
                                        </div>

                                        {/* Right Label */}
                                        <div className="absolute top-4 right-4 bg-purple-600/80 text-white text-xs font-bold px-2 py-1 rounded backdrop-blur pointer-events-none">AFTER</div>

                                        {/* Slider Handle */}
                                        <div
                                            className="absolute top-0 bottom-0 w-1 bg-white shadow-[0_0_10px_rgba(0,0,0,0.5)] pointer-events-none"
                                            style={{ left: `${comparePos}%` }}
                                        >
                                            <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-8 h-8 bg-white rounded-full shadow-lg flex items-center justify-center text-slate-900 font-bold text-xs border-2 border-purple-500">
                                                <Sliders size={14} className="text-purple-600" />
                                            </div>
                                        </div>

                                        {/* Actions Overlay */}
                                        <div className="absolute bottom-6 left-1/2 -translate-x-1/2 flex gap-4 opacity-0 group-hover:opacity-100 transition-all duration-300 translate-y-4 group-hover:translate-y-0">
                                            <a href={result} download="edited_image.png" className="px-6 py-3 bg-white text-slate-900 font-bold rounded-xl shadow-xl hover:bg-slate-200 flex items-center gap-2"><Download size={18} /> Download Result</a>
                                            <button
                                                onClick={handleSaveToLibrary}
                                                className="px-6 py-3 bg-slate-900/90 text-white font-bold rounded-xl shadow-xl border border-slate-700 hover:bg-slate-800 flex items-center gap-2"
                                            >
                                                {isSaved ? <CheckCircle2 size={18} className="text-green-400" /> : <Save size={18} />}
                                                {isSaved ? <span className="text-green-400">Saved!</span> : 'Save to Library'}
                                            </button>
                                        </div>
                                    </div>
                                )}
                            </div>

                            {/* Session History Strip */}
                            {sessionHistory.length > 0 && (
                                <div className="h-24 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-2 flex gap-2 overflow-x-auto custom-scrollbar shadow-sm dark:shadow-none">
                                    {sessionHistory.map((url, i) => (
                                        <div key={i} className="relative aspect-square rounded-lg overflow-hidden cursor-pointer border border-slate-200 dark:border-slate-700 hover:border-purple-500 transition" onClick={() => setResult(url)}>
                                            <img src={url} className="w-full h-full object-cover" />
                                            <span className="absolute bottom-0 left-0 w-full bg-black/60 text-white text-[8px] text-center py-0.5">v{i + 1}</span>
                                        </div>
                                    ))}
                                    <button onClick={() => { setResult(null); setSessionHistory([]); }} className="w-20 flex flex-col items-center justify-center text-xs text-slate-500 border border-dashed border-slate-300 dark:border-slate-700 rounded-lg hover:bg-slate-100 dark:hover:bg-slate-800 hover:text-slate-900 dark:hover:text-white transition">
                                        <RefreshCw size={14} className="mb-1" /> Reset
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>
                )}

                {activeTab === 'bulk' && (
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 h-full overflow-y-auto pb-20 custom-scrollbar">
                        {/* LEFT: BULK CONFIG */}
                        <div className="lg:col-span-4 space-y-6 h-fit">
                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-3 block">Batch Upload (Max 10)</label>
                                <div
                                    onClick={() => document.getElementById('bulk-upload')?.click()}
                                    className="relative border-2 border-dashed border-slate-300 dark:border-slate-700 rounded-xl h-48 flex flex-col items-center justify-center cursor-pointer hover:bg-slate-50 dark:hover:bg-slate-900/50 transition overflow-hidden group bg-slate-50 dark:bg-slate-950"
                                >
                                    <Layers className="text-slate-400 dark:text-slate-500 mb-2 w-8 h-8 group-hover:text-purple-600 dark:group-hover:text-purple-400 transition-colors" />
                                    <span className="text-sm font-bold text-slate-500 dark:text-slate-400">Upload Multiple</span>
                                    <span className="text-[10px] text-slate-400 dark:text-slate-600 mt-1">{bulkFiles.length} files selected</span>
                                    <input id="bulk-upload" type="file" className="hidden" onChange={handleBulkUpload} multiple accept="image/*" />
                                </div>
                            </div>

                            <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-5 space-y-4 shadow-sm dark:shadow-none">
                                <label className="text-sm font-bold text-slate-700 dark:text-slate-300">Apply to All</label>
                                <textarea
                                    value={prompt} onChange={(e) => setPrompt(e.target.value)}
                                    className="w-full h-24 bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-900 dark:text-slate-200 focus:border-purple-500 focus:ring-1 focus:ring-purple-500 outline-none resize-none text-sm"
                                    placeholder="Global edit instruction..."
                                />
                                <div>
                                    <label className="text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wider mb-2 block">Strength: {strength}</label>
                                    <input
                                        type="range" min="0" max="1" step="0.1" value={strength} onChange={(e) => setStrength(Number(e.target.value))}
                                        className="w-full h-1.5 bg-slate-200 dark:bg-slate-700 rounded-lg appearance-none cursor-pointer accent-purple-500"
                                    />
                                </div>
                            </div>

                            <button
                                onClick={processBulk}
                                disabled={isBulkProcessing || bulkFiles.length === 0 || !prompt || !hasApiKey}
                                className="w-full py-4 bg-purple-600 hover:bg-purple-500 text-white font-bold rounded-xl transition-all flex items-center justify-center gap-2 disabled:opacity-50"
                            >
                                {isBulkProcessing ? <RefreshCw className="animate-spin" /> : <Play />} Process Batch
                                {!isBulkProcessing && <span className="ml-2 px-2 py-0.5 bg-black/20 dark:bg-white/20 rounded text-[10px] font-bold opacity-90">{bulkFiles.length} API Calls</span>}
                            </button>
                        </div>

                        {/* RIGHT: BULK RESULTS */}
                        <div className="lg:col-span-8">
                            <div className="bg-slate-100 dark:bg-slate-950 rounded-2xl border border-slate-200 dark:border-slate-800 p-6 min-h-[500px] shadow-inner">
                                {bulkResults.some(r => r.status === 'complete') && (
                                    <div className="flex justify-end mb-4">
                                        <button onClick={handleDownloadBatch} className="text-xs bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 px-3 py-1.5 rounded-lg font-bold flex items-center gap-2 transition">
                                            <Download size={14} /> Download All Completed
                                        </button>
                                    </div>
                                )}
                                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4">
                                    {bulkResults.map((item) => (
                                        <div key={item.id} className="relative aspect-square bg-white dark:bg-slate-900 rounded-xl border border-slate-200 dark:border-slate-800 overflow-hidden group">
                                            {item.status === 'complete' ? (
                                                <>
                                                    <img src={item.edited} className="w-full h-full object-cover" />
                                                    <div className="absolute bottom-2 right-2 flex gap-2">
                                                        <a href={item.edited} download className="p-1.5 bg-white/90 text-black rounded-lg shadow-lg"><Download size={14} /></a>
                                                    </div>
                                                </>
                                            ) : item.status === 'processing' ? (
                                                <div className="w-full h-full flex flex-col items-center justify-center">
                                                    <RefreshCw className="animate-spin text-purple-500 mb-2" />
                                                    <span className="text-xs text-slate-500">Processing...</span>
                                                </div>
                                            ) : (
                                                <div className="w-full h-full flex items-center justify-center relative">
                                                    {bulkPreviews.find(p => p.id === item.id) && (
                                                        <img src={bulkPreviews.find(p => p.id === item.id)?.src} className="w-full h-full object-cover opacity-50" />
                                                    )}
                                                    <span className="absolute bg-black/50 text-white text-xs px-2 py-1 rounded">Pending</span>
                                                </div>
                                            )}
                                        </div>
                                    ))}
                                    {bulkResults.length === 0 && (
                                        <div className="col-span-full flex flex-col items-center justify-center text-slate-500 h-64">
                                            <Layers size={32} className="mb-2 opacity-50" />
                                            <p>Bulk queue is empty</p>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {activeTab === 'history' && (
                    <div className="h-full overflow-y-auto pb-20 custom-scrollbar">
                        <div className="flex justify-between items-center mb-4 px-2">
                            <h3 className="font-bold text-slate-700 dark:text-slate-300">Edit History ({history.length})</h3>
                            {history.length > 0 && (
                                <button onClick={handleDownloadAllHistory} className="flex items-center gap-2 px-3 py-1.5 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 text-xs font-bold rounded-lg transition shadow-lg shadow-amber-500/20">
                                    <Download size={14} /> Download All
                                </button>
                            )}
                        </div>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
                            {history.map((item) => (
                                <div key={item.id} className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl overflow-hidden group shadow-sm dark:shadow-none cursor-pointer" onClick={() => { setPreview(item.url); setResult(null); setActiveTab('editor'); }}>
                                    <div className="aspect-square bg-slate-100 dark:bg-slate-950 relative">
                                        <img src={item.url} className="w-full h-full object-cover" />
                                        <div className="absolute inset-0 flex items-center justify-center bg-black/0 group-hover:bg-black/40 transition">
                                            <span className="opacity-0 group-hover:opacity-100 text-white font-bold text-sm border border-white px-3 py-1 rounded-lg">Edit Again</span>
                                        </div>
                                    </div>
                                    <div className="p-3">
                                        <div className="flex justify-between items-center mb-1">
                                            <span className="text-[10px] text-slate-500">{new Date(item.createdAt).toLocaleDateString()}</span>
                                            <span className="text-[10px] bg-purple-100 dark:bg-purple-500/10 text-purple-600 dark:text-purple-400 px-1.5 py-0.5 rounded font-bold">Edited</span>
                                        </div>
                                        <p className="text-xs text-slate-700 dark:text-slate-300 truncate">{item.prompt}</p>
                                    </div>
                                </div>
                            ))}
                            {history.length === 0 && (
                                <div className="col-span-full text-center py-12 text-slate-500">
                                    <History size={32} className="mx-auto mb-2 opacity-30" />
                                    No editing history found.
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </div>
            {/* Rate Limit Popup */}
            <RateLimitPopup
                isOpen={rateLimitError}
                onClose={() => setRateLimitError(false)}
            />
        </div>
    )
};

export default ImageEditor;
