import React, { useState, useEffect } from 'react';
import { BrainCircuit, Sparkles, Clock, CheckCircle2, Loader2, Cpu } from 'lucide-react';

interface EnhancedLoadingOverlayProps {
    isVisible: boolean;
    title?: string;
    description?: string;
    stage?: string;
    progress?: number; // 0-100
    eta?: number; // seconds
    onCancel?: () => void;
    simulate?: boolean; // If true, auto-increments progress
    apiKey?: string;
    model?: string;
}

export const EnhancedLoadingOverlay: React.FC<EnhancedLoadingOverlayProps> = ({
    isVisible,
    title = "Generating Content",
    description = "AI is processing your request...",
    stage: initialStage = "Initializing...",
    progress: initialProgress = 0,
    eta: initialEta = 15,
    onCancel,
    simulate = false,
    apiKey,
    model
}) => {
    const [progress, setProgress] = useState(initialProgress);
    const [eta, setEta] = useState(initialEta);
    const [stage, setStage] = useState(initialStage);

    useEffect(() => {
        if (!isVisible) {
            setProgress(0);
            return;
        }

        if (simulate) {
            setProgress(0);
            setEta(initialEta);

            // Total ticks based on initialEta (assuming 100ms interval for smoothness)
            const totalTicks = initialEta * 10;
            let currentTick = 0;

            const interval = setInterval(() => {
                currentTick++;

                // Calculate progress based on time elapsed
                // Use a slightly non-linear curve: fast start, slower end
                const progressRatio = currentTick / totalTicks;
                let calculatedProgress = 0;

                if (progressRatio < 0.5) {
                    // First half: go to 60%
                    calculatedProgress = progressRatio * 2 * 60;
                } else {
                    // Second half: go from 60% to 95%
                    calculatedProgress = 60 + (progressRatio - 0.5) * 2 * 35;
                }

                // Cap at 95% until complete
                if (calculatedProgress > 95) calculatedProgress = 95;

                setProgress(calculatedProgress);

                // Update ETA every second (every 10 ticks)
                if (currentTick % 10 === 0) {
                    setEta(prev => prev > 1 ? prev - 1 : 1);
                }
            }, 100);

            // Simulate stage changes based on progress
            const stageInterval = setInterval(() => {
                setProgress(prev => {
                    if (prev < 30) setStage("Analyzing context...");
                    else if (prev < 60) setStage("Forging content...");
                    else if (prev < 90) setStage("Finalizing details...");
                    return prev;
                });
            }, 500);

            return () => {
                clearInterval(interval);
                clearInterval(stageInterval);
            };
        } else {
            setProgress(initialProgress);
            setEta(initialEta);
            setStage(initialStage);
        }
    }, [isVisible, simulate, initialEta, initialStage, initialProgress]);

    if (!isVisible) return null;

    // Helper to mask API key
    const maskedKey = apiKey ? `${apiKey.substring(0, 4)}...${apiKey.substring(apiKey.length - 4)}` : 'Using System Key';

    return (
        <div className="fixed inset-0 z-50 bg-black/80 backdrop-blur-sm flex flex-col items-center justify-center p-4 animate-fade-in">
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 p-8 rounded-3xl max-w-md w-full text-center shadow-2xl relative overflow-hidden">

                {/* Background Glow */}
                <div className="absolute top-0 left-1/2 -translate-x-1/2 w-full h-32 bg-amber-500/10 blur-3xl rounded-full pointer-events-none"></div>

                {/* Icon Animation */}
                <div className="w-20 h-20 mx-auto mb-6 relative z-10">
                    <div className="absolute inset-0 border-4 border-slate-100 dark:border-slate-800 rounded-full"></div>
                    <div
                        className="absolute inset-0 border-4 border-amber-500 rounded-full border-t-transparent animate-spin"
                        style={{ animationDuration: '1.5s' }}
                    ></div>
                    <div className="absolute inset-0 flex items-center justify-center">
                        <BrainCircuit className="text-amber-500 animate-pulse" size={32} />
                    </div>
                </div>

                {/* Text Content */}
                <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-2 font-['Space_Grotesk'] relative z-10">{title}</h3>
                <p className="text-slate-500 dark:text-slate-400 mb-6 relative z-10 text-sm">
                    {description}
                </p>

                {/* API & Model Info */}
                {(apiKey || model) && (
                    <div className="flex justify-center gap-3 mb-6 relative z-10">
                        {apiKey && (
                            <div className="px-2 py-1 bg-slate-100 dark:bg-slate-800 rounded text-[10px] font-mono text-slate-500 dark:text-slate-400 border border-slate-200 dark:border-slate-700 flex items-center gap-1">
                                <span className="w-1.5 h-1.5 rounded-full bg-green-500"></span>
                                Key: {maskedKey}
                            </div>
                        )}
                        {model && (
                            <div className="px-2 py-1 bg-slate-100 dark:bg-slate-800 rounded text-[10px] font-mono text-slate-500 dark:text-slate-400 border border-slate-200 dark:border-slate-700 flex items-center gap-1">
                                <Cpu size={10} />
                                {model}
                            </div>
                        )}
                    </div>
                )}

                {/* Progress Bar */}
                <div className="space-y-3 mb-8 relative z-10">
                    <div className="flex justify-between text-xs font-bold text-slate-500 dark:text-slate-400 uppercase mb-1">
                        <span>{stage}</span>
                        <span>{Math.round(progress)}%</span>
                    </div>
                    <div className="h-3 bg-slate-100 dark:bg-slate-800 rounded-full overflow-hidden border border-slate-200 dark:border-slate-700">
                        <div
                            className="h-full bg-gradient-to-r from-amber-500 to-orange-500 transition-all duration-300 ease-out relative"
                            style={{ width: `${progress}%` }}
                        >
                            <div className="absolute inset-0 bg-white/20 animate-pulse"></div>
                        </div>
                    </div>
                    <div className="flex justify-end items-center gap-1.5 text-xs font-medium text-slate-400">
                        <Clock size={12} />
                        <span>~{eta}s remaining</span>
                    </div>
                </div>

                {/* Cancel Button */}
                {onCancel && (
                    <button
                        onClick={onCancel}
                        className="text-slate-400 hover:text-slate-600 dark:hover:text-slate-200 text-sm font-bold transition relative z-10"
                    >
                        Cancel
                    </button>
                )}
            </div>
        </div>
    );
};
