import React, { useState, useEffect } from 'react';
import { Trash2, Loader2 } from 'lucide-react';

interface DoubleConfirmDeleteButtonProps {
    onDelete: () => void;
    className?: string;
    iconSize?: number;
    title?: string;
    label?: string; // Optional label for buttons with text
    children?: React.ReactNode; // Allow custom children
    isLoading?: boolean;
}

export const DoubleConfirmDeleteButton: React.FC<DoubleConfirmDeleteButtonProps> = ({
    onDelete,
    className = "",
    iconSize = 16,
    title = "Delete",
    label,
    children,
    isLoading = false
}) => {
    const [confirming, setConfirming] = useState(false);
    const [countdown, setCountdown] = useState(3);

    useEffect(() => {
        let timer: NodeJS.Timeout;
        if (confirming) {
            setCountdown(3);
            timer = setInterval(() => {
                setCountdown(prev => {
                    if (prev <= 1) {
                        clearInterval(timer);
                        setConfirming(false);
                        return 3;
                    }
                    return prev - 1;
                });
            }, 1000);
        }
        return () => clearInterval(timer);
    }, [confirming]);

    const handleClick = (e: React.MouseEvent) => {
        e.stopPropagation();
        if (isLoading) return;

        if (confirming) {
            onDelete();
            setConfirming(false);
        } else {
            setConfirming(true);
        }
    };

    return (
        <button
            onClick={handleClick}
            disabled={isLoading}
            className={`transition-all duration-200 flex items-center justify-center gap-1 ${className} ${confirming ? 'bg-red-600 text-white hover:bg-red-700 border-red-600' : ''} ${isLoading ? 'opacity-70 cursor-wait' : ''}`}
            title={confirming ? "Click again to confirm" : title}
        >
            {isLoading ? (
                <>
                    <Loader2 size={iconSize} className="animate-spin" />
                    <span className="font-bold whitespace-nowrap">Processing...</span>
                </>
            ) : children ? (
                confirming ? (
                    <>
                        <Trash2 size={iconSize} />
                        <span className="font-bold whitespace-nowrap">Confirm ({countdown})</span>
                    </>
                ) : (
                    children
                )
            ) : (
                <>
                    <Trash2 size={iconSize} />
                    {confirming && <span className="text-xs font-bold whitespace-nowrap">Confirm ({countdown})</span>}
                    {!confirming && label && <span>{label}</span>}
                </>
            )}
        </button>
    );
};
