import React, { useState, useEffect } from 'react';
import { AppView, User, Tip, ActivityLog, Broadcast, SidebarItemSetting } from '../types';
import { UserDatabase } from '../services/USER-database';
import { MODELS, MODEL_TYPE_TO_SETTINGS } from '../constants';
import {
    ArrowUpRight, ArrowDownRight, Image as ImageIcon, Video,
    Zap, Camera, Wand2, Sparkles, Play, MoreHorizontal,
    Clock, Settings, Key, MessageSquare, Lightbulb, ExternalLink,
    ChevronLeft, ChevronRight, ChevronUp, Download, Trash2, Share2, Crown,
    TrendingUp, Globe, MapPin, X, ScanEye, Minus, Calendar, BrainCircuit,
    AlertCircle, Activity, Loader2, Database, FileText, Layers, Palette, ThumbsDown, Megaphone, SearchCheck,
    User as UserIcon, Lock, Shield
} from 'lucide-react';
import {
    LineChart, Line, AreaChart, Area, ResponsiveContainer, Tooltip, XAxis, YAxis, CartesianGrid
} from 'recharts';
import { authService } from '../services/authService';
import { adminService } from '../services/adminService';
import { dbService, Asset } from '../services/dbService';
import { generateDashboardTrends, getPreferredModels } from '../services/geminiService';
import { TutorialWidget } from './TutorialWidget';

interface DashboardProps {
    setView: (view: AppView) => void;
    user: User | null;
    onOpenSettings?: (tab?: string) => void;
}

const formatTimeAgo = (timestamp: number) => {
    const seconds = Math.floor((Date.now() - timestamp) / 1000);
    if (seconds < 60) return 'Just now';
    const minutes = Math.floor(seconds / 60);
    if (minutes < 60) return `${minutes}m ago`;
    const hours = Math.floor(minutes / 60);
    if (hours < 24) return `${hours}h ago`;
    return `${Math.floor(hours / 24)}d ago`;
};

const handleDownload = async (url: string, type: string, prompt: string) => {
    try {
        // Determine file extension based on type
        let extension = '.png';
        if (type === 'video') extension = '.mp4';
        if (type === 'report') extension = '.pdf';
        if (type === 'image') extension = '.png';

        // Generate filename from prompt (sanitized) or use default
        const sanitizedPrompt = prompt ? prompt.substring(0, 30).replace(/[^a-z0-9]/gi, '_') : 'download';
        const filename = `${sanitizedPrompt}_${Date.now()}${extension}`;

        // Fetch the file and create blob
        const response = await fetch(url);
        const blob = await response.blob();

        // Create download link
        const link = document.createElement('a');
        link.href = window.URL.createObjectURL(blob);
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(link.href);
    } catch (error) {
        console.error('Download failed:', error);
        // Fallback to simple download
        const link = document.createElement('a');
        link.href = url;
        link.download = url.split('/').pop() || 'download';
        link.click();
    }
};

const Dashboard: React.FC<DashboardProps> = ({ setView, user, onOpenSettings }) => {
    const [greeting, setGreeting] = useState('');
    const [showUpgrade, setShowUpgrade] = useState(true);

    // Trend State
    const [activeTrendTab, setActiveTrendTab] = useState<'global' | 'regional'>('global');
    const [showRegionModal, setShowRegionModal] = useState(false);
    const [showKeyWarning, setShowKeyWarning] = useState(false);
    const [regionInput, setRegionInput] = useState('');
    const [currentRegion, setCurrentRegion] = useState(''); // The confirmed region

    const [perfMetric, setPerfMetric] = useState<'usage' | 'generated'>('usage');
    const [showAllActions, setShowAllActions] = useState(false);
    const [activities, setActivities] = useState<ActivityLog[]>([]);
    const [isActivityExpanded, setIsActivityExpanded] = useState(false);
    const [totalActivityCount, setTotalActivityCount] = useState(0);
    const [activityError, setActivityError] = useState<string | null>(null);
    const [recentAssets, setRecentAssets] = useState<Asset[]>([]);
    const [sidebarSettings, setSidebarSettings] = useState<{ [key: string]: SidebarItemSetting } | null>(null);

    // Trends & History State
    const [trends, setTrends] = useState({
        trendSearches: 0,
        metadata: 0,
        images: 0,
        imageEditor: 0,
        videos: 0,
        reviewerAi: 0,
        prompts: 0,
        api: 0
    });
    const [histories, setHistories] = useState({
        trendSearches: [0, 0, 0, 0, 0, 0, 0],
        metadata: [0, 0, 0, 0, 0, 0, 0],
        images: [0, 0, 0, 0, 0, 0, 0],
        imageEditor: [0, 0, 0, 0, 0, 0, 0],
        videos: [0, 0, 0, 0, 0, 0, 0],
        reviewerAi: [0, 0, 0, 0, 0, 0, 0],
        prompts: [0, 0, 0, 0, 0, 0, 0],
        api: [0, 0, 0, 0, 0, 0, 0]
    });

    // Time range for stats display (daily vs monthly)
    const [statTimeRange, setStatTimeRange] = useState<'daily' | 'monthly'>('daily');

    // Debug: Log histories when they change
    useEffect(() => {
        console.log('[Dashboard] Histories updated:', histories);
    }, [histories]);

    // Dynamic Trend Data
    const [trendData, setTrendData] = useState<any[]>([]);
    const [marketInsights, setMarketInsights] = useState<any>(null);
    const [isLoadingTrends, setIsLoadingTrends] = useState(true);
    const [trendLastUpdated, setTrendLastUpdated] = useState<number | null>(null);

    // API Call Tracking
    const [dailyApiCalls, setDailyApiCalls] = useState(0);
    const [enabledModelsLimit, setEnabledModelsLimit] = useState<number>(0);

    // Calculate credits from ONLY enabled models (matching TopBar logic)
    useEffect(() => {
        const fetchEnabledModelsLimit = async () => {
            if (!user) return;

            try {
                const siteSettings = await adminService.getSiteSettings();
                if (siteSettings?.sidebarSettings) {
                    setSidebarSettings(siteSettings.sidebarSettings);
                }
                const apiKeys = await authService.getUserApiKeys(user.uid);

                // DEFAULT TIER LIMITS (Must match Server/Admin)
                const DEFAULT_TIER_LIMITS: Record<string, { free: number, paid: number }> = {
                    'pro_3_0': { free: 50, paid: 500 },
                    'pro_2_5': { free: 50, paid: 500 },
                    'flash_2_5': { free: 300, paid: 1000 },
                    'flash_lite_2_5': { free: 0, paid: 1000 }, // Sync with TopBar
                    'imagen_ultra': { free: 10, paid: 500 },
                    'imagen_fast': { free: 10, paid: 500 },
                    'imagen_std': { free: 10, paid: 500 },
                    'gemini_3_pro_image': { free: 50, paid: 500 },
                    'gemini_2_5_flash_image': { free: 100, paid: 500 },
                    'veo_3_1': { free: 20, paid: 500 },
                    'veo_3_1_fast': { free: 50, paid: 500 },
                    'veo_3_0': { free: 10, paid: 500 },
                    'veo_2_0': { free: 5, paid: 500 },
                };

                let total = 0;
                let hasValidKeys = false;

                if (apiKeys && apiKeys.length > 0) {
                    apiKeys.forEach((key: any) => {
                        if ((key.status === 'valid' || key.status === 'unchecked') && key.isEnabled !== false) {
                            hasValidKeys = true;
                            const tier = (key.tier || 'FREE').toUpperCase();
                            const type = (tier === 'FREE') ? 'free' : 'paid';

                            MODELS.forEach((model: any) => {
                                const settingKey = MODEL_TYPE_TO_SETTINGS[model.id];
                                if (!settingKey) return;

                                const isEnabled = siteSettings?.modelAccessTiers?.[model.id]?.[type] !== false;

                                if (isEnabled) {
                                    const limit = siteSettings?.modelLimitsByTier?.[settingKey]?.[type]
                                        ?? DEFAULT_TIER_LIMITS[settingKey]?.[type]
                                        ?? 0;
                                    total += limit;
                                }
                            });
                        }
                    });
                }

                if (hasValidKeys) {
                    setEnabledModelsLimit(total);
                } else {
                    setEnabledModelsLimit(user?.usageStats?.apiCallsLimit || 0);
                }
            } catch (error) {
                console.error('Failed to calculate enabled models limit:', error);
                setEnabledModelsLimit(user?.usageStats?.apiCallsLimit || 0);
            }
        };

        fetchEnabledModelsLimit();
    }, [user]);

    // Performance Analytics Data (Real)
    const [performanceData, setPerformanceData] = useState<any[]>([]);

    // New Stats
    const [totalUsers, setTotalUsers] = useState(0);
    const [totalGeneratedFiles, setTotalGeneratedFiles] = useState(0);

    // Broadcast
    const [activeBroadcast, setActiveBroadcast] = useState<Broadcast | null>(null);

    useEffect(() => {
        const checkBroadcasts = async () => {
            if (!user) return;
            try {
                const broadcasts = await adminService.getBroadcasts();
                const prefs = await UserDatabase.getPreferences(user.uid);
                const dismissed = prefs.dismissedBroadcasts || [];
                const localDismissedId = localStorage.getItem('sf_dismissed_broadcast_id');

                // Find latest active broadcast NOT dismissed
                const active = broadcasts.find(b => {
                    if (b.isDeleted) return false;

                    // Specific Targeting
                    const target = (b.targetAudience || 'all').toLowerCase();
                    const userPlan = (user.subscription || 'free').toLowerCase();
                    const isTargeted = target.includes('all') || target.includes(userPlan) || user.role === 'admin';

                    if (!isTargeted) return false;

                    // Dismissal Check
                    // Check both DB prefs and LocalStorage (for non-logged in or quick dismissals)
                    const isDismissedDB = dismissed.includes(b.id);
                    const isDismissedLocal = localDismissedId === b.id;

                    return !isDismissedDB && !isDismissedLocal;
                });

                if (active) {
                    setActiveBroadcast(active);
                }
            } catch (e) {
                console.error("Failed to load broadcasts", e);
            }
        };
        checkBroadcasts();
    }, [user]);

    // Helper to retrieve user key
    const getUserApiKey = async (): Promise<string | null> => {
        const currentUser = authService.getCurrentUser();
        if (!currentUser) return null;

        // The API keys are stored in IndexedDB, so we must await
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const validKey = keys.find((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        return validKey ? validKey.fullKey : null;
    };

    useEffect(() => {
        const fetchAllData = async () => {
            if (!user) return;
            console.log('[Dashboard] Fast-Loading Data for:', user.displayName);

            try {
                // PARALLEL FETCHING: Fetch everything at once to speed up load time
                // We use the new 'excludeUrl' optimization for the big stats query to avoid 500MB+ downloads
                const [
                    statsMap,
                    statsAssets, // Lightweight (no URL) for calculations
                    recentMedia, // Heavy (with URL) but Limited count for UI
                    favorites,
                    trash,
                    activities,
                    systemStats
                ] = await Promise.all([
                    authService.getDashboardStats(user.uid, 30),
                    dbService.getLibrary(user.uid, { excludeUrl: true }), // FAST: No Base64
                    dbService.getLibrary(user.uid, { type: 'image,video,review', limit: 20 }), // FAST: Only 20 items
                    dbService.getFavorites(user.uid),
                    dbService.getTrash(user.uid),
                    authService.getActivities(user.uid, 30).catch(() => []), // Safe catch
                    dbService.getGlobalStats().catch(() => ({ users: 0 }))
                ]);

                // --- 1. Stats & Trends Processing ---
                const getDateKey = (offset: number) => {
                    const d = new Date();
                    // Shift correctly to Pacific Time
                    const ptDate = new Date(d.toLocaleString("en-US", { timeZone: "America/Los_Angeles" }));
                    ptDate.setDate(ptDate.getDate() + offset);

                    const year = ptDate.getFullYear();
                    const month = String(ptDate.getMonth() + 1).padStart(2, '0');
                    const day = String(ptDate.getDate()).padStart(2, '0');
                    return `${year}-${month}-${day}`;
                };
                const getVal = (dateKey: string, field: string) => {
                    const val = statsMap[dateKey]?.[field];
                    return (typeof val === 'number' && !isNaN(val)) ? val : 0;
                };

                // Trends - Compare this week (last 7 days) vs previous week
                const calcWeeklyTrend = (field: string) => {
                    let thisWeek = 0;
                    for (let i = 0; i < 7; i++) {
                        const val = getVal(getDateKey(-i), field);
                        thisWeek += (typeof val === 'number' && !isNaN(val)) ? val : 0;
                    }

                    let lastWeek = 0;
                    for (let i = 7; i < 14; i++) {
                        const val = getVal(getDateKey(-i), field);
                        lastWeek += (typeof val === 'number' && !isNaN(val)) ? val : 0;
                    }

                    if (lastWeek === 0) return thisWeek > 0 ? 100 : 0;
                    const trend = Math.round(((thisWeek - lastWeek) / lastWeek) * 100);
                    return (typeof trend === 'number' && !isNaN(trend)) ? trend : 0;
                };

                setTrends({
                    trendSearches: calcWeeklyTrend('trendSearches'),
                    metadata: calcWeeklyTrend('metadataGenerated'),
                    images: calcWeeklyTrend('imagesGenerated'),
                    imageEditor: calcWeeklyTrend('imagesEdited'),
                    videos: calcWeeklyTrend('videosCreated'),
                    reviewerAi: calcWeeklyTrend('imagesReviewed'),
                    prompts: calcWeeklyTrend('promptsGenerated'),
                    api: calcWeeklyTrend('apiCalls')
                });

                // Histories - Last 7 days
                const getHistory = (field: string) => {
                    const res = [];
                    for (let i = 6; i >= 0; i--) {
                        res.push(getVal(getDateKey(-i), field));
                    }
                    return res;
                };

                setHistories({
                    trendSearches: getHistory('trendSearches'),
                    metadata: getHistory('metadataGenerated'),
                    images: getHistory('imagesGenerated'),
                    imageEditor: getHistory('imagesEdited'),
                    videos: getHistory('videosCreated'),
                    reviewerAi: getHistory('imagesReviewed'),
                    prompts: getHistory('promptsGenerated'),
                    api: getHistory('apiCalls')
                });

                // Daily API
                setDailyApiCalls(getVal(getDateKey(0), 'apiCalls'));

                // Performance Data
                const perfData = [];
                for (let i = 29; i >= 0; i--) {
                    const key = getDateKey(-i);
                    const [y, m, dStr] = key.split('-').map(Number);
                    const dObj = new Date(y, m - 1, dStr);
                    const label = dObj.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                    const api = getVal(key, 'apiCalls');
                    const gen =
                        getVal(key, 'trendSearches') +
                        getVal(key, 'imagesGenerated') +
                        getVal(key, 'videosCreated') +
                        getVal(key, 'metadataGenerated') +
                        getVal(key, 'imagesReviewed') +
                        getVal(key, 'imagesEdited') +
                        getVal(key, 'promptsGenerated');
                    perfData.push({ name: label, apiCalls: api, generated: gen });
                }
                setPerformanceData(perfData);

                // --- 2. Activities ---
                setTotalActivityCount(activities.length);
                setActivities(activities.slice(0, 5));
                setActivityError(null);

                // --- 3. Recent Assets (Display Logic with Fallback) ---
                // We use recentMedia which contains the URLs for display (New Server)
                // If recentMedia is empty, we check if statsAssets (Old Server fallback) has data with URLs
                if (recentMedia.length > 0) {
                    setRecentAssets(recentMedia.slice(0, 10));
                } else {
                    // Fallback: Check if we are on Legacy Server (statsAssets has URLs despite excludeUrl=true)
                    // If statsAssets has URLs, we can use it to populate the recent list
                    const hasLegacyUrls = statsAssets.length > 0 && statsAssets[0].url && statsAssets[0].url.length > 50;

                    if (hasLegacyUrls) {
                        console.warn('[Dashboard] Legacy Server API detected (Optimizer pending restart). Using fallback.');
                        const visual = statsAssets.filter(a => a.type === 'image' || a.type === 'video' || a.type === 'review');
                        const sorted = visual.sort((a, b) => b.createdAt - a.createdAt).slice(0, 10);
                        setRecentAssets(sorted);
                    } else {
                        setRecentAssets([]);
                    }
                }

                // --- 4. Total Generated Files (Count Logic) ---
                // Use statsAssets (all items, no URL) for accurate count
                const totalFiles = statsAssets.length + favorites.length + trash.length;
                setTotalGeneratedFiles(totalFiles);

                // --- 5. Total Users ---
                setTotalUsers(systemStats?.users || 0);

            } catch (error) {
                console.error('[Dashboard] Critical Data Load Failure:', error);
            }
        };

        fetchAllData();

    }, [user?.uid]); // Changed from [user] to [user?.uid] to prevent infinite loop

    // Refresh charts when user stats change (without infinite loop)
    useEffect(() => {
        const refreshCharts = async () => {
            if (!user) return;

            console.log('[Dashboard] Refreshing charts due to stats update');

            // Only refetch the dashboard stats API data (for sparklines and performance chart)
            const statsMap = await authService.getDashboardStats(user.uid, 30);

            // Helper functions (Pacific Time Aligned)
            const getDateKey = (offset: number) => {
                const d = new Date();
                // Shift correctly to Pacific Time
                const ptDate = new Date(d.toLocaleString("en-US", { timeZone: "America/Los_Angeles" }));
                ptDate.setDate(ptDate.getDate() + offset);

                const year = ptDate.getFullYear();
                const month = String(ptDate.getMonth() + 1).padStart(2, '0');
                const day = String(ptDate.getDate()).padStart(2, '0');
                return `${year}-${month}-${day}`;
            };
            const getVal = (dateKey: string, field: string) => {
                const val = statsMap[dateKey]?.[field];
                return (typeof val === 'number' && !isNaN(val)) ? val : 0;
            };

            // Dynamic history based on time range
            const historyDays = statTimeRange === 'daily' ? 7 : 30;
            const getHistory = (field: string) => {
                const res = [];
                for (let i = historyDays - 1; i >= 0; i--) {
                    res.push(getVal(getDateKey(-i), field));
                }
                return res;
            };

            // Update sparklines with dynamic length
            setHistories({
                trendSearches: getHistory('trendSearches'),
                metadata: getHistory('metadataGenerated'),
                images: getHistory('imagesGenerated'),
                imageEditor: getHistory('imagesEdited'),
                videos: getHistory('videosCreated'),
                reviewerAi: getHistory('imagesReviewed'),
                prompts: getHistory('promptsGenerated'),
                api: getHistory('apiCalls')
            });

            // Update performance chart
            const perfData = [];
            for (let i = 29; i >= 0; i--) {
                const key = getDateKey(-i);
                // We parse the key YYYY-MM-DD manually to create a local date object for the chart 
                // that matches the visual string, regardless of actual timezone
                const [y, m, dStr] = key.split('-').map(Number);
                const dObj = new Date(y, m - 1, dStr);
                const label = dObj.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                const api = getVal(key, 'apiCalls');
                const gen =
                    getVal(key, 'trendSearches') +
                    getVal(key, 'imagesGenerated') +
                    getVal(key, 'videosCreated') +
                    getVal(key, 'metadataGenerated') +
                    getVal(key, 'imagesReviewed') +
                    getVal(key, 'imagesEdited') +
                    getVal(key, 'promptsGenerated');
                perfData.push({ name: label, apiCalls: api, generated: gen });
            }
            setPerformanceData(perfData);
        };

        // Only refresh if user and usageStats exist
        if (user?.usageStats) {
            refreshCharts();
        }
    }, [user?.usageStats?.daily?.date, user?.usageStats?.daily?.apiCalls, user?.usageStats?.daily?.trendSearches, statTimeRange]); // Added statTimeRange dependency

    // Load Market Trends (With 30-Day Caching logic, modified for regional)
    useEffect(() => {
        const loadTrends = async () => {
            setIsLoadingTrends(true);
            const regionKey = currentRegion ? `_${currentRegion.toLowerCase().replace(/\s/g, '_')}` : '';
            const cacheKey = `sf_trends_cache_${activeTrendTab}${regionKey}`;
            // Change to 24 hours for daily regeneration
            const oneDayMs = 24 * 60 * 60 * 1000;

            try {
                // 1. Check LocalStorage Cache
                const cachedStr = localStorage.getItem(cacheKey);
                if (cachedStr) {
                    const cachedObj = JSON.parse(cachedStr);
                    const age = Date.now() - cachedObj.timestamp;

                    // Check if cache is fresh (less than 24 hours)
                    if (age < oneDayMs) {
                        setTrendData(cachedObj.data.chartData);
                        setMarketInsights(cachedObj.data.insights);
                        setTrendLastUpdated(cachedObj.timestamp);
                        setIsLoadingTrends(false);
                        return; // EXIT: Do not call API
                    }
                }

                // 2. Determine API Key Source
                let key = '';
                if (activeTrendTab === 'regional') {
                    // For Regional, MUST use user key
                    key = (await getUserApiKey()) || '';
                    if (!key) {
                        // Should have been caught by handleTabChange, but double check
                        setTrendData([]);
                        setMarketInsights(null);
                        setIsLoadingTrends(false);
                        return;
                    }
                } else {
                    // For Global, use System Key
                    const settings = await adminService.getSiteSettings();
                    key = settings.appStaticApiKey || '';
                }

                if (!key) {
                    setTrendData([]);
                    setMarketInsights(null);
                    setIsLoadingTrends(false);
                    return;
                }

                const result = await generateDashboardTrends(key, activeTrendTab, currentRegion);
                setTrendData(result.chartData);
                setMarketInsights(result.insights);
                setTrendLastUpdated(Date.now());

                // 3. Save to Cache & Update Stats
                if (result.chartData.length > 0) {
                    localStorage.setItem(cacheKey, JSON.stringify({
                        timestamp: Date.now(),
                        data: result
                    }));

                    // Only increment system usage if we used the system key (Global)
                    if (activeTrendTab === 'global') {
                        await adminService.incrementSystemApiUsage();
                    } else if (user) {
                        // Increment user API usage for regional research
                        await authService.incrementUserStat('trendSearches', 1);
                        // Track credit usage if using user key (Regional)
                        if (activeTrendTab === 'regional' && key) {
                            const modelId = getPreferredModels('thinking')[0] || 'gemini-2.0-flash-thinking-exp';
                            await authService.deductCredits(user.uid, 1, key, modelId);
                        }
                    }
                }
            } catch (e) {
                console.error("Failed to load market trends", e);
                setTrendData([]);
                setMarketInsights(null);
            } finally {
                setIsLoadingTrends(false);
            }
        };
        loadTrends();
    }, [activeTrendTab, currentRegion]);

    const handleTabChange = async (tab: 'global' | 'regional') => {
        if (tab === 'regional') {
            // Check for user API key first
            const userKey = await getUserApiKey();
            if (!userKey) {
                setShowKeyWarning(true);
                return;
            }
            setShowRegionModal(true);
        } else {
            setActiveTrendTab('global');
            setCurrentRegion('');
        }
    };

    const handleRegionConfirm = () => {
        if (regionInput.trim()) {
            setCurrentRegion(regionInput);
            setActiveTrendTab('regional');
            setShowRegionModal(false);
        }
    };

    const dismissBroadcast = async () => {
        if (activeBroadcast && user) {
            localStorage.setItem('sf_dismissed_broadcast_id', activeBroadcast.id);
            setActiveBroadcast(null);
            await adminService.markBroadcastRead(activeBroadcast.id, user.uid);
        }
    };


    // Helper to extract stats safely
    const getStatValue = (field: string) => {
        if (!user?.usageStats) return 0;
        if (statTimeRange === 'daily') {
            return user.usageStats.daily?.[field] || 0;
        } else {
            return user.usageStats.monthly?.[field] || 0;
        }
    };

    const getStatTrend = (field: string) => {
        if (!user?.usageStats?.history) return 0;

        const current = getStatValue(field);
        let previous = 0;

        if (statTimeRange === 'daily') {
            previous = user.usageStats.history.lastDay?.[field] || 0;
        } else {
            previous = user.usageStats.history.lastMonth?.[field] || 0;
        }

        if (previous === 0) return current > 0 ? 100 : 0;
        return Math.round(((current - previous) / previous) * 100);
    };

    // Stats counters - Ordered 1-7 as per user requirement
    const stats = [
        // 1. Trend Searched (pink)
        {
            label: 'Trend Searched',
            value: getStatValue('trendSearches'),
            trend: getStatTrend('trendSearches'),
            color: 'pink',
            icon: TrendingUp,
            data: histories.trendSearches
        },
        // 2. Metadata Generated (amber)
        {
            label: 'Metadata Generated',
            value: getStatValue('metadataGenerated'),
            trend: getStatTrend('metadataGenerated'),
            color: 'amber',
            icon: ScanEye,
            data: histories.metadata
        },
        // 3. Images Generated (purple)
        {
            label: 'Images Generated',
            value: getStatValue('imagesGenerated'),
            trend: getStatTrend('imagesGenerated'),
            color: 'purple',
            icon: ImageIcon,
            data: histories.images
        },
        // 4. Image Editor (cyan)
        {
            label: 'Image Editor',
            value: getStatValue('imagesEdited'),
            trend: getStatTrend('imagesEdited'),
            color: 'cyan',
            icon: Wand2,
            data: histories.imageEditor
        },
        // 5. Videos Created (blue)
        {
            label: 'Videos Created',
            value: getStatValue('videosCreated'),
            trend: getStatTrend('videosCreated'),
            color: 'blue',
            icon: Video,
            data: histories.videos
        },
        // 6. Reviewer AI (red)
        {
            label: 'Reviewer AI',
            value: getStatValue('imagesReviewed'),
            trend: getStatTrend('imagesReviewed'),
            color: 'red',
            icon: SearchCheck,
            data: histories.reviewerAi
        },
        // 7. Prompts Generated
        {
            label: 'Prompts Generated',
            value: getStatValue('promptsGenerated'),
            trend: getStatTrend('promptsGenerated'),
            color: 'orange',
            icon: Sparkles,
            data: histories.prompts
        },
        // 8. Api Calls (green) - HIGHLIGHTED  
        {
            label: 'Api Calls',
            value: user?.usageStats?.daily?.apiCalls || 0,
            trend: (() => {
                if (!user?.usageStats?.history) return 0;
                const current = user?.usageStats?.daily?.apiCalls || 0;
                const previous = user.usageStats.history.lastDay?.apiCalls || 0;
                if (previous === 0) return current > 0 ? 100 : 0;
                return Math.round(((current - previous) / previous) * 100);
            })(),
            limit: enabledModelsLimit,
            color: 'green',
            icon: Zap,
            data: histories.api,
            highlighted: true,
            fixedPeriod: 'daily'
        }
    ];

    useEffect(() => {
        console.log('[Dashboard] Stats Map Keys:', Object.keys(trendData)); // trendData is array, need statsMap from fetchAllData scope? 
        // Can't access statsMap here. I'll add log inside fetchAllData.
    }, [trendData]);

    useEffect(() => {
        const hour = new Date().getHours();
        if (hour < 12) setGreeting('Good morning');
        else if (hour < 18) setGreeting('Good afternoon');
        else setGreeting('Good evening');

        const dismissed = localStorage.getItem('sf_upgrade_dismissed');
        if (dismissed) setShowUpgrade(false);
    }, []);

    const toggleActivityHistory = async () => {
        if (user) {
            if (!isActivityExpanded) {
                // Show full 30-day history
                const allLogs = await authService.getActivities(user.uid, 30);
                setActivities(allLogs);
            } else {
                // Revert to showing only the latest 5
                const allLogs = await authService.getActivities(user.uid, 30);
                setActivities(allLogs.slice(0, 5));
            }
            setIsActivityExpanded(!isActivityExpanded);
        }
    };

    const updateRecentAssets = async () => {
        if (!user) return;
        const assets = await dbService.getLibrary(user.uid);
        const visualAssets = assets.filter(a => a.type === 'image' || a.type === 'video' || a.type === 'review');
        setRecentAssets(visualAssets.sort((a, b) => b.createdAt - a.createdAt).slice(0, 10));
    };



    return (
        <div className="p-6 lg:p-10 max-w-[1600px] mx-auto space-y-8 animate-fade-in">
            {/* Welcome Section & Broadcast */}
            <div className="flex flex-col md:flex-row md:items-start justify-between gap-4">
                <div>
                    <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk']">
                        {greeting}, {user?.displayName?.split(' ')[0] || 'Creator'}! 👋
                    </h1>
                    <p className="text-slate-600 dark:text-slate-400 mt-1 flex items-center gap-2 font-medium">
                        <Clock size={14} /> Here's your creative summary for today.
                    </p>
                </div>

                {/* Date Toggle */}
                <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-lg border border-slate-200 dark:border-slate-700 self-start md:self-center">
                    <button
                        onClick={() => setStatTimeRange('daily')}
                        className={`px-4 py-1.5 rounded-md text-sm font-bold transition-all flex items-center gap-2 ${statTimeRange === 'daily'
                            ? 'bg-white dark:bg-slate-700 text-slate-900 dark:text-white shadow-sm'
                            : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <Calendar size={14} /> Daily
                    </button>
                    <button
                        onClick={() => setStatTimeRange('monthly')}
                        className={`px-4 py-1.5 rounded-md text-sm font-bold transition-all flex items-center gap-2 ${statTimeRange === 'monthly'
                            ? 'bg-white dark:bg-slate-700 text-slate-900 dark:text-white shadow-sm'
                            : 'text-slate-500 hover:text-slate-900 dark:hover:text-slate-300'
                            }`}
                    >
                        <Layers size={14} /> Monthly
                    </button>
                </div>

                {/* Broadcast Box (Top Right) */}
                {activeBroadcast && (
                    <div className="flex-1 max-w-2xl animate-slide-down">
                        <div
                            onClick={() => {
                                if (user && activeBroadcast) {
                                    adminService.markBroadcastRead(activeBroadcast.id, user.uid);
                                }
                            }}
                            className="bg-red-500/10 border border-red-500 rounded-xl p-4 relative overflow-hidden cursor-pointer hover:bg-red-500/20 transition-colors"
                        >
                            <div className="absolute top-0 left-0 w-1 h-full bg-red-500"></div>
                            <div className="flex justify-between items-start gap-4">
                                <div>
                                    <h3 className="text-red-600 dark:text-red-400 font-bold text-sm uppercase tracking-wider mb-1 flex items-center gap-2">
                                        <Megaphone size={14} /> ADMIN BROADCAST MESSAGE!
                                    </h3>
                                    <h4 className="font-bold text-slate-900 dark:text-white text-lg mb-1">{activeBroadcast.title}</h4>
                                    <p className="text-sm text-slate-600 dark:text-slate-300 leading-relaxed">{activeBroadcast.content}</p>
                                </div>
                                <button
                                    onClick={(e) => {
                                        e.stopPropagation();
                                        dismissBroadcast();
                                    }}
                                    className="p-1.5 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 rounded-lg hover:bg-red-200 dark:hover:bg-red-900/50 transition-colors"
                                >
                                    <X size={16} />
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            </div>

            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                {stats.map((stat: any, i) => (
                    <StatsCard key={i} {...stat} delay={i * 100} period={stat.fixedPeriod || statTimeRange} />
                ))}
            </div>

            {/* Main Grid Layout */}
            <div className="grid grid-cols-1 xl:grid-cols-3 gap-8">

                {/* Left Column (2/3 width on large screens) */}
                <div className="xl:col-span-2 space-y-8">

                    {/* Quick Actions */}
                    <div className="space-y-4">
                        <div className="flex items-center justify-between">
                            <h2 className="text-xl font-bold text-slate-900 dark:text-slate-100">Quick Actions</h2>
                            <button
                                onClick={() => setShowAllActions(!showAllActions)}
                                className="text-sm text-amber-600 dark:text-amber-500 hover:text-amber-700 dark:hover:text-amber-400 flex items-center gap-1 transition-colors font-bold"
                            >
                                {showAllActions ? 'View Less' : 'View All'} {showAllActions ? <ChevronUp size={14} /> : <ChevronRight size={14} />}
                            </button>
                        </div>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 transition-all">
                            <QuickAction
                                icon={ImageIcon} color="text-purple-600 dark:text-purple-400" bg="bg-purple-100 dark:bg-purple-500/10" border="border-purple-200 dark:border-purple-500/20"
                                title="Image Generation" desc="Imagen 4.0"
                                onClick={() => setView(AppView.IMAGE_GENERATION)}
                                comingSoon={true}
                            />
                            <QuickAction
                                icon={Wand2} color="text-emerald-600 dark:text-emerald-400" bg="bg-emerald-100 dark:bg-emerald-500/10" border="border-emerald-200 dark:border-emerald-500/20"
                                title="Image Editor" desc="Magic Editor"
                                onClick={() => setView(AppView.IMAGE_EDITOR)}
                                comingSoon={true}
                            />
                            <QuickAction
                                icon={Video} color="text-blue-600 dark:text-blue-400" bg="bg-blue-100 dark:bg-blue-500/10" border="border-blue-200 dark:border-blue-500/20"
                                title="Video Generation" desc="Veo Cinema"
                                onClick={() => setView(AppView.VIDEO_GENERATION)}
                                comingSoon={true}
                            />
                            <QuickAction
                                icon={ScanEye} color="text-amber-600 dark:text-amber-400" bg="bg-amber-100 dark:bg-amber-500/10" border="border-amber-200 dark:border-amber-500/20"
                                title="Metadata AI" desc="Stock Csv"
                                onClick={() => setView(AppView.MEDIA_ANALYSIS)}
                                comingSoon={sidebarSettings?.[AppView.MEDIA_ANALYSIS]?.comingSoon ?? true}
                            />

                            {showAllActions && (
                                <>
                                    <QuickAction
                                        icon={TrendingUp} color="text-cyan-600 dark:text-cyan-400" bg="bg-cyan-100 dark:bg-cyan-500/10" border="border-cyan-200 dark:border-cyan-500/20"
                                        title="Market Intelligence" desc="Stock Trends"
                                        onClick={() => setView(AppView.CHAT_INTELLIGENCE)}
                                    />
                                    <QuickAction
                                        icon={SearchCheck} color="text-red-600 dark:text-red-400" bg="bg-red-100 dark:bg-red-500/10" border="border-red-200 dark:border-red-500/20"
                                        title="Reviewer AI" desc="Stock Csv"
                                        onClick={() => setView(AppView.REVIEWER_AI)}
                                    />
                                    <QuickAction
                                        icon={Sparkles} color="text-orange-600 dark:text-orange-400" bg="bg-orange-100 dark:bg-orange-500/10" border="border-orange-200 dark:border-orange-500/20"
                                        title="Auto-Prompt Gen" desc="AI Prompts"
                                        onClick={() => setView(AppView.PROMPT_GENERATOR)}
                                        comingSoon={true}
                                    />
                                </>
                            )}
                        </div>
                    </div>

                    {/* Performance Chart */}
                    <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 backdrop-blur-sm shadow-sm dark:shadow-none">
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="font-bold text-slate-900 dark:text-slate-100">Performance Analytics (Last 30 Days)</h3>
                            <div className="flex bg-slate-100 dark:bg-slate-950 rounded-lg p-1 border border-slate-200 dark:border-slate-800">
                                <button
                                    onClick={() => setPerfMetric('usage')}
                                    className={`px-3 py-1 rounded-md text-xs font-bold transition-all ${perfMetric === 'usage' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                                >
                                    API Calls Usage
                                </button>
                                <button
                                    onClick={() => setPerfMetric('generated')}
                                    className={`px-3 py-1 rounded-md text-xs font-bold transition-all ${perfMetric === 'generated' ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                                >
                                    Generations
                                </button>
                            </div>
                        </div>
                        <div className="h-[300px] w-full">
                            <ResponsiveContainer width="100%" height="100%">
                                <AreaChart data={performanceData}>
                                    <defs>
                                        <linearGradient id="colorUsage" x1="0" y1="0" x2="0" y2="1">
                                            <stop offset="5%" stopColor="#f59e0b" stopOpacity={0.3} />
                                            <stop offset="95%" stopColor="#f59e0b" stopOpacity={0} />
                                        </linearGradient>
                                        <linearGradient id="colorGen" x1="0" y1="0" x2="0" y2="1">
                                            <stop offset="5%" stopColor="#8b5cf6" stopOpacity={0.3} />
                                            <stop offset="95%" stopColor="#8b5cf6" stopOpacity={0} />
                                        </linearGradient>
                                    </defs>
                                    <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                                    <XAxis
                                        dataKey="name"
                                        stroke="#64748b"
                                        fontSize={12}
                                        tickLine={false}
                                        axisLine={false}
                                        minTickGap={30}
                                    />
                                    <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} domain={[0, 'auto']} />
                                    <Tooltip
                                        labelFormatter={(label) => label}
                                        formatter={(value: any, name: string) => {
                                            if (name === 'apiCalls') return [value, 'API Calls'];
                                            if (name === 'generated') return [value, 'Generated'];
                                            return [value, name];
                                        }}
                                        contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#e2e8f0' }}
                                        itemStyle={{ color: '#e2e8f0' }}
                                    />
                                    <Area
                                        type="monotone"
                                        dataKey={perfMetric === 'usage' ? 'apiCalls' : 'generated'}
                                        stroke={perfMetric === 'usage' ? '#f59e0b' : '#8b5cf6'}
                                        fillOpacity={1}
                                        fill={`url(#${perfMetric === 'usage' ? 'colorUsage' : 'colorGen'})`}
                                        strokeWidth={3}
                                    />
                                </AreaChart>
                            </ResponsiveContainer>
                        </div>
                    </div>

                    {/* Recent Creations Carousel (Conditional) */}
                    {
                        recentAssets.length >= 1 && (
                            <RecentCreations assets={recentAssets} onUpdate={updateRecentAssets} />
                        )
                    }

                    {/* Trends Widget */}
                    <div className="bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm dark:shadow-none">
                        <div className="flex items-center justify-between mb-6">
                            <div>
                                <h3 className="font-bold text-slate-900 dark:text-slate-100 flex items-center gap-2">
                                    <TrendingUp size={18} className="text-emerald-500" /> Microstock Market Trends
                                </h3>
                                {trendLastUpdated && (
                                    <p className="text-[10px] text-slate-500 mt-1 flex items-center gap-1">
                                        <Database size={10} /> Updated: {new Date(trendLastUpdated).toLocaleDateString()} {activeTrendTab === 'regional' && currentRegion && `• ${currentRegion}`}
                                    </p>
                                )}
                            </div>
                            <div className="flex gap-2">
                                <button
                                    onClick={() => handleTabChange('global')}
                                    className={`flex items-center gap-1 px-3 py-1.5 rounded-lg text-xs font-bold border transition-colors ${activeTrendTab === 'global' ? 'bg-emerald-100 dark:bg-emerald-500/10 text-emerald-700 dark:text-emerald-400 border-emerald-200 dark:border-emerald-500/20' : 'bg-transparent border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                                >
                                    <Globe size={12} /> Global
                                </button>
                                <button
                                    onClick={() => handleTabChange('regional')}
                                    className={`flex items-center gap-1 px-3 py-1.5 rounded-lg text-xs font-bold border transition-colors ${activeTrendTab === 'regional' ? 'bg-emerald-100 dark:bg-emerald-500/10 text-emerald-700 dark:text-emerald-400 border-emerald-200 dark:border-emerald-500/20' : 'bg-transparent border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                                >
                                    <MapPin size={12} /> {currentRegion ? `Regional (${currentRegion})` : 'Regional'}
                                </button>
                            </div>
                        </div>

                        {isLoadingTrends ? (
                            <div className="flex items-center justify-center h-40 text-slate-500 gap-2">
                                <Loader2 className="animate-spin text-emerald-500" size={24} />
                                Generating forecast...
                            </div>
                        ) : trendData.length > 0 ? (
                            <div className="space-y-4 animate-fade-in">
                                {/* Charts */}
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                    {trendData.slice(0, 3).map((trend, idx) => (
                                        <div key={idx} className="p-4 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800 hover:border-emerald-500/30 transition-colors group">
                                            <div className="flex justify-between items-start mb-2">
                                                <h4 className="font-bold text-slate-700 dark:text-slate-300 text-sm group-hover:text-emerald-600 dark:group-hover:text-emerald-400 transition-colors">{trend.name}</h4>
                                                <span className="text-xs font-bold text-emerald-600 dark:text-emerald-500 flex items-center">
                                                    <ArrowUpRight size={12} /> {trend.growth}%
                                                </span>
                                            </div>
                                            <div className="h-10">
                                                <ResponsiveContainer width="100%" height="100%">
                                                    <LineChart data={trend.data.map((d: any, i: number) => ({ val: d, i }))}>
                                                        <Line type="monotone" dataKey="val" stroke="#10b981" strokeWidth={2} dot={false} />
                                                    </LineChart>
                                                </ResponsiveContainer>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Insights Grid - Now 2 Rows */}
                                {marketInsights && (
                                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3 pt-2">
                                        <MarketInsightItem
                                            label="Top Title Pattern"
                                            value={marketInsights.topPattern}
                                            icon={<FileText size={14} />}
                                            color="text-blue-600 dark:text-blue-400"
                                        />
                                        <MarketInsightItem
                                            label="Best Single Keyword"
                                            value={marketInsights.bestKeyword}
                                            sub="Vol: High"
                                            icon={<Zap size={14} />}
                                            color="text-purple-600 dark:text-purple-400"
                                        />
                                        <MarketInsightItem
                                            label="Best 2-Word Keyword"
                                            value={marketInsights.bestBiKeyword}
                                            sub="+28%"
                                            icon={<TrendingUp size={14} />}
                                            color="text-pink-600 dark:text-pink-400"
                                        />
                                        <MarketInsightItem
                                            label="Hot Keyword Mix"
                                            value={marketInsights.keywordMix}
                                            icon={<Database size={14} />}
                                            color="text-amber-600 dark:text-amber-400"
                                        />
                                        {/* Row 2 */}
                                        <MarketInsightItem
                                            label="Best Upload Time"
                                            value={marketInsights.bestTime}
                                            icon={<Clock size={14} />}
                                            color="text-teal-600 dark:text-teal-400"
                                        />
                                        <MarketInsightItem
                                            label="Top Category"
                                            value={marketInsights.topCategory}
                                            icon={<Layers size={14} />}
                                            color="text-indigo-600 dark:text-indigo-400"
                                        />
                                        <div className="flex flex-col justify-between p-3 rounded-xl bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 hover:border-slate-300 dark:hover:border-slate-700 transition-colors group">
                                            <div className="flex items-center justify-between mb-2">
                                                <span className="text-[10px] uppercase font-bold text-slate-500 flex items-center gap-1.5">
                                                    <span className="text-slate-400"><Palette size={14} /></span>
                                                    Emerging Color
                                                </span>
                                            </div>
                                            <div className="flex items-center gap-2">
                                                <div className="w-6 h-6 rounded-full border border-slate-300 dark:border-slate-600 shadow-sm" style={{ backgroundColor: marketInsights.emergingColor }}></div>
                                                <span className="text-sm font-bold text-slate-700 dark:text-slate-300">{marketInsights.emergingColor}</span>
                                            </div>
                                        </div>
                                        <MarketInsightItem
                                            label="Dead Niche (Avoid)"
                                            value={marketInsights.deadNiche}
                                            icon={<ThumbsDown size={14} />}
                                            color="text-red-600 dark:text-red-400"
                                        />
                                    </div>
                                )}
                            </div>
                        ) : (
                            <div className="flex flex-col items-center justify-center h-40 text-slate-500 gap-1 bg-slate-50 dark:bg-slate-950/30 rounded-xl border border-dashed border-slate-200 dark:border-slate-800">
                                <AlertCircle className="opacity-50 mb-1" size={24} />
                                <p className="text-sm font-medium">Market trends unavailable</p>
                                <p className="text-xs opacity-70">System key required in Admin Settings.</p>
                            </div>
                        )}
                    </div>

                </div>

                {/* Right Column (Widgets) */}
                <div className="space-y-8">

                    {/* Tutorial Widget (Moved from Sidebar) */}
                    <TutorialWidget
                        onOpenSettings={onOpenSettings}
                        setView={setView}
                        className="mb-2 shadow-sm"
                    />

                    {/* Upgrade Prompt */}
                    {
                        showUpgrade && user?.subscription === 'free' && (
                            <div className="relative overflow-hidden rounded-2xl bg-gradient-to-br from-amber-500 to-orange-600 p-6 text-slate-950 shadow-lg animate-slide-up">
                                <button
                                    onClick={() => {
                                        setShowUpgrade(false);
                                        localStorage.setItem('sf_upgrade_dismissed', 'true');
                                    }}
                                    className="absolute top-4 right-4 opacity-60 hover:opacity-100 transition-opacity text-white"
                                >
                                    <X size={18} />
                                </button>
                                <div className="w-12 h-12 bg-white/20 rounded-xl flex items-center justify-center mb-4 backdrop-blur-md">
                                    <Crown size={24} className="text-white" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-white">Unlock Pro Features</h3>
                                <ul className="space-y-2 mb-6 text-sm font-bold text-white/90">
                                    <li className="flex items-center gap-2"><ArrowUpRight size={14} /> Unlimited Generations</li>
                                    <li className="flex items-center gap-2"><ArrowUpRight size={14} /> 4K Video Export</li>
                                    <li className="flex items-center gap-2"><ArrowUpRight size={14} /> Gemini 3.0 Reasoning</li>
                                </ul>
                                <button className="w-full py-3 bg-slate-950 text-white font-bold rounded-xl hover:bg-slate-900 transition-colors shadow-xl">
                                    Upgrade Now
                                </button>
                            </div>
                        )
                    }

                    {/* Tips of the Day */}
                    <TipsCarousel />

                    {/* Recent Activity (LIVE) */}
                    <div className="relative overflow-hidden rounded-2xl border border-slate-200 dark:border-slate-800 bg-white dark:bg-slate-900/50 shadow-lg dark:shadow-2xl transition-all hover:shadow-xl">
                        {/* Colored Header Background */}
                        <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-blue-500 via-purple-500 to-amber-500"></div>

                        {/* Ambient Glow */}
                        <div className="absolute -top-24 -right-24 w-48 h-48 bg-blue-500/10 blur-3xl rounded-full pointer-events-none"></div>

                        <div className="p-6 relative z-10">
                            <h3 className="font-bold text-slate-900 dark:text-slate-100 mb-6 flex items-center gap-2">
                                <Activity className="text-blue-500" size={20} />
                                Recent Activity
                            </h3>

                            {/* Scrollable Container with Conditional Class */}
                            <div className={`relative transition-all duration-500 ease-in-out ${isActivityExpanded ? 'h-[400px] overflow-y-auto pr-2 custom-scrollbar' : 'h-auto'}`}>
                                <div className="relative space-y-8">
                                    {/* Continuous Timeline Line */}
                                    <div className="absolute left-[20px] top-3 bottom-3 w-0.5 bg-slate-100 dark:bg-slate-800 z-0"></div>

                                    {activities.length > 0 ? activities.map((act, idx) => (
                                        <ActivityItem
                                            key={act.id}
                                            type={act.type}
                                            text={act.text}
                                            time={formatTimeAgo(act.timestamp)}
                                            cost={act.cost}
                                        />
                                    )) : (
                                        <div className="flex flex-col items-center justify-center py-8 text-center relative z-10 bg-slate-50/50 dark:bg-slate-900/50 rounded-xl border border-dashed border-slate-200 dark:border-slate-800">
                                            <Clock size={24} className="text-slate-400 mb-2" />
                                            <p className="text-sm text-slate-500 dark:text-slate-400 font-medium">No recent activity.</p>
                                            <p className="text-xs text-slate-400">Start creating to see logs here.</p>
                                        </div>
                                    )}
                                </div>
                            </div>

                            {totalActivityCount > 5 && (
                                <div className="mt-6 pt-4 border-t border-slate-100 dark:border-slate-800/50">
                                    <button
                                        onClick={toggleActivityHistory}
                                        className={`w-full py-3 px-4 rounded-xl text-sm font-bold transition-all duration-300 flex items-center justify-center gap-2
                                      ${isActivityExpanded
                                                ? 'bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 hover:bg-slate-200 dark:hover:bg-slate-700'
                                                : 'bg-gradient-to-r from-slate-900 to-slate-800 dark:from-blue-600 dark:to-blue-700 text-white shadow-lg hover:shadow-xl hover:scale-[1.02]'
                                            }`}
                                    >
                                        {isActivityExpanded ? (
                                            <>Show Less <ChevronUp size={16} /></>
                                        ) : (
                                            <>View 30-Day History <ChevronRight size={16} /></>
                                        )}
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>

                </div >
            </div >

            {/* Key Warning Modal */}
            {
                showKeyWarning && (
                    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm p-4 animate-fade-in">
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 w-full max-w-md shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}>
                            <div className="flex flex-col items-center text-center">
                                <div className="w-16 h-16 bg-amber-100 dark:bg-amber-900/20 rounded-full flex items-center justify-center mb-4 text-amber-600 dark:text-amber-500">
                                    <Key size={32} />
                                </div>
                                <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-2">API Key Required</h3>
                                <p className="text-slate-600 dark:text-slate-400 mb-6 text-sm">
                                    Regional research requires deep reasoning and advanced data processing. Please add your personal Gemini API key in settings to access this feature.
                                </p>
                                <div className="flex gap-3 w-full">
                                    <button
                                        onClick={() => setShowKeyWarning(false)}
                                        className="flex-1 py-2.5 bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 font-bold rounded-xl hover:bg-slate-200 dark:hover:bg-slate-700 transition"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={() => {
                                            setShowKeyWarning(false);
                                            window.dispatchEvent(new CustomEvent('sf_navigate', { detail: 'settings_api' }));
                                        }}
                                        className="flex-1 py-2.5 bg-amber-500 hover:bg-amber-600 text-white dark:text-slate-950 font-bold rounded-xl transition shadow-lg shadow-amber-500/20"
                                    >
                                        Add Key
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )
            }

            {/* Regional Selection Modal */}
            {
                showRegionModal && (
                    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm p-4 animate-fade-in">
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 w-full max-w-md shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}>
                            <div className="flex justify-between items-center mb-4">
                                <h3 className="text-xl font-bold text-slate-900 dark:text-white">Select Region</h3>
                                <button onClick={() => setShowRegionModal(false)} className="p-1 text-slate-500 hover:text-slate-900 dark:hover:text-white rounded-full hover:bg-slate-100 dark:hover:bg-slate-800 transition"><X size={20} /></button>
                            </div>
                            <p className="text-sm text-slate-600 dark:text-slate-400 mb-4">Enter any country to generate localized microstock trends.</p>

                            <div className="space-y-4">
                                <div className="relative">
                                    <MapPin className="absolute left-3 top-3 text-slate-400" size={18} />
                                    <input
                                        type="text"
                                        value={regionInput}
                                        onChange={e => setRegionInput(e.target.value)}
                                        placeholder="e.g. Germany, Japan, Brazil"
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl py-2.5 pl-10 pr-4 text-slate-900 dark:text-white focus:ring-1 focus:ring-emerald-500 outline-none font-medium"
                                        autoFocus
                                        onKeyDown={e => e.key === 'Enter' && handleRegionConfirm()}
                                    />
                                </div>
                                <div className="flex gap-3 pt-2">
                                    <button onClick={() => setShowRegionModal(false)} className="flex-1 py-2.5 bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 font-bold rounded-xl hover:bg-slate-200 dark:hover:bg-slate-700 transition">Cancel</button>
                                    <button onClick={handleRegionConfirm} disabled={!regionInput.trim()} className="flex-1 py-2.5 bg-emerald-500 hover:bg-emerald-600 text-white font-bold rounded-xl transition disabled:opacity-50 shadow-lg shadow-emerald-500/20">Analyze Market</button>
                                </div>
                            </div>
                        </div>
                    </div>
                )
            }
        </div >
    );
};

// --- Sub Components ---

const StatsCard = ({ label, value, trend, color, icon: Icon, data, limit, type, delay, period = 'Daily', featured, highlighted }: any) => {
    const [count, setCount] = useState(0);

    useEffect(() => {
        if (typeof value === 'number') {
            let start = 0;
            const end = value;
            const duration = 1000;
            const timer = setInterval(() => {
                start += Math.ceil(end / 20);
                if (start >= end) {
                    setCount(end);
                    clearInterval(timer);
                } else {
                    setCount(start);
                }
            }, 30);
            return () => clearInterval(timer);
        }
    }, [value]);

    const styles = {
        purple: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-purple-200 dark:border-purple-500/20',
            iconBox: 'bg-purple-100 dark:bg-purple-500/10 border-purple-200 dark:border-purple-500/20',
            icon: 'text-purple-600 dark:text-purple-400',
            stroke: '#c084fc'
        },
        blue: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-blue-200 dark:border-blue-500/20',
            iconBox: 'bg-blue-100 dark:bg-blue-500/10 border-blue-200 dark:border-blue-500/20',
            icon: 'text-blue-600 dark:text-blue-400',
            stroke: '#60a5fa'
        },
        amber: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-amber-200 dark:border-amber-500/20',
            iconBox: 'bg-amber-100 dark:bg-amber-500/10 border-amber-200 dark:border-amber-500/20',
            icon: 'text-amber-600 dark:text-amber-400',
            stroke: '#fbbf24'
        },
        green: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-emerald-200 dark:border-emerald-500/20',
            iconBox: 'bg-emerald-100 dark:bg-emerald-500/10 border-emerald-200 dark:border-emerald-500/20',
            icon: 'text-emerald-600 dark:text-emerald-400',
            stroke: '#34d399'
        },
        red: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-red-200 dark:border-red-500/20',
            iconBox: 'bg-red-100 dark:bg-red-500/10 border-red-200 dark:border-red-500/20',
            icon: 'text-red-600 dark:text-red-400',
            stroke: '#f87171'
        },
        cyan: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-cyan-200 dark:border-cyan-500/20',
            iconBox: 'bg-cyan-100 dark:bg-cyan-500/10 border-cyan-200 dark:border-cyan-500/20',
            icon: 'text-cyan-600 dark:text-cyan-400',
            stroke: '#22d3ee'
        },
        pink: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-pink-200 dark:border-pink-500/20',
            iconBox: 'bg-pink-100 dark:bg-pink-500/10 border-pink-200 dark:border-pink-500/20',
            icon: 'text-pink-600 dark:text-pink-400',
            stroke: '#f472b6'
        },
        indigo: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-indigo-200 dark:border-indigo-500/20',
            iconBox: 'bg-indigo-100 dark:bg-indigo-500/10 border-indigo-200 dark:border-indigo-500/20',
            icon: 'text-indigo-600 dark:text-indigo-400',
            stroke: '#818cf8'
        },
        orange: {
            bg: 'bg-white dark:bg-slate-900/50',
            border: 'border-orange-200 dark:border-orange-500/20',
            iconBox: 'bg-orange-100 dark:bg-orange-500/10 border-orange-200 dark:border-orange-500/20',
            icon: 'text-orange-600 dark:text-orange-400',
            stroke: '#fb923c'
        }
    }[color as string] || { // Fallback
        bg: 'bg-white dark:bg-slate-900/50',
        border: 'border-slate-200 dark:border-slate-800',
        iconBox: 'bg-slate-100 dark:bg-slate-800 border-slate-200 dark:border-slate-700',
        icon: 'text-slate-600 dark:text-slate-400',
        stroke: '#cbd5e1'
    };

    // Calculations for circular progress
    const percentage = Math.min(100, Math.max(0, (value / (limit || 1)) * 100));

    // Featured box styling
    if (featured) {
        return (
            <div
                className="relative rounded-2xl p-6 overflow-hidden group animate-fade-in"
                style={{ animationDelay: `${delay}ms` }}
            >
                {/* Animated gradient background */}
                <div className="absolute inset-0 bg-gradient-to-br from-emerald-500/20 via-green-500/10 to-teal-500/20 dark:from-emerald-500/30 dark:via-green-500/20 dark:to-teal-500/30"></div>

                {/* Animated border glow */}
                <div className="absolute inset-0 rounded-2xl border-2 border-emerald-400/50 dark:border-emerald-500/60 animate-pulse"></div>

                {/* Glowing orbs */}
                <div className="absolute -top-10 -right-10 w-32 h-32 bg-emerald-500/20 dark:bg-emerald-400/30 rounded-full blur-3xl animate-pulse"></div>
                <div className="absolute -bottom-10 -left-10 w-32 h-32 bg-green-500/20 dark:bg-green-400/30 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }}></div>

                <div className="relative z-10 bg-white/80 dark:bg-slate-900/80 backdrop-blur-sm rounded-xl p-5 border border-emerald-200/50 dark:border-emerald-500/30 shadow-lg">
                    <div className="flex justify-between items-start mb-4">
                        <div className="flex items-center gap-3">
                            {/* Main Icon */}
                            <div className="w-10 h-10 rounded-xl flex items-center justify-center border bg-emerald-100 dark:bg-emerald-500/20 border-emerald-300 dark:border-emerald-500/40 shadow-lg shadow-emerald-500/20">
                                <Icon size={20} className="text-emerald-600 dark:text-emerald-400" />
                            </div>

                            {/* Period Badge with glow */}
                            <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-emerald-100 dark:bg-emerald-950/50 border border-emerald-300 dark:border-emerald-700 text-emerald-700 dark:text-emerald-400 shadow-sm">
                                <Calendar size={12} strokeWidth={2.5} />
                                <span className="text-[10px] font-bold uppercase tracking-wide">{period}</span>
                            </div>
                        </div>

                        {limit !== undefined ? (
                            <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-emerald-100 dark:bg-emerald-950/50 text-xs font-bold text-emerald-700 dark:text-emerald-400 border border-emerald-300 dark:border-emerald-700">
                                <Minus size={12} />
                                <span className={`${percentage > 90 ? 'text-red-500' : ''}`}>{Math.round(percentage)}%</span>
                            </div>
                        ) : (
                            <div className={`flex items-center gap-1 text-xs font-bold ${trend > 0 ? 'text-emerald-600 dark:text-emerald-400' : trend < 0 ? 'text-red-600 dark:text-red-400' : 'text-slate-500 dark:text-slate-400'} bg-white/90 dark:bg-slate-950/50 px-2 py-1 rounded-lg border border-emerald-200 dark:border-emerald-700`}>
                                {trend > 0 ? <ArrowUpRight size={12} /> : trend < 0 ? <ArrowDownRight size={12} /> : <Minus size={12} />}
                                {Math.abs(trend)}%
                            </div>
                        )}
                    </div>

                    <div className="flex items-end justify-between">
                        <div>
                            <h3 className="text-emerald-700 dark:text-emerald-400 text-sm font-bold mb-1">{label}</h3>
                            <p className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk']">
                                {typeof value === 'number' ? count.toLocaleString() : value}
                                {limit !== undefined && (
                                    <span className="text-sm text-slate-500 font-normal ml-1">
                                        / {typeof limit === 'number' ? (limit >= 10000 ? (limit / 1000).toFixed(1).replace('.0', '') + 'k' : limit.toLocaleString()) : limit}
                                    </span>
                                )}
                            </p>
                        </div>
                        {data && (
                            <div className="h-12 w-24 -mb-2">
                                <ResponsiveContainer width="100%" height="100%">
                                    <LineChart data={data.map((d: any, i: number) => ({ val: d, i }))}>
                                        <YAxis hide domain={[0, 'auto']} />
                                        <Line type="monotone" dataKey="val" stroke="#10b981" strokeWidth={4} strokeOpacity={1} dot={false} isAnimationActive={false} />
                                    </LineChart>
                                </ResponsiveContainer>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    }

    // Highlighted box - SAME SIZE but with special visual effects
    if (highlighted) {
        return (
            <div className="relative rounded-2xl p-6 overflow-hidden group hover:shadow-2xl transition-all animate-fade-in" style={{ animationDelay: `${delay}ms` }}>
                {/* Gradient background overlay */}
                <div className="absolute inset-0 bg-gradient-to-br from-emerald-500/15 via-green-500/10 to-teal-500/15 dark:from-emerald-500/25 dark:via-green-500/15 dark:to-teal-500/25"></div>

                {/* Animated pulsing border */}
                <div className="absolute inset-0 rounded-2xl border-2 border-emerald-400/40 dark:border-emerald-500/50 animate-pulse"></div>

                {/* Subtle corner glows */}
                <div className="absolute -top-8 -right-8 w-24 h-24 bg-emerald-500/20 dark:bg-emerald-400/30 rounded-full blur-2xl animate-pulse"></div>
                <div className="absolute -bottom-8 -left-8 w-24 h-24 bg-green-500/20 dark:bg-green-400/25 rounded-full blur-2xl animate-pulse" style={{ animationDelay: '1s' }}></div>

                <div className="relative z-10">
                    <div className="flex justify-between items-start mb-4">
                        <div className="flex items-center gap-3">
                            {/* Highlighted Icon */}
                            <div className="w-10 h-10 rounded-xl flex items-center justify-center border bg-emerald-100 dark:bg-emerald-500/20 border-emerald-300 dark:border-emerald-500/30 shadow-md shadow-emerald-500/10">
                                <Icon size={20} className="text-emerald-600 dark:text-emerald-400" />
                            </div>

                            {/* Highlighted Period Badge */}
                            <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-emerald-100 dark:bg-emerald-950/50 border border-emerald-300 dark:border-emerald-700 text-emerald-700 dark:text-emerald-400">
                                <Calendar size={12} strokeWidth={2.5} />
                                <span className="text-[10px] font-bold uppercase tracking-wide">{period}</span>
                            </div>
                        </div>

                        {limit !== undefined ? (
                            <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-emerald-100 dark:bg-emerald-950/50 text-xs font-bold text-emerald-700 dark:text-emerald-400 border border-emerald-300 dark:border-emerald-700">
                                <Minus size={12} />
                                <span className={`${percentage > 90 ? 'text-red-500' : ''}`}>{Math.round(percentage)}%</span>
                            </div>
                        ) : (
                            <div className={`flex items-center gap-1 text-xs font-bold ${trend > 0 ? 'text-emerald-600 dark:text-emerald-400' : trend < 0 ? 'text-red-600 dark:text-red-400' : 'text-slate-500 dark:text-slate-400'} bg-emerald-50 dark:bg-emerald-950/30 border border-emerald-200 dark:border-emerald-700 px-2 py-1 rounded-lg`}>
                                {trend > 0 ? <ArrowUpRight size={12} /> : trend < 0 ? <ArrowDownRight size={12} /> : <Minus size={12} />}
                                {Math.abs(trend)}%
                            </div>
                        )}
                    </div>

                    <div className="flex items-end justify-between">
                        <div>
                            <h3 className="text-emerald-700 dark:text-emerald-400 text-sm font-bold mb-1">{label}</h3>
                            <p className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk']">
                                {typeof value === 'number' ? count.toLocaleString() : value}
                                {limit !== undefined && (
                                    <span className="text-sm text text-slate-500 font-normal ml-1">
                                        / {typeof limit === 'number' ? (limit >= 10000 ? (limit / 1000).toFixed(1).replace('.0', '') + 'k' : limit.toLocaleString()) : limit}
                                    </span>
                                )}
                            </p>
                        </div>
                        {data && (
                            <div className="h-12 w-24 -mb-2">
                                <ResponsiveContainer width="100%" height="100%">
                                    <LineChart data={data.map((d: any, i: number) => ({ val: d, i }))}>
                                        <YAxis hide domain={[0, 'auto']} />
                                        <Line type="monotone" dataKey="val" stroke="#10b981" strokeWidth={5} strokeOpacity={1} dot={false} isAnimationActive={false} />
                                    </LineChart>
                                </ResponsiveContainer>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    }

    // Normal box styling
    return (
        <div className={`${styles.bg} border ${styles.border} rounded-2xl p-6 relative overflow-hidden group hover:shadow-lg transition-all animate-fade-in`} style={{ animationDelay: `${delay}ms` }}>

            <div className="relative z-10">
                <div className="flex justify-between items-start mb-4">
                    <div className="flex items-center gap-3">
                        {/* Main Icon (Swapped to be first) */}
                        <div className={`w-10 h-10 rounded-xl flex items-center justify-center border ${styles.iconBox}`}>
                            <Icon size={20} className={styles.icon} />
                        </div>

                        {/* Period Badge */}
                        <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700 text-slate-500 dark:text-slate-400">
                            <Calendar size={12} strokeWidth={2.5} />
                            <span className="text-[10px] font-bold uppercase tracking-wide">{period}</span>
                        </div>
                    </div>

                    {limit !== undefined ? (
                        <div className="flex items-center gap-1 px-2 py-1 rounded-lg bg-slate-100 dark:bg-slate-950/50 text-xs font-bold text-slate-500 dark:text-slate-400">
                            <Minus size={12} />
                            <span className={`${percentage > 90 ? 'text-red-500' : ''}`}>{Math.round(percentage)}%</span>
                        </div>
                    ) : (
                        <div className={`flex items-center gap-1 text-xs font-bold ${trend > 0 ? 'text-emerald-600 dark:text-emerald-400' : trend < 0 ? 'text-red-600 dark:text-red-400' : 'text-slate-500 dark:text-slate-400'} bg-slate-100 dark:bg-slate-950/50 px-2 py-1 rounded-lg`}>
                            {trend > 0 ? <ArrowUpRight size={12} /> : trend < 0 ? <ArrowDownRight size={12} /> : <Minus size={12} />}
                            {Math.abs(trend)}%
                        </div>
                    )}
                </div>

                <div className="flex items-end justify-between">
                    <div>
                        <h3 className="text-slate-600 dark:text-slate-400 text-sm font-medium mb-1">{label}</h3>
                        <p className="text-3xl font-bold text-slate-900 dark:text-slate-100 font-['Space_Grotesk']">
                            {typeof value === 'number' ? count.toLocaleString() : value}
                            {limit !== undefined && (
                                <span className="text-sm text-slate-500 font-normal ml-1">
                                    / {typeof limit === 'number' ? (limit >= 10000 ? (limit / 1000).toFixed(1).replace('.0', '') + 'k' : limit.toLocaleString()) : limit}
                                </span>
                            )}
                        </p>
                    </div>
                    {data && (
                        <div className="h-12 w-24 -mb-2">
                            <ResponsiveContainer width="100%" height="100%">
                                <LineChart data={data.map((d: any, i: number) => ({ val: d, i }))}>
                                    <YAxis hide domain={[0, 'auto']} />
                                    <Line type="monotone" dataKey="val" stroke={styles.stroke} strokeWidth={4} strokeOpacity={1} dot={false} isAnimationActive={false} />
                                </LineChart>
                            </ResponsiveContainer>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

const QuickAction = ({ icon: Icon, color, bg, border, title, desc, onClick, comingSoon }: any) => (
    <button
        onClick={comingSoon ? undefined : onClick}
        disabled={comingSoon}
        className={`flex flex-col items-center text-center p-4 rounded-2xl bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 transition-all group relative overflow-hidden shadow-sm dark:shadow-none h-full
        ${comingSoon
                ? 'opacity-60 cursor-not-allowed grayscale-[0.5]'
                : 'hover:bg-slate-50 dark:hover:bg-slate-900'
            }`}
    >
        {comingSoon && (
            <div className="absolute top-2 right-2 px-1.5 py-0.5 rounded text-[9px] font-bold bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400 uppercase tracking-wide z-10">
                Soon
            </div>
        )}
        <div className={`w-12 h-12 rounded-2xl ${bg} ${border} border flex items-center justify-center mb-3 ${comingSoon ? '' : 'group-hover:scale-110'} transition-transform`}>
            <Icon size={24} className={color} />
        </div>
        <h3 className="font-bold text-slate-800 dark:text-slate-200 text-sm group-hover:text-slate-900 dark:group-hover:text-white">{title}</h3>
        <p className="text-xs text-slate-500 dark:text-slate-500 mt-1 group-hover:text-slate-600 dark:group-hover:text-slate-400 font-medium">{desc}</p>
    </button>
);

// Helper to resize images for display (prevents dashboard lag)
const createThumbnail = (src: string, maxWidth: number, maxHeight: number): Promise<string> => {
    return new Promise((resolve) => {
        const img = new Image();
        img.crossOrigin = "anonymous";
        img.onload = () => {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');

            // Calculate new dimensions to fit within bounds while maintaining aspect ratio
            let newWidth = img.width;
            let newHeight = img.height;

            // Resize logic: scale down to fit the bounding box
            const ratio = Math.min(1, maxWidth / newWidth, maxHeight / newHeight);

            newWidth = Math.floor(newWidth * ratio);
            newHeight = Math.floor(newHeight * ratio);

            canvas.width = newWidth;
            canvas.height = newHeight;

            if (ctx) {
                ctx.drawImage(img, 0, 0, newWidth, newHeight);
                // Use JPEG 0.7 for efficient thumbnail storage
                resolve(canvas.toDataURL('image/jpeg', 0.7));
            } else {
                resolve(src); // Fallback to original if context fails
            }
        };
        img.onerror = () => resolve(src); // Fallback to original on error
        img.src = src;
    });
};

const RecentCreations = ({ assets, onUpdate }: { assets: Asset[], onUpdate: () => void }) => {
    const [scrollRef, setScrollRef] = useState<HTMLDivElement | null>(null);
    const [thumbnails, setThumbnails] = useState<Record<string, string>>({});

    // Effect to generate thumbnails for new assets
    useEffect(() => {
        const generateThumbnails = async () => {
            // Identify images that don't have a thumbnail yet
            const toProcess = assets.filter(a => a.type === 'image' && !thumbnails[a.id]);

            if (toProcess.length === 0) return;

            const newThumbs: Record<string, string> = {};
            let hasNew = false;

            for (const item of toProcess) {
                // Only process if it has a URL and likely needs resizing (e.g. base64)
                if (item.url) {
                    try {
                        // User requested max size 270x175
                        const thumb = await createThumbnail(item.url, 270, 175);
                        newThumbs[item.id] = thumb;
                        hasNew = true;
                    } catch (e) {
                        console.error('Thumbnail generation failed for', item.id, e);
                    }
                }
            }

            if (hasNew) {
                setThumbnails(prev => ({ ...prev, ...newThumbs }));
            }
        };

        generateThumbnails();
    }, [assets, thumbnails]);

    const scroll = (dir: 'left' | 'right') => {
        if (scrollRef) {
            const scrollAmount = dir === 'left' ? -300 : 300;
            scrollRef.scrollBy({ left: scrollAmount, behavior: 'smooth' });
        }
    };

    return (
        <div className="space-y-4">
            <div className="flex items-center justify-between">
                <h2 className="text-xl font-bold text-slate-900 dark:text-slate-100">Recent Creations</h2>
                <div className="flex gap-2">
                    <button onClick={() => scroll('left')} className="p-2 rounded-lg border border-slate-200 dark:border-slate-800 hover:bg-slate-100 dark:hover:bg-slate-800 text-slate-600 dark:text-slate-400 transition"><ChevronLeft size={16} /></button>
                    <button onClick={() => scroll('right')} className="p-2 rounded-lg border border-slate-200 dark:border-slate-800 hover:bg-slate-100 dark:hover:bg-slate-800 text-slate-600 dark:text-slate-400 transition"><ChevronRight size={16} /></button>
                </div>
            </div>

            <div
                ref={setScrollRef}
                className="flex gap-4 overflow-x-auto pb-4 snap-x hide-scrollbar"
                style={{ scrollbarWidth: 'none', msOverflowStyle: 'none' }}
            >
                {assets.map((item) => (
                    <div key={item.id} className="snap-start shrink-0 w-64 group relative rounded-xl overflow-hidden bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 shadow-sm dark:shadow-none">
                        {item.type === 'video' ? (
                            <video
                                src={item.url}
                                className="w-full h-40 object-cover opacity-90 dark:opacity-70 group-hover:opacity-100 transition-opacity"
                                muted
                                loop
                                onMouseOver={e => e.currentTarget.play()}
                                onMouseOut={e => e.currentTarget.pause()}
                            />
                        ) : (
                            <img
                                src={thumbnails[item.id] || item.url}
                                alt="creation"
                                className="w-full h-40 object-cover opacity-90 dark:opacity-70 group-hover:opacity-100 transition-opacity"
                            />
                        )}

                        <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity flex flex-col justify-end p-4">
                            <div className="flex justify-end gap-2 mb-2">
                                <button onClick={(e) => { e.stopPropagation(); handleDownload(item.url, item.type, item.prompt || ''); }} className="p-1.5 bg-white/10 backdrop-blur rounded hover:bg-white/20 text-white transition"><Download size={14} /></button>
                            </div>
                        </div>
                        <div className="p-3">
                            <div className="flex items-center justify-between mb-1">
                                <span className={`text-[10px] font-bold uppercase px-1.5 py-0.5 rounded ${item.type === 'image' ? 'bg-purple-100 dark:bg-purple-500/10 text-purple-600 dark:text-purple-400' : 'bg-blue-100 dark:bg-blue-500/10 text-blue-600 dark:text-blue-400'}`}>{item.type}</span>
                                <span className="text-[10px] text-slate-500">{formatTimeAgo(item.createdAt)}</span>
                            </div>
                            <p className="text-sm font-bold text-slate-700 dark:text-slate-300 truncate">{item.prompt}</p>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
};

const TipsCarousel = () => {
    const [active, setActive] = useState(0);
    const [tips, setTips] = useState<Tip[]>([]);

    useEffect(() => {
        const loadTips = async () => {
            const allTips = await adminService.getAllActiveTips();
            setTips(allTips);
        };
        loadTips();
    }, []);

    useEffect(() => {
        if (tips.length === 0) return;
        const timer = setInterval(() => setActive(prev => (prev + 1) % tips.length), 8000);
        return () => clearInterval(timer);
    }, [tips.length]);

    if (tips.length === 0) return null;

    return (
        <>
            <div className="bg-gradient-to-br from-slate-800 to-slate-950 dark:from-slate-900 dark:to-slate-950 border border-slate-700 dark:border-slate-800 rounded-2xl p-6 relative overflow-hidden shadow-lg">
                <div className="absolute top-0 right-0 p-10 bg-amber-500/10 blur-3xl rounded-full"></div>
                <div className="flex items-center gap-2 mb-4 text-amber-500">
                    <Lightbulb size={18} />
                    <span className="text-sm font-bold uppercase tracking-wider">Tips of the Day</span>
                </div>
                <div className="relative h-28">
                    {tips.map((tip, i) => (
                        <div
                            key={tip.id}
                            className={`absolute inset-0 transition-all duration-500 ${i === active ? 'opacity-100 translate-x-0' : 'opacity-0 translate-x-8 pointer-events-none'}`}
                        >
                            <h4 className="font-bold text-slate-100 mb-2">{tip.title}</h4>
                            <p className="text-sm text-slate-300 leading-relaxed font-medium line-clamp-3">{tip.text}</p>
                        </div>
                    ))}
                </div>
                <div className="flex justify-center gap-1.5 mt-2">
                    {tips.map((_, i) => (
                        <button
                            key={i}
                            onClick={() => setActive(i)}
                            className={`w-1.5 h-1.5 rounded-full transition-colors ${i === active ? 'bg-amber-500' : 'bg-slate-700'}`}
                        />
                    ))}
                </div>
            </div>

            {/* Full Tip Modal */}
            {/* Full Tip Modal Removed to prevent popup confusion */}
        </>
    )
}

const MarketInsightItem = ({ label, value, sub, color, icon }: any) => (
    <div className="flex flex-col justify-between p-3 rounded-xl bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 hover:border-slate-300 dark:hover:border-slate-700 transition-colors group">
        <div className="flex items-center justify-between mb-2">
            <span className="text-[10px] uppercase font-bold text-slate-500 flex items-center gap-1.5">
                {icon && <span className="text-slate-400">{icon}</span>}
                {label}
            </span>
            {sub && <span className="text-[9px] font-bold bg-emerald-100 dark:bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 px-1.5 py-0.5 rounded">{sub}</span>}
        </div>
        <div className={`text-sm font-bold ${color} truncate`} title={value}>
            {value}
        </div>
    </div>
);

const ActivityItem: React.FC<{ type: string; text: string; time: string; cost?: number }> = ({ type, text, time, cost }) => {
    const icons: any = {
        // Original types
        generate: { icon: Sparkles, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        setting: { icon: Settings, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        system: { icon: Zap, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        error: { icon: AlertCircle, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },

        // Specific activity types
        trend: { icon: TrendingUp, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        review: { icon: SearchCheck, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        image: { icon: ImageIcon, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        video: { icon: Video, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        metadata: { icon: ScanEye, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        prompt: { icon: FileText, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        edit: { icon: Palette, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },

        // Profile & Security activity types
        profile: { icon: UserIcon, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        password: { icon: Lock, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' },
        apikey: { icon: Key, color: 'text-blue-600 dark:text-blue-400', bg: 'bg-blue-100 dark:bg-blue-900', border: 'border-blue-200 dark:border-blue-700' }
    };

    // Detect activity type from description text
    let detectedType = type;
    if (text) {
        const lowerText = text.toLowerCase();
        if (lowerText.includes('password')) detectedType = 'password';
        else if (lowerText.includes('profile') || lowerText.includes('name') || lowerText.includes('photo')) detectedType = 'profile';
        else if (lowerText.includes('api key') || lowerText.includes('validated')) detectedType = 'apikey';
        else if (lowerText.includes('review') || lowerText.includes('reviewer')) detectedType = 'review';
        else if (lowerText.includes('metadata')) detectedType = 'metadata';
        else if (lowerText.includes('image') && !lowerText.includes('edit')) detectedType = 'image';
        else if (lowerText.includes('video')) detectedType = 'video';
        else if (lowerText.includes('prompt')) detectedType = 'prompt';
        else if (lowerText.includes('edit')) detectedType = 'edit';
        else if (lowerText.includes('trend') || lowerText.includes('analysis')) detectedType = 'trend';
    }

    // Fallback for unknown types
    const style = icons[detectedType] || icons.setting;
    const Icon = style.icon;

    return (
        <div className="flex gap-4 relative z-10">
            <div className={`w-10 h-10 rounded-full shrink-0 ${style.bg} ${style.border} border flex items-center justify-center z-10`}>
                <Icon size={16} className={style.color} />
            </div>
            <div className="pt-1 flex-1">
                <p className="text-sm text-slate-800 dark:text-slate-300 font-medium leading-relaxed">{text || 'Activity recorded'}</p>
                <div className="flex items-center gap-2 mt-0.5">
                    <p className="text-xs text-slate-500 dark:text-slate-500 font-medium">{time}</p>
                    {cost !== undefined && cost > 0 && (
                        <span className="text-[10px] font-bold px-1.5 py-0.5 rounded bg-blue-100 dark:bg-blue-900/50 text-blue-600 dark:text-blue-400 border border-blue-200 dark:border-blue-700">
                            -{cost} credit{cost !== 1 ? 's' : ''}
                        </span>
                    )}
                </div>
            </div>
        </div>
    );
};

export default Dashboard;